# backend/app/services/weekly_pdf_generator.py
"""
Generador de PDF Semanal - Issue #552

Genera el "latido" del sistema: un PDF resumen que el farmacéutico
encuentra cada lunes en su carpeta sin necesidad de abrir la app.

Tecnología: ReportLab
Diseño: A4, optimizado para impresión B/N, escaneable en 30 segundos
"""

import io
import structlog
from datetime import date, datetime, timezone
from typing import Optional
from pathlib import Path

from reportlab.lib import colors
from reportlab.lib.pagesizes import A4
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import mm, cm
from reportlab.lib.enums import TA_CENTER, TA_LEFT, TA_RIGHT
from reportlab.platypus import (
    SimpleDocTemplate,
    Paragraph,
    Spacer,
    Table,
    TableStyle,
    Image,
    HRFlowable,
)
from reportlab.graphics.shapes import Drawing, Rect, String
from reportlab.graphics.charts.barcharts import VerticalBarChart

from app.schemas.reports import DireccionReportData, CategoryGrowth, InsightSummary

logger = structlog.get_logger(__name__)

# Colores corporativos kaiFarma
KAIFARMA_PRIMARY = colors.HexColor("#6366F1")  # Indigo
KAIFARMA_SECONDARY = colors.HexColor("#10B981")  # Emerald
KAIFARMA_ACCENT = colors.HexColor("#F59E0B")  # Amber
KAIFARMA_DARK = colors.HexColor("#1F2937")  # Gray 800
KAIFARMA_LIGHT = colors.HexColor("#F3F4F6")  # Gray 100


class WeeklyPDFGenerator:
    """
    Generador de PDF Semanal para farmacias.

    Secciones del PDF:
    1. Header con logo y datos de farmacia
    2. Resumen KPIs: Ventas semana vs año pasado
    3. Top 3 Categorías con crecimiento
    4. Alertas prioritarias (top 3)
    5. Recomendación principal
    6. Footer con fecha de generación
    """

    def __init__(self):
        self.page_width, self.page_height = A4
        self.margin = 2 * cm
        self.styles = self._create_styles()

    def _create_styles(self) -> dict:
        """Crear estilos personalizados para el PDF."""
        base_styles = getSampleStyleSheet()

        styles = {
            "title": ParagraphStyle(
                "title",
                parent=base_styles["Heading1"],
                fontSize=24,
                textColor=KAIFARMA_DARK,
                spaceAfter=6 * mm,
                alignment=TA_CENTER,
            ),
            "subtitle": ParagraphStyle(
                "subtitle",
                parent=base_styles["Heading2"],
                fontSize=14,
                textColor=KAIFARMA_PRIMARY,
                spaceAfter=4 * mm,
                spaceBefore=6 * mm,
            ),
            "section_header": ParagraphStyle(
                "section_header",
                parent=base_styles["Heading3"],
                fontSize=12,
                textColor=KAIFARMA_DARK,
                spaceAfter=3 * mm,
                spaceBefore=4 * mm,
                borderPadding=2 * mm,
            ),
            "body": ParagraphStyle(
                "body",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=KAIFARMA_DARK,
                spaceAfter=2 * mm,
            ),
            "kpi_value": ParagraphStyle(
                "kpi_value",
                parent=base_styles["Normal"],
                fontSize=20,
                textColor=KAIFARMA_PRIMARY,
                alignment=TA_CENTER,
                fontName="Helvetica-Bold",
            ),
            "kpi_label": ParagraphStyle(
                "kpi_label",
                parent=base_styles["Normal"],
                fontSize=9,
                textColor=colors.gray,
                alignment=TA_CENTER,
            ),
            "kpi_change_positive": ParagraphStyle(
                "kpi_change_positive",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=KAIFARMA_SECONDARY,
                alignment=TA_CENTER,
            ),
            "kpi_change_negative": ParagraphStyle(
                "kpi_change_negative",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=colors.red,
                alignment=TA_CENTER,
            ),
            "alert_high": ParagraphStyle(
                "alert_high",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=colors.red,
                leftIndent=5 * mm,
            ),
            "alert_medium": ParagraphStyle(
                "alert_medium",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=KAIFARMA_ACCENT,
                leftIndent=5 * mm,
            ),
            "recommendation": ParagraphStyle(
                "recommendation",
                parent=base_styles["Normal"],
                fontSize=11,
                textColor=KAIFARMA_DARK,
                backColor=KAIFARMA_LIGHT,
                borderPadding=8,
                spaceAfter=4 * mm,
            ),
            "footer": ParagraphStyle(
                "footer",
                parent=base_styles["Normal"],
                fontSize=8,
                textColor=colors.gray,
                alignment=TA_CENTER,
            ),
        }

        return styles

    def generate(
        self,
        data: DireccionReportData,
        output_path: Optional[str] = None,
    ) -> bytes:
        """
        Genera el PDF semanal.

        Args:
            data: Datos del reporte (de ReportService.get_direccion_data)
            output_path: Ruta opcional para guardar el PDF

        Returns:
            bytes: Contenido del PDF
        """
        logger.info(
            "weekly_pdf_generator.generate.start",
            pharmacy=data.pharmacy_name,
            period_start=str(data.period_start),
            period_end=str(data.period_end),
        )

        # Buffer para el PDF
        buffer = io.BytesIO()

        # Crear documento
        doc = SimpleDocTemplate(
            buffer,
            pagesize=A4,
            rightMargin=self.margin,
            leftMargin=self.margin,
            topMargin=self.margin,
            bottomMargin=self.margin,
        )

        # Construir contenido
        elements = []

        # 1. Header
        elements.extend(self._build_header(data))

        # 2. KPIs principales
        elements.extend(self._build_kpis_section(data))

        # 3. Top categorías
        if data.top_categories:
            elements.extend(self._build_categories_section(data.top_categories))

        # 4. Alertas
        if data.alerts:
            elements.extend(self._build_alerts_section(data.alerts[:3]))

        # 5. Recomendación
        if data.recommendation:
            elements.extend(self._build_recommendation_section(data.recommendation))

        # 6. Footer
        elements.extend(self._build_footer(data))

        # Generar PDF
        doc.build(elements)

        # Obtener bytes
        pdf_bytes = buffer.getvalue()
        buffer.close()

        # Guardar si se especificó ruta
        if output_path:
            Path(output_path).parent.mkdir(parents=True, exist_ok=True)
            with open(output_path, "wb") as f:
                f.write(pdf_bytes)
            logger.info(
                "weekly_pdf_generator.generate.saved",
                path=output_path,
                size_kb=len(pdf_bytes) // 1024,
            )

        logger.info(
            "weekly_pdf_generator.generate.complete",
            pharmacy=data.pharmacy_name,
            size_kb=len(pdf_bytes) // 1024,
        )

        return pdf_bytes

    def _build_header(self, data: DireccionReportData) -> list:
        """Construir header con logo y datos de farmacia."""
        elements = []

        # Título principal
        title = Paragraph(
            f"<b>kaiFarma</b> - Resumen Semanal",
            self.styles["title"],
        )
        elements.append(title)

        # Nombre de farmacia
        pharmacy_name = Paragraph(
            f"<b>{data.pharmacy_name}</b>",
            self.styles["subtitle"],
        )
        elements.append(pharmacy_name)

        # Período
        period_text = f"Semana del {data.period_start.strftime('%d/%m/%Y')} al {data.period_end.strftime('%d/%m/%Y')}"
        period = Paragraph(period_text, self.styles["body"])
        elements.append(period)

        # Línea separadora
        elements.append(Spacer(1, 4 * mm))
        elements.append(
            HRFlowable(
                width="100%",
                thickness=1,
                color=KAIFARMA_PRIMARY,
                spaceBefore=2 * mm,
                spaceAfter=4 * mm,
            )
        )

        return elements

    def _build_kpis_section(self, data: DireccionReportData) -> list:
        """Construir sección de KPIs principales."""
        elements = []

        elements.append(
            Paragraph("Resumen de la Semana", self.styles["section_header"])
        )

        # Formatear valores
        def format_currency(value: float) -> str:
            if value >= 1000:
                return f"{value/1000:.1f}K"
            return f"{value:.0f}"

        def format_change(value: Optional[float]) -> tuple:
            if value is None:
                return ("--", self.styles["body"])
            sign = "+" if value >= 0 else ""
            style = (
                self.styles["kpi_change_positive"]
                if value >= 0
                else self.styles["kpi_change_negative"]
            )
            return (f"{sign}{value:.1f}%", style)

        # Datos para la tabla de KPIs
        sales_change_text, sales_change_style = format_change(data.sales_yoy_change)
        margin_change_text, margin_change_style = format_change(data.margin_yoy_change)

        kpi_data = [
            # Fila 1: Valores
            [
                Paragraph(f"{format_currency(data.total_sales)}€", self.styles["kpi_value"]),
                Paragraph(f"{data.margin_percent:.1f}%", self.styles["kpi_value"]),
                Paragraph(f"{format_currency(data.stock_value)}€", self.styles["kpi_value"]),
                Paragraph(f"{data.sku_count:,}", self.styles["kpi_value"]),
            ],
            # Fila 2: Labels
            [
                Paragraph("Ventas", self.styles["kpi_label"]),
                Paragraph("Margen", self.styles["kpi_label"]),
                Paragraph("Stock", self.styles["kpi_label"]),
                Paragraph("Productos", self.styles["kpi_label"]),
            ],
            # Fila 3: Cambios YoY
            [
                Paragraph(sales_change_text, sales_change_style),
                Paragraph(margin_change_text, margin_change_style),
                Paragraph("--", self.styles["body"]),
                Paragraph("--", self.styles["body"]),
            ],
        ]

        # Calcular anchos de columna
        col_width = (self.page_width - 2 * self.margin) / 4

        kpi_table = Table(
            kpi_data,
            colWidths=[col_width] * 4,
            rowHeights=[15 * mm, 6 * mm, 6 * mm],
        )

        kpi_table.setStyle(
            TableStyle(
                [
                    ("ALIGN", (0, 0), (-1, -1), "CENTER"),
                    ("VALIGN", (0, 0), (-1, -1), "MIDDLE"),
                    ("BOX", (0, 0), (-1, -1), 1, KAIFARMA_LIGHT),
                    ("LINEBELOW", (0, 0), (-1, 0), 0.5, KAIFARMA_LIGHT),
                    ("BACKGROUND", (0, 0), (-1, 0), colors.white),
                ]
            )
        )

        elements.append(kpi_table)
        elements.append(Spacer(1, 6 * mm))

        return elements

    def _build_categories_section(self, categories: list[CategoryGrowth]) -> list:
        """Construir sección de top categorías."""
        elements = []

        elements.append(
            Paragraph("Top Categorías (vs Año Anterior)", self.styles["section_header"])
        )

        # Tabla de categorías
        table_data = [["Categoría", "Ventas", "Cambio YoY", "Productos"]]

        for cat in categories[:3]:
            change_color = (
                KAIFARMA_SECONDARY if cat.yoy_change >= 0 else colors.red
            )
            sign = "+" if cat.yoy_change >= 0 else ""

            table_data.append(
                [
                    cat.display_name,
                    f"{cat.sales:,.0f}€",
                    f"{sign}{cat.yoy_change:.1f}%",
                    str(cat.product_count),
                ]
            )

        col_widths = [
            (self.page_width - 2 * self.margin) * 0.35,
            (self.page_width - 2 * self.margin) * 0.25,
            (self.page_width - 2 * self.margin) * 0.20,
            (self.page_width - 2 * self.margin) * 0.20,
        ]

        cat_table = Table(table_data, colWidths=col_widths)
        cat_table.setStyle(
            TableStyle(
                [
                    ("BACKGROUND", (0, 0), (-1, 0), KAIFARMA_PRIMARY),
                    ("TEXTCOLOR", (0, 0), (-1, 0), colors.white),
                    ("ALIGN", (0, 0), (-1, 0), "CENTER"),
                    ("FONTNAME", (0, 0), (-1, 0), "Helvetica-Bold"),
                    ("FONTSIZE", (0, 0), (-1, 0), 9),
                    ("BOTTOMPADDING", (0, 0), (-1, 0), 8),
                    ("TOPPADDING", (0, 0), (-1, 0), 8),
                    ("ALIGN", (1, 1), (-1, -1), "RIGHT"),
                    ("FONTSIZE", (0, 1), (-1, -1), 9),
                    ("BOTTOMPADDING", (0, 1), (-1, -1), 6),
                    ("TOPPADDING", (0, 1), (-1, -1), 6),
                    ("LINEBELOW", (0, 0), (-1, -2), 0.5, KAIFARMA_LIGHT),
                    ("BOX", (0, 0), (-1, -1), 1, KAIFARMA_LIGHT),
                ]
            )
        )

        elements.append(cat_table)
        elements.append(Spacer(1, 6 * mm))

        return elements

    def _build_alerts_section(self, alerts: list[InsightSummary]) -> list:
        """Construir sección de alertas prioritarias."""
        elements = []

        elements.append(
            Paragraph("Alertas Prioritarias", self.styles["section_header"])
        )

        for i, alert in enumerate(alerts[:3], 1):
            # Determinar estilo según severidad
            style = (
                self.styles["alert_high"]
                if alert.severity == "high"
                else self.styles["alert_medium"]
            )

            # Icono según severidad
            icon = "⚠️" if alert.severity == "high" else "💡"

            # Valor económico
            value_text = ""
            if alert.economic_value > 0:
                value_text = f" ({alert.economic_value:,.0f}€)"

            alert_text = f"{icon} <b>{alert.title}</b>{value_text}"
            elements.append(Paragraph(alert_text, style))

            # Acción sugerida
            if alert.action_label:
                action_text = f"    → {alert.action_label}"
                elements.append(Paragraph(action_text, self.styles["body"]))

            elements.append(Spacer(1, 2 * mm))

        elements.append(Spacer(1, 4 * mm))

        return elements

    def _build_recommendation_section(self, recommendation: str) -> list:
        """Construir sección de recomendación principal."""
        elements = []

        elements.append(
            Paragraph("Acción Recomendada", self.styles["section_header"])
        )

        rec_text = f"📋 {recommendation}"
        elements.append(Paragraph(rec_text, self.styles["recommendation"]))

        elements.append(Spacer(1, 6 * mm))

        return elements

    def _build_footer(self, data: DireccionReportData) -> list:
        """Construir footer con información de generación."""
        elements = []

        elements.append(
            HRFlowable(
                width="100%",
                thickness=0.5,
                color=KAIFARMA_LIGHT,
                spaceBefore=6 * mm,
                spaceAfter=4 * mm,
            )
        )

        # Fecha de generación
        generated_text = f"Generado: {datetime.now().strftime('%d/%m/%Y %H:%M')}"
        elements.append(Paragraph(generated_text, self.styles["footer"]))

        # Disclaimer
        disclaimer = (
            "kaiFarma - El valor del momento exacto | "
            "Este informe es orientativo y no sustituye el análisis profesional."
        )
        elements.append(Paragraph(disclaimer, self.styles["footer"]))

        return elements


# Instancia singleton
weekly_pdf_generator = WeeklyPDFGenerator()
