# backend/app/services/venta_libre_catalog_service.py
"""
Servicio para gestión del catálogo de productos Venta Libre.

Issue #473: Fuzzy matching para detectar duplicados semánticos.
Issue #474: Conciliación cross-farmacia por EAN/CN.

Este servicio proporciona la lógica central para:
1. Buscar productos existentes por EAN, CN o nombre
2. Fuzzy matching para detectar duplicados semánticos
3. Crear nuevos productos con tracking de fuentes
4. Actualizar productos cuando se ven en nuevas farmacias
5. Conciliar duplicados detectados

Orden de búsqueda para find_or_create:
1. EAN-13 exacto (máxima confianza - código de barras único)
2. CN en cn_codes (alta confianza - código oficial)
3. Nombre normalizado exacto (confianza media)
4. Fuzzy matching por tokens (Issue #473)
5. Crear nuevo si no hay match
"""

import logging
from dataclasses import dataclass
from datetime import datetime, timezone
from typing import List, Optional, Tuple
from uuid import UUID, uuid4

from sqlalchemy import func, text
from sqlalchemy.orm import Session

from app.models import ProductCatalogVentaLibre

logger = logging.getLogger(__name__)


def get_known_product_type(
    db: Session,
    product_name: str,
    ean: Optional[str] = None,
    cn: Optional[str] = None,
) -> Optional[str]:
    """
    Determina el tipo de producto consultando fuentes autorizadas.

    Issue #476: Clasificación robusta de productos.

    Orden de verificación:
    1. SalesEnrichment: Si ya vendimos el producto, sabemos qué tipo es
    2. ProductCatalog (CIMA): Si existe en CIMA, es medicamento de receta

    Args:
        db: Database session
        product_name: Nombre del producto
        ean: EAN-13 opcional
        cn: Código nacional opcional

    Returns:
        'prescription' | 'venta_libre' | None (si no encontrado en ninguna fuente)
    """
    # === 1. Verificar en SalesEnrichment (historial de ventas) ===
    se_conditions = []
    se_params = {}

    if product_name:
        normalized = product_name.lower().strip()
        se_conditions.append("LOWER(TRIM(sd.product_name)) = :name")
        se_params["name"] = normalized

    if ean and len(ean) == 13:
        se_conditions.append("sd.ean13 = :ean")
        se_params["ean"] = ean

    if cn and len(cn) in (6, 7):
        se_conditions.append("sd.codigo_nacional = :cn")
        se_params["cn"] = cn

    if se_conditions:
        # Check prescription first (priority)
        prescription_query = f"""
            SELECT 1 FROM sales_enrichment se
            JOIN sales_data sd ON se.sales_data_id = sd.id
            WHERE ({" OR ".join(se_conditions)})
            AND se.product_type = 'prescription'
            LIMIT 1
        """
        result = db.execute(text(prescription_query), se_params)
        if result.fetchone():
            logger.debug(
                f"Found product type in SalesEnrichment: {product_name[:30]} -> prescription"
            )
            return "prescription"

        # Then check venta_libre
        vl_query = f"""
            SELECT 1 FROM sales_enrichment se
            JOIN sales_data sd ON se.sales_data_id = sd.id
            WHERE ({" OR ".join(se_conditions)})
            AND se.product_type = 'venta_libre'
            LIMIT 1
        """
        result = db.execute(text(vl_query), se_params)
        if result.fetchone():
            logger.debug(
                f"Found product type in SalesEnrichment: {product_name[:30]} -> venta_libre"
            )
            return "venta_libre"

    # === 2. Verificar en ProductCatalog (CIMA) ===
    # Si existe en CIMA, es medicamento de receta
    cima_conditions = []
    cima_params = {}

    if cn and len(cn) in (6, 7):
        # CN es el identificador más fiable para CIMA
        cima_conditions.append("pc.national_code = :cn")
        cima_params["cn"] = cn

    # Note: ProductCatalog no tiene columna EAN directa
    # Solo usamos national_code para verificación CIMA

    if cima_conditions:
        cima_query = f"""
            SELECT 1
            FROM product_catalog pc
            WHERE ({" OR ".join(cima_conditions)})
            LIMIT 1
        """
        result = db.execute(text(cima_query), cima_params)
        if result.fetchone():
            logger.debug(
                f"Found product in CIMA catalog (prescription): {product_name[:30]}"
            )
            return "prescription"

    # No encontrado en ninguna fuente
    return None


class VentaLibreCatalogService:
    """
    Servicio para operaciones CRUD en ProductCatalogVentaLibre.

    Implementa la lógica de conciliación cross-farmacia usando
    múltiples identificadores (EAN, CN, nombre).
    """

    def __init__(self, db: Session):
        self.db = db

    def find_by_ean(self, ean: str) -> Optional[ProductCatalogVentaLibre]:
        """
        Busca producto por EAN-13 en ean13 principal o ean_codes array.

        Issue #481: Búsqueda en dos etapas para multi-EAN support:
        1. Primero busca en ean13 (B-tree index, más rápido)
        2. Si no encuentra, busca en ean_codes array (GIN index)

        Solo busca si el EAN es válido (pasa checksum).

        Args:
            ean: Código EAN-13 de 13 dígitos

        Returns:
            Producto encontrado o None
        """
        if not ean or not ProductCatalogVentaLibre.is_valid_ean13(ean):
            return None

        # 1. Buscar en ean13 principal (B-tree index, más rápido)
        result = (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.ean13 == ean)
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .first()
        )

        if result:
            return result

        # 2. Buscar en ean_codes array (GIN index con operador @>)
        return (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.ean_codes.contains([ean]))
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .first()
        )

    def find_by_cn(self, cn: str) -> Optional[ProductCatalogVentaLibre]:
        """
        Busca producto por código nacional en el array cn_codes.

        Solo busca CNs válidos (6-7 dígitos numéricos).

        Args:
            cn: Código nacional de 6-7 dígitos

        Returns:
            Producto encontrado o None
        """
        if not cn or len(cn) < 6 or len(cn) > 7 or not cn.isdigit():
            return None

        # Búsqueda en array usando el operador @> (contains)
        return (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.cn_codes.contains([cn]))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .first()
        )

    def find_by_name(self, product_name: str) -> Optional[ProductCatalogVentaLibre]:
        """
        Busca producto por nombre normalizado exacto.

        Args:
            product_name: Nombre del producto (se normalizará)

        Returns:
            Producto encontrado o None
        """
        if not product_name:
            return None

        normalized = ProductCatalogVentaLibre.normalize_product_name(product_name)
        if not normalized:
            return None

        return (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.product_name_normalized == normalized)
            .filter(ProductCatalogVentaLibre.is_active == True)
            .first()
        )

    def find_exact_match(
        self,
        product_name: str | None = None,
        ean13: str | None = None,
        cn: str | None = None,
    ) -> Optional[ProductCatalogVentaLibre]:
        """
        Busca producto por match exacto usando EAN, CN o nombre (en ese orden).

        Issue #476: Método para clasificación de inventario.

        Orden de búsqueda:
        1. EAN-13 exacto (si válido)
        2. Código Nacional en cn_codes array
        3. Nombre normalizado exacto

        Args:
            product_name: Nombre del producto
            ean13: Código EAN-13
            cn: Código Nacional (6-7 dígitos)

        Returns:
            Producto encontrado o None
        """
        # 1. Try EAN first (highest priority)
        if ean13:
            match = self.find_by_ean(ean13)
            if match:
                return match

        # 2. Try CN second
        if cn:
            match = self.find_by_cn(cn)
            if match:
                return match

        # 3. Try normalized name last
        if product_name:
            match = self.find_by_name(product_name)
            if match:
                return match

        return None

    def _find_review_match(
        self,
        product_name: str,
    ) -> Optional[Tuple[ProductCatalogVentaLibre, float]]:
        """
        Busca producto con similitud 70-85% (zona de revisión).

        Issue #475: Hook para detectar duplicados potenciales cuando se crea
        un nuevo producto. No fusiona automáticamente pero marca para revisión.

        Args:
            product_name: Nombre del producto a buscar

        Returns:
            Tuple (producto_similar, similitud) si hay match en zona de revisión, None si no
        """
        if not product_name:
            return None

        try:
            from app.services.product_matching_service import (
                AUTO_MERGE_THRESHOLD,
                REVIEW_THRESHOLD,
                get_product_matching_service,
            )
        except ImportError:
            logger.warning("ProductMatchingService not available, skipping review match")
            return None

        matching_service = get_product_matching_service(self.db)

        # Buscar matches en zona de revisión (70-85%)
        matches = matching_service.find_similar_products(
            product_name=product_name,
            threshold=REVIEW_THRESHOLD,
            limit=1,
        )

        if matches:
            best_match = matches[0]
            # Solo considerar si está en zona de revisión (no auto-merge)
            if best_match.similarity < AUTO_MERGE_THRESHOLD:
                logger.debug(
                    f"Found review match: '{product_name[:30]}' -> "
                    f"'{best_match.product.product_name_display[:30]}' "
                    f"(similarity: {best_match.similarity:.2%})"
                )
                return (best_match.product, best_match.similarity)

        return None

    def find_by_fuzzy_match(
        self,
        product_name: str,
        auto_merge_only: bool = True,
    ) -> Optional[ProductCatalogVentaLibre]:
        """
        Busca producto por fuzzy matching de tokens (Issue #473).

        Args:
            product_name: Nombre del producto a buscar
            auto_merge_only: Si True, solo retorna matches >= 85% (auto-merge)
                           Si False, también incluye matches 70-85% (para review)

        Returns:
            Producto encontrado o None
        """
        if not product_name:
            return None

        try:
            from app.services.product_matching_service import (
                AUTO_MERGE_THRESHOLD,
                MIN_MATCH_THRESHOLD,
                get_product_matching_service,
            )
        except ImportError:
            logger.warning("ProductMatchingService not available, skipping fuzzy match")
            return None

        threshold = AUTO_MERGE_THRESHOLD if auto_merge_only else MIN_MATCH_THRESHOLD
        matching_service = get_product_matching_service(self.db)

        matches = matching_service.find_similar_products(
            product_name=product_name,
            threshold=threshold,
            limit=1,
        )

        if matches:
            best_match = matches[0]
            logger.debug(
                f"Fuzzy match found: '{product_name[:30]}' -> "
                f"'{best_match.product.product_name_display[:30]}' "
                f"(similarity: {best_match.similarity:.2%})"
            )
            return best_match.product

        return None

    def find_or_create(
        self,
        product_name: str,
        pharmacy_id: UUID,
        ean13: Optional[str] = None,
        cn: Optional[str] = None,
        ml_category: Optional[str] = None,
        ml_confidence: Optional[float] = None,
        prediction_source: Optional[str] = None,
        detected_brand: Optional[str] = None,
    ) -> Tuple[Optional[ProductCatalogVentaLibre], bool]:
        """
        Busca o crea un producto en el catálogo.

        Orden de búsqueda:
        1. EAN-13 exacto (máxima confianza)
        2. CN en cn_codes (alta confianza)
        3. Nombre normalizado exacto (confianza media)
        4. Crear nuevo si no hay match

        Returns:
            Tuple (producto, was_created)
            - producto: El ProductCatalogVentaLibre encontrado/creado,
                       o None si es medicamento de receta conocido
            - was_created: True si se creó nuevo, False en otros casos

        Cuando encuentra un match, actualiza:
        - pharmacy_ids_seen
        - product_name_variants (si el nombre es diferente)
        - cn_codes (si el CN es nuevo)
        - ean13 (si no teníamos uno y el nuevo es válido)

        Args:
            product_name: Nombre del producto (requerido)
            pharmacy_id: UUID de la farmacia que reporta
            ean13: Código EAN-13 opcional
            cn: Código nacional opcional
            ml_category: Categoría ML si ya está clasificado
            ml_confidence: Confianza de la clasificación
            prediction_source: Fuente de la predicción
            detected_brand: Marca detectada

        Returns:
            Tuple (producto, was_created)
            - producto: El ProductCatalogVentaLibre encontrado o creado
            - was_created: True si se creó nuevo, False si ya existía
        """
        if not product_name:
            raise ValueError("product_name is required")

        # === 1. Buscar por EAN-13 ===
        if ean13:
            existing = self.find_by_ean(ean13)
            if existing:
                logger.debug(f"Found product by EAN: {ean13} -> {existing.id}")
                existing.add_pharmacy_source(
                    pharmacy_id=pharmacy_id,
                    product_name=product_name,
                    cn=cn,
                    ean=ean13
                )
                self.db.flush()  # Persist ARRAY column changes
                return existing, False

        # === 2. Buscar por CN ===
        if cn:
            existing = self.find_by_cn(cn)
            if existing:
                logger.debug(f"Found product by CN: {cn} -> {existing.id}")
                existing.add_pharmacy_source(
                    pharmacy_id=pharmacy_id,
                    product_name=product_name,
                    cn=cn,
                    ean=ean13
                )
                self.db.flush()  # Persist ARRAY column changes
                return existing, False

        # === 3. Buscar por nombre normalizado ===
        existing = self.find_by_name(product_name)
        if existing:
            logger.debug(f"Found product by name: {product_name[:50]} -> {existing.id}")
            existing.add_pharmacy_source(
                pharmacy_id=pharmacy_id,
                product_name=product_name,
                cn=cn,
                ean=ean13
            )
            self.db.flush()  # Persist ARRAY column changes
            return existing, False

        # === 4. Fuzzy matching por tokens (Issue #473) ===
        # 4a. Primero buscar auto-merge (>85% similitud)
        fuzzy_match = self.find_by_fuzzy_match(product_name, auto_merge_only=True)
        if fuzzy_match:
            logger.debug(
                f"Found product by fuzzy match: {product_name[:50]} -> {fuzzy_match.id} "
                f"(original: {fuzzy_match.product_name_display[:50]})"
            )
            fuzzy_match.add_pharmacy_source(
                pharmacy_id=pharmacy_id,
                product_name=product_name,
                cn=cn,
                ean=ean13
            )
            self.db.flush()  # Persist ARRAY column changes
            return fuzzy_match, False

        # 4b. Buscar matches para revisión (70-85% similitud) - Issue #475
        # Estos no se fusionan automáticamente pero se marcan como posibles duplicados
        review_match = self._find_review_match(product_name)

        # === 5. Verificar si es medicamento de receta (Issue #476) ===
        # Consultar SalesEnrichment - si ya vendimos este producto, sabemos qué es
        known_type = get_known_product_type(self.db, product_name, ean13, cn)
        if known_type == "prescription":
            logger.info(
                f"Skipping VL creation - product is prescription: {product_name[:50]}"
            )
            # Return None to indicate this should NOT be in VentaLibre
            return None, False

        # === 6. Crear nuevo producto VentaLibre ===
        normalized = ProductCatalogVentaLibre.normalize_product_name(product_name)

        # Auto-clasificar si no viene categoría (Issue #476: punto de convergencia)
        if not ml_category or ml_category in ("unknown", "otros"):
            try:
                from app.services.necesidad_classifier_service import classify_product
                ml_category, ml_confidence = classify_product(
                    product_name,
                    default_category="unknown",
                    default_confidence=0.0,
                )
                if ml_category != "unknown":
                    prediction_source = "necesidad_classifier"
                    logger.debug(f"Auto-classified: {product_name[:40]} -> {ml_category} ({ml_confidence:.2f})")
            except ImportError:
                logger.warning("necesidad_classifier_service not available, skipping auto-classification")

        # Validar EAN antes de guardar
        valid_ean = ean13 if ean13 and ProductCatalogVentaLibre.is_valid_ean13(ean13) else None

        # Preparar cn_codes inicial
        cn_codes = []
        if cn and len(cn) >= 6 and len(cn) <= 7 and cn.isdigit():
            cn_codes.append(cn)

        # Extraer CN del EAN si aplica
        if valid_ean:
            extracted_cn = ProductCatalogVentaLibre.extract_cn_from_ean(valid_ean)
            if extracted_cn and extracted_cn not in cn_codes:
                cn_codes.append(extracted_cn)

        # Generar tokens para fuzzy matching (Issue #473)
        tokens = ProductCatalogVentaLibre.normalize_for_matching(product_name)

        # Issue #481: Inicializar ean_codes con el EAN válido
        ean_codes = [valid_ean] if valid_ean else []

        # Issue #475: Determinar si hay duplicado potencial para revisión
        duplicate_group_id = None
        duplicate_review_status = None

        if review_match:
            similar_product, similarity = review_match
            # Usar el grupo existente del producto similar, o crear uno nuevo
            if similar_product.duplicate_group_id:
                duplicate_group_id = similar_product.duplicate_group_id
            else:
                # Crear nuevo grupo y asignarlo también al producto similar
                duplicate_group_id = uuid4()
                similar_product.duplicate_group_id = duplicate_group_id
                if not similar_product.duplicate_review_status:
                    similar_product.duplicate_review_status = "pending_review"
                logger.info(
                    f"Created duplicate group {duplicate_group_id} for "
                    f"'{product_name[:30]}' <-> '{similar_product.product_name_display[:30]}' "
                    f"(similarity: {similarity:.2%})"
                )

            duplicate_review_status = "pending_review"

        new_product = ProductCatalogVentaLibre(
            product_name_normalized=normalized,
            product_name_display=product_name,
            product_name_variants=[product_name],
            product_name_tokens=tokens,  # Issue #473
            ean13=valid_ean,
            ean_codes=ean_codes,  # Issue #481: Multi-EAN support
            cn_codes=cn_codes if cn_codes else [],
            pharmacy_ids_seen=[pharmacy_id],
            first_pharmacy_id=pharmacy_id,
            pharmacies_count=1,
            ml_category=ml_category,
            ml_confidence=ml_confidence,
            prediction_source=prediction_source,
            detected_brand=detected_brand,
            first_seen_at=datetime.now(timezone.utc),
            last_seen_at=datetime.now(timezone.utc),
            total_sales_count=1,
            is_active=True,
            is_outlier=False,
            # Issue #475: Duplicate detection
            duplicate_group_id=duplicate_group_id,
            duplicate_review_status=duplicate_review_status,
        )

        self.db.add(new_product)

        if duplicate_review_status:
            logger.info(
                f"Created VentaLibre product with PENDING REVIEW: {product_name[:50]} "
                f"(group: {duplicate_group_id}, EAN: {valid_ean})"
            )
        else:
            logger.info(f"Created new VentaLibre product: {product_name[:50]} (EAN: {valid_ean}, CN: {cn_codes})")

        return new_product, True

    def get_stats(self) -> dict:
        """
        Obtiene estadísticas del catálogo.

        Returns:
            Dict con:
            - total_products: Total de productos activos
            - with_ean: Productos con EAN-13 válido
            - with_cn: Productos con al menos un CN
            - multi_pharmacy: Productos en 2+ farmacias
            - verified: Productos verificados por humano
            - pending_review: Productos pendientes de revisión
        """
        total = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .scalar()
        )

        with_ean = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.ean13.isnot(None))
            .scalar()
        )

        # Productos con al menos un CN (array no vacío)
        with_cn = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(func.array_length(ProductCatalogVentaLibre.cn_codes, 1) > 0)
            .scalar()
        ) or 0

        # Issue #481: Productos con múltiples EANs (sinónimos)
        multi_ean = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(func.array_length(ProductCatalogVentaLibre.ean_codes, 1) > 1)
            .scalar()
        ) or 0

        multi_pharmacy = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.pharmacies_count >= 2)
            .scalar()
        )

        verified = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.human_verified == True)
            .scalar()
        )

        # Productos que necesitan revisión (clasificación)
        pending_review = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.human_verified == False)
            .filter(
                (ProductCatalogVentaLibre.ml_category.in_(
                    ["otros", "parafarmacia_otros", "unknown", "sin_clasificar"]
                )) |
                (ProductCatalogVentaLibre.ml_confidence < ProductCatalogVentaLibre.LOW_CONFIDENCE_THRESHOLD)
            )
            .scalar()
        )

        # Issue #475: Duplicados pendientes de revisión
        pending_duplicate_review = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.duplicate_review_status == "pending_review")
            .scalar()
        ) or 0

        # Número de grupos de duplicados únicos
        duplicate_groups = (
            self.db.query(func.count(func.distinct(ProductCatalogVentaLibre.duplicate_group_id)))
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.duplicate_group_id.isnot(None))
            .scalar()
        ) or 0

        return {
            "total_products": total or 0,
            "with_ean": with_ean or 0,
            "with_cn": with_cn,
            "multi_ean": multi_ean,  # Issue #481: Productos con múltiples EANs
            "multi_pharmacy": multi_pharmacy or 0,
            "verified": verified or 0,
            "pending_review": pending_review or 0,
            "pending_duplicate_review": pending_duplicate_review,  # Issue #475
            "duplicate_groups": duplicate_groups,  # Issue #475
            "ean_coverage_pct": round((with_ean or 0) / max(total, 1) * 100, 1),
            "cn_coverage_pct": round(with_cn / max(total, 1) * 100, 1),
        }

    def find_potential_duplicates(
        self,
        product: ProductCatalogVentaLibre,
        similarity_threshold: float = 0.7
    ) -> List[Tuple[ProductCatalogVentaLibre, float, str]]:
        """
        Busca posibles duplicados de un producto (Issue #473).

        Orden de búsqueda:
        1. Coincidencias exactas por EAN (similitud 1.0)
        2. Coincidencias exactas por CN (similitud 1.0)
        3. Fuzzy matching por tokens (similitud variable)

        Args:
            product: Producto para buscar duplicados
            similarity_threshold: Umbral mínimo de similitud (0.0-1.0)

        Returns:
            Lista de tuples (producto, similitud, tipo_match)
        """
        duplicates: List[Tuple[ProductCatalogVentaLibre, float, str]] = []
        seen_ids = {product.id}  # Evitar duplicados en resultados

        # 1. Buscar por mismo EAN (similitud 1.0)
        if product.ean13:
            same_ean = (
                self.db.query(ProductCatalogVentaLibre)
                .filter(ProductCatalogVentaLibre.ean13 == product.ean13)
                .filter(ProductCatalogVentaLibre.id != product.id)
                .filter(ProductCatalogVentaLibre.is_active == True)
                .all()
            )
            for dup in same_ean:
                if dup.id not in seen_ids:
                    duplicates.append((dup, 1.0, "ean_exact"))
                    seen_ids.add(dup.id)

        # 2. Buscar por CNs compartidos (similitud 1.0)
        if product.cn_codes:
            for cn in product.cn_codes:
                same_cn = (
                    self.db.query(ProductCatalogVentaLibre)
                    .filter(ProductCatalogVentaLibre.cn_codes.contains([cn]))
                    .filter(ProductCatalogVentaLibre.id != product.id)
                    .filter(ProductCatalogVentaLibre.is_active == True)
                    .all()
                )
                for dup in same_cn:
                    if dup.id not in seen_ids:
                        duplicates.append((dup, 1.0, "cn_exact"))
                        seen_ids.add(dup.id)

        # 3. Fuzzy matching por tokens (Issue #473)
        try:
            from app.services.product_matching_service import get_product_matching_service

            matching_service = get_product_matching_service(self.db)
            fuzzy_matches = matching_service.find_similar_products(
                product_name=product.product_name_display,
                threshold=similarity_threshold,
                limit=20,
                exclude_id=product.id,
            )

            for match in fuzzy_matches:
                if match.product.id not in seen_ids:
                    duplicates.append((
                        match.product,
                        match.similarity,
                        match.match_type
                    ))
                    seen_ids.add(match.product.id)

        except ImportError:
            logger.warning("ProductMatchingService not available for duplicate search")

        # Ordenar por similitud descendente
        duplicates.sort(key=lambda x: x[1], reverse=True)

        return duplicates
