# backend/app/services/user_pharmacy_service.py
"""
Servicio para gestión de usuarios y farmacias.

Issue #348 FASE 2: Backend - Services & APIs
Implementa transacciones atómicas, soft delete GDPR, y consultas de storage.
"""
import logging
from typing import List, Optional, Tuple, Dict
from uuid import UUID

from sqlalchemy import select, func, and_, or_, text
from sqlalchemy.orm import Session, selectinload
from sqlalchemy.exc import IntegrityError
from passlib.context import CryptContext

from app.models.user import User
from app.models.pharmacy import Pharmacy
from app.models.role import Role
from app.schemas.user_pharmacy import (
    UserCreate,
    PharmacyCreate,
    UserUpdate
)
from app.utils.datetime_utils import utc_now
from app.utils.memory_utils import log_memory_usage

logger = logging.getLogger(__name__)
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")


class UserPharmacyService:
    """Servicio para gestión de usuarios y farmacias con RBAC"""

    @staticmethod
    def create_user_with_pharmacy(
        db: Session,
        user_data: UserCreate,
        pharmacy_data: PharmacyCreate,
        role: str = "user",
        subscription_plan: str = "free"
    ) -> Tuple[User, Pharmacy]:
        """
        Crea un usuario con su farmacia en una transacción atómica.

        REGLA #10: Relación 1:1 User-Pharmacy obligatoria.

        Args:
            db: Sesión de base de datos
            user_data: Datos del usuario
            pharmacy_data: Datos de la farmacia
            role: Rol del usuario (user/admin)
            subscription_plan: Plan de suscripción

        Returns:
            Tupla (User, Pharmacy) creados

        Raises:
            ValueError: Si ya existe usuario con email/username
            IntegrityError: Si falla la transacción
        """
        try:
            # Verificar que no exista usuario con mismo email o username
            existing_user = db.execute(
                select(User).where(
                    or_(
                        User.email == user_data.email,
                        User.username == user_data.username
                    )
                )
            )
            if existing_user.scalar_one_or_none():
                raise ValueError(f"Usuario con email {user_data.email} o username {user_data.username} ya existe")

            # Verificar que no exista farmacia con mismo NIF
            existing_pharmacy = db.execute(
                select(Pharmacy).where(Pharmacy.nif == pharmacy_data.nif)
            )
            if existing_pharmacy.scalar_one_or_none():
                # En desarrollo, verificar si ya tiene usuario
                pharmacy = existing_pharmacy.scalar_one()
                existing_user_for_pharmacy = db.execute(
                    select(User).where(User.pharmacy_id == pharmacy.id)
                )
                if existing_user_for_pharmacy.scalar_one_or_none():
                    raise ValueError(f"Farmacia con NIF {pharmacy_data.nif} ya tiene un usuario asignado (REGLA #10)")
                # Si no tiene usuario, podemos asignárselo
                logger.info("Farmacia existente sin usuario, asignando nuevo usuario")
            else:
                # Crear nueva farmacia
                pharmacy = Pharmacy(
                    **pharmacy_data.model_dump(),
                    code=f"PH-{pharmacy_data.nif}",  # Generar código único
                    subscription_plan=subscription_plan,
                    is_active=True,
                    created_at=utc_now(),
                    updated_at=utc_now()
                )
                db.add(pharmacy)
                db.flush()  # Obtener ID de farmacia

            # Obtener role_id basado en el role string
            role_result = db.execute(
                select(Role).where(Role.name == role)
            )
            role_obj = role_result.scalar_one_or_none()

            if not role_obj:
                # Si no existe el rol, usar el rol por defecto 'user'
                logger.warning(f"Rol '{role}' no encontrado, usando 'user' por defecto")
                role_result = db.execute(
                    select(Role).where(Role.name == "user")
                )
                role_obj = role_result.scalar_one()

            # Crear usuario con hash de password
            hashed_password = pwd_context.hash(user_data.password)

            user = User(
                **user_data.model_dump(exclude={'password'}),
                hashed_password=hashed_password,
                pharmacy_id=pharmacy.id,  # REGLA #10: Relación 1:1
                role=role,
                role_id=role_obj.id,
                subscription_plan=subscription_plan,
                is_active=True,
                is_verified=False,
                created_at=utc_now(),
                updated_at=utc_now()
            )
            db.add(user)

            # Commit de la transacción atómica
            db.commit()
            db.refresh(user)
            db.refresh(pharmacy)

            logger.info(
                f"Usuario {user.email} creado con farmacia {pharmacy.name} (NIF: {pharmacy.nif}) - "
                f"Transacción atómica completada"
            )

            return user, pharmacy

        except IntegrityError as e:
            db.rollback()
            logger.error(f"Error de integridad al crear usuario/farmacia: {str(e)}")
            raise
        except Exception as e:
            db.rollback()
            logger.error(f"Error al crear usuario/farmacia: {str(e)}")
            raise

    @staticmethod
    def get_users_paginated(
        db: Session,
        skip: int = 0,
        limit: int = 100,
        role_filter: Optional[str] = None,
        subscription_filter: Optional[str] = None,
        include_deleted: bool = False
    ) -> Tuple[List[User], int]:
        """
        Obtiene usuarios con filtros y paginación.

        Args:
            db: Sesión de base de datos
            skip: Registros a saltar
            limit: Máximo de registros a devolver
            role_filter: Filtrar por rol
            subscription_filter: Filtrar por plan de suscripción
            include_deleted: Incluir usuarios con soft delete

        Returns:
            Tupla (lista de usuarios, total de registros)
        """
        # Construir query base
        query = select(User).options(
            # Eager loading de relaciones
            selectinload(User.pharmacy),
            selectinload(User.role_obj)
        )

        # Aplicar filtros
        filters = []

        if not include_deleted:
            filters.append(User.deleted_at.is_(None))

        if role_filter:
            filters.append(User.role == role_filter)

        if subscription_filter:
            filters.append(User.subscription_plan == subscription_filter)

        if filters:
            query = query.where(and_(*filters))

        # Contar total
        count_query = select(func.count()).select_from(User)
        if filters:
            count_query = count_query.where(and_(*filters))

        total_result = db.execute(count_query)
        total = total_result.scalar()

        # Aplicar paginación y ordenamiento
        query = query.order_by(User.created_at.desc()).offset(skip).limit(limit)

        # Ejecutar query
        result = db.execute(query)
        users = result.scalars().all()

        logger.info(
            f"Listado de usuarios: {len(users)} de {total} total "
            f"(skip={skip}, limit={limit}, role={role_filter}, subscription={subscription_filter})"
        )

        return users, total

    @staticmethod
    def update_user(
        db: Session,
        user_id: UUID,
        user_update: UserUpdate
    ) -> User:
        """
        Actualiza un usuario.

        Args:
            db: Sesión de base de datos
            user_id: ID del usuario
            user_update: Datos a actualizar

        Returns:
            Usuario actualizado

        Raises:
            ValueError: Si el usuario no existe
        """
        # Obtener usuario
        result = db.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()

        if not user:
            raise ValueError(f"Usuario con ID {user_id} no encontrado")

        if user.deleted_at:
            raise ValueError(f"Usuario con ID {user_id} está eliminado (soft delete)")

        # Actualizar campos
        update_data = user_update.model_dump(exclude_unset=True)

        # Si se actualiza el rol, obtener role_id y convertir enum a string
        if 'role' in update_data:
            # Convertir enum a string si es necesario
            role_value = update_data['role']
            if hasattr(role_value, 'value'):  # Es un enum
                logger.info(f"Converting role enum {role_value} to string value")
                role_value = role_value.value
                update_data['role'] = role_value

            role_result = db.execute(
                select(Role).where(Role.name == role_value)
            )
            role_obj = role_result.scalar_one_or_none()
            if role_obj:
                user.role_id = role_obj.id

        # Si se actualiza el subscription_plan, convertir enum a string
        if 'subscription_plan' in update_data:
            from app.core.subscription_limits import SubscriptionPlan
            if isinstance(update_data['subscription_plan'], SubscriptionPlan):
                logger.info(f"Converting subscription_plan enum {update_data['subscription_plan']} to string value")
                update_data['subscription_plan'] = update_data['subscription_plan'].value

        # Si se actualiza el password, hashearlo
        if 'password' in update_data:
            from app.core.security import get_password_hash
            user.hashed_password = get_password_hash(update_data['password'])
            del update_data['password']  # No setear directamente, ya hasheamos

        for field, value in update_data.items():
            setattr(user, field, value)

        user.updated_at = utc_now()

        db.commit()
        db.refresh(user)

        logger.info(f"Usuario {user.email} actualizado: {update_data}")

        return user

    @staticmethod
    def soft_delete_user(
        db: Session,
        user_id: UUID
    ) -> User:
        """
        Soft delete de un usuario (GDPR Article 17).

        Marca el usuario como eliminado con timestamp pero mantiene
        los datos para auditoría y cumplimiento legal.

        Args:
            db: Sesión de base de datos
            user_id: ID del usuario

        Returns:
            Usuario marcado como eliminado

        Raises:
            ValueError: Si el usuario no existe o ya está eliminado
        """
        # Obtener usuario
        result = db.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()

        if not user:
            raise ValueError(f"Usuario con ID {user_id} no encontrado")

        if user.deleted_at:
            raise ValueError(f"Usuario con ID {user_id} ya está eliminado")

        # Marcar como eliminado
        user.deleted_at = utc_now()
        user.is_active = False  # También desactivar
        user.updated_at = utc_now()

        db.commit()
        db.refresh(user)

        logger.info(f"Usuario {user.email} marcado como eliminado (soft delete) - GDPR compliance")

        return user

    @staticmethod
    def restore_user(
        db: Session,
        user_id: UUID
    ) -> User:
        """
        Restaura un usuario con soft delete.

        Args:
            db: Sesión de base de datos
            user_id: ID del usuario

        Returns:
            Usuario restaurado

        Raises:
            ValueError: Si el usuario no existe o no está eliminado
        """
        # Obtener usuario
        result = db.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()

        if not user:
            raise ValueError(f"Usuario con ID {user_id} no encontrado")

        if not user.deleted_at:
            raise ValueError(f"Usuario con ID {user_id} no está eliminado")

        # Restaurar usuario
        user.deleted_at = None
        user.is_active = True
        user.updated_at = utc_now()

        db.commit()
        db.refresh(user)

        logger.info(f"Usuario {user.email} restaurado desde soft delete")

        return user

    @staticmethod
    def get_storage_usage(
        db: Session,
        pharmacy_id: Optional[UUID] = None
    ) -> List[Dict]:
        """
        Consulta el uso de almacenamiento desde la vista materializada.

        OPTIMIZACIÓN v2 (2025-11-02): Reemplazadas subconsultas repetidas con CTEs
        para pre-calcular agregaciones una sola vez. Previene OOM en Render Starter (512MB).

        Mejoras:
        - CTEs calculan file_stats y enrichment_stats UNA VEZ para todas las farmacias
        - Evita N subconsultas (3 × num_pharmacies)
        - Reduce consumo de memoria ~300MB → ~20MB
        - Tiempo: 20s → <500ms

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de farmacia específica (opcional)

        Returns:
            Lista de diccionarios con estadísticas de uso
        """
        log_memory_usage("storage_usage_start")

        # Usar CTEs para pre-calcular agregaciones una sola vez
        base_query = """
            WITH file_stats AS (
                SELECT
                    pharmacy_id,
                    COUNT(*) as total_files,
                    MAX(uploaded_at) as last_upload_date
                FROM file_uploads
                GROUP BY pharmacy_id
            ),
            enrichment_stats AS (
                SELECT
                    sd.pharmacy_id,
                    COUNT(*) as enriched_count
                FROM sales_enrichment se
                JOIN sales_data sd ON se.sales_data_id = sd.id
                GROUP BY sd.pharmacy_id
            )
            SELECT
                pss.pharmacy_id,
                pss.total_sales,
                pss.storage_mb,
                pss.data_from,
                pss.data_to,
                p.name as pharmacy_name,
                p.nif,
                u.email as user_email,
                u.subscription_plan,
                COALESCE(fs.total_files, 0) as total_files,
                fs.last_upload_date,
                COALESCE(es.enriched_count, 0) as enriched_records_count
            FROM pharmacy_storage_stats pss
            JOIN pharmacies p ON pss.pharmacy_id = p.id
            LEFT JOIN users u ON p.id = u.pharmacy_id
            LEFT JOIN file_stats fs ON pss.pharmacy_id = fs.pharmacy_id
            LEFT JOIN enrichment_stats es ON pss.pharmacy_id = es.pharmacy_id
        """

        if pharmacy_id:
            query = text(base_query + """
                WHERE pss.pharmacy_id = :pharmacy_id
            """)
            result = db.execute(query, {"pharmacy_id": str(pharmacy_id)})
        else:
            query = text(base_query + """
                ORDER BY pss.storage_mb DESC
                LIMIT 100
            """)
            result = db.execute(query)

        rows = result.fetchall()

        # Convertir a lista de diccionarios con métricas precisas
        storage_stats = []
        for row in rows:
            storage_stats.append({
                "pharmacy_id": row.pharmacy_id,
                "pharmacy_name": row.pharmacy_name,
                "nif": row.nif,
                "user_email": row.user_email,
                "subscription_plan": row.subscription_plan or "free",
                "total_files": row.total_files,  # COUNT real de file_uploads
                "total_size_mb": round(float(row.storage_mb), 2) if row.storage_mb else 0,
                "sales_records_count": row.total_sales,  # Total de ventas de la vista
                "enriched_records_count": row.enriched_records_count,  # COUNT real de sales_enrichment
                "last_upload_date": row.last_upload_date  # MAX(uploaded_at) de file_uploads
            })

        log_memory_usage("storage_usage_end")
        logger.info(f"Consultado uso de almacenamiento: {len(storage_stats)} registros")

        return storage_stats
