# backend/app/services/upload_processing_service.py
"""
Servicio para procesamiento de datos de ventas durante upload.
Extraído de api/upload.py para mejorar modularidad y reducir tamaño de archivo.

Funciones incluidas:
- Detección de duplicados por ventana temporal
- Construcción de registros SalesData
- Procesamiento de batches con progreso
- Cálculo de estadísticas de enriquecimiento
"""

import time
from datetime import datetime, time as time_type, timedelta
from typing import Optional

import structlog
from sqlalchemy import and_, case, cast, func
from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import Session
from sqlalchemy.types import Interval

from app.models import FileUpload, SalesData, UploadStatus
from app.models.sales_enrichment import SalesEnrichment

logger = structlog.get_logger(__name__)


# ========================================
# DUPLICATE DETECTION
# ========================================


def load_existing_sales_for_duplicate_detection(
    db: Session, pharmacy_id: str, min_date: datetime, max_date: datetime
) -> tuple[datetime, datetime]:
    """
    Obtiene la ventana temporal de ventas existentes para detección de duplicados.

    NUEVA LÓGICA (Issue #330): Detección por ventana temporal en lugar de comparación tupla por tupla.
    - Resuelve falsos positivos (ej: 5 paracetamoles en mismo ticket)
    - Más eficiente: 1 query simple vs cargar todas las ventas
    - Lógica clara: Dentro de ventana = duplicado, Fuera = nuevo

    Args:
        db: Sesión de base de datos
        pharmacy_id: ID de la farmacia
        min_date: Fecha mínima del archivo (no usado, pero mantenido por compatibilidad)
        max_date: Fecha máxima del archivo (no usado, pero mantenido por compatibilidad)

    Returns:
        Tupla (bd_min_datetime, bd_max_datetime):
        - bd_min_datetime: Datetime mínimo existente en BD (fecha + hora)
        - bd_max_datetime: Datetime máximo existente en BD (fecha + hora)
        - Si no hay ventas: (None, None)
    """
    logger.info(
        "[DUPLICATE_CHECK] Obteniendo ventana temporal de ventas existentes",
        extra={"pharmacy_id": pharmacy_id},
    )

    # Obtener MIN y MAX datetime de ventas existentes (1 query simple)
    # Combinamos sale_date + sale_time para tener datetime completo
    result = (
        db.query(
            func.min(SalesData.sale_date + cast(SalesData.sale_time, Interval)),
            func.max(SalesData.sale_date + cast(SalesData.sale_time, Interval)),
        )
        .filter(SalesData.pharmacy_id == pharmacy_id)
        .first()
    )

    bd_min_dt, bd_max_dt = result if result else (None, None)

    if bd_min_dt and bd_max_dt:
        logger.info(
            f"[DUPLICATE_CHECK] Ventana existente: {bd_min_dt} → {bd_max_dt}",
            extra={
                "pharmacy_id": pharmacy_id,
                "window_start": bd_min_dt.isoformat() if bd_min_dt else None,
                "window_end": bd_max_dt.isoformat() if bd_max_dt else None,
                "window_days": (bd_max_dt - bd_min_dt).days if bd_min_dt and bd_max_dt else 0,
            },
        )
    else:
        logger.info("[DUPLICATE_CHECK] No hay ventas existentes - primera carga")

    return bd_min_dt, bd_max_dt


# ========================================
# SALES RECORD BUILDING
# ========================================


def build_sales_record(
    row,  # pd.Series
    pharmacy_id: str,
    upload_id: str
) -> SalesData:
    """
    Construye objeto SalesData desde row del parser.

    NUEVA FUNCIÓN (Issue #325): Extraída de _process_sales_batch() para mejorar legibilidad.
    - Single Responsibility: Solo construir objeto SalesData
    - Facilita testing unitario
    - Reduce nesting level en funciones caller

    Args:
        row: Fila del DataFrame parseado (pd.Series)
        pharmacy_id: ID de la farmacia
        upload_id: ID del upload

    Returns:
        Objeto SalesData listo para guardar en BD
    """
    # Convertir tipos de datos - PRESERVAR timestamp completo (Issue #273)
    sale_date_converted = row.get("sale_date")
    sale_time_value = row.get("sale_time")

    return SalesData(
        pharmacy_id=pharmacy_id,
        upload_id=upload_id,
        sale_date=sale_date_converted,
        sale_time=sale_time_value,
        ean13=row.get("ean13"),
        codigo_nacional=row.get("codigo_nacional"),
        product_name=row.get("product_name"),
        subcategory=row.get("subcategory"),
        quantity=row.get("quantity"),
        unit_price=row.get("unit_price"),
        purchase_price=row.get("purchase_price"),
        sale_price=row.get("sale_price"),
        total_amount=row.get("total_amount"),
        discount_amount=row.get("discount_amount"),
        margin_amount=row.get("margin_amount"),
        margin_percentage=row.get("margin_percentage"),
        supplier=row.get("supplier"),
        employee_code=row.get("employee_code"),
        client_type=row.get("client_type"),
        is_return=row.get("is_return", False),
    )


# ========================================
# BATCH PROCESSING
# ========================================


def process_sales_batch(
    db: Session, batch_df, existing_sales_window: tuple, pharmacy_id: str, upload_id: str
) -> tuple[list, int]:
    """
    Procesa un batch de registros de ventas con detección de duplicados por ventana temporal.

    NUEVA LÓGICA (Issue #330): Detección por ventana temporal.
    - Cambio de firma: existing_sales_set → existing_sales_window (tupla con min/max datetime)
    - Lógica: Dentro de ventana = duplicado, Fuera = nuevo
    - Resuelve falsos positivos (5 paracetamoles mismo ticket ya no es duplicado)

    Args:
        db: Sesión de base de datos
        batch_df: DataFrame con el batch de ventas
        existing_sales_window: Tupla (bd_min_dt, bd_max_dt) con ventana temporal existente
        pharmacy_id: ID de la farmacia
        upload_id: ID del upload

    Returns:
        Tupla (batch_records, batch_duplicates):
        - batch_records: Lista de objetos SalesData para insertar
        - batch_duplicates: Contador de duplicados encontrados en este batch
    """
    batch_records = []
    batch_duplicates = 0

    bd_min_dt, bd_max_dt = existing_sales_window

    for _, row in batch_df.iterrows():
        product_id = row.get("ean13") or row.get("codigo_nacional")

        try:
            # Convertir tipos de datos - PRESERVAR timestamp completo (Issue #273)
            sale_date_converted = row.get("sale_date")
            sale_time_value = row.get("sale_time")

            # NUEVA LÓGICA (Issue #330): Detección por ventana temporal
            # Combinar sale_date + sale_time para datetime completo
            if sale_date_converted and sale_time_value:
                # Convertir date a datetime primero, luego añadir time
                sale_datetime = datetime.combine(sale_date_converted, time_type.min) + timedelta(
                    hours=sale_time_value.hour,
                    minutes=sale_time_value.minute,
                    seconds=sale_time_value.second
                )
            elif sale_date_converted:
                # Si solo hay fecha, convertir a datetime (00:00:00)
                sale_datetime = datetime.combine(sale_date_converted, time_type.min)
            else:
                sale_datetime = None

            # Check si venta está dentro de ventana existente
            if bd_min_dt and bd_max_dt and sale_datetime:
                if bd_min_dt <= sale_datetime <= bd_max_dt:
                    # Registro duplicado: Dentro de ventana temporal
                    batch_duplicates += 1
                    logger.debug(
                        f"[DUPLICATE] Venta dentro de ventana existente: {row.get('product_name')} "
                        f"datetime={sale_datetime} (ventana: {bd_min_dt} - {bd_max_dt})"
                    )
                    continue

            # Crear registro solo si no es duplicado
            sales_record = build_sales_record(row, pharmacy_id, upload_id)
            batch_records.append(sales_record)

        except Exception as e:
            logger.warning(f"[BATCH] Error procesando registro {product_id}: {e}")
            continue

    return batch_records, batch_duplicates


def save_batch_with_progress(
    db: Session,
    sales_batch: list[SalesData],
    upload_id: str,
    processed_count: int,
    total_count: int,
    records_saved_so_far: int,
    records_duplicated_so_far: int
) -> int:
    """
    Guarda batch y actualiza progreso del upload.

    NUEVA FUNCIÓN (Issue #325): Extraída de process_file() para mejorar legibilidad.
    - Single Responsibility: Solo guardar y reportar progreso
    - Bulk insert con fallback automático a inserción one-by-one
    - Actualización de progreso en tiempo real (Issue #274)

    Args:
        db: Sesión de base de datos
        sales_batch: Lista de SalesData a guardar
        upload_id: ID del upload
        processed_count: Registros procesados hasta ahora (incluyendo este batch)
        total_count: Total de registros en el archivo
        records_saved_so_far: Contador total de registros guardados antes de este batch
        records_duplicated_so_far: Contador total de duplicados antes de este batch

    Returns:
        Número de registros guardados exitosamente en este batch
    """
    saved_in_batch = 0

    try:
        # Bulk insert con commit FIRST (Code Review #1)
        db.bulk_save_objects(sales_batch)
        db.commit()
        saved_in_batch = len(sales_batch)

        # NUEVA LÓGICA (Issue #330): Ya no necesitamos actualizar set
        # La ventana temporal no cambia durante el procesamiento del archivo

        # Update progress (Issue #274: Progreso en tiempo real)
        upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
        if upload:
            progress_pct = int(processed_count / total_count * 100)
            upload.rows_processed = processed_count
            upload.processing_notes = (
                f"Guardando datos... {progress_pct}% "
                f"({records_saved_so_far + saved_in_batch:,} nuevos, "
                f"{records_duplicated_so_far:,} duplicados omitidos)"
            )
            db.commit()

        logger.info(
            f"[BATCH] Guardados {saved_in_batch} registros "
            f"(total: {records_saved_so_far + saved_in_batch}, "
            f"duplicados batch: {len(sales_batch) - saved_in_batch})"
        )

    except Exception as e:
        logger.error(f"[BATCH] Error guardando batch: {e}")
        db.rollback()

        # Fallback: Intentar guardar uno por uno
        for record in sales_batch:
            try:
                db.add(record)
                db.commit()
                saved_in_batch += 1

                # NUEVA LÓGICA (Issue #330): Ya no necesitamos actualizar set
                # La ventana temporal no cambia durante el procesamiento

            except (IntegrityError, Exception) as e:
                logger.debug(f"Error saving individual record (likely duplicate): {e}")
                # Este registro ya existe o tiene otro error, no incrementar saved_in_batch
                db.rollback()

    return saved_in_batch


# ========================================
# ENRICHMENT STATUS & PROGRESS
# ========================================


def get_enrichment_status(db: Session, upload_id: str) -> str:
    """
    Determina el estado del enriquecimiento para un upload específico.

    OPTIMIZADO (Issue #252 Code Review): Añade logging estructurado para debugging.

    Returns:
        - "pending": No hay registros de ventas aún o no iniciado
        - "processing": Enriquecimiento en progreso
        - "completed": Todos los registros procesados
        - "failed": Error en el enriquecimiento
    """
    start_time = time.time()

    # Contar registros de ventas para este upload
    total_sales = db.query(SalesData).filter(SalesData.upload_id == upload_id).count()

    if total_sales == 0:
        query_time = time.time() - start_time
        logger.debug(
            "[ENRICHMENT_STATUS] No hay registros de ventas",
            extra={
                "upload_id": upload_id,
                "status": "pending",
                "total_sales": 0,
                "query_time_ms": round(query_time * 1000, 2),
            },
        )
        return "pending"

    # Contar registros ya enriquecidos (exitosos o en revisión manual)
    enriched_count = (
        db.query(SalesData)
        .filter(SalesData.upload_id == upload_id)
        .join(SalesEnrichment)
        .filter(SalesEnrichment.enrichment_status.in_(["enriched", "manual_review"]))
        .count()
    )

    # Contar registros fallidos
    failed_count = (
        db.query(SalesData)
        .filter(SalesData.upload_id == upload_id)
        .join(SalesEnrichment)
        .filter(SalesEnrichment.enrichment_status == "failed")
        .count()
    )

    # Determinar estado
    if enriched_count + failed_count >= total_sales:
        status = "completed"
    elif enriched_count > 0 or failed_count > 0:
        status = "processing"
    else:
        status = "pending"

    # Logging estructurado con métricas
    query_time = time.time() - start_time
    logger.debug(
        "[ENRICHMENT_STATUS] Estado calculado",
        extra={
            "upload_id": upload_id,
            "status": status,
            "total_sales": total_sales,
            "enriched": enriched_count,
            "failed": failed_count,
            "query_time_ms": round(query_time * 1000, 2),
        },
    )

    return status


def get_enrichment_progress_summary(db: Session, upload_id: str) -> dict:
    """
    Obtiene un resumen del progreso de enriquecimiento para un upload.

    OPTIMIZADO (Issue #252 Code Review): Reduce 8 queries a 1 query única usando agregaciones.
    Performance: ~5x más rápido con datasets grandes (50k+ registros).

    Returns:
        {
            "total": Total de registros a enriquecer,
            "processed": Registros ya procesados (exitosos + fallidos),
            "enriched": Registros enriquecidos exitosamente,
            "manual_review": Registros que requieren revisión manual,
            "failed": Registros fallidos,
            "percentage": Porcentaje completado (0-100),
            "phase": Fase actual del enriquecimiento,
            "matches_by_code": Coincidencias por código nacional,
            "matches_by_name": Coincidencias por nombre fuzzy,
            "matches_by_ean": Coincidencias por EAN
        }
    """
    # Una única query con agregaciones condicionales (8 queries → 1 query)
    stats = (
        db.query(
            func.count(SalesData.id).label("total"),
            # Contar por estado usando CASE
            func.sum(case((SalesEnrichment.enrichment_status == "enriched", 1), else_=0)).label("enriched"),
            func.sum(case((SalesEnrichment.enrichment_status == "manual_review", 1), else_=0)).label("manual_review"),
            func.sum(case((SalesEnrichment.enrichment_status == "failed", 1), else_=0)).label("failed"),
            # Contar por método de matching (solo si enriched)
            func.sum(
                case(
                    (
                        and_(
                            SalesEnrichment.enrichment_status == "enriched",
                            SalesEnrichment.match_method == "codigo_nacional",
                        ),
                        1,
                    ),
                    else_=0,
                )
            ).label("matches_by_code"),
            func.sum(
                case(
                    (
                        and_(
                            SalesEnrichment.enrichment_status == "enriched",
                            SalesEnrichment.match_method == "name_fuzzy",
                        ),
                        1,
                    ),
                    else_=0,
                )
            ).label("matches_by_name"),
            func.sum(
                case(
                    (and_(SalesEnrichment.enrichment_status == "enriched", SalesEnrichment.match_method == "ean13"), 1),
                    else_=0,
                )
            ).label("matches_by_ean"),
        )
        .filter(SalesData.upload_id == upload_id)
        .outerjoin(  # CRITICAL: outerjoin para incluir registros sin enriquecimiento
            SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
        )
        .one_or_none()
    )

    if not stats or stats.total == 0:
        logger.debug("[ENRICHMENT_PROGRESS] Sin datos de ventas", extra={"upload_id": upload_id, "total": 0})
        return {
            "total": 0,
            "processed": 0,
            "enriched": 0,
            "manual_review": 0,
            "failed": 0,
            "percentage": 0.0,
            "phase": "Esperando datos",
            "matches_by_code": 0,
            "matches_by_name": 0,
            "matches_by_ean": 0,
        }

    # Calcular derivados
    processed = (stats.enriched or 0) + (stats.manual_review or 0) + (stats.failed or 0)
    percentage = (processed / stats.total * 100) if stats.total > 0 else 0.0

    # Determinar fase actual
    if processed == 0:
        phase = "Iniciando enriquecimiento..."
    elif percentage < 30:
        phase = "Matching por código nacional..."
    elif percentage < 70:
        phase = "Matching por EAN y nombre..."
    elif percentage < 100:
        phase = "Finalizando clasificación..."
    else:
        phase = "Completado"

    # Logging estructurado con métricas detalladas
    logger.debug(
        "[ENRICHMENT_PROGRESS] Progreso calculado",
        extra={
            "upload_id": upload_id,
            "total": stats.total,
            "processed": processed,
            "enriched": stats.enriched or 0,
            "manual_review": stats.manual_review or 0,
            "failed": stats.failed or 0,
            "percentage": round(percentage, 1),
            "phase": phase,
            "matches_by_code": stats.matches_by_code or 0,
            "matches_by_name": stats.matches_by_name or 0,
            "matches_by_ean": stats.matches_by_ean or 0,
        },
    )

    return {
        "total": stats.total,
        "processed": processed,
        "enriched": stats.enriched or 0,
        "manual_review": stats.manual_review or 0,
        "failed": stats.failed or 0,
        "percentage": round(percentage, 1),
        "phase": phase,
        "matches_by_code": stats.matches_by_code or 0,
        "matches_by_name": stats.matches_by_name or 0,
        "matches_by_ean": stats.matches_by_ean or 0,
    }


# ========================================
# UPLOAD PERCENTAGE CALCULATION
# ========================================


def calculate_upload_percentage(upload: FileUpload) -> float:
    """
    Calcular porcentaje de progreso del upload basado en estado y registros.

    IMPLEMENTA ISSUE #274: Feedback de progreso en tiempo real
    IMPLEMENTA ISSUE #329: Progreso granular desde segundo 1

    Rangos de progreso:
    - PENDING: 0% (inicio)
    - PARSING: 0-20% (parseando archivo ERP)
    - VALIDATING: 20-40% (validando estructura y datos)
    - SAVING: 40-50% (guardando en base de datos)
    - PROCESSING: 50-90% (enriquecimiento CIMA/nomenclator)
    - COMPLETED: 100% (finalizado)

    Returns:
        Porcentaje de 0-100
    """
    if upload.status == UploadStatus.PENDING:
        return 0.0

    elif upload.status == UploadStatus.PARSING:
        # 0-20%: Fase de parsing
        # Progreso basado en si ya se detectó tipo ERP
        notes = (upload.processing_notes or "").lower()
        if "detectado" in notes or "formato" in notes:
            return 15.0  # Formato detectado
        return 5.0  # Inicio del parsing

    elif upload.status == UploadStatus.VALIDATING:
        # 20-40%: Fase de validación
        # Progreso basado en validaciones específicas
        notes = (upload.processing_notes or "").lower()
        if "duplicados" in notes:
            return 35.0  # Verificando duplicados (último paso validación)
        elif "códigos" in notes or "nacional" in notes:
            return 30.0  # Validando códigos nacionales
        return 22.0  # Inicio de validación

    elif upload.status == UploadStatus.SAVING:
        # 40-50%: Fase de guardado en BD
        # Progreso basado en rows_processed si está disponible
        if upload.rows_total and upload.rows_processed:
            save_progress = (upload.rows_processed / upload.rows_total) * 10  # 0-10%
            return 40.0 + save_progress  # 40-50%
        return 42.0  # Inicio del guardado

    elif upload.status == UploadStatus.PROCESSING:
        # 50-90%: Fase de enriquecimiento
        # Progreso basado en rows_processed o estimación
        if upload.rows_total and upload.rows_processed:
            # Calcular progreso dentro de la fase de enriquecimiento (50-90%)
            enrichment_progress = (upload.rows_processed / upload.rows_total) * 40  # 0-40%
            return 50.0 + enrichment_progress  # 50-90%
        return 55.0  # Enriquecimiento iniciado

    elif upload.status == UploadStatus.COMPLETED:
        return 100.0

    elif upload.status == UploadStatus.ERROR:
        # En error, retornar último progreso conocido o 0
        if upload.rows_total and upload.rows_processed:
            return (upload.rows_processed / upload.rows_total) * 50  # Max 50% en error
        return 0.0

    # Default para estados desconocidos
    return 0.0
