"""
Semantic Enrichment Service - Sandwich Strategy Step 1

This service acts as a SEMANTIC NORMALIZER, not a classifier.
It takes dirty product names and produces clean, structured descriptions.

Philosophy:
- LLM does what it's good at: understanding messy text
- LLM does NOT classify (that's the Arbiter's job)
- Output is clean text for the rules engine to process

Example:
    Input:  "Arturo A. Rapsodia Fact. Crec"
    Output: {
        "semantic_summary": "Tratamiento capilar con factores de crecimiento para caída cabello",
        "formato": "Tópico",
        "indicaciones": "Prevenir y tratar la caída del cabello",
        "ingredientes": ["factores de crecimiento", "biotina"]
    }

The classification ("caida_cabello") is determined by the rules engine
in the next step, NOT by this LLM.
"""

import json
import logging
import os
import re
from typing import Any, Dict, List, Literal, Optional

from pydantic import BaseModel, Field
from tenacity import (
    retry,
    retry_if_exception_type,
    stop_after_attempt,
    wait_exponential,
)

logger = logging.getLogger(__name__)


# Formato galénico estructurado - clave para discriminar ingredientes secundarios
FormatoGalenico = Literal[
    "oral_solido",       # cápsulas, comprimidos, sobres, granulados
    "oral_liquido",      # jarabes, ampollas bebibles, gotas orales
    "topico_facial",     # cremas, serums, geles para cara
    "topico_corporal",   # lociones, geles cuerpo, aceites
    "topico_capilar",    # champús, lociones cabello, mascarillas pelo
    "topico_labial",     # bálsamos, protectores labiales
    "oftalmico",         # colirios, gotas ojos
    "nasal",             # sprays nasales, lavados
    "otico",             # gotas óticas, sprays oídos
    "bucal",             # colutorios, pastas dentales, geles bucales
    "vaginal",           # óvulos, geles íntimos
    "rectal",            # supositorios, enemas
    "dispositivo",       # tests, jeringuillas, tensiómetros
    "alimentacion",      # leches infantiles, papillas, nutrición
    "otros"
]


class SemanticEnrichment(BaseModel):
    """
    Semantic enrichment result - NO classification.

    This is the cleaned/normalized product description that
    will be fed to the rules engine.

    IMPORTANT: The `formato` field is CRITICAL for the arbiter.
    It determines whether ingredients like "Vitamina E" are:
    - Primary (oral_solido → classify as vitaminas)
    - Secondary (topico_facial → ignore for classification)
    """
    semantic_summary: str = Field(
        ...,
        description="Descripción limpia y expandida del producto en español"
    )
    formato: FormatoGalenico = Field(
        "otros",
        description="Forma galénica estructurada del producto"
    )
    proposito_principal: Optional[str] = Field(
        None,
        description="El 'job to be done' principal: hidratar, proteger, suplementar, etc."
    )
    indicaciones: Optional[str] = Field(
        None,
        description="Para qué sirve el producto (síntomas que trata)"
    )
    ingredientes: List[str] = Field(
        default_factory=list,
        description="Lista de ingredientes activos principales (NO secundarios como vitamina E en cremas)"
    )
    marca_detectada: Optional[str] = Field(
        None,
        description="Marca detectada si es evidente en el nombre"
    )
    es_infantil: bool = Field(
        False,
        description="Si es un producto específico para bebés/niños"
    )
    es_veterinario: bool = Field(
        False,
        description="Si es un producto para animales"
    )
    confianza: float = Field(
        0.85,
        ge=0.0,
        le=1.0,
        description="Confianza en la extracción"
    )

    def es_topico(self) -> bool:
        """Check if product is topical (important for filtering noise)."""
        return self.formato in [
            "topico_facial", "topico_corporal", "topico_capilar",
            "topico_labial", "oftalmico", "bucal"
        ]

    def es_oral(self) -> bool:
        """Check if product is oral (vitamins/supplements make sense here)."""
        return self.formato in ["oral_solido", "oral_liquido", "alimentacion"]


# Hierarchical prompt for the LLM
SYSTEM_PROMPT = """Eres un NORMALIZADOR SEMÁNTICO de productos de farmacia española.

TU TAREA: Tomar nombres de productos sucios/abreviados y generar una descripción limpia y estructurada.
NO DEBES clasificar el producto. Solo describir QUÉ ES y PARA QUÉ SIRVE.

FORMATO DE SALIDA (JSON):
{
    "semantic_summary": "Descripción expandida y limpia en español (30-50 palabras)",
    "formato": "oral_solido|oral_liquido|topico_facial|topico_corporal|topico_capilar|topico_labial|oftalmico|nasal|otico|bucal|vaginal|rectal|dispositivo|alimentacion|otros",
    "proposito_principal": "El job-to-be-done: hidratar, proteger solar, suplementar vitaminas, etc.",
    "indicaciones": "Para qué síntomas/condiciones se usa (una frase)",
    "ingredientes": ["ingrediente_activo_principal"],
    "marca_detectada": "Marca si es evidente, null si no",
    "es_infantil": true/false,
    "es_veterinario": true/false,
    "confianza": 0.0-1.0
}

========== REGLAS CRÍTICAS ANTI-RUIDO ==========

REGLA 1 - FORMATO GALÉNICO (MUY IMPORTANTE):
- oral_solido: cápsulas, comprimidos, sobres, granulados, pastillas
- oral_liquido: jarabes, ampollas bebibles, soluciones orales
- topico_facial: cremas cara, serums, geles faciales, contorno ojos
- topico_corporal: lociones cuerpo, aceites, geles corporales
- topico_capilar: champús, mascarillas pelo, lociones capilares
- topico_labial: bálsamos labiales, protectores labiales
- oftalmico: colirios, gotas ojos, soluciones oftálmicas
- nasal: sprays nasales, lavados nasales, agua de mar nasal
- bucal: colutorios, pastas dientes, geles encías
- dispositivo: tests, jeringuillas, tensiómetros, glucómetros
- alimentacion: leches infantiles, papillas, nutrición enteral

REGLA 2 - INGREDIENTES SECUNDARIOS (CRÍTICO):
NUNCA listes como ingrediente principal algo que es SECUNDARIO en el producto:
- Si es CREMA/GEL/SERUM → Vitamina E, Aloe Vera, Pantenol son SECUNDARIOS, NO los listes
- Si es PROTECTOR SOLAR → Los filtros UV son principales, vitamina E es secundaria
- Si es CHAMPÚ → El agente limpiador es principal, vitaminas son secundarias

SOLO lista como ingrediente cuando ES EL ACTIVO QUE HACE EL TRABAJO:
- Vitamina C en cápsula → SÍ es ingrediente principal
- Vitamina C en serum facial → NO, es secundario (el serum es para arrugas, no para suplementar)

REGLA 3 - PROPÓSITO PRINCIPAL:
Identifica el "job to be done" del cliente:
- "proteger del sol" (no "aportar vitamina E")
- "hidratar la piel" (no "suplementar colágeno")
- "suplementar vitaminas" (solo si es oral)
- "limpiar la cara" (no "nutrir con vitaminas")

REGLA 4 - SÍNTOMAS SECUNDARIOS:
Si un protector solar dice "calma quemaduras" → El propósito es "proteger del sol", NO "tratar quemaduras"
Si un anti-piojos dice "hidrata el pelo" → El propósito es "eliminar piojos", NO "hidratar"

REGLA 5 - EXPANDE ABREVIATURAS:
"Crec" -> "Crecimiento", "Fact" -> "Factores", "Prot" -> "Protector", "Cplx" -> "Complex"

Responde SOLO con JSON válido, sin texto adicional."""


# Few-shot examples with structured formato and anti-noise principles
FEW_SHOT_EXAMPLES = [
    {
        "input": "Arturo A. Rapsodia Fact. Crec 50ml",
        "output": {
            "semantic_summary": "Tratamiento capilar Arturo Alba con factores de crecimiento para estimular el cuero cabelludo y prevenir la caída del cabello, formato tópico 50ml",
            "formato": "topico_capilar",
            "proposito_principal": "prevenir caída del cabello",
            "indicaciones": "Prevenir y tratar la caída del cabello, estimular crecimiento capilar",
            "ingredientes": ["factores de crecimiento"],
            "marca_detectada": "Arturo Alba",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.85
        }
    },
    {
        "input": "ISDIN FUSION WTR MAG SPF50",
        "output": {
            "semantic_summary": "Protector solar facial ISDIN Fusion Water Magic con SPF 50, textura ultraligera de fase acuosa para uso diario",
            "formato": "topico_facial",
            "proposito_principal": "proteger del sol",
            "indicaciones": "Protección solar alta para rostro, prevenir quemaduras y fotoenvejecimiento",
            "ingredientes": ["filtros UVA/UVB"],
            "marca_detectada": "ISDIN",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.95
        }
    },
    {
        "input": "FRENADOL CPLX GRAN 10 SOB",
        "output": {
            "semantic_summary": "Antigripal Frenadol Complex en granulado para solución oral, 10 sobres, combinación analgésica con antitusivo",
            "formato": "oral_solido",
            "proposito_principal": "aliviar síntomas de gripe",
            "indicaciones": "Alivio de síntomas de gripe y resfriado: fiebre, dolor, tos seca, congestión",
            "ingredientes": ["paracetamol", "dextrometorfano", "clorfenamina", "pseudoefedrina"],
            "marca_detectada": "Frenadol",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.95
        }
    },
    {
        "input": "ARNIDOL GEL STCK 15ML",
        "output": {
            "semantic_summary": "Gel stick Arnidol con árnica para aliviar golpes, contusiones y moratones en formato roll-on 15ml",
            "formato": "topico_corporal",
            "proposito_principal": "aliviar golpes y hematomas",
            "indicaciones": "Alivio de golpes, hematomas, moratones y contusiones leves",
            "ingredientes": ["árnica montana"],
            "marca_detectada": "Arnidol",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.95
        }
    },
    {
        "input": "BLEMIL 2 OPTM 800G",
        "output": {
            "semantic_summary": "Leche de continuación Blemil Plus 2 Optimum para lactantes desde 6 meses, formato 800 gramos",
            "formato": "alimentacion",
            "proposito_principal": "alimentar bebé",
            "indicaciones": "Alimentación complementaria para bebés de 6 a 12 meses",
            "ingredientes": ["proteínas de leche", "DHA"],
            "marca_detectada": "Blemil",
            "es_infantil": True,
            "es_veterinario": False,
            "confianza": 0.90
        }
    },
    # EJEMPLO CRÍTICO: Serum con Vitamina C - NO es suplemento
    {
        "input": "ISDIN FLAVO-C SERUM VIT C 30ML",
        "output": {
            "semantic_summary": "Serum facial antioxidante ISDIN Flavo-C con vitamina C para luminosidad y antiedad, 30ml",
            "formato": "topico_facial",
            "proposito_principal": "tratar arrugas y luminosidad",
            "indicaciones": "Antienvejecimiento, luminosidad facial, antioxidante cutáneo",
            "ingredientes": [],
            "marca_detectada": "ISDIN",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.95
        }
    },
    # EJEMPLO CRÍTICO: Vitamina C oral - SÍ es suplemento
    {
        "input": "REDOXON VIT C 1000MG 30 COMP EFERV",
        "output": {
            "semantic_summary": "Suplemento de Vitamina C Redoxon 1000mg en comprimidos efervescentes para reforzar defensas",
            "formato": "oral_solido",
            "proposito_principal": "suplementar vitaminas",
            "indicaciones": "Reforzar sistema inmune, aporte de vitamina C",
            "ingredientes": ["vitamina C"],
            "marca_detectada": "Redoxon",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.95
        }
    },
    # EJEMPLO: Colutorio - es bucal, no vitaminas
    {
        "input": "GUM ACTIVITAL COLUTORIO 500ML",
        "output": {
            "semantic_summary": "Colutorio GUM Activital para higiene bucal diaria con acción antibacteriana, 500ml",
            "formato": "bucal",
            "proposito_principal": "higiene bucal",
            "indicaciones": "Higiene bucal diaria, prevención de caries y placa",
            "ingredientes": ["cloruro de cetilpiridinio"],
            "marca_detectada": "GUM",
            "es_infantil": False,
            "es_veterinario": False,
            "confianza": 0.95
        }
    },
]


class SemanticEnrichmentService:
    """
    Semantic Normalizer using Groq (Llama 3.3 70B).

    This service ONLY extracts and cleans product information.
    It does NOT classify products into categories.

    Classification is done by the SandwichClassifierService
    using deterministic rules on the enriched data.
    """

    GROQ_MODELS = [
        "llama-3.3-70b-versatile",
        "llama-3.1-8b-instant",
        "mixtral-8x7b-32768",
    ]

    def __init__(
        self,
        groq_api_key: Optional[str] = None,
    ):
        self.groq_api_key = groq_api_key or os.getenv("GROQ_API_KEY")
        self._groq_client = None
        self._model = None
        self._few_shot_messages = self._build_few_shot_messages()

        if not self.groq_api_key:
            logger.warning("GROQ_API_KEY not set. Service will not work.")

    @property
    def groq_client(self):
        """Lazy initialization of Groq client."""
        if self._groq_client is None and self.groq_api_key:
            try:
                from groq import Groq
                self._groq_client = Groq(api_key=self.groq_api_key)
                self._model = self.GROQ_MODELS[0]
                logger.info(f"Groq client initialized: model={self._model}")
            except ImportError:
                logger.error("groq package not installed. Run: pip install groq")
        return self._groq_client

    @property
    def model(self) -> str:
        """Get current model name."""
        if self._model is None:
            _ = self.groq_client
        return self._model or "unknown"

    def _build_few_shot_messages(self) -> List[Dict[str, str]]:
        """Build few-shot messages for context."""
        messages = []
        for example in FEW_SHOT_EXAMPLES:
            messages.append({
                "role": "user",
                "content": f"Producto: {example['input']}"
            })
            messages.append({
                "role": "assistant",
                "content": json.dumps(example["output"], ensure_ascii=False)
            })
        return messages

    def _extract_json(self, text: str) -> Dict[str, Any]:
        """Extract JSON from response with fallbacks."""
        # Try direct JSON
        try:
            return json.loads(text)
        except json.JSONDecodeError:
            pass

        # Try ```json block
        match = re.search(r'```json\s*(.*?)\s*```', text, re.DOTALL)
        if match:
            try:
                return json.loads(match.group(1))
            except json.JSONDecodeError:
                pass

        # Try {...}
        match = re.search(r'\{.*\}', text, re.DOTALL)
        if match:
            try:
                return json.loads(match.group(0))
            except json.JSONDecodeError:
                pass

        raise json.JSONDecodeError("No valid JSON found", text, 0)

    @retry(
        stop=stop_after_attempt(3),
        wait=wait_exponential(multiplier=1, min=1, max=10),
        retry=retry_if_exception_type((ValueError, json.JSONDecodeError)),
        reraise=True
    )
    def enrich(self, product_name: str) -> SemanticEnrichment:
        """
        Semantically enrich a product name.

        Args:
            product_name: Raw/dirty product name

        Returns:
            SemanticEnrichment with cleaned/expanded description

        Note:
            This does NOT classify the product. Use SandwichClassifierService
            for classification.
        """
        if not self.groq_client:
            raise ValueError("Groq client not initialized. Check GROQ_API_KEY.")

        messages = [
            {"role": "system", "content": SYSTEM_PROMPT},
            *self._few_shot_messages,
            {"role": "user", "content": f"Producto: {product_name}"},
        ]

        response = self.groq_client.chat.completions.create(
            model=self._model,
            messages=messages,
            temperature=0.1,
            max_tokens=400,
            response_format={"type": "json_object"},
        )

        content = response.choices[0].message.content
        if not content:
            raise ValueError("Empty response from LLM")

        data = self._extract_json(content)
        enrichment = SemanticEnrichment(**data)

        logger.info(
            f"Enriched: {product_name[:40]}... -> "
            f"summary={enrichment.semantic_summary[:50]}..."
        )

        return enrichment

    def enrich_safe(self, product_name: str) -> Optional[SemanticEnrichment]:
        """
        Safe version that returns None on error.
        """
        try:
            return self.enrich(product_name)
        except Exception as e:
            logger.warning(f"Failed to enrich '{product_name[:40]}...': {e}")
            return None

    def health_check(self) -> Dict[str, Any]:
        """Check service availability."""
        try:
            if self.groq_client:
                response = self.groq_client.chat.completions.create(
                    model=self._model,
                    messages=[{"role": "user", "content": "Responde: OK"}],
                    max_tokens=10,
                )
                return {
                    "status": "healthy",
                    "backend": "groq",
                    "model": self._model,
                }
            return {"status": "unhealthy", "error": "No API key"}
        except Exception as e:
            return {"status": "unhealthy", "error": str(e)}


# Singleton
_service: Optional[SemanticEnrichmentService] = None


def get_semantic_enrichment_service() -> SemanticEnrichmentService:
    """Get singleton instance."""
    global _service
    if _service is None:
        _service = SemanticEnrichmentService()
    return _service
