# backend/app/services/seasonality_service.py
"""
Servicio de estacionalidad para ajuste de metricas de momentum.

Issue #507: Indice de Estacionalidad por Categoria.

Permite ajustar expectativas de venta segun temporada, evitando
falsos positivos en Decision Box (ej: protector solar -45% en octubre = normal).

IMPORTANTE: La estacionalidad se aplica SOLO a metricas de corto plazo (momentum),
NO a MAT (Moving Annual Total) que ya tiene estacionalidad "cancelada" por incluir
12 meses completos.
"""

import logging
from typing import Dict, List, Optional
from uuid import UUID

from sqlalchemy.orm import Session

from app.models.seasonality_index import SeasonalityIndex

logger = logging.getLogger(__name__)


class SeasonalityService:
    """
    Servicio para gestion de indices estacionales.

    Proporciona:
    - Batch loading de indices (evita N+1 queries)
    - Calculo de momentum ajustado por estacionalidad
    - Prioridad farmacia > sector > default (1.0)
    """

    def get_indices_batch(
        self,
        db: Session,
        categories: List[str],
        month: int,
        pharmacy_id: Optional[UUID] = None,
    ) -> Dict[str, float]:
        """
        Obtener indices estacionales para multiples categorias en queries eficientes.

        Estrategia "Merge en Python" (sugerida por usuario):
        1. Cargar defaults del sector (pharmacy_id IS NULL)
        2. Cargar especificos de farmacia (si aplica)
        3. Merge en diccionario (especifico sobrescribe default)

        Args:
            db: Sesion de base de datos
            categories: Lista de categorias NECESIDAD L1
            month: Mes actual (1-12)
            pharmacy_id: ID farmacia (opcional, para indices personalizados)

        Returns:
            Dict[category, index_value] - 1.0 si no existe indice
        """
        if not categories:
            return {}

        # Iniciar con default 1.0 para todas las categorias
        result: Dict[str, float] = {cat: 1.0 for cat in categories}
        specifics: list = []  # Inicializar para evitar NameError en logging

        # 1. Obtener defaults del sector (pharmacy_id IS NULL)
        defaults = (
            db.query(SeasonalityIndex)
            .filter(
                SeasonalityIndex.category.in_(categories),
                SeasonalityIndex.month == month,
                SeasonalityIndex.pharmacy_id.is_(None),
            )
            .all()
        )

        # Aplicar defaults
        for row in defaults:
            result[row.category] = float(row.index_value)

        # 2. Obtener especificos de farmacia (si aplica)
        if pharmacy_id:
            specifics = (
                db.query(SeasonalityIndex)
                .filter(
                    SeasonalityIndex.category.in_(categories),
                    SeasonalityIndex.month == month,
                    SeasonalityIndex.pharmacy_id == pharmacy_id,
                )
                .all()
            )

            # Sobrescribir con especificos (la farmacia manda)
            for row in specifics:
                result[row.category] = float(row.index_value)

        logger.debug(
            "[seasonality] batch loaded",
            extra={
                "categories_count": len(categories),
                "month": month,
                "pharmacy_id": str(pharmacy_id) if pharmacy_id else None,
                "defaults_loaded": len(defaults),
                "specifics_loaded": len(specifics),
            },
        )

        return result

    def get_index(
        self,
        db: Session,
        category: str,
        month: int,
        pharmacy_id: Optional[UUID] = None,
    ) -> float:
        """
        Obtener indice estacional para una categoria especifica.

        Prioridad:
        1. Indice especifico de farmacia
        2. Indice sector (pharmacy_id = NULL)
        3. Default 1.0 (sin ajuste)

        Args:
            db: Sesion de base de datos
            category: Categoria NECESIDAD L1
            month: Mes actual (1-12)
            pharmacy_id: ID farmacia (opcional)

        Returns:
            Valor del indice estacional (1.0 = promedio)
        """
        result = self.get_indices_batch(db, [category], month, pharmacy_id)
        return result.get(category, 1.0)

    def calculate_momentum(
        self,
        current_sales: float,
        avg_monthly_sales: float,
        seasonality_index: float,
    ) -> float:
        """
        Calcular momentum ajustado por estacionalidad.

        Formula:
            expected_sales = avg_monthly_sales * seasonality_index
            momentum = ((current_sales / expected_sales) - 1) * 100

        Ejemplo protector solar octubre (index=0.5):
            - avg_monthly_sales = 1000
            - expected_sales = 1000 * 0.5 = 500 (se espera caida!)
            - current_sales = 550
            - momentum = (550/500 - 1) * 100 = +10% (mejor que esperado!)

        Args:
            current_sales: Ventas del periodo actual (ej: ultimos 30 dias)
            avg_monthly_sales: Promedio mensual historico
            seasonality_index: Indice estacional del mes actual

        Returns:
            Momentum en porcentaje (positivo = mejor que esperado)
        """
        expected = avg_monthly_sales * seasonality_index

        # Protección defensiva: evitar división por cero o valores negativos
        if expected <= 0:
            return 0.0

        momentum = ((current_sales / expected) - 1) * 100
        return round(momentum, 1)

    def get_seasonality_context(
        self,
        index: float,
        momentum: Optional[float] = None,
    ) -> Optional[str]:
        """
        Generar mensaje de contexto estacional para UI.

        Args:
            index: Indice estacional
            momentum: Momentum calculado (opcional)

        Returns:
            Mensaje explicativo o None si no hay contexto relevante
        """
        if index == 1.0:
            return None

        if index < 0.8:
            base = f"Temporada baja ({index:.1f}x)"
        elif index > 1.2:
            base = f"Temporada alta ({index:.1f}x)"
        else:
            # Entre 0.8 y 1.2, no mostrar contexto especial
            return None

        if momentum is not None:
            if momentum > 0:
                return f"{base} - rendimiento +{momentum:.0f}% vs esperado"
            else:
                return f"{base} - rendimiento {momentum:.0f}% vs esperado"

        return base


# Singleton para uso global
seasonality_service = SeasonalityService()
