"""
Sandwich Classifier Service - The Complete Pipeline (v2)

This service implements the "Sandwich Strategy" for product classification
with a 5-level priority system:

1. BLACKLIST (Tier 0) - Products to exclude (regalo, servicio, etc.)
2. TIER_1_SPECIFIC - High priority specific rules (aftas > colutorio)
3. TIER_1_GENERIC - Lower priority generic rules (colutorio, pasta dental)
4. TIER_2 (ingredient_rules.py) - ~800 existing rules
5. LLM + ARBITER - Enrichment with rescue logic for suppressed topicals

Philosophy:
- Specific always beats generic (aftas > higiene_dental)
- Blacklist catches non-products early
- Rules run on BOTH raw name AND enriched data
- LLM provides rescue when rules are suppressed for topicals

Example flow:
    Input: "ALOCLAIR PLUS COLUTORIO 120ML"

    Step 1 (Blacklist): No match
    Step 2 (TIER_1_SPECIFIC): "aloclair" -> aftas (0.95)
    Step 3: STOP - specific match found

    Output: necesidad="aftas", source="tier1_specific", confidence=0.95

    vs. old behavior where "colutorio" would win -> higiene_dental
"""

import logging
from dataclasses import dataclass
from enum import Enum
from typing import Dict, List, Optional, Tuple

from .ingredient_rules import RuleMatch, get_rules_engine
from .semantic_enrichment_service import (
    SemanticEnrichment,
    get_semantic_enrichment_service,
)
from .brand_detection_service import SIMPLE_BRANDS, is_service_product
from ..schemas.symptom_taxonomy import (
    CATEGORY_PRIORITY,
    RulePriority,
    get_parent_category,
    get_priority,
)
from ..core.taxonomy_tier1 import (
    match_tier1,
    Tier1Match,
    SUPPRESS_IF_TOPICAL,
    rescue_suppressed_topical,
)
from ..core.category_normalization import normalize_category

logger = logging.getLogger(__name__)


class ClassificationSource(str, Enum):
    """Where the final classification came from."""
    BLACKLIST = "blacklist"       # Matched blacklist (interno_no_venta)
    BRAND = "brand"               # Matched SIMPLE_BRANDS (1000+ brands)
    TIER1_SPECIFIC = "tier1_specific"  # TIER_1 specific rules (aftas, cicatriz)
    TIER1_GENERIC = "tier1_generic"    # TIER_1 generic rules (colutorio)
    TIER2_RULES = "tier2_rules"   # ingredient_rules.py (legacy)
    RULES = "rules"               # Alias for backward compat
    LLM = "llm"                   # LLM suggestion
    ARBITER = "arbiter"           # Arbiter decided between conflicting sources
    RESCUE = "rescue"             # Rescued from suppressed topical
    FALLBACK = "fallback"         # Default when nothing matched
    CACHE = "cache"               # From cache (previously classified)


# Issue #457: Normalized confidence for hybrid scoring
# This ensures the P1/P2/P3 review queue prioritizes correctly
# because raw confidence values from different sources are on different scales
CONFIDENCE_NORMALIZATION: dict[ClassificationSource, float] = {
    ClassificationSource.BLACKLIST: 0.99,       # Almost infallible
    ClassificationSource.TIER1_SPECIFIC: 0.95,  # Very high - specific rules
    ClassificationSource.BRAND: 0.90,           # High but can fail on line extensions
    ClassificationSource.TIER1_GENERIC: 0.75,   # Moderate - false positives common
    ClassificationSource.TIER2_RULES: 0.70,     # Variable quality
    ClassificationSource.RULES: 0.70,           # Alias for TIER2
    ClassificationSource.ARBITER: 0.65,         # Arbiter decisions need review
    ClassificationSource.LLM: 0.60,             # LLM suggestions capped
    ClassificationSource.RESCUE: 0.55,          # Rescued topicals need review
    ClassificationSource.FALLBACK: 0.20,        # Very low - needs review
    ClassificationSource.CACHE: None,           # Preserve original confidence
}


@dataclass
class ClassificationResult:
    """Result of the Sandwich classification."""
    necesidad: str
    parent_category: Optional[str]
    confidence: float
    source: ClassificationSource

    # Details
    rule_match: Optional[RuleMatch] = None
    enrichment: Optional[SemanticEnrichment] = None
    arbiter_reason: Optional[str] = None

    # Metadata
    product_name: str = ""
    processing_time_ms: int = 0

    def to_dict(self) -> Dict:
        """Convert to dictionary."""
        return {
            "necesidad": self.necesidad,
            "parent_category": self.parent_category,
            "confidence": self.confidence,
            "source": self.source.value,
            "arbiter_reason": self.arbiter_reason,
            "product_name": self.product_name,
            "processing_time_ms": self.processing_time_ms,
        }


class SandwichClassifierService:
    """
    The Sandwich Strategy Classifier.

    Orchestrates:
    1. Semantic enrichment (LLM)
    2. Rule matching (deterministic)
    3. Arbitration (priority logic)
    """

    # Confidence thresholds
    RULE_HIGH_CONFIDENCE = 0.80   # Rule wins for TIER_2 if above this
    LLM_MIN_CONFIDENCE = 0.60     # LLM suggestion needs at least this
    FALLBACK_CONFIDENCE = 0.30    # Confidence for fallback category

    # SUPPRESS_IF_TOPICAL is now imported from taxonomy_tier1

    def __init__(
        self,
        use_cache: bool = True,
        cache_ttl_seconds: int = 3600,
    ):
        self._rules_engine = get_rules_engine()
        self._enrichment_service = get_semantic_enrichment_service()
        self._cache: Dict[str, ClassificationResult] = {}
        self._use_cache = use_cache
        self._cache_ttl = cache_ttl_seconds

        # Issue #449: DB overrides (lazy loaded, cached 5 min)
        self._db_overrides: Optional[Dict[str, Dict[str, str]]] = None
        self._db_overrides_loaded_at: Optional[float] = None
        self._db_overrides_ttl = 300  # 5 minutes

    def classify(
        self,
        product_name: str,
        skip_llm: bool = False,
        force_refresh: bool = False,
    ) -> ClassificationResult:
        """
        Classify a product using the 5-Level Sandwich Strategy.

        Flow:
        1. BLACKLIST - Exclude non-products (regalo, servicio)
        2. TIER_1_SPECIFIC - High priority specific rules (aftas, cicatriz)
        3. TIER_1_GENERIC - Lower priority generic rules (colutorio)
        4. TIER_2 (ingredient_rules.py) - ~800 existing rules
        5. LLM + ARBITER - Enrichment with rescue logic

        Args:
            product_name: Raw product name
            skip_llm: Skip LLM enrichment (use rules on raw name only)
            force_refresh: Ignore cache

        Returns:
            ClassificationResult with final classification
        """
        import time
        start_time = time.time()

        # Check cache
        cache_key = product_name.lower().strip()
        if self._use_cache and not force_refresh and cache_key in self._cache:
            cached = self._cache[cache_key]
            cached.source = ClassificationSource.CACHE
            logger.debug(f"Cache hit: {product_name[:30]}... -> {cached.necesidad}")
            return cached

        # =====================================================================
        # STEP 0.3: Service Product Filter (Issue #486)
        # Filter internal pharmacy services (SPD, SERVICIO NUTRICION, etc.)
        # These are not real products and should be classified as interno_no_venta
        # =====================================================================
        if is_service_product(product_name):
            result = self._build_result(
                necesidad="interno_no_venta",
                confidence=0.99,  # Service detection is highly reliable
                source=ClassificationSource.BLACKLIST,
                product_name=product_name,
                arbiter_reason=f"SERVICE_FILTER: {product_name[:30]}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            logger.debug(f"Service product filtered: {product_name[:30]}...")
            return result

        # =====================================================================
        # STEP 0.5: DB Overrides (Issue #449)
        # Priority: DB Blacklist > DB Brands > DB Keywords
        # These override hardcoded rules and allow dynamic keyword management
        # =====================================================================

        # Check DB blacklist first (highest priority)
        db_blacklist = self._match_db_override(product_name, "blacklist")
        if db_blacklist:
            keyword, category, _ = db_blacklist
            result = self._build_result(
                necesidad=category,
                confidence=0.99,  # DB blacklist is definitive
                source=ClassificationSource.BLACKLIST,
                product_name=product_name,
                arbiter_reason=f"DB_BLACKLIST: {keyword}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # Check DB brands (high priority, before TIER_1 specific)
        db_brand = self._match_db_override(product_name, "brands")
        if db_brand:
            keyword, category, _ = db_brand
            result = self._build_result(
                necesidad=category,
                confidence=0.95,  # DB brand override
                source=ClassificationSource.BRAND,
                product_name=product_name,
                arbiter_reason=f"DB_BRAND: {keyword}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # =====================================================================
        # STEP 1: TIER_1 Rules (Blacklist + Specific + Generic)
        # =====================================================================
        tier1_match = match_tier1(product_name)

        if tier1_match:
            # Blacklist match -> internal, skip everything
            if tier1_match.tier == "blacklist":
                result = self._build_result(
                    necesidad=tier1_match.necesidad,
                    confidence=tier1_match.confidence,
                    source=ClassificationSource.BLACKLIST,
                    product_name=product_name,
                    arbiter_reason=f"BLACKLIST: {tier1_match.matched_term}",
                    start_time=start_time,
                )
                self._cache[cache_key] = result
                return result

            # Specific match (aftas, cicatriz, etc.) -> high priority, return immediately
            if tier1_match.tier == "specific":
                result = self._build_result(
                    necesidad=tier1_match.necesidad,
                    confidence=tier1_match.confidence,
                    source=ClassificationSource.TIER1_SPECIFIC,
                    product_name=product_name,
                    arbiter_reason=f"TIER1_SPECIFIC: {tier1_match.matched_term}",
                    start_time=start_time,
                )
                self._cache[cache_key] = result
                return result

        # =====================================================================
        # STEP 2: BRAND Layer (1000+ brands from brand_detection_service)
        # Brand is a powerful predictor - Suavinex=infantil, Cumlaude=gineco
        # Runs AFTER TIER1_SPECIFIC (specific rules are more precise)
        # =====================================================================
        brand_match = self._match_brand(product_name)
        if brand_match:
            brand_name, necesidad = brand_match
            result = self._build_result(
                necesidad=necesidad,
                confidence=0.92,  # Brand confidence
                source=ClassificationSource.BRAND,
                product_name=product_name,
                arbiter_reason=f"BRAND: {brand_name}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # NOTE: TIER_1 GENERIC is handled later (after TIER_2 check)
        # Generic match (colutorio, champu) -> TIER_2 might have more specific match

        # =====================================================================
        # STEP 2.5: DB Keywords (Issue #449)
        # Check DB keyword overrides before TIER_2 rules
        # =====================================================================
        db_keyword = self._match_db_override(product_name, "keywords")
        if db_keyword:
            keyword, category, _ = db_keyword
            result = self._build_result(
                necesidad=category,
                confidence=0.90,  # DB keyword override
                source=ClassificationSource.TIER1_SPECIFIC,  # Treat as specific rule
                product_name=product_name,
                arbiter_reason=f"DB_KEYWORD: {keyword}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # =====================================================================
        # STEP 3: TIER_2 Rules (ingredient_rules.py - ~800 rules)
        # =====================================================================
        tier2_match = self._rules_engine.match_text(product_name)
        tier2_best = tier2_match[0] if tier2_match else None

        # If TIER_2 has a high-confidence match, use it
        if tier2_best and tier2_best.confidence >= 0.85:
            result = self._build_result(
                necesidad=tier2_best.necesidad,
                confidence=tier2_best.confidence,
                source=ClassificationSource.TIER2_RULES,
                rule_match=tier2_best,
                product_name=product_name,
                arbiter_reason=f"TIER2 ({tier2_best.rule_type}): {tier2_best.matched_term}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # If TIER_1 generic had a match, use it (fallback)
        if tier1_match and tier1_match.tier == "generic":
            result = self._build_result(
                necesidad=tier1_match.necesidad,
                confidence=tier1_match.confidence,
                source=ClassificationSource.TIER1_GENERIC,
                product_name=product_name,
                arbiter_reason=f"TIER1_GENERIC: {tier1_match.matched_term}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # If TIER_2 had a lower confidence match, still use it
        if tier2_best:
            result = self._build_result(
                necesidad=tier2_best.necesidad,
                confidence=tier2_best.confidence,
                source=ClassificationSource.TIER2_RULES,
                rule_match=tier2_best,
                product_name=product_name,
                arbiter_reason=f"TIER2 low-conf ({tier2_best.rule_type}): {tier2_best.matched_term}",
                start_time=start_time,
            )
            self._cache[cache_key] = result
            return result

        # =====================================================================
        # STEP 5: LLM Enrichment + Arbiter (if not skipped)
        # =====================================================================
        enrichment: Optional[SemanticEnrichment] = None
        if not skip_llm:
            enrichment = self._enrichment_service.enrich_safe(product_name)

        # =====================================================================
        # STEP 5.5: LLM Brand Detection (Issue #486)
        # If LLM extracted a brand name and it matches SIMPLE_BRANDS, use it.
        # This captures brands that hardcoded detection missed because of
        # unusual product naming, typos, or abbreviations.
        # Confidence capped at 0.65 (LLM ceiling) to trigger review queue.
        # =====================================================================
        if enrichment and enrichment.marca_detectada:
            llm_brand = enrichment.marca_detectada.lower().strip()
            # Validate: at least 3 chars, alphanumeric, and exists in known brands
            is_valid_brand = (
                len(llm_brand) >= 3
                and llm_brand in SIMPLE_BRANDS
                and llm_brand.replace(' ', '').replace('-', '').isalnum()
            )
            if is_valid_brand:
                result = self._build_result(
                    necesidad=SIMPLE_BRANDS[llm_brand],
                    confidence=0.65,  # LLM brand = moderate confidence
                    source=ClassificationSource.LLM,
                    product_name=product_name,
                    arbiter_reason=f"LLM_BRAND: {llm_brand}",
                    start_time=start_time,
                )
                self._cache[cache_key] = result
                logger.debug(f"LLM brand match: {product_name[:30]}... -> {llm_brand} -> {SIMPLE_BRANDS[llm_brand]}")
                return result

        # Try rules on enriched data
        enriched_match: Optional[RuleMatch] = None
        if enrichment:
            enriched_match = self._rules_engine.match_enriched(
                product_name=product_name,
                semantic_summary=enrichment.semantic_summary,
                indicaciones=enrichment.indicaciones,
                ingredientes=", ".join(enrichment.ingredientes) if enrichment.ingredientes else None,
            )

        # Arbiter decides based on enrichment
        result = self._arbitrate(
            product_name=product_name,
            raw_match=None,  # No raw match at this point
            enriched_match=enriched_match,
            enrichment=enrichment,
            start_time=start_time,
        )

        # Cache result
        if self._use_cache:
            self._cache[cache_key] = result

        return result

    def _arbitrate(
        self,
        product_name: str,
        raw_match: Optional[RuleMatch],
        enriched_match: Optional[RuleMatch],
        enrichment: Optional[SemanticEnrichment],
        start_time: float,
    ) -> ClassificationResult:
        """
        The Arbiter: Decides which classification wins.

        Priority logic:
        1. TIER_1 categories: Rule ALWAYS wins
        2. TIER_2 categories: Rule wins if confidence > 0.8
        3. TIER_3 categories: LLM can override

        If no matches, fall back to "otros".

        IMPORTANT: When comparing raw_match vs enriched_match, we prefer
        based on rule_type priority (keyword > brand > product_type > ingredient).
        This prevents LLM-extracted generic ingredients like "vitamina E" from
        overriding specific brand matches like "aposan" -> material_sanitario.
        """
        # Rule type priority (lower = better)
        RULE_TYPE_PRIORITY = {"keyword": 0, "product_type": 1, "brand": 2, "ingredient": 3}

        # Choose best rule: Compare raw_match vs enriched_match intelligently
        best_rule = None
        if raw_match and enriched_match:
            # Compare by rule_type priority first, then confidence
            raw_priority = RULE_TYPE_PRIORITY.get(raw_match.rule_type, 99)
            enriched_priority = RULE_TYPE_PRIORITY.get(enriched_match.rule_type, 99)

            if raw_priority < enriched_priority:
                # Raw match has higher priority (e.g., brand vs ingredient)
                best_rule = raw_match
            elif enriched_priority < raw_priority:
                # Enriched match has higher priority
                best_rule = enriched_match
            else:
                # Same priority, prefer higher confidence
                best_rule = raw_match if raw_match.confidence >= enriched_match.confidence else enriched_match
        else:
            # Only one match exists
            best_rule = enriched_match or raw_match

        # SUPPRESS_IF_TOPICAL Filter: If product is topical and match is for
        # an oral supplement category from ingredient extraction, suppress it.
        # E.g., "vitamina E" in a moisturizer shouldn't classify as vitaminas.
        suppressed_reason = None
        if best_rule and enrichment and enrichment.es_topico():
            if (
                best_rule.rule_type == "ingredient"
                and best_rule.necesidad in SUPPRESS_IF_TOPICAL
            ):
                suppressed_reason = (
                    f"SUPPRESSED: {best_rule.necesidad} ({best_rule.matched_term}) "
                    f"invalid for topical product (formato={enrichment.formato})"
                )
                logger.debug(suppressed_reason)
                best_rule = None  # Nullify the match

        # Case 1: No matches at all (or match was suppressed)
        if not best_rule and not enrichment:
            reason = suppressed_reason or "No rule matches and no enrichment"
            return self._build_result(
                necesidad="otros",
                confidence=self.FALLBACK_CONFIDENCE,
                source=ClassificationSource.FALLBACK,
                product_name=product_name,
                arbiter_reason=reason,
                start_time=start_time,
            )

        # Case 2: Rule match exists
        if best_rule:
            priority = get_priority(best_rule.necesidad)

            # TIER_1: Rule ALWAYS wins
            if priority == RulePriority.TIER_1:
                return self._build_result(
                    necesidad=best_rule.necesidad,
                    confidence=best_rule.confidence,
                    source=ClassificationSource.RULES,
                    rule_match=best_rule,
                    enrichment=enrichment,
                    product_name=product_name,
                    arbiter_reason=f"TIER_1 ({best_rule.rule_type}): {best_rule.matched_term}",
                    start_time=start_time,
                )

            # TIER_2: Rule wins if confidence > threshold
            if priority == RulePriority.TIER_2:
                if best_rule.confidence >= self.RULE_HIGH_CONFIDENCE:
                    return self._build_result(
                        necesidad=best_rule.necesidad,
                        confidence=best_rule.confidence,
                        source=ClassificationSource.RULES,
                        rule_match=best_rule,
                        enrichment=enrichment,
                        product_name=product_name,
                        arbiter_reason=f"TIER_2 high-conf ({best_rule.confidence:.2f}): {best_rule.matched_term}",
                        start_time=start_time,
                    )

            # TIER_3 or low-confidence TIER_2: LLM can override
            # But we don't have LLM classification, only enrichment
            # So we still use the rule, but with lower confidence
            return self._build_result(
                necesidad=best_rule.necesidad,
                confidence=best_rule.confidence * 0.9,  # Slightly reduce confidence
                source=ClassificationSource.ARBITER,
                rule_match=best_rule,
                enrichment=enrichment,
                product_name=product_name,
                arbiter_reason=f"TIER_3/low-conf rule used: {best_rule.matched_term}",
                start_time=start_time,
            )

        # Case 3: No rule match, but have enrichment
        # Use enrichment to make educated guess based on keywords
        if enrichment:
            # Try to infer from semantic_summary
            inferred = self._infer_from_enrichment(enrichment)
            if inferred:
                return self._build_result(
                    necesidad=inferred[0],
                    confidence=inferred[1],
                    source=ClassificationSource.ARBITER,
                    enrichment=enrichment,
                    product_name=product_name,
                    arbiter_reason=f"Inferred from enrichment: {inferred[2]}",
                    start_time=start_time,
                )

        # Fallback
        return self._build_result(
            necesidad="otros",
            confidence=self.FALLBACK_CONFIDENCE,
            source=ClassificationSource.FALLBACK,
            enrichment=enrichment,
            product_name=product_name,
            arbiter_reason="No clear classification",
            start_time=start_time,
        )

    def _infer_from_enrichment(
        self,
        enrichment: SemanticEnrichment
    ) -> Optional[Tuple[str, float, str]]:
        """
        Infer category from enrichment when rules don't match.

        Returns (necesidad, confidence, reason) or None.
        """
        # Check flags first
        if enrichment.es_infantil:
            return ("alimentacion_bebe", 0.70, "es_infantil=True")

        if enrichment.es_veterinario:
            return ("otros", 0.60, "es_veterinario=True")

        # Check structured formato - with new format values from SemanticEnrichment
        formato_inferences = {
            "topico_facial": ("hidratacion_facial", 0.50, "formato=topico_facial"),
            "topico_corporal": ("hidratacion_corporal", 0.50, "formato=topico_corporal"),
            "topico_capilar": ("caida_cabello", 0.45, "formato=topico_capilar"),
            "topico_labial": ("labios_secos", 0.55, "formato=topico_labial"),
            "oral_solido": ("vitaminas_minerales", 0.40, "formato=oral_solido"),
            "oral_liquido": ("vitaminas_minerales", 0.40, "formato=oral_liquido"),
            "oftalmico": ("ojo_seco", 0.55, "formato=oftalmico"),
            "nasal": ("congestion_nasal", 0.55, "formato=nasal"),
            "bucal": ("higiene_dental", 0.50, "formato=bucal"),
        }

        if enrichment.formato and enrichment.formato in formato_inferences:
            return formato_inferences[enrichment.formato]

        return None

    def _match_brand(self, product_name: str) -> Optional[Tuple[str, str]]:
        """
        Match product name against SIMPLE_BRANDS dictionary.

        Uses longest-match strategy to avoid false positives.
        E.g., "vichy dercos" should match before "vichy".

        Args:
            product_name: Raw product name

        Returns:
            Tuple of (matched_brand, necesidad) or None
        """
        name_lower = product_name.lower()

        # Sort brands by length (longest first) to match most specific brand
        sorted_brands = sorted(SIMPLE_BRANDS.keys(), key=len, reverse=True)

        for brand in sorted_brands:
            # Use word boundary check to avoid partial matches
            # E.g., "vit" shouldn't match in "vitamin"
            import re
            pattern = r'\b' + re.escape(brand) + r'\b'
            if re.search(pattern, name_lower):
                return (brand, SIMPLE_BRANDS[brand])

        return None

    def _build_result(
        self,
        necesidad: str,
        confidence: float,
        source: ClassificationSource,
        rule_match: Optional[RuleMatch] = None,
        enrichment: Optional[SemanticEnrichment] = None,
        product_name: str = "",
        arbiter_reason: Optional[str] = None,
        start_time: float = 0,
    ) -> ClassificationResult:
        """Build a ClassificationResult with normalized confidence.

        Issue #457: Applies confidence normalization to ensure the review queue
        (P1/P2/P3) prioritizes correctly across different classification sources.
        """
        import time
        processing_time_ms = int((time.time() - start_time) * 1000) if start_time else 0

        # Normalize category to match DB naming conventions
        normalized_necesidad = normalize_category(necesidad)

        # Issue #457: Apply confidence normalization based on source reliability
        # This ensures TIER1_GENERIC (0.75 normalized) products get reviewed
        # before BRAND (0.90 normalized), even if raw confidence was similar
        normalized_confidence = self._normalize_confidence(confidence, source)

        return ClassificationResult(
            necesidad=normalized_necesidad,
            parent_category=get_parent_category(normalized_necesidad),
            confidence=normalized_confidence,
            source=source,
            rule_match=rule_match,
            enrichment=enrichment,
            arbiter_reason=arbiter_reason,
            product_name=product_name,
            processing_time_ms=processing_time_ms,
        )

    def _normalize_confidence(
        self,
        raw_confidence: float,
        source: ClassificationSource,
    ) -> float:
        """Normalize confidence based on source reliability.

        Issue #457: Different sources have different reliability profiles:
        - TIER1_SPECIFIC: Very reliable, rarely wrong
        - BRAND: High reliability but can fail on line extensions
        - TIER1_GENERIC: Moderate, has false positives (e.g., "gel" -> higiene)
        - LLM: Variable, needs human validation

        Returns normalized confidence capped by source ceiling.
        """
        ceiling = CONFIDENCE_NORMALIZATION.get(source)

        # CACHE: Preserve original confidence (already normalized)
        if ceiling is None:
            return raw_confidence

        # For LLM: Use minimum of raw confidence and ceiling
        # This allows good LLM predictions to keep high confidence
        if source == ClassificationSource.LLM:
            return min(raw_confidence, ceiling)

        # For rules: Use ceiling as the normalized value
        # Raw confidence from rules indicates match quality, but
        # the source ceiling reflects historical accuracy
        return ceiling

    def classify_batch(
        self,
        product_names: List[str],
        skip_llm: bool = False,
        progress_callback: Optional[callable] = None,
    ) -> List[ClassificationResult]:
        """
        Classify a batch of products.

        Args:
            product_names: List of product names
            skip_llm: Skip LLM enrichment for all
            progress_callback: Called with (current, total) after each product

        Returns:
            List of ClassificationResult
        """
        results = []
        total = len(product_names)

        for i, name in enumerate(product_names):
            result = self.classify(name, skip_llm=skip_llm)
            results.append(result)

            if progress_callback:
                progress_callback(i + 1, total)

        return results

    # =========================================================================
    # Issue #449: DB Overrides Support
    # =========================================================================

    def _get_db_overrides(self) -> Dict[str, Dict[str, str]]:
        """
        Load keyword overrides from database (cached 5 min).

        Returns:
            {
                "brands": {"olistic": "caida_cabello", ...},
                "keywords": {"anticaida": "caida_cabello", ...},
                "blacklist": {"regalo": "interno_no_venta", ...}
            }
        """
        import time

        # Check cache
        now = time.time()
        if (
            self._db_overrides is not None
            and self._db_overrides_loaded_at
            and now - self._db_overrides_loaded_at < self._db_overrides_ttl
        ):
            return self._db_overrides

        # Load from DB
        try:
            from app.database import SessionLocal
            from app.services.keyword_override_service import KeywordOverrideService

            with SessionLocal() as db:
                service = KeywordOverrideService(db)
                self._db_overrides = service.get_active_overrides()
                self._db_overrides_loaded_at = now

                total = sum(len(v) for v in self._db_overrides.values())
                if total > 0:
                    logger.info(f"Loaded {total} DB keyword overrides")

        except Exception as e:
            logger.warning(f"Could not load DB overrides: {e}")
            self._db_overrides = {"brands": {}, "keywords": {}, "blacklist": {}}
            self._db_overrides_loaded_at = now

        return self._db_overrides

    def _match_db_override(
        self, product_name: str, override_type: str
    ) -> Optional[Tuple[str, str, str]]:
        """
        Match product name against DB overrides.

        Args:
            product_name: Raw product name
            override_type: "blacklist", "brands", or "keywords"

        Returns:
            Tuple of (matched_keyword, category, override_type) or None
        """
        overrides = self._get_db_overrides()
        type_overrides = overrides.get(override_type, {})

        if not type_overrides:
            return None

        name_lower = product_name.lower()

        # Sort by length (longest first) for most specific match
        sorted_keywords = sorted(type_overrides.keys(), key=len, reverse=True)

        for keyword in sorted_keywords:
            if keyword in name_lower:
                return (keyword, type_overrides[keyword], override_type)

        return None

    def get_stats(self) -> Dict:
        """Get classification statistics from cache."""
        if not self._cache:
            return {"cached_products": 0}

        source_counts = {}
        category_counts = {}
        total_confidence = 0.0

        for result in self._cache.values():
            source_counts[result.source.value] = source_counts.get(result.source.value, 0) + 1
            category_counts[result.necesidad] = category_counts.get(result.necesidad, 0) + 1
            total_confidence += result.confidence

        return {
            "cached_products": len(self._cache),
            "avg_confidence": total_confidence / len(self._cache) if self._cache else 0,
            "source_distribution": source_counts,
            "top_categories": sorted(category_counts.items(), key=lambda x: -x[1])[:10],
        }

    def clear_cache(self):
        """Clear the classification cache."""
        self._cache.clear()
        logger.info("Classification cache cleared")

    def health_check(self) -> Dict:
        """Check service health."""
        llm_health = self._enrichment_service.health_check()
        return {
            "status": "healthy" if llm_health.get("status") == "healthy" else "degraded",
            "llm_backend": llm_health,
            "rules_engine": "healthy",
            "cached_products": len(self._cache),
        }


# Singleton
_service: Optional[SandwichClassifierService] = None


def get_sandwich_classifier() -> SandwichClassifierService:
    """Get singleton instance."""
    global _service
    if _service is None:
        _service = SandwichClassifierService()
    return _service
