# backend/app/services/roi_summary_service.py
"""
Service para cálculo y gestión de resúmenes de ROI mensuales.

Issue #514: Feedback Loop de Acciones - ROI Tracker.
Agrega impacto mensual para demostrar valor de suscripción xFarma.
"""

import logging
from calendar import monthrange
from datetime import date, datetime, timezone
from typing import Dict, List, Optional
from uuid import UUID

from sqlalchemy import and_, desc, func
from sqlalchemy.orm import Session

from app.models.action_tracking import ActionStatus, ActionTracking, MonthlyROISummary
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


# Coste de suscripción por defecto (EUR/mes)
DEFAULT_SUBSCRIPTION_COST = 99.0


class ROISummaryService:
    """
    Service para gestión de resúmenes de ROI mensuales.

    Calcula y almacena métricas mensuales de ROI:
    - Conteo de acciones por estado
    - Impacto esperado vs real
    - Desglose por tipo de acción
    - ROI % y ROI neto
    """

    def __init__(self, db: Session):
        self.db = db

    # =========================================================================
    # SUMMARY RETRIEVAL
    # =========================================================================

    def get_summary(
        self,
        pharmacy_id: UUID,
        year: int,
        month: int,
    ) -> Optional[MonthlyROISummary]:
        """
        Obtener resumen ROI de un mes específico.

        Args:
            pharmacy_id: ID de la farmacia
            year: Año
            month: Mes (1-12)

        Returns:
            MonthlyROISummary o None si no existe
        """
        month_date = date(year, month, 1)
        return self.db.query(MonthlyROISummary).filter(
            and_(
                MonthlyROISummary.pharmacy_id == pharmacy_id,
                MonthlyROISummary.month == month_date,
            )
        ).first()

    def get_current_month_summary(
        self,
        pharmacy_id: UUID,
    ) -> Optional[MonthlyROISummary]:
        """Obtener resumen del mes actual."""
        now = utc_now()
        return self.get_summary(pharmacy_id, now.year, now.month)

    def get_last_n_months(
        self,
        pharmacy_id: UUID,
        months: int = 6,
    ) -> List[MonthlyROISummary]:
        """
        Obtener resúmenes de los últimos N meses.

        Args:
            pharmacy_id: ID de la farmacia
            months: Número de meses a obtener

        Returns:
            Lista de resúmenes ordenados por mes DESC
        """
        return self.db.query(MonthlyROISummary).filter(
            MonthlyROISummary.pharmacy_id == pharmacy_id
        ).order_by(
            desc(MonthlyROISummary.month)
        ).limit(months).all()

    # =========================================================================
    # SUMMARY CALCULATION
    # =========================================================================

    def calculate_monthly_summary(
        self,
        pharmacy_id: UUID,
        year: int,
        month: int,
        subscription_cost: Optional[float] = None,
    ) -> MonthlyROISummary:
        """
        Calcular y guardar resumen ROI para un mes.

        Args:
            pharmacy_id: ID de la farmacia
            year: Año
            month: Mes (1-12)
            subscription_cost: Coste de suscripción (default: 99€)

        Returns:
            MonthlyROISummary creado o actualizado
        """
        month_date = date(year, month, 1)
        cost = subscription_cost or DEFAULT_SUBSCRIPTION_COST

        # Definir rango del mes
        _, last_day = monthrange(year, month)
        start_date = datetime(year, month, 1, 0, 0, 0, tzinfo=timezone.utc)
        end_date = datetime(year, month, last_day, 23, 59, 59, tzinfo=timezone.utc)

        # Contar acciones por estado
        counts = self._count_actions_by_status(pharmacy_id, start_date, end_date)

        # Calcular impactos
        expected_impact = self._sum_expected_impact(pharmacy_id, start_date, end_date)
        actual_impact = self._sum_actual_impact(pharmacy_id, start_date, end_date)

        # Desglose por tipo
        impact_by_type = self._calculate_impact_by_type(pharmacy_id, start_date, end_date)

        # Calcular ROI
        roi_percentage = None
        if cost > 0:
            roi_percentage = (actual_impact / cost) * 100

        # Crear o actualizar resumen
        summary = self.get_summary(pharmacy_id, year, month)

        if summary:
            # Actualizar existente
            summary.actions_suggested = counts["total"]
            summary.actions_executed = counts[ActionStatus.EXECUTED.value]
            summary.actions_discarded = counts[ActionStatus.DISCARDED.value]
            summary.actions_postponed = counts[ActionStatus.POSTPONED.value]
            summary.total_expected_impact = expected_impact
            summary.total_actual_impact = actual_impact
            summary.impact_by_type = impact_by_type
            summary.subscription_cost = cost
            summary.roi_percentage = roi_percentage
            summary.calculated_at = utc_now()
        else:
            # Crear nuevo
            summary = MonthlyROISummary(
                pharmacy_id=pharmacy_id,
                month=month_date,
                actions_suggested=counts["total"],
                actions_executed=counts[ActionStatus.EXECUTED.value],
                actions_discarded=counts[ActionStatus.DISCARDED.value],
                actions_postponed=counts[ActionStatus.POSTPONED.value],
                total_expected_impact=expected_impact,
                total_actual_impact=actual_impact,
                impact_by_type=impact_by_type,
                subscription_cost=cost,
                roi_percentage=roi_percentage,
                calculated_at=utc_now(),
            )
            self.db.add(summary)

        self.db.commit()
        self.db.refresh(summary)

        if roi_percentage is not None:
            logger.info(
                f"Calculated ROI summary for pharmacy {pharmacy_id}, "
                f"month {year}-{month:02d}: {roi_percentage:.1f}% ROI"
            )
        else:
            logger.info(
                f"Calculated ROI summary for pharmacy {pharmacy_id}, "
                f"month {year}-{month:02d}: No ROI yet"
            )

        return summary

    def get_or_calculate(
        self,
        pharmacy_id: UUID,
        year: int,
        month: int,
        force_recalculate: bool = False,
    ) -> MonthlyROISummary:
        """
        Obtener resumen existente o calcularlo si no existe.

        Args:
            pharmacy_id: ID de la farmacia
            year: Año
            month: Mes (1-12)
            force_recalculate: Forzar recálculo aunque exista

        Returns:
            MonthlyROISummary
        """
        if not force_recalculate:
            existing = self.get_summary(pharmacy_id, year, month)
            if existing:
                return existing

        return self.calculate_monthly_summary(pharmacy_id, year, month)

    # =========================================================================
    # DASHBOARD WIDGET DATA
    # =========================================================================

    def get_dashboard_data(
        self,
        pharmacy_id: UUID,
    ) -> Dict:
        """
        Obtener datos para el widget del dashboard.

        Returns:
            Dict con datos del widget ROI
        """
        now = utc_now()
        current_year, current_month = now.year, now.month

        # Obtener o calcular resumen actual
        current_summary = self.get_or_calculate(
            pharmacy_id, current_year, current_month
        )

        # Obtener resumen del mes anterior para trend
        if current_month == 1:
            prev_year, prev_month = current_year - 1, 12
        else:
            prev_year, prev_month = current_year, current_month - 1

        prev_summary = self.get_summary(pharmacy_id, prev_year, prev_month)

        # Calcular trend
        roi_trend = None
        trend_direction = None
        if prev_summary and prev_summary.roi_percentage and current_summary.roi_percentage:
            roi_trend = current_summary.roi_percentage - prev_summary.roi_percentage
            if roi_trend > 5:
                trend_direction = "up"
            elif roi_trend < -5:
                trend_direction = "down"
            else:
                trend_direction = "stable"

        # Contar acciones pendientes
        pending_count = self._count_pending_actions(pharmacy_id)
        pending_opportunity = self._sum_pending_opportunity(pharmacy_id)

        # Top tipos de acción
        top_types = self._get_top_action_types(current_summary)

        return {
            "current_month": f"{current_year}-{current_month:02d}",
            "roi_percentage": current_summary.roi_percentage,
            "net_roi_eur": current_summary.net_roi,
            "execution_rate": current_summary.execution_rate,
            "pending_actions": pending_count,
            "total_opportunity_eur": pending_opportunity,
            "roi_trend": roi_trend,
            "trend_direction": trend_direction,
            "top_action_types": top_types,
        }

    # =========================================================================
    # HELPER METHODS
    # =========================================================================

    def _count_actions_by_status(
        self,
        pharmacy_id: UUID,
        start_date: datetime,
        end_date: datetime,
    ) -> Dict[str, int]:
        """Contar acciones por estado en un periodo."""
        results = self.db.query(
            ActionTracking.status,
            func.count(ActionTracking.id).label("count"),
        ).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.created_at >= start_date,
                ActionTracking.created_at <= end_date,
            )
        ).group_by(ActionTracking.status).all()

        counts = {
            ActionStatus.PENDING.value: 0,
            ActionStatus.EXECUTED.value: 0,
            ActionStatus.DISCARDED.value: 0,
            ActionStatus.POSTPONED.value: 0,
            "total": 0,
        }

        for status, count in results:
            counts[status] = count
            counts["total"] += count

        return counts

    def _sum_expected_impact(
        self,
        pharmacy_id: UUID,
        start_date: datetime,
        end_date: datetime,
    ) -> float:
        """Sumar impacto esperado de acciones ejecutadas en el periodo."""
        result = self.db.query(
            func.sum(ActionTracking.expected_impact_eur)
        ).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.status == ActionStatus.EXECUTED.value,
                ActionTracking.executed_at >= start_date,
                ActionTracking.executed_at <= end_date,
            )
        ).scalar()

        return float(result or 0.0)

    def _sum_actual_impact(
        self,
        pharmacy_id: UUID,
        start_date: datetime,
        end_date: datetime,
    ) -> float:
        """Sumar impacto real de acciones ejecutadas en el periodo."""
        result = self.db.query(
            func.sum(ActionTracking.actual_impact_eur)
        ).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.status == ActionStatus.EXECUTED.value,
                ActionTracking.executed_at >= start_date,
                ActionTracking.executed_at <= end_date,
                ActionTracking.actual_impact_eur.isnot(None),
            )
        ).scalar()

        return float(result or 0.0)

    def _calculate_impact_by_type(
        self,
        pharmacy_id: UUID,
        start_date: datetime,
        end_date: datetime,
    ) -> Dict[str, float]:
        """Calcular impacto desglosado por tipo de acción."""
        results = self.db.query(
            ActionTracking.action_type,
            func.sum(ActionTracking.actual_impact_eur).label("impact"),
        ).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.status == ActionStatus.EXECUTED.value,
                ActionTracking.executed_at >= start_date,
                ActionTracking.executed_at <= end_date,
                ActionTracking.actual_impact_eur.isnot(None),
            )
        ).group_by(ActionTracking.action_type).all()

        return {
            action_type: float(impact or 0)
            for action_type, impact in results
        }

    def _count_pending_actions(self, pharmacy_id: UUID) -> int:
        """Contar acciones pendientes actuales."""
        return self.db.query(func.count(ActionTracking.id)).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.status == ActionStatus.PENDING.value,
            )
        ).scalar() or 0

    def _sum_pending_opportunity(self, pharmacy_id: UUID) -> float:
        """Sumar impacto esperado de acciones pendientes."""
        result = self.db.query(
            func.sum(ActionTracking.expected_impact_eur)
        ).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.status == ActionStatus.PENDING.value,
            )
        ).scalar()

        return float(result or 0.0)

    def _get_top_action_types(
        self,
        summary: MonthlyROISummary,
        limit: int = 3,
    ) -> List[Dict]:
        """Obtener top tipos de acción por impacto."""
        if not summary.impact_by_type:
            return []

        # Ordenar por impacto descendente
        sorted_types = sorted(
            summary.impact_by_type.items(),
            key=lambda x: x[1],
            reverse=True,
        )[:limit]

        # Calculate month range from summary
        _, last_day = monthrange(summary.month.year, summary.month.month)
        start_date = datetime(
            summary.month.year, summary.month.month, 1, 0, 0, 0, tzinfo=timezone.utc
        )
        end_date = datetime(
            summary.month.year, summary.month.month, last_day, 23, 59, 59, tzinfo=timezone.utc
        )

        # Contar acciones por tipo (del mes específico)
        type_counts = self.db.query(
            ActionTracking.action_type,
            func.count(ActionTracking.id).label("count"),
        ).filter(
            and_(
                ActionTracking.pharmacy_id == summary.pharmacy_id,
                ActionTracking.status == ActionStatus.EXECUTED.value,
                ActionTracking.executed_at >= start_date,
                ActionTracking.executed_at <= end_date,
            )
        ).group_by(ActionTracking.action_type).all()

        count_map = {t: c for t, c in type_counts}

        return [
            {
                "type": action_type,
                "impact": impact,
                "count": count_map.get(action_type, 0),
            }
            for action_type, impact in sorted_types
        ]

    # =========================================================================
    # HISTORICAL AGGREGATION
    # =========================================================================

    def get_historical_stats(
        self,
        pharmacy_id: UUID,
        months: int = 12,
    ) -> Dict:
        """
        Obtener estadísticas históricas de ROI.

        Args:
            pharmacy_id: ID de la farmacia
            months: Número de meses a analizar

        Returns:
            Dict con estadísticas agregadas
        """
        summaries = self.get_last_n_months(pharmacy_id, months)

        if not summaries:
            return {
                "total_months": 0,
                "avg_roi_percentage": None,
                "total_accumulated_impact": 0.0,
                "total_subscription_cost": 0.0,
            }

        total_impact = sum(s.total_actual_impact for s in summaries)
        total_cost = sum(s.subscription_cost for s in summaries)

        roi_values = [s.roi_percentage for s in summaries if s.roi_percentage]
        avg_roi = sum(roi_values) / len(roi_values) if roi_values else None

        return {
            "total_months": len(summaries),
            "avg_roi_percentage": round(avg_roi, 1) if avg_roi else None,
            "total_accumulated_impact": round(total_impact, 2),
            "total_subscription_cost": round(total_cost, 2),
        }
