# backend/app/services/report_service.py
"""
Servicio de generación de reportes (Issue #511).

Orquesta la recopilación de datos para:
- PDF Dirección: KPIs ejecutivos + alertas
- Excel Suelo: Pick-lists operativas de inventario
"""

import structlog
from datetime import date, datetime, timezone
from typing import Optional
from uuid import UUID

from dateutil.relativedelta import relativedelta
from sqlalchemy import func, and_, case, text
from sqlalchemy.orm import Session

from app.models import SalesData, SalesEnrichment, InventorySnapshot, Pharmacy
from app.schemas.reports import (
    DireccionReportData,
    SueloReportData,
    CategoryGrowth,
    InsightSummary,
    ProductPickItem,
)
from app.schemas.action_plans import (
    ActionPlanData,
    ActionPlanItem,
    ActionPlanSummary,
    ActionPriority,
)
from app.services.insight_engine import InsightEngineService
from app.services.ventalibre_service import VentaLibreService

logger = structlog.get_logger(__name__)


class ReportService:
    """
    Servicio para generar datos de reportes.

    Usa:
    - VentaLibreService para KPIs y categorías
    - InsightEngineService para alertas
    - Queries directas para pick-lists de inventario
    """

    def __init__(self):
        self.ventalibre_service = VentaLibreService()
        self.insight_engine = InsightEngineService()

    async def get_direccion_data(
        self,
        db: Session,
        pharmacy_id: UUID,
        start_date: date,
        end_date: date,
    ) -> DireccionReportData:
        """
        Recopila datos para PDF One-Pager de Dirección.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            start_date: Inicio del período
            end_date: Fin del período

        Returns:
            DireccionReportData con KPIs, categorías, alertas y recomendación
        """
        logger.info(
            "report_service.get_direccion_data.start",
            pharmacy_id=str(pharmacy_id),
            start_date=str(start_date),
            end_date=str(end_date),
        )

        # 1. Obtener nombre de farmacia
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        pharmacy_name = pharmacy.name if pharmacy else "Farmacia"

        # 2. KPIs del período actual
        current_kpis = self._get_period_kpis(db, pharmacy_id, start_date, end_date)

        # 3. KPIs del período anterior (YoY) - usando relativedelta para leap year safety
        prev_start = start_date - relativedelta(years=1)
        prev_end = end_date - relativedelta(years=1)
        prev_kpis = self._get_period_kpis(db, pharmacy_id, prev_start, prev_end)

        # 4. Calcular cambios YoY
        sales_yoy = None
        margin_yoy = None
        if prev_kpis["total_sales"] > 0:
            sales_yoy = (
                (current_kpis["total_sales"] - prev_kpis["total_sales"])
                / prev_kpis["total_sales"]
            ) * 100
        if prev_kpis["margin_percent"] > 0:
            margin_yoy = current_kpis["margin_percent"] - prev_kpis["margin_percent"]

        # 5. Top categorías con crecimiento
        top_categories = self._get_top_categories_yoy(
            db, pharmacy_id, start_date, end_date
        )

        # 6. Obtener insights (alertas)
        try:
            audit_result = await self.insight_engine.run_audit(
                db, pharmacy_id, force=False
            )
            # Ordenar por economic_value y tomar top 5
            insights_sorted = sorted(
                audit_result.insights, key=lambda x: x.economic_value, reverse=True
            )[:5]
            alerts = [
                InsightSummary(
                    rule_code=i.rule_code,
                    severity=i.severity,
                    title=i.title,
                    economic_value=i.economic_value,
                    action_label=i.action_label,
                )
                for i in insights_sorted
            ]
        except Exception as e:
            logger.warning("report_service.insights_error", error=str(e))
            alerts = []

        # 7. Generar recomendación basada en insight principal
        recommendation = ""
        if alerts:
            top_alert = alerts[0]
            recommendation = f"Prioridad: {top_alert.title}. {top_alert.action_label}."

        # 8. Valor de stock actual
        stock_value = self._get_stock_value(db, pharmacy_id)

        logger.info(
            "report_service.get_direccion_data.complete",
            pharmacy_id=str(pharmacy_id),
            sales=current_kpis["total_sales"],
            alerts_count=len(alerts),
        )

        return DireccionReportData(
            pharmacy_name=pharmacy_name,
            period_start=start_date,
            period_end=end_date,
            generated_at=datetime.now(timezone.utc).isoformat(),
            total_sales=current_kpis["total_sales"],
            margin_percent=current_kpis["margin_percent"],
            stock_value=stock_value,
            sku_count=current_kpis["sku_count"],
            sales_yoy_change=sales_yoy,
            margin_yoy_change=margin_yoy,
            stock_yoy_change=None,  # Stock histórico no siempre disponible
            top_categories=top_categories,
            alerts=alerts,
            recommendation=recommendation,
        )

    async def get_suelo_data(
        self,
        db: Session,
        pharmacy_id: UUID,
        include_liquidation: bool = True,
        include_restock: bool = True,
        include_promotion: bool = True,
    ) -> SueloReportData:
        """
        Recopila datos para Excel Pick-Lists de Suelo.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            include_liquidation: Incluir productos a liquidar
            include_restock: Incluir productos a reponer
            include_promotion: Incluir productos a promocionar

        Returns:
            SueloReportData con listas de productos por acción
        """
        logger.info(
            "report_service.get_suelo_data.start",
            pharmacy_id=str(pharmacy_id),
        )

        # Obtener nombre de farmacia
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        pharmacy_name = pharmacy.name if pharmacy else "Farmacia"

        liquidation_items = []
        restock_items = []
        promotion_items = []

        # 1. Productos a liquidar (dead stock > 180 días)
        if include_liquidation:
            liquidation_items = self._get_liquidation_items(db, pharmacy_id)

        # 2. Productos a reponer (cobertura < 7 días)
        if include_restock:
            restock_items = self._get_restock_items(db, pharmacy_id)

        # 3. Productos a promocionar (alto margen, baja rotación)
        if include_promotion:
            promotion_items = self._get_promotion_items(db, pharmacy_id)

        # Calcular totales
        liquidation_total = sum(item.value_eur for item in liquidation_items)
        restock_urgency = sum(1 for item in restock_items if item.priority == 1)
        promotion_margin = sum(item.value_eur for item in promotion_items)

        logger.info(
            "report_service.get_suelo_data.complete",
            pharmacy_id=str(pharmacy_id),
            liquidation_count=len(liquidation_items),
            restock_count=len(restock_items),
            promotion_count=len(promotion_items),
        )

        return SueloReportData(
            pharmacy_name=pharmacy_name,
            generated_at=datetime.now(timezone.utc).isoformat(),
            liquidation_items=liquidation_items,
            restock_items=restock_items,
            promotion_items=promotion_items,
            liquidation_total_value=liquidation_total,
            restock_urgency_count=restock_urgency,
            promotion_potential_margin=promotion_margin,
        )

    async def get_action_plan_data(
        self,
        db: Session,
        pharmacy_id: UUID,
        start_date: date,
        end_date: date,
        max_actions: int = 20,
        min_economic_value: float = 0.0,
    ) -> ActionPlanData:
        """
        Recopila datos para documento Word de Plan de Acción (Issue #513).

        Transforma insights del InsightEngine en acciones priorizadas
        con pasos concretos y campos para seguimiento.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            start_date: Inicio del período analizado
            end_date: Fin del período analizado
            max_actions: Máximo de acciones a incluir
            min_economic_value: Valor económico mínimo para incluir

        Returns:
            ActionPlanData con acciones priorizadas y resumen
        """
        logger.info(
            "report_service.get_action_plan_data.start",
            pharmacy_id=str(pharmacy_id),
            start_date=str(start_date),
            end_date=str(end_date),
            max_actions=max_actions,
        )

        # 1. Obtener nombre y NIF de farmacia (con validación defensiva)
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            logger.warning(
                "report_service.action_plan.pharmacy_not_found",
                pharmacy_id=str(pharmacy_id)
            )
        pharmacy_name = pharmacy.name if pharmacy else "Farmacia"
        pharmacy_nif = getattr(pharmacy, 'nif', None) if pharmacy else None

        # 2. Obtener insights del InsightEngine
        try:
            audit_result = await self.insight_engine.run_audit(
                db, pharmacy_id, force=False
            )
            insights = audit_result.insights
        except Exception as e:
            logger.warning(
                "report_service.action_plan.insights_error",
                error=str(e)
            )
            insights = []

        # 3. Filtrar por valor económico mínimo
        filtered_insights = [
            i for i in insights
            if i.economic_value >= min_economic_value
        ]

        # 4. Ordenar por economic_value (mayor primero)
        sorted_insights = sorted(
            filtered_insights,
            key=lambda x: x.economic_value,
            reverse=True
        )[:max_actions]

        # 5. Convertir insights a ActionPlanItems
        actions = []
        for insight in sorted_insights:
            action = self._insight_to_action_item(insight)
            actions.append(action)

        # 6. Construir resumen
        summary = self._build_action_summary(actions)

        logger.info(
            "report_service.get_action_plan_data.complete",
            pharmacy_id=str(pharmacy_id),
            actions_count=len(actions),
            total_opportunity=summary.total_economic_opportunity,
        )

        return ActionPlanData(
            pharmacy_name=pharmacy_name,
            pharmacy_nif=pharmacy_nif,
            period_start=start_date,
            period_end=end_date,
            generated_at=datetime.now(timezone.utc).isoformat(),
            summary=summary,
            actions=actions,
            insights_source="InsightEngine v1.0",
            export_version="1.0",
        )

    def _insight_to_action_item(self, insight) -> ActionPlanItem:
        """
        Convierte un InsightResult en un ActionPlanItem.

        Añade pasos concretos basados en la categoría del insight
        y determina la prioridad según severidad y valor económico.
        """
        # Determinar prioridad basada en severidad y valor económico
        priority = self._determine_priority(
            insight.severity,
            insight.economic_value
        )

        # Generar pasos concretos según categoría
        action_steps = self._generate_action_steps(
            insight.category,
            insight.rule_code
        )

        # Determinar plazo sugerido
        suggested_deadline = self._get_suggested_deadline(priority)

        # Extraer productos afectados (top 3)
        affected_count = len(insight.affected_items) if insight.affected_items else 0
        top_affected = []
        if insight.affected_items:
            for item in insight.affected_items[:3]:
                name = item.get("product_name") or item.get("name") or "Producto"
                top_affected.append(name[:30])

        return ActionPlanItem(
            rule_code=insight.rule_code,
            category=insight.category,
            severity=insight.severity,
            title=insight.title,
            description=insight.description,
            economic_impact=insight.economic_impact,
            economic_value=insight.economic_value,
            action_label=insight.action_label,
            action_steps=action_steps,
            priority=priority,
            suggested_deadline=suggested_deadline,
            responsible="",  # Campo vacío para rellenar
            notes="",        # Campo vacío para rellenar
            affected_count=affected_count,
            top_affected=top_affected,
        )

    def _determine_priority(
        self,
        severity: str,
        economic_value: float
    ) -> ActionPriority:
        """Determinar prioridad basada en severidad y valor económico."""
        # Valor alto (>1000€) siempre es prioritario
        if economic_value > 1500:
            if severity == "high":
                return ActionPriority.CRITICAL
            return ActionPriority.HIGH

        if economic_value > 500:
            if severity == "high":
                return ActionPriority.HIGH
            return ActionPriority.MEDIUM

        # Valor bajo pero severidad alta
        if severity == "high":
            return ActionPriority.MEDIUM

        return ActionPriority.LOW

    def _generate_action_steps(
        self,
        category: str,
        rule_code: str
    ) -> list[str]:
        """
        Generar pasos concretos según categoría del insight.

        Basado en mejores prácticas de gestión de farmacia.
        """
        # Pasos genéricos por categoría
        steps_by_category = {
            "stock": [
                "Revisar listado de productos afectados",
                "Verificar stock físico en estanterías",
                "Contactar proveedores para pedido urgente",
                "Programar recepción y colocación",
            ],
            "margin": [
                "Analizar productos con margen bajo",
                "Revisar precios de compra con proveedores",
                "Evaluar oportunidades de negociación",
                "Actualizar estrategia de precios si aplica",
            ],
            "hhi": [
                "Identificar laboratorios/marcas dominantes",
                "Buscar alternativas de otros proveedores",
                "Evaluar impacto de diversificación",
                "Planificar transición gradual",
            ],
            "trend": [
                "Revisar tendencia de categoría/producto",
                "Identificar causas del cambio",
                "Adaptar surtido según demanda",
                "Monitorizar evolución próximos 30 días",
            ],
            "surtido": [
                "Evaluar amplitud de surtido actual",
                "Identificar gaps respecto a competencia",
                "Seleccionar productos a incorporar/retirar",
                "Planificar espacios y merchandising",
            ],
        }

        # Pasos específicos por rule_code (sobreescribir si existe)
        steps_by_rule = {
            "STOCK_001": [
                "Revisar productos con stock crítico (<3 días)",
                "Priorizar pedido a mayorista principal",
                "Confirmar fechas de entrega",
                "Colocar productos al recibir",
            ],
            "STOCK_002": [
                "Identificar productos sin rotación >90 días",
                "Evaluar posibilidad de devolución a proveedor",
                "Considerar promociones de liquidación",
                "Ajustar ubicación en estanterías",
            ],
            "MARGIN_001": [
                "Analizar productos con margen <15%",
                "Comparar precios con laboratorios alternativos",
                "Considerar sustitución por equivalentes",
                "Revisar acuerdos comerciales vigentes",
            ],
        }

        # Usar steps específicos si existen, sino genéricos por categoría
        if rule_code in steps_by_rule:
            return steps_by_rule[rule_code]

        return steps_by_category.get(category, [
            "Revisar detalle del insight",
            "Identificar productos/categorías afectadas",
            "Definir acción correctiva",
            "Programar seguimiento",
        ])

    def _get_suggested_deadline(self, priority: ActionPriority) -> str:
        """Obtener plazo sugerido según prioridad."""
        deadlines = {
            ActionPriority.CRITICAL: "Inmediato (24-48h)",
            ActionPriority.HIGH: "Esta semana",
            ActionPriority.MEDIUM: "Próximos 15 días",
            ActionPriority.LOW: "Próximo mes",
        }
        return deadlines.get(priority, "Sin plazo definido")

    def _build_action_summary(
        self,
        actions: list[ActionPlanItem]
    ) -> ActionPlanSummary:
        """Construir resumen del plan de acción."""
        # Conteos por prioridad
        critical_count = sum(
            1 for a in actions if a.priority == ActionPriority.CRITICAL
        )
        high_count = sum(
            1 for a in actions if a.priority == ActionPriority.HIGH
        )
        medium_count = sum(
            1 for a in actions if a.priority == ActionPriority.MEDIUM
        )
        low_count = sum(
            1 for a in actions if a.priority == ActionPriority.LOW
        )

        # Oportunidad total
        total_opportunity = sum(a.economic_value for a in actions)

        # Conteo por categoría
        by_category = {}
        for action in actions:
            cat = action.category
            by_category[cat] = by_category.get(cat, 0) + 1

        # Recomendación principal
        recommendation = ""
        if critical_count > 0:
            recommendation = (
                f"URGENTE: {critical_count} acciones críticas requieren "
                "atención inmediata."
            )
        elif high_count > 0:
            recommendation = (
                f"Priorizar {high_count} acciones de alta prioridad "
                "esta semana."
            )
        elif actions:
            recommendation = (
                f"Plan de mejora con {len(actions)} acciones identificadas."
            )

        return ActionPlanSummary(
            total_actions=len(actions),
            critical_count=critical_count,
            high_count=high_count,
            medium_count=medium_count,
            low_count=low_count,
            total_economic_opportunity=total_opportunity,
            by_category=by_category,
            main_recommendation=recommendation,
        )

    # =========================================================================
    # Métodos privados para KPIs
    # =========================================================================

    def _get_period_kpis(
        self,
        db: Session,
        pharmacy_id: UUID,
        start_date: date,
        end_date: date,
    ) -> dict:
        """Obtener KPIs básicos para un período."""
        result = (
            db.query(
                func.coalesce(func.sum(SalesData.total_amount), 0).label("total_sales"),
                func.coalesce(
                    func.avg(SalesData.margin_percentage), 0
                ).label("margin_percent"),
                func.count(func.distinct(SalesData.codigo_nacional)).label("sku_count"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= start_date,
                SalesData.sale_date <= end_date,
                SalesEnrichment.product_type == "venta_libre",
            )
            .first()
        )

        return {
            "total_sales": float(result.total_sales or 0),
            "margin_percent": float(result.margin_percent or 0),
            "sku_count": int(result.sku_count or 0),
        }

    def _get_stock_value(self, db: Session, pharmacy_id: UUID) -> float:
        """Obtener valor de stock actual."""
        # Intentar desde inventory_snapshot si existe
        try:
            result = (
                db.query(func.sum(InventorySnapshot.stock_value))
                .filter(InventorySnapshot.pharmacy_id == pharmacy_id)
                .scalar()
            )
            return float(result or 0)
        except Exception:
            return 0.0

    def _get_top_categories_yoy(
        self,
        db: Session,
        pharmacy_id: UUID,
        start_date: date,
        end_date: date,
        limit: int = 3,
    ) -> list[CategoryGrowth]:
        """Obtener top categorías por crecimiento YoY."""
        # Período actual
        current = (
            db.query(
                SalesEnrichment.ml_category,
                func.sum(SalesData.total_amount).label("sales"),
                func.count(func.distinct(SalesData.codigo_nacional)).label("count"),
            )
            .join(SalesData, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= start_date,
                SalesData.sale_date <= end_date,
                SalesEnrichment.product_type == "venta_libre",
                SalesEnrichment.ml_category.isnot(None),
            )
            .group_by(SalesEnrichment.ml_category)
            .all()
        )

        current_dict = {r.ml_category: r for r in current}

        # Período anterior (YoY) - relativedelta para leap year safety
        prev_start = start_date - relativedelta(years=1)
        prev_end = end_date - relativedelta(years=1)
        previous = (
            db.query(
                SalesEnrichment.ml_category,
                func.sum(SalesData.total_amount).label("sales"),
            )
            .join(SalesData, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= prev_start,
                SalesData.sale_date <= prev_end,
                SalesEnrichment.product_type == "venta_libre",
                SalesEnrichment.ml_category.isnot(None),
            )
            .group_by(SalesEnrichment.ml_category)
            .all()
        )

        prev_dict = {r.ml_category: float(r.sales or 0) for r in previous}

        # Calcular crecimiento
        categories = []
        for cat, data in current_dict.items():
            if cat in ["otros", "interno_no_venta", "servicios_cat"]:
                continue
            current_sales = float(data.sales or 0)
            prev_sales = prev_dict.get(cat, 0)

            if prev_sales > 0:
                yoy_change = ((current_sales - prev_sales) / prev_sales) * 100
            elif current_sales > 0:
                yoy_change = 100.0  # Nuevo
            else:
                yoy_change = 0.0

            # Nombre display
            display_names = {
                "dolor_fiebre": "Dolor y Fiebre",
                "respiratorio": "Respiratorio",
                "digestivo": "Digestivo",
                "piel": "Dermocosmética",
                "bucal": "Bucal",
                "nutricion": "Suplementos",
                "infantil": "Infantil",
                "sexual": "Sexual",
                "capilar": "Capilar",
                "ocular": "Ocular",
            }

            categories.append(
                CategoryGrowth(
                    category=cat,
                    display_name=display_names.get(cat, cat.replace("_", " ").title()),
                    sales=current_sales,
                    yoy_change=round(yoy_change, 1),
                    product_count=int(data.count or 0),
                )
            )

        # Ordenar por crecimiento (descendente) y tomar top N
        categories.sort(key=lambda x: x.yoy_change, reverse=True)
        return categories[:limit]

    # =========================================================================
    # Métodos privados para Pick-Lists
    # =========================================================================

    def _get_liquidation_items(
        self, db: Session, pharmacy_id: UUID, limit: int = 50
    ) -> list[ProductPickItem]:
        """
        Obtener productos a liquidar (dead stock > 180 días).

        Usa InventorySnapshot si hay datos, sino usa última venta.
        """
        # Productos con stock pero sin ventas recientes (180 días)
        from datetime import timedelta

        cutoff_date = date.today()
        days_ago_180 = cutoff_date - timedelta(days=180)

        # Query: productos con stock en inventario sin ventas recientes
        try:
            results = (
                db.query(
                    InventorySnapshot.product_name,
                    InventorySnapshot.product_code,
                    InventorySnapshot.ean13,
                    InventorySnapshot.stock_quantity,
                    InventorySnapshot.stock_value,
                    func.max(SalesData.sale_date).label("last_sale"),
                )
                .outerjoin(
                    SalesData,
                    and_(
                        SalesData.codigo_nacional == InventorySnapshot.product_code,
                        SalesData.pharmacy_id == InventorySnapshot.pharmacy_id,
                    ),
                )
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.stock_quantity > 0,
                )
                .group_by(
                    InventorySnapshot.product_name,
                    InventorySnapshot.product_code,
                    InventorySnapshot.ean13,
                    InventorySnapshot.stock_quantity,
                    InventorySnapshot.stock_value,
                )
                .having(
                    (func.max(SalesData.sale_date) < days_ago_180)
                    | (func.max(SalesData.sale_date).is_(None))
                )
                .order_by(InventorySnapshot.stock_value.desc())
                .limit(limit)
                .all()
            )

            items = []
            for r in results:
                days_since = 0
                if r.last_sale:
                    days_since = (cutoff_date - r.last_sale).days
                else:
                    days_since = 999

                priority = 1 if days_since > 270 else (2 if days_since > 180 else 3)

                items.append(
                    ProductPickItem(
                        product_name=r.product_name or "Desconocido",
                        codigo_nacional=r.product_code or "",
                        ean=r.ean13,
                        location_hint="N/A",
                        value_eur=float(r.stock_value or 0),
                        quantity=int(r.stock_quantity or 0),
                        priority=priority,
                        reason=f"Sin ventas hace {days_since} días",
                    )
                )
            return items
        except Exception as e:
            logger.warning("report_service.liquidation_query_error", error=str(e))
            return []

    def _get_restock_items(
        self, db: Session, pharmacy_id: UUID, limit: int = 50
    ) -> list[ProductPickItem]:
        """
        Obtener productos a reponer (cobertura < 7 días).

        Calcula velocidad de venta y compara con stock actual.
        """
        try:
            from datetime import timedelta

            # Ventas últimos 30 días por producto
            end_date = date.today()
            start_date = end_date - timedelta(days=30)

            # Subquery: velocidad de venta diaria
            velocity_subq = (
                db.query(
                    SalesData.codigo_nacional,
                    (func.sum(SalesData.quantity) / 30.0).label("daily_velocity"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesData.sale_date <= end_date,
                )
                .group_by(SalesData.codigo_nacional)
                .subquery()
            )

            # Join con inventario
            results = (
                db.query(
                    InventorySnapshot.product_name,
                    InventorySnapshot.product_code,
                    InventorySnapshot.ean13,
                    InventorySnapshot.stock_quantity,
                    velocity_subq.c.daily_velocity,
                )
                .join(
                    velocity_subq,
                    InventorySnapshot.product_code == velocity_subq.c.codigo_nacional,
                )
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.stock_quantity > 0,
                    velocity_subq.c.daily_velocity > 0,
                )
                .all()
            )

            items = []
            for r in results:
                daily_vel = float(r.daily_velocity or 0)
                stock = int(r.stock_quantity or 0)

                if daily_vel > 0:
                    coverage_days = stock / daily_vel
                else:
                    continue  # Sin velocidad, no hay urgencia

                if coverage_days > 7:
                    continue  # Suficiente stock

                priority = 1 if coverage_days < 3 else (2 if coverage_days < 5 else 3)

                items.append(
                    ProductPickItem(
                        product_name=r.product_name or "Desconocido",
                        codigo_nacional=r.product_code or "",
                        ean=r.ean13,
                        location_hint="N/A",
                        value_eur=0.0,  # No aplica para reposición
                        quantity=stock,
                        priority=priority,
                        reason=f"Cobertura: {coverage_days:.0f} días",
                    )
                )

            # Ordenar por prioridad y limitar
            items.sort(key=lambda x: (x.priority, -len(x.reason)))
            return items[:limit]

        except Exception as e:
            logger.warning("report_service.restock_query_error", error=str(e))
            return []

    def _get_promotion_items(
        self, db: Session, pharmacy_id: UUID, limit: int = 30
    ) -> list[ProductPickItem]:
        """
        Obtener productos candidatos a promoción.

        Criterio: Alto margen (>30%) pero baja rotación.
        """
        try:
            from datetime import timedelta

            end_date = date.today()
            start_date = end_date - timedelta(days=90)

            # Productos con alto margen pero pocas ventas
            results = (
                db.query(
                    SalesData.product_name,
                    SalesData.codigo_nacional,
                    func.avg(SalesData.margin_percentage).label("avg_margin"),
                    func.sum(SalesData.quantity).label("total_units"),
                    func.sum(SalesData.margin_amount).label("total_margin"),
                )
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesData.sale_date <= end_date,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesData.margin_percentage > 30,  # Alto margen
                )
                .group_by(SalesData.product_name, SalesData.codigo_nacional)
                .having(func.sum(SalesData.quantity) < 10)  # Baja rotación
                .order_by(func.avg(SalesData.margin_percentage).desc())
                .limit(limit)
                .all()
            )

            items = []
            for r in results:
                margin = float(r.avg_margin or 0)
                priority = 1 if margin > 40 else (2 if margin > 35 else 3)

                items.append(
                    ProductPickItem(
                        product_name=r.product_name or "Desconocido",
                        codigo_nacional=r.codigo_nacional or "",
                        ean=None,
                        location_hint="N/A",
                        value_eur=float(r.total_margin or 0),
                        quantity=int(r.total_units or 0),
                        priority=priority,
                        reason=f"Margen {margin:.0f}%, {r.total_units} uds/trim",
                    )
                )

            return items

        except Exception as e:
            logger.warning("report_service.promotion_query_error", error=str(e))
            return []


# Instancia singleton
report_service = ReportService()
