# backend/app/services/product_matching_service.py
"""
ProductMatchingService - Fuzzy matching para ProductCatalogVentaLibre.

Issue #473: Servicio para detectar duplicados semánticos usando:
- Tokens ordenados alfabéticamente (normalización agresiva)
- rapidfuzz para matching difuso con umbral configurable
- Validación crítica de números (180g ≠ 120g)

Thresholds:
- 100% tokens: Match automático
- >85%: Auto-merge (añadir a variants del existente)
- 70-85%: Marcar para revisión manual (admin W14)
- <70%: Crear nuevo producto

CRÍTICO: Números deben coincidir exactamente para evitar fusionar:
    "Aboca Grintuss Adultos 180g" ≠ "Aboca Grintuss Adultos 120g"

Ejemplo de uso:
    service = ProductMatchingService(db)
    matches = service.find_similar_products("TIRAS CONTOUR NEXT 50U")
    # Encuentra "CONTOUR NEXT TIRAS 50 U" con 100% similitud
"""

import logging
import re
from dataclasses import dataclass
from typing import List, Optional, Set, Tuple
from uuid import UUID

from rapidfuzz import fuzz
from sqlalchemy import func, or_, text
from sqlalchemy.orm import Session

from app.models import ProductCatalogVentaLibre

logger = logging.getLogger(__name__)


# === THRESHOLDS ===
AUTO_MERGE_THRESHOLD = 0.85    # >85%: fusionar automáticamente
REVIEW_THRESHOLD = 0.70        # 70-85%: marcar para revisión
MIN_MATCH_THRESHOLD = 0.70     # <70%: no es match


def extract_numbers(s: str) -> Set[str]:
    """
    Extrae todos los números de un string normalizado.

    CRÍTICO: Usado para validar que los productos tienen los mismos números.
    Ej: "180g" y "120g" tienen números diferentes → NO SON EL MISMO PRODUCTO.

    Returns:
        Set de números encontrados (como strings para preservar leading zeros)
    """
    if not s:
        return set()
    # Extraer números que pueden estar pegados a unidades (180g, 50u, etc.)
    numbers = re.findall(r'\d+', s)
    return set(numbers)


def numbers_match(tokens1: str, tokens2: str) -> bool:
    """
    Verifica que dos strings de tokens tengan exactamente los mismos números.

    CRÍTICO: Esta validación previene falsos positivos como:
        "Aboca Grintuss Adultos 180g" vs "Aboca Grintuss Adultos 120g"

    Returns:
        True si los números coinciden exactamente, False en caso contrario
    """
    nums1 = extract_numbers(tokens1)
    nums2 = extract_numbers(tokens2)
    return nums1 == nums2


@dataclass
class MatchResult:
    """Resultado de matching de un producto."""
    product: ProductCatalogVentaLibre
    similarity: float
    match_type: str  # "exact", "fuzzy_auto", "fuzzy_review", "tokens_exact"

    @property
    def should_auto_merge(self) -> bool:
        """True si el match es suficientemente alto para fusión automática."""
        return self.similarity >= AUTO_MERGE_THRESHOLD

    @property
    def needs_review(self) -> bool:
        """True si el match está en zona de revisión manual."""
        return REVIEW_THRESHOLD <= self.similarity < AUTO_MERGE_THRESHOLD


class ProductMatchingService:
    """
    Servicio para detección de duplicados semánticos en ProductCatalogVentaLibre.

    Usa una combinación de:
    1. Matching exacto por tokens ordenados (más rápido)
    2. Fuzzy matching con rapidfuzz (más flexible)
    """

    def __init__(self, db: Session):
        self.db = db

    def find_similar_products(
        self,
        product_name: str,
        threshold: float = MIN_MATCH_THRESHOLD,
        limit: int = 10,
        exclude_id: Optional[UUID] = None,
    ) -> List[MatchResult]:
        """
        Busca productos similares usando fuzzy matching.

        Proceso:
        1. Normalizar nombre a tokens ordenados
        2. Buscar match exacto de tokens (rápido)
        3. Si no hay match exacto, buscar candidatos y aplicar fuzzy

        Args:
            product_name: Nombre del producto a buscar
            threshold: Umbral mínimo de similitud (0.0-1.0)
            limit: Máximo de resultados
            exclude_id: UUID de producto a excluir (útil para buscar duplicados)

        Returns:
            Lista de MatchResult ordenada por similitud descendente
        """
        if not product_name or not product_name.strip():
            return []

        # Normalizar para matching
        tokens = ProductCatalogVentaLibre.normalize_for_matching(product_name)
        if not tokens:
            return []

        results: List[MatchResult] = []

        # === 1. Buscar match exacto de tokens ===
        exact_query = (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.product_name_tokens == tokens)
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
        )
        if exclude_id:
            exact_query = exact_query.filter(ProductCatalogVentaLibre.id != exclude_id)

        exact_matches = exact_query.all()
        for product in exact_matches:
            results.append(MatchResult(
                product=product,
                similarity=1.0,
                match_type="tokens_exact"
            ))

        # Si ya tenemos matches exactos, no necesitamos fuzzy
        if exact_matches:
            return results[:limit]

        # === 2. Fuzzy matching ===
        # Obtener candidatos: productos que comparten al menos un token
        candidates = self._get_candidates_by_tokens(tokens, exclude_id)

        for candidate in candidates:
            if candidate.product_name_tokens:
                # CRÍTICO: Validar que los números coinciden exactamente
                # Esto previene fusionar "180g" con "120g"
                if not numbers_match(tokens, candidate.product_name_tokens):
                    logger.debug(
                        f"Skipping candidate due to number mismatch: "
                        f"'{tokens}' vs '{candidate.product_name_tokens}'"
                    )
                    continue

                # Usar ratio de rapidfuzz (0-100) convertido a 0.0-1.0
                similarity = fuzz.ratio(tokens, candidate.product_name_tokens) / 100.0

                if similarity >= threshold:
                    match_type = "fuzzy_auto" if similarity >= AUTO_MERGE_THRESHOLD else "fuzzy_review"
                    results.append(MatchResult(
                        product=candidate,
                        similarity=similarity,
                        match_type=match_type
                    ))

        # Ordenar por similitud descendente
        results.sort(key=lambda x: x.similarity, reverse=True)

        return results[:limit]

    def _get_candidates_by_tokens(
        self,
        tokens: str,
        exclude_id: Optional[UUID] = None,
        max_candidates: int = 50,
        similarity_threshold: float = 0.3,
    ) -> List[ProductCatalogVentaLibre]:
        """
        Obtiene candidatos que podrían ser similares usando trigramas.

        Estrategia escalable:
        1. Usar pg_trgm (trigramas) para búsqueda eficiente en PostgreSQL
        2. Fallback a LIKE si pg_trgm no está disponible

        Esto reduce 50,000 productos a ~50 candidatos en milisegundos.

        Args:
            tokens: String de tokens normalizados
            exclude_id: UUID a excluir de resultados
            max_candidates: Máximo de candidatos a retornar
            similarity_threshold: Umbral mínimo de similitud de trigramas (0.0-1.0)
        """
        if not tokens:
            return []

        # === Intentar usar pg_trgm (más eficiente) ===
        # Usamos savepoint para poder hacer rollback sin afectar la transacción principal
        try:
            # Crear savepoint para poder revertir solo esta query
            savepoint = self.db.begin_nested()
            try:
                # Query usando similitud de trigramas
                # Requiere extensión pg_trgm e índice GIN
                query = (
                    self.db.query(ProductCatalogVentaLibre)
                    .filter(ProductCatalogVentaLibre.is_active.is_(True))
                    .filter(ProductCatalogVentaLibre.product_name_tokens.isnot(None))
                    .filter(
                        text(
                            "similarity(product_name_tokens, :search_tokens) > :threshold"
                        ).bindparams(search_tokens=tokens, threshold=similarity_threshold)
                    )
                    .order_by(
                        text("similarity(product_name_tokens, :search_tokens) DESC")
                        .bindparams(search_tokens=tokens)
                    )
                    .limit(max_candidates)
                )

                if exclude_id:
                    query = query.filter(ProductCatalogVentaLibre.id != exclude_id)

                candidates = query.all()
                savepoint.commit()

                if candidates:
                    logger.debug(f"Found {len(candidates)} candidates using pg_trgm")
                    return candidates

            except Exception as e:
                # pg_trgm no disponible o error de query
                # Rollback del savepoint solamente, no de la transacción completa
                savepoint.rollback()
                logger.debug(f"pg_trgm not available, falling back to LIKE: {e}")

        except Exception as e:
            # Error al crear savepoint - continuar con fallback
            logger.debug(f"Could not create savepoint, falling back to LIKE: {e}")

        # === Fallback a LIKE (menos eficiente pero siempre funciona) ===
        token_list = tokens.split()
        if not token_list:
            return []

        # Buscar productos que contengan alguno de los primeros tokens
        conditions = []
        for token in token_list[:3]:  # Solo los 3 primeros para eficiencia
            conditions.append(
                ProductCatalogVentaLibre.product_name_tokens.like(f"{token}%")
            )
            conditions.append(
                ProductCatalogVentaLibre.product_name_tokens.like(f"% {token}%")
            )

        query = (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .filter(ProductCatalogVentaLibre.product_name_tokens.isnot(None))
            .filter(or_(*conditions))
        )

        # Aplicar exclude_id ANTES de limit (SQLAlchemy requiere filters antes de limit)
        if exclude_id:
            query = query.filter(ProductCatalogVentaLibre.id != exclude_id)

        return query.limit(max_candidates).all()

    def find_best_match(
        self,
        product_name: str,
        exclude_id: Optional[UUID] = None,
    ) -> Optional[MatchResult]:
        """
        Busca el mejor match para un producto.

        Returns:
            El mejor MatchResult si supera el threshold mínimo, None si no hay match
        """
        matches = self.find_similar_products(
            product_name=product_name,
            threshold=MIN_MATCH_THRESHOLD,
            limit=1,
            exclude_id=exclude_id
        )
        return matches[0] if matches else None

    def find_duplicates_for_product(
        self,
        product: ProductCatalogVentaLibre,
    ) -> List[MatchResult]:
        """
        Busca duplicados potenciales para un producto existente.

        Args:
            product: El producto para el que buscar duplicados

        Returns:
            Lista de productos que podrían ser duplicados
        """
        return self.find_similar_products(
            product_name=product.product_name_display,
            threshold=REVIEW_THRESHOLD,
            exclude_id=product.id
        )

    def get_duplicate_statistics(self) -> dict:
        """
        Obtiene estadísticas de duplicados potenciales en el catálogo.

        Returns:
            Dict con estadísticas de duplicados
        """
        # Productos con tokens
        total_with_tokens = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .filter(ProductCatalogVentaLibre.product_name_tokens.isnot(None))
            .scalar()
        ) or 0

        # Productos sin tokens (pendientes de backfill)
        total_without_tokens = (
            self.db.query(func.count(ProductCatalogVentaLibre.id))
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .filter(ProductCatalogVentaLibre.product_name_tokens.is_(None))
            .scalar()
        ) or 0

        # Tokens duplicados exactos (grupos de duplicados)
        duplicate_tokens = (
            self.db.query(
                ProductCatalogVentaLibre.product_name_tokens,
                func.count(ProductCatalogVentaLibre.id).label('count')
            )
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .filter(ProductCatalogVentaLibre.product_name_tokens.isnot(None))
            .group_by(ProductCatalogVentaLibre.product_name_tokens)
            .having(func.count(ProductCatalogVentaLibre.id) > 1)
            .all()
        )

        duplicate_groups = len(duplicate_tokens)
        total_duplicates = sum(count for _, count in duplicate_tokens)

        return {
            "total_with_tokens": total_with_tokens,
            "total_without_tokens": total_without_tokens,
            "duplicate_groups": duplicate_groups,
            "total_duplicates": total_duplicates,
            "coverage_pct": round(
                total_with_tokens / max(total_with_tokens + total_without_tokens, 1) * 100, 1
            ),
        }

    def backfill_tokens(self, batch_size: int = 1000) -> Tuple[int, int]:
        """
        Rellena product_name_tokens para productos que no lo tienen.

        Args:
            batch_size: Número de productos a procesar por lote

        Returns:
            Tuple (productos_actualizados, productos_con_error)
        """
        updated = 0
        errors = 0

        # Obtener productos sin tokens
        products = (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.is_active.is_(True))
            .filter(ProductCatalogVentaLibre.product_name_tokens.is_(None))
            .limit(batch_size)
            .all()
        )

        for product in products:
            try:
                tokens = ProductCatalogVentaLibre.normalize_for_matching(
                    product.product_name_display
                )
                if tokens:
                    product.product_name_tokens = tokens
                    updated += 1
            except Exception as e:
                logger.warning(f"Error tokenizing product {product.id}: {e}")
                errors += 1

        if updated > 0:
            self.db.commit()
            logger.info(f"Backfill tokens: {updated} actualizados, {errors} errores")

        return updated, errors


def get_product_matching_service(db: Session) -> ProductMatchingService:
    """Factory function para obtener el servicio."""
    return ProductMatchingService(db)
