"""
Servicio para análisis de tendencias y estacionalidad de prescripción.

Issue #489: Tab 3 "Tendencias y Estacionalidad" en /prescription.
Issue #498: STL Decomposition + Forecast con Holt-Winters.

Responsabilidades:
- Heatmap por día de semana × hora (gestión de turnos)
- Índice de estacionalidad mensual (planificación de compras)
- Descomposición STL y forecast (Issue #498)
- (Futuro) Detección de anomalías (Issue #501)
"""

import logging
from datetime import date, datetime, timedelta
from decimal import Decimal
from typing import Any, Dict, List, Optional
from uuid import UUID

import numpy as np
import pandas as pd
from sqlalchemy import and_, extract, func
from sqlalchemy.orm import Session
from statsmodels.tsa.holtwinters import ExponentialSmoothing
from statsmodels.tsa.seasonal import STL

from app.models.dim_date import DimDate
from app.models.inventory_snapshot import InventorySnapshot
from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.utils.spanish_holidays import get_spanish_holidays_range

logger = logging.getLogger(__name__)

# ====================================================================
# DEPRECATED: Usar dim_date para queries SQL (Issue #503)
# Estas constantes se mantienen SOLO para formateo de respuestas fuera
# de queries SQL. Para nuevos desarrollos, usar JOINs con dim_date:
#
#   .join(DimDate, func.date(SalesData.sale_date) == DimDate.date_key)
#   .query(DimDate.month_name, DimDate.weekday_name, ...)
#
# MÉTODOS MIGRADOS a dim_date:
#   - get_monthly_index()
#   - _get_monthly_series()
#   - get_category_patterns()
#   - _get_current_month_seasonality_index()
#
# MÉTODOS NO MIGRADOS (requieren granularidad horaria - ver Issue #515):
#   - get_hourly_heatmap() - Usa SalesData.weekday + EXTRACT(hour)
#
# Los valores son idénticos a dim_date (misma fuente en populate script)
# ====================================================================

# Nombres de días en español (ISO-8601: 1=Lunes, 7=Domingo)
WEEKDAY_NAMES = {
    1: "Lunes",
    2: "Martes",
    3: "Miércoles",
    4: "Jueves",
    5: "Viernes",
    6: "Sábado",
    7: "Domingo",
}

# Nombres de meses en español
MONTH_NAMES = {
    1: "Enero",
    2: "Febrero",
    3: "Marzo",
    4: "Abril",
    5: "Mayo",
    6: "Junio",
    7: "Julio",
    8: "Agosto",
    9: "Septiembre",
    10: "Octubre",
    11: "Noviembre",
    12: "Diciembre",
}


class PrescriptionSeasonalityService:
    """
    Servicio para análisis de tendencias y estacionalidad.

    Proporciona cálculos de patrones temporales para el dashboard
    de prescripción Tab 3 (Issue #489).
    """

    def __init__(self):
        """Inicializar servicio con configuración por defecto."""
        self.max_months_range = 24  # Máximo rango permitido para queries

    def get_hourly_heatmap(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        categories: Optional[List[str]] = None,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Genera heatmap de ventas por día de semana y hora.

        Útil para gestión de turnos y personal.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            categories: Filtrar por categorías específicas (opcional)

        Returns:
            Dict con heatmap_data, summary y period
        """
        logger.info(
            f"[SEASONALITY] Generando heatmap horario para farmacia {pharmacy_id} "
            f"desde {date_from} hasta {date_to}"
        )

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Query base con JOINs
        # NOTA: Usar sale_time para la hora (no sale_date que tiene hora 00:00:00)
        query = (
            db.query(
                SalesData.weekday.label("weekday"),
                extract("hour", SalesData.sale_time).label("hour"),
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
                func.count(SalesData.id).label("transaction_count"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                    SalesData.weekday.isnot(None),  # Requiere weekday
                    SalesData.sale_time.isnot(None),  # Requiere sale_time para hora
                )
            )
            .group_by(SalesData.weekday, extract("hour", SalesData.sale_time))
        )

        # Filtrar por categorías si se especifican
        if categories:
            query = query.filter(ProductCatalog.xfarma_prescription_category.in_(categories))

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            query = query.filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        results = query.all()

        # Construir heatmap_data
        heatmap_data = []
        total_sales = Decimal("0")
        peak_sales = Decimal("0")
        peak_hour = None
        peak_day = None

        for row in results:
            weekday = int(row.weekday) if row.weekday else 1
            hour = int(row.hour) if row.hour is not None else 0
            sales = Decimal(str(row.sales or 0))
            units = int(row.units or 0)
            transaction_count = int(row.transaction_count or 0)

            heatmap_data.append({
                "weekday": weekday,
                "weekday_name": WEEKDAY_NAMES.get(weekday, f"Día {weekday}"),
                "hour": hour,
                "sales": float(sales),
                "units": units,
                "transaction_count": transaction_count,
            })

            total_sales += sales

            # Actualizar picos
            if sales > peak_sales:
                peak_sales = sales
                peak_hour = hour
                peak_day = weekday

        # Ordenar por día y hora
        heatmap_data.sort(key=lambda x: (x["weekday"], x["hour"]))

        # Construir summary
        summary = {
            "total_sales": float(total_sales),
            "total_cells": len(heatmap_data),
            "peak_hour": peak_hour,
            "peak_day": peak_day,
            "peak_day_name": WEEKDAY_NAMES.get(peak_day) if peak_day else None,
            "peak_sales": float(peak_sales),
            "avg_per_slot": float(total_sales / len(heatmap_data)) if heatmap_data else 0,
        }

        logger.info(
            f"[SEASONALITY] Heatmap generado: {len(heatmap_data)} celdas, "
            f"pico: {peak_day_name} {peak_hour}h ({float(peak_sales):.2f}€)"
            if (peak_day_name := WEEKDAY_NAMES.get(peak_day)) else
            f"[SEASONALITY] Heatmap vacío"
        )

        return {
            "heatmap_data": heatmap_data,
            "summary": summary,
            "period": {
                "date_from": date_from.isoformat(),
                "date_to": date_to.isoformat(),
            },
        }

    def get_monthly_index(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        category: Optional[str] = None,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Calcula índice de estacionalidad mensual.

        El índice representa la intensidad relativa de ventas para cada mes:
        - 1.0 = promedio anual
        - >1.0 = mes con ventas superiores al promedio
        - <1.0 = mes con ventas inferiores al promedio

        Issue #529: Meses incompletos se marcan como parciales y se excluyen
        del cálculo del índice para evitar distorsiones.

        Útil para planificación de compras y previsión de demanda.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            category: Categoría específica (opcional)

        Returns:
            Dict con monthly_index, summary, period, data_quality y partial_months
        """
        import calendar

        logger.info(
            f"[SEASONALITY] Calculando índice mensual para farmacia {pharmacy_id} "
            f"desde {date_from} hasta {date_to}"
        )

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Issue #529: Query para contar días con datos por mes/año
        # Detectar meses incompletos (< 25 días de datos)
        days_query = (
            db.query(
                DimDate.month,
                DimDate.year,
                func.count(func.distinct(DimDate.date_key)).label("days_with_data"),
            )
            .join(SalesData, func.date(SalesData.sale_date) == DimDate.date_key)
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
            .group_by(DimDate.month, DimDate.year)
        )

        if category:
            days_query = days_query.filter(ProductCatalog.xfarma_prescription_category == category)

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            days_query = days_query.filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        days_results = days_query.all()

        # Crear mapa de días con datos por (month, year)
        days_map: Dict[tuple, int] = {}
        for row in days_results:
            key = (int(row.month), int(row.year))
            days_map[key] = int(row.days_with_data)

        # Query base con JOINs
        # Issue #503: Usar dim_date en lugar de EXTRACT() para mejor performance y nombres españoles
        query = (
            db.query(
                DimDate.month,
                DimDate.year,
                DimDate.month_name,
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
            )
            .join(DimDate, func.date(SalesData.sale_date) == DimDate.date_key)
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
            .group_by(
                DimDate.month,
                DimDate.year,
                DimDate.month_name,
            )
        )

        # Filtrar por categoría si se especifica
        if category:
            query = query.filter(ProductCatalog.xfarma_prescription_category == category)

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            query = query.filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        results = query.all()

        # Issue #529: Detectar meses parciales
        # Un mes es parcial si tiene < 25 días de datos
        MIN_DAYS_FOR_COMPLETE = 25
        partial_months_info: List[Dict] = []

        # Agregar por mes (promediando años), EXCLUYENDO meses incompletos del cálculo
        monthly_totals: Dict[int, Dict] = {}
        monthly_totals_all: Dict[int, Dict] = {}  # Incluye parciales para mostrar
        years_by_month: Dict[int, set] = {}
        partial_month_data: Dict[int, Dict] = {}  # Para mostrar mes parcial

        for row in results:
            month = int(row.month) if row.month else 1
            year = int(row.year) if row.year else date_from.year
            sales = Decimal(str(row.sales or 0))
            units = int(row.units or 0)

            # Obtener días en el mes y días con datos
            days_in_month = calendar.monthrange(year, month)[1]
            days_with_data = days_map.get((month, year), 0)
            is_partial = days_with_data < MIN_DAYS_FOR_COMPLETE

            if is_partial:
                # Guardar info del mes parcial
                partial_months_info.append({
                    "month": month,
                    "year": year,
                    "month_name": MONTH_NAMES.get(month, f"Mes {month}"),
                    "days_in_month": days_in_month,
                    "days_with_data": days_with_data,
                })
                # Guardar datos para mostrar (pero no incluir en cálculo de índice)
                if month not in partial_month_data:
                    partial_month_data[month] = {
                        "sales": Decimal("0"),
                        "units": 0,
                        "days_in_month": days_in_month,
                        "days_with_data": days_with_data,
                        "year": year,
                    }
                partial_month_data[month]["sales"] += sales
                partial_month_data[month]["units"] += units
                logger.info(
                    f"[SEASONALITY] Issue #529: Mes parcial detectado - "
                    f"{MONTH_NAMES.get(month)} {year}: {days_with_data}/{days_in_month} días"
                )
            else:
                # Mes completo: incluir en cálculo del índice
                if month not in monthly_totals:
                    monthly_totals[month] = {"sales": Decimal("0"), "units": 0, "count": 0}
                    years_by_month[month] = set()

                monthly_totals[month]["sales"] += sales
                monthly_totals[month]["units"] += units
                monthly_totals[month]["count"] += 1
                years_by_month[month].add(year)

        # Calcular promedios por mes (solo meses completos)
        monthly_averages = {}
        for month, data in monthly_totals.items():
            years_count = len(years_by_month.get(month, {1}))
            monthly_averages[month] = {
                "avg_sales": data["sales"] / years_count if years_count > 0 else Decimal("0"),
                "avg_units": data["units"] // years_count if years_count > 0 else 0,
                "years_included": years_count,
            }

        # Calcular promedio anual general (solo con meses completos)
        total_avg_sales = sum(m["avg_sales"] for m in monthly_averages.values())
        months_with_complete_data = len(monthly_averages)
        grand_avg = total_avg_sales / months_with_complete_data if months_with_complete_data > 0 else Decimal("1")

        # Calcular índices (normalizar para que promedio = 1.0)
        monthly_index = []
        peak_month = None
        peak_index = 0.0
        low_month = None
        low_index = float("inf")

        for month in range(1, 13):
            # Issue #529: Verificar si este mes tiene datos parciales
            partial_info = partial_month_data.get(month)
            is_partial = partial_info is not None

            if is_partial:
                # Mes parcial: mostrar datos pero marcar como incompleto
                # NO calcular índice real (sería distorsionado)
                monthly_index.append({
                    "month": month,
                    "month_name": MONTH_NAMES.get(month, f"Mes {month}"),
                    "index": None,  # No calculable con datos parciales
                    "avg_sales": float(partial_info["sales"]),
                    "avg_units": partial_info["units"],
                    "years_included": 1,
                    "is_partial": True,
                    "days_in_month": partial_info["days_in_month"],
                    "days_with_data": partial_info["days_with_data"],
                })
            else:
                avg_data = monthly_averages.get(month, {"avg_sales": Decimal("0"), "avg_units": 0, "years_included": 0})

                # Issue #529: Meses sin datos también son "parciales" (0 días = incompleto)
                has_no_data = avg_data["years_included"] == 0

                if has_no_data:
                    # Mes sin datos: marcarlo como parcial sin índice calculable
                    monthly_index.append({
                        "month": month,
                        "month_name": MONTH_NAMES.get(month, f"Mes {month}"),
                        "index": None,  # No calculable (sin datos)
                        "avg_sales": 0.0,
                        "avg_units": 0,
                        "years_included": 0,
                        "is_partial": True,  # Sin datos = incompleto
                        "days_in_month": 0,
                        "days_with_data": 0,
                    })
                else:
                    # Índice = ventas_mes / promedio_general
                    index = float(avg_data["avg_sales"] / grand_avg) if grand_avg > 0 else 0.0

                    monthly_index.append({
                        "month": month,
                        "month_name": MONTH_NAMES.get(month, f"Mes {month}"),
                        "index": round(index, 3),
                        "avg_sales": float(avg_data["avg_sales"]),
                        "avg_units": avg_data["avg_units"],
                        "years_included": avg_data["years_included"],
                        "is_partial": False,
                    })

                    # Actualizar pico y valle (solo meses completos con datos)
                    if index > peak_index:
                        peak_index = index
                        peak_month = month
                    if index < low_index and index > 0:
                        low_index = index
                        low_month = month

        # Summary
        summary = {
            "peak_month": peak_month,
            "peak_month_name": MONTH_NAMES.get(peak_month) if peak_month else None,
            "peak_index": round(peak_index, 3),
            "low_month": low_month,
            "low_month_name": MONTH_NAMES.get(low_month) if low_month else None,
            "low_index": round(low_index, 3) if low_index != float("inf") else 0.0,
            "avg_annual_sales": float(total_avg_sales),
        }

        # Data quality - Issue #529: Incluir info de meses parciales
        months_with_data = len([m for m in monthly_index if m.get("years_included", 0) > 0])
        partial_count = len([m for m in monthly_index if m.get("is_partial", False)])

        data_quality = {
            "months_with_data": months_with_data,
            "months_complete": months_with_complete_data,
            "months_partial": partial_count,
            "min_months_recommended": 12,
            "sufficient_data": months_with_complete_data >= 12,
            "years_range": f"{date_from.year}-{date_to.year}",
            "partial_months": partial_months_info,  # Detalles de meses parciales
        }

        logger.info(
            f"[SEASONALITY] Índice mensual calculado: {months_with_complete_data}/12 meses completos, "
            f"{partial_count} parciales, pico: {MONTH_NAMES.get(peak_month)} ({peak_index:.2f})"
        )

        return {
            "monthly_index": monthly_index,
            "category": category,
            "summary": summary,
            "period": {
                "date_from": date_from.isoformat(),
                "date_to": date_to.isoformat(),
            },
            "data_quality": data_quality,
        }

    def get_seasonality_kpis(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Calcula KPIs principales de estacionalidad para el header del Tab 3.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            atc_code: Código ATC nivel 1 para filtrar (A-V). Issue #532

        Returns:
            Dict con:
            - trend_yoy: Variación porcentual año actual vs anterior
            - trend_direction: "up", "down" o "stable"
            - next_peak_category: Categoría del próximo pico estacional
            - next_peak_months: Meses del pico (ej: "Nov-Feb")
            - days_until_peak: Días hasta inicio del próximo pico
            - anomalies_count: Número de anomalías (TODO: Issue #501)
            - insufficient_data: True si < 18 meses de datos (Issue #530)
            - data_span_months: Meses de datos disponibles
            - trend_mom: Alternativa Month-over-Month si YoY no disponible

        Algoritmo de Próximo Pico Estacional (Issue #527):
        ------------------------------------------------
        Utiliza un modelo heurístico basado en patrones estacionales conocidos
        en farmacias españolas:

        1. ANTIGRIPALES (Código ATC R05 - Antitusivos/antigripales):
           - Pico: Noviembre a Febrero (temporada de gripe/resfriados)
           - Se predice cuando el mes actual está entre Marzo-Octubre

        2. ANTIHISTAMÍNICOS (Código ATC R06 - Antihistamínicos sistémicos):
           - Pico: Marzo a Junio (temporada de alergias primaverales)
           - Se predice cuando el mes actual está entre Noviembre-Febrero

        Cálculo de días_hasta_pico:
        - Si próximo pico es Antigripales → días hasta 1 de Noviembre
        - Si próximo pico es Antihistamínicos → días hasta 1 de Marzo

        Nota: Este es un modelo simplificado. Una versión futura podría
        analizar los índices estacionales históricos reales por categoría
        ATC de la farmacia específica para predicciones más precisas.
        """
        logger.info(f"[SEASONALITY] Calculando KPIs para farmacia {pharmacy_id}")

        # Validar rango de fechas (consistencia con otros métodos)
        self._validate_date_range(date_from, date_to)

        # Issue #530: Obtener fecha mínima de datos para verificar suficiencia
        min_date_query = (
            db.query(func.min(SalesData.sale_date))
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
        )

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            min_date_query = min_date_query.filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        min_date_result = min_date_query.scalar()

        # Calcular meses de datos disponibles
        if min_date_result:
            min_data_date = min_date_result.date() if hasattr(min_date_result, 'date') else min_date_result
            data_span_months = (
                (date_to.year - min_data_date.year) * 12
                + (date_to.month - min_data_date.month)
            )
        else:
            data_span_months = 0

        # Issue #530: Mínimo 18 meses para YoY significativo
        MIN_MONTHS_FOR_YOY = 18
        insufficient_data = data_span_months < MIN_MONTHS_FOR_YOY

        # Calcular tendencia YoY (solo si hay datos suficientes)
        current_year = date_to.year
        prev_year = current_year - 1

        trend_yoy = None
        trend_direction = "stable"
        trend_mom = None  # Alternativa: Month-over-Month

        if not insufficient_data:
            # Ventas año actual
            current_sales = self._get_total_sales(
                db, pharmacy_id,
                date(current_year, 1, 1),
                date(current_year, 12, 31),
                atc_code=atc_code,
            )

            # Ventas año anterior
            prev_sales = self._get_total_sales(
                db, pharmacy_id,
                date(prev_year, 1, 1),
                date(prev_year, 12, 31),
                atc_code=atc_code,
            )

            # Calcular tendencia YoY
            if prev_sales > 0:
                trend_yoy = float((current_sales - prev_sales) / prev_sales * 100)
                trend_direction = "up" if trend_yoy > 1 else ("down" if trend_yoy < -1 else "stable")
        else:
            # Issue #530: Calcular alternativa MoM si hay al menos 2 meses
            if data_span_months >= 2:
                # Mes actual
                current_month_start = date(date_to.year, date_to.month, 1)
                current_month_sales = self._get_total_sales(
                    db, pharmacy_id,
                    current_month_start,
                    date_to,
                    atc_code=atc_code,
                )

                # Mes anterior
                if date_to.month == 1:
                    prev_month_start = date(date_to.year - 1, 12, 1)
                    prev_month_end = date(date_to.year - 1, 12, 31)
                else:
                    prev_month_start = date(date_to.year, date_to.month - 1, 1)
                    # Último día del mes anterior
                    prev_month_end = current_month_start - timedelta(days=1)

                prev_month_sales = self._get_total_sales(
                    db, pharmacy_id,
                    prev_month_start,
                    prev_month_end,
                    atc_code=atc_code,
                )

                if prev_month_sales > 0:
                    trend_mom = float((current_month_sales - prev_month_sales) / prev_month_sales * 100)
                    trend_direction = "up" if trend_mom > 1 else ("down" if trend_mom < -1 else "stable")

        # =====================================================================
        # ALGORITMO PRÓXIMO PICO ESTACIONAL (Issue #527)
        # =====================================================================
        # Modelo heurístico basado en patrones estacionales en farmacias españolas.
        # Ver docstring para documentación completa del algoritmo.
        #
        # Dos categorías principales con estacionalidad marcada:
        # - ANTIGRIPALES (R05): Pico Nov-Feb (gripe/resfriados invernales)
        # - ANTIHISTAMÍNICOS (R06): Pico Mar-Jun (alergias primaverales)
        # =====================================================================
        current_month = date_to.month

        if current_month in [3, 4, 5, 6, 7, 8, 9, 10]:
            # Meses Mar-Oct: El próximo pico será ANTIGRIPALES en Noviembre
            next_peak_category = "Antigripales"
            next_peak_months = "Nov-Feb"
            # Días hasta el 1 de Noviembre del año actual
            days_until = (date(current_year, 11, 1) - date_to).days
        else:
            # Meses Nov-Feb: El próximo pico será ANTIHISTAMÍNICOS en Marzo
            next_peak_category = "Antihistamínicos"
            next_peak_months = "Mar-Jun"
            # Días hasta el 1 de Marzo (próximo año si estamos en Nov-Dic)
            if current_month >= 11:
                days_until = (date(current_year + 1, 3, 1) - date_to).days
            else:
                # Enero o Febrero: Marzo del mismo año
                days_until = (date(current_year, 3, 1) - date_to).days

        result = {
            "trend_yoy": round(trend_yoy, 1) if trend_yoy is not None else None,
            "trend_direction": trend_direction,
            "next_peak_category": next_peak_category,
            "next_peak_months": next_peak_months,
            "days_until_peak": max(0, days_until),
            "anomalies_count": 0,  # TODO: Implementar en Issue #501
            # Issue #530: Campos para datos insuficientes
            "insufficient_data": insufficient_data,
            "data_span_months": data_span_months,
            "min_months_required": MIN_MONTHS_FOR_YOY,
        }

        # Issue #530: Añadir alternativa MoM si YoY no disponible
        if trend_mom is not None:
            result["trend_mom"] = round(trend_mom, 1)

        return result

    def _get_total_sales(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        atc_code: Optional[str] = None,
    ) -> Decimal:
        """Helper para obtener ventas totales de prescripción en un período."""
        query = (
            db.query(func.sum(SalesData.total_amount))
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
        )

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            query = query.filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        result = query.scalar()
        return Decimal(str(result or 0))

    def _get_monthly_series(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        atc_code: Optional[str] = None,
    ) -> pd.Series:
        """
        Obtiene serie temporal mensual de ventas de prescripción.

        Returns:
            Serie pandas con índice DatetimeIndex mensual y ventas como valores.
        """
        # Query para obtener ventas mensuales
        # Issue #503: Usar dim_date en lugar de EXTRACT() para mejor performance
        query = (
            db.query(
                DimDate.year,
                DimDate.month,
                func.sum(SalesData.total_amount).label("sales"),
            )
            .join(DimDate, func.date(SalesData.sale_date) == DimDate.date_key)
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
        )

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            query = query.filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        results = (
            query
            .group_by(
                DimDate.year,
                DimDate.month,
            )
            .order_by(
                DimDate.year,
                DimDate.month,
            )
            .all()
        )

        # Convertir a DataFrame y luego a Series con índice temporal
        if not results:
            return pd.Series(dtype=float)

        data = []
        for row in results:
            if row.year and row.month:
                data.append({
                    "date": pd.Timestamp(year=int(row.year), month=int(row.month), day=1),
                    "sales": float(row.sales or 0),
                })

        if not data:
            return pd.Series(dtype=float)

        df = pd.DataFrame(data)
        df.set_index("date", inplace=True)
        df.index = pd.DatetimeIndex(df.index, freq="MS")  # Month Start frequency

        return df["sales"]

    def get_trend_decomposition(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Descompone la serie temporal usando STL (Seasonal-Trend decomposition using LOESS).

        STL separa la serie en:
        - Tendencia: dirección general de largo plazo
        - Estacionalidad: patrón que se repite cada 12 meses
        - Residuo: variación no explicada

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio
            date_to: Fecha fin

        Returns:
            Dict con decomposition, summary, period, data_quality
        """
        logger.info(
            f"[SEASONALITY] STL decomposition para farmacia {pharmacy_id} "
            f"desde {date_from} hasta {date_to}"
        )

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Obtener serie mensual
        series = self._get_monthly_series(db, pharmacy_id, date_from, date_to, atc_code=atc_code)

        # Verificar datos suficientes (mínimo 12 meses para 1 ciclo estacional)
        # Issue #501: Reducido de 24 a 12 para mostrar resultados con menos datos
        min_periods = 12
        if len(series) < min_periods:
            logger.warning(
                f"[SEASONALITY] Datos insuficientes para STL: {len(series)} meses "
                f"(mínimo requerido: {min_periods})"
            )
            return {
                "decomposition": [],
                "period": {
                    "date_from": date_from.isoformat(),
                    "date_to": date_to.isoformat(),
                },
                "summary": {
                    "trend_direction": "unknown",
                    "seasonal_strength": 0.0,
                    "data_points": len(series),
                },
                "data_quality": {
                    "sufficient_data": False,
                    "months_available": len(series),
                    "min_periods_required": min_periods,
                    "message": f"Se requieren al menos {min_periods} meses de datos para descomposición STL",
                },
            }

        # Aplicar STL
        try:
            stl = STL(series, period=12, robust=True)
            result = stl.fit()
        except Exception as e:
            logger.error(f"[SEASONALITY] Error en STL: {e}")
            return {
                "decomposition": [],
                "period": {
                    "date_from": date_from.isoformat(),
                    "date_to": date_to.isoformat(),
                },
                "summary": {"error": str(e)},
                "data_quality": {"sufficient_data": False},
            }

        # Construir respuesta
        decomposition = []
        for idx in result.observed.index:
            decomposition.append({
                "date": idx.strftime("%Y-%m-%d"),
                "observed": round(float(result.observed[idx]), 2),
                "trend": round(float(result.trend[idx]), 2),
                "seasonal": round(float(result.seasonal[idx]), 2),
                "residual": round(float(result.resid[idx]), 2),
            })

        # Calcular fuerza estacional (0-1)
        var_seasonal = np.var(result.seasonal)
        var_resid = np.var(result.resid)
        seasonal_strength = max(0, 1 - var_resid / (var_seasonal + var_resid)) if (var_seasonal + var_resid) > 0 else 0

        # Determinar dirección de tendencia
        trend_values = result.trend.dropna()
        if len(trend_values) >= 2:
            first_half = trend_values[:len(trend_values)//2].mean()
            second_half = trend_values[len(trend_values)//2:].mean()
            if second_half > first_half * 1.02:
                trend_direction = "up"
            elif second_half < first_half * 0.98:
                trend_direction = "down"
            else:
                trend_direction = "stable"
        else:
            trend_direction = "unknown"

        logger.info(
            f"[SEASONALITY] STL completado: {len(decomposition)} puntos, "
            f"tendencia: {trend_direction}, fuerza estacional: {seasonal_strength:.2f}"
        )

        return {
            "decomposition": decomposition,
            "period": {
                "date_from": date_from.isoformat(),
                "date_to": date_to.isoformat(),
            },
            "summary": {
                "trend_direction": trend_direction,
                "seasonal_strength": round(seasonal_strength, 3),
                "data_points": len(decomposition),
                "peak_seasonal_month": int(result.seasonal.idxmax().month) if len(result.seasonal) > 0 else None,
                "low_seasonal_month": int(result.seasonal.idxmin().month) if len(result.seasonal) > 0 else None,
            },
            "data_quality": {
                "sufficient_data": True,
                "months_available": len(series),
                "min_periods_required": min_periods,
            },
        }

    def get_forecast(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        periods_ahead: int = 3,
        confidence: float = 0.95,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Genera forecast con Holt-Winters Exponential Smoothing.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del histórico
            date_to: Fecha fin del histórico
            periods_ahead: Meses a predecir (default: 3)
            confidence: Nivel de confianza (default: 0.95)

        Returns:
            Dict con forecast, historical, model_info, summary
        """
        logger.info(
            f"[SEASONALITY] Forecast para farmacia {pharmacy_id}, "
            f"{periods_ahead} meses adelante"
        )

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Limitar períodos de predicción
        periods_ahead = min(periods_ahead, 12)  # Máximo 12 meses

        # Obtener serie mensual
        series = self._get_monthly_series(db, pharmacy_id, date_from, date_to, atc_code=atc_code)

        # Verificar datos mínimos (mínimo 6 meses para cualquier forecast)
        min_periods = 6
        if len(series) < min_periods:
            logger.warning(
                f"[SEASONALITY] Datos insuficientes para forecast: {len(series)} meses"
            )
            return {
                "forecast": [],
                "historical": [],
                "confidence_level": confidence,
                "periods_ahead": periods_ahead,
                "model_info": {"error": "insufficient_data"},
                "summary": {
                    "message": f"Se requieren al menos {min_periods} meses de datos",
                    "months_available": len(series),
                },
            }

        # Seleccionar modelo según cantidad de datos:
        # - >= 24 meses: Holt-Winters completo con estacionalidad (2 ciclos)
        # - 12-23 meses: Holt-Winters solo con tendencia (sin estacionalidad)
        # - 6-11 meses: Simple Exponential Smoothing
        try:
            if len(series) >= 24:
                # Modelo completo con estacionalidad
                logger.info(f"[SEASONALITY] Usando Holt-Winters completo ({len(series)} meses)")
                model = ExponentialSmoothing(
                    series,
                    seasonal_periods=12,
                    trend="add",
                    seasonal="add",
                    initialization_method="estimated",
                ).fit(optimized=True)
                model_type = "Holt-Winters (estacional)"
            elif len(series) >= 12:
                # Modelo con tendencia pero sin estacionalidad
                logger.info(f"[SEASONALITY] Usando Holt-Winters sin estacionalidad ({len(series)} meses, <24 para estacional)")
                model = ExponentialSmoothing(
                    series,
                    trend="add",
                    seasonal=None,
                    initialization_method="estimated",
                ).fit(optimized=True)
                model_type = "Holt-Winters (tendencia)"
            else:
                # Simple Exponential Smoothing para pocos datos
                logger.info(f"[SEASONALITY] Usando SES ({len(series)} meses)")
                model = ExponentialSmoothing(
                    series,
                    trend=None,
                    seasonal=None,
                    initialization_method="estimated",
                ).fit(optimized=True)
                model_type = "Suavizado Exponencial Simple"

            # Generar forecast
            forecast_values = model.forecast(periods_ahead)

            # Calcular intervalo de confianza basado en residuos
            residuals_std = float(model.resid.std())  # Convertir a float nativo
            z_score = 1.96 if confidence == 0.95 else 2.576  # 95% o 99%

        except Exception as e:
            logger.error(f"[SEASONALITY] Error en Holt-Winters: {e}")
            return {
                "forecast": [],
                "historical": [],
                "confidence_level": confidence,
                "periods_ahead": periods_ahead,
                "model_info": {"error": str(e)},
                "summary": {},
            }

        # Construir forecast response
        forecast_list = []
        for idx in forecast_values.index:
            forecast_val = float(forecast_values[idx])
            forecast_list.append({
                "date": idx.strftime("%Y-%m-%d"),
                "month_name": MONTH_NAMES.get(idx.month, f"Mes {idx.month}"),
                "forecast": round(forecast_val, 2),
                "lower_bound": round(forecast_val - z_score * residuals_std, 2),
                "upper_bound": round(forecast_val + z_score * residuals_std, 2),
            })

        # Incluir últimos 6 meses de histórico para contexto
        historical_list = []
        recent_series = series.tail(6)
        for idx in recent_series.index:
            historical_list.append({
                "date": idx.strftime("%Y-%m-%d"),
                "month_name": MONTH_NAMES.get(idx.month, f"Mes {idx.month}"),
                "value": round(float(recent_series[idx]), 2),
            })

        # Calcular resumen
        total_forecast = sum(f["forecast"] for f in forecast_list)
        avg_monthly = total_forecast / periods_ahead if periods_ahead > 0 else 0

        # Determinar tendencia del forecast
        if len(forecast_list) >= 2:
            if forecast_list[-1]["forecast"] > forecast_list[0]["forecast"] * 1.02:
                forecast_trend = "up"
            elif forecast_list[-1]["forecast"] < forecast_list[0]["forecast"] * 0.98:
                forecast_trend = "down"
            else:
                forecast_trend = "stable"
        else:
            forecast_trend = "unknown"

        logger.info(
            f"[SEASONALITY] Forecast completado: {periods_ahead} meses, "
            f"total: {total_forecast:.2f}€, tendencia: {forecast_trend}"
        )

        return {
            "forecast": forecast_list,
            "historical": historical_list,
            "confidence_level": confidence,
            "periods_ahead": periods_ahead,
            "model_info": {
                "type": model_type,
                "seasonal_periods": 12 if len(series) >= 24 else None,
                "trend_type": "additive" if len(series) >= 12 else None,
                "seasonal_type": "additive" if len(series) >= 24 else None,
                "residuals_std": round(float(residuals_std), 2),
                "months_available": len(series),
            },
            "summary": {
                "total_forecast": round(total_forecast, 2),
                "avg_monthly": round(avg_monthly, 2),
                "forecast_trend": forecast_trend,
                "confidence_range": f"±{round(z_score * residuals_std, 2)}€",
            },
        }

    def get_category_patterns(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        min_sales_threshold: Decimal = Decimal("1000"),
        top_n: int = 10,
    ) -> Dict[str, Any]:
        """
        Obtiene patrones de estacionalidad para las principales categorías de prescripción.

        Issue #499: Fase 3 - Estacionalidad por Categoría ATC.

        Casos de uso:
        - Antigripales: Pico Nov-Feb
        - Antihistamínicos: Pico Mar-Jun
        - Protectores solares: Pico Jun-Ago

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            min_sales_threshold: Ventas mínimas para incluir categoría
            top_n: Número máximo de categorías a retornar

        Returns:
            Dict con category_patterns, summary y period
        """
        logger.info(
            f"[SEASONALITY] Calculando patrones por categoría para farmacia {pharmacy_id} "
            f"desde {date_from} hasta {date_to}"
        )

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # 1. Obtener categorías con ventas suficientes
        category_query = (
            db.query(
                ProductCatalog.xfarma_prescription_category.label("category"),
                func.sum(SalesData.total_amount).label("total_sales"),
                func.count(SalesData.id).label("transaction_count"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
            .group_by(ProductCatalog.xfarma_prescription_category)
            .having(func.sum(SalesData.total_amount) >= min_sales_threshold)
            .order_by(func.sum(SalesData.total_amount).desc())
            .limit(top_n)
            .all()
        )

        if not category_query:
            logger.warning(f"[SEASONALITY] No hay categorías con ventas suficientes")
            return {
                "category_patterns": [],
                "summary": {
                    "categories_analyzed": 0,
                    "message": "No hay suficientes datos para análisis por categoría",
                },
                "period": {
                    "date_from": date_from.isoformat(),
                    "date_to": date_to.isoformat(),
                },
            }

        categories = [row.category for row in category_query]
        logger.info(f"[SEASONALITY] Analizando {len(categories)} categorías: {categories}")

        # 2. Calcular índice mensual para cada categoría
        category_patterns = []

        for category in categories:
            # Obtener ventas mensuales por categoría
            # Issue #503: Usar dim_date en lugar de EXTRACT()
            monthly_data = (
                db.query(
                    DimDate.month,
                    DimDate.month_name,
                    func.sum(SalesData.total_amount).label("sales"),
                    func.sum(SalesData.quantity).label("units"),
                )
                .join(DimDate, func.date(SalesData.sale_date) == DimDate.date_key)
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
                .filter(
                    and_(
                        SalesData.pharmacy_id == pharmacy_id,
                        SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                        SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                        ProductCatalog.xfarma_prescription_category == category,
                    )
                )
                .group_by(DimDate.month, DimDate.month_name)
                .order_by(DimDate.month)
                .all()
            )

            if not monthly_data:
                continue

            # Calcular promedio anual
            total_sales = sum(float(row.sales) for row in monthly_data)
            avg_monthly = total_sales / 12 if total_sales > 0 else 0

            # Calcular índice por mes (1.0 = promedio)
            monthly_indices = []
            peak_month = None
            peak_index = 0
            low_month = None
            low_index = float('inf')

            for month_num in range(1, 13):
                month_row = next((r for r in monthly_data if r.month == month_num), None)
                if month_row and avg_monthly > 0:
                    index = float(month_row.sales) / avg_monthly
                    # Issue #503: Usar month_name de dim_date cuando disponible
                    month_name = month_row.month_name
                else:
                    index = 0.0
                    month_name = MONTH_NAMES[month_num]  # Fallback

                monthly_indices.append({
                    "month": month_num,
                    "month_name": month_name,
                    "index": round(index, 2),
                })

                if index > peak_index:
                    peak_index = index
                    peak_month = month_num
                if 0 < index < low_index:
                    low_index = index
                    low_month = month_num

            # Determinar patrón estacional (ej: "Nov-Feb" para antigripales)
            peak_months = self._identify_peak_months(monthly_indices)

            category_patterns.append({
                "category": category,
                "category_display": self._format_category_name(category),
                "total_sales": round(total_sales, 2),
                "monthly_indices": monthly_indices,
                "peak_month": peak_month,
                "peak_month_name": MONTH_NAMES.get(peak_month, "N/A") if peak_month else "N/A",
                "peak_index": round(peak_index, 2),
                "low_month": low_month,
                "low_month_name": MONTH_NAMES.get(low_month, "N/A") if low_month else "N/A",
                "low_index": round(low_index, 2) if low_index != float('inf') else 0,
                "peak_period": peak_months,
                "seasonality_strength": round(peak_index - low_index, 2) if low_index != float('inf') else 0,
            })

        # 3. Ordenar por fuerza estacional (más estacional primero)
        category_patterns.sort(key=lambda x: x["seasonality_strength"], reverse=True)

        logger.info(
            f"[SEASONALITY] Patrones calculados para {len(category_patterns)} categorías"
        )

        return {
            "category_patterns": category_patterns,
            "summary": {
                "categories_analyzed": len(category_patterns),
                "most_seasonal_category": category_patterns[0]["category"] if category_patterns else None,
                "least_seasonal_category": category_patterns[-1]["category"] if category_patterns else None,
            },
            "period": {
                "date_from": date_from.isoformat(),
                "date_to": date_to.isoformat(),
            },
        }

    def _identify_peak_months(self, monthly_indices: List[Dict]) -> str:
        """
        Identifica el período de pico estacional (ej: "Nov-Feb").

        Un mes es parte del pico si su índice > 1.1 (10% sobre promedio).
        """
        peak_months = [m for m in monthly_indices if m["index"] > 1.1]

        if not peak_months:
            return "Sin pico definido"

        # Ordenar por mes
        peak_months.sort(key=lambda x: x["month"])

        # Detectar rangos contiguos (considerando wrap-around Dic-Ene)
        if len(peak_months) == 1:
            return peak_months[0]["month_name"][:3]

        # Simplificación: retornar primer y último mes del pico
        first = peak_months[0]["month_name"][:3]
        last = peak_months[-1]["month_name"][:3]

        if first == last:
            return first

        return f"{first}-{last}"

    def _format_category_name(self, category: str) -> str:
        """Formatea nombre de categoría para display."""
        if not category:
            return "Sin categoría"

        # Convertir snake_case a título
        formatted = category.replace("_", " ").title()

        # Correcciones específicas
        corrections = {
            "Formulas Magistrales": "Fórmulas Magistrales",
            "Dietoterapicos": "Dietoterápicos",
            "Ortopedia Financiada": "Ortopedia Financiada",
        }

        return corrections.get(formatted, formatted)

    def _validate_date_range(self, date_from: date, date_to: date) -> None:
        """Validar que el rango de fechas sea válido y no exceda el límite."""
        if date_from >= date_to:
            raise ValueError(f"date_from ({date_from}) debe ser menor que date_to ({date_to})")

        # Calcular diferencia en meses
        months_diff = (date_to.year - date_from.year) * 12 + (date_to.month - date_from.month)

        if months_diff > self.max_months_range:
            raise ValueError(
                f"El rango máximo permitido es de {self.max_months_range} meses, "
                f"pero se solicitaron {months_diff} meses"
            )

    # =========================================================================
    # STOCK-OUT RISK MATRIX (Issue #500)
    # =========================================================================

    def get_stockout_risk(
        self,
        db: Session,
        pharmacy_id: UUID,
        days_until_restock: int = 7,
        top_n: int = 50,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Calcula la matriz de riesgo de rotura de stock.

        Cruza el índice de estacionalidad del mes actual con el inventario
        para identificar productos con riesgo de stockout.

        Issue #500: Fase 3.5 - Stock-out Risk Matrix.

        Args:
            db: Sesión de base de datos
            pharmacy_id: UUID de la farmacia
            days_until_restock: Días hasta la próxima reposición (default 7)
            top_n: Número máximo de productos a retornar

        Returns:
            Dict con risk_items, summary y metadata
        """
        logger.info(
            f"[STOCKOUT] Calculando riesgo para pharmacy {pharmacy_id}, "
            f"restock_days={days_until_restock}, top_n={top_n}"
        )

        current_month = datetime.now().month
        current_date = date.today()

        # 1. Obtener último snapshot de inventario (solo prescripción)
        latest_snapshot_date = db.query(
            func.max(InventorySnapshot.snapshot_date)
        ).filter(
            InventorySnapshot.pharmacy_id == pharmacy_id,
            InventorySnapshot.product_type == "prescription",
        ).scalar()

        if not latest_snapshot_date:
            logger.warning(f"[STOCKOUT] No inventory snapshot found for {pharmacy_id}")
            return {
                "risk_items": [],
                "summary": {
                    "critical_count": 0,
                    "high_count": 0,
                    "ok_count": 0,
                    "medium_count": 0,
                    "excess_count": 0,
                    "total_products": 0,
                    "message": "No se encontró inventario de prescripción",
                },
                "period": {},
                "inventory_date": None,
            }

        # 2. Obtener productos del inventario con stock
        inventory_query = db.query(
            InventorySnapshot.product_code,
            InventorySnapshot.product_name,
            InventorySnapshot.stock_quantity,
            InventorySnapshot.product_catalog_id,
        ).filter(
            InventorySnapshot.pharmacy_id == pharmacy_id,
            InventorySnapshot.snapshot_date == latest_snapshot_date,
            InventorySnapshot.product_type == "prescription",
            InventorySnapshot.stock_quantity > 0,
        )

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            inventory_query = inventory_query.join(
                ProductCatalog, InventorySnapshot.product_catalog_id == ProductCatalog.id
            ).filter(ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%"))

        inventory_items = inventory_query.all()

        if not inventory_items:
            logger.warning(f"[STOCKOUT] No prescription products in inventory")
            return {
                "risk_items": [],
                "summary": {
                    "critical_count": 0,
                    "high_count": 0,
                    "ok_count": 0,
                    "medium_count": 0,
                    "excess_count": 0,
                    "total_products": 0,
                    "message": "No hay productos de prescripción en inventario",
                },
                "period": {},
                "inventory_date": latest_snapshot_date.isoformat(),
            }

        # 3. Calcular ventas promedio diarias (últimos 30 días) por producto
        thirty_days_ago = current_date - timedelta(days=30)

        # Query de ventas por codigo nacional
        product_codes = [item.product_code for item in inventory_items if item.product_code]

        avg_sales_query = db.query(
            SalesData.codigo_nacional,
            func.sum(SalesData.quantity).label("total_units"),
            func.count(func.distinct(SalesData.sale_date)).label("days_with_sales"),
        ).filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesData.sale_date >= thirty_days_ago,
            SalesData.codigo_nacional.in_(product_codes),
        ).group_by(
            SalesData.codigo_nacional
        ).all()

        # Crear dict de ventas promedio
        avg_sales_map = {}
        for row in avg_sales_query:
            if row.total_units and row.days_with_sales:
                # Promedio diario = total / 30 días
                avg_sales_map[row.codigo_nacional] = float(row.total_units) / 30.0

        # 4. Obtener índice de estacionalidad del mes actual
        # Calcular índice mensual para el mes actual usando datos históricos
        monthly_index = self._get_current_month_seasonality_index(
            db, pharmacy_id, current_month, atc_code=atc_code
        )

        # 5. Batch fetch de categorías para evitar N+1 queries
        catalog_ids = [item.product_catalog_id for item in inventory_items if item.product_catalog_id]
        category_map = {}
        if catalog_ids:
            category_results = db.query(
                ProductCatalog.id, ProductCatalog.xfarma_prescription_category
            ).filter(ProductCatalog.id.in_(catalog_ids)).all()
            category_map = {row.id: row.xfarma_prescription_category for row in category_results}

        # 6. Calcular riesgo para cada producto
        risk_items = []

        for item in inventory_items:
            if not item.product_code:
                continue

            avg_daily = avg_sales_map.get(item.product_code, 0)

            # Si no hay ventas, asumir bajo riesgo (producto nuevo o sin rotación)
            if avg_daily <= 0:
                continue

            # Ajustar ventas por estacionalidad
            adjusted_daily = avg_daily * monthly_index

            # Calcular días de cobertura
            if adjusted_daily > 0:
                days_coverage = item.stock_quantity / adjusted_daily
            else:
                days_coverage = float('inf')

            # Calcular ratio de cobertura vs período de reposición
            if days_until_restock > 0:
                coverage_ratio = days_coverage / days_until_restock
            else:
                coverage_ratio = days_coverage

            # Determinar nivel de riesgo según el algoritmo del issue
            risk_level, risk_color = self._calculate_risk_level(coverage_ratio)

            # Calcular pedido recomendado si hay riesgo
            recommended_order = 0
            if coverage_ratio < 1.0:
                # Necesitamos stock para cubrir días de reposición
                target_stock = adjusted_daily * days_until_restock * 1.2  # 20% margen
                recommended_order = max(0, int(target_stock - item.stock_quantity))

            # Obtener categoría del producto desde batch pre-cargado
            category = category_map.get(item.product_catalog_id) if item.product_catalog_id else None

            risk_items.append({
                "product_code": item.product_code,
                "product_name": item.product_name,
                "category": category,
                "current_stock": item.stock_quantity,
                "avg_daily_sales": round(avg_daily, 2),
                "seasonality_index": round(monthly_index, 2),
                "adjusted_daily_sales": round(adjusted_daily, 2),
                "days_of_coverage": round(days_coverage, 1),
                "coverage_ratio": round(coverage_ratio, 2),
                "risk_level": risk_level,
                "risk_color": risk_color,
                "recommended_order": recommended_order,
            })

        # 6. Ordenar por riesgo (más crítico primero)
        risk_order = {"CRÍTICO": 0, "ALTO": 1, "OK": 2, "MEDIO": 3, "EXCESO": 4}
        risk_items.sort(key=lambda x: (risk_order.get(x["risk_level"], 5), -x["coverage_ratio"]))

        # Limitar a top_n
        risk_items = risk_items[:top_n]

        # 7. Calcular resumen
        summary = {
            "critical_count": sum(1 for r in risk_items if r["risk_level"] == "CRÍTICO"),
            "high_count": sum(1 for r in risk_items if r["risk_level"] == "ALTO"),
            "ok_count": sum(1 for r in risk_items if r["risk_level"] == "OK"),
            "medium_count": sum(1 for r in risk_items if r["risk_level"] == "MEDIO"),
            "excess_count": sum(1 for r in risk_items if r["risk_level"] == "EXCESO"),
            "total_products": len(risk_items),
            "current_month_index": round(monthly_index, 2),
            "current_month": MONTH_NAMES.get(current_month, str(current_month)),
        }

        logger.info(
            f"[STOCKOUT] Análisis completado: {summary['critical_count']} críticos, "
            f"{summary['high_count']} altos de {summary['total_products']} productos"
        )

        return {
            "risk_items": risk_items,
            "summary": summary,
            "period": {
                "sales_from": thirty_days_ago.isoformat(),
                "sales_to": current_date.isoformat(),
            },
            "inventory_date": latest_snapshot_date.isoformat(),
        }

    def _get_current_month_seasonality_index(
        self,
        db: Session,
        pharmacy_id: UUID,
        current_month: int,
        atc_code: Optional[str] = None,
    ) -> float:
        """
        Calcula el índice de estacionalidad del mes actual.

        Usa datos históricos para determinar si el mes actual tiene
        ventas por encima o debajo del promedio anual.

        Returns:
            float: Índice de estacionalidad (1.0 = promedio)
        """
        # Obtener ventas por mes de los últimos 2 años
        # Issue #503: Usar dim_date en lugar de EXTRACT()
        two_years_ago = date.today() - timedelta(days=730)

        monthly_sales_query = db.query(
            DimDate.month,
            func.sum(SalesData.sale_price * SalesData.quantity).label('total_sales'),
        ).join(
            DimDate, func.date(SalesData.sale_date) == DimDate.date_key
        ).join(
            SalesEnrichment,
            SalesData.id == SalesEnrichment.sales_data_id
        ).join(
            ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id
        ).filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesData.sale_date >= two_years_ago,
            SalesEnrichment.product_type == 'medicamento',
        )

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            monthly_sales_query = monthly_sales_query.filter(
                ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%")
            )

        monthly_sales = monthly_sales_query.group_by(DimDate.month).all()

        if not monthly_sales:
            return 1.0  # Sin datos, asumir promedio

        # Calcular promedio mensual
        sales_by_month = {int(row.month): float(row.total_sales or 0) for row in monthly_sales}
        total_sales = sum(sales_by_month.values())

        if total_sales <= 0:
            return 1.0

        avg_monthly = total_sales / 12

        # Índice del mes actual
        current_month_sales = sales_by_month.get(current_month, avg_monthly)

        if avg_monthly > 0:
            return current_month_sales / avg_monthly

        return 1.0

    def _calculate_risk_level(self, coverage_ratio: float) -> tuple:
        """
        Calcula nivel de riesgo basado en ratio de cobertura.

        Algoritmo del Issue #500:
        - <0.5: CRÍTICO (danger)
        - 0.5-0.8: ALTO (warning)
        - 0.8-1.2: OK (success)
        - 1.2-2.0: MEDIO - exceso leve (info)
        - >2.0: EXCESO (secondary)

        Returns:
            tuple: (risk_level, risk_color)
        """
        if coverage_ratio < 0.5:
            return ("CRÍTICO", "danger")
        elif coverage_ratio < 0.8:
            return ("ALTO", "warning")
        elif coverage_ratio <= 1.2:
            return ("OK", "success")
        elif coverage_ratio <= 2.0:
            return ("MEDIO", "info")
        else:
            return ("EXCESO", "secondary")

    # =========================================================================
    # ANOMALY DETECTION (Issue #501)
    # =========================================================================

    def detect_anomalies(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: Optional[date] = None,
        date_to: Optional[date] = None,
        sensitivity: float = 2.0,
        exclude_holidays: bool = True,
        exclude_zero_stock: bool = True,
        exclude_bulk_sales_pct: float = 0.3,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Detecta anomalías estadísticas en ventas con filtros anti-falsos positivos.

        Issue #501: Fase 4 - Detector de Anomalías + Alertas.

        Algoritmo:
        1. Calcula ventas diarias
        2. Aplica filtros anti-falsos positivos
        3. Calcula Z-score sobre datos limpios
        4. Identifica anomalías (|z-score| > sensitivity)

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio (default: 90 días atrás)
            date_to: Fecha fin (default: hoy)
            sensitivity: Umbral Z-score (default: 2.0)
            exclude_holidays: Excluir festivos españoles
            exclude_zero_stock: Excluir días sin stock (roturas)
            exclude_bulk_sales_pct: Excluir ventas institucionales (>X% del diario)

        Returns:
            Dict con anomalies, summary, period, filters_applied, data_quality
        """
        # Import moved to top of file (Issue #501 review fix)

        # Fechas por defecto
        current_date = date.today()
        if not date_to:
            date_to = current_date
        if not date_from:
            date_from = date_to - timedelta(days=90)

        # 1. Obtener ventas diarias agregadas
        daily_sales_query = (
            db.query(
                SalesData.sale_date,
                func.sum(SalesData.sale_price * SalesData.quantity).label("total_sales"),
                func.sum(SalesData.quantity).label("total_units"),
                func.count(SalesData.id).label("transaction_count"),
                func.max(SalesData.sale_price * SalesData.quantity).label("max_transaction"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= date_from,
                SalesData.sale_date <= date_to,
                ProductCatalog.xfarma_prescription_category.isnot(None),
            )
        )

        # Issue #532: Filtrar por código ATC nivel 1 (normalizado a mayúsculas)
        if atc_code:
            daily_sales_query = daily_sales_query.filter(
                ProductCatalog.cima_atc_code.like(f"{atc_code.upper()}%")
            )

        daily_sales_query = daily_sales_query.group_by(SalesData.sale_date).all()

        if not daily_sales_query:
            return {
                "anomalies": [],
                "summary": {
                    "total_anomalies": 0,
                    "spikes": 0,
                    "drops": 0,
                    "high_severity_count": 0,
                },
                "period": {
                    "date_from": date_from.isoformat(),
                    "date_to": date_to.isoformat(),
                },
                "filters_applied": {
                    "sensitivity": sensitivity,
                    "exclude_holidays": exclude_holidays,
                    "exclude_zero_stock": exclude_zero_stock,
                    "exclude_bulk_sales_pct": exclude_bulk_sales_pct,
                },
                "data_quality": {
                    "days_analyzed": 0,
                    "days_excluded_holidays": 0,
                    "days_excluded_bulk": 0,
                    "insufficient_data": True,
                },
            }

        # Convertir a DataFrame para análisis
        daily_data = []
        for row in daily_sales_query:
            daily_data.append({
                "sale_date": row.sale_date,
                "total_sales": float(row.total_sales or 0),
                "total_units": int(row.total_units or 0),
                "transaction_count": int(row.transaction_count or 0),
                "max_transaction": float(row.max_transaction or 0),
            })

        df = pd.DataFrame(daily_data)
        original_count = len(df)

        # 2. Aplicar filtros anti-falsos positivos

        # 2a. Excluir festivos españoles
        days_excluded_holidays = 0
        if exclude_holidays:
            holidays = get_spanish_holidays_range(date_from, date_to)
            before_count = len(df)
            # Convertir holidays (Set[date]) a datetime64 para evitar FutureWarning
            # (df["sale_date"] es datetime64[ns], no compatible directamente con date objects)
            holidays_dt = pd.to_datetime(list(holidays))
            df = df[~df["sale_date"].isin(holidays_dt)]
            days_excluded_holidays = before_count - len(df)

        # 2b. Excluir ventas en lote (institucionales)
        days_excluded_bulk = 0
        if exclude_bulk_sales_pct > 0:
            # Si la transacción más grande es > X% del total diario, excluir
            # Mantener días con total_sales=0 (no aplica filtro de bulk)
            before_count = len(df)
            df = df[
                (df["total_sales"] == 0)
                | (df["max_transaction"] < (df["total_sales"] * exclude_bulk_sales_pct))
            ]
            days_excluded_bulk = before_count - len(df)

        # 3. Verificar datos suficientes
        if len(df) < 14:  # Mínimo 2 semanas de datos
            return {
                "anomalies": [],
                "summary": {
                    "total_anomalies": 0,
                    "spikes": 0,
                    "drops": 0,
                    "high_severity_count": 0,
                },
                "period": {
                    "date_from": date_from.isoformat(),
                    "date_to": date_to.isoformat(),
                },
                "filters_applied": {
                    "sensitivity": sensitivity,
                    "exclude_holidays": exclude_holidays,
                    "exclude_zero_stock": exclude_zero_stock,
                    "exclude_bulk_sales_pct": exclude_bulk_sales_pct,
                },
                "data_quality": {
                    "days_analyzed": len(df),
                    "days_excluded_holidays": days_excluded_holidays,
                    "days_excluded_bulk": days_excluded_bulk,
                    "insufficient_data": True,
                    "min_days_required": 14,
                },
            }

        # 4. Calcular estadísticas y Z-score
        mean_sales = df["total_sales"].mean()
        std_sales = df["total_sales"].std()

        if std_sales == 0 or pd.isna(std_sales):
            # Sin variación, no hay anomalías
            return {
                "anomalies": [],
                "summary": {
                    "total_anomalies": 0,
                    "spikes": 0,
                    "drops": 0,
                    "high_severity_count": 0,
                    "note": "No hay variación suficiente en las ventas",
                },
                "period": {
                    "date_from": date_from.isoformat(),
                    "date_to": date_to.isoformat(),
                },
                "filters_applied": {
                    "sensitivity": sensitivity,
                    "exclude_holidays": exclude_holidays,
                    "exclude_zero_stock": exclude_zero_stock,
                    "exclude_bulk_sales_pct": exclude_bulk_sales_pct,
                },
                "data_quality": {
                    "days_analyzed": len(df),
                    "days_excluded_holidays": days_excluded_holidays,
                    "days_excluded_bulk": days_excluded_bulk,
                    "mean_daily_sales": round(mean_sales, 2),
                    "std_daily_sales": 0,
                },
            }

        df["z_score"] = (df["total_sales"] - mean_sales) / std_sales

        # 5. Identificar anomalías
        anomalies_df = df[abs(df["z_score"]) > sensitivity].copy()

        # Ordenar por fecha descendente
        anomalies_df = anomalies_df.sort_values("sale_date", ascending=False)

        # 6. Construir lista de anomalías
        anomalies = []
        weekday_names = {
            0: "Lunes", 1: "Martes", 2: "Miércoles", 3: "Jueves",
            4: "Viernes", 5: "Sábado", 6: "Domingo"
        }

        for _, row in anomalies_df.iterrows():
            z_score = row["z_score"]
            observed = row["total_sales"]

            # Tipo de anomalía
            if z_score > 0:
                anomaly_type = "PICO"
            else:
                anomaly_type = "CAÍDA"

            # Desviación porcentual
            deviation_pct = ((observed - mean_sales) / mean_sales) * 100 if mean_sales > 0 else 0

            # Severidad basada en Z-score
            abs_z = abs(z_score)
            if abs_z > 3:
                severity = "ALTA"
                severity_color = "danger"
            elif abs_z > 2.5:
                severity = "MEDIA"
                severity_color = "warning"
            else:
                severity = "BAJA"
                severity_color = "info"

            # Posibles causas según tipo y contexto
            possible_causes = self._infer_anomaly_causes(
                anomaly_type, observed, mean_sales, row["sale_date"]
            )

            anomalies.append({
                "date": row["sale_date"].isoformat(),
                "weekday_name": weekday_names.get(row["sale_date"].weekday(), ""),
                "anomaly_type": anomaly_type,
                "observed_sales": round(observed, 2),
                "expected_sales": round(mean_sales, 2),
                "deviation_pct": round(deviation_pct, 1),
                "z_score": round(z_score, 2),
                "severity": severity,
                "severity_color": severity_color,
                "possible_causes": possible_causes,
            })

        # 7. Construir resumen
        spikes = len([a for a in anomalies if a["anomaly_type"] == "PICO"])
        drops = len([a for a in anomalies if a["anomaly_type"] == "CAÍDA"])
        high_severity = len([a for a in anomalies if a["severity"] == "ALTA"])

        return {
            "anomalies": anomalies,
            "summary": {
                "total_anomalies": len(anomalies),
                "spikes": spikes,
                "drops": drops,
                "high_severity_count": high_severity,
                "avg_daily_sales": round(mean_sales, 2),
                "std_daily_sales": round(std_sales, 2),
            },
            "period": {
                "date_from": date_from.isoformat(),
                "date_to": date_to.isoformat(),
            },
            "filters_applied": {
                "sensitivity": sensitivity,
                "exclude_holidays": exclude_holidays,
                "exclude_zero_stock": exclude_zero_stock,
                "exclude_bulk_sales_pct": exclude_bulk_sales_pct,
            },
            "data_quality": {
                "days_analyzed": len(df),
                "days_original": original_count,
                "days_excluded_holidays": days_excluded_holidays,
                "days_excluded_bulk": days_excluded_bulk,
            },
        }

    def _infer_anomaly_causes(
        self,
        anomaly_type: str,
        observed: float,
        expected: float,
        anomaly_date: date,
    ) -> List[str]:
        """
        Infiere posibles causas de una anomalía basándose en el contexto.

        Args:
            anomaly_type: 'PICO' o 'CAÍDA'
            observed: Ventas observadas
            expected: Ventas esperadas
            anomaly_date: Fecha de la anomalía

        Returns:
            Lista de posibles causas
        """
        causes = []

        if anomaly_type == "PICO":
            # Causas de picos
            ratio = observed / expected if expected > 0 else 0

            if ratio > 3:
                causes.append("Posible venta institucional o pedido especial")

            if anomaly_date.weekday() == 0:  # Lunes
                causes.append("Acumulación de demanda del fin de semana")

            if anomaly_date.month in [1, 2, 11, 12]:
                causes.append("Posible pico estacional (temporada alta)")

            if anomaly_date.day <= 5:
                causes.append("Inicio de mes (renovación de recetas)")

            if not causes:
                causes.append("Demanda inusualmente alta")

        else:  # CAÍDA
            if observed < expected * 0.3:
                causes.append("Posible cierre o incidencia operativa")

            if anomaly_date.weekday() in [5, 6]:  # Sábado, Domingo
                causes.append("Fin de semana con menor actividad")

            if anomaly_date.month in [7, 8]:
                causes.append("Período vacacional")

            if not causes:
                causes.append("Demanda inusualmente baja")

        return causes

    # =========================================================================
    # EXPORT FORECAST (Issue #502)
    # =========================================================================

    def export_forecast(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        periods_ahead: int = 3,
        top_categories: int = 10,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Genera datos de forecast exportables a CSV.

        Issue #502: Fase 5 - Exportación CSV + Badges.

        Combina forecast general con patrones por categoría para generar
        una tabla exportable con previsiones por categoría y mes.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del histórico
            date_to: Fecha fin del histórico
            periods_ahead: Meses a predecir
            top_categories: Número de categorías a incluir

        Returns:
            Dict con data (filas exportables), summary, export_info
        """
        logger.info(
            f"[EXPORT] Generando forecast exportable para pharmacy {pharmacy_id}, "
            f"periods_ahead={periods_ahead}, top_categories={top_categories}"
        )

        # 1. Obtener patrones por categoría
        category_patterns = self.get_category_patterns(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            top_n=top_categories,
        )

        if not category_patterns.get("category_patterns"):
            return {
                "data": [],
                "summary": {
                    "total_categories": 0,
                    "total_forecast": 0,
                    "message": "No hay datos suficientes para exportar",
                },
                "export_info": {
                    "generated_at": datetime.now().isoformat(),
                    "pharmacy_id": str(pharmacy_id),
                    "periods": periods_ahead,
                },
            }

        # 2. Obtener forecast general para escalar
        forecast_result = self.get_forecast(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            periods_ahead=periods_ahead,
            atc_code=atc_code,
        )

        # 3. Construir filas de exportación
        export_rows = []
        total_forecast = 0.0

        for pattern in category_patterns["category_patterns"]:
            category_display = pattern["category_display"]
            category_total_sales = pattern.get("total_sales", 0)

            # Calcular porcentaje de esta categoría sobre el total
            total_all_categories = sum(
                p.get("total_sales", 0)
                for p in category_patterns["category_patterns"]
            )
            category_share = (
                category_total_sales / total_all_categories
                if total_all_categories > 0
                else 0
            )

            # Generar fila por cada mes del forecast
            for forecast_point in forecast_result.get("forecast", []):
                forecast_date = datetime.fromisoformat(forecast_point["date"])
                month_num = forecast_date.month

                # Obtener índice de estacionalidad de esta categoría para este mes
                monthly_index = next(
                    (m["index"] for m in pattern.get("monthly_indices", [])
                     if m["month"] == month_num),
                    1.0
                )

                # Calcular forecast para esta categoría
                # = forecast_general * share_categoría * índice_estacional
                base_forecast = forecast_point["forecast"] * category_share
                adjusted_forecast = base_forecast * monthly_index
                lower = forecast_point["lower_bound"] * category_share * monthly_index
                upper = forecast_point["upper_bound"] * category_share * monthly_index

                export_rows.append({
                    "category": category_display,
                    "month": MONTH_NAMES.get(month_num, f"Mes {month_num}"),
                    "month_num": month_num,
                    "forecast": round(adjusted_forecast, 2),
                    "lower_bound": round(lower, 2),
                    "upper_bound": round(upper, 2),
                    "seasonality_index": round(monthly_index, 2),
                })

                total_forecast += adjusted_forecast

        # 4. Ordenar por categoría y mes
        export_rows.sort(key=lambda x: (x["category"], x["month_num"]))

        logger.info(
            f"[EXPORT] Exportación generada: {len(export_rows)} filas, "
            f"total forecast: {total_forecast:.2f}€"
        )

        return {
            "data": export_rows,
            "summary": {
                "total_categories": len(category_patterns["category_patterns"]),
                "total_forecast": round(total_forecast, 2),
                "periods": periods_ahead,
            },
            "export_info": {
                "generated_at": datetime.now().isoformat(),
                "pharmacy_id": str(pharmacy_id),
                "date_from": date_from.isoformat(),
                "date_to": date_to.isoformat(),
                "periods_ahead": periods_ahead,
            },
        }

    # =========================================================================
    # TAB BADGES (Issue #502)
    # =========================================================================

    def get_badges(
        self,
        db: Session,
        pharmacy_id: UUID,
        atc_code: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Obtiene badges para el header del tab de estacionalidad.

        Issue #502: Fase 5 - Badges visuales en tab header.

        Muestra información rápida:
        - Número de anomalías en últimos 30 días
        - Próximo pico estacional
        - Alertas de stockout

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia

        Returns:
            Dict con anomalies_count, next_peak_label, stockout_alerts y colores
        """
        logger.info(f"[BADGES] Calculando badges para pharmacy {pharmacy_id}")

        current_date = date.today()

        # 1. Contar anomalías últimos 30 días
        anomalies_result = self.detect_anomalies(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=current_date - timedelta(days=30),
            date_to=current_date,
            atc_code=atc_code,
        )
        anomalies_count = anomalies_result.get("summary", {}).get("total_anomalies", 0)
        high_severity = anomalies_result.get("summary", {}).get("high_severity_count", 0)

        # Color de badge según severidad
        if high_severity > 0:
            anomalies_badge_color = "danger"
        elif anomalies_count > 0:
            anomalies_badge_color = "warning"
        else:
            anomalies_badge_color = "secondary"

        # 2. Calcular próximo pico estacional
        kpis = self.get_seasonality_kpis(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date(current_date.year - 1, 1, 1),
            date_to=current_date,
            atc_code=atc_code,
        )

        next_peak_category = kpis.get("next_peak_category")
        days_until_peak = kpis.get("days_until_peak", 0)

        if next_peak_category and days_until_peak > 0:
            if days_until_peak <= 30:
                next_peak_label = f"¡{next_peak_category} en {days_until_peak}d!"
                next_peak_badge_color = "warning"
            elif days_until_peak <= 60:
                next_peak_label = f"{next_peak_category} en {days_until_peak}d"
                next_peak_badge_color = "info"
            else:
                next_peak_label = f"{next_peak_category} ({kpis.get('next_peak_months', '')})"
                next_peak_badge_color = "secondary"
        else:
            next_peak_label = None
            next_peak_badge_color = "secondary"

        # 3. Contar alertas de stockout
        try:
            stockout_result = self.get_stockout_risk(
                db=db,
                pharmacy_id=pharmacy_id,
                days_until_restock=7,
                top_n=100,
                atc_code=atc_code,
            )
            critical_count = stockout_result.get("summary", {}).get("critical_count", 0)
            high_count = stockout_result.get("summary", {}).get("high_count", 0)
            stockout_alerts = critical_count + high_count

            if critical_count > 0:
                stockout_badge_color = "danger"
            elif high_count > 0:
                stockout_badge_color = "warning"
            else:
                stockout_badge_color = "secondary"

        except Exception as e:
            logger.warning(f"[BADGES] Error obteniendo stockout: {e}")
            stockout_alerts = 0
            stockout_badge_color = "secondary"

        return {
            "anomalies_count": anomalies_count,
            "anomalies_badge_color": anomalies_badge_color,
            "next_peak_label": next_peak_label,
            "next_peak_badge_color": next_peak_badge_color,
            "stockout_alerts": stockout_alerts,
            "stockout_badge_color": stockout_badge_color,
        }


# Instancia singleton del servicio
prescription_seasonality_service = PrescriptionSeasonalityService()
