# backend/app/services/prescription_reference_service.py
"""
Servicio para gestion de listados de referencia de prescripcion.

Issue #445: Carga sostenible de listados oficiales (dietas, tiras reactivas,
efectos y accesorios) via API en lugar de scripts manuales.

Flujo:
1. Admin sube Excel oficial via endpoint
2. Servicio parsea hojas y carga en prescription_reference_list
3. Opcionalmente crea entradas en ProductCatalog para joins
4. Enrichment usa estos listados automaticamente durante procesamiento

Este servicio reemplaza el script load_prescription_references.py
para un flujo mas sostenible y trazable.
"""
import logging
import time
from datetime import datetime, timezone
from io import BytesIO
from typing import Dict, List, Optional, Tuple

import pandas as pd
from sqlalchemy import func
from sqlalchemy.orm import Session

from app.models.enums import PrescriptionCategory
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.models.product_catalog import ProductCatalog

logger = logging.getLogger(__name__)


class PrescriptionReferenceService:
    """
    Servicio para cargar y gestionar listados de referencia de prescripcion.

    Soporta:
    - Carga de Excel oficial con multiples hojas
    - Truncado previo opcional (para actualizaciones completas)
    - Creacion automatica de entradas en ProductCatalog
    - Estadisticas de cobertura
    """

    # Prefijos para clasificar efectos y accesorios
    ORTOPEDIA_PREFIXES = ['MEDIA', 'MUNEQUERA', 'MUSLERA', 'RODILLERA', 'TOBILLERA', 'MUÑEQUERA']

    def __init__(self, db: Session):
        self.db = db
        self._catalog_entries_created = 0
        self._warnings: List[str] = []

    def load_from_excel(
        self,
        file_content: bytes,
        truncate_before: bool = False
    ) -> Dict:
        """
        Cargar listados de referencia desde archivo Excel.

        Args:
            file_content: Contenido del archivo Excel en bytes
            truncate_before: Si True, elimina registros existentes antes de cargar

        Returns:
            Dict con estadisticas de carga

        Raises:
            ValueError: Si el archivo no tiene las hojas esperadas
        """
        start_time = time.time()
        self._catalog_entries_created = 0
        self._warnings = []

        # Verificar estado previo
        previous_count = self.db.query(func.count(PrescriptionReferenceList.id)).scalar() or 0

        # Truncar si se solicita
        if truncate_before and previous_count > 0:
            deleted = self.db.query(PrescriptionReferenceList).delete()
            self.db.commit()
            logger.info(f"[PRESCRIPTION_REFERENCE] Truncated {deleted} existing records")

        # Leer Excel
        try:
            excel_file = pd.ExcelFile(BytesIO(file_content))
        except Exception as e:
            raise ValueError(f"Error leyendo archivo Excel: {str(e)}")

        # Validar hojas esperadas
        expected_sheets = ['Dietas', 'Tiras reactivas', 'Efectos y accesorios']
        missing_sheets = [s for s in expected_sheets if s not in excel_file.sheet_names]
        if missing_sheets:
            self._warnings.append(f"Hojas no encontradas: {missing_sheets}")

        # Cargar cada tipo de listado
        counts = {
            'dietas': 0,
            'tiras': 0,
            'incontinencia': 0,
            'ortopedia': 0,
            'efectos': 0,
        }

        # 1. Dietas (DIETOTERAPICOS)
        if 'Dietas' in excel_file.sheet_names:
            df_dietas = pd.read_excel(excel_file, sheet_name='Dietas')
            counts['dietas'] = self._load_dietas(df_dietas)
            logger.info(f"[PRESCRIPTION_REFERENCE] Loaded {counts['dietas']} dietas")

        # 2. Tiras reactivas (TIRAS_REACTIVAS_GLUCOSA)
        if 'Tiras reactivas' in excel_file.sheet_names:
            df_tiras = pd.read_excel(excel_file, sheet_name='Tiras reactivas')
            counts['tiras'] = self._load_tiras_reactivas(df_tiras)
            logger.info(f"[PRESCRIPTION_REFERENCE] Loaded {counts['tiras']} tiras reactivas")

        # 3. Efectos y accesorios (INCONTINENCIA, ORTOPEDIA, EFECTOS)
        if 'Efectos y accesorios' in excel_file.sheet_names:
            df_efectos = pd.read_excel(excel_file, sheet_name='Efectos y accesorios')
            efectos_counts = self._load_efectos_accesorios(df_efectos)
            counts['incontinencia'] = efectos_counts['incontinencia']
            counts['ortopedia'] = efectos_counts['ortopedia']
            counts['efectos'] = efectos_counts['efectos']
            logger.info(
                f"[PRESCRIPTION_REFERENCE] Loaded efectos: "
                f"incontinencia={counts['incontinencia']}, "
                f"ortopedia={counts['ortopedia']}, "
                f"efectos={counts['efectos']}"
            )

        # 4. Codigos especiales (formulas magistrales, vacunas)
        special_count = self._add_special_codes()
        logger.info(f"[PRESCRIPTION_REFERENCE] Added {special_count} special codes")

        # Commit
        self.db.commit()

        execution_time = time.time() - start_time
        total_loaded = sum(counts.values()) + special_count

        logger.info(
            f"[PRESCRIPTION_REFERENCE] Load completed",
            extra={
                'total_loaded': total_loaded,
                'catalog_entries_created': self._catalog_entries_created,
                'execution_time_seconds': execution_time,
                'truncated': truncate_before,
                'previous_count': previous_count
            }
        )

        return {
            'total_loaded': total_loaded,
            'dietas_count': counts['dietas'],
            'tiras_count': counts['tiras'],
            'incontinencia_count': counts['incontinencia'],
            'ortopedia_count': counts['ortopedia'],
            'efectos_count': counts['efectos'],
            'special_codes_count': special_count,
            'catalog_entries_created': self._catalog_entries_created,
            'truncated_before': truncate_before,
            'previous_count': previous_count,
            'execution_time_seconds': round(execution_time, 2),
            'warnings': self._warnings
        }

    def _load_dietas(self, df: pd.DataFrame) -> int:
        """Cargar productos dieteticos."""
        count = 0
        category = PrescriptionCategory.DIETOTERAPICOS

        # Buscar columna de descripcion (puede tener espacio al final)
        desc_col = self._find_column(df, ['DESCRIPCIÓN ', 'DESCRIPCION ', 'DESCRIPCIÓN', 'DESCRIPCION'])

        for _, row in df.iterrows():
            try:
                cn_value = row.get('CN')
                if pd.isna(cn_value):
                    continue

                national_code = str(int(cn_value) if isinstance(cn_value, float) else cn_value).strip()
                product_name = str(row.get(desc_col, ''))[:500] if desc_col else ''

                if not national_code or not product_name:
                    continue

                ref = PrescriptionReferenceList(
                    national_code=national_code,
                    product_name=product_name,
                    product_description_long=str(row.get('descripcion2', ''))[:500] if pd.notna(row.get('descripcion2')) else None,
                    category=category,
                    reference_source='Dietas',
                    pf_code=str(row.get('PF')) if pd.notna(row.get('PF')) else None,
                    pvp_iva=float(row.get('PMF')) if pd.notna(row.get('PMF')) else None,
                    status=str(row.get('situacion ', row.get('situacion'))) if pd.notna(row.get('situacion ', row.get('situacion'))) else None
                )
                self.db.add(ref)

                # Crear entrada en ProductCatalog si no existe
                self._ensure_product_catalog(national_code, product_name, category, 'Dietas')

                count += 1
            except Exception as e:
                self._warnings.append(f"Error en dieta row: {e}")

        return count

    def _load_tiras_reactivas(self, df: pd.DataFrame) -> int:
        """Cargar tiras reactivas de glucosa."""
        count = 0
        category = PrescriptionCategory.TIRAS_REACTIVAS_GLUCOSA

        desc_col = self._find_column(df, ['DESCRIPCIÓN ', 'DESCRIPCION ', 'DESCRIPCIÓN', 'DESCRIPCION'])

        for _, row in df.iterrows():
            try:
                cn_value = row.get('CN')
                if pd.isna(cn_value):
                    continue

                national_code = str(int(cn_value) if isinstance(cn_value, float) else cn_value).strip()
                product_name = str(row.get(desc_col, ''))[:500] if desc_col else ''

                if not national_code or not product_name:
                    continue

                ref = PrescriptionReferenceList(
                    national_code=national_code,
                    product_name=product_name,
                    category=category,
                    reference_source='Tiras reactivas',
                    pf_code=str(row.get('PF')) if pd.notna(row.get('PF')) else None,
                    pvp_iva=float(row.get('PMF')) if pd.notna(row.get('PMF')) else None,
                )
                self.db.add(ref)

                # Crear entrada en ProductCatalog si no existe
                self._ensure_product_catalog(national_code, product_name, category, 'Tiras reactivas')

                count += 1
            except Exception as e:
                self._warnings.append(f"Error en tira reactiva row: {e}")

        return count

    def _load_efectos_accesorios(self, df: pd.DataFrame) -> Dict[str, int]:
        """Cargar efectos y accesorios (incontinencia, ortopedia, otros)."""
        counts = {'incontinencia': 0, 'ortopedia': 0, 'efectos': 0}

        desc_col = self._find_column(df, ['DESCRIPCIÓN ', 'DESCRIPCION ', 'DESCRIPCIÓN', 'DESCRIPCION'])

        for _, row in df.iterrows():
            try:
                cn_value = row.get('CN')
                if pd.isna(cn_value):
                    continue

                national_code = str(int(cn_value) if isinstance(cn_value, float) else cn_value).strip()
                product_name = str(row.get(desc_col, ''))[:500] if desc_col else ''

                if not national_code or not product_name:
                    continue

                # Columna 5 (Unnamed: 4) contiene el tipo de producto
                col5 = str(row.get('Unnamed: 4', ''))

                # Clasificar por columna 5
                if col5.startswith('ABSORB INC'):
                    category_enum = PrescriptionCategory.INCONTINENCIA_FINANCIADA
                    counts['incontinencia'] += 1
                elif any(col5.upper().startswith(prefix) for prefix in self.ORTOPEDIA_PREFIXES):
                    category_enum = PrescriptionCategory.ORTOPEDIA_FINANCIADA
                    counts['ortopedia'] += 1
                else:
                    category_enum = PrescriptionCategory.EFECTOS_FINANCIADOS
                    counts['efectos'] += 1

                ref = PrescriptionReferenceList(
                    national_code=national_code,
                    product_name=product_name,
                    product_description_long=col5[:500] if col5 else None,
                    category=category_enum,
                    reference_source='Efectos y accesorios',
                    pf_code=str(row.get('PF')) if pd.notna(row.get('PF')) else None,
                    pvp_iva=float(row.get('PVP + iva')) if pd.notna(row.get('PVP + iva')) else None,
                    status=str(row.get('situación', row.get('situacion'))) if pd.notna(row.get('situación', row.get('situacion'))) else None
                )
                self.db.add(ref)

                # Crear entrada en ProductCatalog si no existe
                self._ensure_product_catalog(national_code, product_name, category_enum, 'Efectos y accesorios')

            except Exception as e:
                self._warnings.append(f"Error en efecto/accesorio row: {e}")

        return counts

    def _add_special_codes(self) -> int:
        """Agregar codigos especiales (formulas magistrales, vacunas)."""
        special_codes = [
            {
                'national_code': '500017',
                'product_name': 'FORMULAS MAGISTRALES',
                'category': PrescriptionCategory.FORMULAS_MAGISTRALES,
                'reference_source': 'Codigos especiales'
            },
            {
                'national_code': '500009',
                'product_name': 'EXTRACTOS HIPOSENSIBILIZANTES Y VACUNAS BACTERIANAS',
                'category': PrescriptionCategory.VACUNAS_INDIVIDUALIZADAS,
                'reference_source': 'Codigos especiales'
            }
        ]

        count = 0
        for code_info in special_codes:
            # Verificar si ya existe
            existing = self.db.query(PrescriptionReferenceList).filter(
                PrescriptionReferenceList.national_code == code_info['national_code'],
                PrescriptionReferenceList.category == code_info['category']
            ).first()

            if not existing:
                ref = PrescriptionReferenceList(
                    national_code=code_info['national_code'],
                    product_name=code_info['product_name'],
                    category=code_info['category'],
                    reference_source=code_info['reference_source']
                )
                self.db.add(ref)
                count += 1

        return count

    def _ensure_product_catalog(
        self,
        national_code: str,
        product_name: str,
        category: PrescriptionCategory,
        reference_source: str
    ) -> None:
        """
        Crear entrada en ProductCatalog si no existe.

        Issue #449: Los productos de prescription_reference_list necesitan existir en
        ProductCatalog para aparecer en analytics (joins con SalesEnrichment).
        """
        existing = self.db.query(ProductCatalog).filter(
            ProductCatalog.national_code == national_code
        ).first()

        if not existing:
            product = ProductCatalog(
                national_code=national_code,
                cima_nombre_comercial=product_name,
                nomen_nombre=product_name,
                xfarma_prescription_category=category,
                data_sources=f"prescription_reference_list:{reference_source}",
            )
            self.db.add(product)
            self._catalog_entries_created += 1

    def _find_column(self, df: pd.DataFrame, candidates: List[str]) -> Optional[str]:
        """Encontrar columna por nombre (maneja variaciones de espacios/acentos)."""
        for col in candidates:
            if col in df.columns:
                return col
        return None

    def get_stats(self) -> Dict:
        """
        Obtener estadisticas de listados de referencia cargados.

        Returns:
            Dict con estadisticas de cobertura
        """
        # Total de entradas
        total_entries = self.db.query(func.count(PrescriptionReferenceList.id)).scalar() or 0

        if total_entries == 0:
            return {
                'total_entries': 0,
                'category_breakdown': {},
                'source_breakdown': {},
                'last_loaded_at': None,
                'catalog_coverage': 0
            }

        # Desglose por categoria
        category_counts = self.db.query(
            PrescriptionReferenceList.category,
            func.count(PrescriptionReferenceList.id)
        ).group_by(PrescriptionReferenceList.category).all()

        category_breakdown = {
            str(cat.value if hasattr(cat, 'value') else cat): count
            for cat, count in category_counts
        }

        # Desglose por fuente
        source_counts = self.db.query(
            PrescriptionReferenceList.reference_source,
            func.count(PrescriptionReferenceList.id)
        ).group_by(PrescriptionReferenceList.reference_source).all()

        source_breakdown = {src: count for src, count in source_counts}

        # Ultima carga
        last_loaded = self.db.query(
            func.max(PrescriptionReferenceList.loaded_at)
        ).scalar()

        # Cobertura en ProductCatalog
        reference_codes = self.db.query(PrescriptionReferenceList.national_code).distinct().subquery()
        catalog_coverage = self.db.query(func.count(ProductCatalog.id)).filter(
            ProductCatalog.national_code.in_(
                self.db.query(reference_codes)
            )
        ).scalar() or 0

        return {
            'total_entries': total_entries,
            'category_breakdown': category_breakdown,
            'source_breakdown': source_breakdown,
            'last_loaded_at': last_loaded,
            'catalog_coverage': catalog_coverage
        }
