# backend/app/services/prescription_classification_service.py
"""
Servicio de Clasificación de Productos de Prescripción (Issue #16 - Fase 1)

Motor de clasificación de productos farmacéuticos en 15 categorías según:
- Nomenclator oficial del Ministerio de Sanidad
- Base de datos CIMA de la AEMPS
- Reglas de negocio farmacéuticas españolas

Este servicio implementa la lógica de clasificación jerárquica prioritaria
para asignar productos a categorías de prescripción basándose en:
- Estado de comercialización (NO_COMERCIALIZADO - Issue #432)
- Códigos especiales (500017, 500009)
- Características del producto (nombre, tipo, PVP)
- Datos regulatorios (receta, veterinaria, financiación)
"""

import logging
from decimal import Decimal
from typing import Any, Dict, List, Optional

from sqlalchemy.orm import Session

from app.models.enums import PrescriptionCategory
from app.models.product_catalog import ProductCatalog
from app.models.prescription_reference_list import PrescriptionReferenceList

# Configuración de logging
logger = logging.getLogger(__name__)


class PrescriptionClassificationService:
    """
    Servicio de clasificación de productos de prescripción farmacéutica

    Proporciona métodos para:
    - Determinar si un producto es de prescripción
    - Clasificar productos en 14 categorías según reglas de negocio
    - Clasificación masiva con estadísticas detalladas
    """

    def __init__(
        self,
        db: Optional[Session] = None,
        reference_map: Optional[Dict[str, PrescriptionCategory]] = None
    ):
        """
        Inicializa el servicio de clasificación

        Args:
            db: Sesión SQLAlchemy opcional para consultas DB
            reference_map: Mapa opcional de código nacional → categoría para evitar N+1 queries.
                          Si se proporciona, se usa en lugar de consultas individuales a DB.
                          Formato: {"123456": PrescriptionCategory.TIRAS_REACTIVAS_GLUCOSA, ...}

        Performance:
            Sin reference_map: 1 query por producto (N+1 problem)
            Con reference_map: 0 queries adicionales (lookup O(1) en memoria)
        """
        self.db = db
        self.reference_map = reference_map

    def _lookup_reference_list(self, national_code: str) -> Optional[PrescriptionCategory]:
        """
        Consulta la tabla prescription_reference_list para obtener clasificación oficial

        Esta tabla contiene listados oficiales del Ministerio de Sanidad:
        - Dietoterapéuticos
        - Tiras reactivas glucosa
        - Incontinencia financiada
        - Ortopedia financiada
        - Efectos financiados
        - Fórmulas magistrales (500017)
        - Vacunas individualizadas (500009)

        Args:
            national_code: Código nacional del producto

        Returns:
            PrescriptionCategory si existe en listado oficial, None si no

        Note:
            Si self.reference_map está disponible, usa lookup O(1) en memoria.
            De lo contrario, consulta DB (requiere self.db inicializado).

        Performance:
            Con reference_map: O(1) dict lookup (~0.001ms)
            Sin reference_map: O(1) DB query indexed (~5ms)
        """
        if not national_code:
            logger.warning(f"[REFERENCE_LOOKUP] national_code es None/empty")
            return None

        # OPTIMIZACIÓN: Usar reference_map si está disponible (evita N+1 queries)
        if self.reference_map is not None:
            category = self.reference_map.get(national_code)
            if category:
                logger.debug(
                    f"[REFERENCE_LOOKUP] Producto {national_code} encontrado en map: {category}"
                )
            return category

        # FALLBACK: Consultar DB (modo legacy o single product lookups)
        if not self.db:
            logger.warning(f"[REFERENCE_LOOKUP] self.db es None - no se puede consultar")
            return None

        logger.info(f"[REFERENCE_LOOKUP] Consultando código en DB: {national_code}")

        try:
            reference = (
                self.db.query(PrescriptionReferenceList)
                .filter(PrescriptionReferenceList.national_code == national_code)
                .first()
            )

            if reference:
                logger.debug(
                    f"[REFERENCE_LOOKUP] Producto {national_code} encontrado en listado oficial: "
                    f"{reference.category} (fuente: {reference.reference_source})"
                )
                return PrescriptionCategory(reference.category)

            return None

        except Exception as e:
            logger.warning(
                f"[REFERENCE_LOOKUP] Error consultando listado de referencia para {national_code}: {e}"
            )
            return None

    def _has_generic_in_homogeneous_group(self, codigo_homogeneo: str) -> bool:
        """
        Verifica si un conjunto homogéneo tiene al menos un producto genérico COMERCIALIZADO.

        Un conjunto homogéneo "real" debe tener al menos un genérico (EFG/GENERICO)
        que esté comercializado (nomen_estado = 'ALTA').
        Si todos los genéricos están de baja o no hay genéricos, no se considera
        un conjunto homogéneo a efectos de clasificación de prescripción.

        Args:
            codigo_homogeneo: Código del conjunto homogéneo

        Returns:
            True si hay al menos un genérico comercializado, False en caso contrario

        Note:
            Requiere self.db inicializado para consultar la base de datos.
        """
        if not codigo_homogeneo or not self.db:
            return False

        try:
            # Contar productos genéricos COMERCIALIZADOS en el conjunto homogéneo
            generic_count = (
                self.db.query(ProductCatalog)
                .filter(
                    ProductCatalog.nomen_codigo_homogeneo == codigo_homogeneo,
                    ProductCatalog.nomen_tipo_farmaco.in_(["EFG", "GENERICO"]),
                    ProductCatalog.nomen_estado == "ALTA"  # Solo comercializados
                )
                .count()
            )
            return generic_count > 0
        except Exception as e:
            logger.warning(
                f"[HOMOGENEOUS_CHECK] Error verificando genéricos en conjunto {codigo_homogeneo}: {e}"
            )
            # En caso de error, asumir que sí hay genéricos (comportamiento conservador)
            return True

    def is_prescription_product(self, product: ProductCatalog) -> bool:
        """
        Determina si un producto es de prescripción farmacéutica

        Un producto es considerado de prescripción si cumple ALGUNO de los siguientes criterios:
        1. Está en el Nomenclator (tiene código homogéneo)
        2. Requiere receta según CIMA
        3. Tiene código especial (500017, 500009)
        4. Está en listados oficiales (prescription_reference_list) - NUEVO
        5. Es producto financiado con nombre específico (tiras, incontinencia, efectos, ortopedia)

        Args:
            product: Instancia de ProductCatalog a evaluar

        Returns:
            True si el producto es de prescripción, False en caso contrario

        Ejemplos:
            >>> product = ProductCatalog(nomen_codigo_homogeneo="800585")
            >>> service.is_prescription_product(product)
            True

            >>> product = ProductCatalog(cima_requiere_receta=True)
            >>> service.is_prescription_product(product)
            True

            >>> product = ProductCatalog(cima_requiere_receta=False, nomen_codigo_homogeneo=None)
            >>> service.is_prescription_product(product)
            False
        """
        # Criterio 1: Código especial (siempre prescripción)
        if product.national_code in ["500017", "500009"]:
            return True

        # Criterio 2: Producto en Nomenclator (tiene conjunto homogéneo)
        if product.nomen_codigo_homogeneo is not None:
            return True

        # Criterio 3: Requiere receta según CIMA
        if product.cima_requiere_receta is True:
            return True

        # Criterio 4: Producto en listados oficiales (prescription_reference_list)
        # Incluye dietoterapéuticos, tiras reactivas, incontinencia, ortopedia, efectos
        # que pueden NO tener receta ni nomenclator pero SÍ son de prescripción
        if self._lookup_reference_list(product.national_code) is not None:
            return True

        # Criterio 5: Productos financiados por nombre (aunque no requieran receta)
        nombre_lower = (product.nomen_nombre or "").lower()
        keywords_financiados = [
            "tiras reactivas",
            "tiras glucemia",
            "incontinencia",
            "sonda",
            "pesario",
            "braguero",
            "ortopedia",
        ]

        if any(keyword in nombre_lower for keyword in keywords_financiados):
            return True

        return False

    def classify_product(self, product: ProductCatalog) -> Optional[PrescriptionCategory]:
        """
        Clasifica un producto según las reglas de negocio farmacéuticas

        Evaluación por orden de prioridad:
        1. FORMULAS_MAGISTRALES: CN = 500017 (código especial)
        2. VACUNAS_INDIVIDUALIZADAS: CN = 500009 (código especial)
        3. NO_COMERCIALIZADO: cima_estado_registro="NO_COMERCIALIZADO" (Issue #432)
        ** Consulta a prescription_reference_list **
           - Listados oficiales del Ministerio de Sanidad
           - Incluye: Dietoterapéuticos, Tiras reactivas, Incontinencia, Ortopedia, Efectos
           - Prevalece sobre reglas heurísticas
        4. TIRAS_REACTIVAS_GLUCOSA: nombre contiene "tiras reactivas" o "tiras glucemia" (heurística)
        5. INCONTINENCIA_FINANCIADA: sin tipo_farmaco Y nombre contiene "incontinencia" (heurística)
        6. EFECTOS_FINANCIADOS: sin tipo_farmaco Y (sonda/pesario/braguero) (heurística)
        7. ORTOPEDIA_FINANCIADA: sin tipo_farmaco Y nombre contiene "ortopedia" (heurística)
        8. DIETOTERAPICOS: grupo terapéutico LIKE 'A09%' (heurística)
        9. CONJUNTO_HOMOGENEO: tiene código homogéneo CON genéricos COMERCIALIZADOS
        10. MARGEN_ESPECIAL_3: PVL > 500€ (PVP ≥ 727,66€, solo NO sustituibles)
        11. MARGEN_ESPECIAL_2: 200€ < PVL ≤ 500€ (PVP 287,21€-727,66€, solo NO sustituibles)
        12. MARGEN_ESPECIAL_1: 91,63€ < PVL ≤ 200€ (PVP 126,12€-287,21€, solo NO sustituibles)
        13. VETERINARIA: uso veterinario CON receta veterinaria (Issue #354)
        14. MEDICAMENTOS: con receta + en nomenclator
        15. USO_HUMANO_NO_FINANCIADO: con receta + sin nomenclator (fallback final)

        NOTA VETERINARIA: Solo se clasifican como VETERINARIA los productos con
        cima_uso_veterinario=True Y cima_requiere_receta=True. Productos veterinarios
        de venta libre (cima_requiere_receta=False) no son productos de prescripción.

        Args:
            product: Instancia de ProductCatalog a clasificar

        Returns:
            PrescriptionCategory correspondiente o None si no es producto de prescripción

        Ejemplos:
            >>> product = ProductCatalog(national_code="500017", cima_requiere_receta=True)
            >>> service.classify_product(product)
            PrescriptionCategory.FORMULAS_MAGISTRALES

            >>> product = ProductCatalog(cima_estado_registro="NO_COMERCIALIZADO", cima_requiere_receta=True)
            >>> service.classify_product(product)
            PrescriptionCategory.NO_COMERCIALIZADO

            >>> product = ProductCatalog(cima_requiere_receta=False, nomen_codigo_homogeneo=None)
            >>> service.classify_product(product)
            None  # No es producto de prescripción
        """
        # PASO 0: Verificar elegibilidad como producto de prescripción
        if not self.is_prescription_product(product):
            return None

        # ===================================================================
        # FASE 1: CÓDIGOS ESPECIALES (Prioridad 1-2)
        # ===================================================================

        # REGLA 1: Fórmulas magistrales (máxima prioridad)
        if product.national_code == "500017":
            return PrescriptionCategory.FORMULAS_MAGISTRALES

        # REGLA 2: Vacunas individualizadas
        if product.national_code == "500009":
            return PrescriptionCategory.VACUNAS_INDIVIDUALIZADAS

        # ===================================================================
        # FASE 1.3: PRODUCTOS NO COMERCIALIZADOS (Prioridad 3 - Issue #432)
        # ===================================================================
        # Productos con cima_estado_registro="NO_COMERCIALIZADO" se clasifican
        # aquí antes de otras reglas funcionales porque no están disponibles
        # en el mercado y no deberían clasificarse como CONJUNTO_HOMOGENEO,
        # MEDICAMENTOS, etc.

        if product.cima_estado_registro == "NO_COMERCIALIZADO":
            return PrescriptionCategory.NO_COMERCIALIZADO

        # ===================================================================
        # FASE 1.5: LISTADOS OFICIALES DE REFERENCIA (Prioridad inmediata)
        # ===================================================================
        # Consulta tabla prescription_reference_list con listados del Ministerio
        # (Dietoterapéuticos, Tiras reactivas, Incontinencia, Ortopedia, Efectos)
        # Los listados oficiales tienen prioridad sobre reglas heurísticas

        if product.national_code:
            reference_category = self._lookup_reference_list(product.national_code)
            if reference_category:
                return reference_category

        # ===================================================================
        # FASE 2: CLASIFICACIÓN POR NOMBRE (Prioridad 3-6)
        # ===================================================================

        # Helper: normalizar nombre para búsquedas case-insensitive
        nombre_lower = (product.nomen_nombre or "").lower()

        # REGLA 3: Tiras reactivas glucosa
        if "tiras reactivas" in nombre_lower or "tiras glucemia" in nombre_lower:
            return PrescriptionCategory.TIRAS_REACTIVAS_GLUCOSA

        # REGLAS 4-6: Productos financiados SIN tipo_farmaco
        if product.nomen_tipo_farmaco is None:
            # REGLA 4: Incontinencia financiada
            if "incontinencia" in nombre_lower:
                return PrescriptionCategory.INCONTINENCIA_FINANCIADA

            # REGLA 5: Efectos financiados (sonda, pesario, braguero)
            if any(keyword in nombre_lower for keyword in ["sonda", "pesario", "braguero"]):
                return PrescriptionCategory.EFECTOS_FINANCIADOS

            # REGLA 6: Ortopedia financiada
            if "ortopedia" in nombre_lower:
                return PrescriptionCategory.ORTOPEDIA_FINANCIADA

        # ===================================================================
        # FASE 3: CLASIFICACIÓN POR GRUPO TERAPÉUTICO (Prioridad 7)
        # ===================================================================

        # REGLA 7: Dietoterápicos (grupo A09%)
        if product.cima_grupo_terapeutico:
            if product.cima_grupo_terapeutico.startswith("A09"):
                return PrescriptionCategory.DIETOTERAPICOS

        # ===================================================================
        # FASE 4: CONJUNTO HOMOGÉNEO (Prioridad 8)
        # ===================================================================

        # REGLA 8: Conjunto homogéneo (sustituibles - ANTES de márgenes especiales)
        # Prioridad alta: productos sustituibles se clasifican aquí independiente del PVP
        # EXCEPCIÓN: Si todos los productos del conjunto son MARCA (sin genéricos),
        # no se considera conjunto homogéneo y continúa evaluación normal
        if product.nomen_codigo_homogeneo is not None:
            if self._has_generic_in_homogeneous_group(product.nomen_codigo_homogeneo):
                return PrescriptionCategory.CONJUNTO_HOMOGENEO
            # Si no hay genéricos, continuar con las siguientes reglas
            logger.debug(
                f"[CLASSIFICATION] Producto {product.national_code} tiene código homogéneo "
                f"{product.nomen_codigo_homogeneo} pero sin genéricos - no es CONJUNTO_HOMOGENEO"
            )

        # ===================================================================
        # FASE 4.5: VETERINARIA (Prioridad 8) - Issue #354
        # ===================================================================
        # Productos veterinarios de prescripción (requieren receta veterinaria)
        # Datos provienen de CIMAVet (base de datos separada de CIMA humano)
        # Un producto veterinario con homogéneo sigue siendo VETERINARIA
        #
        # IMPORTANTE: cima_uso_veterinario=True se setea en enrichment desde CIMAVet
        # El campo cima_requiere_receta determina si es prescripción o venta libre:
        # - cima_requiere_receta=True → VETERINARIA (prescripción)
        # - cima_requiere_receta=False → No clasificado aquí (venta libre, no es prescripción)
        if product.cima_uso_veterinario is True and product.cima_requiere_receta is True:
            return PrescriptionCategory.VETERINARIA

        # ===================================================================
        # FASE 5: MÁRGENES ESPECIALES (Prioridad 9-11)
        # ===================================================================

        # Márgenes especiales RD 823/2008: PVL + NO sustituible
        # Productos con código homogéneo YA fueron clasificados como CONJUNTO_HOMOGENEO
        # Solo llegan aquí productos SIN código homogéneo (NO sustituibles)
        # Aplica a medicamentos financiados con prescripción (no dietoterápicos)
        if product.nomen_pvp is not None:
            pvp = Decimal(str(product.nomen_pvp))

            # REGLA 9: Margen especial 3 (PVL > 500€ → PVP ≥ 727,66€)
            if pvp >= Decimal("727.66"):
                return PrescriptionCategory.MARGEN_ESPECIAL_3

            # REGLA 10: Margen especial 2 (200€ < PVL ≤ 500€ → 287,21€ ≤ PVP < 727,66€)
            if Decimal("287.21") <= pvp < Decimal("727.66"):
                return PrescriptionCategory.MARGEN_ESPECIAL_2

            # REGLA 11: Margen especial 1 (91,63€ < PVL ≤ 200€ → 126,12€ ≤ PVP < 287,21€)
            if Decimal("126.12") <= pvp < Decimal("287.21"):
                return PrescriptionCategory.MARGEN_ESPECIAL_1

        # ===================================================================
        # FASE 6: FALLBACKS FINALES (Prioridad 12-13)
        # ===================================================================

        # REGLA 12: Medicamentos (con receta Y financiados SNS)
        # Productos con receta que están en nomenclator (financiados)
        # Incluye tanto productos con homogéneo como sin homogéneo
        # Ejemplos: Hydrea (sin homogéneo), Omeprazol (con homogéneo pero PVP bajo)
        if product.cima_requiere_receta is True and product.nomen_nombre is not None:
            return PrescriptionCategory.MEDICAMENTOS

        # REGLA 13: Uso humano no financiado (fallback final)
        # Productos con receta pero NO en nomenclator (no financiados SNS)
        # Ejemplos: Positon (CN 694141)
        if product.cima_requiere_receta is True:
            return PrescriptionCategory.USO_HUMANO_NO_FINANCIADO

        # Fallback absoluto: productos que llegaron aquí siendo de prescripción
        # según is_prescription_product() pero sin receta (ej: productos financiados sin receta)
        return PrescriptionCategory.USO_HUMANO_NO_FINANCIADO

    def bulk_classify(
        self, products: List[ProductCatalog], dry_run: bool = False
    ) -> Dict[str, Any]:
        """
        Clasifica múltiples productos en batch con estadísticas detalladas

        Procesa cada producto individualmente, clasificando solo productos de prescripción
        y generando estadísticas completas por categoría.

        Args:
            products: Lista de productos a clasificar
            dry_run: Si True, no persiste cambios en DB (solo estadísticas)

        Returns:
            Diccionario con estadísticas de clasificación:
            {
                "total_products": int,           # Total productos evaluados
                "classified_count": int,         # Productos clasificados
                "skipped_otc_count": int,        # Productos OTC (no prescripción)
                "category_breakdown": {          # Contadores por categoría
                    "FORMULAS_MAGISTRALES": int,
                    "VACUNAS_INDIVIDUALIZADAS": int,
                    ...
                },
                "dry_run": bool                  # Indicador dry_run
            }

        Ejemplos:
            >>> productos = [
            ...     ProductCatalog(national_code="500017", cima_requiere_receta=True),
            ...     ProductCatalog(cima_requiere_receta=False)  # OTC
            ... ]
            >>> resultado = service.bulk_classify(productos, dry_run=False)
            >>> resultado["total_products"]
            2
            >>> resultado["classified_count"]
            1
            >>> resultado["skipped_otc_count"]
            1
        """
        logger.info(
            f"[CLASSIFICATION] Iniciando clasificación masiva: {len(products)} productos "
            f"(dry_run={dry_run})"
        )

        # Inicializar contadores
        total_products = len(products)
        classified_count = 0
        skipped_otc_count = 0
        category_breakdown: Dict[str, int] = {}

        # Procesar cada producto
        for product in products:
            try:
                # Clasificar producto
                categoria = self.classify_product(product)

                if categoria is None:
                    # Producto no es de prescripción (OTC/parafarmacia)
                    skipped_otc_count += 1
                    continue

                # Producto clasificado exitosamente
                classified_count += 1

                # Actualizar contadores por categoría
                categoria_str = categoria.name
                category_breakdown[categoria_str] = category_breakdown.get(categoria_str, 0) + 1

                # Persistir en DB si NO es dry_run
                if not dry_run and self.db:
                    product.xfarma_prescription_category = categoria

            except Exception as e:
                # Log error pero continuar con siguiente producto (resiliencia)
                logger.warning(
                    f"[CLASSIFICATION] Error clasificando producto {product.national_code}: {e}",
                    exc_info=True,
                )
                continue

        # Commit transaccional si NO es dry_run
        if not dry_run and self.db:
            try:
                self.db.commit()
                logger.info(
                    f"[CLASSIFICATION] Clasificación persistida en DB: "
                    f"{classified_count} productos actualizados"
                )
            except Exception as e:
                logger.error(f"[CLASSIFICATION] Error al persistir clasificación: {e}", exc_info=True)
                self.db.rollback()
                raise

        # Estadísticas finales
        resultado = {
            "total_products": total_products,
            "classified_count": classified_count,
            "skipped_otc_count": skipped_otc_count,
            "category_breakdown": category_breakdown,
            "dry_run": dry_run,
        }

        logger.info(
            f"[CLASSIFICATION] Clasificación completada: "
            f"{classified_count}/{total_products} productos clasificados, "
            f"{skipped_otc_count} productos OTC omitidos"
        )

        return resultado
