# backend/app/services/prescription_category_backfill_service.py
"""
Servicio de Backfill para xfarma_prescription_category (Issue #451)

Corrige productos con receta que tienen xfarma_prescription_category = NULL
mediante clasificación automática usando PrescriptionClassificationService.

Root Cause:
- Productos creados ANTES de implementar event listeners de auto-clasificación
- Productos creados por enrichment_service sin llamar explícitamente a clasificación
- ~2,933 productos afectados (principalmente medicamentos sin financiar SNS)

Solución:
- Clasificación masiva usando PrescriptionClassificationService
- Pre-carga de prescription_reference_list para evitar N+1 queries
- Procesamiento en batches para performance
- Estadísticas detalladas de categorías asignadas
"""

import logging
from typing import Any, Dict, List, Optional

from sqlalchemy.orm import Session

from app.models.enums import PrescriptionCategory
from app.models.product_catalog import ProductCatalog
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.services.prescription_classification_service import PrescriptionClassificationService

# Configuración de logging
logger = logging.getLogger(__name__)


class PrescriptionCategoryBackfillService:
    """
    Servicio de backfill para corregir productos con categoría de prescripción NULL

    Issue #451: Medicamentos con cima_requiere_receta=TRUE pero xfarma_prescription_category=NULL

    Features:
    - Clasificación masiva optimizada con reference_map pre-cargado
    - Procesamiento en batches para evitar memory issues
    - Estadísticas detalladas por categoría
    - Modo dry_run para validación
    - Filtros flexibles (por pharmacy, por upload, por criterios específicos)
    """

    def __init__(self, db: Session):
        """
        Inicializa el servicio de backfill

        Args:
            db: Sesión SQLAlchemy para consultas y updates
        """
        self.db = db

    def _build_reference_map(self) -> Dict[str, PrescriptionCategory]:
        """
        Pre-carga prescription_reference_list en memoria para evitar N+1 queries

        Returns:
            Diccionario {national_code: PrescriptionCategory} con todos los listados oficiales

        Performance:
            1 query único vs N queries (uno por producto)
        """
        logger.info("[BACKFILL] Pre-cargando prescription_reference_list en memoria...")

        references = self.db.query(PrescriptionReferenceList).all()

        reference_map = {}
        for ref in references:
            try:
                # Convertir string a enum si es necesario
                if isinstance(ref.category, str):
                    category_enum = PrescriptionCategory(ref.category)
                else:
                    category_enum = ref.category

                reference_map[ref.national_code] = category_enum
            except ValueError as e:
                logger.warning(
                    f"[BACKFILL] Categoría inválida en reference_list para {ref.national_code}: "
                    f"{ref.category} - {e}"
                )
                continue

        logger.info(
            f"[BACKFILL] Reference map cargado: {len(reference_map)} productos en listados oficiales"
        )
        return reference_map

    def backfill_missing_categories(
        self,
        batch_size: int = 500,
        dry_run: bool = False,
        filter_criteria: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Corrige productos con xfarma_prescription_category = NULL mediante clasificación automática

        Args:
            batch_size: Tamaño de batch para procesamiento (default: 500)
            dry_run: Si True, no persiste cambios (solo estadísticas)
            filter_criteria: Filtros opcionales para limitar scope:
                - national_codes: List[str] - Códigos nacionales específicos
                - requires_prescription: bool - Solo productos con receta
                - has_nomenclator: bool - Solo productos en nomenclator
                - limit: int - Límite máximo de productos a procesar

        Returns:
            Diccionario con estadísticas de backfill:
            {
                "total_evaluated": int,
                "total_classified": int,
                "total_skipped_otc": int,
                "category_breakdown": Dict[str, int],
                "batch_count": int,
                "dry_run": bool
            }

        Examples:
            >>> # Backfill completo (todos los productos con NULL category)
            >>> service.backfill_missing_categories(dry_run=False)

            >>> # Backfill solo productos con receta
            >>> service.backfill_missing_categories(
            ...     filter_criteria={"requires_prescription": True},
            ...     dry_run=False
            ... )

            >>> # Dry run para validar clasificación
            >>> stats = service.backfill_missing_categories(dry_run=True)
            >>> print(f"Se clasificarían {stats['total_classified']} productos")
        """
        logger.info(
            f"[BACKFILL] Iniciando backfill de xfarma_prescription_category "
            f"(batch_size={batch_size}, dry_run={dry_run})"
        )

        # Estadísticas globales
        total_stats = {
            "total_evaluated": 0,
            "total_classified": 0,
            "total_skipped_otc": 0,
            "category_breakdown": {},
            "batch_count": 0,
            "dry_run": dry_run,
        }

        # Pre-cargar reference_map para optimizar performance
        reference_map = self._build_reference_map()

        # Construir query base
        query = self.db.query(ProductCatalog).filter(
            ProductCatalog.xfarma_prescription_category.is_(None)
        )

        # Aplicar filtros opcionales
        if filter_criteria:
            if filter_criteria.get("national_codes"):
                query = query.filter(
                    ProductCatalog.national_code.in_(filter_criteria["national_codes"])
                )

            if filter_criteria.get("requires_prescription") is True:
                query = query.filter(ProductCatalog.cima_requiere_receta == True)

            if filter_criteria.get("has_nomenclator") is True:
                query = query.filter(ProductCatalog.nomen_nombre.isnot(None))

            if filter_criteria.get("limit"):
                query = query.limit(filter_criteria["limit"])

        # Contar total de productos a procesar
        total_products = query.count()
        logger.info(f"[BACKFILL] Productos a evaluar: {total_products}")

        if total_products == 0:
            logger.info("[BACKFILL] No hay productos para procesar")
            return total_stats

        # Procesar en batches
        offset = 0
        batch_number = 0

        while offset < total_products:
            batch_number += 1
            logger.info(
                f"[BACKFILL] Procesando batch {batch_number} "
                f"(offset={offset}, size={batch_size})"
            )

            # Obtener batch de productos
            products_batch = query.offset(offset).limit(batch_size).all()

            if not products_batch:
                break

            # Crear servicio de clasificación con reference_map optimizado
            classification_service = PrescriptionClassificationService(
                db=self.db,
                reference_map=reference_map
            )

            # Clasificar batch
            batch_stats = classification_service.bulk_classify(
                products=products_batch,
                dry_run=dry_run
            )

            # Acumular estadísticas
            total_stats["total_evaluated"] += batch_stats["total_products"]
            total_stats["total_classified"] += batch_stats["classified_count"]
            total_stats["total_skipped_otc"] += batch_stats["skipped_otc_count"]

            # Acumular breakdown por categoría
            for category, count in batch_stats["category_breakdown"].items():
                total_stats["category_breakdown"][category] = (
                    total_stats["category_breakdown"].get(category, 0) + count
                )

            total_stats["batch_count"] = batch_number

            # Log progreso
            logger.info(
                f"[BACKFILL] Batch {batch_number} completado - "
                f"Clasificados: {batch_stats['classified_count']}/{batch_stats['total_products']} - "
                f"Total acumulado: {total_stats['total_classified']}/{total_stats['total_evaluated']}"
            )

            # Avanzar offset
            offset += batch_size

        # Log resumen final
        logger.info(
            f"[BACKFILL] Backfill completado - "
            f"Total evaluado: {total_stats['total_evaluated']} - "
            f"Total clasificado: {total_stats['total_classified']} - "
            f"Total OTC (omitido): {total_stats['total_skipped_otc']} - "
            f"Batches procesados: {total_stats['batch_count']}"
        )

        # Log breakdown por categoría
        if total_stats["category_breakdown"]:
            logger.info("[BACKFILL] Distribución por categoría:")
            for category, count in sorted(
                total_stats["category_breakdown"].items(),
                key=lambda x: x[1],
                reverse=True
            ):
                percentage = (count / total_stats["total_classified"] * 100) if total_stats["total_classified"] > 0 else 0
                logger.info(f"  - {category}: {count} ({percentage:.1f}%)")

        return total_stats

    def backfill_by_national_codes(
        self,
        national_codes: List[str],
        dry_run: bool = False
    ) -> Dict[str, Any]:
        """
        Backfill específico para lista de códigos nacionales

        Args:
            national_codes: Lista de códigos nacionales a clasificar
            dry_run: Si True, no persiste cambios

        Returns:
            Diccionario con estadísticas de backfill

        Example:
            >>> # Corregir productos específicos reportados en Issue #451
            >>> codes = ["895805", "929653", "722396", "874776"]
            >>> service.backfill_by_national_codes(codes, dry_run=False)
        """
        logger.info(
            f"[BACKFILL] Backfill específico para {len(national_codes)} códigos nacionales"
        )

        return self.backfill_missing_categories(
            batch_size=100,  # Batches más pequeños para listas específicas
            dry_run=dry_run,
            filter_criteria={"national_codes": national_codes}
        )

    def validate_classification(self, limit: int = 100) -> Dict[str, Any]:
        """
        Valida que la clasificación automática funciona correctamente

        Ejecuta dry_run sobre muestra de productos y retorna estadísticas
        para verificación manual.

        Args:
            limit: Número de productos a validar (default: 100)

        Returns:
            Diccionario con estadísticas de validación + muestra de productos clasificados
        """
        logger.info(f"[BACKFILL] Validando clasificación automática ({limit} productos)")

        # Ejecutar dry_run
        stats = self.backfill_missing_categories(
            batch_size=limit,
            dry_run=True,
            filter_criteria={"limit": limit, "requires_prescription": True}
        )

        # Obtener muestra de productos para inspección manual
        sample_products = (
            self.db.query(ProductCatalog)
            .filter(
                ProductCatalog.xfarma_prescription_category.is_(None),
                ProductCatalog.cima_requiere_receta == True
            )
            .limit(10)
            .all()
        )

        # Clasificar muestra y mostrar resultados
        classification_service = PrescriptionClassificationService(db=self.db)

        sample_results = []
        for product in sample_products:
            category = classification_service.classify_product(product)
            sample_results.append({
                "national_code": product.national_code,
                "name": product.get_display_name(),
                "requires_prescription": product.cima_requiere_receta,
                "has_homogeneous": product.nomen_codigo_homogeneo is not None,
                "has_pvp": product.nomen_pvp is not None,
                "predicted_category": category.value if category else None
            })

        logger.info("[BACKFILL] Muestra de clasificaciones:")
        for result in sample_results:
            logger.info(
                f"  - {result['national_code']}: {result['name'][:50]} "
                f"→ {result['predicted_category']}"
            )

        stats["sample_results"] = sample_results
        return stats
