"""
Servicio para análisis de ventas de prescripción con códigos ATC.

Issue #400 Sprint 1: Backend service para dashboard de análisis terapéutico.

Responsabilidades:
- Cálculo de KPIs de prescripción (ventas, unidades, coverage ATC)
- Agregación temporal (series mensuales en formato long/flat)
- Distribución por categorías de prescripción (14 categorías)
- Drill-down jerárquico por códigos ATC (5 niveles)
- Análisis waterfall para comparaciones YoY/MoM/QoQ
"""

import logging
import time
from datetime import date, datetime, timedelta
from decimal import Decimal
from typing import Any, Dict, List, Optional, Tuple
from uuid import UUID

from sqlalchemy import and_, case, func, literal, literal_column, or_, text
from sqlalchemy.orm import Session

from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.schemas.prescription_analytics import (
    ATCDistributionResponse,
    ATCNode,
    CategoryChange,
    CategorySummary,
    PrescriptionKPIs,
    PrescriptionOverviewResponse,
    TimeSeriesPoint,
    TopContributor,
    TopContributorsResponse,
    WaterfallAnalysisResponse,
)

logger = logging.getLogger(__name__)


class PrescriptionAnalyticsService:
    """
    Servicio para análisis de ventas de prescripción con códigos ATC.

    Proporciona cálculos agregados y análisis temporal para el dashboard
    de prescripción del Issue #400.
    """

    def __init__(self):
        """Inicializar servicio con configuración por defecto."""
        self.max_months_range = 24  # Máximo rango permitido para queries
        self.default_atc_level = 1  # Nivel jerárquico ATC por defecto

    def calculate_overview_kpis(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        categories: Optional[List[str]] = None,
        atc_codes: Optional[List[str]] = None,  # Issue #441
        laboratory_codes: Optional[List[str]] = None,  # Issue #444
    ) -> PrescriptionOverviewResponse:
        """
        Calcula KPIs + time series + resumen por categoría.

        OPTIMIZADO: Single query con CTEs para reducir I/O de 1.8GB a ~450MB.
        Issue #516: Performance optimization - de 4 queries a 1.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            categories: Filtrar por categorías específicas (opcional)
            atc_codes: Filtrar por códigos ATC nivel 1 (A-V) (opcional)
            laboratory_codes: Filtrar por códigos de laboratorio (opcional) - Issue #444

        Returns:
            PrescriptionOverviewResponse con KPIs, time_series y category_summary

        Query Strategy (OPTIMIZED):
            - CTE base_data: JOIN una sola vez (sales_data + enrichment + catalog)
            - CTE total_pharmacy: Total ventas farmacia (sin JOIN a catalog)
            - SELECT: Agregaciones paralelas sobre base_data
            - Resultado: 1 query vs 4 queries anteriores
        """
        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Calculando overview KPIs (OPTIMIZED) para farmacia {pharmacy_id} "
            f"desde {date_from} hasta {date_to}, atc_codes={atc_codes}, laboratory_codes={laboratory_codes}"
        )

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Usar implementación optimizada con CTEs
        return self._calculate_overview_optimized_cte(
            db, pharmacy_id, date_from, date_to, categories, atc_codes, laboratory_codes
        )

    def _calculate_overview_optimized_cte(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        categories: Optional[List[str]] = None,
        atc_codes: Optional[List[str]] = None,
        laboratory_codes: Optional[List[str]] = None,
    ) -> PrescriptionOverviewResponse:
        """
        Implementación optimizada usando CTEs para consolidar queries.

        Performance: Reduce de 4 queries separadas a 1 query con CTEs.
        I/O estimado: ~450MB vs 1.8GB original (75% reducción).
        """
        t_start = time.perf_counter()

        # Construir filtros dinámicos
        category_filter = ""
        atc_filter = ""
        lab_filter = ""
        params = {
            "pharmacy_id": str(pharmacy_id),
            "date_from": datetime.combine(date_from, datetime.min.time()),
            "date_to": datetime.combine(date_to, datetime.max.time()),
        }

        if categories:
            # Cast enum to text for comparison with text array
            category_filter = "AND pc.xfarma_prescription_category::text = ANY(:categories)"
            params["categories"] = categories

        if atc_codes:
            atc_conditions = " OR ".join([f"LEFT(pc.cima_atc_code, 1) = '{code.upper()}'" for code in atc_codes])
            atc_filter = f"AND ({atc_conditions})"

        if laboratory_codes:
            lab_filter = "AND pc.nomen_codigo_laboratorio = ANY(:laboratory_codes)"
            params["laboratory_codes"] = laboratory_codes

        # Query unificada con CTEs
        sql = text(f"""
            WITH base_data AS (
                -- CTE 1: JOIN único - ejecutado una sola vez
                SELECT
                    sd.total_amount,
                    sd.quantity,
                    sd.sale_date,
                    pc.xfarma_prescription_category AS category,
                    pc.cima_atc_code
                FROM sales_data sd
                JOIN sales_enrichment se ON sd.id = se.sales_data_id
                JOIN product_catalog pc ON se.product_catalog_id = pc.id
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND sd.sale_date >= :date_from
                  AND sd.sale_date <= :date_to
                  AND pc.xfarma_prescription_category IS NOT NULL
                  {category_filter}
                  {atc_filter}
                  {lab_filter}
            ),
            total_pharmacy AS (
                -- CTE 2: Total ventas farmacia (sin JOIN a catalog - más rápido)
                SELECT COALESCE(SUM(total_amount), 0) AS total
                FROM sales_data
                WHERE pharmacy_id = :pharmacy_id
                  AND sale_date >= :date_from
                  AND sale_date <= :date_to
            ),
            kpi_data AS (
                -- Agregación KPIs
                SELECT
                    COALESCE(SUM(total_amount), 0) AS total_sales,
                    COALESCE(SUM(quantity), 0) AS total_units,
                    COALESCE(AVG(total_amount), 0) AS avg_ticket,
                    COUNT(*) AS total_records,
                    COALESCE(SUM(CASE WHEN cima_atc_code IS NOT NULL THEN total_amount ELSE 0 END), 0) AS sales_with_atc
                FROM base_data
            ),
            time_series_data AS (
                -- Agregación time series mensual
                SELECT
                    TO_CHAR(sale_date, 'YYYY-MM') AS period,
                    category,
                    SUM(total_amount) AS sales,
                    SUM(quantity) AS units
                FROM base_data
                GROUP BY TO_CHAR(sale_date, 'YYYY-MM'), category
            ),
            category_summary_data AS (
                -- Agregación por categoría
                SELECT
                    category,
                    SUM(total_amount) AS total_sales,
                    SUM(quantity) AS total_units
                FROM base_data
                GROUP BY category
            )
            SELECT
                -- KPIs
                (SELECT total_sales FROM kpi_data) AS kpi_total_sales,
                (SELECT total_units FROM kpi_data) AS kpi_total_units,
                (SELECT avg_ticket FROM kpi_data) AS kpi_avg_ticket,
                (SELECT sales_with_atc FROM kpi_data) AS kpi_sales_with_atc,
                (SELECT total FROM total_pharmacy) AS pharmacy_total_sales,
                -- Time series como JSON array
                (SELECT COALESCE(JSON_AGG(
                    JSON_BUILD_OBJECT(
                        'period', period,
                        'category', category,
                        'sales', sales,
                        'units', units
                    ) ORDER BY period, category
                ), '[]'::json) FROM time_series_data) AS time_series_json,
                -- Category summary como JSON array
                (SELECT COALESCE(JSON_AGG(
                    JSON_BUILD_OBJECT(
                        'category', category,
                        'total_sales', total_sales,
                        'total_units', total_units
                    ) ORDER BY total_sales DESC
                ), '[]'::json) FROM category_summary_data) AS category_summary_json
        """)

        t_query_start = time.perf_counter()
        result = db.execute(sql, params).fetchone()
        t_query_end = time.perf_counter()

        # Parsear resultados
        total_sales = Decimal(str(result.kpi_total_sales or 0))
        total_units = int(result.kpi_total_units or 0)
        avg_ticket = Decimal(str(result.kpi_avg_ticket or 0))
        sales_with_atc = Decimal(str(result.kpi_sales_with_atc or 0))
        pharmacy_total_sales = Decimal(str(result.pharmacy_total_sales or 0))

        # Calcular porcentajes
        prescription_percentage = (
            float(total_sales / pharmacy_total_sales) if pharmacy_total_sales > 0 else 0.0
        )
        atc_coverage = float(sales_with_atc / total_sales) if total_sales > 0 else 0.0

        # Construir KPIs
        kpis = PrescriptionKPIs(
            total_sales=total_sales,
            total_units=total_units,
            prescription_percentage=round(prescription_percentage, 4),
            avg_ticket=avg_ticket,
            atc_coverage=round(atc_coverage, 4),
        )

        # Parsear time series JSON
        import json
        time_series_raw = result.time_series_json
        if isinstance(time_series_raw, str):
            time_series_raw = json.loads(time_series_raw)

        time_series = [
            TimeSeriesPoint(
                period=item["period"],
                category=str(item["category"]),
                sales=Decimal(str(item["sales"])),
                units=int(item["units"]),
            )
            for item in time_series_raw
        ]

        # Parsear category summary JSON
        category_summary_raw = result.category_summary_json
        if isinstance(category_summary_raw, str):
            category_summary_raw = json.loads(category_summary_raw)

        # Calcular porcentajes para category summary
        grand_total = sum(Decimal(str(item["total_sales"] or 0)) for item in category_summary_raw)
        category_summary = [
            CategorySummary(
                category=str(item["category"]),
                total_sales=Decimal(str(item["total_sales"])),
                total_units=int(item["total_units"]),
                percentage=round(
                    float((Decimal(str(item["total_sales"])) / grand_total) * 100) if grand_total > 0 else 0.0,
                    2
                ),
            )
            for item in category_summary_raw
        ]

        t_end = time.perf_counter()
        query_ms = (t_query_end - t_query_start) * 1000
        parse_ms = (t_end - t_query_end) * 1000
        total_ms = (t_end - t_start) * 1000

        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Overview OPTIMIZADO: {kpis.total_sales}€, "
            f"{len(time_series)} puntos time series, {len(category_summary)} categorías "
            f"(1 query CTE vs 4 queries) | "
            f"TIMING: query={query_ms:.0f}ms, parse={parse_ms:.0f}ms, total={total_ms:.0f}ms"
        )

        return PrescriptionOverviewResponse(
            kpis=kpis, time_series=time_series, category_summary=category_summary
        )

    def get_atc_distribution(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        atc_level: int = 1,
        categories: Optional[List[str]] = None,
        atc_codes: Optional[List[str]] = None,  # Issue #444
        laboratory_codes: Optional[List[str]] = None,  # Issue #444
    ) -> ATCDistributionResponse:
        """
        Distribución jerárquica por niveles ATC.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            atc_level: Nivel de detalle ATC (1-5)
            categories: Filtrar por categorías específicas (opcional)
            atc_codes: Filtrar por códigos ATC nivel 1 (opcional) - Issue #444
            laboratory_codes: Filtrar por códigos de laboratorio (opcional) - Issue #444

        Returns:
            ATCDistributionResponse con nodos jerárquicos

        Niveles ATC:
            - Nivel 1: Primer carácter (A-V) - Grupo anatómico
            - Nivel 2: Primeros 3 caracteres (A02) - Grupo terapéutico
            - Nivel 3: Primeros 4 caracteres (A02B) - Subgrupo farmacológico
            - Nivel 4: Primeros 5 caracteres (A02BC) - Subgrupo químico
            - Nivel 5: Completo 7 caracteres (A02BC01) - Principio activo

        Query Strategy:
            - Usar SUBSTRING(cima_atc_code, 1, N) para extraer nivel
            - GROUP BY nivel ATC
            - Calcular porcentajes relativos
            - Construir estructura jerárquica (parent-child)
        """
        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Calculando distribución ATC nivel {atc_level} "
            f"para farmacia {pharmacy_id}"
        )

        # Validar nivel ATC
        if not 1 <= atc_level <= 5:
            raise ValueError(f"atc_level debe estar entre 1 y 5, recibido: {atc_level}")

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Construir query base (Issue #444: añadir atc_codes y laboratory_codes)
        base_query = self._build_base_query(
            db, pharmacy_id, date_from, date_to, categories, atc_codes, laboratory_codes
        )

        # Calcular distribución por nivel ATC
        atc_nodes = self._calculate_atc_distribution_by_level(base_query, atc_level)

        # Calcular productos sin código ATC (uncategorized)
        uncategorized = self._calculate_uncategorized_products(base_query)

        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Distribución ATC calculada: {len(atc_nodes)} nodos, "
            f"uncategorized={uncategorized.get('percentage', 0) if uncategorized else 0}%"
        )

        return ATCDistributionResponse(atc_distribution=atc_nodes, uncategorized=uncategorized)

    def calculate_waterfall(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        comparison_period: str = "yoy",
        categories: Optional[List[str]] = None,
        comparison_date_from: Optional[date] = None,  # Issue #441
        comparison_date_to: Optional[date] = None,  # Issue #441
        laboratory_codes: Optional[List[str]] = None,  # Issue #444
    ) -> WaterfallAnalysisResponse:
        """
        Análisis waterfall de crecimiento por categoría.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período actual
            date_to: Fecha fin del período actual
            comparison_period: Tipo de comparación ("yoy", "mom", "qoq", "custom")
            categories: Filtrar por categorías específicas (opcional)
            comparison_date_from: Fecha inicio período comparación (solo si custom)
            comparison_date_to: Fecha fin período comparación (solo si custom)
            laboratory_codes: Filtrar por códigos de laboratorio (opcional) - Issue #444

        Returns:
            WaterfallAnalysisResponse con cambios por categoría

        Comparaciones:
            - YoY: Año sobre año (mismo mes año anterior)
            - MoM: Mes sobre mes (mes anterior)
            - QoQ: Quarter sobre quarter (trimestre anterior)
            - custom: Períodos explícitos via comparison_date_from/to

        Query Strategy:
            - Calcular ventas período actual
            - Calcular ventas período anterior (usando helper _get_comparison_dates o explícito)
            - Calcular cambios absolutos y porcentuales por categoría
            - Calcular contribución al crecimiento total
        """
        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Calculando waterfall {comparison_period.upper()} "
            f"para farmacia {pharmacy_id}"
        )

        # Validar comparison_period
        if comparison_period not in ["yoy", "mom", "qoq", "custom"]:
            raise ValueError(
                f"comparison_period debe ser 'yoy', 'mom', 'qoq' o 'custom', recibido: {comparison_period}"
            )

        # Validar rango de fechas período actual
        self._validate_date_range(date_from, date_to)

        # Issue #441: Calcular fechas del período anterior (explícito o calculado)
        if comparison_period == "custom":
            if not comparison_date_from or not comparison_date_to:
                raise ValueError(
                    "comparison_date_from y comparison_date_to son requeridos cuando comparison_period='custom'"
                )
            self._validate_date_range(comparison_date_from, comparison_date_to)
            prev_from, prev_to = comparison_date_from, comparison_date_to
        else:
            prev_from, prev_to = self._get_comparison_dates(date_from, date_to, comparison_period)

        logger.debug(
            f"[PRESCRIPTION_ANALYTICS] Comparando período actual ({date_from} - {date_to}) "
            f"vs período anterior ({prev_from} - {prev_to})"
        )

        # OPTIMIZED: Single dual-period query (Issue #516 - 50% faster)
        current_sales_by_category, previous_sales_by_category = self._get_sales_by_category_dual_period(
            db,
            pharmacy_id,
            current_date_from=date_from,
            current_date_to=date_to,
            prev_date_from=prev_from,
            prev_date_to=prev_to,
            categories=categories,
            laboratory_codes=laboratory_codes,
        )

        # Calcular cambios por categoría
        waterfall_data = self._calculate_category_changes(
            current_sales_by_category, previous_sales_by_category
        )

        # Calcular resumen total
        total_current = sum(Decimal(str(d["sales"])) for d in current_sales_by_category.values())
        total_previous = sum(Decimal(str(d["sales"])) for d in previous_sales_by_category.values())
        total_growth = total_current - total_previous

        summary = {
            "total_current": total_current,
            "total_previous": total_previous,
            "total_growth": total_growth,
        }

        # Metadata de comparación
        comparison = {
            "type": comparison_period,
            "current_period": {"from": date_from.isoformat(), "to": date_to.isoformat()},
            "previous_period": {"from": prev_from.isoformat(), "to": prev_to.isoformat()},
        }

        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Waterfall calculado: {len(waterfall_data)} categorías, "
            f"crecimiento total={float(total_growth):.2f}"
        )

        return WaterfallAnalysisResponse(
            comparison=comparison, waterfall_data=waterfall_data, summary=summary
        )

    def get_top_contributors(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        comparison_period: str = "yoy",
        comparison_date_from: Optional[date] = None,  # Issue #441
        comparison_date_to: Optional[date] = None,  # Issue #441
        limit: int = 10,
        direction: str = "all",
        categories: Optional[List[str]] = None,  # Issue #457: Filtro por categorías
    ) -> TopContributorsResponse:
        """
        Obtiene los principios activos que mas contribuyen al cambio en ventas.

        Issue #436 Fase 2: Top Contributors por Principio Activo.
        Issue #441: Soporte para períodos de comparación explícitos.

        Args:
            db: Sesion de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del periodo actual
            date_to: Fecha fin del periodo actual
            comparison_period: Tipo de comparacion ("yoy", "mom", "qoq", "custom")
            comparison_date_from: Fecha inicio período comparación (solo si custom)
            comparison_date_to: Fecha fin período comparación (solo si custom)
            limit: Numero de resultados (10, 20, 50)
            direction: Filtro de direccion ("all", "up", "down")

        Returns:
            TopContributorsResponse con lista de contributors ordenados por impacto

        Query Strategy:
            - Agrupar ventas por principio activo (nomen_principio_activo)
            - Calcular variacion vs periodo anterior
            - Ordenar por impacto absoluto (contribucion al cambio total)
            - Aplicar filtros (limit, direction)
        """
        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Calculando top contributors {comparison_period.upper()} "
            f"para farmacia {pharmacy_id} (limit={limit}, direction={direction})"
        )

        # Validar comparison_period
        if comparison_period not in ["yoy", "mom", "qoq", "custom"]:
            raise ValueError(
                f"comparison_period debe ser 'yoy', 'mom', 'qoq' o 'custom', recibido: {comparison_period}"
            )

        # Validar direction
        if direction not in ["all", "up", "down"]:
            raise ValueError(
                f"direction debe ser 'all', 'up' o 'down', recibido: {direction}"
            )

        # Validar limit
        if limit not in [10, 20, 50]:
            raise ValueError(f"limit debe ser 10, 20 o 50, recibido: {limit}")

        # Validar rango de fechas
        self._validate_date_range(date_from, date_to)

        # Issue #441: Calcular fechas del período anterior (explícito o calculado)
        if comparison_period == "custom":
            if not comparison_date_from or not comparison_date_to:
                raise ValueError(
                    "comparison_date_from y comparison_date_to son requeridos cuando comparison_period='custom'"
                )
            self._validate_date_range(comparison_date_from, comparison_date_to)
            prev_from, prev_to = comparison_date_from, comparison_date_to
        else:
            prev_from, prev_to = self._get_comparison_dates(date_from, date_to, comparison_period)

        logger.debug(
            f"[PRESCRIPTION_ANALYTICS] Top contributors: periodo actual ({date_from} - {date_to}) "
            f"vs periodo anterior ({prev_from} - {prev_to})"
        )

        # Performance optimization: single query for both periods (~50% faster)
        # Issue #457: Pasar filtro de categorías
        current_sales, previous_sales = self._get_sales_by_active_ingredient_dual_period(
            db,
            pharmacy_id,
            current_date_from=date_from,
            current_date_to=date_to,
            prev_date_from=prev_from,
            prev_date_to=prev_to,
            categories=categories,
        )

        # Calcular cambios por principio activo
        contributors = self._calculate_ingredient_changes(
            current_sales, previous_sales, direction, limit
        )

        # Calcular cambio total
        total_current = sum(Decimal(str(d["sales"])) for d in current_sales.values())
        total_previous = sum(Decimal(str(d["sales"])) for d in previous_sales.values())
        total_change = total_current - total_previous

        # Metadata de filtros aplicados
        filters_applied = {
            "limit": limit,
            "direction": direction,
            "date_from": date_from.isoformat(),
            "date_to": date_to.isoformat(),
            "prev_from": prev_from.isoformat(),
            "prev_to": prev_to.isoformat(),
            "categories": categories,  # Issue #457: Incluir filtro de categorías
        }

        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Top contributors calculados: {len(contributors)} items, "
            f"cambio total={float(total_change):.2f}"
        )

        return TopContributorsResponse(
            contributors=contributors,
            total_change=total_change,
            comparison_period=comparison_period,
            filters_applied=filters_applied,
        )

    def get_products_by_category(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        category: Optional[str] = None,
        limit: int = 10,
    ) -> Dict:
        """
        Obtiene productos agrupados por categoría de prescripción.

        Issue #441 Fase 4: Acordeón drill-down de categorías a productos.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período
            date_to: Fecha fin del período
            category: Categoría específica (None = todas)
            limit: Top N productos por categoría

        Returns:
            Dict con categories y summary
        """
        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Obteniendo productos por categoría para farmacia {pharmacy_id}"
        )

        # Validar fechas
        self._validate_date_range(date_from, date_to)

        # Mapeo de nombres legibles para categorías (sincronizado con frontend)
        CATEGORY_NAMES = {
            "formulas_magistrales": "Fórmulas Magistrales",
            "vacunas_individualizadas": "Vacunas Individualizadas",
            "tiras_reactivas_glucosa": "Tiras Reactivas de Glucosa",
            "incontinencia_financiada": "Incontinencia Financiada",
            "efectos_financiados": "Efectos Financiados",
            "ortopedia_financiada": "Ortopedia Financiada",
            "dietoterapicos": "Dietoterapéuticos",
            "veterinaria": "Veterinaria",
            "conjunto_homogeneo": "Conjunto Homogéneo",
            "margen_especial_3": "Margen Especial 3 (≥500€)",
            "margen_especial_2": "Margen Especial 2 (200-500€)",
            "margen_especial_1": "Margen Especial 1 (91-200€)",
            "medicamentos": "Medicamentos",
            "no_comercializado": "No Comercializado",
            "uso_humano_no_financiado": "Uso Humano No Financiado",
        }

        # Issue #449 + Issue #451: COALESCE para nombre de producto con múltiples fallbacks
        # Prioridad: cima_nombre_comercial > nomen_nombre > SalesData.product_name
        # Esto asegura que productos de nomenclator (vacunas, ortopedia, etc.) muestren nombre
        product_name_expr = func.coalesce(
            ProductCatalog.cima_nombre_comercial,
            ProductCatalog.nomen_nombre,
            SalesData.product_name,  # Issue #451: Fallback final desde datos de venta
        ).label("product_name")

        query = (
            db.query(
                ProductCatalog.xfarma_prescription_category.label("category"),
                ProductCatalog.national_code,
                product_name_expr,
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
        )

        # Filtrar por categoría específica si se proporciona
        if category:
            query = query.filter(ProductCatalog.xfarma_prescription_category == category)

        # Agrupar por categoría y producto
        # Issue #449 + Issue #451: Usar EXACTAMENTE misma expresión COALESCE en group_by
        # PostgreSQL requiere que todas las columnas en SELECT estén en GROUP BY o agregadas
        query = query.group_by(
            ProductCatalog.xfarma_prescription_category,
            ProductCatalog.national_code,
            func.coalesce(
                ProductCatalog.cima_nombre_comercial,
                ProductCatalog.nomen_nombre,
                SalesData.product_name,  # Issue #451: Incluir fallback para coincidir con SELECT
            ),
        )

        # Ejecutar query
        results = query.all()

        # Organizar por categoría
        categories_dict: Dict[str, Dict] = {}
        for row in results:
            cat_key = row.category
            if cat_key not in categories_dict:
                categories_dict[cat_key] = {
                    "category_key": cat_key,
                    "category_name": CATEGORY_NAMES.get(cat_key, cat_key),
                    "total_sales": Decimal("0"),
                    "total_units": 0,
                    "products": [],
                }

            # Issue #449 + Issue #451: Usar product_name del COALESCE con filtrado de valores vacíos/nan
            raw_name = row.product_name
            if raw_name is None or (isinstance(raw_name, str) and raw_name.strip().lower() in ("", "nan", "none")):
                product_name = "Sin nombre"
            else:
                product_name = raw_name.strip() if isinstance(raw_name, str) else str(raw_name)

            categories_dict[cat_key]["products"].append({
                "national_code": row.national_code,
                "name": product_name,
                "sales": float(row.sales) if row.sales else 0.0,
                "units": int(row.units) if row.units else 0,
            })
            categories_dict[cat_key]["total_sales"] += Decimal(str(row.sales or 0))
            categories_dict[cat_key]["total_units"] += int(row.units or 0)

        # Ordenar productos por ventas y aplicar limit
        for cat_data in categories_dict.values():
            cat_data["products"].sort(key=lambda x: x["sales"], reverse=True)
            total_sales = float(cat_data["total_sales"])
            for product in cat_data["products"]:
                product["percentage"] = round((product["sales"] / total_sales * 100), 2) if total_sales > 0 else 0.0
            cat_data["products"] = cat_data["products"][:limit]
            cat_data["total_sales"] = float(cat_data["total_sales"])

        # Ordenar categorías por ventas
        categories_list = sorted(
            categories_dict.values(),
            key=lambda x: x["total_sales"],
            reverse=True
        )

        # Calcular summary
        total_products = sum(len(c["products"]) for c in categories_list)
        total_sales = sum(c["total_sales"] for c in categories_list)

        return {
            "categories": categories_list,
            "summary": {
                "total_categories": len(categories_list),
                "total_products": total_products,
                "total_sales": total_sales,
            }
        }

    def get_products_by_active_ingredient(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        active_ingredient: str,
        limit: int = 10,
        comparison_date_from: Optional[date] = None,  # Issue #444: YoY comparison
        comparison_date_to: Optional[date] = None,
        grouping_type: Optional[str] = None,  # Issue #451: Tipo de agrupación para drill-down correcto
    ) -> Dict:
        """
        Obtiene productos de un grupo de análisis específico.

        Issue #441 Fase 4: Acordeón drill-down de principios activos a productos.
        Issue #444: Añade comparación YoY con % contribución al cambio por producto.
        Issue #451: Drill-down adaptativo según grouping_type (homogeneous_group, active_ingredient, product).

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del período actual
            date_to: Fecha fin del período actual
            active_ingredient: Nombre del grupo de análisis (puede ser grupo homogéneo, principio activo, o producto)
            limit: Top N productos
            comparison_date_from: Fecha inicio período comparación (YoY)
            comparison_date_to: Fecha fin período comparación (YoY)
            grouping_type: Tipo de agrupación ("homogeneous_group", "active_ingredient", "product")

        Returns:
            Dict con products (incluyendo variation_euros, contribution_percent) y summary
        """
        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Obteniendo productos para {grouping_type or 'active_ingredient'}: {active_ingredient}"
        )

        # Validar fechas
        self._validate_date_range(date_from, date_to)

        # Issue #451: Determinar filtro según tipo de agrupación
        # Issue #452: Caso especial "Sin identificar" - productos con NULL en campos de agrupación
        is_unidentified = active_ingredient.lower() in ("sin identificar", "sin_identificar", "unidentified")

        if is_unidentified:
            # Buscar productos donde todos los campos de agrupación son NULL o vacíos
            grouping_filter = and_(
                or_(
                    ProductCatalog.nomen_nombre_homogeneo.is_(None),
                    ProductCatalog.nomen_nombre_homogeneo == "",
                ),
                or_(
                    ProductCatalog.nomen_principio_activo.is_(None),
                    ProductCatalog.nomen_principio_activo == "",
                ),
            )
            logger.info("[PRESCRIPTION_ANALYTICS] Filtering for unidentified products (NULL grouping fields)")
        elif grouping_type == "homogeneous_group":
            grouping_filter = ProductCatalog.nomen_nombre_homogeneo == active_ingredient
        elif grouping_type == "product":
            # Para productos únicos, buscar por nombre comercial o producto de venta
            grouping_filter = or_(
                ProductCatalog.cima_nombre_comercial == active_ingredient,
                ProductCatalog.nomen_nombre == active_ingredient,
            )
        else:
            # Default: principio activo (retrocompatibilidad)
            grouping_filter = ProductCatalog.nomen_principio_activo == active_ingredient

        # Query período actual
        # Issue #452: Añadir SalesData.product_name para fallback en productos sin catálogo completo
        query = (
            db.query(
                ProductCatalog.national_code,
                ProductCatalog.cima_nombre_comercial,
                ProductCatalog.nomen_principio_activo,
                ProductCatalog.nomen_nombre,  # Issue #451: Añadir para fallback de nombre
                func.max(SalesData.product_name).label("sales_product_name"),  # Issue #452: Fallback desde ERP
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                    grouping_filter,  # Issue #451: Filtro adaptativo
                )
            )
            .group_by(
                ProductCatalog.national_code,
                ProductCatalog.cima_nombre_comercial,
                ProductCatalog.nomen_principio_activo,
                ProductCatalog.nomen_nombre,
            )
            .order_by(func.sum(SalesData.total_amount).desc())
            .limit(limit)
        )

        results = query.all()

        # Issue #444: Si hay fechas de comparación, obtener ventas y unidades del período anterior
        prev_sales_by_code: Dict[str, Decimal] = {}
        prev_units_by_code: Dict[str, int] = {}  # Issue #453: Unidades período anterior
        if comparison_date_from and comparison_date_to:
            self._validate_date_range(comparison_date_from, comparison_date_to)

            prev_query = (
                db.query(
                    ProductCatalog.national_code,
                    func.sum(SalesData.total_amount).label("sales"),
                    func.sum(SalesData.quantity).label("units"),  # Issue #453: Añadir unidades
                )
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
                .filter(
                    and_(
                        SalesData.pharmacy_id == pharmacy_id,
                        SalesData.sale_date >= datetime.combine(comparison_date_from, datetime.min.time()),
                        SalesData.sale_date <= datetime.combine(comparison_date_to, datetime.max.time()),
                        ProductCatalog.xfarma_prescription_category.isnot(None),
                        grouping_filter,  # Issue #451: Usar mismo filtro adaptativo
                    )
                )
                .group_by(ProductCatalog.national_code)
            )

            for row in prev_query.all():
                prev_sales_by_code[row.national_code] = Decimal(str(row.sales or 0))
                prev_units_by_code[row.national_code] = int(row.units or 0)  # Issue #453

            logger.debug(
                f"[PRESCRIPTION_ANALYTICS] YoY comparison: {len(prev_sales_by_code)} productos en período anterior"
            )

        products = []
        total_sales = Decimal("0")
        total_units = 0
        total_variation = Decimal("0")  # Issue #444: Para calcular contribución

        for row in results:
            sales = Decimal(str(row.sales or 0))
            units = int(row.units or 0)
            prev_sales = prev_sales_by_code.get(row.national_code, Decimal("0"))
            prev_units = prev_units_by_code.get(row.national_code, 0)  # Issue #453
            variation = sales - prev_sales
            units_variation = units - prev_units  # Issue #453

            # Issue #451 + #452: Fallback de nombre con filtrado de valores vacíos/nan
            # Prioridad: cima_nombre_comercial > nomen_nombre > sales_product_name (del ERP)
            raw_name = row.cima_nombre_comercial or row.nomen_nombre or row.sales_product_name
            if raw_name is None or (isinstance(raw_name, str) and raw_name.strip().lower() in ("", "nan", "none")):
                product_name = f"Producto CN {row.national_code}" if row.national_code else "Sin nombre"
            else:
                product_name = raw_name.strip() if isinstance(raw_name, str) else str(raw_name)

            products.append({
                "national_code": row.national_code,
                "product_name": product_name,
                "total_sales": float(sales),
                "total_units": units,
                "prev_sales": float(prev_sales),  # Issue #444
                "prev_units": prev_units,  # Issue #453: Unidades período anterior
                "variation_euros": float(variation),  # Issue #444
                "variation_units": units_variation,  # Issue #453: Variación unidades
            })
            total_sales += sales
            total_units += units
            total_variation += variation

        # Calcular porcentajes y contribución al cambio
        for product in products:
            product["percentage"] = round(
                (product["total_sales"] / float(total_sales) * 100), 2
            ) if total_sales > 0 else 0.0

            # Issue #444: % contribución al cambio total del principio activo
            if total_variation != 0:
                product["contribution_percent"] = round(
                    (Decimal(str(product["variation_euros"])) / total_variation * 100), 2
                )
            else:
                product["contribution_percent"] = 0.0

        logger.info(
            f"[PRESCRIPTION_ANALYTICS] Encontrados {len(products)} productos para {active_ingredient}, "
            f"variación total: {float(total_variation):.2f}€"
        )

        return {
            "active_ingredient": active_ingredient,
            "products": products,
            "summary": {
                "total_products": len(products),
                "total_sales": float(total_sales),
                "total_units": total_units,
                "total_variation": float(total_variation),  # Issue #444
            }
        }

    # ========================================================================
    # HELPER METHODS - Query Building
    # ========================================================================

    def _build_base_query(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        categories: Optional[List[str]] = None,
        atc_codes: Optional[List[str]] = None,  # Issue #441
        laboratory_codes: Optional[List[str]] = None,  # Issue #444: Filtro laboratorios CIMA
    ):
        """
        Construir query base con JOIN sales_data, sales_enrichment, product_catalog.

        Filtra por:
        - pharmacy_id
        - rango de fechas
        - productos de prescripción (xfarma_prescription_category IS NOT NULL)
        - categorías específicas (opcional)
        - códigos ATC nivel 1 (opcional) - Issue #441
        - códigos de laboratorio (opcional) - Issue #444

        Returns:
            Query SQLAlchemy sin ejecutar (permite agregar más filtros/agregaciones)
        """
        query = (
            db.query(
                SalesData.id.label("sale_id"),
                SalesData.sale_date,
                SalesData.quantity,
                SalesData.total_amount,
                ProductCatalog.national_code,
                ProductCatalog.cima_atc_code,
                ProductCatalog.xfarma_prescription_category,
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
        )

        # Filtrar por categorías específicas si se proporcionan
        if categories:
            query = query.filter(ProductCatalog.xfarma_prescription_category.in_(categories))

        # Issue #441: Filtrar por códigos ATC nivel 1 (primera letra del código ATC)
        if atc_codes:
            # Construir filtro OR para cada código ATC nivel 1
            from sqlalchemy import func
            atc_conditions = [
                func.left(ProductCatalog.cima_atc_code, 1) == code.upper()
                for code in atc_codes
            ]
            from sqlalchemy import or_
            query = query.filter(or_(*atc_conditions))

        # Issue #444: Filtrar por códigos de laboratorio (CIMA/nomenclator)
        if laboratory_codes:
            query = query.filter(ProductCatalog.nomen_codigo_laboratorio.in_(laboratory_codes))

        return query

    def _validate_date_range(self, date_from: date, date_to: date):
        """Validar que el rango de fechas sea válido y no exceda el límite."""
        if date_from >= date_to:
            raise ValueError(f"date_from ({date_from}) debe ser menor que date_to ({date_to})")

        # Calcular diferencia en meses
        months_diff = (date_to.year - date_from.year) * 12 + (date_to.month - date_from.month)

        if months_diff > self.max_months_range:
            raise ValueError(
                f"El rango máximo permitido es de {self.max_months_range} meses, "
                f"pero se solicitaron {months_diff} meses"
            )

    # ========================================================================
    # HELPER METHODS - KPIs Calculation
    # ========================================================================

    def _calculate_kpis(
        self,
        db: Session,
        base_query,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
    ) -> PrescriptionKPIs:
        """
        Calcular KPIs principales de prescripción.

        KPIs:
        - total_sales: Suma de total_amount
        - total_units: Suma de quantity
        - prescription_percentage: % del total de ventas de la farmacia
        - avg_ticket: Promedio de total_amount por venta
        - atc_coverage: % de ventas con código ATC asignado
        """
        # Agregación de ventas de prescripción
        prescription_agg = base_query.with_entities(
            func.sum(SalesData.total_amount).label("total_sales"),
            func.sum(SalesData.quantity).label("total_units"),
            func.avg(SalesData.total_amount).label("avg_ticket"),
            func.count(SalesData.id).label("total_records"),
        ).first()

        total_sales = Decimal(str(prescription_agg.total_sales or 0))
        total_units = int(prescription_agg.total_units or 0)
        avg_ticket = Decimal(str(prescription_agg.avg_ticket or 0))

        # Calcular prescription_percentage (% del total de ventas de la farmacia)
        total_pharmacy_sales = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= datetime.combine(date_from, datetime.min.time()),
                    SalesData.sale_date <= datetime.combine(date_to, datetime.max.time()),
                )
            )
            .scalar()
        )

        total_pharmacy_sales = Decimal(str(total_pharmacy_sales or 0))
        # Retornar como decimal (0.1974) en lugar de porcentaje (19.74)
        # El frontend usa format_percentage() que multiplica por 100
        prescription_percentage = (
            float(total_sales / total_pharmacy_sales) if total_pharmacy_sales > 0 else 0.0
        )

        # Calcular atc_coverage (% de ventas con código ATC)
        sales_with_atc = (
            base_query.filter(ProductCatalog.cima_atc_code.isnot(None))
            .with_entities(func.sum(SalesData.total_amount))
            .scalar()
        )

        sales_with_atc = Decimal(str(sales_with_atc or 0))
        # Retornar como decimal (0.1974) en lugar de porcentaje (19.74)
        atc_coverage = float(sales_with_atc / total_sales) if total_sales > 0 else 0.0

        return PrescriptionKPIs(
            total_sales=total_sales,
            total_units=total_units,
            prescription_percentage=round(prescription_percentage, 4),  # 4 decimales para precisión
            avg_ticket=avg_ticket,
            atc_coverage=round(atc_coverage, 4),  # 4 decimales para precisión
        )

    # ========================================================================
    # HELPER METHODS - Time Series
    # ========================================================================

    def _calculate_time_series(self, base_query) -> List[TimeSeriesPoint]:
        """
        Calcular serie temporal mensual en formato long/flat.

        Formato:
            [
                {"period": "2025-01", "category": "ANTIBIOTICOS", "sales": 5000, "units": 150},
                {"period": "2025-01", "category": "CARDIACOS", "sales": 3000, "units": 100},
                {"period": "2025-02", "category": "ANTIBIOTICOS", "sales": 5200, "units": 155},
                ...
            ]

        Returns:
            Lista de TimeSeriesPoint (cada punto = mes + categoría)
        """
        time_series_query = base_query.with_entities(
            func.to_char(SalesData.sale_date, "YYYY-MM").label("period"),
            ProductCatalog.xfarma_prescription_category.label("category"),
            func.sum(SalesData.total_amount).label("sales"),
            func.sum(SalesData.quantity).label("units"),
        ).group_by(
            func.to_char(SalesData.sale_date, "YYYY-MM"), ProductCatalog.xfarma_prescription_category
        )

        results = time_series_query.all()

        time_series = []
        for row in results:
            time_series.append(
                TimeSeriesPoint(
                    period=row.period,
                    category=str(row.category.value if hasattr(row.category, "value") else row.category),
                    sales=Decimal(str(row.sales or 0)),
                    units=int(row.units or 0),
                )
            )

        # Ordenar por período y categoría
        time_series.sort(key=lambda x: (x.period, x.category))

        return time_series

    # ========================================================================
    # HELPER METHODS - Category Summary
    # ========================================================================

    def _calculate_category_summary(self, base_query) -> List[CategorySummary]:
        """
        Calcular resumen agregado por categoría de prescripción.

        Returns:
            Lista de CategorySummary ordenada por ventas descendente
        """
        category_query = base_query.with_entities(
            ProductCatalog.xfarma_prescription_category.label("category"),
            func.sum(SalesData.total_amount).label("total_sales"),
            func.sum(SalesData.quantity).label("total_units"),
        ).group_by(ProductCatalog.xfarma_prescription_category)

        results = category_query.all()

        # Calcular total para porcentajes
        grand_total = sum(Decimal(str(row.total_sales or 0)) for row in results)

        category_summary = []
        for row in results:
            total_sales = Decimal(str(row.total_sales or 0))
            percentage = float((total_sales / grand_total) * 100) if grand_total > 0 else 0.0

            category_summary.append(
                CategorySummary(
                    category=str(row.category.value if hasattr(row.category, "value") else row.category),
                    total_sales=total_sales,
                    total_units=int(row.total_units or 0),
                    percentage=round(percentage, 2),
                )
            )

        # Ordenar por ventas descendente
        category_summary.sort(key=lambda x: x.total_sales, reverse=True)

        return category_summary

    # ========================================================================
    # HELPER METHODS - ATC Distribution
    # ========================================================================

    def _calculate_atc_distribution_by_level(self, base_query, atc_level: int) -> List[ATCNode]:
        """
        Calcular distribución por nivel ATC específico.

        Args:
            base_query: Query base filtrada
            atc_level: Nivel de detalle (1-5)

        Returns:
            Lista de ATCNode con agregación por nivel
        """
        # Determinar longitud de código según nivel
        atc_lengths = {1: 1, 2: 3, 3: 4, 4: 5, 5: 7}
        code_length = atc_lengths[atc_level]

        # Query con SUBSTRING para extraer nivel
        atc_query = (
            base_query.filter(ProductCatalog.cima_atc_code.isnot(None))
            .with_entities(
                func.substring(ProductCatalog.cima_atc_code, 1, code_length).label("atc_code"),
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
            )
            .group_by(func.substring(ProductCatalog.cima_atc_code, 1, code_length))
        )

        results = atc_query.all()

        # Calcular total para porcentajes
        grand_total = sum(Decimal(str(row.sales or 0)) for row in results)

        atc_nodes = []
        for row in results:
            sales = Decimal(str(row.sales or 0))
            percentage = float((sales / grand_total) * 100) if grand_total > 0 else 0.0

            # TODO: En producción, buscar nombre ATC desde tabla de nomenclatura ATC
            # Por ahora, usar código como nombre placeholder
            atc_name = f"ATC {row.atc_code}"

            atc_nodes.append(
                ATCNode(
                    atc_code=row.atc_code,
                    atc_name=atc_name,
                    sales=sales,
                    units=int(row.units or 0),
                    percentage=round(percentage, 2),
                    level=atc_level,
                    children=None,  # TODO: Implementar drill-down jerárquico en Sprint 2
                )
            )

        # Ordenar por ventas descendente
        atc_nodes.sort(key=lambda x: x.sales, reverse=True)

        return atc_nodes

    def _calculate_uncategorized_products(self, base_query) -> Optional[Dict[str, Any]]:
        """
        Calcular estadísticas de productos sin código ATC.

        Returns:
            Dict con sales, units y percentage, o None si no hay uncategorized
        """
        uncategorized_query = base_query.filter(ProductCatalog.cima_atc_code.is_(None)).with_entities(
            func.sum(SalesData.total_amount).label("sales"),
            func.sum(SalesData.quantity).label("units"),
        )

        result = uncategorized_query.first()

        if not result or not result.sales:
            return None

        # Calcular total para porcentaje
        total_query = base_query.with_entities(func.sum(SalesData.total_amount)).scalar()
        grand_total = Decimal(str(total_query or 0))

        uncategorized_sales = Decimal(str(result.sales or 0))
        percentage = (
            float((uncategorized_sales / grand_total) * 100) if grand_total > 0 else 0.0
        )

        return {
            "sales": float(uncategorized_sales),
            "units": int(result.units or 0),
            "percentage": round(percentage, 2),
        }

    # ========================================================================
    # HELPER METHODS - Waterfall Analysis
    # ========================================================================

    def _get_comparison_dates(
        self, date_from: date, date_to: date, comparison_type: str
    ) -> Tuple[date, date]:
        """
        Calcular fechas del período de comparación.

        Args:
            date_from: Fecha inicio período actual
            date_to: Fecha fin período actual
            comparison_type: "yoy", "mom" o "qoq"

        Returns:
            Tupla (prev_from, prev_to) con fechas del período anterior
        """
        from dateutil.relativedelta import relativedelta

        if comparison_type == "yoy":
            # Año sobre año (restar 1 año)
            prev_from = date_from - relativedelta(years=1)
            prev_to = date_to - relativedelta(years=1)
        elif comparison_type == "mom":
            # Mes sobre mes (restar 1 mes)
            prev_from = date_from - relativedelta(months=1)
            prev_to = date_to - relativedelta(months=1)
        elif comparison_type == "qoq":
            # Quarter sobre quarter (restar 3 meses)
            prev_from = date_from - relativedelta(months=3)
            prev_to = date_to - relativedelta(months=3)
        else:
            raise ValueError(f"comparison_type inválido: {comparison_type}")

        return prev_from, prev_to

    def _get_sales_by_category(self, base_query) -> Dict[str, Dict[str, Any]]:
        """
        Obtener ventas agregadas por categoría.

        Returns:
            Dict con categoría como key y {sales, units} como value
        """
        category_query = base_query.with_entities(
            ProductCatalog.xfarma_prescription_category.label("category"),
            func.sum(SalesData.total_amount).label("sales"),
            func.sum(SalesData.quantity).label("units"),
        ).group_by(ProductCatalog.xfarma_prescription_category)

        results = category_query.all()

        sales_by_category = {}
        for row in results:
            category_str = str(row.category.value if hasattr(row.category, "value") else row.category)
            sales_by_category[category_str] = {
                "sales": Decimal(str(row.sales or 0)),
                "units": int(row.units or 0),
            }

        return sales_by_category

    def _get_sales_by_category_dual_period(
        self,
        db: Session,
        pharmacy_id: UUID,
        current_date_from: date,
        current_date_to: date,
        prev_date_from: date,
        prev_date_to: date,
        categories: Optional[List[str]] = None,
        laboratory_codes: Optional[List[str]] = None,
    ) -> Tuple[Dict[str, Dict[str, Any]], Dict[str, Dict[str, Any]]]:
        """
        Optimized dual-period query for waterfall analysis.

        Performance: ~50% faster than 2 separate queries (Issue #516).
        Uses CASE statement to label rows by period, avoiding duplicate JOIN costs.

        Args:
            db: Database session
            pharmacy_id: Pharmacy ID
            current_date_from: Current period start
            current_date_to: Current period end
            prev_date_from: Previous period start
            prev_date_to: Previous period end
            categories: Optional category filter
            laboratory_codes: Optional laboratory filter (Issue #444)

        Returns:
            Tuple of (current_sales_by_category, previous_sales_by_category) dicts
        """
        # Build conditional filters
        # SECURITY NOTE: These are hardcoded SQL fragments, NOT user-supplied values.
        # Actual filter values use parameterized queries via :params (safe from injection).
        category_filter = ""
        lab_filter = ""
        params = {
            "pharmacy_id": str(pharmacy_id),
            "current_from": datetime.combine(current_date_from, datetime.min.time()),
            "current_to": datetime.combine(current_date_to, datetime.max.time()),
            "prev_from": datetime.combine(prev_date_from, datetime.min.time()),
            "prev_to": datetime.combine(prev_date_to, datetime.max.time()),
        }

        if categories:
            category_filter = "AND pc.xfarma_prescription_category::text = ANY(:categories)"
            params["categories"] = categories

        if laboratory_codes:
            lab_filter = "AND pc.nomen_codigo_laboratorio = ANY(:laboratory_codes)"
            params["laboratory_codes"] = laboratory_codes

        # Single query for both periods using CASE with BETWEEN for readability
        sql = text(f"""
            SELECT
                CASE
                    WHEN sd.sale_date BETWEEN :current_from AND :current_to THEN 'current'
                    WHEN sd.sale_date BETWEEN :prev_from AND :prev_to THEN 'previous'
                    ELSE 'other'
                END AS period,
                pc.xfarma_prescription_category AS category,
                SUM(sd.total_amount) AS sales,
                SUM(sd.quantity) AS units
            FROM sales_data sd
            JOIN sales_enrichment se ON sd.id = se.sales_data_id
            JOIN product_catalog pc ON se.product_catalog_id = pc.id
            WHERE sd.pharmacy_id = :pharmacy_id
              AND se.enrichment_status = 'enriched'
              AND pc.xfarma_prescription_category IS NOT NULL
              AND (
                  sd.sale_date BETWEEN :current_from AND :current_to
                  OR
                  sd.sale_date BETWEEN :prev_from AND :prev_to
              )
              {category_filter}
              {lab_filter}
            GROUP BY
                CASE
                    WHEN sd.sale_date BETWEEN :current_from AND :current_to THEN 'current'
                    WHEN sd.sale_date BETWEEN :prev_from AND :prev_to THEN 'previous'
                    ELSE 'other'
                END,
                pc.xfarma_prescription_category
        """)

        results = db.execute(sql, params).fetchall()

        # Split results into current and previous dicts
        current_sales: Dict[str, Dict[str, Any]] = {}
        previous_sales: Dict[str, Dict[str, Any]] = {}

        for row in results:
            period = row.period
            if period == "other":
                continue

            category_str = str(row.category.value if hasattr(row.category, "value") else row.category)
            data = {
                "sales": Decimal(str(row.sales or 0)),
                "units": int(row.units or 0),
            }

            if period == "current":
                current_sales[category_str] = data
            else:
                previous_sales[category_str] = data

        logger.debug(
            f"[PRESCRIPTION_ANALYTICS] Waterfall dual-period: "
            f"current={len(current_sales)} categories, previous={len(previous_sales)} categories"
        )

        return current_sales, previous_sales

    def _calculate_category_changes(
        self, current_sales: Dict[str, Dict[str, Any]], previous_sales: Dict[str, Dict[str, Any]]
    ) -> List[CategoryChange]:
        """
        Calcular cambios por categoría entre dos períodos.

        Args:
            current_sales: Ventas período actual por categoría
            previous_sales: Ventas período anterior por categoría

        Returns:
            Lista de CategoryChange ordenada por contribución descendente
        """
        # Unir todas las categorías de ambos períodos
        all_categories = set(current_sales.keys()) | set(previous_sales.keys())

        # Calcular cambio total para contribución
        total_current = sum(Decimal(str(d["sales"])) for d in current_sales.values())
        total_previous = sum(Decimal(str(d["sales"])) for d in previous_sales.values())
        total_change = total_current - total_previous

        category_changes = []
        for category in all_categories:
            current_data = current_sales.get(category, {"sales": Decimal(0), "units": 0})
            previous_data = previous_sales.get(category, {"sales": Decimal(0), "units": 0})

            current_amount = Decimal(str(current_data["sales"]))
            previous_amount = Decimal(str(previous_data["sales"]))

            absolute_change = current_amount - previous_amount

            # Calcular cambio porcentual (evitar división por cero)
            if previous_amount > 0:
                percentage_change = float((absolute_change / previous_amount) * 100)
            elif current_amount > 0:
                percentage_change = 100.0  # Categoría nueva
            else:
                percentage_change = 0.0

            # Calcular contribución al crecimiento total
            if total_change != 0:
                contribution_to_growth = float((absolute_change / total_change) * 100)
            else:
                contribution_to_growth = 0.0

            category_changes.append(
                CategoryChange(
                    category=category,
                    current_sales=current_amount,
                    previous_sales=previous_amount,
                    absolute_change=absolute_change,
                    percentage_change=round(percentage_change, 2),
                    contribution_to_growth=round(contribution_to_growth, 2),
                )
            )

        # Ordenar por contribución al crecimiento (descendente)
        category_changes.sort(key=lambda x: abs(x.contribution_to_growth), reverse=True)

        return category_changes

    # ========================================================================
    # HELPER METHODS - Top Contributors (Issue #436 Fase 2)
    # ========================================================================

    def _get_sales_by_active_ingredient(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        categories: Optional[List[str]] = None,  # Issue #457: Filtro por categorías
    ) -> Dict[str, Dict[str, Any]]:
        """
        Obtener ventas agregadas por grupo de análisis adaptativo.

        Issue #449: Agrupación jerárquica adaptativa:
        1. Si tiene nomen_nombre_homogeneo → usar nombre del grupo homogéneo
        2. Si tiene nomen_principio_activo → usar principio activo
        3. Si no tiene ninguno → usar nombre del producto INDIVIDUALMENTE

        Issue #455: Productos sin HG ni PA se muestran individualmente, no agrupados
        bajo "Sin identificar". Esto es especialmente importante para categorías
        como vacunas, ortopedia, dietoterapéuticos que no tienen grupos homogéneos.

        Args:
            db: Sesion de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio del periodo
            date_to: Fecha fin del periodo

        Returns:
            Dict con key=(grupo_analisis, categoria) y value={sales, units, grouping_type}
        """
        from datetime import datetime as dt

        # Issue #449 + Issue #450: COALESCE adaptativo para agrupación
        # Prioridad: Grupo Homogéneo > Principio Activo > Nombre Producto (catálogo) > Nombre Producto (ventas)
        # CRÍTICO: sd.product_name es el fallback final para productos sin datos en catálogo
        grouping_name_expr = func.coalesce(
            ProductCatalog.nomen_nombre_homogeneo,
            ProductCatalog.nomen_principio_activo,
            ProductCatalog.cima_nombre_comercial,
            ProductCatalog.nomen_nombre,
            SalesData.product_name,  # Issue #450: Fallback final desde datos de venta
        ).label("grouping_name")

        # Campo para identificar qué tipo de agrupación se usó
        # IMPORTANTE: Verificar tanto NULL como string vacío para HG y PA
        grouping_type_expr = case(
            (
                and_(
                    ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                    ProductCatalog.nomen_nombre_homogeneo != "",
                ),
                literal("homogeneous_group")
            ),
            (
                and_(
                    ProductCatalog.nomen_principio_activo.isnot(None),
                    ProductCatalog.nomen_principio_activo != "",
                ),
                literal("active_ingredient")
            ),
            else_=literal("product"),
        ).label("grouping_type")

        # Issue #455: Para productos individuales (sin HG ni PA), necesitamos que cada uno
        # tenga un nombre único. Usamos COALESCE con national_code como último fallback
        # en el propio nombre de agrupación (no en GROUP BY) para mantener la agregación
        # de HG y PA intacta.
        #
        # Estrategia:
        # - HG: grouping_name = nomen_nombre_homogeneo (agrupa todos los productos del grupo)
        # - PA: grouping_name = nomen_principio_activo (agrupa todos los productos del PA)
        # - Producto: grouping_name = CONCAT(product_name, ' (CN ', national_code, ')') para unicidad
        product_unique_name_expr = func.concat(
            func.coalesce(
                ProductCatalog.cima_nombre_comercial,
                ProductCatalog.nomen_nombre,
                SalesData.product_name,
                literal("Producto"),
            ),
            literal(" (CN "),
            ProductCatalog.national_code,
            literal(")"),
        )

        # Issue #455: Nombre de agrupación con fallback a nombre único de producto
        # IMPORTANTE: Verificar tanto NULL como string vacío para HG y PA
        grouping_name_with_cn_expr = case(
            (
                and_(
                    ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                    ProductCatalog.nomen_nombre_homogeneo != "",
                ),
                ProductCatalog.nomen_nombre_homogeneo
            ),
            (
                and_(
                    ProductCatalog.nomen_principio_activo.isnot(None),
                    ProductCatalog.nomen_principio_activo != "",
                ),
                ProductCatalog.nomen_principio_activo
            ),
            # Para productos sin HG ni PA (NULL o vacío), usar nombre + CN para unicidad
            else_=product_unique_name_expr,
        ).label("grouping_name")

        query = (
            db.query(
                grouping_name_with_cn_expr,  # Issue #455: Nombre único para productos
                grouping_type_expr,
                ProductCatalog.xfarma_prescription_category.label("category"),
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= dt.combine(date_from, dt.min.time()),
                    SalesData.sale_date <= dt.combine(date_to, dt.max.time()),
                    # Solo productos de prescripcion (con categoria asignada)
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                    # Issue #449: NO filtrar por principio activo - incluir todas las categorías
                )
            )
        )

        # Issue #457: Filtrar por categorías específicas si se proporciona
        if categories:
            query = query.filter(
                ProductCatalog.xfarma_prescription_category.in_(categories)
            )

        query = query.group_by(
                # Issue #455: GROUP BY con la misma expresión de nombre único
                # IMPORTANTE: Verificar tanto NULL como string vacío para HG y PA
                case(
                    (
                        and_(
                            ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                            ProductCatalog.nomen_nombre_homogeneo != "",
                        ),
                        ProductCatalog.nomen_nombre_homogeneo
                    ),
                    (
                        and_(
                            ProductCatalog.nomen_principio_activo.isnot(None),
                            ProductCatalog.nomen_principio_activo != "",
                        ),
                        ProductCatalog.nomen_principio_activo
                    ),
                    else_=product_unique_name_expr,
                ),
                case(
                    (
                        and_(
                            ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                            ProductCatalog.nomen_nombre_homogeneo != "",
                        ),
                        literal("homogeneous_group")
                    ),
                    (
                        and_(
                            ProductCatalog.nomen_principio_activo.isnot(None),
                            ProductCatalog.nomen_principio_activo != "",
                        ),
                        literal("active_ingredient")
                    ),
                    else_=literal("product"),
                ),
                ProductCatalog.xfarma_prescription_category,
            )

        results = query.all()

        # Crear diccionario con clave compuesta (grupo_analisis, categoria)
        sales_by_ingredient = {}
        for row in results:
            # Issue #455: El nombre ya viene único desde SQL (con CN para productos)
            raw_name = row.grouping_name
            grouping_type = row.grouping_type or "product"
            category_str = str(
                row.category.value if hasattr(row.category, "value") else row.category
            )

            # Issue #455: Limpiar nombre (ya viene único desde SQL)
            # Para HG y PA sin nombre, usar "Sin identificar" (caso raro)
            # Para productos, el SQL ya incluye el CN en el nombre
            if raw_name is None or (isinstance(raw_name, str) and raw_name.strip().lower() in ("", "nan", "none")):
                grouping_name = "Sin identificar" if grouping_type != "product" else "Producto desconocido"
            else:
                grouping_name = raw_name.strip() if isinstance(raw_name, str) else str(raw_name)

            # Clave compuesta: (grupo_analisis, categoria)
            key = (grouping_name, category_str)
            sales_by_ingredient[key] = {
                "sales": Decimal(str(row.sales or 0)),
                "units": int(row.units or 0),
                "grouping_type": grouping_type,  # Issue #449: Tipo de agrupación usado
            }

        return sales_by_ingredient

    def _get_sales_by_active_ingredient_dual_period(
        self,
        db: Session,
        pharmacy_id: UUID,
        current_date_from: date,
        current_date_to: date,
        prev_date_from: date,
        prev_date_to: date,
        categories: Optional[List[str]] = None,
    ) -> Tuple[Dict[str, Dict[str, Any]], Dict[str, Dict[str, Any]]]:
        """
        Optimized dual-period query - fetches both periods in single query.

        Performance: ~50% faster than 2 separate queries (3.4s → 1.7s).
        Uses CASE statement to label rows by period, avoiding duplicate JOIN costs.

        Args:
            db: Database session
            pharmacy_id: Pharmacy ID
            current_date_from: Current period start
            current_date_to: Current period end
            prev_date_from: Previous period start
            prev_date_to: Previous period end
            categories: Optional category filter (Issue #457)

        Returns:
            Tuple of (current_sales, previous_sales) dicts
        """
        from datetime import datetime as dt

        # Period identification CASE
        period_expr = case(
            (
                and_(
                    SalesData.sale_date >= dt.combine(current_date_from, dt.min.time()),
                    SalesData.sale_date <= dt.combine(current_date_to, dt.max.time()),
                ),
                literal("current"),
            ),
            (
                and_(
                    SalesData.sale_date >= dt.combine(prev_date_from, dt.min.time()),
                    SalesData.sale_date <= dt.combine(prev_date_to, dt.max.time()),
                ),
                literal("previous"),
            ),
            else_=literal("other"),
        ).label("period")

        # Reuse expressions from _get_sales_by_active_ingredient
        product_unique_name_expr = func.concat(
            func.coalesce(
                ProductCatalog.cima_nombre_comercial,
                ProductCatalog.nomen_nombre,
                SalesData.product_name,
                literal("Producto"),
            ),
            literal(" (CN "),
            ProductCatalog.national_code,
            literal(")"),
        )

        grouping_name_expr = case(
            (
                and_(
                    ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                    ProductCatalog.nomen_nombre_homogeneo != "",
                ),
                ProductCatalog.nomen_nombre_homogeneo,
            ),
            (
                and_(
                    ProductCatalog.nomen_principio_activo.isnot(None),
                    ProductCatalog.nomen_principio_activo != "",
                ),
                ProductCatalog.nomen_principio_activo,
            ),
            else_=product_unique_name_expr,
        ).label("grouping_name")

        grouping_type_expr = case(
            (
                and_(
                    ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                    ProductCatalog.nomen_nombre_homogeneo != "",
                ),
                literal("homogeneous_group"),
            ),
            (
                and_(
                    ProductCatalog.nomen_principio_activo.isnot(None),
                    ProductCatalog.nomen_principio_activo != "",
                ),
                literal("active_ingredient"),
            ),
            else_=literal("product"),
        ).label("grouping_type")

        # Single query for both periods
        query = (
            db.query(
                period_expr,
                grouping_name_expr,
                grouping_type_expr,
                ProductCatalog.xfarma_prescription_category.label("category"),
                func.sum(SalesData.total_amount).label("sales"),
                func.sum(SalesData.quantity).label("units"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    # Combined date filter: either in current OR previous period
                    or_(
                        and_(
                            SalesData.sale_date >= dt.combine(current_date_from, dt.min.time()),
                            SalesData.sale_date <= dt.combine(current_date_to, dt.max.time()),
                        ),
                        and_(
                            SalesData.sale_date >= dt.combine(prev_date_from, dt.min.time()),
                            SalesData.sale_date <= dt.combine(prev_date_to, dt.max.time()),
                        ),
                    ),
                    ProductCatalog.xfarma_prescription_category.isnot(None),
                )
            )
        )

        # Category filter (Issue #457)
        if categories:
            query = query.filter(
                ProductCatalog.xfarma_prescription_category.in_(categories)
            )

        # Group by period + grouping expressions
        query = query.group_by(
            period_expr,
            case(
                (
                    and_(
                        ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                        ProductCatalog.nomen_nombre_homogeneo != "",
                    ),
                    ProductCatalog.nomen_nombre_homogeneo,
                ),
                (
                    and_(
                        ProductCatalog.nomen_principio_activo.isnot(None),
                        ProductCatalog.nomen_principio_activo != "",
                    ),
                    ProductCatalog.nomen_principio_activo,
                ),
                else_=product_unique_name_expr,
            ),
            case(
                (
                    and_(
                        ProductCatalog.nomen_nombre_homogeneo.isnot(None),
                        ProductCatalog.nomen_nombre_homogeneo != "",
                    ),
                    literal("homogeneous_group"),
                ),
                (
                    and_(
                        ProductCatalog.nomen_principio_activo.isnot(None),
                        ProductCatalog.nomen_principio_activo != "",
                    ),
                    literal("active_ingredient"),
                ),
                else_=literal("product"),
            ),
            ProductCatalog.xfarma_prescription_category,
        )

        results = query.all()

        # Split results into current and previous dicts
        current_sales: Dict[str, Dict[str, Any]] = {}
        previous_sales: Dict[str, Dict[str, Any]] = {}

        for row in results:
            period = row.period
            if period == "other":
                continue

            raw_name = row.grouping_name
            grouping_type = row.grouping_type or "product"
            category_str = str(
                row.category.value if hasattr(row.category, "value") else row.category
            )

            if raw_name is None or (isinstance(raw_name, str) and raw_name.strip().lower() in ("", "nan", "none")):
                grouping_name = "Sin identificar" if grouping_type != "product" else "Producto desconocido"
            else:
                grouping_name = raw_name.strip() if isinstance(raw_name, str) else str(raw_name)

            key = (grouping_name, category_str)
            data = {
                "sales": Decimal(str(row.sales or 0)),
                "units": int(row.units or 0),
                "grouping_type": grouping_type,
            }

            if period == "current":
                current_sales[key] = data
            else:
                previous_sales[key] = data

        logger.debug(
            f"[PRESCRIPTION_ANALYTICS] Dual-period query: "
            f"current={len(current_sales)} items, previous={len(previous_sales)} items"
        )

        return current_sales, previous_sales

    def _calculate_ingredient_changes(
        self,
        current_sales: Dict[tuple, Dict[str, Any]],
        previous_sales: Dict[tuple, Dict[str, Any]],
        direction: str,
        limit: int,
    ) -> List[TopContributor]:
        """
        Calcular cambios por grupo de analisis entre dos periodos.

        Issue #449: Soporta agrupacion adaptativa (grupo homogeneo, principio activo, producto).

        Args:
            current_sales: Ventas periodo actual por (grupo_analisis, categoria)
            previous_sales: Ventas periodo anterior por (grupo_analisis, categoria)
            direction: Filtro de direccion ("all", "up", "down")
            limit: Numero maximo de resultados

        Returns:
            Lista de TopContributor ordenada por impacto absoluto
        """
        # Unir todas las claves de ambos periodos
        all_keys = set(current_sales.keys()) | set(previous_sales.keys())

        # Calcular cambio total para calcular impacto
        total_current = sum(Decimal(str(d["sales"])) for d in current_sales.values())
        total_previous = sum(Decimal(str(d["sales"])) for d in previous_sales.values())
        total_change = total_current - total_previous

        contributors = []
        for key in all_keys:
            grouping_name, category = key
            current_data = current_sales.get(key, {"sales": Decimal(0), "units": 0, "grouping_type": "product"})
            previous_data = previous_sales.get(key, {"sales": Decimal(0), "units": 0, "grouping_type": "product"})

            current_amount = Decimal(str(current_data["sales"]))
            previous_amount = Decimal(str(previous_data["sales"]))
            variation = current_amount - previous_amount

            # Issue #449: Obtener grouping_type (priorizar current, fallback a previous)
            grouping_type = current_data.get("grouping_type") or previous_data.get("grouping_type") or "product"

            # Filtrar por direccion
            if direction == "up" and variation <= 0:
                continue
            if direction == "down" and variation >= 0:
                continue

            # Calcular variacion porcentual (evitar division por cero)
            if previous_amount > 0:
                variation_percent = float((variation / previous_amount) * 100)
            elif current_amount > 0:
                variation_percent = 100.0  # Producto nuevo
            else:
                variation_percent = 0.0

            # Calcular impacto sobre cambio total
            if total_change != 0:
                impact = float((variation / abs(total_change)) * 100)
            else:
                impact = 0.0

            # Issue #457: Calcular variación de unidades (solo para productos individuales)
            current_units = int(current_data.get("units", 0) or 0)
            previous_units = int(previous_data.get("units", 0) or 0)
            variation_units = current_units - previous_units

            # Solo incluir unidades si es un producto individual (grouping_type="product")
            include_units = grouping_type == "product"

            contributors.append(
                TopContributor(
                    active_ingredient=grouping_name,
                    category=category,
                    grouping_type=grouping_type,  # Issue #449: Tipo de agrupacion
                    current_sales=current_amount,
                    previous_sales=previous_amount,
                    variation_euros=variation,
                    variation_percent=round(variation_percent, 2),
                    impact_percent=round(impact, 2),
                    # Issue #457: Unidades solo para productos individuales
                    current_units=current_units if include_units else None,
                    previous_units=previous_units if include_units else None,
                    variation_units=variation_units if include_units else None,
                )
            )

        # Ordenar por impacto absoluto (descendente)
        contributors.sort(key=lambda x: abs(x.impact_percent), reverse=True)

        # Aplicar limite
        return contributors[:limit]
