﻿# backend/app/services/pharmacy_partners_service.py
"""
Servicio para auto-calcular laboratorios partners de farmacias.
Define los top 8 laboratorios genéricos por volumen de ventas.
"""

import logging
from datetime import timedelta
from typing import Any, Dict, List

from sqlalchemy import desc, func
from sqlalchemy.orm import Session

from app.models.pharmacy import Pharmacy
from app.models.sales_data import SalesData
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class PharmacyPartnersService:
    """
    Servicio para gestionar laboratorios partners de farmacias
    """

    def calculate_and_update_partners(self, db: Session, pharmacy_id: str) -> Dict[str, Any]:
        """
        Calcula automáticamente los top 8 laboratorios partners basándose en ventas
        y actualiza la farmacia con estos datos.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia

        Returns:
            Diccionario con resultados del cálculo
        """

        logger.info(f"[PARTNERS] Calculando partners para farmacia {pharmacy_id}")

        try:
            # Obtener farmacia
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
            if not pharmacy:
                return {"error": "Farmacia no encontrada"}

            # Calcular período de análisis (últimos 13 meses)
            analysis_months = int(pharmacy.analysis_period_months or 13)
            end_date = utc_now()
            start_date = end_date - timedelta(days=analysis_months * 30)

            # 1. SOLUCIÓN DEFINITIVA: Usar nomenclator_local para identificar laboratorios
            from app.models.nomenclator_local import NomenclatorLocal

            # Calcular ventas por laboratorio usando nomenclator_local
            partners_data = (
                db.query(
                    NomenclatorLocal.laboratory,
                    func.sum(SalesData.total_amount).label("total_sales"),
                    func.sum(SalesData.quantity).label("total_quantity"),
                    func.count(SalesData.id).label("total_transactions"),
                    func.count(func.distinct(NomenclatorLocal.id)).label("unique_products"),
                )
                .join(
                    SalesData,
                    SalesData.codigo_nacional == NomenclatorLocal.national_code,
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date.date(),
                    SalesData.sale_date <= end_date.date(),
                    NomenclatorLocal.laboratory.isnot(None),
                    NomenclatorLocal.laboratory != "",
                    NomenclatorLocal.is_generic.is_(True),  # Solo laboratorios de genéricos
                )
                .group_by(NomenclatorLocal.laboratory)
                .having(func.sum(SalesData.total_amount) > 50)  # Mínimo 50€ en ventas
                .order_by(desc("total_sales"))
                .limit(8)
                .all()
            )

            # Si no encontramos suficientes con is_generic=True, usar todos los laboratorios
            if len(partners_data) < 3:
                logger.info("Pocos laboratorios genéricos encontrados, expandiendo búsqueda a todos los laboratorios")
                partners_data = (
                    db.query(
                        NomenclatorLocal.laboratory,
                        func.sum(SalesData.total_amount).label("total_sales"),
                        func.sum(SalesData.quantity).label("total_quantity"),
                        func.count(SalesData.id).label("total_transactions"),
                        func.count(func.distinct(NomenclatorLocal.id)).label("unique_products"),
                    )
                    .join(
                        SalesData,
                        SalesData.codigo_nacional == NomenclatorLocal.national_code,
                    )
                    .filter(
                        SalesData.pharmacy_id == pharmacy_id,
                        SalesData.sale_date >= start_date.date(),
                        SalesData.sale_date <= end_date.date(),
                        NomenclatorLocal.laboratory.isnot(None),
                        NomenclatorLocal.laboratory != "",
                    )
                    .group_by(NomenclatorLocal.laboratory)
                    .having(func.sum(SalesData.total_amount) > 50)  # Mínimo 50€ en ventas
                    .order_by(desc("total_sales"))
                    .limit(8)
                    .all()
                )

            # 3. Formatear laboratorios partners
            partners = []
            total_partner_sales = sum(float(lab.total_sales or 0) for lab in partners_data)

            for i, lab in enumerate(partners_data):
                sales_amount = float(lab.total_sales or 0)
                percentage = (sales_amount / total_partner_sales * 100) if total_partner_sales > 0 else 0

                partner = {
                    "rank": i + 1,
                    "laboratory_name": lab.laboratory,  # Cambiado de nomen_laboratorio a laboratory
                    "total_sales": sales_amount,
                    "total_quantity": int(lab.total_quantity or 0),
                    "total_transactions": int(lab.total_transactions or 0),
                    "unique_products": int(lab.unique_products or 0),
                    "percentage_of_sales": round(percentage, 2),
                    "avg_transaction": round(sales_amount / int(lab.total_transactions or 1), 2),
                    "calculated_at": utc_now().isoformat(),
                }
                partners.append(partner)

            # 4. Actualizar farmacia con partners calculados
            pharmacy.partner_laboratories = partners
            pharmacy.updated_at = utc_now()
            db.commit()

            logger.info(f"[PARTNERS] Actualizados {len(partners)} partners para farmacia {pharmacy.name}")

            return {
                "success": True,
                "pharmacy_id": pharmacy_id,
                "pharmacy_name": pharmacy.name,
                "analysis_period": {
                    "months": analysis_months,
                    "start_date": start_date.date().isoformat(),
                    "end_date": end_date.date().isoformat(),
                },
                "partners_count": len(partners),
                "total_partner_sales": round(total_partner_sales, 2),
                "partners": partners,
            }

        except Exception as e:
            logger.error(f"[PARTNERS] Error calculando partners para farmacia {pharmacy_id}: {str(e)}")
            db.rollback()
            return {"error": str(e)}

    def get_pharmacy_partners(self, db: Session, pharmacy_id: str) -> List[Dict[str, Any]]:
        """
        Obtiene los laboratorios partners de una farmacia

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia

        Returns:
            Lista de laboratorios partners
        """

        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            return []

        return pharmacy.partner_laboratories or []

    def update_partners_for_all_pharmacies(self, db: Session) -> Dict[str, Any]:
        """
        Actualiza partners para todas las farmacias activas

        Args:
            db: Sesión de base de datos

        Returns:
            Resumen de actualizaciones realizadas
        """

        logger.info("[PARTNERS] Iniciando actualización masiva de partners")

        pharmacies = db.query(Pharmacy).filter(Pharmacy.is_active.is_(True)).all()

        results = {
            "total_pharmacies": len(pharmacies),
            "updated": 0,
            "errors": 0,
            "details": [],
        }

        for pharmacy in pharmacies:
            result = self.calculate_and_update_partners(db, str(pharmacy.id))

            if result.get("success"):
                results["updated"] += 1
                results["details"].append(
                    {
                        "pharmacy_id": str(pharmacy.id),
                        "pharmacy_name": pharmacy.name,
                        "partners_count": result.get("partners_count", 0),
                    }
                )
            else:
                results["errors"] += 1
                logger.error(f"Error actualizando partners para {pharmacy.name}: {result.get('error')}")

        logger.info(
            f"[PARTNERS] Actualización completada - {results['updated']}/{results['total_pharmacies']} exitosas"
        )

        return results


# Instancia global del servicio
pharmacy_partners_service = PharmacyPartnersService()
