# backend/app/services/pdf_generator.py
"""
Generador de PDF One-Pager para Dirección (Issue #511).

Usa reportlab para generar un resumen ejecutivo de una página A4:
- Header: Nombre farmacia + período
- KPIs: 4 cards con flechas YoY
- Top Categories: Tabla de categorías por crecimiento
- Alertas: Top insights del InsightEngine
- Footer: Recomendación principal
"""

import io
import structlog
from datetime import date

from reportlab.lib import colors
from reportlab.lib.pagesizes import A4
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import cm, mm
from reportlab.platypus import (
    SimpleDocTemplate,
    Paragraph,
    Spacer,
    Table,
    TableStyle,
    HRFlowable,
)
from reportlab.lib.enums import TA_CENTER, TA_LEFT, TA_RIGHT

from app.schemas.reports import DireccionReportData

logger = structlog.get_logger(__name__)

# Colores corporativos (basados en design tokens)
COLORS = {
    "primary": colors.HexColor("#2c3e50"),      # Dark blue
    "success": colors.HexColor("#27ae60"),      # Green
    "danger": colors.HexColor("#e74c3c"),       # Red
    "warning": colors.HexColor("#f39c12"),      # Orange
    "muted": colors.HexColor("#7f8c8d"),        # Gray
    "light": colors.HexColor("#ecf0f1"),        # Light gray
    "white": colors.white,
}


class DireccionPDFGenerator:
    """
    Genera PDF One-Pager para Dirección.

    Layout A4 vertical:
    - Header (2cm): Título + período + timestamp
    - KPIs (4cm): 4 cards en fila con valor + YoY
    - Categories (5cm): Tabla top 3 categorías
    - Alerts (6cm): Lista de alertas con severidad
    - Recommendation (2cm): Texto de recomendación principal
    - Footer (1cm): Generado por xFarma
    """

    def __init__(self):
        self.page_width, self.page_height = A4
        self.margin = 1.5 * cm
        self.styles = self._create_styles()

    def _create_styles(self) -> dict:
        """Crear estilos personalizados."""
        base_styles = getSampleStyleSheet()

        return {
            "title": ParagraphStyle(
                "title",
                parent=base_styles["Heading1"],
                fontSize=18,
                textColor=COLORS["primary"],
                spaceAfter=2 * mm,
                alignment=TA_CENTER,
            ),
            "subtitle": ParagraphStyle(
                "subtitle",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=COLORS["muted"],
                alignment=TA_CENTER,
                spaceAfter=5 * mm,
            ),
            "section_header": ParagraphStyle(
                "section_header",
                parent=base_styles["Heading2"],
                fontSize=12,
                textColor=COLORS["primary"],
                spaceBefore=8 * mm,
                spaceAfter=3 * mm,
            ),
            "kpi_value": ParagraphStyle(
                "kpi_value",
                parent=base_styles["Normal"],
                fontSize=16,
                textColor=COLORS["primary"],
                alignment=TA_CENTER,
            ),
            "kpi_label": ParagraphStyle(
                "kpi_label",
                parent=base_styles["Normal"],
                fontSize=8,
                textColor=COLORS["muted"],
                alignment=TA_CENTER,
            ),
            "kpi_change_up": ParagraphStyle(
                "kpi_change_up",
                parent=base_styles["Normal"],
                fontSize=9,
                textColor=COLORS["success"],
                alignment=TA_CENTER,
            ),
            "kpi_change_down": ParagraphStyle(
                "kpi_change_down",
                parent=base_styles["Normal"],
                fontSize=9,
                textColor=COLORS["danger"],
                alignment=TA_CENTER,
            ),
            "body": ParagraphStyle(
                "body",
                parent=base_styles["Normal"],
                fontSize=9,
                textColor=COLORS["primary"],
                spaceAfter=2 * mm,
            ),
            "recommendation": ParagraphStyle(
                "recommendation",
                parent=base_styles["Normal"],
                fontSize=10,
                textColor=COLORS["primary"],
                backColor=COLORS["light"],
                borderPadding=5 * mm,
                spaceBefore=5 * mm,
            ),
            "footer": ParagraphStyle(
                "footer",
                parent=base_styles["Normal"],
                fontSize=7,
                textColor=COLORS["muted"],
                alignment=TA_CENTER,
            ),
        }

    def generate(self, data: DireccionReportData) -> bytes:
        """
        Genera PDF One-Pager.

        Args:
            data: DireccionReportData con todos los datos

        Returns:
            bytes del PDF generado
        """
        logger.info(
            "pdf_generator.generate.start",
            pharmacy=data.pharmacy_name,
        )

        buffer = io.BytesIO()
        doc = SimpleDocTemplate(
            buffer,
            pagesize=A4,
            leftMargin=self.margin,
            rightMargin=self.margin,
            topMargin=self.margin,
            bottomMargin=self.margin,
        )

        # Construir contenido
        story = []

        # 1. Header
        story.extend(self._build_header(data))

        # 2. KPIs
        story.extend(self._build_kpis(data))

        # 3. Top Categories
        story.extend(self._build_categories(data))

        # 4. Alerts
        story.extend(self._build_alerts(data))

        # 5. Recommendation
        story.extend(self._build_recommendation(data))

        # 6. Footer
        story.extend(self._build_footer(data))

        # Generar PDF
        doc.build(story)

        logger.info("pdf_generator.generate.complete")

        buffer.seek(0)
        return buffer.getvalue()

    def _build_header(self, data: DireccionReportData) -> list:
        """Construir sección de header."""
        elements = []

        # Título
        title = Paragraph(
            f"INFORME MENSUAL PARAFARMACIA",
            self.styles["title"],
        )
        elements.append(title)

        # Subtítulo con farmacia y período
        period_str = self._format_period(data.period_start, data.period_end)
        subtitle = Paragraph(
            f"{data.pharmacy_name} | {period_str}",
            self.styles["subtitle"],
        )
        elements.append(subtitle)

        # Línea separadora
        elements.append(HRFlowable(width="100%", color=COLORS["light"], thickness=1))
        elements.append(Spacer(1, 3 * mm))

        return elements

    def _build_kpis(self, data: DireccionReportData) -> list:
        """Construir sección de KPIs en cards."""
        elements = []

        # Preparar datos de KPIs
        kpis = [
            {
                "label": "VENTAS",
                "value": self._format_currency(data.total_sales),
                "change": data.sales_yoy_change,
                "suffix": "%",
            },
            {
                "label": "MARGEN",
                "value": f"{data.margin_percent:.1f}%",
                "change": data.margin_yoy_change,
                "suffix": "pp",
            },
            {
                "label": "STOCK",
                "value": self._format_currency(data.stock_value),
                "change": data.stock_yoy_change,
                "suffix": "%",
            },
            {
                "label": "SURTIDO",
                "value": f"{data.sku_count:,}".replace(",", "."),
                "change": None,
                "suffix": "",
            },
        ]

        # Construir tabla de KPIs
        kpi_data = []
        for kpi in kpis:
            cell = self._build_kpi_cell(
                kpi["label"], kpi["value"], kpi["change"], kpi["suffix"]
            )
            kpi_data.append(cell)

        # Tabla de una fila con 4 columnas
        col_width = (self.page_width - 2 * self.margin) / 4
        table = Table([kpi_data], colWidths=[col_width] * 4)
        table.setStyle(
            TableStyle(
                [
                    ("ALIGN", (0, 0), (-1, -1), "CENTER"),
                    ("VALIGN", (0, 0), (-1, -1), "MIDDLE"),
                    ("BOX", (0, 0), (-1, -1), 0.5, COLORS["light"]),
                    ("INNERGRID", (0, 0), (-1, -1), 0.5, COLORS["light"]),
                    ("BACKGROUND", (0, 0), (-1, -1), COLORS["white"]),
                    ("TOPPADDING", (0, 0), (-1, -1), 8),
                    ("BOTTOMPADDING", (0, 0), (-1, -1), 8),
                ]
            )
        )

        elements.append(table)
        elements.append(Spacer(1, 5 * mm))

        return elements

    def _build_kpi_cell(
        self, label: str, value: str, change: float | None, suffix: str
    ) -> list:
        """Construir contenido de una celda de KPI."""
        parts = []

        # Valor
        parts.append(Paragraph(value, self.styles["kpi_value"]))

        # Label
        parts.append(Paragraph(label, self.styles["kpi_label"]))

        # Cambio YoY
        if change is not None:
            arrow = "▲" if change >= 0 else "▼"
            style = "kpi_change_up" if change >= 0 else "kpi_change_down"
            change_text = f"{arrow} {abs(change):.1f}{suffix}"
            parts.append(Paragraph(change_text, self.styles[style]))
        else:
            parts.append(Paragraph("-", self.styles["kpi_label"]))

        # Retornar como mini-tabla para apilar verticalmente
        return Table([[p] for p in parts], colWidths=["100%"])

    def _build_categories(self, data: DireccionReportData) -> list:
        """Construir sección de top categorías."""
        elements = []

        elements.append(
            Paragraph("TOP CATEGORÍAS POR CRECIMIENTO", self.styles["section_header"])
        )

        if not data.top_categories:
            elements.append(Paragraph("Sin datos de categorías", self.styles["body"]))
            return elements

        # Header de tabla
        table_data = [["Categoría", "Ventas", "Var. YoY", "SKUs"]]

        for cat in data.top_categories:
            arrow = "▲" if cat.yoy_change >= 0 else "▼"
            change_color = "green" if cat.yoy_change >= 0 else "red"
            yoy_text = f'<font color="{change_color}">{arrow} {abs(cat.yoy_change):.1f}%</font>'

            table_data.append(
                [
                    cat.display_name,
                    self._format_currency(cat.sales),
                    Paragraph(yoy_text, self.styles["body"]),
                    str(cat.product_count),
                ]
            )

        col_widths = [6 * cm, 3 * cm, 3 * cm, 2 * cm]
        table = Table(table_data, colWidths=col_widths)
        table.setStyle(
            TableStyle(
                [
                    ("BACKGROUND", (0, 0), (-1, 0), COLORS["primary"]),
                    ("TEXTCOLOR", (0, 0), (-1, 0), COLORS["white"]),
                    ("FONTSIZE", (0, 0), (-1, 0), 9),
                    ("FONTSIZE", (0, 1), (-1, -1), 9),
                    ("ALIGN", (1, 0), (-1, -1), "RIGHT"),
                    ("ALIGN", (0, 0), (0, -1), "LEFT"),
                    ("VALIGN", (0, 0), (-1, -1), "MIDDLE"),
                    ("INNERGRID", (0, 0), (-1, -1), 0.5, COLORS["light"]),
                    ("BOX", (0, 0), (-1, -1), 0.5, COLORS["primary"]),
                    ("ROWBACKGROUNDS", (0, 1), (-1, -1), [COLORS["white"], COLORS["light"]]),
                    ("TOPPADDING", (0, 0), (-1, -1), 4),
                    ("BOTTOMPADDING", (0, 0), (-1, -1), 4),
                ]
            )
        )

        elements.append(table)
        elements.append(Spacer(1, 5 * mm))

        return elements

    def _build_alerts(self, data: DireccionReportData) -> list:
        """Construir sección de alertas."""
        elements = []

        elements.append(Paragraph("ALERTAS ACTIVAS", self.styles["section_header"]))

        if not data.alerts:
            elements.append(Paragraph("Sin alertas activas", self.styles["body"]))
            return elements

        # Iconos de severidad (texto en lugar de emojis para compatibilidad PDF)
        severity_icons = {
            "high": "[!]",     # Alta prioridad
            "medium": "[~]",   # Media prioridad
            "low": "[ok]",     # Baja prioridad
        }

        table_data = [["Sev.", "Alerta", "Impacto", "Acción"]]

        for alert in data.alerts:
            icon = severity_icons.get(alert.severity, "[-]")
            impact = self._format_currency(alert.economic_value) + "/mes"

            table_data.append(
                [
                    icon,
                    alert.title,
                    impact,
                    alert.action_label,
                ]
            )

        col_widths = [0.8 * cm, 7 * cm, 2.5 * cm, 4 * cm]
        table = Table(table_data, colWidths=col_widths)
        table.setStyle(
            TableStyle(
                [
                    ("BACKGROUND", (0, 0), (-1, 0), COLORS["primary"]),
                    ("TEXTCOLOR", (0, 0), (-1, 0), COLORS["white"]),
                    ("FONTSIZE", (0, 0), (-1, -1), 8),
                    ("ALIGN", (0, 0), (0, -1), "CENTER"),
                    ("ALIGN", (2, 0), (2, -1), "RIGHT"),
                    ("VALIGN", (0, 0), (-1, -1), "MIDDLE"),
                    ("INNERGRID", (0, 0), (-1, -1), 0.5, COLORS["light"]),
                    ("BOX", (0, 0), (-1, -1), 0.5, COLORS["primary"]),
                    ("ROWBACKGROUNDS", (0, 1), (-1, -1), [COLORS["white"], COLORS["light"]]),
                    ("TOPPADDING", (0, 0), (-1, -1), 3),
                    ("BOTTOMPADDING", (0, 0), (-1, -1), 3),
                ]
            )
        )

        elements.append(table)
        elements.append(Spacer(1, 5 * mm))

        return elements

    def _build_recommendation(self, data: DireccionReportData) -> list:
        """Construir sección de recomendación."""
        elements = []

        if data.recommendation:
            elements.append(
                Paragraph("RECOMENDACIÓN PRINCIPAL", self.styles["section_header"])
            )

            # Box con fondo
            rec_table = Table(
                [[Paragraph(data.recommendation, self.styles["body"])]],
                colWidths=[self.page_width - 2 * self.margin - 10 * mm],
            )
            rec_table.setStyle(
                TableStyle(
                    [
                        ("BACKGROUND", (0, 0), (-1, -1), COLORS["light"]),
                        ("BOX", (0, 0), (-1, -1), 1, COLORS["primary"]),
                        ("TOPPADDING", (0, 0), (-1, -1), 8),
                        ("BOTTOMPADDING", (0, 0), (-1, -1), 8),
                        ("LEFTPADDING", (0, 0), (-1, -1), 8),
                        ("RIGHTPADDING", (0, 0), (-1, -1), 8),
                    ]
                )
            )
            elements.append(rec_table)

        return elements

    def _build_footer(self, data: DireccionReportData) -> list:
        """Construir footer."""
        elements = []
        elements.append(Spacer(1, 10 * mm))
        elements.append(HRFlowable(width="100%", color=COLORS["light"], thickness=0.5))
        elements.append(Spacer(1, 2 * mm))

        footer_text = f"Generado por xFarma | {data.generated_at[:16].replace('T', ' ')}"
        elements.append(Paragraph(footer_text, self.styles["footer"]))

        return elements

    # =========================================================================
    # Helpers de formateo
    # =========================================================================

    def _format_currency(self, value: float) -> str:
        """Formatear valor como moneda."""
        if value >= 1000:
            return f"{value/1000:.1f}K€"
        return f"{value:.0f}€"

    def _format_period(self, start: date, end: date) -> str:
        """Formatear período para mostrar."""
        months = [
            "Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio",
            "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"
        ]

        if start.month == end.month and start.year == end.year:
            return f"{months[start.month - 1]} {start.year}"
        else:
            return f"{start.day}/{start.month}/{start.year} - {end.day}/{end.month}/{end.year}"


# Instancia singleton
pdf_generator = DireccionPDFGenerator()
