﻿# backend/app/services/partner_analysis_service.py
"""
Servicio de Análisis de Partners - Backend para Panel Partners Redesign

Implementa la lógica específica para el nuevo panel de análisis de partners:
- Universo sustituible fijo (conjuntos con AL MENOS un lab genérico)
- Análisis dinámico basado en partners seleccionados
- Cálculos de ahorro potencial optimizados para grandes datasets
- Drill-down temporal y detalle de conjuntos

Arquitectura Power BI-style con filtros composables y separación contexto/análisis
"""

import asyncio
import hashlib
import json
import logging
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Tuple
from uuid import UUID

from sqlalchemy import text
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now
from app.utils.memory_utils import log_memory_usage
from app.utils.subscription_helpers import get_data_date_limit_for_user

logger = logging.getLogger(__name__)

# Cache TTL for context treemap (5 minutes - Issue #443)
CONTEXT_TREEMAP_CACHE_TTL = 300


class PartnerAnalysisService:
    """
    Servicio especializado para análisis de partners farmacéuticos

    Características clave:
    - Separación clara entre contexto fijo y análisis dinámico
    - Filtros composables para alta performance
    - Cálculos basados en conjuntos homogéneos precalculados
    - Soporte para drill-down temporal (trimestre → mes → quincena)
    """

    def __init__(self):
        self.default_period_months = 12  # Últimos 12 meses máximo
        self.temporal_levels = ["quarter", "month", "fortnight"]

    # ========== CONTEXTO FIJO DEL UNIVERSO SUSTITUIBLE ==========

    def get_substitutable_universe_context(
        self,
        db: Session,
        pharmacy_id: UUID,
        start_date: str = None,
        end_date: str = None,
        period_months: int = None,
        limit: int = 1000,
        offset: int = 0,
    ) -> Dict[str, Any]:
        """
        Obtiene el contexto fijo del universo sustituible para la farmacia.

        DEFINICIÓN UNIVERSO SUSTITUIBLE:
        - Solo conjuntos homogéneos que tienen AL MENOS un laboratorio genérico
        - Métricas que NO cambian con selección de partners
        - Base para todos los análisis posteriores

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de farmacia
            start_date: Fecha inicio análisis (YYYY-MM-DD). Prioridad sobre period_months
            end_date: Fecha fin análisis (YYYY-MM-DD). Prioridad sobre period_months
            period_months: Período análisis en meses (DEPRECADO, fallback si no hay fechas)
            limit: Límite de grupos a retornar (default: 1000, para prevenir OOM en Render)
            offset: Offset para paginación (default: 0)

        Returns:
            Contexto fijo: ventas totales, conjuntos disponibles, etc.
        """
        logger.info(f"[PARTNER_ANALYSIS] Obteniendo universo sustituible para farmacia {pharmacy_id}")

        # Issue #xxx: Usar fechas directas si están presentes, sino fallback a period_months
        start_date, end_date = self._resolve_date_range(
            start_date_str=start_date,
            end_date_str=end_date,
            period_months=period_months,
            pharmacy_id=pharmacy_id,
            db=db,
        )

        # Calcular período en meses para el response
        period = max(1, (end_date.year - start_date.year) * 12 + (end_date.month - start_date.month))

        try:
            # 1. Conjuntos homogéneos sustituibles (tienen genéricos disponibles)
            # OPTIMIZATION: Use MATERIALIZED CTEs for better performance on 72k products
            substitutable_query = text(
                """
                WITH generic_homogeneous AS MATERIALIZED (
                    -- Conjuntos que tienen AL MENOS un producto genérico
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                ),
                pharmacy_sales_in_period AS MATERIALIZED (
                    -- Ventas de la farmacia en período para conjuntos sustituibles
                    SELECT
                        pc.nomen_codigo_homogeneo,
                        pc.nomen_nombre_homogeneo,
                        -- CORRECCIÓN: Cantidad neta considerando devoluciones (total_amount < 0)
                        SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) as total_units,
                        SUM(sd.total_amount) as total_revenue,
                        COUNT(sd.id) as total_transactions,
                        MIN(sd.sale_date) as first_sale,
                        MAX(sd.sale_date) as last_sale
                    FROM sales_data sd
                    JOIN sales_enrichment se ON sd.id = se.sales_data_id
                    JOIN product_catalog pc ON se.product_catalog_id = pc.id
                    JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                    WHERE sd.pharmacy_id = :pharmacy_id
                      AND sd.sale_date >= :start_date
                      AND sd.sale_date <= :end_date
                      AND pc.xfarma_prescription_category IS NOT NULL
                    GROUP BY pc.nomen_codigo_homogeneo, pc.nomen_nombre_homogeneo
                )
                SELECT
                    COUNT(*) as total_substitutable_groups,
                    SUM(total_units) as total_substitutable_units,
                    SUM(total_revenue) as total_substitutable_revenue,
                    SUM(total_transactions) as total_substitutable_transactions,
                    MIN(first_sale) as earliest_sale,
                    MAX(last_sale) as latest_sale
                FROM pharmacy_sales_in_period
                """
            )

            result = db.execute(
                substitutable_query,
                {
                    "pharmacy_id": str(pharmacy_id),
                    "start_date": start_date,
                    "end_date": end_date,
                },
            ).fetchone()

            # 1.5. Total REAL de la farmacia (TODAS las ventas, no solo sustituibles) - Issue #330
            total_pharmacy_query = text(
                """
                SELECT
                    SUM(total_amount) as total_pharmacy_revenue
                FROM sales_data
                WHERE pharmacy_id = :pharmacy_id
                  AND sale_date >= :start_date
                  AND sale_date <= :end_date
                """
            )

            total_pharmacy_result = db.execute(
                total_pharmacy_query,
                {
                    "pharmacy_id": str(pharmacy_id),
                    "start_date": start_date,
                    "end_date": end_date,
                },
            ).fetchone()

            # 2. Lista detallada de conjuntos sustituibles (con paginación)
            # OPTIMIZATION: Use MATERIALIZED CTE and efficient pagination
            groups_detail_query = text(
                """
                WITH generic_homogeneous AS MATERIALIZED (
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                )
                SELECT
                    pc.nomen_codigo_homogeneo,
                    pc.nomen_nombre_homogeneo,
                    SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) as total_units,
                    SUM(sd.total_amount) as total_revenue,
                    COUNT(sd.id) as total_transactions
                FROM sales_data sd
                JOIN sales_enrichment se ON sd.id = se.sales_data_id
                JOIN product_catalog pc ON se.product_catalog_id = pc.id
                JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND sd.sale_date >= :start_date
                  AND sd.sale_date <= :end_date
                GROUP BY pc.nomen_codigo_homogeneo, pc.nomen_nombre_homogeneo
                ORDER BY SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) DESC
                LIMIT :limit OFFSET :offset
                """
            )

            groups_result = db.execute(
                groups_detail_query,
                {
                    "pharmacy_id": str(pharmacy_id),
                    "start_date": start_date,
                    "end_date": end_date,
                    "limit": limit,
                    "offset": offset,
                },
            ).fetchall()

            # 3. Análisis de laboratorios disponibles en el universo
            # OPTIMIZATION: Use MATERIALIZED CTE for complex joins
            labs_query = text(
                """
                WITH generic_homogeneous AS MATERIALIZED (
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                )
                SELECT
                    pc.nomen_laboratorio,
                    COUNT(DISTINCT pc.nomen_codigo_homogeneo) as homogeneous_groups_count,
                    SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) as total_units,
                    SUM(sd.total_amount) as total_revenue
                FROM sales_data sd
                JOIN sales_enrichment se ON sd.id = se.sales_data_id
                JOIN product_catalog pc ON se.product_catalog_id = pc.id
                JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND sd.sale_date >= :start_date
                  AND sd.sale_date <= :end_date
                  AND pc.nomen_laboratorio IS NOT NULL
                  AND pc.nomen_laboratorio != ''
                GROUP BY pc.nomen_laboratorio
                ORDER BY SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) DESC
                LIMIT 100  -- OPTIMIZATION: Limit labs to top 100 for Render performance
                """
            )

            labs_result = db.execute(
                labs_query,
                {
                    "pharmacy_id": str(pharmacy_id),
                    "start_date": start_date,
                    "end_date": end_date,
                },
            ).fetchall()

            # Construir respuesta
            universe_context = {
                "pharmacy_id": str(pharmacy_id),
                "analysis_period": {
                    "start_date": start_date.isoformat(),
                    "end_date": end_date.isoformat(),
                    "months": period,
                },
                "universe_summary": {
                    "total_substitutable_groups": result.total_substitutable_groups or 0,
                    "total_substitutable_units": result.total_substitutable_units or 0,
                    "total_substitutable_revenue": float(result.total_substitutable_revenue or 0),
                    "total_substitutable_transactions": result.total_substitutable_transactions or 0,
                    "total_pharmacy_revenue": float(total_pharmacy_result.total_pharmacy_revenue or 0),  # Issue #330: Total REAL
                    "earliest_sale": (result.earliest_sale.isoformat() if result.earliest_sale else None),
                    "latest_sale": (result.latest_sale.isoformat() if result.latest_sale else None),
                },
                "homogeneous_groups": [
                    {
                        "homogeneous_code": row.nomen_codigo_homogeneo,
                        "homogeneous_name": row.nomen_nombre_homogeneo,
                        "total_units": row.total_units,
                        "total_revenue": float(row.total_revenue),
                        "total_transactions": row.total_transactions,
                    }
                    for row in groups_result
                ],
                "laboratories_in_universe": [
                    {
                        "laboratory_name": row.nomen_laboratorio,
                        "homogeneous_groups_count": row.homogeneous_groups_count,
                        "total_units": row.total_units,
                        "total_revenue": float(row.total_revenue),
                    }
                    for row in labs_result
                ],
                "context_type": "substitutable_universe_fixed",
                "calculated_at": utc_now().isoformat(),
            }

            logger.info(
                f"[PARTNER_ANALYSIS] Universo sustituible: {universe_context['universe_summary']['total_substitutable_groups']} conjuntos, {universe_context['universe_summary']['total_substitutable_units']} unidades"
            )

            return universe_context

        except Exception as e:
            logger.error(f"[PARTNER_ANALYSIS] Error obteniendo universo sustituible: {e}")
            raise

    # ========== ANÁLISIS DINÁMICO CON PARTNERS SELECCIONADOS ==========

    def calculate_partner_dynamic_analysis(
        self,
        db: Session,
        pharmacy_id: UUID,
        selected_partner_codes: List[str],
        start_date: str = None,
        end_date: str = None,
        period_months: int = None,
        discount_percentage: float = None,
        limit: Optional[int] = None,  # P1.3: Paginación opcional
        offset: int = 0,  # P1.3: Offset para paginación
        selected_employee_names: Optional[List[str]] = None,  # Issue #402
    ) -> Dict[str, Any]:
        """
        Realiza análisis dinámico basado en códigos de partners seleccionados.

        LÓGICA DE FILTRADO:
        - Solo conjuntos donde existe AL MENOS UNA alternativa de partners seleccionados
        - Ahorro potencial calculado únicamente sobre ventas NO-partners
        - Exclusión automática de conjuntos sin partners disponibles

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de farmacia
            selected_partner_codes: Lista de códigos de laboratorios partners seleccionados
            start_date: Fecha inicio análisis (YYYY-MM-DD). Prioridad sobre period_months
            end_date: Fecha fin análisis (YYYY-MM-DD). Prioridad sobre period_months
            period_months: Período análisis en meses (DEPRECADO, fallback si no hay fechas)
            discount_percentage: Porcentaje de descuento esperado (default: 10%)
            limit: Límite de grupos a retornar (None = sin límite) - P1.3
            offset: Número de grupos a saltar (default: 0) - P1.3
            selected_employee_names: Lista de nombres de empleados a filtrar (None o vacío = TODOS)

        Returns:
            Análisis dinámico: ventas analizables, ahorro potencial con descuento, penetración

        Note:
            Issue #402: Employee filtering (PRO feature)
            DECISIÓN #2: Filtrado solo por employee_name (no employee_code)
            DECISIÓN UX #2: Lista vacía o None = TODOS los empleados (default)
            DECISIÓN UX #3: Incluye ventas sin empleado como "__sin_empleado__"
        """
        log_memory_usage("partner_dynamic_start")
        logger.info(
            f"[PARTNER_ANALYSIS] Análisis dinámico para farmacia {pharmacy_id} con partner codes: {selected_partner_codes}"
        )

        # Issue #xxx: Usar fechas directas si están presentes, sino fallback a period_months
        start_date, end_date = self._resolve_date_range(
            start_date_str=start_date,
            end_date_str=end_date,
            period_months=period_months,
            pharmacy_id=pharmacy_id,
            db=db,
        )

        # Calcular período en meses para el response
        period = max(1, (end_date.year - start_date.year) * 12 + (end_date.month - start_date.month))

        # ✅ FIX: Aplicar descuento por defecto del 10% si no se especifica
        discount_pct = discount_percentage if discount_percentage is not None else 10.0

        if not selected_partner_codes:
            return self._empty_dynamic_analysis(pharmacy_id, start_date, end_date, discount_pct)

        # Issue #402: Construir filtro de empleados
        employee_filter = ""
        params = {
            "pharmacy_id": str(pharmacy_id),
            "start_date": start_date,
            "end_date": end_date,
            "selected_partner_codes": tuple(selected_partner_codes),
        }

        # DECISIÓN UX #2: Lista vacía o None = mostrar TODOS
        if selected_employee_names:
            # DECISIÓN UX #3: COALESCE para incluir "__sin_empleado__"
            employee_filter = "AND COALESCE(sd.employee_name, '__sin_empleado__') = ANY(:employee_names)"
            params["employee_names"] = selected_employee_names

        try:
            # Query principal para análisis dinámico
            dynamic_query = text(
                f"""
                WITH generic_homogeneous AS (
                    -- Conjuntos sustituibles (tienen genéricos)
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                ),
                partner_availability AS (
                    -- Conjuntos que SÍ tienen partners seleccionados disponibles
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_laboratorio IN :selected_partner_codes
                      AND pc.nomen_estado = 'ALTA'
                ),
                analyzable_sales AS (
                    -- Ventas en conjuntos analizables (sustituibles + partners disponibles)
                    SELECT
                        pc.nomen_codigo_homogeneo,
                        pc.nomen_nombre_homogeneo,
                        pc.nomen_laboratorio,
                        pc.nomen_precio_referencia,
                        pc.nomen_pvl_calculado,
                        hgm.calculated_pvl as group_calculated_pvl,
                        sd.quantity,
                        sd.total_amount,
                        sd.sale_date,
                        CASE
                            WHEN pc.nomen_codigo_laboratorio IN :selected_partner_codes THEN 'partner'
                            ELSE 'non_partner'
                        END as partner_status
                    FROM sales_data sd
                    JOIN sales_enrichment se ON sd.id = se.sales_data_id
                    JOIN product_catalog pc ON se.product_catalog_id = pc.id
                    LEFT JOIN homogeneous_groups_master hgm ON pc.nomen_codigo_homogeneo = hgm.homogeneous_code
                    JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                    JOIN partner_availability pa ON pc.nomen_codigo_homogeneo = pa.nomen_codigo_homogeneo
                    WHERE sd.pharmacy_id = :pharmacy_id
                      AND sd.sale_date >= :start_date
                      AND sd.sale_date <= :end_date
                      AND pc.xfarma_prescription_category IS NOT NULL
                      {employee_filter}
                )
                SELECT
                    -- Métricas totales analizables
                    COUNT(DISTINCT nomen_codigo_homogeneo) as analyzable_groups_count,
                    -- CORRECCIÓN: Cantidad neta considerando devoluciones
                    SUM(CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END) as total_analyzable_units,
                    SUM(total_amount) as total_analyzable_revenue,
                    COUNT(*) as total_analyzable_transactions,

                    -- Métricas de partners
                    SUM(CASE WHEN partner_status = 'partner' THEN
                        CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                    ELSE 0 END) as partner_units,
                    SUM(CASE WHEN partner_status = 'partner' THEN total_amount ELSE 0 END) as partner_revenue,
                    COUNT(CASE WHEN partner_status = 'partner' THEN 1 END) as partner_transactions,

                    -- Métricas de oportunidad (NO-partners)
                    SUM(CASE WHEN partner_status = 'non_partner' THEN
                        CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                    ELSE 0 END) as opportunity_units,
                    SUM(CASE WHEN partner_status = 'non_partner' THEN total_amount ELSE 0 END) as opportunity_revenue,
                    COUNT(CASE WHEN partner_status = 'non_partner' THEN 1 END) as opportunity_transactions,

                    -- ✅ CRÍTICO: Base para cálculo de ahorro (PVL de ventas NO-partner)
                    -- 3-tier fallback: individual product PVL → product precio_referencia → group calculated_pvl
                    -- Resolves Issue #343: Ticagrelor products with NULL individual prices use group PVL
                    SUM(CASE
                        WHEN partner_status = 'non_partner' AND COALESCE(nomen_pvl_calculado, nomen_precio_referencia, group_calculated_pvl) > 0
                        THEN (CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END) * COALESCE(nomen_pvl_calculado, nomen_precio_referencia, group_calculated_pvl)
                        ELSE 0
                    END) as potential_savings_base
                FROM analyzable_sales
                """
            )

            result = db.execute(dynamic_query, params).fetchone()

            # Detalle por conjunto homogéneo
            # P1.3: Construir query con paginación opcional
            pagination_clause = ""
            if limit is not None:
                pagination_clause = f"LIMIT {limit} OFFSET {offset}"

            groups_detail_query = text(
                f"""
                WITH generic_homogeneous AS (
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                ),
                partner_availability AS (
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_laboratorio IN :selected_partner_codes
                      AND pc.nomen_estado = 'ALTA'
                ),
                analyzable_sales AS (
                    SELECT
                        pc.nomen_codigo_homogeneo,
                        pc.nomen_nombre_homogeneo,
                        pc.nomen_laboratorio,
                        pc.nomen_precio_referencia,
                        pc.nomen_pvl_calculado,
                        hgm.calculated_pvl as group_calculated_pvl,
                        sd.quantity,
                        sd.total_amount,
                        CASE
                            WHEN pc.nomen_codigo_laboratorio IN :selected_partner_codes THEN 'partner'
                            ELSE 'non_partner'
                        END as partner_status
                    FROM sales_data sd
                    JOIN sales_enrichment se ON sd.id = se.sales_data_id
                    JOIN product_catalog pc ON se.product_catalog_id = pc.id
                    LEFT JOIN homogeneous_groups_master hgm ON pc.nomen_codigo_homogeneo = hgm.homogeneous_code
                    JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                    JOIN partner_availability pa ON pc.nomen_codigo_homogeneo = pa.nomen_codigo_homogeneo
                    WHERE sd.pharmacy_id = :pharmacy_id
                      AND sd.sale_date >= :start_date
                      AND sd.sale_date <= :end_date
                      AND pc.xfarma_prescription_category IS NOT NULL
                      {employee_filter}
                )
                SELECT
                    nomen_codigo_homogeneo,
                    nomen_nombre_homogeneo,
                    SUM(CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END) as total_units,
                    SUM(total_amount) as total_revenue,
                    SUM(CASE WHEN partner_status = 'partner' THEN
                        CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                    ELSE 0 END) as partner_units,
                    SUM(CASE WHEN partner_status = 'non_partner' THEN
                        CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                    ELSE 0 END) as opportunity_units,
                    -- 3-tier fallback: individual product PVL → product precio_referencia → group calculated_pvl
                    -- Resolves Issue #343: Ticagrelor products with NULL individual prices use group PVL
                    SUM(CASE
                        WHEN partner_status = 'non_partner' AND COALESCE(nomen_pvl_calculado, nomen_precio_referencia, group_calculated_pvl) > 0
                        THEN (CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END) * COALESCE(nomen_pvl_calculado, nomen_precio_referencia, group_calculated_pvl)
                        ELSE 0
                    END) as savings_base
                FROM analyzable_sales
                GROUP BY nomen_codigo_homogeneo, nomen_nombre_homogeneo
                HAVING SUM(CASE WHEN partner_status = 'non_partner' THEN
                    CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                ELSE 0 END) > 0
                ORDER BY SUM(CASE WHEN partner_status = 'non_partner' THEN
                    CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                ELSE 0 END) DESC
                {pagination_clause}
                """
            )

            groups_result = db.execute(groups_detail_query, params).fetchall()

            # Calcular métricas finales
            total_analyzable = result.total_analyzable_units or 0
            partner_units = result.partner_units or 0
            opportunity_units = result.opportunity_units or 0

            penetration_percentage = round((partner_units / total_analyzable) * 100, 2) if total_analyzable > 0 else 0.0

            dynamic_analysis = {
                "pharmacy_id": str(pharmacy_id),
                "selected_partner_codes": selected_partner_codes,
                "analysis_period": {
                    "start_date": start_date.isoformat(),
                    "end_date": end_date.isoformat(),
                    "months": period,
                },
                "analyzable_universe": {
                    "groups_count": result.analyzable_groups_count or 0,
                    "total_units": total_analyzable,
                    "total_revenue": float(result.total_analyzable_revenue or 0),
                    "total_transactions": result.total_analyzable_transactions or 0,
                },
                "partner_performance": {
                    "partner_units": partner_units,
                    "partner_revenue": float(result.partner_revenue or 0),
                    "partner_transactions": result.partner_transactions or 0,
                    "penetration_percentage": penetration_percentage,
                },
                "opportunity_metrics": {
                    "opportunity_units": opportunity_units,
                    "opportunity_revenue": float(result.opportunity_revenue or 0),
                    "opportunity_transactions": result.opportunity_transactions or 0,
                    "potential_savings_base": float(result.potential_savings_base or 0),
                    # ✅ FIX Issue #339: Calcular ahorro estimado con descuento
                    # Validación explícita con logging detallado
                    "estimated_savings_with_discount": self._calculate_estimated_savings(
                        pharmacy_id=pharmacy_id,
                        potential_savings_base=result.potential_savings_base,
                        discount_pct=discount_pct
                    ),
                    "applied_discount_percentage": discount_pct,
                },
                "homogeneous_groups_detail": [
                    {
                        "homogeneous_code": row.nomen_codigo_homogeneo,
                        "homogeneous_name": row.nomen_nombre_homogeneo,
                        "total_units": row.total_units,
                        "total_revenue": float(row.total_revenue),
                        "partner_units": row.partner_units,
                        "opportunity_units": row.opportunity_units,
                        "savings_base": float(row.savings_base),
                        "partner_penetration": (
                            round((row.partner_units / row.total_units) * 100, 2) if row.total_units > 0 else 0.0
                        ),
                    }
                    for row in groups_result
                ],
                "analysis_type": "partner_dynamic",
                "calculated_at": utc_now().isoformat(),
            }

            log_memory_usage("partner_dynamic_end")
            logger.info(
                f"[PARTNER_ANALYSIS] Análisis dinámico completado: {dynamic_analysis['analyzable_universe']['groups_count']} conjuntos analizables, {penetration_percentage}% penetración partners"
            )

            return dynamic_analysis

        except Exception as e:
            logger.error(f"[PARTNER_ANALYSIS] Error en análisis dinámico: {e}")
            raise

    # ========== DRILL-DOWN TEMPORAL ==========

    def get_temporal_breakdown(
        self,
        db: Session,
        pharmacy_id: UUID,
        selected_partner_codes: List[str],
        level: str = "quarter",
        period_months: int = None,
        homogeneous_code: Optional[str] = None,  # ✅ Issue #346: Filtro opcional
        selected_employee_names: Optional[List[str]] = None,  # Issue #402
    ) -> Dict[str, Any]:
        """
        Obtiene breakdown temporal para drill-down (trimestre → mes → quincena).

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de farmacia
            selected_partners: Partners seleccionados
            level: Nivel temporal ('quarter', 'month', 'fortnight')
            period_months: Período análisis
            homogeneous_code: Código conjunto homogéneo para filtrar (opcional)
            selected_employee_names: Lista de nombres de empleados a filtrar (None o vacío = TODOS)

        Returns:
            Datos temporales con métricas por período

        Note:
            Issue #402: Employee filtering (PRO feature)
            DECISIÓN #2: Filtrado solo por employee_name (no employee_code)
            DECISIÓN UX #2: Lista vacía o None = TODOS los empleados (default)
            DECISIÓN UX #3: Incluye ventas sin empleado como "__sin_empleado__"
        """
        filter_msg = f" [filtro: conjunto {homogeneous_code}]" if homogeneous_code else ""
        logger.info(f"[PARTNER_ANALYSIS] Breakdown temporal nivel '{level}' para farmacia {pharmacy_id}{filter_msg}")

        if level not in self.temporal_levels:
            raise ValueError(f"Nivel temporal '{level}' no válido. Usar: {self.temporal_levels}")

        period = period_months or self.default_period_months
        start_date, end_date = self._get_analysis_period(period, pharmacy_id, db)

        # ✅ SECURITY: Usar mapeo controlado para prevenir SQL injection
        # Solo niveles válidos predefinidos, no input directo del usuario
        temporal_expressions = {
            "quarter": "DATE_TRUNC('quarter', sd.sale_date)",
            "month": "DATE_TRUNC('month', sd.sale_date)",
            "fortnight": """
                CASE
                    WHEN EXTRACT(DAY FROM sd.sale_date) <= 15 THEN
                        DATE_TRUNC('month', sd.sale_date)
                    ELSE
                        DATE_TRUNC('month', sd.sale_date) + INTERVAL '15 days'
                END
            """,
        }
        date_expression = temporal_expressions[level]  # Safe: solo valores del dict

        # ✅ Issue #346: Construir filtro opcional de conjunto homogéneo
        # SEGURO: No hay inyección SQL, homogeneous_code se pasa como parámetro binding
        homogeneous_filter = (
            "AND pc.nomen_codigo_homogeneo = :homogeneous_code"
            if homogeneous_code else ""
        )

        # Issue #402: Construir filtro de empleados
        # DECISIÓN UX #2: Lista vacía [] o None = TODOS (no filtrar)
        employee_filter = ""
        if selected_employee_names and len(selected_employee_names) > 0:
            employee_filter = "AND COALESCE(sd.employee_name, '__sin_empleado__') = ANY(:employee_names)"

        try:
            # ✅ SEGURO: SQL construido de forma segura sin inyección
            temporal_query = text(
                f"""
                WITH generic_homogeneous AS (
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                ),
                partner_availability AS (
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_laboratorio IN :selected_partner_codes
                      AND pc.nomen_estado = 'ALTA'
                ),
                analyzable_sales AS (
                    SELECT
                        {date_expression} as period_start,
                        pc.nomen_laboratorio,
                        pc.nomen_codigo_homogeneo,
                        sd.quantity,
                        sd.total_amount,
                        pc.nomen_precio_referencia,
                        pc.nomen_pvl_calculado,
                        hgm.calculated_pvl as group_calculated_pvl,
                        CASE
                            WHEN pc.nomen_codigo_laboratorio IN :selected_partner_codes THEN 'partner'
                            ELSE 'non_partner'
                        END as partner_status
                    FROM sales_data sd
                    JOIN sales_enrichment se ON sd.id = se.sales_data_id
                    JOIN product_catalog pc ON se.product_catalog_id = pc.id
                    LEFT JOIN homogeneous_groups_master hgm ON pc.nomen_codigo_homogeneo = hgm.homogeneous_code
                    JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                    JOIN partner_availability pa ON pc.nomen_codigo_homogeneo = pa.nomen_codigo_homogeneo
                    WHERE sd.pharmacy_id = :pharmacy_id
                      AND sd.sale_date >= :start_date
                      AND sd.sale_date <= :end_date
                      {homogeneous_filter}
                      {employee_filter}
                )
                SELECT
                    period_start,
                    SUM(CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END) as total_units,
                    SUM(total_amount) as total_revenue,
                    SUM(CASE WHEN partner_status = 'partner' THEN
                        CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                    ELSE 0 END) as partner_units,
                    SUM(CASE WHEN partner_status = 'non_partner' THEN
                        CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END
                    ELSE 0 END) as opportunity_units,
                    -- 3-tier fallback: individual product PVL → product precio_referencia → group calculated_pvl
                    -- Resolves Issue #343: Ticagrelor products with NULL individual prices use group PVL
                    SUM(CASE
                        WHEN partner_status = 'non_partner' AND COALESCE(nomen_pvl_calculado, nomen_precio_referencia, group_calculated_pvl) > 0
                        THEN (CASE WHEN total_amount >= 0 THEN quantity ELSE -quantity END) * COALESCE(nomen_pvl_calculado, nomen_precio_referencia, group_calculated_pvl)
                        ELSE 0
                    END) as savings_base
                FROM analyzable_sales
                GROUP BY period_start
                ORDER BY period_start
                """
            )

            # ✅ Issue #346: Construir parámetros dinámicamente
            query_params = {
                "pharmacy_id": str(pharmacy_id),
                "start_date": start_date,
                "end_date": end_date,
                "selected_partner_codes": tuple(selected_partner_codes),
            }

            # Agregar homogeneous_code solo si está presente
            if homogeneous_code:
                query_params["homogeneous_code"] = homogeneous_code

            # Issue #402: Agregar employee_names solo si está presente
            # DECISIÓN UX #2: Lista vacía [] o None = TODOS (no agregar parámetro)
            if selected_employee_names and len(selected_employee_names) > 0:
                query_params["employee_names"] = selected_employee_names

            result = db.execute(temporal_query, query_params).fetchall()

            # ✅ Issue #346: Logging específico cuando homogeneous_code no produce resultados
            if not result and homogeneous_code:
                logger.warning(
                    f"[PARTNER_ANALYSIS] No data found for homogeneous_code: {homogeneous_code}. "
                    f"Pharmacy: {pharmacy_id}, Period: {start_date} to {end_date}"
                )

            # Procesar resultados
            temporal_data = []
            for row in result:
                total_units = row.total_units or 0
                partner_units = row.partner_units or 0

                penetration = round((partner_units / total_units) * 100, 2) if total_units > 0 else 0.0

                temporal_data.append(
                    {
                        "period_start": row.period_start.isoformat(),
                        "period_label": self._format_period_label(row.period_start, level),
                        "total_units": total_units,
                        "total_revenue": float(row.total_revenue or 0),
                        "partner_units": partner_units,
                        "opportunity_units": row.opportunity_units or 0,
                        "savings_base": float(row.savings_base or 0),
                        "partner_penetration": penetration,
                    }
                )

            temporal_breakdown = {
                "pharmacy_id": str(pharmacy_id),
                "selected_partner_codes": selected_partner_codes,
                "temporal_level": level,
                "analysis_period": {
                    "start_date": start_date.isoformat(),
                    "end_date": end_date.isoformat(),
                    "months": period,
                },
                "temporal_data": temporal_data,
                "summary": {
                    "periods_count": len(temporal_data),
                    "avg_partner_penetration": (
                        round(sum(p["partner_penetration"] for p in temporal_data) / len(temporal_data), 2)
                        if temporal_data
                        else 0.0
                    ),
                    "total_opportunity_units": sum(p["opportunity_units"] for p in temporal_data),
                    "total_savings_base": sum(p["savings_base"] for p in temporal_data),
                },
                "analysis_type": "temporal_breakdown",
                "calculated_at": utc_now().isoformat(),
            }

            # ✅ Issue #346: Incluir filtro aplicado si existe
            if homogeneous_code:
                temporal_breakdown["homogeneous_filter"] = homogeneous_code

            logger.info(f"[PARTNER_ANALYSIS] Breakdown temporal: {len(temporal_data)} períodos, nivel '{level}'")

            return temporal_breakdown

        except Exception as e:
            logger.error(f"[PARTNER_ANALYSIS] Error en breakdown temporal: {e}")
            raise

    # ========== DETALLE DE CONJUNTO HOMOGÉNEO ==========

    def get_homogeneous_group_detail(
        self,
        db: Session,
        pharmacy_id: UUID,
        homogeneous_code: str,
        selected_partner_codes: List[str],
        period_months: int = None,
        selected_employee_names: Optional[List[str]] = None,  # Issue #402
    ) -> Dict[str, Any]:
        """
        Obtiene detalle completo de un conjunto homogéneo específico.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de farmacia
            homogeneous_code: Código del conjunto homogéneo
            selected_partners: Partners seleccionados
            period_months: Período análisis
            selected_employee_names: Lista de nombres de empleados a filtrar (None o vacío = TODOS)

        Returns:
            Detalle con productos, laboratorios, unidades y ahorro individual

        Note:
            Issue #402: Employee filtering (PRO feature)
            DECISIÓN #2: Filtrado solo por employee_name (no employee_code)
            DECISIÓN UX #2: Lista vacía o None = TODOS los empleados (default)
            DECISIÓN UX #3: Incluye ventas sin empleado como "__sin_empleado__"
        """
        logger.info(f"[PARTNER_ANALYSIS] Detalle conjunto {homogeneous_code} para farmacia {pharmacy_id}")

        period = period_months or self.default_period_months
        start_date, end_date = self._get_analysis_period(period, pharmacy_id, db)

        # Issue #402: Construir filtro de empleados
        # DECISIÓN UX #2: Lista vacía [] o None = TODOS (no filtrar)
        employee_filter = ""
        if selected_employee_names and len(selected_employee_names) > 0:
            employee_filter = "AND COALESCE(sd.employee_name, '__sin_empleado__') = ANY(:employee_names)"

        # Convertir códigos de partner a nombres para comparaciones
        partner_names = []
        if selected_partner_codes:
            codes_query = text(
                """
                SELECT DISTINCT nomen_laboratorio
                FROM product_catalog
                WHERE nomen_codigo_laboratorio IN :codes
                AND nomen_laboratorio IS NOT NULL
            """
            )
            codes_result = db.execute(codes_query, {"codes": tuple(selected_partner_codes)}).fetchall()
            partner_names = [row[0] for row in codes_result]

        try:
            # Query para productos en el conjunto
            # ✅ MEJORA: Mostrar TODOS los productos partners disponibles (aunque no tengamos ventas)
            # ❌ Para NO-partners: Solo mostrar los que SÍ vendimos (no todo el catálogo)
            products_query = text(
                f"""
                SELECT
                    pc.national_code,
                    pc.nomen_nombre,
                    pc.nomen_laboratorio,
                    pc.nomen_pvp,
                    pc.nomen_precio_referencia,
                    pc.nomen_tipo_farmaco,
                    COALESCE(SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END), 0) as units_sold,
                    COALESCE(SUM(sd.total_amount), 0) as revenue_sold,
                    COUNT(sd.id) as transactions_count,
                    CASE
                        WHEN pc.nomen_laboratorio IN :selected_partners THEN 'partner'
                        ELSE 'non_partner'
                    END as partner_status,
                    -- 3-tier fallback: individual product PVL → product precio_referencia → group calculated_pvl
                    -- Resolves Issue #343: Ticagrelor products with NULL individual prices use group PVL
                    COALESCE(SUM(CASE
                        WHEN pc.nomen_codigo_laboratorio NOT IN :selected_partner_codes
                             AND COALESCE(pc.nomen_pvl_calculado, pc.nomen_precio_referencia, hgm.calculated_pvl) > 0
                        THEN (CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) * COALESCE(pc.nomen_pvl_calculado, pc.nomen_precio_referencia, hgm.calculated_pvl)
                        ELSE 0
                    END), 0) as individual_savings_base
                FROM product_catalog pc
                LEFT JOIN homogeneous_groups_master hgm ON pc.nomen_codigo_homogeneo = hgm.homogeneous_code
                LEFT JOIN sales_enrichment se ON se.product_catalog_id = pc.id
                LEFT JOIN sales_data sd ON sd.id = se.sales_data_id
                    AND sd.pharmacy_id = :pharmacy_id
                    AND sd.sale_date >= :start_date
                    AND sd.sale_date <= :end_date
                    {employee_filter}
                WHERE pc.nomen_codigo_homogeneo = :homogeneous_code
                  AND pc.nomen_estado = 'ALTA'
                  AND (
                    -- PARTNERS: Mostrar todos (con o sin ventas)
                    pc.nomen_laboratorio IN :selected_partners
                    OR
                    -- NO-PARTNERS: Solo los que vendimos (sd.id IS NOT NULL)
                    (pc.nomen_laboratorio NOT IN :selected_partners AND sd.id IS NOT NULL)
                  )
                GROUP BY
                    pc.national_code,
                    pc.nomen_nombre,
                    pc.nomen_laboratorio,
                    pc.nomen_pvp,
                    pc.nomen_precio_referencia,
                    pc.nomen_tipo_farmaco
                ORDER BY COALESCE(SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END), 0) DESC,
                         CASE WHEN pc.nomen_laboratorio IN :selected_partners THEN 0 ELSE 1 END,
                         pc.nomen_laboratorio
                """
            )

            query_params = {
                "pharmacy_id": str(pharmacy_id),
                "start_date": start_date,
                "end_date": end_date,
                "homogeneous_code": homogeneous_code,
                "selected_partner_codes": tuple(selected_partner_codes),
                "selected_partners": tuple(partner_names),  # Añadir parámetro faltante
            }

            # Issue #402: Agregar employee_names solo si está presente
            # DECISIÓN UX #2: Lista vacía [] o None = TODOS (no agregar parámetro)
            if selected_employee_names and len(selected_employee_names) > 0:
                query_params["employee_names"] = selected_employee_names

            products_result = db.execute(products_query, query_params).fetchall()

            # Resumen del conjunto
            total_units = sum(p.units_sold for p in products_result)
            total_revenue = sum(p.revenue_sold for p in products_result)
            partner_units = sum(p.units_sold for p in products_result if p.partner_status == "partner")
            opportunity_units = sum(p.units_sold for p in products_result if p.partner_status == "non_partner")
            total_savings_base = sum(p.individual_savings_base for p in products_result)

            penetration = round((partner_units / total_units) * 100, 2) if total_units > 0 else 0.0

            # Información de laboratorios en el conjunto
            labs_in_group = {}
            for product in products_result:
                lab = product.nomen_laboratorio
                if lab not in labs_in_group:
                    labs_in_group[lab] = {
                        "laboratory_name": lab,
                        "is_partner": lab in partner_names,
                        "products_count": 0,
                        "total_units": 0,
                        "total_revenue": 0.0,
                    }

                labs_in_group[lab]["products_count"] += 1
                labs_in_group[lab]["total_units"] += product.units_sold
                labs_in_group[lab]["total_revenue"] += float(product.revenue_sold)

            group_detail = {
                "pharmacy_id": str(pharmacy_id),
                "homogeneous_code": homogeneous_code,
                "selected_partner_codes": selected_partner_codes,
                "analysis_period": {
                    "start_date": start_date.isoformat(),
                    "end_date": end_date.isoformat(),
                    "months": period,
                },
                "group_summary": {
                    "total_products": len(products_result),
                    "total_units": total_units,
                    "total_revenue": float(total_revenue),
                    "partner_units": partner_units,
                    "opportunity_units": opportunity_units,
                    "partner_penetration": penetration,
                    "total_savings_base": float(total_savings_base),
                },
                "products_detail": [
                    {
                        "national_code": row.national_code,
                        "product_name": row.nomen_nombre,
                        "laboratory": row.nomen_laboratorio,
                        "pvp": float(row.nomen_pvp or 0),
                        "precio_referencia": float(row.nomen_precio_referencia or 0),
                        "drug_type": row.nomen_tipo_farmaco,
                        "units_sold": row.units_sold,
                        "revenue_sold": float(row.revenue_sold),
                        "transactions_count": row.transactions_count,
                        "is_partner": row.partner_status == "partner",
                        "individual_savings_base": float(row.individual_savings_base),
                    }
                    for row in products_result
                ],
                "laboratories_in_group": list(labs_in_group.values()),
                "analysis_type": "homogeneous_group_detail",
                "calculated_at": utc_now().isoformat(),
            }

            logger.info(
                f"[PARTNER_ANALYSIS] Detalle conjunto {homogeneous_code}: {len(products_result)} productos, {penetration}% penetración"
            )

            return group_detail

        except Exception as e:
            logger.error(f"[PARTNER_ANALYSIS] Error en detalle conjunto: {e}")
            raise

    # ========== MÉTODOS AUXILIARES ==========

    def _resolve_date_range(
        self,
        start_date_str: Optional[str],
        end_date_str: Optional[str],
        period_months: Optional[int],
        pharmacy_id: Optional[UUID] = None,
        db: Optional[Session] = None,
    ) -> Tuple[datetime, datetime]:
        """
        Resuelve el rango de fechas para análisis.

        Issue #xxx: Prioridad a fechas directas sobre period_months para consistencia
        con el treemap que ya usa fechas directas.

        Args:
            start_date_str: Fecha inicio (YYYY-MM-DD) - tiene prioridad
            end_date_str: Fecha fin (YYYY-MM-DD) - tiene prioridad
            period_months: Fallback - meses hacia atrás desde hoy
            pharmacy_id: Para aplicar restricción FREE tier
            db: Sesión de BD

        Returns:
            Tupla (start_date, end_date) como datetime objects
        """
        from datetime import timezone

        # Si tenemos fechas directas, usarlas
        if start_date_str and end_date_str:
            try:
                start_date = datetime.fromisoformat(start_date_str[:10])
                end_date = datetime.fromisoformat(end_date_str[:10])

                # Asegurar timezone-aware (UTC)
                if start_date.tzinfo is None:
                    start_date = start_date.replace(tzinfo=timezone.utc)
                if end_date.tzinfo is None:
                    end_date = end_date.replace(tzinfo=timezone.utc)

                logger.info(
                    f"[PARTNER_ANALYSIS] Using direct date range: {start_date.date()} to {end_date.date()}"
                )

                # Aplicar restricción FREE tier si corresponde
                if pharmacy_id and db:
                    start_date, end_date = self._apply_free_tier_restriction(
                        start_date, end_date, pharmacy_id, db
                    )

                return start_date, end_date

            except ValueError as e:
                logger.warning(f"[PARTNER_ANALYSIS] Invalid date format: {e}. Falling back to period_months")

        # Fallback: usar period_months (comportamiento legacy)
        period = period_months or self.default_period_months
        return self._get_analysis_period(period, pharmacy_id, db)

    def _apply_free_tier_restriction(
        self,
        start_date: datetime,
        end_date: datetime,
        pharmacy_id: UUID,
        db: Session,
    ) -> Tuple[datetime, datetime]:
        """
        Aplica restricción FREE tier a un rango de fechas.

        REGLA #18 (Issue #142): Usuarios FREE limitados a 3 meses desde última venta.
        """
        from app.models.user import User

        user = db.query(User).filter(User.pharmacy_id == pharmacy_id).first()
        if user:
            free_tier_date_limit = get_data_date_limit_for_user(user, db)
            if free_tier_date_limit:
                original_start = start_date
                start_date = max(start_date, free_tier_date_limit)

                if start_date != original_start:
                    logger.info(
                        f"[PARTNER_ANALYSIS] FREE tier restriction applied: "
                        f"start_date adjusted from {original_start.date()} to {start_date.date()}"
                    )

        return start_date, end_date

    def _get_analysis_period(
        self, months: int, pharmacy_id: Optional[UUID] = None, db: Optional[Session] = None
    ) -> Tuple[datetime, datetime]:
        """
        Calcula período de análisis, aplicando restricciones de subscription tier.

        Args:
            months: Número de meses de análisis
            pharmacy_id: ID de farmacia (opcional, para aplicar restricción FREE)
            db: Sesión de BD (opcional, requerida si pharmacy_id está presente)

        Returns:
            Tupla (start_date, end_date)

        Note:
            REGLA #18 (Issue #142): Usuarios FREE limitados a 3 meses desde última venta
        """
        end_date = utc_now()
        start_date = end_date - timedelta(days=months * 30)

        # REGLA #18 (Issue #142): Restricción de 3 meses para usuarios FREE
        if pharmacy_id and db:
            from app.models.user import User

            user = db.query(User).filter(User.pharmacy_id == pharmacy_id).first()
            if user:
                free_tier_date_limit = get_data_date_limit_for_user(user, db)
                if free_tier_date_limit:
                    # Usuario FREE: aplicar límite de 3 meses desde última venta
                    original_start = start_date
                    start_date = max(start_date, free_tier_date_limit)

                    if start_date != original_start:
                        logger.info(
                            f"[PARTNER_ANALYSIS] FREE tier restriction applied for pharmacy {pharmacy_id}: "
                            f"start_date adjusted from {original_start.date()} to {start_date.date()} "
                            f"(3 months from last sale)"
                        )

        return start_date, end_date

    def _empty_dynamic_analysis(self, pharmacy_id: UUID, start_date: datetime, end_date: datetime, discount_pct: float = 10.0) -> Dict[str, Any]:
        """Retorna análisis dinámico vacío cuando no hay partners seleccionados"""
        # Calcular meses del período
        months_diff = (end_date.year - start_date.year) * 12 + (end_date.month - start_date.month)

        # Warning cuando el fallback se activa
        if months_diff <= 0:
            logger.warning(
                f"Invalid date range for pharmacy {pharmacy_id}: "
                f"start={start_date.isoformat()}, end={end_date.isoformat()}, "
                f"months_diff={months_diff}. Using 12-month fallback."
            )
            months_diff = 12

        return {
            "pharmacy_id": str(pharmacy_id),
            "selected_partner_codes": [],
            "analysis_period": {
                "start_date": start_date.isoformat(),
                "end_date": end_date.isoformat(),
                "months": months_diff,
            },
            "analyzable_universe": {
                "groups_count": 0,
                "total_units": 0,
                "total_revenue": 0.0,
                "total_transactions": 0,
            },
            "partner_performance": {
                "partner_units": 0,
                "partner_revenue": 0.0,
                "partner_transactions": 0,
                "penetration_percentage": 0.0,
            },
            "opportunity_metrics": {
                "opportunity_units": 0,
                "opportunity_revenue": 0.0,
                "opportunity_transactions": 0,
                "potential_savings_base": 0.0,
                "estimated_savings_with_discount": 0.0,  # Campo requerido por schema
                "applied_discount_percentage": discount_pct,  # Usar el valor pasado como parámetro
            },
            "homogeneous_groups_detail": [],
            "analysis_type": "partner_dynamic",
            "calculated_at": utc_now().isoformat(),
        }

    def _calculate_estimated_savings(
        self, pharmacy_id: UUID, potential_savings_base: Optional[float], discount_pct: float
    ) -> float:
        """
        Calcula el ahorro estimado aplicando el descuento.

        Issue #339: Validación explícita con logging detallado para diagnosticar
        casos donde el cálculo podría retornar 0 incorrectamente.

        Args:
            pharmacy_id: ID de la farmacia (para logging)
            potential_savings_base: PVL de ventas no-partners (puede ser None)
            discount_pct: Tasa de descuento a aplicar (ej: 10.0 para 10%)

        Returns:
            Ahorro estimado con descuento aplicado (0.0 si base <= 0 o None)
        """
        # Logging PRE-cálculo
        logger.debug(
            f"[SAVINGS_CALC] pharmacy_id={pharmacy_id}, "
            f"potential_base={potential_savings_base}, "
            f"discount_pct={discount_pct}, "
            f"type_base={type(potential_savings_base)}"
        )

        # Validación explícita
        if potential_savings_base is None or potential_savings_base <= 0:
            logger.debug("[SAVINGS_CALC] Base is None or <= 0, returning 0.0")
            return 0.0

        # Cálculo con conversión explícita
        base_float = float(potential_savings_base)
        discount_rate = discount_pct / 100
        estimated_savings = base_float * discount_rate

        # Logging POST-cálculo
        logger.debug(
            f"[SAVINGS_CALC] estimated_savings={estimated_savings:.2f} "
            f"(base={base_float:.2f} × rate={discount_rate:.2f})"
        )

        return estimated_savings

    def _format_period_label(self, period_start: datetime, level: str) -> str:
        """Formatea etiqueta del período según nivel temporal"""
        if level == "quarter":
            quarter = (period_start.month - 1) // 3 + 1
            return f"Q{quarter} {period_start.year}"
        elif level == "month":
            return period_start.strftime("%b %Y")
        elif level == "fortnight":
            # ✅ FIX: Formato quincenas españolas (1ª quincena: días 1-15, 2ª quincena: días 16-fin)
            day = period_start.day
            if day <= 15:
                return f"1ª Quincena {period_start.strftime('%b %Y')}"
            else:
                return f"2ª Quincena {period_start.strftime('%b %Y')}"
        else:
            return period_start.strftime("%Y-%m-%d")

    # ========== PARTNER PRODUCT REFERENCES ==========

    def get_partner_product_references(
        self,
        db: Session,
        pharmacy_id: UUID,
        selected_partner_codes: List[str],
        homogeneous_code: Optional[str] = None,
        period_months: int = None,
        selected_employee_names: Optional[List[str]] = None,  # Issue #402
    ) -> Dict[str, Any]:
        """
        Obtiene referencias de productos vendidos de partners seleccionados.

        Muestra tabla detallada con:
        - Código nacional, descripción, laboratorio
        - Unidades vendidas, ventas totales
        - Opcionalmente filtrado por conjunto homogéneo

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de farmacia
            selected_partners: Partners seleccionados para mostrar
            homogeneous_code: Código conjunto homogéneo opcional para filtrar
            period_months: Período análisis (default: 12 meses)
            selected_employee_names: Lista de nombres de empleados a filtrar (None o vacío = TODOS)

        Returns:
            Lista de productos de partners con métricas de venta

        Note:
            Issue #402: Employee filtering (PRO feature)
            DECISIÓN #2: Filtrado solo por employee_name (no employee_code)
            DECISIÓN UX #2: Lista vacía o None = TODOS los empleados (default)
            DECISIÓN UX #3: Incluye ventas sin empleado como "__sin_empleado__"
        """
        logger.info(
            f"[PARTNER_ANALYSIS] Referencias productos partners para farmacia {pharmacy_id}, partners: {selected_partner_codes}"
        )

        period = period_months or self.default_period_months
        start_date, end_date = self._get_analysis_period(period, pharmacy_id, db)

        if not selected_partner_codes:
            return {
                "pharmacy_id": str(pharmacy_id),
                "selected_partner_codes": [],
                "homogeneous_code": homogeneous_code,
                "analysis_period": {
                    "start_date": start_date.isoformat(),
                    "end_date": end_date.isoformat(),
                    "months": period,
                },
                "total_products": 0,
                "total_units": 0,
                "total_sales": 0.0,
                "products": [],
                "analysis_type": "partner_product_references",
                "calculated_at": utc_now().isoformat(),
            }

        try:
            # Base query con filtro opcional por conjunto homogéneo
            homogeneous_filter = ""
            if homogeneous_code:
                homogeneous_filter = "AND pc.nomen_codigo_homogeneo = :homogeneous_code"

            # Issue #402: Construir filtro de empleados
            employee_filter = ""
            if selected_employee_names:
                employee_filter = "AND COALESCE(sd.employee_name, '__sin_empleado__') = ANY(:employee_names)"

            # ✅ CORRECCIÓN: Construir placeholders IN dinámicamente
            # Para evitar problemas con ANY() y arrays en SQLAlchemy text()
            placeholders = ", ".join([f":partner_{i}" for i in range(len(selected_partner_codes))])

            products_query = text(
                f"""
                WITH partner_products AS (
                    SELECT
                        pc.national_code,
                        pc.cima_nombre_comercial as product_name,
                        pc.nomen_laboratorio as laboratory,
                        pc.nomen_codigo_homogeneo as homogeneous_code,
                        pc.nomen_nombre_homogeneo as homogeneous_name,
                        SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) as units_sold,
                        SUM(sd.total_amount) as total_sales,
                        COUNT(DISTINCT sd.id) as transactions_count,
                        AVG(sd.total_amount / NULLIF(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END, 0)) as avg_price
                    FROM sales_data sd
                    JOIN sales_enrichment se ON sd.id = se.sales_data_id
                    JOIN product_catalog pc ON se.product_catalog_id = pc.id
                    WHERE sd.pharmacy_id = :pharmacy_id
                      AND sd.sale_date >= :start_date
                      AND sd.sale_date <= :end_date
                      AND pc.nomen_codigo_laboratorio IN ({placeholders})
                      {homogeneous_filter}
                      {employee_filter}
                    GROUP BY
                        pc.national_code,
                        pc.cima_nombre_comercial,
                        pc.nomen_laboratorio,
                        pc.nomen_codigo_homogeneo,
                        pc.nomen_nombre_homogeneo
                )
                SELECT
                    national_code,
                    COALESCE(product_name, 'Producto sin nombre') as product_name,
                    laboratory,
                    homogeneous_code,
                    homogeneous_name,
                    units_sold,
                    total_sales,
                    transactions_count,
                    avg_price
                FROM partner_products
                ORDER BY units_sold DESC, total_sales DESC
            """
            )

            # Parámetros base con placeholders dinámicos para IN clause
            params = {
                "pharmacy_id": str(pharmacy_id),
                "start_date": start_date,
                "end_date": end_date,
            }

            # Añadir parámetros individuales para cada partner (evita problemas con arrays)
            for i, code in enumerate(selected_partner_codes):
                params[f"partner_{i}"] = code

            # Añadir filtro de conjunto homogéneo si aplica
            if homogeneous_code:
                params["homogeneous_code"] = homogeneous_code

            # Issue #402: Agregar employee_names solo si está presente
            if selected_employee_names:
                params["employee_names"] = selected_employee_names

            result = db.execute(products_query, params).fetchall()

            # Procesar resultados
            products = []
            total_units = 0
            total_sales = 0.0

            for row in result:
                product = {
                    "national_code": row.national_code,
                    "product_name": row.product_name,
                    "laboratory": row.laboratory,
                    "homogeneous_code": row.homogeneous_code,
                    "homogeneous_name": row.homogeneous_name,
                    "units_sold": row.units_sold or 0,
                    "total_sales": float(row.total_sales or 0),
                    "transactions_count": row.transactions_count or 0,
                    "avg_price": float(row.avg_price or 0),
                }
                products.append(product)
                total_units += product["units_sold"]
                total_sales += product["total_sales"]

            response = {
                "pharmacy_id": str(pharmacy_id),
                "selected_partner_codes": selected_partner_codes,
                "homogeneous_code": homogeneous_code,
                "analysis_period": {
                    "start_date": start_date.isoformat(),
                    "end_date": end_date.isoformat(),
                    "months": period,
                },
                "total_products": len(products),
                "total_units": total_units,
                "total_sales": total_sales,
                "products": products,
                "analysis_type": "partner_product_references",
                "calculated_at": utc_now().isoformat(),
            }

            logger.info(
                f"[PARTNER_ANALYSIS] Referencias obtenidas: {len(products)} productos, {total_units} unidades totales"
            )

            return response

        except Exception as e:
            logger.error(f"[PARTNER_ANALYSIS] Error obteniendo referencias productos partners: {e}")
            raise


    # ========== CONTEXT TREEMAP (Issue #415) ==========

    def _get_treemap_cache_key(
        self,
        pharmacy_id: UUID,
        start_date: Optional[datetime],
        end_date: Optional[datetime],
        employee_names: Optional[List[str]],
    ) -> str:
        """
        Genera clave de cache única para el treemap (Issue #443).

        Args:
            pharmacy_id: ID de la farmacia
            start_date: Fecha inicio
            end_date: Fecha fin
            employee_names: Filtro de empleados

        Returns:
            Clave de cache única basada en parámetros
        """
        # Normalizar parámetros para generar hash consistente
        start_str = start_date.strftime("%Y-%m-%d") if start_date else "default"
        end_str = end_date.strftime("%Y-%m-%d") if end_date else "default"
        emp_str = ",".join(sorted(employee_names)) if employee_names else "all"

        # Hash de los parámetros para evitar claves muy largas
        params_hash = hashlib.md5(
            f"{start_str}:{end_str}:{emp_str}".encode(),
            usedforsecurity=False
        ).hexdigest()[:12]

        return f"treemap:{pharmacy_id}:{params_hash}"

    async def _get_cached_treemap(self, cache_key: str) -> Optional[Dict[str, Any]]:
        """
        Obtiene datos de treemap desde cache Redis (Issue #443).

        Args:
            cache_key: Clave de cache

        Returns:
            Datos cacheados o None si no hay cache/expirado
        """
        try:
            from app.services.enrichment_cache import enrichment_cache

            if not enrichment_cache.enabled:
                return None

            cached = await enrichment_cache.get(cache_key)
            if cached:
                logger.info(f"[TREEMAP_CACHE] HIT for key {cache_key[:30]}...")
                return cached

            logger.debug(f"[TREEMAP_CACHE] MISS for key {cache_key[:30]}...")
            return None

        except Exception as e:
            logger.warning(f"[TREEMAP_CACHE] Error reading cache: {e}")
            return None

    async def _set_cached_treemap(self, cache_key: str, data: Dict[str, Any]) -> None:
        """
        Guarda datos de treemap en cache Redis (Issue #443).

        Args:
            cache_key: Clave de cache
            data: Datos a cachear
        """
        try:
            from app.services.enrichment_cache import enrichment_cache

            if not enrichment_cache.enabled:
                return

            await enrichment_cache.set(cache_key, data, ttl=CONTEXT_TREEMAP_CACHE_TTL)
            logger.info(f"[TREEMAP_CACHE] SET key {cache_key[:30]}... TTL={CONTEXT_TREEMAP_CACHE_TTL}s")

        except Exception as e:
            logger.warning(f"[TREEMAP_CACHE] Error setting cache: {e}")

    async def invalidate_treemap_cache(self, pharmacy_id: UUID) -> int:
        """
        Invalida todo el cache de treemap para una farmacia (Issue #443).

        Se debe llamar cuando:
        - Se sube un nuevo archivo de ventas
        - Se modifican los partners seleccionados
        - Se ejecuta re-enrichment

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            Número de claves invalidadas
        """
        try:
            from app.services.enrichment_cache import enrichment_cache

            if not enrichment_cache.enabled or not enrichment_cache.redis:
                logger.debug("[TREEMAP_CACHE] Cache not enabled, skipping invalidation")
                return 0

            # Buscar todas las claves de treemap para esta farmacia
            pattern = f"treemap:{pharmacy_id}:*"
            cursor = 0
            deleted_count = 0

            while True:
                cursor, keys = await enrichment_cache.redis.scan(
                    cursor, match=pattern, count=100
                )

                if keys:
                    deleted_count += await enrichment_cache.redis.delete(*keys)

                if cursor == 0:
                    break

            if deleted_count > 0:
                logger.info(
                    f"[TREEMAP_CACHE] Invalidated {deleted_count} cache entries for pharmacy {pharmacy_id}"
                )

            return deleted_count

        except Exception as e:
            logger.warning(f"[TREEMAP_CACHE] Error invalidating cache for pharmacy {pharmacy_id}: {e}")
            return 0

    def get_context_treemap_data(
        self,
        db: Session,
        pharmacy_id: UUID,
        start_date: Optional[datetime] = None,
        end_date: Optional[datetime] = None,
        employee_names: Optional[List[str]] = None,
    ) -> Dict[str, Any]:
        """
        Obtiene datos jerárquicos para treemap de contexto de ventas (Issue #415, #426).

        Issue #443: Implementa cache Redis con TTL de 5 minutos para optimizar performance.
        Cache se invalida automáticamente cuando hay nuevo upload de ventas.

        Jerarquía del treemap (Issue #426 - reestructurada):
        - Total Ventas (100%)
          - Prescripción (requiere receta)
            - Sustituible (conjuntos con AL MENOS un genérico disponible)
              - Analizable (cubierta por partners seleccionados)
                - Ya Partners (ventas actuales de partners)
                - Oportunidad (ventas NO-partners en conjuntos analizables)
              - No Analizable (sin partners que cubran estos conjuntos)
            - No Sustituible (sin genéricos disponibles)
          - Venta Libre (no requiere receta)

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            start_date: Fecha inicio análisis (opcional, default: 24 meses atrás)
            end_date: Fecha fin análisis (opcional, default: hoy)
            employee_names: Filtro de empleados (Issue #402, PRO feature)

        Returns:
            Datos estructurados para Plotly treemap con hierarchy y summary

        Note:
            Issue #402: Employee filtering (PRO feature)
            Issue #426: Jerarquía Prescripción/Venta Libre como primer nivel
            Issue #443: Cache Redis para optimización de performance
            DECISIÓN UX #2: Lista vacía o None = TODOS los empleados (default)
        """
        # Issue #443: Intentar obtener datos de cache primero
        cache_key = self._get_treemap_cache_key(pharmacy_id, start_date, end_date, employee_names)

        # Intentar obtener de cache (sync wrapper para método async)
        try:
            loop = asyncio.get_event_loop()
            if loop.is_running():
                # Ya estamos en un event loop (FastAPI async context)
                import concurrent.futures
                with concurrent.futures.ThreadPoolExecutor() as executor:
                    future = executor.submit(
                        asyncio.run,
                        self._get_cached_treemap(cache_key)
                    )
                    cached_data = future.result(timeout=2)
            else:
                cached_data = asyncio.run(self._get_cached_treemap(cache_key))

            if cached_data:
                # Cache hit - retornar datos cacheados
                cached_data["_cache_hit"] = True
                return cached_data
        except Exception as e:
            logger.debug(f"[TREEMAP_CACHE] Cache lookup failed, computing: {e}")

        logger.info(f"[PARTNER_ANALYSIS] Calculando context treemap para farmacia {pharmacy_id}")

        # Determinar período de análisis
        if not end_date:
            end_date = utc_now()
        if not start_date:
            # Default: últimos 24 meses para análisis de contexto
            start_date = end_date - timedelta(days=24 * 30)

        # Aplicar restricción FREE tier si aplica (REGLA #18)
        start_date, end_date = self._apply_free_tier_restriction(
            start_date, end_date, pharmacy_id, db
        )

        # Construir filtro de empleados
        employee_filter = ""
        params: Dict[str, Any] = {
            "pharmacy_id": str(pharmacy_id),
            "start_date": start_date,
            "end_date": end_date,
        }

        if employee_names and len(employee_names) > 0:
            employee_filter = "AND COALESCE(sd.employee_name, '__sin_empleado__') = ANY(:employee_names)"
            params["employee_names"] = employee_names

        try:
            # Obtener partners seleccionados de la farmacia
            selected_partner_codes = self._get_pharmacy_selected_partner_codes(db, pharmacy_id)

            # ================================================================
            # Issue #426: Nueva jerarquía Prescripción/Venta Libre
            # ================================================================

            # 1. Total de ventas y desglose por tipo de dispensación
            total_and_type_query = text(
                f"""
                SELECT
                    COALESCE(SUM(sd.total_amount), 0) as total_revenue,
                    COALESCE(SUM(CASE
                        WHEN pc.xfarma_prescription_category IS NOT NULL THEN sd.total_amount
                        ELSE 0
                    END), 0) as prescription_revenue,
                    COALESCE(SUM(CASE
                        WHEN pc.xfarma_prescription_category IS NULL
                        THEN sd.total_amount
                        ELSE 0
                    END), 0) as otc_revenue
                FROM sales_data sd
                LEFT JOIN sales_enrichment se ON sd.id = se.sales_data_id
                LEFT JOIN product_catalog pc ON se.product_catalog_id = pc.id
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND sd.sale_date >= :start_date
                  AND sd.sale_date <= :end_date
                  {employee_filter}
                """
            )
            total_result = db.execute(total_and_type_query, params).fetchone()
            total_sales = float(total_result.total_revenue or 0)
            prescription_sales = float(total_result.prescription_revenue or 0)
            otc_sales = float(total_result.otc_revenue or 0)

            # 2. Dentro de PRESCRIPCIÓN: Sustituible vs No Sustituible
            prescription_breakdown_query = text(
                f"""
                WITH generic_homogeneous AS MATERIALIZED (
                    -- Conjuntos homogéneos que tienen AL MENOS un genérico disponible
                    SELECT DISTINCT pc.nomen_codigo_homogeneo
                    FROM product_catalog pc
                    WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                      AND pc.nomen_codigo_homogeneo != ''
                      AND pc.nomen_tipo_farmaco = 'GENERICO'
                      AND pc.nomen_estado = 'ALTA'
                )
                SELECT
                    -- Sustituible dentro de prescripción
                    COALESCE(SUM(CASE
                        WHEN pc.nomen_codigo_homogeneo IN (SELECT nomen_codigo_homogeneo FROM generic_homogeneous)
                        THEN sd.total_amount
                        ELSE 0
                    END), 0) as rx_substitutable_revenue,
                    -- No sustituible dentro de prescripción
                    COALESCE(SUM(CASE
                        WHEN pc.nomen_codigo_homogeneo IS NULL
                          OR pc.nomen_codigo_homogeneo = ''
                          OR pc.nomen_codigo_homogeneo NOT IN (SELECT nomen_codigo_homogeneo FROM generic_homogeneous)
                        THEN sd.total_amount
                        ELSE 0
                    END), 0) as rx_non_substitutable_revenue
                FROM sales_data sd
                LEFT JOIN sales_enrichment se ON sd.id = se.sales_data_id
                LEFT JOIN product_catalog pc ON se.product_catalog_id = pc.id
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND sd.sale_date >= :start_date
                  AND sd.sale_date <= :end_date
                  AND pc.xfarma_prescription_category IS NOT NULL
                  {employee_filter}
                """
            )
            rx_breakdown = db.execute(prescription_breakdown_query, params).fetchone()
            rx_substitutable_sales = float(rx_breakdown.rx_substitutable_revenue or 0)
            rx_non_substitutable_sales = float(rx_breakdown.rx_non_substitutable_revenue or 0)

            # 3. Dentro de PRESCRIPCIÓN > SUSTITUIBLE: Analizable (Ya Partners, Oportunidad, No Analizable)
            rx_analyzable_sales = 0.0
            rx_partner_sales = 0.0
            rx_opportunity_sales = 0.0
            rx_non_analyzable_sales = rx_substitutable_sales  # Default si no hay partners

            if selected_partner_codes:
                params["selected_partner_codes"] = tuple(selected_partner_codes)

                rx_analyzable_query = text(
                    f"""
                    WITH generic_homogeneous AS MATERIALIZED (
                        SELECT DISTINCT pc.nomen_codigo_homogeneo
                        FROM product_catalog pc
                        WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                          AND pc.nomen_codigo_homogeneo != ''
                          AND pc.nomen_tipo_farmaco = 'GENERICO'
                          AND pc.nomen_estado = 'ALTA'
                    ),
                    partner_availability AS MATERIALIZED (
                        SELECT DISTINCT pc.nomen_codigo_homogeneo
                        FROM product_catalog pc
                        WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                          AND pc.nomen_codigo_laboratorio IN :selected_partner_codes
                          AND pc.nomen_estado = 'ALTA'
                    )
                    SELECT
                        COALESCE(SUM(sd.total_amount), 0) as analyzable_revenue,
                        COALESCE(SUM(CASE
                            WHEN pc.nomen_codigo_laboratorio IN :selected_partner_codes
                            THEN sd.total_amount ELSE 0
                        END), 0) as partner_revenue,
                        COALESCE(SUM(CASE
                            WHEN pc.nomen_codigo_laboratorio NOT IN :selected_partner_codes
                            THEN sd.total_amount ELSE 0
                        END), 0) as opportunity_revenue
                    FROM sales_data sd
                    JOIN sales_enrichment se ON sd.id = se.sales_data_id
                    JOIN product_catalog pc ON se.product_catalog_id = pc.id
                    JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                    JOIN partner_availability pa ON pc.nomen_codigo_homogeneo = pa.nomen_codigo_homogeneo
                    WHERE sd.pharmacy_id = :pharmacy_id
                      AND sd.sale_date >= :start_date
                      AND sd.sale_date <= :end_date
                      AND pc.xfarma_prescription_category IS NOT NULL
                      {employee_filter}
                    """
                )
                rx_analyzable_result = db.execute(rx_analyzable_query, params).fetchone()
                rx_analyzable_sales = float(rx_analyzable_result.analyzable_revenue or 0)
                rx_partner_sales = float(rx_analyzable_result.partner_revenue or 0)
                rx_opportunity_sales = float(rx_analyzable_result.opportunity_revenue or 0)
                rx_non_analyzable_sales = rx_substitutable_sales - rx_analyzable_sales

            # Variables de compatibilidad para summary (mantener nombres existentes)
            substitutable_sales = rx_substitutable_sales
            non_substitutable_sales = rx_non_substitutable_sales
            analyzable_sales = rx_analyzable_sales
            non_analyzable_sales = rx_non_analyzable_sales
            partner_sales = rx_partner_sales
            opportunity_sales = rx_opportunity_sales

            # Calcular porcentajes
            def safe_percentage(value: float, total: float) -> float:
                """Calcula porcentaje de forma segura (evita división por cero)"""
                return round((value / total) * 100, 2) if total > 0 else 0.0

            # Construir estructura del treemap para Plotly
            # Formato compatible con go.Treemap (labels, parents, values, text)
            # Issue #426: Jerarquía Prescripción/Venta Libre como primer nivel
            treemap_data = {
                "labels": [
                    "Total Ventas",
                    "Prescripción",       # Nivel 1: Requiere receta
                    "Venta Libre",        # Nivel 1: Sin receta
                    "Sustituible",        # Nivel 2 (dentro de Prescripción)
                    "No Sustituible",     # Nivel 2 (dentro de Prescripción)
                    "Analizable",         # Nivel 3 (dentro de Sustituible)
                    "No Analizable",      # Nivel 3 (dentro de Sustituible)
                    "Ya Partners",        # Nivel 4 (dentro de Analizable)
                    "Oportunidad",        # Nivel 4 (dentro de Analizable)
                ],
                "parents": [
                    "",                   # Total Ventas es root
                    "Total Ventas",       # Prescripción → Total
                    "Total Ventas",       # Venta Libre → Total
                    "Prescripción",       # Sustituible → Prescripción
                    "Prescripción",       # No Sustituible → Prescripción
                    "Sustituible",        # Analizable → Sustituible
                    "Sustituible",        # No Analizable → Sustituible
                    "Analizable",         # Ya Partners → Analizable
                    "Analizable",         # Oportunidad → Analizable
                ],
                "values": [
                    total_sales,
                    prescription_sales,
                    otc_sales,
                    rx_substitutable_sales,
                    rx_non_substitutable_sales,
                    rx_analyzable_sales,
                    rx_non_analyzable_sales,
                    rx_partner_sales,
                    rx_opportunity_sales,
                ],
                "text": [
                    f"EUR{total_sales:,.0f} (100%)",
                    f"EUR{prescription_sales:,.0f} ({safe_percentage(prescription_sales, total_sales)}%)",
                    f"EUR{otc_sales:,.0f} ({safe_percentage(otc_sales, total_sales)}%)",
                    f"EUR{rx_substitutable_sales:,.0f} ({safe_percentage(rx_substitutable_sales, prescription_sales)}%)",
                    f"EUR{rx_non_substitutable_sales:,.0f} ({safe_percentage(rx_non_substitutable_sales, prescription_sales)}%)",
                    f"EUR{rx_analyzable_sales:,.0f} ({safe_percentage(rx_analyzable_sales, rx_substitutable_sales)}%)",
                    f"EUR{rx_non_analyzable_sales:,.0f} ({safe_percentage(rx_non_analyzable_sales, rx_substitutable_sales)}%)",
                    f"EUR{rx_partner_sales:,.0f} ({safe_percentage(rx_partner_sales, rx_analyzable_sales)}%)",
                    f"EUR{rx_opportunity_sales:,.0f} ({safe_percentage(rx_opportunity_sales, rx_analyzable_sales)}%)",
                ],
                # Colores siguiendo Design System
                "colors": [
                    "#6c757d",  # Total: gris neutro
                    "#6F42C1",  # Prescripción: púrpura (pharmacy-specific)
                    "#FD7E14",  # Venta Libre: naranja (diferenciación)
                    "#0d6efd",  # Sustituible: azul primario
                    "#adb5bd",  # No Sustituible: gris claro
                    "#17a2b8",  # Analizable: cyan/info
                    "#6c757d",  # No Analizable: gris medio
                    "#28a745",  # Ya Partners: verde success
                    "#ffc107",  # Oportunidad: amarillo warning (acción)
                ],
            }

            # Estructura jerárquica alternativa (para visualizaciones custom)
            # Issue #426: Nueva jerarquía con Prescripción/Venta Libre como primer nivel
            hierarchy = {
                "name": "Total Ventas",
                "value": total_sales,
                "percentage": 100.0,
                "children": [
                    {
                        "name": "Prescripción",
                        "value": prescription_sales,
                        "percentage": safe_percentage(prescription_sales, total_sales),
                        "children": [
                            {
                                "name": "Sustituible",
                                "value": rx_substitutable_sales,
                                "percentage": safe_percentage(rx_substitutable_sales, prescription_sales),
                                "children": [
                                    {
                                        "name": "Analizable",
                                        "value": rx_analyzable_sales,
                                        "percentage": safe_percentage(rx_analyzable_sales, rx_substitutable_sales),
                                        "children": [
                                            {
                                                "name": "Ya Partners",
                                                "value": rx_partner_sales,
                                                "percentage": safe_percentage(rx_partner_sales, rx_analyzable_sales),
                                            },
                                            {
                                                "name": "Oportunidad",
                                                "value": rx_opportunity_sales,
                                                "percentage": safe_percentage(rx_opportunity_sales, rx_analyzable_sales),
                                            },
                                        ],
                                    },
                                    {
                                        "name": "No Analizable",
                                        "value": rx_non_analyzable_sales,
                                        "percentage": safe_percentage(rx_non_analyzable_sales, rx_substitutable_sales),
                                    },
                                ],
                            },
                            {
                                "name": "No Sustituible",
                                "value": rx_non_substitutable_sales,
                                "percentage": safe_percentage(rx_non_substitutable_sales, prescription_sales),
                            },
                        ],
                    },
                    {
                        "name": "Venta Libre",
                        "value": otc_sales,
                        "percentage": safe_percentage(otc_sales, total_sales),
                    },
                ],
            }

            # Summary con métricas clave
            # Issue #426: Actualizado para nueva jerarquía
            summary = {
                "total_sales": total_sales,
                # Nivel 1: Prescripción vs Venta Libre
                "prescription_sales": prescription_sales,
                "prescription_percentage": safe_percentage(prescription_sales, total_sales),
                "otc_sales": otc_sales,
                "otc_percentage": safe_percentage(otc_sales, total_sales),
                # Nivel 2 (dentro de Prescripción): Sustituible vs No Sustituible
                "substitutable_sales": substitutable_sales,
                "substitutable_percentage": safe_percentage(substitutable_sales, prescription_sales),
                "non_substitutable_sales": non_substitutable_sales,
                "non_substitutable_percentage": safe_percentage(non_substitutable_sales, prescription_sales),
                # Nivel 3 (dentro de Sustituible): Analizable vs No Analizable
                "analyzable_sales": analyzable_sales,
                "analyzable_percentage": safe_percentage(analyzable_sales, substitutable_sales),
                "non_analyzable_sales": non_analyzable_sales,
                "non_analyzable_percentage": safe_percentage(non_analyzable_sales, substitutable_sales),
                # Nivel 4 (dentro de Analizable): Ya Partners vs Oportunidad
                "partner_sales": partner_sales,
                "partner_percentage": safe_percentage(partner_sales, analyzable_sales),
                "opportunity_sales": opportunity_sales,
                "opportunity_percentage": safe_percentage(opportunity_sales, analyzable_sales),
            }

            response = {
                "pharmacy_id": str(pharmacy_id),
                "treemap_data": treemap_data,
                "hierarchy": hierarchy,
                "summary": summary,
                "period": {
                    "start_date": start_date.strftime("%Y-%m-%d"),
                    "end_date": end_date.strftime("%Y-%m-%d"),
                },
                "filters_applied": {
                    "employee_names": employee_names if employee_names else "all",
                    "selected_partner_codes": selected_partner_codes,
                },
                "analysis_type": "context_treemap",
                "calculated_at": utc_now().isoformat(),
            }

            logger.info(
                f"[PARTNER_ANALYSIS] Context treemap completado (Issue #426): "
                f"Total={total_sales:.2f}, Prescripción={prescription_sales:.2f}, "
                f"OTC={otc_sales:.2f}, Sustituible={substitutable_sales:.2f}, "
                f"Analizable={analyzable_sales:.2f}, Oportunidad={opportunity_sales:.2f}"
            )

            # Issue #443: Guardar resultado en cache para futuras requests
            try:
                loop = asyncio.get_event_loop()
                if loop.is_running():
                    import concurrent.futures
                    with concurrent.futures.ThreadPoolExecutor() as executor:
                        executor.submit(
                            asyncio.run,
                            self._set_cached_treemap(cache_key, response)
                        )
                else:
                    asyncio.run(self._set_cached_treemap(cache_key, response))
            except Exception as cache_error:
                logger.debug(f"[TREEMAP_CACHE] Failed to cache result: {cache_error}")

            return response

        except Exception as e:
            logger.error(f"[PARTNER_ANALYSIS] Error calculando context treemap: {e}")
            raise

    def _get_pharmacy_selected_partner_codes(
        self, db: Session, pharmacy_id: UUID
    ) -> List[str]:
        """
        Obtiene los códigos de laboratorios partners seleccionados para la farmacia.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia

        Returns:
            Lista de códigos de laboratorios partners seleccionados
        """
        from app.models.pharmacy_partners import PharmacyPartner

        partners = (
            db.query(PharmacyPartner)
            .filter(
                PharmacyPartner.pharmacy_id == pharmacy_id,
                PharmacyPartner.is_selected == True,  # noqa: E712
            )
            .all()
        )

        if not partners:
            logger.warning(f"[PARTNER_ANALYSIS] No selected partners found for pharmacy {pharmacy_id}")
            return []

        # Obtener códigos de laboratorio desde el catálogo usando los nombres
        lab_names = [p.laboratory_name for p in partners]

        if not lab_names:
            return []

        codes_query = text(
            """
            SELECT DISTINCT nomen_codigo_laboratorio
            FROM product_catalog
            WHERE nomen_laboratorio IN :lab_names
              AND nomen_codigo_laboratorio IS NOT NULL
              AND nomen_codigo_laboratorio != ''
            """
        )
        codes_result = db.execute(codes_query, {"lab_names": tuple(lab_names)}).fetchall()

        return [row[0] for row in codes_result]


# Instancia global del servicio
partner_analysis_service = PartnerAnalysisService()
