﻿"""
Servicio de notificaciones inteligentes para administradores de farmacia.
FASE 5: Sistema proactivo de alertas específicas para el contexto farmacéutico español.
"""

import os
from datetime import timedelta
from typing import Dict, Optional

import structlog
from sqlalchemy.orm import Session

from app.models.product_catalog import ProductCatalog
from app.models.system_health import PerformanceSnapshot, SystemAlert
from app.models.system_status import SystemComponent, SystemStatus, SystemStatusEnum
from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)


class NotificationService:
    """
    Servicio de notificaciones inteligentes específicas para farmacias españolas.
    Diseñado para minimizar ruido y maximizar utilidad para administradores no técnicos.
    """

    def __init__(self, db: Session):
        self.db = db
        self.notification_rules = self._load_notification_rules()

    def _load_notification_rules(self) -> Dict:
        """
        Carga las reglas de notificación específicas para farmacia.
        """
        return {
            # Alertas críticas para farmacia
            "cima_outdated": {
                "threshold_hours": 240,  # 10 días
                "severity": "high",
                "title": "Base CIMA Desactualizada",
                "description": "La base de datos CIMA no se ha actualizado en más de 10 días",
                "impact": "Los precios y datos de medicamentos pueden estar obsoletos",
                "action": "Ejecutar sincronización CIMA desde el panel de administración",
                "frequency_limit_hours": 24,  # Solo una notificación por día
            },
            "nomenclator_stale": {
                "threshold_hours": 168,  # 7 días
                "severity": "medium",
                "title": "Nomenclátor Ministerio Desactualizado",
                "description": "Los datos oficiales del Ministerio no se han actualizado en más de 7 días",
                "impact": "La clasificación de genéricos puede estar desactualizada",
                "action": "Ejecutar sincronización de Nomenclátor",
                "frequency_limit_hours": 12,
            },
            "product_mismatch": {
                "threshold_percent": 85,
                "severity": "medium",
                "title": "Baja Cobertura de Productos",
                "description": "Menos del 85% de productos tienen datos CIMA completos",
                "impact": "Los análisis de genéricos y precios pueden ser incompletos",
                "action": "Revisar y ejecutar re-enriquecimiento de catálogo",
                "frequency_limit_hours": 6,
            },
            "sync_failure": {
                "threshold_count": 3,
                "severity": "high",
                "title": "Múltiples Fallos de Sincronización",
                "description": "Se han detectado múltiples errores consecutivos en sincronización",
                "impact": "El sistema puede no estar actualizado correctamente",
                "action": "Revisar logs en modo desarrollador y contactar soporte técnico",
                "frequency_limit_hours": 2,
            },
            "memory_warning": {
                "threshold_percent": 80,  # 80% en Render
                "severity": "high",
                "title": "Uso Alto de Memoria",
                "description": "El sistema está usando más del 80% de memoria disponible",
                "impact": "Posibles ralentizaciones o fallos en operaciones pesadas",
                "action": "Considerar ejecutar limpieza de memoria o contactar soporte",
                "frequency_limit_hours": 1,
            },
            "api_errors": {
                "threshold_count": 10,
                "severity": "medium",
                "title": "Errores en APIs Externas",
                "description": "Se han detectado múltiples errores en APIs de CIMA o Ministerio",
                "impact": "Las sincronizaciones pueden fallar temporalmente",
                "action": "Esperar a que se resuelvan los problemas externos",
                "frequency_limit_hours": 4,
            },
        }

    def check_and_send_notifications(self) -> Dict:
        """
        Ejecuta todos los chequeos y envía notificaciones necesarias.
        Diseñado para ejecutarse periódicamente (cada 30 minutos).
        """
        logger.info("Starting intelligent notification check")

        results = {
            "checks_performed": 0,
            "notifications_sent": 0,
            "alerts_created": 0,
            "checks": {},
        }

        try:
            # Chequeo 1: Frescura del catálogo CIMA
            cima_result = self._check_cima_freshness()
            results["checks"]["cima_freshness"] = cima_result
            results["checks_performed"] += 1

            # Chequeo 2: Frescura del Nomenclátor
            nomenclator_result = self._check_nomenclator_freshness()
            results["checks"]["nomenclator_freshness"] = nomenclator_result
            results["checks_performed"] += 1

            # Chequeo 3: Cobertura de productos
            coverage_result = self._check_product_coverage()
            results["checks"]["product_coverage"] = coverage_result
            results["checks_performed"] += 1

            # Chequeo 4: Fallos de sincronización
            sync_result = self._check_sync_failures()
            results["checks"]["sync_failures"] = sync_result
            results["checks_performed"] += 1

            # Chequeo 5: Uso de memoria
            memory_result = self._check_memory_usage()
            results["checks"]["memory_usage"] = memory_result
            results["checks_performed"] += 1

            # Chequeo 6: Errores en APIs externas
            api_result = self._check_api_errors()
            results["checks"]["api_errors"] = api_result
            results["checks_performed"] += 1

            logger.info(
                "Notification check completed",
                checks_performed=results["checks_performed"],
                notifications_sent=results["notifications_sent"],
            )

        except Exception as e:
            logger.error("Error during notification check", error=str(e))
            results["error"] = str(e)

        return results

    def _check_cima_freshness(self) -> Dict:
        """
        Chequea si la base CIMA está actualizada.
        """
        try:
            rule = self.notification_rules["cima_outdated"]

            # Buscar último estado de CIMA
            cima_status = self.db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()

            if not cima_status or not cima_status.updated_at:
                # No hay información de sincronización
                self._create_notification_if_needed(
                    "cima_outdated",
                    "No hay información de sincronización CIMA",
                    {"reason": "no_sync_info"},
                    "high",
                )
                return {"status": "error", "message": "No sync info"}

            hours_since_update = (utc_now() - cima_status.updated_at).total_seconds() / 3600

            if hours_since_update > rule["threshold_hours"]:
                # CIMA desactualizado
                context = {
                    "hours_since_update": hours_since_update,
                    "last_update": cima_status.updated_at.isoformat(),
                    "threshold_hours": rule["threshold_hours"],
                }

                self._create_notification_if_needed(
                    "cima_outdated",
                    f"CIMA no actualizado en {hours_since_update:.1f} horas",
                    context,
                    rule["severity"],
                )

                return {"status": "outdated", "hours": hours_since_update}

            return {"status": "ok", "hours": hours_since_update}

        except Exception as e:
            logger.error("Error checking CIMA freshness", error=str(e))
            return {"status": "error", "error": str(e)}

    def _check_nomenclator_freshness(self) -> Dict:
        """
        Chequea si el Nomenclátor está actualizado.
        """
        try:
            rule = self.notification_rules["nomenclator_stale"]

            nomenclator_status = self.db.query(SystemStatus).filter_by(component=SystemComponent.NOMENCLATOR).first()

            if not nomenclator_status or not nomenclator_status.updated_at:
                self._create_notification_if_needed(
                    "nomenclator_stale",
                    "No hay información de sincronización Nomenclátor",
                    {"reason": "no_sync_info"},
                    "medium",
                )
                return {"status": "error", "message": "No sync info"}

            hours_since_update = (utc_now() - nomenclator_status.updated_at).total_seconds() / 3600

            if hours_since_update > rule["threshold_hours"]:
                context = {
                    "hours_since_update": hours_since_update,
                    "last_update": nomenclator_status.updated_at.isoformat(),
                    "threshold_hours": rule["threshold_hours"],
                }

                self._create_notification_if_needed(
                    "nomenclator_stale",
                    f"Nomenclátor no actualizado en {hours_since_update:.1f} horas",
                    context,
                    rule["severity"],
                )

                return {"status": "outdated", "hours": hours_since_update}

            return {"status": "ok", "hours": hours_since_update}

        except Exception as e:
            logger.error("Error checking Nomenclator freshness", error=str(e))
            return {"status": "error", "error": str(e)}

    def _check_product_coverage(self) -> Dict:
        """
        Chequea la cobertura de productos en el catálogo.
        """
        try:
            rule = self.notification_rules["product_mismatch"]

            total_products = self.db.query(ProductCatalog).count()
            if total_products == 0:
                return {"status": "empty", "coverage": 0}

            with_cima = self.db.query(ProductCatalog).filter(ProductCatalog.cima_national_code.isnot(None)).count()

            coverage_percent = (with_cima / total_products) * 100

            if coverage_percent < rule["threshold_percent"]:
                context = {
                    "coverage_percent": coverage_percent,
                    "total_products": total_products,
                    "with_cima": with_cima,
                    "threshold_percent": rule["threshold_percent"],
                }

                self._create_notification_if_needed(
                    "product_mismatch",
                    f"Cobertura CIMA baja: {coverage_percent:.1f}%",
                    context,
                    rule["severity"],
                )

                return {"status": "low_coverage", "coverage": coverage_percent}

            return {"status": "ok", "coverage": coverage_percent}

        except Exception as e:
            logger.error("Error checking product coverage", error=str(e))
            return {"status": "error", "error": str(e)}

    def _check_sync_failures(self) -> Dict:
        """
        Chequea fallos consecutivos de sincronización.
        """
        try:
            rule = self.notification_rules["sync_failure"]

            # Buscar errores de sincronización en las últimas 24 horas
            since = utc_now() - timedelta(hours=24)

            error_count = (
                self.db.query(SystemStatus)
                .filter(
                    SystemStatus.updated_at >= since,
                    SystemStatus.status == SystemStatusEnum.ERROR,
                )
                .count()
            )

            if error_count >= rule["threshold_count"]:
                context = {
                    "error_count": error_count,
                    "threshold_count": rule["threshold_count"],
                    "period_hours": 24,
                }

                self._create_notification_if_needed(
                    "sync_failure",
                    f"{error_count} fallos de sincronización en 24h",
                    context,
                    rule["severity"],
                )

                return {"status": "multiple_failures", "count": error_count}

            return {"status": "ok", "count": error_count}

        except Exception as e:
            logger.error("Error checking sync failures", error=str(e))
            return {"status": "error", "error": str(e)}

    def _check_memory_usage(self) -> Dict:
        """
        Chequea el uso actual de memoria.
        """
        try:
            rule = self.notification_rules["memory_warning"]

            # Obtener último snapshot de performance
            latest_snapshot = (
                self.db.query(PerformanceSnapshot).order_by(PerformanceSnapshot.snapshot_at.desc()).first()
            )

            if not latest_snapshot:
                return {"status": "no_data"}

            if not latest_snapshot.memory_usage_mb or not latest_snapshot.memory_available_mb:
                return {"status": "incomplete_data"}

            total_memory = latest_snapshot.memory_usage_mb + latest_snapshot.memory_available_mb
            usage_percent = (latest_snapshot.memory_usage_mb / total_memory) * 100

            # En Render, usar umbral más bajo
            is_render = os.getenv("RENDER") == "true"
            threshold = rule["threshold_percent"] if is_render else 90

            if usage_percent > threshold:
                context = {
                    "usage_percent": usage_percent,
                    "usage_mb": latest_snapshot.memory_usage_mb,
                    "available_mb": latest_snapshot.memory_available_mb,
                    "threshold_percent": threshold,
                    "is_render": is_render,
                }

                self._create_notification_if_needed(
                    "memory_warning",
                    f"Memoria alta: {usage_percent:.1f}%",
                    context,
                    rule["severity"],
                )

                return {"status": "high_usage", "usage_percent": usage_percent}

            return {"status": "ok", "usage_percent": usage_percent}

        except Exception as e:
            logger.error("Error checking memory usage", error=str(e))
            return {"status": "error", "error": str(e)}

    def _check_api_errors(self) -> Dict:
        """
        Chequea errores en APIs externas.
        """
        try:
            rule = self.notification_rules["api_errors"]

            # Obtener último snapshot para verificar errores de API
            latest_snapshot = (
                self.db.query(PerformanceSnapshot).order_by(PerformanceSnapshot.snapshot_at.desc()).first()
            )

            if not latest_snapshot:
                return {"status": "no_data"}

            error_count = latest_snapshot.api_errors_last_hour or 0

            if error_count >= rule["threshold_count"]:
                context = {
                    "error_count": error_count,
                    "threshold_count": rule["threshold_count"],
                    "period": "última hora",
                }

                self._create_notification_if_needed(
                    "api_errors",
                    f"{error_count} errores de API en última hora",
                    context,
                    rule["severity"],
                )

                return {"status": "high_errors", "count": error_count}

            return {"status": "ok", "count": error_count}

        except Exception as e:
            logger.error("Error checking API errors", error=str(e))
            return {"status": "error", "error": str(e)}

    def _create_notification_if_needed(
        self, alert_type: str, message: str, context: Dict, severity: str
    ) -> Optional[SystemAlert]:
        """
        Crea una notificación solo si no existe una reciente del mismo tipo.
        """
        try:
            rule = self.notification_rules.get(alert_type, {})
            frequency_limit = rule.get("frequency_limit_hours", 24)

            # Buscar alertas recientes del mismo tipo
            since = utc_now() - timedelta(hours=frequency_limit)
            recent_alert = (
                self.db.query(SystemAlert)
                .filter(
                    SystemAlert.alert_type == alert_type,
                    SystemAlert.created_at >= since,
                    SystemAlert.is_active.is_(True),
                )
                .first()
            )

            if recent_alert:
                # Ya existe una alerta reciente, solo actualizar contador
                recent_alert.occurrence_count += 1
                recent_alert.last_occurred = utc_now()
                recent_alert.updated_at = utc_now()
                self.db.commit()

                logger.debug(
                    "Updated existing alert",
                    alert_type=alert_type,
                    occurrences=recent_alert.occurrence_count,
                )

                return recent_alert

            # Crear nueva alerta
            alert = SystemAlert(
                alert_type=alert_type,
                severity=severity,
                title=rule.get("title", f"Alerta {alert_type}"),
                message=message,
                context=context,
            )

            self.db.add(alert)
            self.db.commit()

            # Enviar notificación si está configurado
            self._send_notification_if_configured(alert, rule)

            logger.info(
                "Created new alert",
                alert_type=alert_type,
                severity=severity,
                title=alert.title,
            )

            return alert

        except Exception as e:
            logger.error("Error creating notification", error=str(e), alert_type=alert_type)
            self.db.rollback()
            return None

    def _send_notification_if_configured(self, alert: SystemAlert, rule: Dict):
        """
        Envía la notificación por los canales configurados.
        """
        try:
            # Por ahora, solo logging estructurado
            # En el futuro se pueden añadir:
            # - Emails
            # - Webhooks
            # - Push notifications
            # - Slack/Teams

            logger.warning(
                "PHARMACY ALERT",
                alert_type=alert.alert_type,
                severity=alert.severity,
                title=alert.title,
                message=alert.message,
                impact=rule.get("impact", ""),
                action=rule.get("action", ""),
                context=alert.context,
            )

            # TODO: Implementar email notifications
            # self._send_email_notification(alert, rule)

            # TODO: Implementar push notifications
            # self._send_push_notification(alert, rule)

        except Exception as e:
            logger.error("Error sending notification", error=str(e), alert_id=str(alert.id))

    def get_active_alerts_summary(self) -> Dict:
        """
        Obtiene un resumen de alertas activas para el dashboard.
        """
        try:
            active_alerts = (
                self.db.query(SystemAlert)
                .filter(SystemAlert.is_active.is_(True), SystemAlert.is_acknowledged.is_(False))
                .order_by(SystemAlert.last_occurred.desc())
                .all()
            )

            summary = {
                "total_active": len(active_alerts),
                "by_severity": {"critical": 0, "high": 0, "medium": 0, "low": 0},
                "recent_alerts": [],
            }

            for alert in active_alerts:
                summary["by_severity"][alert.severity] += 1

                if len(summary["recent_alerts"]) < 5:
                    summary["recent_alerts"].append(
                        {
                            "id": str(alert.id),
                            "type": alert.alert_type,
                            "severity": alert.severity,
                            "title": alert.title,
                            "message": alert.message,
                            "count": alert.occurrence_count,
                            "last_occurred": alert.last_occurred.isoformat(),
                        }
                    )

            return summary

        except Exception as e:
            logger.error("Error getting alerts summary", error=str(e))
            return {"total_active": 0, "by_severity": {}, "recent_alerts": []}

    def acknowledge_alert(self, alert_id: str, acknowledged_by: str) -> bool:
        """
        Reconoce una alerta específica.
        """
        try:
            alert = self.db.query(SystemAlert).filter(SystemAlert.id == alert_id).first()

            if not alert:
                return False

            alert.is_acknowledged = True
            alert.acknowledged_by = acknowledged_by
            alert.acknowledged_at = utc_now()
            alert.updated_at = utc_now()

            self.db.commit()

            logger.info(
                "Alert acknowledged",
                alert_id=alert_id,
                alert_type=alert.alert_type,
                acknowledged_by=acknowledged_by,
            )

            return True

        except Exception as e:
            logger.error("Error acknowledging alert", error=str(e), alert_id=alert_id)
            self.db.rollback()
            return False


def schedule_notification_checks():
    """
    Función para programar chequeos periódicos de notificaciones.
    Diseñada para ser llamada por un scheduler (cron, celery, etc.).
    """
    from app.database import get_db

    try:
        db = next(get_db())
        notification_service = NotificationService(db)
        results = notification_service.check_and_send_notifications()

        logger.info("Scheduled notification check completed", results=results)

        return results

    except Exception as e:
        logger.error("Error in scheduled notification check", error=str(e))
        return {"error": str(e)}

    finally:
        if "db" in locals():
            db.close()
