"""
Servicio de Clasificación NECESIDAD para Productos VentaLibre.

Issue #446 - Enfoque "¿Qué me das para...?"
Issue #476 - Unificación de clasificadores (migrado desde notebooks/)

Este es el ÚNICO clasificador autorizado para productos VentaLibre.
Categorías orientadas a la necesidad del cliente, no al formato del producto.

PUNTO DE CONVERGENCIA: Tanto ventas como inventario usan este clasificador.

Uso:
    from app.services.necesidad_classifier_service import (
        classify_product,
        extract_necesidad,
        get_necesidad_label,
    )

    # API principal (retorna categoría + confianza)
    category, confidence = classify_product("FRENADOL COMPLEX")
    # Returns: ("congestion_nasal", 0.75)

    # API legacy (solo categoría, para compatibilidad)
    category = extract_necesidad("FRENADOL COMPLEX")
    # Returns: "congestion_nasal"

================================================================================
WORKFLOW PARA AÑADIR NUEVOS MAPPINGS
================================================================================

1. IDENTIFICAR productos candidatos CON FILTRO OTC:

    SELECT sd.product_name, se.product_type, SUM(sd.total_amount) as revenue
    FROM sales_data sd
    JOIN sales_enrichment se ON sd.id = se.sales_data_id
    WHERE se.product_type = 'free_sale'  -- CRÍTICO: Solo OTC
    GROUP BY sd.product_name, se.product_type
    ORDER BY revenue DESC
    LIMIT 100;

2. VERIFICAR producto contra BD antes de añadir:

    from app.database import SessionLocal
    from app.models import SalesData
    from app.models.sales_enrichment import SalesEnrichment
    from sqlalchemy import func

    db = SessionLocal()
    result = db.query(
        SalesEnrichment.product_type,
        func.count(SalesEnrichment.id)
    ).join(
        SalesData, SalesData.id == SalesEnrichment.sales_data_id
    ).filter(
        func.lower(SalesData.product_name).contains('nombre_producto')
    ).group_by(SalesEnrichment.product_type).all()

    # Si product_type='prescription' → NO AÑADIR

3. REGLA: Solo añadir productos con product_type='free_sale' en BD

================================================================================
"""

from typing import Tuple

# =============================================================================
# CATEGORÍAS DE NECESIDAD
# =============================================================================

NECESIDADES = {
    # CAPILAR
    "caida_cabello": "Caída del cabello / Alopecia",
    "caspa": "Caspa / Descamación",
    "cabello_graso": "Cabello graso",
    "cabello_seco": "Cabello seco / Dañado",

    # FACIAL - PIEL
    "arrugas_antiedad": "Arrugas / Antiedad",
    "manchas": "Manchas / Hiperpigmentación",
    "acne": "Acné / Imperfecciones",
    "rojeces_rosacea": "Rojeces / Rosácea",
    "piel_seca": "Piel seca / Deshidratación",
    "piel_sensible": "Piel sensible / Irritada",
    "piel_atopica": "Piel atópica / Dermatitis",
    "proteccion_solar": "Protección solar",
    "limpieza_facial": "Limpieza facial",

    # CORPORAL
    "dolor_muscular": "Dolor muscular / Articular",
    "circulacion_piernas": "Circulación / Piernas cansadas",
    "cicatrices_estrias": "Cicatrices / Estrías",
    "quemaduras": "Quemaduras / Aftersun",
    "hongos_pies": "Hongos / Micosis pies",
    "sudoracion": "Sudoración excesiva",

    # OCULAR
    "ojo_seco": "Ojo seco / Lágrimas artificiales",
    "fatiga_visual": "Fatiga visual",

    # DENTAL
    "protesis_dental": "Prótesis dental / Fijación",
    "encias": "Encías sensibles / Sangrado",
    "sensibilidad_dental": "Sensibilidad dental / Dientes sensibles",
    "higiene_dental": "Higiene dental general",
    "blanqueamiento": "Blanqueamiento dental",
    "interdental": "Cepillos interdentales / Higiene interproximal",
    "caries": "Prevención caries",
    "aftas": "Aftas / Llagas bucales",

    # BEBE / INFANTIL
    "alimentacion_bebe": "Alimentación bebé / Fórmulas",
    "colicos_bebe": "Cólicos bebé / Digestivo infantil",
    "irritacion_panal": "Irritación pañal / Culito bebé",
    "higiene_bebe": "Higiene bebé",
    "fiebre_infantil": "Fiebre / Dolor infantil",

    # DIGESTIVO
    "estrenimiento": "Estreñimiento",
    "acidez_reflujo": "Acidez / Reflujo",
    "gases_digestion": "Gases / Digestión pesada",
    "flora_intestinal": "Flora intestinal / Probióticos",
    "diarrea": "Diarrea",

    # RESPIRATORIO
    "tos_garganta": "Tos / Garganta irritada",
    "caramelos_garganta": "Caramelos garganta / Pastillas",
    "congestion_nasal": "Congestión nasal / Resfriado",
    "alergia": "Alergia",
    "defensas_inmunidad": "Defensas / Sistema inmune",

    # ENERGIA / SISTEMA NERVIOSO
    "energia_cansancio": "Energía / Cansancio",
    "sueno_insomnio": "Sueño / Insomnio",
    "estres_ansiedad": "Estrés / Ansiedad",
    "memoria_concentracion": "Memoria / Concentración",

    # MUJER
    "menopausia": "Menopausia / Sofocos",
    "embarazo_prenatal": "Embarazo / Prenatal",
    "lactancia": "Lactancia",
    "cistitis": "Cistitis / Infección urinaria",
    "higiene_intima": "Higiene íntima",
    "sequedad_vaginal": "Sequedad vaginal",

    # CARDIOVASCULAR / METABOLICO
    "colesterol": "Colesterol alto",
    "tension_arterial": "Tensión arterial",
    "glucosa": "Control glucosa",

    # PIOJOS / PARASITOS
    "piojos": "Piojos / Liendres",
    "repelente_insectos": "Repelente insectos",
    "parasitos": "Parásitos intestinales / Lombrices",

    # HERIDAS / PRIMEROS AUXILIOS
    "heridas_apositos": "Heridas / Apósitos",
    "herpes_labial": "Herpes labial",

    # VITAMINAS / SUPLEMENTOS
    "vitaminas_general": "Vitaminas / Multivitamínico",
    "hierro": "Anemia / Hierro",
    "omega3": "Omega 3 / Salud cardiovascular",
    "colageno": "Colágeno / Articulaciones",
    "magnesio": "Magnesio / Relajación muscular",
    "calcio_huesos": "Calcio / Salud ósea",
    "probioticos_general": "Probióticos generales",

    # DISPOSITIVOS / EQUIPAMIENTO
    "tension_medicion": "Medición tensión arterial",
    "glucosa_medicion": "Medición glucosa",
    "ortopedia": "Ortopedia / Movilidad",
    "test_diagnostico": "Test diagnóstico",

    # DENTAL AVANZADO
    "cepillo_electrico": "Cepillos eléctricos / Irrigadores",
    "irrigador_dental": "Irrigador dental",
    "blanqueamiento_dental": "Blanqueamiento dental profesional",

    # COSMÉTICA / CUIDADO PERSONAL
    "cuidado_facial_general": "Cuidado facial general",
    "cuidado_corporal": "Cuidado corporal / Hidratación",
    "perfumeria": "Perfumería / Fragancias",
    "maquillaje": "Maquillaje / Cosmética color",
    "afeitado": "Afeitado / Grooming",

    # LENTES / OPTICA
    "lentes_contacto": "Lentes de contacto / Líquidos",
    "optica": "Óptica / Gafas lectura",

    # CONTROL PESO
    "perdida_peso": "Control de peso / Adelgazar",
    "drenante": "Drenante / Retención líquidos",

    # NUTRICIÓN DEPORTIVA
    "deporte_recuperacion": "Deporte / Recuperación",

    # NUTRICIÓN CLÍNICA / HOSPITALARIA
    "nutricion_clinica": "Nutrición clínica / Suplementos médicos",
    "fertilidad": "Fertilidad / Reproducción",

    # HIGIENE PERSONAL
    "higiene_cabello": "Higiene capilar / Champú",
    "higiene_corporal": "Higiene corporal / Gel ducha",
    "desodorante": "Desodorante / Antitranspirante",

    # PIEL - ADICIONALES
    "picaduras": "Picaduras / Calmar irritación",
    "unas": "Uñas / Fortalecimiento",

    # HIGIENE PERSONAL - ADICIONALES
    "higiene_manos": "Higiene manos / Jabones",
    "accesorios_belleza": "Accesorios / Utensilios belleza",

    # SALUD SEXUAL
    "salud_sexual": "Salud sexual / Preservativos",

    # MATERIAL SANITARIO
    "material_sanitario": "Material sanitario / Fungibles",
    "dispositivos_medicos": "Dispositivos médicos / Aparatos",

    # CUIDADO DEPENDIENTES
    "cuidado_dependientes": "Cuidado personas dependientes",

    # VETERINARIA
    "veterinaria": "Productos veterinarios / Mascotas",

    # HOMEOPATÍA
    "homeopatia": "Homeopatía / Medicina alternativa",

    # JOYERIA
    "joyeria": "Joyería / Pendientes",

    # OTROS
    "termometro": "Termómetro / Medición temperatura",
    "test_embarazo": "Test embarazo",
    "incontinencia": "Incontinencia urinaria",
    "ostomia": "Ostomía",
    "vendaje_compresion": "Vendaje / Compresión",

    # SERVICIOS (a filtrar)
    "servicio": "Servicio farmacia",

    # =========================================================================
    # NUEVAS CATEGORÍAS (Fase 5b - Q33-Q65)
    # =========================================================================

    # OIDOS
    "proteccion_auditiva": "Protección auditiva / Tapones oídos",
    "higiene_oidos": "Higiene oídos / Limpieza ótica",

    # FACIAL ESPECÍFICO
    "contorno_ojos": "Contorno de ojos",
    "labios": "Labios / Bálsamo labial",

    # TERMOTERAPIA
    "termoterapia": "Termoterapia / Almohadillas eléctricas",

    # OCULAR GENERAL
    "salud_ocular": "Salud ocular / Colirios",

    # NUTRICION DEPORTIVA (expandido)
    "nutricion_deportiva": "Nutrición deportiva / Suplementos deporte",

    # MINERALES (separado de vitaminas)
    "minerales": "Minerales / Oligoelementos",

    # COMPLEMENTOS FARMACIA
    "complementos": "Complementos / Merchandising",

    # PAÑALES (separado de higiene)
    "panales": "Pañales / Absorbentes bebé",

    # =========================================================================
    # NUEVAS CATEGORÍAS (Fase 5c - Q66-Q79)
    # =========================================================================

    # HIGADO
    "detox_higado": "Detox hepático / Salud hígado",

    # OIDOS ESPECIFICO
    "acufenos": "Acúfenos / Tinnitus",

    # RENAL
    "renal": "Salud renal / Riñón",

    # MANOS
    "manos": "Cuidado manos / Cremas manos",

    # NEUROPATIA (Q87)
    "neuropatia": "Neuropatía / Dolor nervioso",

    # HEMORROIDES (Lote 100)
    "hemorroides": "Hemorroides / Almorranas",

    # PIES (Lote LLM #3)
    "pies_talones": "Cuidado pies / Talones agrietados",

    # MAREO (Lote LLM #3)
    "mareo_viaje": "Mareo / Cinetosis",

    # DIABETES (Lote LLM #3)
    "diabetes": "Diabetes / Control glucémico",

    # BEBÉS/INFANTIL GENERAL
    "bebes_infantil": "Bebés / Cuidado infantil general",

    # PRIMEROS AUXILIOS (Lote LLM #3)
    "primeros_auxilios": "Primeros auxilios / Curas",

    # QUEMADURAS AFTERSUN (Lote LLM #3)
    "quemaduras_aftersun": "Quemaduras / Aftersun",

    # DEFENSAS (alias)
    "defensas": "Defensas / Inmunidad",

    # DERMATITIS (alias)
    "dermatitis": "Dermatitis / Piel irritada",

    # =========================================================================
    # NUEVAS CATEGORÍAS (Fase 6 - Medicamentos OTC)
    # =========================================================================

    # GRIPE / RESFRIADO
    "gripe_resfriado": "Gripe / Resfriado / Catarro",

    # DOLOR GARGANTA (diferente de tos_garganta - más específico)
    "dolor_garganta": "Dolor de garganta / Faringitis",

    # DOLOR / ANALGÉSICOS (diferente de dolor_muscular - más general)
    "dolor_analgesico": "Dolor / Analgésicos / Antiinflamatorios",

    # ANTICONCEPCIÓN EMERGENCIA
    "anticoncepcion": "Anticoncepción de emergencia / Píldora del día después",

    # TABAQUISMO
    "dejar_fumar": "Dejar de fumar / Deshabituación tabáquica",

    # VERRUGAS / CALLOS
    "verrugas_callos": "Verrugas / Callos / Ojos de gallo",

    # FALLBACK
    "otros": "Otros / Sin clasificar",
}


# =============================================================================
# KEYWORDS POR NECESIDAD
# =============================================================================

NECESIDAD_KEYWORDS = {
    # CAPILAR
    "caida_cabello": [
        "anticaida", "anti-caida", "caida", "alopecia", "olistic", "pilexil",
        "iraltone", "tricovit", "aminexil", "neogenic", "triphasic", "aminohair",
        "vitalfan", "klorane anticaida", "dercos aminexil", "anacaps",
        "keratincaps", "keratina", "keracnyl hair", "moncho moreno"
    ],
    "caspa": [
        "caspa", "anticaspa", "descamacion", "kelual", "squanorm", "node ds",
        "pitiriasis"
    ],

    # FACIAL
    "arrugas_antiedad": [
        "antiedad", "antiarrugas", "anti-age", "retinol", "retinal", "colageno",
        "reafirmante", "lifting", "pro-collagen", "peptido", "age reverse",
        "vital age", "filorga", "endocare", "martiderm", "esthederm",
        "isdinceutics", "a.g.e.", "anti-wrinkle", "firmeza", "flacidez"
    ],
    "manchas": [
        "despigmentante", "antimanchas", "manchas", "melasma", "pigment",
        "bella aurora", "neoretin", "melascreen", "azelac ru", "spot prevent",
        "c-vit", "vitamina c", "radiance", "luminosidad"
    ],
    "acne": [
        "acne", "granos", "espinillas", "imperfecciones", "effaclar", "sebium",
        "cleanance", "normaderm", "keracnyl", "acniben", "seborreica"
    ],
    "rojeces_rosacea": [
        "rosacea", "rojeces", "cuperosis", "rosacure", "rosaderm", "anti-rougeurs",
        "antirougeurs", "sensibio ar"
    ],
    "piel_seca": [
        "piel seca", "hidratante", "hidratacion", "xerosis", "nutritiva",
        "funcion barrera", "hydrabio", "hydraphase", "xeracalm", "vea lipo",
        "vea olio", "aceite emoliente", "emoliente"
    ],
    "piel_sensible": [
        "piel sensible", "calmante", "reparador", "toleriane", "sensibio",
        "cicaplast", "bepanthol", "avene", "cicabio", "cicatrizante",
        "reparadora", "cica cream", "gel bioadhesivo"
    ],
    "piel_atopica": [
        "atopica", "atopico", "atoderm", "lipikar", "xeracalm", "exomega",
        "topialyse", "svr topialyse", "a-derma exomega"
    ],
    "proteccion_solar": [
        "solar", "spf", "fps", "fotoprotector", "heliocare", "fusion water",
        "sunscreen", "protector solar", "eryfotona", "ak-nmsc"
    ],
    "limpieza_facial": [
        "limpiador", "desmaquillante", "micelar", "gel limpiador", "espuma limpiadora",
        "gel moussant", "limpieza facial"
    ],

    # OCULAR
    "ojo_seco": [
        "ojo seco", "lagrimas", "thealoz", "hylo", "optiben", "aquoral",
        "lubristil", "systane", "gotas ojos", "sequedad ocular", "relive",
        "ojos secos", "lephanet", "blephasol", "blefaritis", "ozonest",
        "toallitas oftalmicas", "solucion oftalmica", "gotas oftalmicas"
    ],
    "fatiga_visual": [
        "fatiga visual", "oftan", "macula", "ojos cansados",
        "luteina", "zeaxantina", "nutrof", "visirelax"
    ],

    # DENTAL
    "protesis_dental": [
        "protesis", "corega", "kukident", "dentadura", "adhesivo dental", "fijacion dental"
    ],
    "encias": [
        "encias", "gingivitis", "sangrado", "gingilacer", "bexident encias",
        "parodontax", "paroex", "oralkin", "gum activital", "phb total",
        "enjuague bucal", "colutorio"
    ],
    "aftas": [
        "aftas", "llagas", "aftaclear", "afta clear"
    ],

    # BEBE
    "alimentacion_bebe": [
        "blemil", "pedialac", "nutriben", "almiron", "nan optipro", "hero baby",
        "formula infantil", "leche continuacion", "lata 800"
    ],
    "colicos_bebe": [
        "colico", "colimil", "bivos", "reuteri", "colico lactante"
    ],
    "irritacion_panal": [
        "panal", "mitosyl", "culito", "nutraisdin pomada", "pasta al agua",
        "pomada protectora"
    ],
    "higiene_bebe": [
        "gel bebe", "jabon bebe", "champu bebe", "bano bebe", "biberon",
        "tetina", "chupete", "suavinex", "nuk", "avent", "esterilizador"
    ],

    # DIGESTIVO
    "estrenimiento": [
        "estrenimiento", "laxante", "dulcosoft", "melilax", "vilardell",
        "transito intestinal"
    ],
    "acidez_reflujo": [
        "acidez", "reflujo", "arcid", "neobianacid", "esoxx", "antiácido"
    ],
    "flora_intestinal": [
        "probiotico", "florabiotic", "megalevure", "prodefen", "lactibiane",
        "ultralevura", "flora intestinal"
    ],
    "diarrea": [
        "diarrea", "dialiv", "smecta", "megasmect"
    ],

    # RESPIRATORIO
    "tos_garganta": [
        "tos", "garganta", "zarbees", "grintuss", "nartuss", "propolis",
        "expectorante", "antitusivo", "jarabe tos"
    ],
    "congestion_nasal": [
        "nasal", "congestion", "normomar", "respimar", "rhinomer", "agua mar",
        "suero fisiologico"
    ],
    "alergia": [
        "alergia", "antiacaros", "antihistaminico", "rinitis", "alergico",
        "antipolvo", "acaros", "alergias"
    ],

    # ENERGIA / NERVIOSO
    "energia_cansancio": [
        "energia", "cansancio", "fatiga", "pharmaton", "supradyn", "multicentrum",
        "vitalidad", "vitans"
    ],
    "sueno_insomnio": [
        "sueno", "dormir", "insomnio", "melatonina", "zzzquil", "melamil",
        "passiflorine", "valeriana", "somnífero"
    ],
    "estres_ansiedad": [
        "estres", "ansiedad", "relax", "stress", "calmante nervioso",
        "calmarelax", "equirelax"
    ],
    "memoria_concentracion": [
        "memoria", "concentracion", "cognex", "memorup", "aminobrain"
    ],

    # MUJER
    "menopausia": [
        "menopausia", "sofoco", "flavia", "menopro", "climaterio", "ciminocta"
    ],
    "embarazo_prenatal": [
        "embarazo", "prenatal", "natalben", "fisiogen", "femibion", "exelvit",
        "gestacion", "acido folico", "seidibion mater", "mater", "maternity"
    ],
    "cistitis": [
        "cistitis", "infeccion orina", "cispren", "d-manosa"
    ],
    "higiene_intima": [
        "higiene intima", "cumlaude", "lactacyd", "zelesse", "intimo",
        "flora intima", "donna plus flora"
    ],
    "sequedad_vaginal": [
        "sequedad vaginal", "lubripiu", "ovulos vaginales", "mucus"
    ],

    # DOLOR
    "dolor_muscular": [
        "muscular", "articular", "fisiocrem", "voltadol", "arnidol",
        "cannabiben", "dolor espalda", "contractura"
    ],
    "circulacion_piernas": [
        "circulacion", "piernas cansadas", "venovid", "legvass", "varices"
    ],

    # PIEL CORPORAL
    "cicatrices_estrias": [
        "cicatriz", "estrias", "trofolastin", "repavar"
    ],
    "quemaduras": [
        "quemadura", "aftersun", "be+ med quemaduras"
    ],
    "hongos_pies": [
        "hongos", "micosis", "farmafeet", "funsol", "pie atleta"
    ],

    # PIOJOS
    "piojos": [
        "piojos", "liendres", "goibi", "fullmarks", "paranix", "neositrin",
        "antipiojos", "kitapic"
    ],

    # HERIDAS
    "heridas_apositos": [
        "aposito", "tiritas", "compeed", "herida", "gasa", "venda"
    ],
    "herpes_labial": [
        "herpes", "calentura", "herpix", "compeed herpes"
    ],

    # VITAMINAS
    "vitaminas_general": [
        "vitamina", "multivitaminico", "multicentrum", "supradyn", "pharmaton",
        "vitamins", "ivb", "nutrof", "centrum", "dayamin", "becozyme",
        "aminocomplex", "aminovital", "aminopause", "amino acid", "aminoacido",
        "soniase", "supra d", "complex 30", "complex 60"
    ],
    "hierro": [
        "hierro", "anemia", "fisiogen ferro", "floradix", "ferrum"
    ],
    "omega3": [
        "omega 3", "omega3", "dha", "epa", "nuadha", "brudy"
    ],
    "colageno": [
        "colageno", "collagen", "articulaciones", "colnatur", "colagenova",
        "collagen synergy", "epaplus colageno"
    ],

    # COLESTEROL
    "colesterol": [
        "colesterol", "armolipid", "lipidcol"
    ],

    # TEST
    "test_embarazo": [
        "test embarazo", "predictor", "clearblue"
    ],

    # INCONTINENCIA
    "incontinencia": [
        "incontinencia", "tena", "indasec", "absorbente"
    ],

    # MAGNESIO
    "magnesio": [
        "magnesio", "magnesium", "mg citrato", "bisglicinato", "quelado"
    ],

    # CALCIO / HUESOS
    "calcio_huesos": [
        "calcio", "calcium", "huesos", "osteoporosis", "osteogen"
    ],

    # DISPOSITIVOS / MEDICIÓN
    "tension_medicion": [
        "tensimetro", "tensiometro", "monitor presion", "omron m3", "omron m6",
        "omron m7", "omron m10", "esfigmomanometro", "presion arterial"
    ],
    "glucosa_medicion": [
        "glucometro", "medidor glucosa", "freestyle", "accu-chek", "lancetas",
        "tiras reactivas glucosa"
    ],
    "test_diagnostico": [
        "test rapido", "test antigenos", "autotest", "diagnostico rapido"
    ],
    "ortopedia": [
        "muleta", "andador", "silla ruedas", "baston", "tobillera", "rodillera",
        "munequera", "faja lumbar", "cabestrillo", "collar cervical", "plantillas",
        "ferula", "inmovilizador"
    ],

    # DENTAL AVANZADO
    "cepillo_electrico": [
        "oral-b", "oral b", "braun oral", "cepillo electrico", "recambio oral",
        "sonicare", "philips sonicare", "cabezal recambio"
    ],
    "irrigador_dental": [
        "irrigador", "waterpik", "dental jet", "hidropulsor"
    ],

    # CUIDADO FACIAL / CORPORAL GENERAL
    "cuidado_facial_general": [
        "serum facial", "crema facial", "contorno ojos", "tónico facial",
        "mascarilla facial", "peeling facial", "ampolla facial"
    ],
    "cuidado_corporal": [
        "locion corporal", "crema corporal", "body milk", "aceite corporal",
        "exfoliante corporal", "reafirmante corporal"
    ],

    # MAQUILLAJE
    "maquillaje": [
        "maquillaje", "base maquillaje", "corrector", "colorete", "pintalabios",
        "lapiz labios", "mascara pestanas", "sombra ojos", "iluminador",
        "coverage", "isdin coverage", "foundation", "powder"
    ],

    # PERFUMERÍA
    "perfumeria": [
        "eau de toilette", "eau de parfum", "colonia", "perfume", "body spray",
        "fragancia", "edt", "edp", "iap pharma", "pour femme", "pour homme"
    ],

    # AFEITADO
    "afeitado": [
        "espuma afeitar", "gel afeitar", "aftershave", "maquinilla", "cuchilla",
        "recambio maquinilla", "recortadora", "barba"
    ],

    # LENTES CONTACTO
    "lentes_contacto": [
        "lentilla", "lente contacto", "solucion lentes", "renu", "optifree",
        "biotrue", "liquido lentillas", "farline optica", "solucion unica"
    ],

    # DEPORTE
    "deporte_recuperacion": [
        "recuperacion muscular", "proteina", "bcaa", "creatina", "isotonica",
        "bebida deportiva", "gel energetico", "barrita energetica"
    ],

    # VENDAJE
    "vendaje_compresion": [
        "media compresion", "calcetin compresion", "venda elastica",
        "vendaje neuromuscular", "kinesio", "tape", "esparadrapo"
    ],

    # CONTROL PESO
    "perdida_peso": [
        "adelgazar", "slim", "quemagrasas", "fat burner", "reductor",
        "dieta", "xls medical", "bie3 slim", "reductor grasa", "saciante"
    ],
    "drenante": [
        "drenante", "retencion liquidos", "diuretico", "depurativo",
        "detox", "eliminacion toxinas"
    ],

    # NUTRICION CLINICA / HOSPITALARIA
    "nutricion_clinica": [
        "fresubin", "ensure", "fortimel", "nutricomp", "fontactiv",
        "meritene", "nutri", "resource", "isosource", "kabi control",
        "diasip", "glucerna", "cubitan", "enterex", "nutrison", "barimix",
        "bariatric", "souvenaid", "supportan"
    ],

    # FERTILIDAD
    "fertilidad": [
        "fertilidad", "ferty", "ovulacion", "esperma", "concebir",
        "seidivid", "fertybiotic", "ovusitol"
    ],

    # HIGIENE CAPILAR
    "higiene_cabello": [
        "champu", "shampoo", "acondicionador", "mascarilla capilar"
    ],

    # HIGIENE CORPORAL
    "higiene_corporal": [
        "gel ducha", "gel bano", "shower gel", "jabon liquido",
        "body wash"
    ],

    # DESODORANTE
    "desodorante": [
        "desodorante", "antitranspirante", "roll-on", "spray axila"
    ],

    # TERMOMETRO
    "termometro": [
        "termometro", "termómetro", "temperatura corporal", "febril"
    ],

    # HONGOS PIES (ampliado)
    "hongos_pies": [
        "hongos", "micosis", "farmafeet", "funsol", "pie atleta",
        "polvos pedicos", "antifungico pie"
    ],

    # SERVICIOS (filtrar)
    "servicio": [
        "consulta", "servicio", "spd", "nutricion personalizado"
    ],

    # NUEVAS CATEGORIAS (Fase 5)

    # DENTAL - NUEVOS
    "higiene_dental": [
        "pasta dentifrica", "pasta dental", "dentifrico", "cepillo dental",
        "cepillo manual", "kin pasta", "lacer pasta", "vitis pasta", "gum pasta",
        "phb pasta", "fluocaril"
    ],
    "sensibilidad_dental": [
        "sensodyne", "sensibilidad dental", "dientes sensibles", "sensible dientes"
    ],
    "interdental": [
        "interdental", "interprox", "tepe", "cepillo interdental", "seda dental",
        "hilo dental", "flosser"
    ],

    # RESPIRATORIO - NUEVOS
    "caramelos_garganta": [
        "caramelos", "pastillas garganta", "juanola", "ricola", "strepsils",
        "lizipaina", "angileptol", "faringesic pastillas"
    ],
    "defensas_inmunidad": [
        "inmunidad", "defensas", "inmuno", "imunoglukan", "inmunoferon",
        "immunoferron", "echinacea", "propolis defensas", "sistema inmune"
    ],

    # OPTICA
    "optica": [
        "gafas", "gafas lectura", "gafas sol", "gafas presbicia", "farmamoda"
    ],

    # SALUD SEXUAL
    "salud_sexual": [
        "durex", "preservativo", "preservativos", "condon", "lubricante intimo",
        "control preserv", "sico preserv"
    ],

    # MATERIAL SANITARIO
    "material_sanitario": [
        "aposan", "gasa esteril", "gasa", "algodon", "esparadrapo", "jeringa",
        "jeringuilla", "aguja", "lanceta", "guantes latex", "guantes nitrilo",
        "suero fisiologico monodosis"
    ],

    # DISPOSITIVOS MEDICOS
    "dispositivos_medicos": [
        "pulsioximetro", "saturimetro", "camara inhalacion", "nebulizador",
        "aerosol camara", "optichamber", "prochamber", "spacer"
    ],

    # CUIDADO DEPENDIENTES
    "cuidado_dependientes": [
        "lindor", "dependientes", "encamados", "higiene encamados"
    ],

    # VETERINARIA
    "veterinaria": [
        "scalibor", "collar antiparasitario", "seresto", "frontline", "advantix",
        "nexgard", "bravecto", "veterinario", "mascotas", "perro", "gato"
    ],

    # JOYERIA
    "joyeria": [
        "pipia", "pipía", "pendientes", "pendiente", "aros", "joyeria"
    ],

    # PICADURAS
    "picaduras": [
        "talquistina", "after bite", "afterbite", "picadura", "picaduras",
        "mosquitos calmar", "alivio picadura"
    ],

    # UNAS
    "unas": [
        "si-nails", "sinails", "unas fragiles", "fortalecedor unas",
        "endurecedor unas", "tratamiento unas"
    ],

    # HIGIENE MANOS
    "higiene_manos": [
        "jabon manos", "gel manos", "hidroalcoholico", "hand soap",
        "lavado manos", "higiene manos"
    ],

    # ACCESORIOS BELLEZA
    "accesorios_belleza": [
        "beter", "pinzas", "lima unas", "cortaunas", "cepillo pelo",
        "peine", "espejo", "esponja maquillaje"
    ],

    # =========================================================================
    # NUEVOS KEYWORDS (Fase 5b - Q33-Q65)
    # =========================================================================

    # OIDOS
    "proteccion_auditiva": [
        "tapones oidos", "tapones silicona", "tapones espuma",
        "proteccion auditiva", "ear plugs"
    ],
    "higiene_oidos": [
        "audispray", "otosan", "cerumenol", "limpieza oidos",
        "gotas oticas", "spray oidos", "cerumen"
    ],

    # FACIAL ESPECIFICO
    "contorno_ojos": [
        "contorno ojos", "contorno de ojos", "eye contour",
        "ojeras", "bolsas ojos", "patas gallo"
    ],
    "labios": [
        "balsamo labios", "balsamo labial", "protector labial",
        "labios agrietados", "labios secos", "lip balm", "stick labial"
    ],

    # TERMOTERAPIA
    "termoterapia": [
        "almohadilla electrica", "manta electrica", "cojin termico",
        "saco termico", "bolsa semillas"
    ],

    # SALUD OCULAR
    "salud_ocular": [
        "colirio", "gotas ojos", "yeloin", "coqun", "dx drops",
        "solucion oftalmica"
    ],

    # NUTRICION DEPORTIVA
    "nutricion_deportiva": [
        "creatina", "proteina whey", "bcaa", "aminoacidos deporte",
        "pre-workout", "post-workout", "isotonica", "gel energetico",
        "barrita energetica", "recuperador muscular"
    ],

    # MINERALES
    "minerales": [
        "zinc", "selenio", "cobre", "manganeso", "cromo",
        "oligoelementos", "mineral complex"
    ],

    # COMPLEMENTOS
    "complementos": [
        "adorno", "decoracion", "merchandising", "regalo",
        "navidad", "papa noel", "elfo"
    ],

    # PANALES
    "panales": [
        "panal", "pañal", "dodot", "huggies", "pampers",
        "panal infantil", "pañal bebe"
    ],

    # PARCHES TERMICOS (añadir a dolor_muscular)
    "dolor_muscular": [
        "muscular", "articular", "fisiocrem", "voltadol", "arnidol",
        "cannabiben", "dolor espalda", "contractura",
        "parche termico", "thermacare", "bolsa frio", "bolsa calor",
        "bolsa agua caliente", "frio calor", "hielo instantaneo"
    ],

    # TOALLITAS (por uso - añadir keywords)
    "limpieza_facial": [
        "limpiador", "desmaquillante", "micelar", "gel limpiador",
        "espuma limpiadora", "gel moussant", "limpieza facial",
        "toallitas desmaquillantes"
    ],

    # SVR LINEAS
    "acne": [
        "acne", "granos", "espinillas", "imperfecciones", "effaclar",
        "sebium", "cleanance", "normaderm", "keracnyl", "acniben",
        "seborreica", "sebiaclear", "boreade"
    ],

    # ENEMAS
    "estrenimiento": [
        "estrenimiento", "laxante", "dulcosoft", "melilax", "vilardell",
        "transito intestinal", "enema", "fisioenema", "casen"
    ],

    # RETINOL
    "arrugas_antiedad": [
        "antiedad", "antiarrugas", "anti-age", "retinol", "retinal",
        "colageno facial", "reafirmante", "lifting", "pro-collagen",
        "peptido", "age reverse", "vital age", "filorga", "endocare",
        "martiderm", "esthederm", "isdinceutics", "a.g.e.",
        "anti-wrinkle", "firmeza", "flacidez", "retincare"
    ],

    # =========================================================================
    # NUEVOS KEYWORDS (Fase 5c - Q66-Q79)
    # =========================================================================

    # SERVICIOS (Q66)
    "servicio": [
        "consulta", "servicio", "spd", "nutricion personalizado",
        "peso", "fianza", "alquiler"
    ],

    # DETOX HIGADO (Q69b)
    "detox_higado": [
        "higado", "hepatico", "detox", "cardo mariano", "silimarina",
        "depurativo hepatico"
    ],

    # ACUFENOS (Q74)
    "acufenos": [
        "tinnitan", "acufenos", "tinnitus", "zumbido oidos"
    ],

    # RENAL (Q76)
    "renal": [
        "herbensurina", "renal", "rinon", "litiasis", "calculos renales",
        "vias urinarias"
    ],

    # MANOS (Q78)
    "manos": [
        "crema manos", "crema de manos", "manos secas", "manos agrietadas",
        "hand cream"
    ],

    # =========================================================================
    # NUEVOS KEYWORDS (Fase 5d - Q80-Q87)
    # =========================================================================

    # NEUROPATIA (Q87)
    "neuropatia": [
        "neuropatia", "neuropatico", "nervala", "ciática", "tunel carpiano",
        "dolor nervioso", "acido lipoico"
    ],

    # DIARREA (ampliar Q82)
    "diarrea": [
        "diarrea", "dialiv", "smecta", "megasmect", "suerobiotic",
        "suero oral", "rehidratacion"
    ],

    # SALUD OCULAR (ampliar Q85)
    "salud_ocular": [
        "colirio", "gotas ojos", "yeloin", "coqun", "dx drops",
        "solucion oftalmica", "dx wipes", "toallitas ocular", "higiene ocular"
    ],
}


# =============================================================================
# MARCAS → NECESIDAD (inferencia directa)
# =============================================================================

MARCA_NECESIDAD = {
    # CAIDA CABELLO
    "olistic": "caida_cabello",
    "pilexil": "caida_cabello",
    "iraltone": "caida_cabello",
    "tricovit": "caida_cabello",
    "aminexil": "caida_cabello",
    "neogenic": "caida_cabello",
    "triphasic": "caida_cabello",
    "vitalfan": "caida_cabello",

    # ANTIEDAD
    "endocare": "arrugas_antiedad",
    "esthederm": "arrugas_antiedad",
    "filorga": "arrugas_antiedad",
    "martiderm": "arrugas_antiedad",
    "medik8": "arrugas_antiedad",

    # MANCHAS
    "bella aurora": "manchas",
    "neoretin": "manchas",
    "melascreen": "manchas",

    # ACNE
    "effaclar": "acne",
    "sebium": "acne",
    "cleanance": "acne",
    "normaderm": "acne",
    "acniben": "acne",

    # ROJECES
    "rosacure": "rojeces_rosacea",
    "rosaderm": "rojeces_rosacea",

    # PIEL ATOPICA
    "atoderm": "piel_atopica",
    "lipikar": "piel_atopica",
    "xeracalm": "piel_atopica",
    "exomega": "piel_atopica",

    # SOLAR
    "heliocare": "proteccion_solar",
    "isdin fotoprotector": "proteccion_solar",
    "fusion water": "proteccion_solar",

    # OJO SECO
    "thealoz": "ojo_seco",
    "hylo": "ojo_seco",
    "optiben": "ojo_seco",
    "aquoral": "ojo_seco",
    "lubristil": "ojo_seco",

    # PROTESIS DENTAL
    "corega": "protesis_dental",
    "kukident": "protesis_dental",

    # BEBE ALIMENTACION
    "blemil": "alimentacion_bebe",
    "pedialac": "alimentacion_bebe",
    "nutriben": "alimentacion_bebe",
    "almiron": "alimentacion_bebe",
    "nan": "alimentacion_bebe",
    "hero baby": "alimentacion_bebe",

    # BEBE COLICOS
    "colimil": "colicos_bebe",
    "bivos": "colicos_bebe",

    # PANAL
    "mitosyl": "irritacion_panal",

    # ESTRENIMIENTO
    "dulcosoft": "estrenimiento",
    "melilax": "estrenimiento",

    # PROBIOTICOS
    "megalevure": "flora_intestinal",
    "florabiotic": "flora_intestinal",
    "prodefen": "flora_intestinal",
    "lactibiane": "flora_intestinal",

    # TOS
    "zarbees": "tos_garganta",
    "grintuss": "tos_garganta",

    # SUENO
    "zzzquil": "sueno_insomnio",
    "melamil": "sueno_insomnio",
    "passiflorine": "sueno_insomnio",

    # ENERGIA
    "pharmaton": "energia_cansancio",
    "supradyn": "energia_cansancio",
    "multicentrum": "energia_cansancio",

    # MENOPAUSIA
    "flavia": "menopausia",

    # EMBARAZO
    "natalben": "embarazo_prenatal",
    "fisiogen": "embarazo_prenatal",
    "femibion": "embarazo_prenatal",
    "exelvit": "embarazo_prenatal",

    # DOLOR
    "fisiocrem": "dolor_muscular",
    "arnidol": "dolor_muscular",
    "cannabiben": "dolor_muscular",

    # PIOJOS
    "goibi": "piojos",
    "fullmarks": "piojos",
    "paranix": "piojos",
    "kitapic": "piojos",

    # CISTITIS
    "cistitus": "cistitis",

    # COLESTEROL
    "armolipid": "colesterol",

    # MAGNESIO / SUPLEMENTOS
    "aquilea magnesio": "magnesio",
    "ana maria lajusticia": "magnesio",

    # DISPOSITIVOS MEDICIÓN
    "omron": "tension_medicion",
    "freestyle": "glucosa_medicion",
    "accu-chek": "glucosa_medicion",
    "contour": "glucosa_medicion",

    # CEPILLOS ELECTRICOS
    "oral-b": "cepillo_electrico",
    "oral b": "cepillo_electrico",
    "sonicare": "cepillo_electrico",
    "waterpik": "irrigador_dental",

    # =========================================================================
    # GH GEMA HERRERÍAS (por línea de producto - web scraping 2024-12)
    # =========================================================================
    "gh azelaic": "acne",  # GH 12 AZELAIC-N - imperfecciones, acné, rosácea
    "gh 12 azelaic": "acne",
    "gh aza-gz": "acne",  # Multifunción imperfecciones
    "gh retinal": "arrugas_antiedad",  # GH 600/1600/2000 RETINAL - antiedad
    "gh retinol": "arrugas_antiedad",  # GH 3000 RETINOL-NP
    "gh péptidos": "arrugas_antiedad",  # GH 25 PÉPTIDOS-TH
    "gh peptidos": "arrugas_antiedad",
    "gh antiox": "arrugas_antiedad",  # GH 15 ANTIOX-C - antioxidante
    "gh función barrera": "piel_sensible",  # Función barrera - piel sensible
    "gh funcion barrera": "piel_sensible",
    "gh cica": "piel_sensible",  # GH 5 CICA-E - piel sensible
    "gh cica-efc": "piel_sensible",  # GH CICA-EFC multifunción
    "gh pha-pg": "exfoliacion",  # GH PHA-PG crema renovadora
    "gh pha": "exfoliacion",  # GH PHA exfoliante suave
    "gh renovadora": "exfoliacion",  # Crema renovadora
    "gh función global": "contorno_ojos",  # Contorno ojos y labios
    "gh funcion global": "contorno_ojos",
    "gh k-at sérum": "contorno_ojos",  # Ojeras y bolsas
    "gh k-at serum": "contorno_ojos",
    "gh contorno": "contorno_ojos",
    "gh peeling": "exfoliacion",  # GH PEELING AHA/PHA
    "gh limpiador": "limpieza_facial",  # Limpiadores
    "gh aceite limpiador": "limpieza_facial",
    "gh gel limpiador": "limpieza_facial",
    "gh espumante": "limpieza_facial",
    "gh sistema bifase": "limpieza_facial",
    "gh fps50": "proteccion_solar",  # GH FPS50
    "gh fps": "proteccion_solar",  # GH FPS general
    "gh factor fotobioma": "proteccion_solar",
    "gh doble d": "proteccion_solar",  # GH Doble D FPS50
    "gh triple ceramida": "piel_seca",  # Sérum piel seca
    "gh ultrahidratante": "piel_seca",
    "gh bruma": "piel_sensible",  # Bruma universal dermobiota
    "gh nam-td": "manchas",  # GH 10 NAM-TD Despigmentante
    "gh in-898": "vitaminas_general",  # Complemento alimenticio antiedad
    # Issue #446: Productos GH adicionales (gap analysis 2024-12)
    "gh 25 peptidos": "arrugas_antiedad",  # GH 25 PEPTIDOS-TH SERUM
    "gh 25-peptidos": "arrugas_antiedad",  # Variante con guión
    "gh mascarilla": "hidratacion_facial",  # GH MASCARILLA ULTRAHIDRATANTE
    "gh 5 azelaico": "acne",  # GH 5 AZELAICO S-GH
    "gh azelaico s-gh": "acne",
    "gh kiró": "limpieza_facial",  # GH KIRÓ CRISTINA MITRE
    "gh kiro": "limpieza_facial",  # Sin tilde
    "cristina mitre": "limpieza_facial",  # Línea Cristina Mitre (limpieza japonesa)
    "gh cofre": "set_regalo",  # Cofres/kits GH
    "gh 10 aha": "exfoliacion",  # GH 10 AHA PEELING
    "gh aha": "exfoliacion",  # GH AHA general
    "gh esencia": "limpieza_facial",  # GH ESENCIA TRIPLE H
    "gh triple h": "hidratacion_facial",  # Triple H = hidratación

    # =========================================================================
    # ARTURO ALBA / DR ARTHOUROS (por línea - web scraping 2024-12)
    # =========================================================================
    # Con "FORMULA" intermedio
    "arturo alba formula rapsodia": "piel_sensible",  # Alta recuperación
    "arturo alba formula despigmentante": "manchas",  # Despigmentante
    "arturo alba formula barra solar": "proteccion_solar",
    "arturo alba formula manteca": "limpieza_facial",  # Primera limpieza
    "arturo alba formula chronos": "vitaminas_general",  # Complemento
    "arturo alba formula bruma": "piel_sensible",  # Calmante
    "arturo alba formula renovadora": "arrugas_antiedad",  # Antiedad
    # Con "FORMULA" + línea
    "arturo alba formula regenerante": "piel_seca",  # Hidrolipídica
    "arturo alba formula limpiadora": "limpieza_facial",  # Espumosa
    "arturo alba formula retex": "arrugas_antiedad",  # Retinol texturizante
    # Sin "FORMULA"
    "arturo alba rapsodia": "piel_sensible",  # Alta recuperación, piel dañada
    "arturo alba retinol": "arrugas_antiedad",  # Retinol 0.5% y 1%
    "arturo alba renovadora": "arrugas_antiedad",  # Renovadora noche
    "arturo alba transformadora": "arrugas_antiedad",  # Transformadora noche
    "arturo alba contorno": "contorno_ojos",  # Contorno ojos antiedad
    "arturo alba crema contorno": "contorno_ojos",  # Crema contorno
    "arturo alba despigmentante": "manchas",  # Sérum despigmentante melasma
    "arturo alba antioxidante": "arrugas_antiedad",  # SOD-Ferulico
    "arturo alba firmeza": "arrugas_antiedad",  # DMAE-Ursolico
    "arturo alba limpiadora": "limpieza_facial",  # Limpiadora espumosa
    "arturo alba manteca": "limpieza_facial",  # Manteca de primera limpieza
    "arturo alba regenerante": "piel_seca",  # Hidrolipídica
    "arturo alba locion": "exfoliacion",  # Loción ácida exfoliante
    "arturo alba solucion exfoliante": "exfoliacion",
    "arturo alba desodorante": "desodorante",  # Desodorante escudo
    "arturo alba pomada": "accesorios_belleza",  # Fortalecedora pestañas/cejas
    "arturo alba chronos": "vitaminas_general",  # Complemento alimenticio
    "arturo alba pâte": "exfoliacion",  # Pâte Exfoliante Físico
    "arturo alba pate": "exfoliacion",
    "arturo alba iride": "arrugas_antiedad",  # Tono radiante, luminosidad
    "arturo alba infusión": "arrugas_antiedad",  # Exo Peptídica
    "arturo alba infusion": "arrugas_antiedad",
    "arturo alba escudo": "desodorante",  # Fórmula Escudo roll-on
    "arturo alba vela": "perfumeria",  # Vela perfumada
    "dr arthouros alba": "arrugas_antiedad",  # Variant name - general antiedad
    "arturo alba": "arrugas_antiedad",  # Catch-all for brand (mainly antiedad)

    # =========================================================================
    # NUTRACEL (ISDIN pomada reparadora)
    # =========================================================================
    "nutracel": "cicatrices_estrias",  # ISDIN pomada reparadora piel

    # =========================================================================
    # OTRAS MARCAS PREMIUM (mantener específicas por línea)
    # =========================================================================
    "lindor crema": "piel_sensible",
    "montibello": "higiene_cabello",  # Principalmente cabello
    "germinal accion": "arrugas_antiedad",  # Germinal Acción Inmediata
    "germinal essential": "arrugas_antiedad",
    "germinal ampollas": "arrugas_antiedad",
    "natura bisse diamond": "arrugas_antiedad",  # Línea Diamond antiedad
    "natura bisse c+c": "arrugas_antiedad",  # Vitamina C
    "perricone md": "arrugas_antiedad",  # Principalmente antiedad
    "sensilis upgrade": "arrugas_antiedad",  # Línea antiedad
    "sensilis pure perfection": "acne",  # Línea acné
    "sensilis photocorrection": "manchas",  # Despigmentante
    "sensilis ritual care": "limpieza_facial",  # Limpieza
    "skeyndor power retinol": "arrugas_antiedad",
    "skeyndor corrective": "manchas",
    "casmara": "cuidado_facial_general",  # Mascarillas profesionales

    # =========================================================================
    # 5PUNTO5 / 5.5 (dermocosméticos pH 5.5)
    # =========================================================================
    "5punto5 contorno": "contorno_ojos",  # Absolut contorno ojos
    "5punto5 calmante": "piel_sensible",  # Bálsamo calmante
    "5punto5 lip booster": "labios",  # Repulpante labios
    "lip booster": "labios",  # Repulpante labios (sin marca)
    "balsamo piel reactiva": "piel_sensible",  # 5punto5 bálsamo
    "aceite desmaquillante": "limpieza_facial",  # Aceite desmaquillante
    "5punto5": "piel_sensible",  # Catch-all (especialista pH 5.5)
    "absolut contorno": "contorno_ojos",  # Productos Absolut

    # =========================================================================
    # WELEDA (cosmética natural)
    # =========================================================================
    "weleda": "cuidado_corporal",  # Cosmética natural

    # =========================================================================
    # SESDERMA (por línea de producto - web scraping 2024-12)
    # =========================================================================
    "sesderma c-vit": "arrugas_antiedad",  # Vitamina C, antioxidante, luminosidad
    "sesderma cvit": "arrugas_antiedad",
    "sesderma azelac": "manchas",  # Despigmentante, manchas
    "sesderma azelac ru": "manchas",
    "sesderma hidraderm": "piel_seca",  # Hidratación, piel seca
    "sesderma reti age": "arrugas_antiedad",  # Retinol antiedad
    "sesderma retisil": "arrugas_antiedad",
    "sesderma daeses": "arrugas_antiedad",  # Reafirmante antiedad
    "sesderma factor g": "arrugas_antiedad",  # Factor de crecimiento
    "sesderma fillderma": "arrugas_antiedad",  # Relleno arrugas
    "sesderma btses": "arrugas_antiedad",  # Efecto botox
    "sesderma ferulac": "arrugas_antiedad",  # Ácido ferúlico antioxidante
    "sesderma acglicolic": "exfoliacion",  # Ácido glicólico exfoliante
    "sesderma repaskin": "proteccion_solar",  # Fotoprotección
    "sesderma estryses": "cicatrices_estrias",  # Estrías
    "sesderma celulex": "circulacion_piernas",  # Celulitis
    "sesderma hidraven": "limpieza_facial",  # Limpieza
    "sesderma hidraloe": "piel_seca",  # Aloe hidratante
    "sesderma glicare": "contorno_ojos",  # Contorno ojos
    "sesderma k-vit": "contorno_ojos",  # Ojeras, vitamina K
    "sesderma oceanskin": "piel_seca",  # Hidratación marina
    "sesderma hidroquin": "manchas",  # Blanqueamiento
    "sesderma": "arrugas_antiedad",  # Catch-all (principalmente antiedad)

    # =========================================================================
    # SENSILIS (por línea - web scraping 2024-12)
    # =========================================================================
    "sensilis eternalist": "arrugas_antiedad",  # Retinol antiedad
    "sensilis origin pro": "arrugas_antiedad",  # EGF-5 factores crecimiento
    "sensilis supreme": "arrugas_antiedad",  # Alta gama antiedad
    "sensilis skin delight": "arrugas_antiedad",  # Luminosidad antiedad
    "sensilis skin glow": "arrugas_antiedad",
    "sensilis pure age": "acne",  # Cicacne - acné adulto
    "sensilis purify": "acne",  # Acné, piel grasa
    "sensilis d-pigment": "manchas",  # Despigmentante
    "sensilis skin d-pigment": "manchas",
    "sensilis hydra essence": "piel_seca",  # Hidratación
    "sensilis peptide ar": "rojeces_rosacea",  # Rosácea, rojeces
    "sensilis resurfacing": "exfoliacion",  # Exfoliación
    "sensilis": "piel_sensible",  # Catch-all (especialista piel sensible)

    # LENTES CONTACTO
    "renu": "lentes_contacto",
    "optifree": "lentes_contacto",
    "biotrue": "lentes_contacto",

    # ORTOPEDIA
    "farmalastic": "ortopedia",
    "prim": "ortopedia",
    "orliman": "ortopedia",

    # VITAMINAS / MINERALES
    "solgar": "vitaminas_general",
    "nature's bounty": "vitaminas_general",
    "solaray": "vitaminas_general",

    # DEPORTE
    "isostar": "deporte_recuperacion",
    "powerbar": "deporte_recuperacion",
    "226ers": "deporte_recuperacion",

    # NUTRICION CLINICA
    "fresubin": "nutricion_clinica",
    "ensure": "nutricion_clinica",
    "fortimel": "nutricion_clinica",
    "fontactiv": "nutricion_clinica",
    "meritene": "nutricion_clinica",
    "resource": "nutricion_clinica",
    "glucerna": "nutricion_clinica",

    # FERTILIDAD
    "seidivid": "fertilidad",
    "ovusitol": "fertilidad",
    "fertybiotic": "fertilidad",

    # KLORANE (cabello)
    "klorane": "higiene_cabello",
    "lazartigue": "caida_cabello",
    "rene furterer": "caida_cabello",

    # TERMOMETROS
    "aposan termometro": "termometro",
    "termometro clinico": "termometro",

    # =========================================================================
    # NUEVAS MARCAS (Fase 5 - Revisión farmacéutica)
    # =========================================================================

    # HIGIENE DENTAL
    "gum ": "higiene_dental",
    "kin ": "higiene_dental",
    "lacer ": "encias",  # Lacer principalmente encias
    "vitis ": "higiene_dental",
    "phb ": "higiene_dental",
    "fluocaril": "higiene_dental",

    # SENSIBILIDAD DENTAL
    "sensodyne": "sensibilidad_dental",

    # INTERDENTAL
    "interprox": "interdental",
    "tepe": "interdental",

    # SALUD SEXUAL
    "durex": "salud_sexual",
    "control preserv": "salud_sexual",

    # MATERIAL SANITARIO
    "aposan": "material_sanitario",

    # ALIMENTACION BEBE
    "smileat": "alimentacion_bebe",
    # hero baby ya está en línea 719

    # HIGIENE BEBE
    "mustela": "higiene_bebe",
    "suavinex": "higiene_bebe",
    "nuk ": "higiene_bebe",
    "avent": "higiene_bebe",

    # CONGESTION NASAL
    "vicks": "congestion_nasal",
    "rhinomer": "congestion_nasal",
    "respir": "congestion_nasal",

    # ESTRENIMIENTO
    "roha max": "estrenimiento",
    "vilardell": "estrenimiento",

    # LACTANCIA
    "purelan": "lactancia",
    "medela": "lactancia",
    "suavinex lactancia": "lactancia",

    # CUIDADO DEPENDIENTES
    "lindor": "cuidado_dependientes",
    "tena ": "incontinencia",
    "indasec": "incontinencia",

    # VETERINARIA
    "scalibor": "veterinaria",
    "seresto": "veterinaria",
    "frontline": "veterinaria",
    "advantix": "veterinaria",

    # PICADURAS
    "talquistina": "picaduras",
    "after bite": "picaduras",

    # ACCESORIOS BELLEZA
    "beter": "accesorios_belleza",

    # OJO SECO (adicionales)
    "vita-pos": "ojo_seco",
    "vitapos": "ojo_seco",
    "systane": "ojo_seco",
    "relive": "ojo_seco",

    # SUENO
    "ns sonaben": "sueno_insomnio",
    "soñaben": "sueno_insomnio",
    "sonaben": "sueno_insomnio",

    # DEFENSAS/INMUNIDAD
    "imunoglukan": "defensas_inmunidad",
    "inmunoferon": "defensas_inmunidad",

    # HIGIENE INTIMA
    "papilocare": "higiene_intima",
    "cumlaude": "higiene_intima",
    "lactacyd": "higiene_intima",

    # PROTECCION SOLAR (marcas adicionales)
    "fotoultra": "proteccion_solar",
    "eryfotona": "proteccion_solar",

    # OPTICA
    "farmamoda": "optica",

    # DISPOSITIVOS MEDICOS
    "optichamber": "dispositivos_medicos",
    "prochamber": "dispositivos_medicos",

    # JOYERIA
    "pipia": "joyeria",

    # CARAMELOS GARGANTA
    "juanola": "caramelos_garganta",
    "ricola": "caramelos_garganta",
    "strepsils": "caramelos_garganta",

    # UÑAS
    "si-nails": "unas",

    # =========================================================================
    # NUEVAS MARCAS (Fase 5b - Q33-Q65)
    # =========================================================================

    # SVR LINEAS (Q34)
    "svr sebiaclear": "acne",
    "svr topialyse": "piel_atopica",
    "svr sensifine": "piel_sensible",
    "svr ": "cuidado_facial_general",

    # ROGER & GALLET (Q35)
    "roger gallet eau": "perfumeria",
    "roger gallet parfum": "perfumeria",
    "roger gallet jabon": "higiene_corporal",
    "roger gallet gel": "higiene_corporal",

    # OZOAQUA (Q36)
    "ozoaqua crema": "cuidado_corporal",
    "ozoaqua gel intimo": "higiene_intima",
    "ozoaqua jabon": "higiene_corporal",
    "ozoaqua": "higiene_corporal",

    # PARCHES/BOLSAS TERMICAS (Q37, Q40)
    "thermacare": "dolor_muscular",
    "thermo dr": "dolor_muscular",

    # OIDOS (Q38, Q41)
    "audispray": "higiene_oidos",
    "otosan": "higiene_oidos",
    "cerumenol": "higiene_oidos",

    # SUPLEMENTOS (Q42, Q43)
    "aminobalance": "vitaminas_general",
    "aminoenergy": "energia_cansancio",
    "aminosleep": "sueno_insomnio",
    "astenolit": "energia_cansancio",
    "mindex": "vitaminas_general",
    "sibok": "vitaminas_general",

    # ALOCLAIR (Q56)
    "aloclair": "aftas",

    # ISDIN LINEAS (Q53)
    "isdin ureadin": "piel_seca",
    "isdin nutratopic": "piel_atopica",
    "isdin lambdapil": "caida_cabello",
    "isdin hydration": "piel_seca",

    # BE+ MED (Q62)
    "be+ med escoceduras": "irritacion_panal",
    "be+med escoceduras": "irritacion_panal",
    "be+ med balsamo labios": "labios",
    "be+med balsamo labios": "labios",

    # OTRAS MARCAS (Q55, Q61, Q63, Q64, Q65)
    "libicare": "salud_sexual",
    "novalac": "alimentacion_bebe",
    "inibsa": "higiene_corporal",
    "boreade": "acne",
    "eroxon": "salud_sexual",
    "gelsectan": "flora_intestinal",
    "retincare": "arrugas_antiedad",

    # FARLINE/ACOFARMA por producto (Q33)
    "farline gel derma": "higiene_corporal",
    "farline gel intimo": "higiene_intima",
    "farline champu": "higiene_cabello",
    "acofarma mascarilla": "material_sanitario",
    "acofar mascarilla": "material_sanitario",
    "acofarma nesira": "cuidado_corporal",

    # PANALES (corregido)
    "dodot": "panales",
    "huggies": "panales",
    "pampers": "panales",

    # TOALLITAS por uso (Q39)
    "toallitas bebe": "higiene_bebe",
    "toallitas intimas": "higiene_intima",
    "toallitas desmaquillantes": "limpieza_facial",

    # BOLSO PASEO (Q60)
    "bolso paseo": "higiene_bebe",

    # FINISHER (Q52)
    "finisher": "nutricion_deportiva",

    # SULFATO ZINC (Q58)
    "sulfato de zinc": "minerales",

    # =========================================================================
    # NUEVAS MARCAS (Fase 5c - Q66-Q79)
    # =========================================================================

    # HERO (Q68) - capturar todo
    "hero": "alimentacion_bebe",

    # SUPLEMENTOS ESPECIFICOS (Q69)
    "nexala": "colageno",
    "livex": "detox_higado",
    "iromax": "flora_intestinal",
    "superlativa": "estres_ansiedad",

    # ENFAMIL (Q70)
    "enfamil": "alimentacion_bebe",

    # ISDIN GERMISDIN (Q71)
    "germisdin gel": "higiene_corporal",
    "germisdin aloe": "higiene_corporal",
    "isdindeo": "desodorante",
    "germisdin deo": "desodorante",

    # YOVIS (Q72)
    "yovis": "flora_intestinal",

    # MELAGYN (Q73)
    "melagyn": "higiene_intima",

    # TINNITAN (Q74)
    "tinnitan": "acufenos",

    # NEURALEX (Q75)
    "neuralex": "memoria_concentracion",

    # HERBENSURINA (Q76)
    "herbensurina": "renal",

    # SUNISDIN (Q77)
    "sunisdin": "proteccion_solar",

    # NEUTROGENA MANOS (Q78)
    "neutrogena crema manos": "manos",
    "neutrogena manos": "manos",

    # MYCOBEN (Q79)
    "mycoben": "hongos_pies",

    # =========================================================================
    # NUEVAS MARCAS (Fase 5d - Q80-Q87)
    # =========================================================================

    # BABY NATURALS (Q80)
    "baby naturals": "irritacion_panal",
    "zn40": "irritacion_panal",

    # POVIDONA (Q81)
    "povidona": "heridas_apositos",
    "betadine": "heridas_apositos",

    # NS SUEROBIOTIC (Q82)
    "ns suerobiotic": "diarrea",
    "suerobiotic": "diarrea",

    # BE+ MED PEDIATRICS (Q83)
    "be+ med pediatrics": "higiene_bebe",
    "be+med pediatrics": "higiene_bebe",

    # PRODUO (Q84)
    "produo": "flora_intestinal",

    # DX WIPES (Q85)
    "dx wipes": "salud_ocular",

    # PERMEACARE (Q86)
    "permeacare": "flora_intestinal",

    # NERVALA (Q87)
    "nervala": "neuropatia",

    # =========================================================================
    # NUEVAS MARCAS (Fase 5e - Q88-Q97)
    # =========================================================================

    # AMBIENTADOR (Q88)
    "ambientador": "complementos",

    # GENOTERGENTE (Q89)
    "genotergente": "higiene_corporal",

    # NHCO (Q90-Q91)
    "aminotriveplan": "drenante",
    "nhco aminoflex": "colageno",
    "aminoflex": "colageno",

    # PEDIAKID (Q92)
    "pediakid": "alimentacion_bebe",

    # 5PUNTO5 (Q93)
    "photoaging": "proteccion_solar",
    "5punto5": "cuidado_facial_general",
    "tonico hialuronico": "arrugas_antiedad",

    # KREAFLORA (Q94)
    "kreaflora": "gases_digestion",

    # AZEANE (Q95)
    "azeane": "acne",
    "acido azelaico": "acne",

    # BE+MED HIDRACALM (Q96)
    "hidracalm": "piel_seca",

    # HYDRA BEBE (Q97)
    "hydra bebe": "higiene_bebe",

    # RETILUT (Q98)
    "retilut": "fatiga_visual",

    # =========================================================================
    # LOTE 100 - CLASIFICACIÓN MASIVA
    # =========================================================================

    # EMBARAZO/FERTILIDAD
    "gestacare": "embarazo_prenatal",
    "gestagyn": "fertilidad",
    "inositol": "fertilidad",

    # DIGESTIVO
    "gavidigest": "estrenimiento",
    "digestconfort": "gases_digestion",
    "naturflat": "gases_digestion",
    "coligas": "gases_digestion",
    "boldo": "gases_digestion",
    "esophacare": "acidez_reflujo",
    "oliovita": "acidez_reflujo",
    "eno": "acidez_reflujo",

    # FLORA INTESTINAL
    "regenintest": "flora_intestinal",
    "alflorex": "flora_intestinal",
    "profaes": "flora_intestinal",
    "cleanintest": "flora_intestinal",
    "kijimea": "flora_intestinal",
    "l-glutamine": "flora_intestinal",
    "glutamine": "flora_intestinal",

    # DIARREA
    "frenodiar": "diarrea",
    "probiobal": "diarrea",

    # HEMORROIDES
    "hemoless": "hemorroides",
    "neofitoroid": "hemorroides",

    # DOLOR MUSCULAR
    "acticrem": "dolor_muscular",
    "bonflex cbd": "dolor_muscular",
    "luxoben": "dolor_muscular",
    "alcohol romero": "dolor_muscular",

    # TOS/RESPIRATORIO
    "sediflu": "tos_garganta",
    "aluneb": "congestion_nasal",
    "rhinodouche": "congestion_nasal",

    # MANCHAS/ANTIEDAD
    "mela b3": "manchas",
    "basiko": "arrugas_antiedad",
    "be+ booster": "arrugas_antiedad",
    "be+ener": "arrugas_antiedad",

    # UÑAS
    "mordex": "unas",

    # LACTANCIA
    "ginecomplex": "lactancia",

    # PIEL
    "abs skincare": "piel_seca",
    "keltia": "piel_seca",
    "policalm": "piel_sensible",
    "emolienta": "piel_sensible",
    "calamina": "piel_sensible",
    "psorisdin": "piel_sensible",
    "acmed": "acne",
    "acnicontrol": "acne",

    # CAIDA CABELLO
    "plenocapil": "caida_cabello",
    "complidermol": "caida_cabello",

    # DENTAL
    "denzimax": "encias",
    "perio aid": "encias",
    "curasept": "encias",
    "lacerblanc": "higiene_dental",
    "bexident": "higiene_dental",

    # OMEGA/COLESTEROL
    "krill": "omega3",
    "alicina": "colesterol",

    # ALIMENTACION BEBE
    "blevit": "alimentacion_bebe",
    "gerber": "alimentacion_bebe",
    "pediasure": "alimentacion_bebe",

    # ORTOPEDIA
    "silla wc": "ortopedia",
    "faja sacrolumbar": "ortopedia",
    "varisan": "ortopedia",

    # CICATRICES
    "bio-oil": "cicatrices_estrias",
    "bio oil": "cicatrices_estrias",
    "rosa mosqueta": "cicatrices_estrias",

    # OSTOMIA
    "adhesive remover": "ostomia",

    # OJO
    "gse eye": "ojo_seco",
    "orcex": "ojo_seco",
    "nurane": "ojo_seco",

    # COLAGENO/ARTICULACIONES
    "bonflex": "colageno",
    "aminocollax": "colageno",
    "siik": "colageno",

    # DEFENSAS
    "aminodefense": "defensas_inmunidad",
    "oscillococcinum": "defensas_inmunidad",

    # MANOS
    "o'keeffe": "manos",
    "okeeffe": "manos",
    "working hands": "manos",

    # HIGIENE CORPORAL
    "vivera": "higiene_corporal",
    "sebamed": "higiene_corporal",
    "two poles": "higiene_corporal",

    # HIGIENE INTIMA
    "colpofix": "higiene_intima",
    "femibiotic": "higiene_intima",

    # VITAMINAS
    "calbidol": "vitaminas_general",
    "buonavit": "vitaminas_general",
    "antiox": "vitaminas_general",

    # HIERRO
    "ferroprotect": "hierro",

    # MATERIAL SANITARIO
    "molicare": "material_sanitario",
    "alcohol 70": "material_sanitario",

    # HERIDAS
    "septomida": "heridas_apositos",
    "parche protector": "heridas_apositos",

    # =========================================================================
    # LOTE 200 - CLASIFICACIÓN MASIVA
    # =========================================================================

    # HIGIENE OIDOS
    "taponox": "higiene_oidos",
    "normocare otico": "higiene_oidos",
    "otialiv": "higiene_oidos",

    # CABELLO
    "nosa": "higiene_cabello",
    "desulpir": "higiene_cabello",
    "vitacrecil": "caida_cabello",
    "pilopeptan": "caida_cabello",
    "novophane": "caida_cabello",
    "luxmetique": "caida_cabello",
    "be+ med capilar": "caida_cabello",
    "be+ capilar": "caida_cabello",
    "be+capilar": "caida_cabello",

    # MENOPAUSIA
    "oilnagra": "menopausia",
    "donna plus": "menopausia",
    "seidibion": "menopausia",

    # PERDIDA PESO
    "aminoappetite": "perdida_peso",
    "metabok": "perdida_peso",
    "adelmed": "perdida_peso",
    "lipograsil": "perdida_peso",
    "dietcontrol": "perdida_peso",
    "somatoline": "perdida_peso",

    # ANTIEDAD/PIEL
    "xhekpon": "arrugas_antiedad",
    "galenic": "arrugas_antiedad",
    "retiage": "arrugas_antiedad",
    "halibut": "piel_seca",
    "nevasona": "piel_sensible",
    "vea spray": "piel_seca",
    "vea ": "piel_seca",
    "vaselina": "piel_seca",
    "vaselix": "piel_seca",
    "corpitol": "piel_seca",
    "cerave": "piel_seca",
    "mahiou": "piel_seca",
    "novamed skincare": "piel_seca",
    "matricium": "piel_sensible",

    # PIEL ATOPICA
    "leti at4": "piel_atopica",
    "letiat4": "piel_atopica",

    # VAGINAL
    "hidrafem": "sequedad_vaginal",
    "muvagyn": "sequedad_vaginal",

    # DIGESTIVO
    "aliviolas": "estrenimiento",
    "ns fibra": "estrenimiento",
    "reflualiv": "acidez_reflujo",
    "enzimdigest": "gases_digestion",
    "nutira": "gases_digestion",
    "eupeptin": "gases_digestion",
    "aminodigest": "gases_digestion",
    "lactoben": "gases_digestion",
    "digesconfort": "gases_digestion",
    "aero-net": "gases_digestion",

    # DIARREA
    "electrolit": "diarrea",
    "apoxkid": "diarrea",
    "ergial": "diarrea",
    "cito-oral": "diarrea",

    # FLORA
    "rotagermine": "flora_intestinal",
    "donacilus": "flora_intestinal",
    "probiasor": "flora_intestinal",
    "motilinitest": "flora_intestinal",
    "bifiselle": "flora_intestinal",
    "simbiosistem": "flora_intestinal",
    "adomelle": "flora_intestinal",
    "probiac": "flora_intestinal",
    "colilen": "flora_intestinal",
    "symbiotenex": "flora_intestinal",

    # RESPIRATORIO
    "hialix": "tos_garganta",
    "isla medic": "tos_garganta",
    "propol2": "tos_garganta",
    "faringedol": "tos_garganta",

    # PICADURAS
    "neoviderm": "picaduras",
    "calmatopic": "picaduras",
    "ozopick": "picaduras",

    # DOLOR
    "nexcare cold": "dolor_muscular",
    "cremactiv": "dolor_muscular",
    "levedol": "dolor_muscular",

    # OCULAR
    "omnitears": "ojo_seco",
    "visuevo": "ojo_seco",
    "neovis": "ojo_seco",
    "visuectoine": "ojo_seco",
    "eyestil": "ojo_seco",
    "blefarix": "salud_ocular",
    "estila toallitas": "salud_ocular",
    "naviblef": "salud_ocular",
    "vitreoclar": "salud_ocular",
    "blephaderm": "salud_ocular",
    "eye revival": "fatiga_visual",

    # DEPORTIVO
    "leotron": "energia_cansancio",
    "corny": "nutricion_deportiva",
    "paleobull": "nutricion_deportiva",

    # CISTITIS
    "urocran": "cistitis",
    "uromannosa": "cistitis",
    "cistiseid": "cistitis",
    "urosens": "cistitis",

    # FERTILIDAD
    "ovosicare": "fertilidad",

    # INTIMA
    "femconfort": "higiene_intima",
    "ginecanesflor": "higiene_intima",
    "physioflor": "higiene_intima",
    "actifemme": "higiene_intima",
    "saugella": "higiene_intima",
    "silveractive": "higiene_intima",
    "cum laude": "higiene_intima",

    # DEFENSAS
    "glizigen": "defensas_inmunidad",
    "bion 3": "defensas_inmunidad",
    "animobium": "defensas_inmunidad",

    # DESODORANTE
    "perspirex": "desodorante",
    "vichy clinical": "desodorante",
    "lambda control": "desodorante",

    # COLAGENO
    "dolocare": "colageno",
    "artiok": "colageno",
    "fisioflex": "colageno",

    # MEMORIA
    "recordum": "memoria_concentracion",
    "dememory": "memoria_concentracion",

    # OMEGA
    "omega kids": "omega3",
    "omegakids": "omega3",

    # VITAMINAS
    "complexteen": "vitaminas_general",
    "multiguard": "vitaminas_general",
    "jalea multivit": "vitaminas_general",

    # HIERRO
    "vigorix": "hierro",

    # ORTOPEDIA
    "salvelox callos": "ortopedia",
    "cojin postural": "ortopedia",
    "almohadilla plantar": "ortopedia",
    "munequera": "ortopedia",
    "caminador": "ortopedia",

    # DRENANTE
    "aredsan": "drenante",

    # HERIDAS
    "unigel": "heridas_apositos",
    "nexcare duo": "heridas_apositos",

    # MATERIAL SANITARIO
    "mascarilla quirurgica": "material_sanitario",
    "alcohol 96": "material_sanitario",
    "agua oxigenada": "material_sanitario",
    "agua destilada": "material_sanitario",
    "test covid": "material_sanitario",
    "suero fisiologico": "material_sanitario",

    # LACTANCIA
    "pezonera": "lactancia",

    # BEBE
    "nidina": "alimentacion_bebe",
    "babynaturals": "higiene_bebe",

    # HIGIENE CORPORAL
    "esponja enjabonada": "higiene_corporal",
    "germisdin": "higiene_corporal",
    "pon-emo": "higiene_corporal",

    # HEMORROIDES
    "procten": "hemorroides",
    "proctoial": "hemorroides",

    # AFTAS
    "vea oris": "aftas",
    "oddent": "aftas",

    # DENTAL
    "oraldine": "higiene_dental",
    "halita": "higiene_dental",

    # UÑAS
    "tijeras pieles": "unas",

    # OSTOMIA
    "adapt pasta": "ostomia",

    # COLESTEROL
    "metarecod": "colesterol",

    # SUEÑO
    "gaba": "sueno_insomnio",

    # PROTECCION SOLAR
    "prototype": "proteccion_solar",

    # ACNE
    "biretix": "acne",

    # CICATRICES
    "cicapost": "cicatrices_estrias",

    # NUTRICION CLINICA
    "kabi complet": "nutricion_clinica",
    "diamil": "nutricion_clinica",

    # CURCUMA/COMPLEMENTOS
    "curcuma": "colageno",

    # VETERINARIA
    "optimmune": "veterinaria",
    "impromune": "veterinaria",
    "upcard": "veterinaria",

    # PROTECCION AUDITIVA
    "tapones moldex": "proteccion_auditiva",
    "audiovit": "acufenos",

    # MANCHAS
    "depiwhite": "manchas",

    # =========================================================================
    # LOTE FINAL - 657 PRODUCTOS CLASIFICADOS
    # =========================================================================

    # ESTRES/ANSIEDAD
    "bachrescue": "estres_ansiedad",
    "bach rescue": "estres_ansiedad",
    "rescue remedy": "estres_ansiedad",
    "rescue plus": "estres_ansiedad",
    "zenex": "estres_ansiedad",

    # DOLOR MUSCULAR
    "alcohol de romero": "dolor_muscular",
    "alcohol romero": "dolor_muscular",
    "cannabix": "dolor_muscular",
    "cannaben": "dolor_muscular",
    "fibroal": "dolor_muscular",
    "herdibel": "dolor_muscular",
    "voltanatura": "dolor_muscular",
    "tythedol": "dolor_muscular",
    "oseo3": "dolor_muscular",

    # CIRCULACION PIERNAS
    "venaliv": "circulacion_piernas",
    "aminoveins": "circulacion_piernas",
    "circuactiv": "circulacion_piernas",
    "vid roja": "circulacion_piernas",
    "vidroja": "circulacion_piernas",

    # LABIOS
    "letibalm": "labios",

    # PIEL SECA
    "o´keeffe´s": "piel_seca",
    "okeeffes": "piel_seca",
    "o'keeffe's": "piel_seca",
    "vitamono": "piel_seca",
    "mepentol": "piel_seca",
    "queraderm": "piel_seca",
    "tractopon": "piel_seca",
    "lensabel": "piel_seca",
    "neusc pies": "piel_seca",
    "kneipp foot": "piel_seca",
    "vitry hydra": "piel_seca",

    # ORTOPEDIA
    "actimove": "ortopedia",
    "actius": "ortopedia",
    "flexipro": "ortopedia",
    "fixtoe": "ortopedia",
    "comforsil": "ortopedia",
    "comforgel": "ortopedia",
    "scholl party": "ortopedia",
    "dr scholl": "ortopedia",
    "herbi feet": "ortopedia",
    "futuro 3m": "ortopedia",
    "ventubel": "ortopedia",
    "pedalier": "ortopedia",
    "talonera": "ortopedia",
    "plantilla silicona": "ortopedia",

    # MAQUILLAJE
    "rougj": "maquillaje",
    "abeñula": "maquillaje",
    "martinelia": "maquillaje",
    "snails gloss": "maquillaje",

    # SALUD SEXUAL
    "satisfyer": "salud_sexual",
    "energisil vigor": "salud_sexual",
    "vagisil": "salud_sexual",
    "lubets": "salud_sexual",
    "woman isdin lubricante": "salud_sexual",

    # HIGIENE INTIMA
    "enna cycle": "higiene_intima",
    "copa menstrual": "higiene_intima",
    "seidigyn": "higiene_intima",
    "seidibiotics": "higiene_intima",
    "vulvisens": "higiene_intima",
    "palomacare": "higiene_intima",
    "feminabiane intima": "higiene_intima",
    "spagyn": "higiene_intima",
    "donnaplus ginegel": "higiene_intima",
    "ginecanesbalance": "higiene_intima",
    "clogin ovulos": "higiene_intima",
    "letifem": "higiene_intima",
    "urogelus": "higiene_intima",

    # SEQUEDAD VAGINAL
    "idracare": "sequedad_vaginal",
    "palomacare gel vaginal": "sequedad_vaginal",

    # FLORA INTESTINAL
    "butirato": "flora_intestinal",
    "sibo-k": "flora_intestinal",
    "sibok": "flora_intestinal",
    "iberobiotics": "flora_intestinal",
    "candi defens": "flora_intestinal",
    "candidefens": "flora_intestinal",
    "femelle": "flora_intestinal",

    # DISPOSITIVOS MEDICOS
    "aerochamber": "dispositivos_medicos",
    "alcoholimetro": "dispositivos_medicos",
    "fonendoscopio": "dispositivos_medicos",

    # CICATRICES
    "linovera": "cicatrices_estrias",
    "exscar": "cicatrices_estrias",

    # TOS/GARGANTA
    "paratuss": "tos_garganta",
    "farinchup": "tos_garganta",
    "strepherbal": "tos_garganta",
    "probactis strep": "tos_garganta",
    "propol2": "tos_garganta",
    "propol 2": "tos_garganta",
    "santasapina": "tos_garganta",

    # VETERINARIA
    "hyalophyt": "veterinaria",
    "fortekor": "veterinaria",
    "cardisure": "veterinaria",
    "fortiflora": "veterinaria",
    "dentican": "veterinaria",
    "tusheel": "veterinaria",
    "anima strath": "veterinaria",
    "animastrath": "veterinaria",
    "glycozoo": "veterinaria",
    "dc pharm": "veterinaria",

    # HERPES LABIAL
    "hupavir": "herpes_labial",
    "disavir": "herpes_labial",

    # ROJECES/ROSACEA
    "leti sr": "rojeces_rosacea",
    "letisr": "rojeces_rosacea",

    # FERTILIDAD
    "ciclogyne": "fertilidad",

    # NEUROPATIA
    "nouro": "neuropatia",
    "nucleodol": "neuropatia",

    # MEMORIA/CONCENTRACION
    "acutil": "memoria_concentracion",
    "melena de leon": "memoria_concentracion",
    "melenadleon": "memoria_concentracion",
    "ginko": "memoria_concentracion",
    "ginkoplus": "memoria_concentracion",
    "apiserum intelecto": "memoria_concentracion",

    # HIGIENE OIDOS
    "gse ear": "higiene_oidos",
    "dryotix": "higiene_oidos",
    "otorig": "higiene_oidos",
    "borisec": "higiene_oidos",
    "cleanotix": "higiene_oidos",
    "filme oto": "higiene_oidos",

    # PROBIOTICOS
    "neuraxbiotic": "probioticos_general",
    "osmobiotic": "probioticos_general",
    "symbioram": "probioticos_general",
    "kaleidon": "probioticos_general",
    "vivomixx": "probioticos_general",

    # LIMPIEZA FACIAL
    "dr. arthouros": "limpieza_facial",
    "loavi": "limpieza_facial",
    "awa solucion": "limpieza_facial",
    "agua de rosas": "limpieza_facial",
    "atache essentielle": "limpieza_facial",

    # CONTORNO OJOS
    "hyalu b5 serum ojos": "contorno_ojos",

    # MANOS
    "panreumol": "manos",
    "fisioreumol": "manos",
    "urgo grietas": "manos",

    # GLUCOSA
    "disibex": "glucosa",
    "glucody pastilla": "glucosa",
    "glucody hipo": "glucosa",

    # GLUCOSA MEDICION
    "glucody parches": "glucosa_medicion",
    "glucoject": "glucosa_medicion",
    "bolsa isotermica kit diabetico": "glucosa_medicion",

    # UNAS
    "unglax": "unas",
    "betalfatrus": "unas",
    "regenail": "unas",
    "mussvital base endurecedora": "unas",
    "laca de uñas vitry": "unas",
    "soivre amargo": "unas",

    # AFEITADO
    "foamtastic": "afeitado",
    "medicis dermoespuma": "afeitado",
    "vichy homme": "afeitado",

    # CAIDA CABELLO
    "trihair": "caida_cabello",
    "hairgen": "caida_cabello",
    "frikton": "caida_cabello",

    # GASES/DIGESTION
    "fructaid": "gases_digestion",
    "betaindigest": "gases_digestion",
    "enzymedica": "gases_digestion",

    # LACTANCIA
    "lactanza": "lactancia",
    "discos de lactancia": "lactancia",

    # HIERRO
    "ferrodue": "hierro",
    "ferrosol": "hierro",
    "gynefer": "hierro",

    # HONGOS
    "gse mycos": "hongos_pies",

    # OJO SECO
    "alaon": "ojo_seco",
    "iridya": "ojo_seco",
    "cristalregen": "ojo_seco",

    # DETOX HIGADO
    "lipok": "detox_higado",

    # CASPA
    "sebumlaude": "caspa",

    # DEFENSAS/INMUNIDAD
    "buona difesa": "defensas_inmunidad",
    "lactoferrina": "defensas_inmunidad",
    "oleocaps": "defensas_inmunidad",
    "oscillococcinum": "defensas_inmunidad",
    "oscilococcinum": "defensas_inmunidad",
    "bronactive": "defensas_inmunidad",
    "aceite de oregano": "defensas_inmunidad",

    # COLESTEROL
    "oxicol": "colesterol",

    # COMPLEMENTOS
    "tiobec": "complementos",
    "homocystrol": "complementos",
    "berberina": "complementos",
    "l-lysine": "complementos",
    "lysine": "complementos",
    "zibric": "complementos",
    "liberum": "complementos",
    "boiron": "complementos",

    # MINERALES
    "quinton isotonic": "minerales",

    # ARRUGAS/ANTIEDAD
    "healthy aging": "arrugas_antiedad",
    "healthyaging": "arrugas_antiedad",

    # PIEL SENSIBLE
    "kertyol": "piel_sensible",
    "veraderm": "piel_sensible",
    "dexyane": "piel_sensible",
    "quinton dermo": "piel_sensible",

    # FATIGA VISUAL
    "blephaeyebag": "fatiga_visual",

    # PROTESIS DENTAL
    "bruxicalm": "protesis_dental",

    # DIARREA
    "resok": "diarrea",

    # SALUD OCULAR
    "retinorm": "salud_ocular",
    "preservision": "salud_ocular",
    "lephasol": "salud_ocular",
    "estila forte": "salud_ocular",

    # ACUFENOS
    "acumax": "acufenos",

    # ESTRENIMIENTO
    "protransitus": "estrenimiento",
    "psyllium": "estrenimiento",
    "laxafibra": "estrenimiento",
    "manasul": "estrenimiento",

    # PERDIDA PESO
    "saciaplen": "perdida_peso",
    "reducarb": "perdida_peso",

    # NUTRICION DEPORTIVA
    "barebell": "nutricion_deportiva",
    "barebells": "nutricion_deportiva",
    "protein plus": "nutricion_deportiva",
    "proteinplus": "nutricion_deportiva",

    # TERMOTERAPIA
    "fito cold": "termoterapia",
    "fitocold": "termoterapia",
    "dermaplast active": "termoterapia",
    "instant coldhot": "termoterapia",
    "bolsa de frio": "termoterapia",

    # MENOPAUSIA
    "menalya": "menopausia",
    "harmolya": "menopausia",

    # CONGESTION NASAL
    "nebianax": "congestion_nasal",
    "neilmed": "congestion_nasal",
    "marimer": "congestion_nasal",

    # ACIDEZ/REFLUJO
    "sikapur": "acidez_reflujo",

    # QUEMADURAS
    "dermaloe": "quemaduras",
    "babe aloe": "quemaduras",

    # CABELLO SECO
    "sensinol": "cabello_seco",

    # ALERGIA
    "alernine": "alergia",
    "sentabex": "alergia",

    # SUENO/INSOMNIO
    "soñodina": "sueno_insomnio",
    "sonodina": "sueno_insomnio",

    # DRENANTE
    "cola de caballo": "drenante",
    "coladecaballo": "drenante",

    # DESODORANTE
    "piedra de alumbre": "desodorante",
    "alumbre": "desodorante",

    # HIGIENE DENTAL
    "placcontrol": "higiene_dental",
    "plac control": "higiene_dental",
    "cb12": "higiene_dental",
    "kemphor": "higiene_dental",
    "vamboo": "higiene_dental",
    "oralchup": "higiene_dental",
    "ohlala": "higiene_dental",

    # CARIES
    "duraphat": "caries",

    # BLANQUEAMIENTO DENTAL
    "blanx": "blanqueamiento_dental",
    "bucomed pasta blanqueadora": "blanqueamiento_dental",

    # SENSIBILIDAD DENTAL
    "elmex sensitive": "sensibilidad_dental",

    # SUDORACION
    "anhidrol": "sudoracion",

    # LENTES CONTACTO
    "nesira": "lentes_contacto",
    "acuvue": "lentes_contacto",

    # REPELENTE INSECTOS
    "repel bite": "repelente_insectos",
    "repelbite": "repelente_insectos",

    # PROTECCION AUDITIVA
    "biotap": "proteccion_auditiva",

    # HIGIENE MANOS
    "sterillium": "higiene_manos",
    "naps toallitas": "higiene_manos",
    "jabon de manos en espuma": "higiene_manos",

    # HIGIENE CORPORAL
    "sato detergente": "higiene_corporal",
    "boi thermal": "higiene_corporal",
    "bilca": "higiene_corporal",
    "suavina": "higiene_corporal",
    "jabon de azufre": "higiene_corporal",
    "jabon azufre": "higiene_corporal",
    "eudermin talco": "higiene_corporal",

    # HIGIENE BEBE
    "ozobaby": "higiene_bebe",
    "muselina mushie": "higiene_bebe",
    "playgro": "higiene_bebe",
    "fehn": "higiene_bebe",
    "nuby": "higiene_bebe",
    "chicco mordedor": "higiene_bebe",
    "chicco refreshing": "higiene_bebe",
    "sophie la girafe": "higiene_bebe",
    "frigg": "higiene_bebe",
    "weleda bebe": "higiene_bebe",
    "weleda calendula bebe": "higiene_bebe",

    # IRRITACION PANAL
    "almidona": "irritacion_panal",
    "weleda crema protectora calendula": "irritacion_panal",

    # CUIDADO DEPENDIENTES
    "conveen": "cuidado_dependientes",
    "menalind": "cuidado_dependientes",
    "cojin antiescaras": "cuidado_dependientes",
    "orinal masculino": "cuidado_dependientes",
    "orinal sabot": "cuidado_dependientes",
    "cuña corysan": "cuidado_dependientes",
    "vaso recortado": "cuidado_dependientes",

    # TEST DIAGNOSTICO
    "combur": "test_diagnostico",
    "prezisa": "test_diagnostico",
    "hba1c": "test_diagnostico",

    # TENSION MEDICION
    "toma tension": "tension_medicion",
    "toma tensión": "tension_medicion",

    # TENSION ARTERIAL
    "cardioprotect": "tension_arterial",

    # MATERIAL SANITARIO
    "tijera recta": "material_sanitario",
    "pinza diseccion": "material_sanitario",
    "pera de irrigacion": "material_sanitario",
    "depresores": "material_sanitario",
    "sonda vesical": "material_sanitario",
    "sonda palex": "material_sanitario",
    "sonda gastrica": "material_sanitario",
    "cepillo limpiacanulas": "material_sanitario",
    "clorhexidina": "material_sanitario",
    "cinta smarch": "material_sanitario",
    "bisturi": "material_sanitario",
    "compresa indas": "material_sanitario",
    "envase recogida": "material_sanitario",
    "envase cuentagotas": "material_sanitario",
    "frasco laboratorio": "material_sanitario",

    # OPTICA
    "twins sole": "optica",
    "twins platinum": "optica",
    "cordones gafas": "optica",

    # PERFUMERIA
    "roger&gallet cologne": "perfumeria",
    "roger&gallet set": "perfumeria",

    # =========================================================================
    # KEYWORDS EXTRAÍDOS DE WEBS DE MARCAS
    # =========================================================================

    # --- BIODERMA ---
    "sensibio": "piel_sensible",
    "atoderm": "piel_atopica",
    "sebium": "acne",
    "hydrabio": "piel_seca",
    "pigmentbio": "manchas",
    "cicabio": "cicatrices_estrias",
    "photoderm": "proteccion_solar",
    "matricium": "arrugas_antiedad",
    "nodé": "higiene_cabello",
    "node": "higiene_cabello",
    "abcderm": "higiene_bebe",

    # --- INSTITUT ESTHEDERM ---
    "esthederm": "arrugas_antiedad",
    "intensive hyaluronic": "arrugas_antiedad",
    "age proteom": "arrugas_antiedad",
    "adaptasun": "proteccion_solar",
    "bronze repair": "proteccion_solar",
    "photo reverse": "manchas",
    "osmoclean": "limpieza_facial",
    "sensi system": "piel_sensible",
    "pure system": "acne",
    "nutri system": "piel_seca",
    "lift & repair": "arrugas_antiedad",

    # --- NHCO NUTRITION (Aminoscience) ---
    "nhco": "complementos",
    "aminosleep": "sueno_insomnio",
    "amino sleep": "sueno_insomnio",
    "l-night": "sueno_insomnio",
    "aminoflex": "colageno",
    "amino flex": "colageno",
    "aminojoint": "colageno",
    "amino joint": "colageno",
    "aminodetox": "detox_higado",
    "amino detox": "detox_higado",
    "aminoveins": "circulacion_piernas",
    "amino veins": "circulacion_piernas",
    "aminobrain": "memoria_concentracion",
    "amino brain": "memoria_concentracion",
    "aminoskin": "arrugas_antiedad",
    "amino skin": "arrugas_antiedad",
    "aminohair": "caida_cabello",
    "amino hair": "caida_cabello",
    "aminomeno": "menopausia",
    "amino meno": "menopausia",
    "aminodefense": "defensas_inmunidad",
    "amino defense": "defensas_inmunidad",

    # --- AQUILEA ---
    "aquilea sueño": "sueno_insomnio",
    "aquilea sueno": "sueno_insomnio",
    "aquilea articulaciones": "colageno",
    "aquilea mobility": "colageno",
    "aquilea digestivo": "acidez_reflujo",
    "aquilea digest": "acidez_reflujo",
    "aquilea piernas": "circulacion_piernas",
    "aquilea enrelax": "estres_ansiedad",
    "aquilea qbiotics": "flora_intestinal",
    "aquilea magnesio": "magnesio",
    "aquilea tranquilidad": "estres_ansiedad",

    # --- CINFA NS NUTRITIONAL SYSTEM ---
    "ns florabiotic": "probioticos_general",
    "ns lactoben": "gases_digestion",
    "ns vitans energy": "energia_cansancio",
    "ns vitans vitalidad": "vitaminas_general",
    "ns saciaplen": "perdida_peso",
    "ns digestconfort": "acidez_reflujo",
    "ns melatonina": "sueno_insomnio",

    # --- CANTABRIA LABS ---
    "heliocare": "proteccion_solar",
    "endocare": "arrugas_antiedad",
    "endocare tensage": "arrugas_antiedad",
    "endocare radiance": "manchas",
    "endocare hydractive": "piel_seca",
    "biretix": "acne",
    "neoretin": "manchas",

    # --- PRODUCTOS ESPECÍFICOS DE BÚSQUEDA ---
    "lactoflora": "flora_intestinal",
    "enterogermina": "flora_intestinal",
    "normolabial": "herpes_labial",

    # =========================================================================
    # KEYWORDS FROM LLM BATCH CLASSIFICATION #3 (309 productos)
    # =========================================================================

    # --- HIGIENE DENTAL ---
    "gum": "higiene_dental",
    "interprox": "higiene_dental",
    "phb": "higiene_dental",
    "oraldine": "higiene_dental",
    "bucal": "higiene_dental",
    "enjuague": "higiene_dental",

    # --- INCONTINENCIA ---
    "ausonia": "incontinencia",
    "tena": "incontinencia",
    "lindor": "incontinencia",
    "discreet": "incontinencia",

    # --- HIGIENE ÍNTIMA ---
    "evax": "higiene_intima",
    "tampax": "higiene_intima",
    "carefree": "higiene_intima",
    "suavinex copa": "higiene_intima",
    "copa menstrual": "higiene_intima",

    # --- PIEL SECA / HIDRATACIÓN ---
    "ureadin": "piel_seca",
    "leti at4": "piel_atopica",
    "letiat4": "piel_atopica",
    "ozoaqua": "piel_seca",
    "repavar": "cicatrices_estrias",

    # --- MENOPAUSIA ---
    "vitawoman": "menopausia",
    "menopausia": "menopausia",
    "femarelle": "menopausia",
    "ymea": "menopausia",
    "gynecomplex": "menopausia",

    # --- OJO SECO ---
    "lipolac": "ojo_seco",
    "hyabak": "ojo_seco",
    "optiben": "ojo_seco",
    "lacrifresh": "ojo_seco",
    "lubristil": "ojo_seco",

    # --- CISTITIS ---
    "cysticlean": "cistitis",
    "urocran": "cistitis",
    "cistifort": "cistitis",
    "cistitus": "cistitis",

    # --- CASPA ---
    "oliprox": "caspa",
    "kelual": "caspa",
    "sebiprox": "caspa",

    # --- COLÁGENO / ARTICULACIONES ---
    "artilane": "colageno",
    "articolageno": "colageno",
    "movial": "colageno",
    "cartilago": "colageno",

    # --- DOLOR MUSCULAR ---
    "traumeel": "dolor_muscular",
    "contractubex": "dolor_muscular",
    "voltadol": "dolor_muscular",
    "radio salil": "dolor_muscular",
    "radiosalil": "dolor_muscular",

    # --- MAGNESIO ---
    "formag": "magnesio",
    "magnesium": "magnesio",
    "mag 2": "magnesio",

    # --- HIGIENE CORPORAL ---
    "mussvital botanics": "higiene_corporal",
    "mussvital essentials": "higiene_corporal",
    "gel baño": "higiene_corporal",
    "gel ducha": "higiene_corporal",
    "jabon liquido": "higiene_corporal",

    # --- PROBIÓTICOS ---
    "bivos": "probioticos_general",
    "reuteri": "probioticos_general",
    "vivomixx": "probioticos_general",

    # --- DEFENSAS / INMUNIDAD ---
    "inmuno": "defensas",
    "inmunoferon": "defensas",
    "immunomix": "defensas",

    # --- DERMATITIS / ECCEMA ---
    "leti sr": "dermatitis",
    "dexyane": "dermatitis",

    # --- VITAMINAS ESPECÍFICAS ---
    "vitamina d": "vitaminas_general",
    "vitamina c": "vitaminas_general",
    "vitamina b": "vitaminas_general",
    "ibsa vitamina": "vitaminas_general",

    # --- DIGESTIVO / ACIDEZ ---
    "gaviscon": "acidez_reflujo",
    "almax": "acidez_reflujo",
    "rennie": "acidez_reflujo",
    "bemolan": "acidez_reflujo",  # Protector gástrico
    "gastropan": "acidez_reflujo",  # Digestivo

    # --- GASES / DIGESTIÓN ---
    "disolgas": "gases_digestion",
    "carbon vegetal": "gases_digestion",
    "salvarina": "gases_digestion",  # Digestivo

    # --- FLORA INTESTINAL ---
    "infloran": "flora_intestinal",

    # --- TOS ---
    "sekisan": "tos_garganta",  # Antitusivo

    # --- VÍAS URINARIAS ---
    "canephron": "cistitis",  # Infección urinaria

    # --- CONGESTIÓN NASAL ---
    "sinus": "congestion_nasal",  # Inhalaciones

    # --- DOLOR / ANALGÉSICO ---
    "algifast": "dolor_analgesico",  # Analgésico

    # --- LAXANTES ---
    "dulcolax": "estrenimiento",
    "movicol": "estrenimiento",
    "duphalac": "estrenimiento",
    "hodernal": "estrenimiento",  # Parafina líquida
    "aceite de ricino": "estrenimiento",  # Laxante tradicional

    # --- TENSIÓN ARTERIAL ---
    "tensiometro": "tension_arterial",
    "omron": "tension_arterial",

    # --- PRIMEROS AUXILIOS ---
    "tiritas": "primeros_auxilios",
    "hansaplast": "primeros_auxilios",
    "betadine": "primeros_auxilios",
    "cristalmina": "primeros_auxilios",

    # --- QUEMADURAS / AFTERSUN ---
    "aftersun": "quemaduras_aftersun",
    "after sun": "quemaduras_aftersun",
    "post solar": "quemaduras_aftersun",

    # --- ORTOPEDIA ---
    "farmalastic": "ortopedia",
    "orliman": "ortopedia",
    "rodillera": "ortopedia",
    "tobillera": "ortopedia",
    "muñequera": "ortopedia",
    "media compresion": "ortopedia",

    # --- INFANTIL ---
    "pediatric": "bebes_infantil",
    "pediatrico": "bebes_infantil",
    "mustela": "bebes_infantil",
    "suavinex": "bebes_infantil",

    # --- ANTIFUNGICOS ---
    "canesten": "hongos_pies",
    "fungarest": "hongos_pies",
    "lamisil": "hongos_pies",

    # --- REPELENTES ---
    "relec": "repelente_insectos",
    "goibi": "repelente_insectos",
    "antimosquitos": "repelente_insectos",

    # --- HIGIENE ÍNTIMA ADICIONAL ---
    "compresas higienicas": "higiene_intima",
    "compresas femeninas": "higiene_intima",
    "farmaconfort": "higiene_intima",

    # --- ALIMENTACIÓN BEBÉ ADICIONAL ---
    "capricare": "alimentacion_bebe",
    "leche crecimiento": "alimentacion_bebe",

    # --- PRIMEROS AUXILIOS ADICIONAL ---
    "nobecutan": "primeros_auxilios",
    "nexcare": "primeros_auxilios",
    "leukosan": "primeros_auxilios",
    "sutura cutanea": "primeros_auxilios",

    # --- MATERIAL SANITARIO ADICIONAL ---
    "guantes nitrilo": "material_sanitario",
    "guantes latex": "material_sanitario",
    "guantes vinilo": "material_sanitario",
    "abena": "material_sanitario",

    # --- CUIDADO LABIOS ---
    "balsamo labios": "herpes_labial",
    "balsamo labial": "herpes_labial",
    "labial hidratante": "herpes_labial",

    # --- COSMÉTICA / HIGIENE CORPORAL ---
    "roger gallet": "higiene_corporal",
    "roger&gallet": "higiene_corporal",

    # --- CUIDADO PIES ---
    "crema pies": "pies_talones",
    "farline pies": "pies_talones",

    # --- MAREO ---
    "stop mareo": "mareo_viaje",
    "biodramina": "mareo_viaje",

    # --- GLUCOSA / DIABETES ---
    "test glucosa": "diabetes",
    "glucometro": "diabetes",
    "tiras reactivas": "diabetes",

    # --- SUERO FISIOLÓGICO ---
    "solucion fisiologica": "congestion_nasal",
    "suero fisiologico": "congestion_nasal",

    # --- ACCESORIOS BELLEZA ---
    "cepillo de pelo": "accesorios_belleza",
    "peine": "accesorios_belleza",
    "pinza depilacion": "accesorios_belleza",
    "tijeras pieles": "accesorios_belleza",
    "tijeras unas": "accesorios_belleza",
    "vitry": "accesorios_belleza",
    "lima unas": "accesorios_belleza",
    "cortaunas": "accesorios_belleza",

    # --- HIGIENE CORPORAL ADICIONAL ---
    "floralba": "higiene_corporal",
    "gel de baño": "higiene_corporal",
    "gel baño": "higiene_corporal",
    "mayolica": "higiene_corporal",
    "oleogel ducha": "higiene_corporal",

    # --- MANCHAS / VITAMINA C ---
    "pure vitamin c": "manchas",
    "vitamin c10": "manchas",

    # --- AGUA TERMAL ---
    "agua termal": "piel_sensible",

    # --- BEBÉ ADICIONAL ---
    "baby cologne": "bebes_infantil",
    "neo peques": "bebes_infantil",
    "nutratopic": "piel_atopica",

    # --- ONAGRA / MENOPAUSIA ---
    "aceite onagra": "menopausia",
    "aceite de onagra": "menopausia",

    # --- TEST GLUCOSA ---
    "test de glucosa": "diabetes",

    # --- HEMORROIDES ---
    "hemofarm": "hemorroides",
    "toallitas hemorroides": "hemorroides",

    # --- DETOX / HIGADO ---
    "alcachofa": "detox_higado",
    "arkopharma alcachofa": "detox_higado",
    "cardo mariano": "detox_higado",

    # --- ENERGIA / JALEA REAL ---
    "jalea real": "energia_cansancio",
    "black bee": "energia_cansancio",

    # --- HIGIENE DENTAL ADICIONAL ---
    "xeros dentaid": "higiene_dental",
    "xerostomia": "higiene_dental",
    "dentaid": "higiene_dental",
    "mushie": "bebes_infantil",

    # --- PIEL SENSIBLE ADICIONAL ---
    "cetaphil": "piel_sensible",
    "pan dermatologico": "piel_sensible",

    # --- NEUTROGENA ---
    "neutrogena hydro boost": "piel_seca",
    "neutrogena formula noruega": "piel_seca",
    "neutrogena hidrat": "piel_seca",
    "neutrogena corp": "higiene_corporal",
    "neutrogena locion": "piel_seca",

    # --- EUCERIN ---
    "eucerin oleogel": "higiene_corporal",
    "eucerin piel seca": "piel_seca",
    "eucerin ph5": "higiene_corporal",

    # --- PIEL SECA / ATOPICA ADICIONAL ---
    "balneum plus": "piel_atopica",
    "balneum": "piel_atopica",
    "aceite almendras": "piel_seca",
    "aceite de almendras": "piel_seca",

    # --- SOPORTE / ORTOPEDIA ---
    "futuro": "ortopedia",
    "soporte terapeutico": "ortopedia",
    "arco plantar": "ortopedia",

    # --- OSTOMIA ---
    "ostomia": "ostomia",
    "bolsas ostomia": "ostomia",
    "stomahesive": "ostomia",
    "sensura": "ostomia",
    "coloplast ostomia": "ostomia",
    "periostomal": "ostomia",

    # --- SEQUEDAD VAGINAL ---
    "hid vulvar": "sequedad_vaginal",
    "hidratante vulvar": "sequedad_vaginal",
    "woman duo": "sequedad_vaginal",

    # --- DERMATITIS SEBORREICA ---
    "uriage ds": "caspa",
    "uriage d s": "caspa",
    "d s emulsion": "caspa",

    # --- PIEL SECA (UREA) ---
    "manto acido": "piel_seca",
    "urea 20": "piel_seca",
    "urea 30": "piel_seca",

    # --- INCONTINENCIA ADICIONAL ---
    "karinas": "incontinencia",
    "acofar karinas": "incontinencia",

    # --- PROSTATA ---

    # --- PROTECCION SOLAR ADICIONAL ---
    "fotop isdin": "proteccion_solar",
    "isdin stick": "proteccion_solar",

    # --- PIEL ATOPICA / ACNE ADICIONAL ---
    "be+ atopia": "piel_atopica",
    "eucerin dermopure": "acne",
    "dermopure oil": "acne",

    # --- DIGESTIVO / HANGOVER ---
    "resalim": "gases_digestion",

    # --- SUDORACION ---
    "hiposudol": "sudoracion",

    # --- PIEL SECA FINAL ---
    "rilastil xerolact": "piel_seca",
    "xerolact": "piel_seca",
    "dexeryl": "piel_seca",

    # --- BEBES/INFANTIL PROTECCION ---
    "be+ skinprotect infantil": "bebes_infantil",

    # --- ACNE / TEA TREE ---
    "arbol del te": "acne",
    "tea tree": "acne",
    "aceite arbol": "acne",

    # --- VERRUGAS (ya marcados otros) ---

    # --- OTROS (explícitamente marcados - solo muy específicos) ---

    # =========================================================================
    # MARCAS DETECTADAS POR ANÁLISIS DE VENTAS (Issue #446 - Fase 2.6)
    # =========================================================================

    # TOP MARCAS POR IMPORTE (análisis sistemático)
    # GH Cosmetics (€21,907) - Facial care
    "gh 12": "cuidado_facial_general",
    "gh pha": "cuidado_facial_general",
    "gh cica": "cuidado_facial_general",
    "gh funcion": "cuidado_facial_general",
    "gh limpiador": "limpieza_facial",
    "gh serum": "arrugas_antiedad",

    # NS Nutritional System (€12,881)
    "ns nutritional": "vitaminas_general",
    "ns vitans": "vitaminas_general",
    "ns complet": "vitaminas_general",
    "ns florabiotic": "flora_intestinal",
    "ns digestconfort": "gases_digestion",
    "ns omega": "omega3",
    "ns melatonina": "sueno_insomnio",
    # Issue #446: Productos NS adicionales (gap analysis 2024-12)
    "ns gineprotect": "infeccion_urinaria",  # NS GINEPROTECT CISPRENICINA - cistitis
    "ns cisprenicina": "infeccion_urinaria",  # Variante directa
    "ns soñadol": "sueno_insomnio",  # NS SOÑADOL
    "ns sonadol": "sueno_insomnio",  # Sin tilde
    "ns lipoless": "perdida_peso",  # NS LIPOLESS
    "ns lipocarbo": "perdida_peso",  # NS LIPOCARBO BI-EFFECT
    "ns endulzante": "edulcorante",  # NS ENDULZANTE

    # IVB Vitaminas (€12,567)
    "ivb magnesio": "magnesio",
    "ivb vitamina": "vitaminas_general",
    "ivb omega": "omega3",
    "ivb colageno": "colageno",
    "ivb ": "vitaminas_general",

    # OLISTIC (€10,901) - Caída cabello
    "olistic": "caida_cabello",

    # ARTURO ALBA / DR ARTHOUROS (€8,849)
    "arturo alba": "cuidado_facial_general",
    "dr arthouros": "cuidado_facial_general",

    # KUKIDENT (€6,843) - Prótesis dental
    "kukident": "protesis_dental",

    # THEALOZ (€3,022) - Ojo seco
    "thealoz": "ojo_seco",

    # HYLO (€2,069) - Ojo seco
    "hylo": "ojo_seco",

    # COMPEED (€2,113) - Apósitos/heridas
    "compeed": "heridas_apositos",

    # OPTIBEN (€2,798) - Ocular
    "optiben": "ojo_seco",

    # MEGALEVURE (€3,145) - Probióticos
    "megalevure": "flora_intestinal",

    # IRALTONE (€1,937) - Caída cabello
    "iraltone": "caida_cabello",

    # OTRAS MARCAS DETECTADAS
    "dulcosoft": "estrenimiento",
    "fisiocrem": "dolor_muscular",
    "arnidol": "dolor_muscular",
    "rhinomer": "congestion_nasal",
    "rhinospray": "congestion_nasal",
    "sterimar": "congestion_nasal",
    "physiomer": "congestion_nasal",
    "respibien": "congestion_nasal",
    "utabon": "congestion_nasal",

    # DISPOSITIVOS
    "omron": "tension_medicion",
    "predictor": "test_embarazo",
    "clearblue": "test_embarazo",

    # =========================================================================
    # MARCAS DESDE WEB RESEARCH (Issue #446 - Fase 2.5b)
    # =========================================================================

    # BIODERMA (bioderma.es) - Líneas de producto
    "sensibio": "piel_sensible",
    "pigmentbio": "manchas",
    "hydrabio": "piel_seca",
    "cicabio": "cicatrices_estrias",
    "matricium": "arrugas_antiedad",
    "node ": "higiene_cabello",  # Capilar
    "abcderm": "higiene_bebe",

    # AQUILEA (aquilea.com) - Por categoría
    "aquilea sueño": "sueno_insomnio",
    "aquilea sueno": "sueno_insomnio",
    "aquilea forte": "sueno_insomnio",
    "aquilea digest": "gases_digestion",
    "aquilea movilidad": "dolor_muscular",
    "aquilea articulaciones": "dolor_muscular",
    "aquilea enrelax": "estres_ansiedad",
    "aquilea relax": "estres_ansiedad",
    "aquilea circulacion": "circulacion_piernas",
    "aquilea piernas": "circulacion_piernas",
    "aquilea defensas": "defensas_inmunidad",
    "aquilea corazon": "colesterol",
    "aquilea vitamina": "vitaminas_general",
    "aquilea energia": "energia_cansancio",
    "aquilea beauty": "colageno",

    # KLORANE (klorane.com) - Líneas anticaída
    "klorane quinine": "caida_cabello",
    "klorane quinina": "caida_cabello",
    "keratincaps": "caida_cabello",
    "klorane mango": "cabello_seco",
    "klorane ortiga": "cabello_graso",
    "klorane avena": "piel_sensible",

    # MUSTELA (mustela.es) - Líneas específicas
    "stelatopia": "piel_atopica",
    "mustela solar": "proteccion_solar",
    "mustela bebe": "higiene_bebe",
    "mustela estrias": "cicatrices_estrias",

    # LACER (lacer.com) - Líneas no dentales
    "thrombocid": "circulacion_piernas",
    "thrombactiv": "dolor_muscular",
    "anso ": "hemorroides",
    "lacer pilexil": "caida_cabello",

    # ISDIN LINEAS adicionales (isdin.com)
    "bexident": "encias",
    "babynaturals": "higiene_bebe",
    "isdin acniben": "acne",
    "isdin fotoprotector": "proteccion_solar",
    "isdinceutics retinal": "arrugas_antiedad",
    "isdinceutics melaclear": "manchas",
    "isdinceutics glicoisdin": "arrugas_antiedad",

    # NHCO (nhco-nutrition.com) - Por categoría
    "nhco aqualyse": "drenante",
    "nhco aminomax": "deporte_recuperacion",
    "nhco aminovita": "vitaminas_general",
    "nhco aminoactif": "energia_cansancio",
    "nhco omega": "cardiovascular",
    "nhco synerbeautific": "colageno",
    "nhco trioptec": "fatiga_visual",

    # GUM (gum.es) - Líneas específicas
    "gum soft-picks": "interdental",
    "gum trav-ler": "interdental",
    "gum paroex": "encias",
    "gum afta": "aftas",
    "gum ortho": "ortodoncia",
    "gum halicontrol": "halitosis",

    # KIN (kin.es) - Líneas específicas
    "kin hidrat": "ojo_seco",
    "kin oro": "protesis_dental",
    "kin perio": "encias",
    "kin fluor": "caries",

    # DUCRAY (ducray.com) - Líneas específicas
    "ducray anacaps": "caida_cabello",
    "ducray anaphase": "caida_cabello",
    "ducray squanorm": "caspa",
    "ducray kelual": "caspa",
    "ducray kertyol": "caspa",
    "ducray sensinol": "cuero_cabelludo",

    # RENE FURTERER (renefurterer.com)
    "forticea": "caida_cabello",
    "triphasic": "caida_cabello",
    "curbicia": "cabello_graso",
    "astera": "cuero_cabelludo",
    "karinga": "cabello_seco",

    # =========================================================================
    # ISDIN COMPLETO (isdin.com) - Web Scraping Dic 2024
    # =========================================================================
    # Fotoprotección
    "isdin fusion water": "proteccion_solar",
    "isdin foto ultra": "proteccion_solar",
    "isdin eryfotona": "proteccion_solar",
    "eryfotona": "proteccion_solar",
    "fusion water": "proteccion_solar",
    "foto ultra": "proteccion_solar",
    "isdin pediatrics": "proteccion_solar",
    "fotoprotector isdin": "proteccion_solar",
    "isdin sunbrush": "proteccion_solar",
    # Antiedad / Isdinceutics
    "isdinceutics": "arrugas_antiedad",
    "isdin flavo-c": "arrugas_antiedad",
    "flavo-c": "arrugas_antiedad",
    "isdin k-ox": "contorno_ojos",
    "k-ox eyes": "contorno_ojos",
    "isdin vital eyes": "contorno_ojos",
    "vital eyes": "contorno_ojos",
    "isdin hyaluronic": "piel_seca",
    "isdin melatonik": "arrugas_antiedad",
    "melatonik": "arrugas_antiedad",
    "isdin ultraglican": "arrugas_antiedad",
    "ultraglican": "arrugas_antiedad",
    # Hidratación / Ureadin
    "ureadin": "piel_seca",
    "isdin ureadin": "piel_seca",
    "ureadin ultra": "piel_seca",
    "ureadin podos": "pies_secos",
    "ureadin calm": "piel_sensible",
    # Capilar
    "lambdapil": "caida_cabello",
    "isdin lambdapil": "caida_cabello",
    # Acné
    "acniben": "acne",
    "isdin acniben": "acne",
    # Atopia
    "nutratopic": "piel_atopica",
    "isdin nutratopic": "piel_atopica",
    # Psoriasis
    "psorisdin": "psoriasis",
    "isdin psorisdin": "psoriasis",
    # Higiene
    "germisdin": "limpieza_facial",
    "isdin germisdin": "limpieza_facial",
    # Bebé
    "isdin baby": "higiene_bebe",
    "babynaturals isdin": "higiene_bebe",
    # Dental
    "bexident encias": "encias",
    "bexident fresh": "halitosis",
    "bexident anticaries": "caries",
    "bexident blanqueante": "blanqueamiento",
    # Otros
    "isdin woman": "menopausia",
    "velastisa": "cicatrices_estrias",
    "isdin velastisa": "cicatrices_estrias",

    # =========================================================================
    # HELIOCARE (heliocare.com / Cantabria Labs) - Web Scraping Dic 2024
    # =========================================================================
    "heliocare 360": "proteccion_solar",
    "heliocare ultra": "proteccion_solar",
    "heliocare color": "proteccion_solar",
    "heliocare advanced": "proteccion_solar",
    "heliocare pediatrics": "proteccion_solar",
    "heliocare gel": "proteccion_solar",
    "heliocare spray": "proteccion_solar",
    "heliocare airgel": "proteccion_solar",
    "heliocare mineral": "proteccion_solar",
    "heliocare water": "proteccion_solar",
    "heliocare cushion": "proteccion_solar",
    "heliocare pigment": "manchas",  # Heliocare Pigment Solution
    "heliocare oral": "proteccion_solar",  # Cápsulas orales
    "heliocare purewhite": "manchas",  # Antimanchas oral
    "fernblock": "proteccion_solar",  # Activo Heliocare

    # =========================================================================
    # BE+ / CINFA (bemaslab.cinfa.com) - Web Scraping Dic 2024
    # =========================================================================
    # Antiedad
    "be+ antiarrugas": "arrugas_antiedad",
    "be+ antiedad": "arrugas_antiedad",
    "be+ reafirmante": "arrugas_antiedad",
    "be+ lifting": "arrugas_antiedad",
    "be+ energifique": "arrugas_antiedad",
    "be+ ultra concentrados": "arrugas_antiedad",
    # Piel sensible
    "be+ piel intolerante": "piel_sensible",
    "be+ anti-rojeces": "rojeces_rosacea",
    "be+ antirojeces": "rojeces_rosacea",
    "be+ cuperosis": "rojeces_rosacea",
    "be+ sensible": "piel_sensible",
    # Capilar
    "be+ capilar": "caida_cabello",
    "be+ anticaida": "caida_cabello",
    "be+ anticaída": "caida_cabello",
    "be+ champu": "caida_cabello",
    "be+ locion capilar": "caida_cabello",
    # Hidratación
    "be+ hidratante": "piel_seca",
    "be+ nutritiva": "piel_seca",
    # Solar
    "be+ fotoprotector": "proteccion_solar",
    "be+ spf": "proteccion_solar",
    "be+ solar": "proteccion_solar",
    # Bebé/Pediatría
    "be+ pediatrics": "higiene_bebe",
    "be+ bebe": "higiene_bebe",
    "be+ bebé": "higiene_bebe",
    # Contorno ojos
    "be+ contorno": "contorno_ojos",
    # Hombre
    "be+ hombre": "cuidado_facial_general",
    # Labios
    "be+ labios": "labios_secos",

    # =========================================================================
    # MEDIK8 (medik8.com) - Web Scraping Dic 2024
    # =========================================================================
    # Retinol/Vitamina A (Antiedad)
    "medik8 retinal": "arrugas_antiedad",
    "medik8 crystal retinal": "arrugas_antiedad",
    "crystal retinal": "arrugas_antiedad",
    "medik8 retinol": "arrugas_antiedad",
    "medik8 intelligent retinol": "arrugas_antiedad",
    "intelligent retinol": "arrugas_antiedad",
    "medik8 r-retinoate": "arrugas_antiedad",
    "r-retinoate": "arrugas_antiedad",
    # Vitamina C
    "medik8 c-tetra": "arrugas_antiedad",
    "c-tetra": "arrugas_antiedad",
    "medik8 super c": "arrugas_antiedad",
    "medik8 daily radiance": "arrugas_antiedad",
    # Manchas
    "medik8 white balance": "manchas",
    "medik8 clarity peptides": "manchas",
    # Hidratación
    "medik8 hydr8": "piel_seca",
    "medik8 advanced day": "piel_seca",
    "medik8 ultimate recovery": "piel_sensible",
    # Exfoliación
    "medik8 press refresh": "exfoliacion",
    "medik8 pore refining": "poros",
    # Ojos
    "medik8 eyelift": "contorno_ojos",
    "medik8 eye": "contorno_ojos",
    # Solar
    "medik8 advanced day total protect": "proteccion_solar",
    # Limpieza
    "medik8 surface radiance": "limpieza_facial",
    "medik8 micellar mousse": "limpieza_facial",
    "medik8 cream cleanse": "limpieza_facial",

    # =========================================================================
    # BIODERMA (bioderma.es) - Web Scraping Dic 2024
    # =========================================================================
    # SENSIBIO - Piel sensible, rojeces
    "bioderma sensibio": "piel_sensible",
    "sensibio h2o": "limpieza_facial",  # Agua micelar icónica
    "sensibio gel": "limpieza_facial",
    "sensibio foaming": "limpieza_facial",
    "sensibio defensive": "piel_sensible",
    "sensibio forte": "rojeces_rosacea",
    "sensibio ar": "rojeces_rosacea",  # Anti-redness
    "sensibio tolerance": "piel_sensible",
    "sensibio eye": "contorno_ojos",
    "sensibio": "piel_sensible",  # Catch-all línea
    # ATODERM - Piel atópica, muy seca
    "bioderma atoderm": "piel_atopica",
    "atoderm intensive": "piel_atopica",
    "atoderm baume": "piel_atopica",
    "atoderm sos spray": "piel_atopica",
    "atoderm preventive": "piel_atopica",
    "atoderm gel douche": "higiene_corporal",
    "atoderm shower": "higiene_corporal",
    "atoderm huile": "higiene_corporal",
    "atoderm creme": "piel_seca",
    "atoderm": "piel_atopica",  # Catch-all línea
    # SEBIUM - Acné, piel grasa
    "bioderma sebium": "acne",
    "sebium h2o": "limpieza_facial",  # Agua micelar piel grasa
    "sebium gel": "limpieza_facial",
    "sebium foaming": "limpieza_facial",
    "sebium mat": "acne",  # Control brillo
    "sebium pore refiner": "poros",
    "sebium global": "acne",
    "sebium sensitive": "acne",
    "sebium night": "acne",
    "sebium serum": "acne",
    "sebium": "acne",  # Catch-all línea
    # PHOTODERM - Protección solar
    "bioderma photoderm": "proteccion_solar",
    "photoderm max": "proteccion_solar",
    "photoderm spot": "manchas",  # Solar + antimanchas
    "photoderm spot age": "manchas",
    "photoderm nude": "proteccion_solar",
    "photoderm bronz": "proteccion_solar",
    "photoderm kid": "proteccion_solar",
    "photoderm pediatrics": "proteccion_solar",
    "photoderm aquafluide": "proteccion_solar",
    "photoderm mineral": "proteccion_solar",
    "photoderm cover touch": "proteccion_solar",
    "photoderm ar": "rojeces_rosacea",  # Solar + anti-rojeces
    "photoderm m": "manchas",  # Solar + melasma
    "photoderm laser": "cicatrices_estrias",  # Post-procedimiento
    "photoderm": "proteccion_solar",  # Catch-all línea
    # HYDRABIO - Piel deshidratada
    "bioderma hydrabio": "piel_seca",
    "hydrabio h2o": "limpieza_facial",
    "hydrabio gel": "limpieza_facial",
    "hydrabio serum": "piel_seca",
    "hydrabio perfecteur": "piel_seca",
    "hydrabio creme": "piel_seca",
    "hydrabio legere": "piel_seca",
    "hydrabio masque": "piel_seca",
    "hydrabio": "piel_seca",  # Catch-all línea
    # CICABIO - Reparación, heridas
    "bioderma cicabio": "cicatrices_estrias",
    "cicabio creme": "cicatrices_estrias",
    "cicabio lotion": "cicatrices_estrias",
    "cicabio pommade": "cicatrices_estrias",
    "cicabio arnica": "cicatrices_estrias",
    "cicabio": "cicatrices_estrias",  # Catch-all línea
    # PIGMENTBIO - Manchas, hiperpigmentación
    "bioderma pigmentbio": "manchas",
    "pigmentbio daily": "manchas",
    "pigmentbio sensitive": "manchas",
    "pigmentbio night": "manchas",
    "pigmentbio c-concentrate": "manchas",
    "pigmentbio foaming": "limpieza_facial",
    "pigmentbio h2o": "limpieza_facial",
    "pigmentbio": "manchas",  # Catch-all línea
    # NODE - Cabello
    "bioderma node": "higiene_cabello",
    "node a": "higiene_cabello",  # Calmante
    "node ds": "caspa",  # Anticaspa
    "node k": "caspa",  # Queratorregulador
    "node fluid": "higiene_cabello",
    "node g": "higiene_cabello",  # Grasa
    # ABCDerm - Bebé
    "bioderma abcderm": "higiene_bebe",
    "abcderm": "higiene_bebe",

    # =========================================================================
    # INSTITUT ESTHEDERM (esthederm.es) - Web Scraping Dic 2024
    # =========================================================================
    # Limpieza - Osmoclean
    "esthederm osmoclean": "limpieza_facial",
    "osmoclean": "limpieza_facial",
    # Agua Celular - Base
    "esthederm eau cellulaire": "piel_seca",
    "eau cellulaire": "piel_seca",
    "esthederm cellular water": "piel_seca",
    # Age Proteom - Antiedad celular
    "esthederm age proteom": "arrugas_antiedad",
    "age proteom": "arrugas_antiedad",
    # Active Repair - Reparación
    "esthederm active repair": "arrugas_antiedad",
    "active repair": "arrugas_antiedad",
    # Lift & Repair - Firmeza
    "esthederm lift repair": "arrugas_antiedad",
    "lift repair": "arrugas_antiedad",
    "esthederm lift and repair": "arrugas_antiedad",
    # Excellage - Piel madura
    "esthederm excellage": "arrugas_antiedad",
    "excellage": "arrugas_antiedad",
    # Intensive Collection
    "esthederm intensive retinol": "arrugas_antiedad",
    "intensive retinol": "arrugas_antiedad",
    "esthederm intensive hyaluronic": "piel_seca",
    "intensive hyaluronic": "piel_seca",
    "esthederm intensive vitamine": "arrugas_antiedad",
    "intensive vitamin": "arrugas_antiedad",
    "esthederm intensive c": "arrugas_antiedad",
    # Pro-Collagen - Colágeno
    "esthederm pro-collagen": "arrugas_antiedad",
    "esthederm pro collagen": "arrugas_antiedad",
    "pro-collagen": "arrugas_antiedad",
    # Esthe-White - Manchas
    "esthederm esthe-white": "manchas",
    "esthederm esthe white": "manchas",
    "esthe-white": "manchas",
    "esthe white": "manchas",
    # Solares - Adaptasun, Bronz Repair
    "esthederm adaptasun": "proteccion_solar",
    "adaptasun": "proteccion_solar",
    "esthederm bronz repair": "proteccion_solar",
    "bronz repair": "proteccion_solar",
    "esthederm photo reverse": "manchas",  # Fotoenvejecimiento
    "photo reverse": "manchas",
    "esthederm sun ultimate": "proteccion_solar",
    "esthederm no sun": "autobronceador",
    "no sun": "autobronceador",
    # Cuerpo
    "esthederm body": "cuidado_corporal",
    # Catch-all
    "institut esthederm": "arrugas_antiedad",
    "esthederm": "arrugas_antiedad",

    # =========================================================================
    # CUMLAUDE LAB (cumlaudelab.com) - Web Scraping Dic 2024
    # =========================================================================
    # CLX - Antiséptico (clorhexidina)
    "cumlaude clx": "higiene_intima",
    "cumlaude higiene intima clx": "higiene_intima",
    "clx higiene": "higiene_intima",
    "clx mousse": "higiene_intima",
    "clx gel": "higiene_intima",
    "clx ovulos": "higiene_intima",
    "clx lavado vaginal": "higiene_intima",
    # Gynelaude - Línea ginecológica
    "gynelaude": "higiene_intima",
    "cumlaude gynelaude": "higiene_intima",
    # Higiene íntima diaria
    "cumlaude higiene intima": "higiene_intima",
    "cumlaude origyn": "higiene_intima",
    "origyn higiene": "higiene_intima",
    "cumlaude deligyn": "higiene_intima",
    "deligyn": "higiene_intima",
    # Pediatric
    "cumlaude pediatrics": "higiene_bebe",
    "cumlaude pediatric": "higiene_bebe",
    # Hidratación vaginal
    "cumlaude hidratante interno": "sequedad_vaginal",
    "cumlaude hidratante externo": "sequedad_vaginal",
    "cumlaude lubricante": "salud_sexual",
    # Lubripiù - Sequedad
    "cumlaude lubripiu": "sequedad_vaginal",
    "lubripiu": "sequedad_vaginal",
    "lubripiù": "sequedad_vaginal",
    # Hydragyn - Hidratación íntima
    "cumlaude hydragyn": "sequedad_vaginal",
    "hydragyn": "sequedad_vaginal",
    # Prebiotic - Flora íntima
    "cumlaude prebiotic": "higiene_intima",
    # Catch-all (ya existe, pero más específico)
    "cumlaude lab": "higiene_intima",

    # =========================================================================
    # SVR LABORATORY (svr.com) - Web Scraping Dic 2024
    # =========================================================================
    # SEBIACLEAR - Acné, piel grasa
    "svr sebiaclear active": "acne",
    "svr sebiaclear serum": "acne",
    "svr sebiaclear mat": "acne",
    "svr sebiaclear hydra": "acne",
    "sebiaclear active": "acne",
    "sebiaclear serum": "acne",
    "sebiaclear mat": "acne",
    "sebiaclear gel": "limpieza_facial",
    "sebiaclear foaming": "limpieza_facial",
    # TOPIALYSE - Piel atópica
    "svr topialyse baume": "piel_atopica",
    "svr topialyse creme": "piel_atopica",
    "svr topialyse gel": "higiene_corporal",
    "topialyse baume": "piel_atopica",
    "topialyse creme": "piel_atopica",
    "topialyse gel lavant": "higiene_corporal",
    "topialyse intensive": "piel_atopica",
    # XERIAL - Piel muy seca, hiperqueratosis
    "svr xerial": "piel_seca",
    "xerial 10": "piel_seca",
    "xerial 30": "piel_seca",
    "xerial 40": "piel_seca",
    "xerial 50": "piel_seca",
    "xerial pied": "piel_seca",
    # CICAVIT+ - Reparación
    "svr cicavit": "cicatrices_estrias",
    "cicavit+": "cicatrices_estrias",
    "cicavit sos": "cicatrices_estrias",
    "cicavit creme": "cicatrices_estrias",
    # SUN SECURE - Solar
    "svr sun secure": "proteccion_solar",
    "sun secure blur": "proteccion_solar",
    "sun secure fluide": "proteccion_solar",
    "sun secure extreme": "proteccion_solar",
    # CLAIRIAL - Manchas
    "svr clairial": "manchas",
    "clairial day": "manchas",
    "clairial night": "manchas",
    "clairial serum": "manchas",
    "clairial creme": "manchas",
    # DENSITIUM - Antiedad piel madura
    "svr densitium": "arrugas_antiedad",
    "densitium creme": "arrugas_antiedad",
    "densitium contour": "contorno_ojos",
    "densitium serum": "arrugas_antiedad",
    # SENSIFINE - Piel sensible
    "svr sensifine ar": "rojeces_rosacea",
    "sensifine ar": "rojeces_rosacea",
    "svr sensifine dermo": "piel_sensible",
    "sensifine dermo": "piel_sensible",
    # HYDRALIANE - Hidratación
    "svr hydraliane": "piel_seca",
    "hydraliane legere": "piel_seca",
    "hydraliane riche": "piel_seca",
    # SPIRIAL - Desodorante
    "svr spirial": "desodorante",
    "spirial deo": "desodorante",
    "spirial extreme": "desodorante",
    # ABC (Antiedad)
    "svr abc": "arrugas_antiedad",
    "svr a": "arrugas_antiedad",  # Vitamina A
    "svr b3": "arrugas_antiedad",  # Niacinamida
    "svr c": "arrugas_antiedad",  # Vitamina C

    # =========================================================================
    # CANTABRIA LABS (cantabrialabs.com) - Web Scraping Dic 2024
    # =========================================================================
    # HELIOCARE - Solar (ya tiene entradas, añadir más específicas)
    "heliocare water gel": "proteccion_solar",
    "heliocare mineral": "proteccion_solar",
    "heliocare airgel": "proteccion_solar",
    "heliocare compact": "proteccion_solar",
    "heliocare pigment": "manchas",  # Pigment Solution
    "heliocare age active": "arrugas_antiedad",  # Age Active Fluid
    # ENDOCARE - Antiedad (expandir)
    "endocare renewal": "arrugas_antiedad",
    "endocare cellage": "arrugas_antiedad",
    "endocare c ferulic": "arrugas_antiedad",
    "endocare essential": "arrugas_antiedad",
    "endocare eye contour": "contorno_ojos",
    "endocare day": "arrugas_antiedad",
    "endocare night": "arrugas_antiedad",
    # NEORETIN - Manchas (expandir)
    "neoretin discrom": "manchas",
    "neoretin gelcream": "manchas",
    "neoretin serum": "manchas",
    "neoretin pigment": "manchas",
    "neoretin ultra": "manchas",
    # BIRETIX - Acné (expandir)
    "biretix tri-active": "acne",
    "biretix duo": "acne",
    "biretix mask": "acne",
    "biretix gel": "limpieza_facial",
    "biretix cleanser": "limpieza_facial",
    # IFC - Cicatrices
    "ifc derma": "cicatrices_estrias",
    # IRALTONE - Cabello (Cantabria Labs)
    "iraltone forte": "caida_cabello",
    "iraltone aga": "caida_cabello",
    "iraltone ds": "caspa",
    "iraltone champu": "higiene_cabello",

    # =========================================================================
    # LA ROCHE-POSAY (laroche-posay.es) - Web Scraping Dic 2024
    # =========================================================================
    # EFFACLAR - Acné, piel grasa
    "la roche posay effaclar": "acne",
    "la roche-posay effaclar": "acne",
    "effaclar duo": "acne",
    "effaclar h": "acne",
    "effaclar k": "acne",
    "effaclar mat": "acne",
    "effaclar serum": "acne",
    "effaclar gel": "limpieza_facial",
    "effaclar agua micelar": "limpieza_facial",
    "effaclar purifying": "limpieza_facial",
    "effaclar": "acne",  # Catch-all línea
    # TOLERIANE - Piel sensible
    "la roche posay toleriane": "piel_sensible",
    "la roche-posay toleriane": "piel_sensible",
    "toleriane ultra": "piel_sensible",
    "toleriane sensitive": "piel_sensible",
    "toleriane rosaliac": "rojeces_rosacea",
    "toleriane dermallergo": "piel_sensible",
    "toleriane double repair": "piel_sensible",
    "toleriane caring wash": "limpieza_facial",
    "toleriane dermo": "limpieza_facial",
    "toleriane": "piel_sensible",  # Catch-all línea
    # LIPIKAR - Piel atópica, muy seca
    "la roche posay lipikar": "piel_atopica",
    "la roche-posay lipikar": "piel_atopica",
    "lipikar baume": "piel_atopica",
    "lipikar ap+": "piel_atopica",
    "lipikar ap": "piel_atopica",
    "lipikar eczema": "piel_atopica",
    "lipikar syndet": "higiene_corporal",
    "lipikar surgras": "higiene_corporal",
    "lipikar huile": "higiene_corporal",
    "lipikar lait": "piel_seca",
    "lipikar stick": "piel_atopica",
    "lipikar xerand": "manos",
    "lipikar": "piel_atopica",  # Catch-all línea
    # ANTHELIOS - Protección solar
    "la roche posay anthelios": "proteccion_solar",
    "la roche-posay anthelios": "proteccion_solar",
    "anthelios uvmune": "proteccion_solar",
    "anthelios age correct": "proteccion_solar",
    "anthelios pigment correct": "manchas",  # Solar + antimanchas
    "anthelios oil correct": "proteccion_solar",
    "anthelios invisible": "proteccion_solar",
    "anthelios mineral": "proteccion_solar",
    "anthelios hydrating": "proteccion_solar",
    "anthelios shaka": "proteccion_solar",
    "anthelios pocket": "proteccion_solar",
    "anthelios dermo": "proteccion_solar",
    "anthelios spray": "proteccion_solar",
    "anthelios xl": "proteccion_solar",
    "anthelios": "proteccion_solar",  # Catch-all línea
    # CICAPLAST - Reparación
    "la roche posay cicaplast": "cicatrices_estrias",
    "la roche-posay cicaplast": "cicatrices_estrias",
    "cicaplast baume": "cicatrices_estrias",
    "cicaplast b5": "cicatrices_estrias",
    "cicaplast gel": "cicatrices_estrias",
    "cicaplast mains": "manos",
    "cicaplast lavant": "limpieza_facial",
    "cicaplast": "cicatrices_estrias",  # Catch-all línea
    # REDERMIC - Antiedad
    "la roche posay redermic": "arrugas_antiedad",
    "la roche-posay redermic": "arrugas_antiedad",
    "redermic r": "arrugas_antiedad",
    "redermic c": "arrugas_antiedad",
    "redermic retinol": "arrugas_antiedad",
    "redermic": "arrugas_antiedad",  # Catch-all línea
    # HYALU B5 - Hidratación antiedad
    "la roche posay hyalu b5": "arrugas_antiedad",
    "hyalu b5 serum": "arrugas_antiedad",
    "hyalu b5 aquagel": "piel_seca",
    "hyalu b5 ojos": "contorno_ojos",
    "hyalu b5": "arrugas_antiedad",
    # PIGMENTCLAR - Manchas
    "la roche posay pigmentclar": "manchas",
    "pigmentclar serum": "manchas",
    "pigmentclar eyes": "contorno_ojos",
    "pigmentclar": "manchas",
    # ISO-UREA - Piel muy seca
    "la roche posay iso-urea": "piel_seca",
    "iso-urea md": "piel_seca",
    "iso-urea": "piel_seca",
    # PURE VITAMIN C - Antioxidante
    "la roche posay pure vitamin c": "arrugas_antiedad",
    "pure vitamin c10": "arrugas_antiedad",
    "pure vitamin c": "arrugas_antiedad",
    # SUBSTIANE - Densidad piel madura
    "la roche posay substiane": "arrugas_antiedad",
    "substiane": "arrugas_antiedad",
    # KERIUM - Cabello
    "la roche posay kerium": "higiene_cabello",
    "kerium ds": "caspa",
    "kerium anticaida": "caida_cabello",
    "kerium doux": "higiene_cabello",
    "kerium": "higiene_cabello",
    # RESPECTISSIME - Maquillaje sensible
    "respectissime": "maquillaje",
    # Catch-all marca
    "la roche-posay": "piel_sensible",
    "la roche posay": "piel_sensible",
    "laroche posay": "piel_sensible",

    # =========================================================================
    # VICHY (vichy.es) - Web Scraping Dic 2025
    # =========================================================================
    # LIFTACTIV - Antiedad
    "vichy liftactiv": "arrugas_antiedad",
    "liftactiv supreme": "arrugas_antiedad",
    "liftactiv collagen": "arrugas_antiedad",
    "liftactiv specialist": "arrugas_antiedad",
    "liftactiv peptide-c": "arrugas_antiedad",
    "liftactiv retinol": "arrugas_antiedad",
    "liftactiv h.a.": "arrugas_antiedad",
    "liftactiv noche": "arrugas_antiedad",
    "liftactiv ojos": "contorno_ojos",
    "liftactiv": "arrugas_antiedad",
    # MINERAL 89 - Hidratación, barrera
    "vichy mineral 89": "piel_seca",
    "mineral 89 booster": "piel_seca",
    "mineral 89 serum": "piel_seca",
    "mineral 89 ojos": "contorno_ojos",
    "mineral 89 probiotic": "piel_sensible",
    "mineral 89": "piel_seca",
    # NORMADERM - Acné, piel grasa
    "vichy normaderm": "acne",
    "normaderm phytosolution": "acne",
    "normaderm probio-bha": "acne",
    "normaderm double correction": "acne",
    "normaderm sos": "acne",
    "normaderm gel": "limpieza_facial",
    "normaderm cleanser": "limpieza_facial",
    "normaderm": "acne",
    # CAPITAL SOLEIL - Solar
    "vichy capital soleil": "proteccion_solar",
    "capital soleil uv-age": "proteccion_solar",
    "capital soleil dry touch": "proteccion_solar",
    "capital soleil invisible": "proteccion_solar",
    "capital soleil beach": "proteccion_solar",
    "capital soleil cell protect": "proteccion_solar",
    "capital soleil anti-dark": "manchas",
    "capital soleil": "proteccion_solar",
    # NEOVADIOL - Menopausia, piel madura
    "vichy neovadiol": "arrugas_antiedad",
    "neovadiol meno": "arrugas_antiedad",
    "neovadiol peri": "arrugas_antiedad",
    "neovadiol post": "arrugas_antiedad",
    "neovadiol rose": "arrugas_antiedad",
    "neovadiol magistral": "arrugas_antiedad",
    "neovadiol": "arrugas_antiedad",
    # AQUALIA THERMAL - Hidratación
    "vichy aqualia": "piel_seca",
    "aqualia thermal": "piel_seca",
    "aqualia serum": "piel_seca",
    "aqualia": "piel_seca",
    # IDEALIA - Luminosidad
    "vichy idealia": "arrugas_antiedad",
    "idealia": "arrugas_antiedad",
    # SLOW AGE - Prevención antiedad
    "vichy slow age": "arrugas_antiedad",
    "slow age": "arrugas_antiedad",
    # PURETE THERMALE - Limpieza
    "vichy purete thermale": "limpieza_facial",
    "purete thermale": "limpieza_facial",
    # DERCOS - Cabello
    "vichy dercos": "higiene_cabello",
    "dercos aminexil": "caida_cabello",
    "dercos densi-solutions": "caida_cabello",
    "dercos energisant": "caida_cabello",
    "dercos anti-caspa": "caspa",
    "dercos antipelliculaire": "caspa",
    "dercos kera-solutions": "higiene_cabello",
    "dercos nutrients": "higiene_cabello",
    "dercos": "higiene_cabello",
    # HOMME - Masculino
    "vichy homme hydra mag": "cuidado_facial_general",
    "vichy homme sensi baume": "afeitado",
    # Catch-all
    "vichy": "cuidado_facial_general",

    # =========================================================================
    # AVÈNE (avene.es) - Web Scraping Dic 2025
    # =========================================================================
    # CICALFATE+ - Reparación
    "avene cicalfate": "cicatrices_estrias",
    "cicalfate+": "cicatrices_estrias",
    "cicalfate plus": "cicatrices_estrias",
    "cicalfate creme": "cicatrices_estrias",
    "cicalfate mains": "manos",
    "cicalfate levres": "labios",
    "cicalfate": "cicatrices_estrias",
    # TOLERANCE - Piel hipersensible
    "avene tolerance": "piel_sensible",
    "tolerance control": "piel_sensible",
    "tolerance extreme": "piel_sensible",
    "tolerance hydra-10": "piel_sensible",
    "tolerance": "piel_sensible",
    # CLEANANCE - Acné, piel grasa
    "avene cleanance": "acne",
    "cleanance comedomed": "acne",
    "cleanance expert": "acne",
    "cleanance mat": "acne",
    "cleanance hydra": "acne",
    "cleanance solaire": "proteccion_solar",
    "cleanance gel": "limpieza_facial",
    "cleanance agua micelar": "limpieza_facial",
    "cleanance": "acne",
    # SOLAIRE - Solar
    "avene solaire": "proteccion_solar",
    "avene spf": "proteccion_solar",
    "avene solar": "proteccion_solar",
    "avene sun": "proteccion_solar",
    "very high protection": "proteccion_solar",
    "tres haute protection": "proteccion_solar",
    # HYDRANCE - Hidratación
    "avene hydrance": "piel_seca",
    "hydrance optimale": "piel_seca",
    "hydrance aqua-gel": "piel_seca",
    "hydrance legere": "piel_seca",
    "hydrance riche": "piel_seca",
    "hydrance": "piel_seca",
    # XERACALM - Piel atópica
    "avene xeracalm": "piel_atopica",
    "xeracalm a.d": "piel_atopica",
    "xeracalm lipid": "piel_atopica",
    "xeracalm": "piel_atopica",
    # A-OXITIVE - Antioxidante antiedad
    "avene a-oxitive": "arrugas_antiedad",
    "a-oxitive serum": "arrugas_antiedad",
    "a-oxitive": "arrugas_antiedad",
    # PHYSIOLIFT - Antiedad
    "avene physiolift": "arrugas_antiedad",
    "physiolift": "arrugas_antiedad",
    # DERMABSOLU - Piel madura
    "avene dermabsolu": "arrugas_antiedad",
    "dermabsolu": "arrugas_antiedad",
    # EAU THERMALE - Agua termal
    "avene eau thermale": "piel_sensible",
    "eau thermale avene": "piel_sensible",
    # PEDIATRIL - Bebé
    "avene pediatril": "higiene_bebe",
    "pediatril": "higiene_bebe",
    # COLD CREAM - Piel seca
    "avene cold cream": "piel_seca",
    "cold cream avene": "piel_seca",
    # COUVRANCE - Maquillaje
    "avene couvrance": "maquillaje",
    "couvrance": "maquillaje",
    # Catch-all
    "avene": "piel_sensible",
    "avène": "piel_sensible",

    # =========================================================================
    # EUCERIN (eucerin.es) - Web Scraping Dic 2025
    # =========================================================================
    # HYALURON-FILLER - Antiedad
    "eucerin hyaluron-filler": "arrugas_antiedad",
    "eucerin hyaluron filler": "arrugas_antiedad",
    "hyaluron-filler 3x effect": "arrugas_antiedad",
    "hyaluron-filler elasticity": "arrugas_antiedad",
    "hyaluron-filler volume": "arrugas_antiedad",
    "hyaluron-filler vitamin c": "arrugas_antiedad",
    "hyaluron-filler moisture": "arrugas_antiedad",
    "hyaluron-filler ojos": "contorno_ojos",
    "hyaluron-filler eye": "contorno_ojos",
    "hyaluron-filler urea": "piel_seca",
    "hyaluron-filler": "arrugas_antiedad",
    # DERMOPURE / DERMOPURIFYER - Acné
    "eucerin dermopure": "acne",
    "dermopure oil control": "acne",
    "dermopure mat": "acne",
    "dermopure serum": "acne",
    "dermopure cleanser": "limpieza_facial",
    "dermopure gel": "limpieza_facial",
    "dermopure clinical": "acne",
    "dermopurifyer": "acne",
    "dermopure": "acne",
    # ATOPICONTROL - Piel atópica
    "eucerin atopicontrol": "piel_atopica",
    "atopicontrol balsamo": "piel_atopica",
    "atopicontrol crema": "piel_atopica",
    "atopicontrol locion": "piel_atopica",
    "atopicontrol aceite": "higiene_corporal",
    "atopicontrol": "piel_atopica",
    # UREAREPAIR - Piel muy seca
    "eucerin urearepair": "piel_seca",
    "urearepair plus": "piel_seca",
    "urearepair original": "piel_seca",
    "urearepair pies": "pies_talones",
    "urearepair manos": "manos",
    "urearepair": "piel_seca",
    # SUN - Solar
    "eucerin sun": "proteccion_solar",
    "eucerin solar": "proteccion_solar",
    "eucerin spf": "proteccion_solar",
    "sun fluid": "proteccion_solar",
    "sun gel-cream": "proteccion_solar",
    "sun pigment control": "manchas",
    "sun actinic control": "proteccion_solar",
    "sun photoaging": "proteccion_solar",
    # AQUAPHOR - Reparación
    "eucerin aquaphor": "cicatrices_estrias",
    "aquaphor": "cicatrices_estrias",
    # PH5 - Higiene
    "eucerin ph5": "higiene_corporal",
    "ph5 gel": "higiene_corporal",
    "ph5 locion": "piel_seca",
    # ANTI-PIGMENT - Manchas
    "eucerin anti-pigment": "manchas",
    "eucerin antipigment": "manchas",
    "anti-pigment dual serum": "manchas",
    "anti-pigment": "manchas",
    # Catch-all
    "eucerin": "cuidado_facial_general",

    # =========================================================================
    # CERAVE (cerave.es) - Web Scraping Dic 2025
    # =========================================================================
    # SA - Ácido salicílico (textura, acné leve)
    "cerave sa": "acne",
    "cerave sa smoothing": "acne",
    "cerave sa cleanser": "limpieza_facial",
    "cerave sa lotion": "acne",
    "cerave sa cream": "acne",
    "sa smoothing": "acne",
    "sa cleanser": "limpieza_facial",
    # MOISTURIZING - Hidratación
    "cerave moisturizing": "piel_seca",
    "cerave moisturising": "piel_seca",
    "cerave crema hidratante": "piel_seca",
    "cerave locion hidratante": "piel_seca",
    # FOAMING - Limpieza piel normal/grasa
    "cerave foaming": "limpieza_facial",
    "cerave foaming cleanser": "limpieza_facial",
    "cerave espumoso": "limpieza_facial",
    # HYDRATING - Limpieza piel seca
    "cerave hydrating": "limpieza_facial",
    "cerave hydrating cleanser": "limpieza_facial",
    "cerave cream-to-foam": "limpieza_facial",
    # PM/AM - Facial
    "cerave pm": "piel_seca",
    "cerave am": "proteccion_solar",
    "cerave facial moisturising": "piel_seca",
    "cerave facial moisturizing": "piel_seca",
    # EYE REPAIR - Contorno ojos
    "cerave eye repair": "contorno_ojos",
    "cerave ojos": "contorno_ojos",
    # ADVANCED REPAIR - Reparación
    "cerave advanced repair": "piel_seca",
    "cerave healing": "cicatrices_estrias",
    # BLEMISH CONTROL - Acné
    "cerave blemish control": "acne",
    "cerave acne": "acne",
    # BABY - Bebé
    "cerave baby": "higiene_bebe",
    # RESURFACING - Retinol
    "cerave resurfacing": "arrugas_antiedad",
    "cerave retinol": "arrugas_antiedad",
    "cerave skin renewing": "arrugas_antiedad",
    # Catch-all
    "cerave": "piel_seca",

    # =========================================================================
    # MEDICAMENTOS OTC - GRIPE/RESFRIADO/CATARRO
    # =========================================================================
    "frenadol": "gripe_resfriado",
    "frenadol complex": "gripe_resfriado",
    "frenadol descongestivo": "gripe_resfriado",
    "frenadol forte": "gripe_resfriado",
    "normogrip": "gripe_resfriado",
    "ilvico": "gripe_resfriado",
    "couldina con ibuprofeno": "gripe_resfriado",
    "couldina ibuprofeno": "gripe_resfriado",
    "couldina instant": "gripe_resfriado",
    "couldina": "gripe_resfriado",
    "pharmagrip": "gripe_resfriado",
    "bisolgrip": "gripe_resfriado",
    "gripavicks": "gripe_resfriado",
    "vincigrip": "gripe_resfriado",
    "termalgin gripe": "gripe_resfriado",
    "antigripal": "gripe_resfriado",
    "grippal": "gripe_resfriado",
    "propalgina": "gripe_resfriado",
    "desenfriol": "gripe_resfriado",

    # =========================================================================
    # MEDICAMENTOS OTC - TOS / MUCOLÍTICOS / EXPECTORANTES
    # =========================================================================
    "fluimucil": "tos_garganta",
    "fluimucil forte": "tos_garganta",
    "flutox": "tos_garganta",
    "cinfamucol": "tos_garganta",
    "mucosan": "tos_garganta",
    "mucosolvan": "tos_garganta",
    "pectox": "tos_garganta",
    "bisolvon": "tos_garganta",
    "romilar": "tos_garganta",
    "iniston": "tos_garganta",
    "tussidril": "tos_garganta",
    "histiacil": "tos_garganta",
    "cinfatos": "tos_garganta",
    "robitussin": "tos_garganta",
    "cloperastina": "tos_garganta",
    "acetilcisteina": "tos_garganta",
    "ambroxol": "tos_garganta",
    "dextrometorfano": "tos_garganta",
    "frionex": "tos_garganta",

    # =========================================================================
    # MEDICAMENTOS OTC - DOLOR DE GARGANTA / FARINGITIS
    # =========================================================================
    "bucomax": "dolor_garganta",
    "strefen": "dolor_garganta",
    "strepsils": "dolor_garganta",
    "lizipaina": "dolor_garganta",
    "angileptol": "dolor_garganta",
    "faringesic": "dolor_garganta",
    "faringospray": "dolor_garganta",
    "bucometasana": "dolor_garganta",
    "mebucaina": "dolor_garganta",
    "drill": "dolor_garganta",
    "pharysol": "dolor_garganta",
    "tantum verde": "dolor_garganta",

    # =========================================================================
    # MEDICAMENTOS OTC - DOLOR / ANTIINFLAMATORIOS / ANALGÉSICOS
    # =========================================================================
    # Ibuprofeno
    "nurofen": "dolor_analgesico",
    "nurofen rapid": "dolor_analgesico",
    "espididol": "dolor_analgesico",
    "ibudol": "dolor_analgesico",
    "ibufen": "dolor_analgesico",
    "dalsy": "dolor_analgesico",
    "junifen": "dolor_analgesico",
    "ibuprofeno": "dolor_analgesico",
    "advil": "dolor_analgesico",
    "saetil": "dolor_analgesico",
    # Paracetamol
    "gelocatil": "dolor_analgesico",
    "termalgin": "dolor_analgesico",
    "dolostop": "dolor_analgesico",
    "efferalgan": "dolor_analgesico",
    "paracetamol": "dolor_analgesico",
    "apiretal": "dolor_analgesico",
    "febrectal": "dolor_analgesico",
    # Combinados
    "actron": "dolor_analgesico",
    "actron compuesto": "dolor_analgesico",
    "nolotil": "dolor_analgesico",
    "metamizol": "dolor_analgesico",
    "buscapina": "dolor_analgesico",
    "dolo-neurobion": "dolor_analgesico",
    # Tópicos antiinflamatorios
    "reflex": "dolor_muscular",
    "reflex spray": "dolor_muscular",
    "diclokern": "dolor_muscular",
    "voltaren": "dolor_muscular",
    "voltadol": "dolor_muscular",
    "flogoprofen": "dolor_muscular",
    # calmiox REMOVED - es hidrocortisona (dermatitis), no dolor muscular
    "calmatel": "dolor_muscular",
    "fastum": "dolor_muscular",
    "osteoren": "dolor_muscular",
    "flekosteel": "dolor_muscular",
    # Migraña
    "migraleve": "dolor_cabeza",
    "migralivia": "dolor_cabeza",
    "hemicraneal": "dolor_cabeza",

    # =========================================================================
    # MEDICAMENTOS OTC - CIRCULACIÓN / PIERNAS CANSADAS / VARICES
    # =========================================================================
    "daflon": "circulacion_piernas",
    "variliv": "circulacion_piernas",
    "venoruton": "circulacion_piernas",
    "venosmil": "circulacion_piernas",
    "diosmina": "circulacion_piernas",
    "flebinec": "circulacion_piernas",
    "fabovenol": "circulacion_piernas",
    "venosmine": "circulacion_piernas",
    "fleboside": "circulacion_piernas",
    "diosmiven": "circulacion_piernas",
    "venolep": "circulacion_piernas",

    # =========================================================================
    # MEDICAMENTOS OTC - DIARREA / ANTIDIARREICOS
    # =========================================================================
    "fortasec": "diarrea",
    "salvacolina": "diarrea",
    "loperamida": "diarrea",
    "tiorfan": "diarrea",
    "tanagel": "diarrea",
    "ultra-levura": "flora_intestinal",
    "ultralevura": "flora_intestinal",

    # =========================================================================
    # MEDICAMENTOS OTC - HEMORROIDES
    # =========================================================================
    "anso pomada": "hemorroides",
    "hemoal": "hemorroides",
    "hemoal forte": "hemorroides",
    "hemoclin": "hemorroides",
    "proctolog": "hemorroides",
    "ruscus llorens": "hemorroides",
    "preparation h": "hemorroides",

    # =========================================================================
    # MEDICAMENTOS OTC - ESTREÑIMIENTO / LAXANTES
    # =========================================================================
    "dulcolaxo": "estrenimiento",
    "dulcolax": "estrenimiento",
    "bisacodilo": "estrenimiento",
    "micralax": "estrenimiento",
    "enema casen": "estrenimiento",
    "movicol": "estrenimiento",
    "duphalac": "estrenimiento",
    "lactulosa": "estrenimiento",
    "laxante": "estrenimiento",
    "evacuol": "estrenimiento",
    "plantaben": "estrenimiento",
    "agiolax": "estrenimiento",
    "fave de fuca": "estrenimiento",

    # =========================================================================
    # MEDICAMENTOS OTC - ACIDEZ / ANTIÁCIDOS / DIGESTIVO
    # =========================================================================
    "omeprazol": "acidez_reflujo",
    "omereflux": "acidez_reflujo",
    "almax": "acidez_reflujo",
    "gaviscon": "acidez_reflujo",
    "rennie": "acidez_reflujo",
    "maalox": "acidez_reflujo",
    "almagato": "acidez_reflujo",
    "pantoprazol": "acidez_reflujo",
    "nexium": "acidez_reflujo",
    "aero red": "gases_digestion",
    "aerored": "gases_digestion",
    "simeticona": "gases_digestion",
    "meteospasmyl": "gases_digestion",
    "carminativo": "gases_digestion",
    "digest": "gases_digestion",
    "digestivo": "gases_digestion",
    "magnogene": "acidez_reflujo",
    "sal de fruta": "acidez_reflujo",

    # =========================================================================
    # MEDICAMENTOS OTC - ALERGIAS / ANTIHISTAMÍNICOS
    # =========================================================================
    "reactine": "alergia",
    "cetirizina": "alergia",
    "loratadina": "alergia",
    "ebastel": "alergia",
    "ebastina": "alergia",
    "aerius": "alergia",
    "desloratadina": "alergia",
    "clarityne": "alergia",
    "polaramine": "alergia",
    "atarax": "alergia",
    "bilaxten": "alergia",
    "zyrtec": "alergia",
    "xazal": "alergia",
    "rupafin": "alergia",
    "antihistaminico": "alergia",

    # =========================================================================
    # MEDICAMENTOS OTC - SUEÑO / INSOMNIO
    # =========================================================================
    "dormidina": "sueno_insomnio",
    "doxilamina": "sueno_insomnio",
    "soñodor": "sueno_insomnio",
    "sominex": "sueno_insomnio",
    "nytol": "sueno_insomnio",

    # =========================================================================
    # MEDICAMENTOS OTC - HERPES LABIAL
    # =========================================================================
    "zovirax": "herpes_labial",
    "zoviduo": "herpes_labial",
    "lavirk": "herpes_labial",
    "aciclovir labial": "herpes_labial",
    "erazaban": "herpes_labial",
    "virucalm": "herpes_labial",
    "penciclovir": "herpes_labial",

    # =========================================================================
    # MEDICAMENTOS OTC - CONGESTIÓN NASAL / DESCONGESTIONANTES
    # =========================================================================
    "respibien": "congestion_nasal",
    "utabon": "congestion_nasal",
    "rhinospray": "congestion_nasal",
    "rhinodina": "congestion_nasal",
    "nasodren": "congestion_nasal",
    "oximetazolina": "congestion_nasal",
    "xilometazolina": "congestion_nasal",
    "vicks inhaler": "congestion_nasal",
    "vaporub": "congestion_nasal",
    "vibrocil": "congestion_nasal",
    "otrivina": "congestion_nasal",
    "sinomarin": "congestion_nasal",

    # =========================================================================
    # MEDICAMENTOS OTC - VITAMINAS B / ENERGÍA
    # =========================================================================
    "hidroxil": "vitaminas_general",
    "hidroxil b1-b6-b12": "vitaminas_general",
    "neurobion": "vitaminas_general",
    "medebiotin": "vitaminas_general",
    "becozyme": "vitaminas_general",
    "dayamineral": "vitaminas_general",
    "vigortonic": "energia_cansancio",

    # =========================================================================
    # MEDICAMENTOS OTC - ANTICONCEPTIVOS EMERGENCIA
    # =========================================================================
    "ellaone": "anticoncepcion",
    "postinor": "anticoncepcion",
    "norlevo": "anticoncepcion",
    "levonorgestrel": "anticoncepcion",
    "postday": "anticoncepcion",

    # =========================================================================
    # MEDICAMENTOS OTC - DEJAR DE FUMAR
    # =========================================================================
    "nicorette": "dejar_fumar",
    "nicotinell": "dejar_fumar",
    "niquitin": "dejar_fumar",
    "parche nicotina": "dejar_fumar",
    "chicle nicotina": "dejar_fumar",

    # =========================================================================
    # MEDICAMENTOS OTC - VERRUGAS / CALLOS
    # =========================================================================
    "antiverrugas": "verrugas_callos",
    "wartner": "verrugas_callos",
    "isdin verrugas": "verrugas_callos",
    "callicida": "verrugas_callos",
    "duofilm": "verrugas_callos",
    "scholl callos": "verrugas_callos",
    "scholl verrugas": "verrugas_callos",

    # =========================================================================
    # MEDICAMENTOS OTC - MAREO / CINETOSIS
    # =========================================================================
    "biodramina": "mareo_viaje",
    "cinfamar": "mareo_viaje",
    "dimenhidrinato": "mareo_viaje",
    "nauserina": "mareo_viaje",

    # =========================================================================
    # MEDICAMENTOS OTC - HONGOS / ANTIFÚNGICOS
    # =========================================================================
    "canesten": "hongos_pies",
    "clotrimazol": "hongos_pies",
    "fungarest": "hongos_pies",
    "lamisil": "hongos_pies",
    "terbinafina": "hongos_pies",
    "pevaryl": "hongos_pies",
    "daktarin": "hongos_pies",
    "gine-canesten": "hongos_pies",
    "canestene": "hongos_pies",
    "gine canesten": "hongos_pies",
    "brentan": "hongos_pies",

    # =========================================================================
    # ANTIINFLAMATORIOS TÓPICOS / DOLOR MUSCULAR Y ARTICULAR
    # =========================================================================
    "voltadol": "dolor_muscular",
    "voltadol forte": "dolor_muscular",
    "voltaren": "dolor_muscular",
    "voltaren emulgel": "dolor_muscular",
    "diclofenaco": "dolor_muscular",
    "diclofenaco gel": "dolor_muscular",
    "diclokern": "dolor_muscular",
    "diclokern forte": "dolor_muscular",
    "reflex": "dolor_muscular",
    "reflex spray": "dolor_muscular",
    "flogoprofen": "dolor_muscular",
    "radio salil": "dolor_muscular",
    "radio-salil": "dolor_muscular",
    "fastum": "dolor_muscular",
    "fastum gel": "dolor_muscular",
    "ketoprofeno": "dolor_muscular",
    "piroxicam gel": "dolor_muscular",
    "feldene": "dolor_muscular",
    "calmatel": "dolor_muscular",
    "fisiocrem": "dolor_muscular",
    "arnidol": "dolor_muscular",
    "arnica": "dolor_muscular",
    "arnica gel": "dolor_muscular",
    "arnica montana": "dolor_muscular",
    "traumeel": "dolor_muscular",
    "kytta": "dolor_muscular",
    "artrotec": "dolor_muscular",

    # =========================================================================
    # ANTISÉPTICOS / HERIDAS / PRIMEROS AUXILIOS
    # =========================================================================
    "cristalmina": "heridas_apositos",
    "cristalmina film": "heridas_apositos",
    "betadine": "heridas_apositos",
    "povidona": "heridas_apositos",
    "povidona yodada": "heridas_apositos",
    "clorhexidina": "heridas_apositos",
    "clorhexidina spray": "heridas_apositos",
    "alcohol 96": "heridas_apositos",
    "alcohol sanitario": "heridas_apositos",
    "agua oxigenada": "heridas_apositos",
    "mercromina": "heridas_apositos",
    "tiritas": "heridas_apositos",
    "aposito": "heridas_apositos",
    "apósito": "heridas_apositos",
    "hansaplast": "heridas_apositos",
    "suero fisiologico": "heridas_apositos",
    "suero fisiológico": "heridas_apositos",
    "gasas": "heridas_apositos",
    "vendas": "heridas_apositos",
    "esparadrapo": "heridas_apositos",

    # =========================================================================
    # HIGIENE ÓTICA / OÍDOS
    # =========================================================================
    "otocerum": "higiene_oidos",
    "audispray": "higiene_oidos",
    "audiclean": "higiene_oidos",
    "cerulysin": "higiene_oidos",
    "tapon oido": "higiene_oidos",
    "tapones oido": "higiene_oidos",
    "gotas oido": "higiene_oidos",
    "gotas oidos": "higiene_oidos",
    "cerumen": "higiene_oidos",

    # =========================================================================
    # DESCONGESTIONANTES TÓPICOS (PECHO/NASAL)
    # =========================================================================
    "vaporub": "congestion_nasal",
    "vicks vaporub": "congestion_nasal",
    "respirfix": "congestion_nasal",
    "utabon": "congestion_nasal",
    "rhinomer": "congestion_nasal",
    "rhinospray": "congestion_nasal",
    "respir": "congestion_nasal",

    # =========================================================================
    # CIRCULACIÓN / HEMATOMAS / VARICES (TÓPICO)
    # =========================================================================
    "thrombocid": "circulacion_piernas",
    "thrombocid forte": "circulacion_piernas",
    "hirudoid": "circulacion_piernas",
    "hirudoid forte": "circulacion_piernas",
    "hemovas": "circulacion_piernas",
    "venaliv": "circulacion_piernas",
    "venaliv gel": "circulacion_piernas",
    "venosmil": "circulacion_piernas",
    "tavonin": "circulacion_piernas",

    # =========================================================================
    # GASES / DIGESTIVO
    # =========================================================================
    "aero red": "gases_digestion",
    "aerored": "gases_digestion",
    "simeticona": "gases_digestion",
    "benegast": "gases_digestion",
    "gas-x": "gases_digestion",
    "imogas": "gases_digestion",
    "digestivo": "gases_digestion",
    "antiácido": "acidez_reflujo",

    # =========================================================================
    # PROBIÓTICOS / FLORA INTESTINAL
    # =========================================================================
    "ultra levura": "flora_intestinal",
    "ultra-levura": "flora_intestinal",
    "ultralevura": "flora_intestinal",
    "prodefen": "flora_intestinal",
    "lactoflora": "flora_intestinal",
    "nutriflora": "flora_intestinal",
    "florase": "flora_intestinal",
    "florastor": "flora_intestinal",
    "rotabiotic": "flora_intestinal",
    "reuteri": "flora_intestinal",
    "enterogermina": "flora_intestinal",
    "bivos": "flora_intestinal",
    "vivomixx": "flora_intestinal",

    # =========================================================================
    # INFANTIL / FIEBRE Y DOLOR BEBÉS
    # =========================================================================
    "dalsy": "fiebre_infantil",
    "junifen": "fiebre_infantil",
    "apiretal": "fiebre_infantil",
    "febrectal": "fiebre_infantil",
    "ibuprofeno pediatrico": "fiebre_infantil",
    "paracetamol pediatrico": "fiebre_infantil",
    "dalsy pediatrico": "fiebre_infantil",

    # =========================================================================
    # VETERINARIA / MASCOTAS
    # =========================================================================
    "seresto": "veterinaria",
    "seresto perros": "veterinaria",
    "seresto gatos": "veterinaria",
    "frontline": "veterinaria",
    "frontline combo": "veterinaria",
    "frontline tri-act": "veterinaria",
    "advantix": "veterinaria",
    "advantage": "veterinaria",
    "bravecto": "veterinaria",
    "nexgard": "veterinaria",
    "nexgard spectra": "veterinaria",
    "scalibor": "veterinaria",
    "broadline": "veterinaria",
    "drontal": "veterinaria",
    "milbemax": "veterinaria",
    "collar antiparasitario": "veterinaria",
    "pipetas perro": "veterinaria",
    "pipetas gato": "veterinaria",

    # =========================================================================
    # VITAMINAS / SUPLEMENTOS ESPECÍFICOS
    # =========================================================================
    "medebiotin": "caida_cabello",
    "biotina": "caida_cabello",
    "magnogene": "magnesio",
    "magnesium": "magnesio",
    "magnesioboi": "magnesio",
    "ana maria lajusticia magnesio": "magnesio",
    "hidroxil": "vitaminas_general",
    "hidroxil b1-b6-b12": "vitaminas_general",
    "berocca": "vitaminas_general",
    "supradyn": "vitaminas_general",
    "centrum": "vitaminas_general",
    "dayamineral": "vitaminas_general",

    # =========================================================================
    # OMEPRAZOL / ANTIÁCIDOS OTC
    # =========================================================================
    "omereflux": "acidez_reflujo",
    "omeprazol cinfa": "acidez_reflujo",
    "omeprazol kern": "acidez_reflujo",
    "losec": "acidez_reflujo",
    "maalox": "acidez_reflujo",
    "rennie": "acidez_reflujo",
    "alka-seltzer": "acidez_reflujo",

    # =========================================================================
    # ANTITUSIVOS / EXPECTORANTES ESPECÍFICOS
    # =========================================================================
    "normotus": "tos_garganta",
    "cloperastina": "tos_garganta",
    "cinfatós": "tos_garganta",
    "cinfatos": "tos_garganta",
    "bisolvon": "tos_garganta",
    "bisolvon expectorante": "tos_garganta",
    "romilar": "tos_garganta",
    "inistón": "tos_garganta",
    "iniston": "tos_garganta",
    "actithiol": "tos_garganta",
    "fluidasa": "tos_garganta",
    "pectoral": "tos_garganta",

    # =========================================================================
    # ANALGÉSICOS ESPECÍFICOS
    # =========================================================================
    "actromadol": "dolor_analgesico",
    "difenadol": "dolor_analgesico",
    "difenadex": "dolor_analgesico",
    "dexketoprofeno": "dolor_analgesico",
    "endolex": "dolor_analgesico",
    "enantyum": "dolor_analgesico",
    "dolostop": "dolor_analgesico",
    "dolostop plus": "dolor_analgesico",
    "gelocatil plus": "dolor_analgesico",

    # =========================================================================
    # HERPES LABIAL ESPECÍFICO
    # =========================================================================
    "zoviduo": "herpes_labial",
    "lavirk": "herpes_labial",
    "compeed calenturas": "herpes_labial",

    # =========================================================================
    # GRIPE ESPECÍFICO
    # =========================================================================
    "frionex": "gripe_resfriado",
    "ilvigrip": "gripe_resfriado",

    # =========================================================================
    # PICADURAS / ANTIPRURIGINOSOS
    # =========================================================================
    "calmiox": "dermatitis",  # Hidrocortisona - corticoide suave para piel irritada general
    "dermapic": "picaduras",
    "after bite": "picaduras",
    "afterbite": "picaduras",
    "fenistil": "picaduras",
    "fenistil gel": "picaduras",
    "polaramine crema": "picaduras",
    "repelente mosquitos": "repelente_insectos",
    "antimosquitos": "repelente_insectos",
    "relec": "repelente_insectos",
    "goibi": "repelente_insectos",

    # =========================================================================
    # CICATRIZANTES / REGENERADORES
    # =========================================================================
    "blastoestimulina": "cicatrices_estrias",
    "bepanthol cicatrices": "cicatrices_estrias",
    "mederma": "cicatrices_estrias",
    "contractubex": "cicatrices_estrias",
    "kelo-cote": "cicatrices_estrias",
    "kelocote": "cicatrices_estrias",
    "strataderm": "cicatrices_estrias",
    "dermatix": "cicatrices_estrias",
    "cicalfate": "cicatrices_estrias",

    # =========================================================================
    # HIGIENE BUCAL / ENJUAGUES
    # =========================================================================
    "normobucal": "encias",
    "listerine": "higiene_dental",
    "oraldine": "higiene_dental",
    "fluocaril": "higiene_dental",
    "colutorio": "higiene_dental",
    "enjuague bucal": "higiene_dental",
    "perio-aid": "encias",
    "perioaid": "encias",
    "paroex": "encias",
    "gingilacer": "encias",
    "dentaid": "higiene_dental",
    "dentispray": "aftas",
    "oddent": "aftas",
    "aftasone": "aftas",

    # =========================================================================
    # ANTIHISTAMÍNICOS ORALES ESPECÍFICOS
    # =========================================================================
    "bactil": "alergia",
    "bactil forte": "alergia",
    "abrilia": "alergia",
    "telfast": "alergia",
    "fexofenadina": "alergia",
    "rino-ebastel": "alergia",
    "rinoebastel": "alergia",
    "rinobloc": "alergia",
    "flonase": "alergia",
    "fluticasona nasal": "alergia",
    "avamys": "alergia",
    "nasonex": "alergia",
    "optialerg": "alergia",
    "opatanol": "alergia",
    "zaditen": "alergia",

    # =========================================================================
    # LAXANTES VEGETALES / FIBRA
    # =========================================================================
    "fave de fuca": "estrenimiento",
    "favedifuca": "estrenimiento",
    "plantaben": "estrenimiento",
    "plantago": "estrenimiento",
    "metamucil": "estrenimiento",
    "movicol": "estrenimiento",
    "linomed": "estrenimiento",
    "emportal": "estrenimiento",
    "lactitol": "estrenimiento",
    "duphalac": "estrenimiento",
    "laevolac": "estrenimiento",
    "lactulosa": "estrenimiento",
    "laxante": "estrenimiento",
    "supositorios glicerina": "estrenimiento",
    "vilardell": "estrenimiento",

    # =========================================================================
    # ANTIDIARREICOS ESPECÍFICOS
    # =========================================================================
    "tiorfan": "diarrea",
    "racecadotrilo": "diarrea",
    "sindiar": "diarrea",
    "tanagel": "diarrea",
    "smecta": "diarrea",

    # =========================================================================
    # TOS / EXPECTORANTES ESPECÍFICOS
    # =========================================================================
    "bronchostop": "tos_garganta",
    "mucibron": "tos_garganta",
    "prospantus": "tos_garganta",
    "prospan": "tos_garganta",
    "terbasmin": "tos_garganta",
    "cinfatusina": "tos_garganta",
    "iniston tos": "tos_garganta",
    "iniston mucosidad": "tos_garganta",
    "pharmagrip tos": "tos_garganta",

    # =========================================================================
    # HEMORROIDES ESPECÍFICOS
    # =========================================================================
    "hemoliv": "hemorroides",
    "detramax": "hemorroides",
    "ruscus llorens": "hemorroides",
    "ruscus": "hemorroides",

    # =========================================================================
    # SALES REHIDRATACIÓN
    # =========================================================================
    "sueroral": "diarrea",
    "sueroral hiposodico": "diarrea",
    "bioralsuero": "diarrea",
    "citorsal": "diarrea",

    # =========================================================================
    # DOLOR MUSCULAR SPRAY
    # =========================================================================
    "cinfadol": "dolor_muscular",
    "cinfadol spray": "dolor_muscular",
    "ibustick": "dolor_muscular",
    "reflex spray": "dolor_muscular",

    # =========================================================================
    # HERPES ESPECÍFICO
    # =========================================================================
    "viruderm": "herpes_labial",
    "zovicrem": "herpes_labial",
    "zovicrem labial": "herpes_labial",
    "penciclovir": "herpes_labial",
    "aciclovir labial": "herpes_labial",

    # =========================================================================
    # VITAMINAS ESPECÍFICAS
    # =========================================================================
    "vitagobens": "vitaminas_general",
    "becozyme": "vitaminas_general",
    "auxina": "vitaminas_general",
    "dayamineral": "vitaminas_general",
    "multicentrum": "vitaminas_general",
    "pharmaton": "vitaminas_general",

    # =========================================================================
    # CAÍDA CABELLO ESPECÍFICO
    # =========================================================================
    "normopil": "caida_cabello",
    "minoxidil": "caida_cabello",
    "regaxidil": "caida_cabello",
    "lacovin": "caida_cabello",
    "regaine": "caida_cabello",

    # =========================================================================
    # GASES / ENZIMAS DIGESTIVAS
    # =========================================================================
    "pankreoflat": "gases_digestion",
    "noraflat": "gases_digestion",
    "entero silicona": "gases_digestion",
    "enterosilicona": "gases_digestion",
    "kreon": "gases_digestion",
    "creon": "gases_digestion",

    # =========================================================================
    # CIRCULACIÓN ESPECÍFICO
    # =========================================================================
    "menaven": "circulacion_piernas",
    "tanakene": "circulacion_piernas",
    "ginkgo": "circulacion_piernas",

    # =========================================================================
    # CONGESTIÓN NASAL ESPECÍFICO
    # =========================================================================
    "respidina": "congestion_nasal",
    "pseudoefedrina": "congestion_nasal",
    "pranzo": "congestion_nasal",
    "disneumon": "congestion_nasal",
    "rinocusi": "congestion_nasal",
    "nasalvicks": "congestion_nasal",
    "inhalvicks": "congestion_nasal",
    "normonasal": "congestion_nasal",
    "respibien antialergico": "congestion_nasal",

    # =========================================================================
    # ESTRÉS / ANSIEDAD
    # =========================================================================
    "lasea": "estres_ansiedad",
    "lavanda": "estres_ansiedad",
    "sedatif pc": "estres_ansiedad",
    "sed-tif": "estres_ansiedad",
    "passiflorine": "estres_ansiedad",
    "valeriana": "estres_ansiedad",
    "aquilea sueno": "sueno_insomnio",

    # =========================================================================
    # HIGIENE ÍNTIMA ESPECÍFICO
    # =========================================================================
    "rosalgin": "higiene_intima",
    "muvagyn": "higiene_intima",
    "ginecanesflor": "higiene_intima",
    "ginecanestest": "higiene_intima",
    "lactacyd": "higiene_intima",
    "chilly": "higiene_intima",
    "isdin intim": "higiene_intima",

    # =========================================================================
    # OJO SECO ESPECÍFICO
    # =========================================================================
    "lipolasic": "ojo_seco",
    "acuolens": "ojo_seco",
    "lagriben": "ojo_seco",
    "lagroben": "ojo_seco",
    "viscofresh": "ojo_seco",
    "vispring": "ojo_seco",
    "optava": "ojo_seco",
    "hyabak": "ojo_seco",
    "siccafluid": "ojo_seco",
    "yellox": "ojo_seco",

    # =========================================================================
    # DOLOR GARGANTA ESPECÍFICO
    # =========================================================================
    "deratin": "dolor_garganta",
    "strefen spray": "dolor_garganta",
    "pharysol": "dolor_garganta",

    # =========================================================================
    # HERIDAS / APÓSITOS ESPECÍFICO
    # =========================================================================
    "tisuderma": "heridas_apositos",
    "linitul": "heridas_apositos",
    "bexidermil": "hongos_pies",

    # =========================================================================
    # ANALGÉSICOS ESPECÍFICOS (PARACETAMOL)
    # =========================================================================
    "antidol": "dolor_analgesico",
    "dolovanz": "dolor_analgesico",
    "geloprofen": "dolor_analgesico",
    "efferaldol": "dolor_analgesico",
    "aspirina": "dolor_analgesico",
    "aspirina efervescente": "dolor_analgesico",

    # =========================================================================
    # FASE 9 - PRODUCTOS VERIFICADOS (200-300 por volumen)
    # =========================================================================

    # PÉRDIDA DE PESO / OBESIDAD (solo OTC)
    "alli": "perdida_peso",  # Orlistat 60mg - único OTC para pérdida peso
    "xls medical": "perdida_peso",
    "xls": "perdida_peso",

    # MENOPAUSIA (suplementos, no hormonales)
    "femarelle": "menopausia",
    "ymea": "menopausia",
    "estromineral": "menopausia",
    "serelys": "menopausia",

    # SUEÑO ADICIONAL
    "dormirel": "sueno_insomnio",  # Difenhidramina
    "dormirel forte": "sueno_insomnio",
    "sominex": "sueno_insomnio",
    "nytol": "sueno_insomnio",

    # ALERGIA ADICIONAL
    "alercina": "alergia",  # Cetirizina
    "virlix": "alergia",  # Cetirizina
    "alerdina": "alergia",
    "allergodil": "alergia",  # Azelastina

    # ACIDEZ ADICIONAL
    "pepcid": "acidez_reflujo",  # Famotidina
    "famotidina": "acidez_reflujo",
    "tagamet": "acidez_reflujo",  # Cimetidina
    "omekaste": "acidez_reflujo",  # Omeprazol

    # MAREO ADICIONAL
    "normostop cafeina": "mareo_viaje",  # Dimenhidrinato + cafeína
    "normostop": "mareo_viaje",  # Dimenhidrinato

    # HONGOS ADICIONAL
    "zeliderm": "hongos_pies",  # Clotrimazol 20%

    # PARÁSITOS INTESTINALES
    "trilombrin": "parasitos",  # Pirantel - lombrices (OTC confirmado)

    # DOLOR MUSCULAR ADICIONAL
    "algesal": "dolor_muscular",  # Dietilamina salicilato
    "gelodol": "dolor_muscular",  # Aceite romero

    # DOLOR GARGANTA ADICIONAL
    "propalgar": "dolor_garganta",  # Clorfenamina
    "lizifen": "dolor_garganta",  # Flurbiprofeno

    # ESTREÑIMIENTO ADICIONAL
    "glycilax": "estrenimiento",  # Glicerina
    "cuve": "estrenimiento",  # Glicerina supositorios

    # PICADURAS ADICIONAL
    "neosayomol": "picaduras",  # Difenhidramina crema

    # OJO SECO/COLIRIOS ADICIONAL
    "ojosbel": "ojo_seco",  # Colirio vasoconstrictor

    # GRIPE ADICIONAL
    "grippal": "gripe_resfriado",  # Similar a Frenadol
    "grippal fenilefrina": "gripe_resfriado",

    # ANTISÉPTICOS ADICIONAL
    "betadine gel": "heridas_apositos",
    "betadine solucion": "heridas_apositos",

    # PARCHES DOLOR
    "voltadol parche": "dolor_muscular",
    "voltadol aposito": "dolor_muscular",

    # =========================================================================
    # FASE 10 - PRODUCTOS OTC GAPS FINALES (cobertura 96%+)
    # =========================================================================

    # HONGOS
    "canespie": "hongos_pies",  # Bifonazol antifúngico
    "canesten": "hongos_pies",

    # DIGESTIVO / HERBAL
    "iberogast": "gases_digestion",  # Gotas digestivas herbales

    # ESTREÑIMIENTO
    "supositorios glicerina": "estrenimiento",
    "supositorios de glicerina": "estrenimiento",
    "glicerina rovi": "estrenimiento",
    "bekunis": "estrenimiento",  # Laxante estimulante

    # ANTIÁCIDOS
    "rennie": "acidez_reflujo",  # Antiácido masticable

    # DOLOR DE GARGANTA
    "garydol": "dolor_garganta",  # Comprimidos para chupar

    # CONGESTIÓN
    "anticongestiva cusi": "congestion_nasal",  # Pasta nasal

    # HERPES LABIAL
    "relepost": "herpes",  # Barra labial cold sore

    # CAÍDA CABELLO SUPLEMENTOS
    "regender": "caida_cabello",  # Suplemento capilar

    # HERIDAS / APÓSITOS
    "betatul": "heridas_apositos",  # Apósitos impregnados

    # HEMORROIDES
    "hemorrane": "hemorroides",  # Pomada rectal

    # DOLOR / ANALGÉSICOS
    "voltaduo": "dolor_analgesico",  # Paracetamol + ibuprofeno

    # ACNÉ
    "peroxiben": "acne",  # Peróxido de benzoilo

    # HIGIENE ÍNTIMA / PROBIÓTICOS
    "intibiotics": "higiene_intima",  # Probióticos vaginales
    "lactibiane": "higiene_intima",  # Probióticos vaginales

    # ANALGÉSICOS ADICIONALES
    "saridon": "dolor_analgesico",  # Paracetamol + cafeína
    "difenatil": "dolor_analgesico",  # Paracetamol

    # TOS / EXPECTORANTES
    "expectodhu": "tos_garganta",  # Jarabe tos

    # DOLOR GARGANTA
    "bucopro": "dolor_garganta",  # Pastillas antiinflamatorias

    # =========================================================================
    # FASE 11 - OTC GENUINOS RESTANTES (cima_requiere_receta=False)
    # =========================================================================

    # ANTICONCEPCIÓN EMERGENCIA (OTC en España desde 2009)
    "ulipristal": "anticoncepcion",  # ellaOne genérico
    "ellaone": "anticoncepcion",
    "postinor": "anticoncepcion",
    "norlevo": "anticoncepcion",

    # ANTIÁCIDOS / PROTECTORES GÁSTRICOS
    "astrilax": "acidez_reflujo",  # Omeprazol OTC
    "omeprazol": "acidez_reflujo",
    "secrepat": "acidez_reflujo",  # Antiácido

    # DESPIGMENTANTES
    "melanasa": "manchas",  # Hidroquinona tópica

    # CALCIO / HUESOS
    "mastical": "vitaminas_general",  # Calcio
    "carbocal": "vitaminas_general",  # Calcio + Vit D
    "caosina": "vitaminas_general",  # Calcio
    "calcium": "vitaminas_general",

    # ANTICASPA
    "bioselenium": "caspa",  # Sulfuro de selenio

    # APETITO / NUTRICIÓN
    "mosegor": "energia_cansancio",  # Estimulante apetito

    # HIERRO / ÁCIDO FÓLICO
    "foliferron": "vitaminas_general",  # Hierro + ácido fólico
    "ferbisol": "vitaminas_general",
    "tardyferon": "vitaminas_general",

    # LAXANTES
    "magnesia san pellegrino": "estrenimiento",
    "magnesia cinfa": "estrenimiento",
    "emuliquen": "estrenimiento",

    # VETERINARIOS
    "adtab": "veterinaria",  # Antiparasitario perros
    "capstar": "veterinaria",  # Antipulgas
    "frontline": "veterinaria",
    "advantix": "veterinaria",
    "nexgard": "veterinaria",
    "seresto": "veterinaria",
    "scalibor": "veterinaria",

    # DOLOR GARGANTA ADICIONAL
    "juanolgar": "dolor_garganta",  # Pastillas chupar
    "strepsils": "dolor_garganta",
    "lizipaina": "dolor_garganta",

    # ANALGÉSICOS ADICIONALES
    "dolalgial": "dolor_analgesico",  # Clonixino
    "enandol": "dolor_analgesico",  # Dexketoprofeno
    "enantyum": "dolor_analgesico",
    "nolotil": "dolor_analgesico",

    # DOLOR MUSCULAR ADICIONAL
    "tantum": "dolor_muscular",  # Bencidamina tópica

    # ANTIDIARREICOS
    "diarfin": "diarrea",  # Loperamida
    "fortasec": "diarrea",
    "salvacolina": "diarrea",

    # ANTIHISTAMÍNICOS
    "ebacina": "alergia",  # Ebastina
    "ebastel": "alergia",
    "bilaxten": "alergia",

    # TOS ADICIONAL
    "propalcof": "tos_garganta",  # Dextrometorfano
    "romilar": "tos_garganta",
    "bisolvon": "tos_garganta",

    # SUEÑO / RELAJACIÓN
    "amapola de california": "sueno_insomnio",
    "amapola california": "sueno_insomnio",
    "valeriana": "sueno_insomnio",
    "pasiflora": "sueno_insomnio",

    # AFTAS / BUCAL
    "oralsone": "aftas",  # Triamcinolona bucal
    "aloclair": "aftas",
    "aftasone": "aftas",

    # =========================================================================
    # FASE 12 - OTC FINALES (<1000 EUR restantes)
    # =========================================================================

    # DOLOR MUSCULAR TÓPICO
    "ibucalm": "dolor_muscular",  # Ibuprofeno gel
    "zenavan": "dolor_muscular",  # Diclofenaco gel
    "novidol": "dolor_muscular",  # Ibuprofeno spray

    # TOS / MUCOLÍTICOS
    "tusclin": "tos_garganta",  # Antitusivo
    "normofludil": "tos_garganta",  # Mucolítico
    "angituss": "tos_garganta",  # Jarabe tos

    # VITAMINAS
    "benerva": "vitaminas_general",  # Vitamina B1
    "benadon": "vitaminas_general",  # Vitamina B6
    "potenciator": "energia_cansancio",

    # GRIPE
    "bronchogrip": "gripe_resfriado",
    "pharmafren": "gripe_resfriado",  # Paracetamol + pseudoefedrina

    # ALERGIA TÓPICA
    "fenergan": "alergia",  # Prometazina crema

    # LAXANTES
    "lubrilax": "estrenimiento",  # Bisacodilo
    "aceite ricino": "estrenimiento",
    "recolax": "estrenimiento",

    # VETERINARIOS
    "vectra": "veterinaria",
    "effinol": "veterinaria",

    # HONGOS UÑAS
    "locetar": "hongos_pies",  # Amorolfina

    # HEMORROIDES
    "cohortan": "hemorroides",

    # HERIDAS / ANTISÉPTICOS
    "normovidona": "heridas_apositos",  # Povidona yodada

    # VERRUGAS
    "verufil": "verrugas_callos",

    # SUEÑO
    "normodorm": "sueno_insomnio",  # Doxilamina

    # CAÍDA CABELLO
    "alocutan": "caida_cabello",  # Minoxidil

    # =========================================================================
    # HOMEOPATÍA
    # =========================================================================
    "lycopodium": "homeopatia",
    "symphytum": "homeopatia",
    "nervoheel": "homeopatia",
    "engystol": "homeopatia",
    "viburcol": "homeopatia",
    "hypericum": "homeopatia",
    "phosphorus iberica": "homeopatia",
    "arsenicum": "homeopatia",
    "heel": "homeopatia",  # Laboratorio Heel
    "boiron": "homeopatia",  # Laboratorio Boiron
    "iberica c": "homeopatia",  # Diluciones C
    "iberica k": "homeopatia",  # Diluciones K
    "globulos": "homeopatia",

    # =========================================================================
    # PRODUCTOS ESPECÍFICOS - Issue #446 Fase 1
    # =========================================================================
    # Primeros auxilios / Antisépticos
    "alcomon": "primeros_auxilios",  # Alcohol antiséptico
    "epistaxol": "primeros_auxilios",  # Hemorragias nasales

    # Aminoácidos / Cabello y uñas
    "cistina": "caida_cabello",  # Aminoácido para cabello/uñas

    # Picaduras / Alergias cutáneas
    "polaracrem": "picaduras",  # Picaduras insectos / Alergias cutáneas

    # Antihistamínicos sistémicos
    "periactin": "alergia",  # Antihistamínico

    # Antiparasitarios intestinales
    "zeninas": "parasitos",  # Lombrices intestinales

    # Cicatrizantes / Reparadores cutáneos
    "amniolina": "cicatrices_estrias",  # Pomada cicatrizante

    # Antiespasmódicos / Dolor abdominal
    "buscopresc": "dolor_analgesico",  # Antiespasmódico (cólicos/espasmos)

    # =========================================================================
    # ISSUE #476: Reglas añadidas para reducir unknown rate
    # =========================================================================

    # ÓPTICA (singular y plural)
    "gafa": "optica",  # Gafas lectura (singular)
    "gafas": "optica",  # Gafas lectura (plural)

    # COSMÉTICA FACIAL
    "be+": "cuidado_facial_general",  # Marca Be+ cosmética
    "be +": "cuidado_facial_general",  # Variante con espacio
    "soivre": "cuidado_facial_general",  # Marca cosmética

    # HIGIENE NASAL
    "nasalmer": "higiene_nasal",  # Spray nasal higiene

    # ORTOPEDIA (prendas de soporte)
    "dedil": "ortopedia",  # Protector dedos
    "codera": "ortopedia",  # Soporte codo
    "suspensorio": "ortopedia",  # Soporte atlético
    "faja": "ortopedia",  # Faja lumbar/abdominal
    "panty": "circulacion_piernas",  # Medias compresión

    # VITAMINAS
    "bion3": "vitaminas_general",  # Marca vitaminas
    "bion 3": "vitaminas_general",  # Variante con espacio

    # CHICLES/CARAMELOS
    "trident": "caramelos_garganta",  # Chicles
    "alprausch": "caramelos_garganta",  # Chicles naturales

    # BIENESTAR/SPA
    "kneipp": "higiene_corporal",  # Productos baño/bienestar

    # MATERIAS PRIMAS FARMACIA
    "acido borico": "material_sanitario",  # Materia prima
    "acido citrico": "material_sanitario",  # Materia prima
    "bicarbonato": "acidez_reflujo",  # Antiácido casero
    "glicerina": "material_sanitario",  # Materia prima
    "acetona": "accesorios_belleza",  # Quitaesmalte

    # ACCESORIOS
    "pastillero": "dispositivos_medicos",  # Organizador medicación
    "bolsa acofar": "material_sanitario",  # Bolsas farmacia
}


import re

# Cache para patrones regex compilados (performance)
_MARCA_PATTERNS = {}
_KEYWORD_PATTERNS = {}


def _get_word_pattern(keyword: str) -> re.Pattern:
    """
    Crea un patrón regex que matchea el keyword como palabra completa.

    Ejemplos:
    - "prim" matchea "PRIM FAJA" pero NO "comprimidos"
    - "be+ capilar" matchea "BE+ CAPILAR USO"
    """
    # Escapar caracteres especiales de regex (como +)
    escaped = re.escape(keyword)
    # Word boundary: inicio/fin de string o no-alfanumérico
    # Usamos (?:^|[^a-záéíóúñ]) en lugar de \b para mejor soporte español
    pattern = rf'(?:^|[^a-záéíóúñ0-9]){escaped}(?:[^a-záéíóúñ0-9]|$)'
    return re.compile(pattern, re.IGNORECASE)


def _matches_word(keyword: str, text: str, cache: dict) -> bool:
    """Verifica si keyword aparece como palabra completa en text."""
    if keyword not in cache:
        cache[keyword] = _get_word_pattern(keyword)
    return bool(cache[keyword].search(text))


def extract_necesidad(product_name: str) -> str:
    """
    Extrae la NECESIDAD de un producto basándose en keywords y marcas.

    Args:
        product_name: Nombre del producto

    Returns:
        Código de necesidad (ej: "caida_cabello", "proteccion_solar")
    """
    if not product_name:
        return "otros"

    name_lower = product_name.lower()

    # 1. Primero buscar por MARCA (más preciso)
    # Ordenar por longitud DESC para priorizar matches más específicos
    sorted_marcas = sorted(MARCA_NECESIDAD.keys(), key=len, reverse=True)
    for marca in sorted_marcas:
        # Usar word boundary matching para evitar falsos positivos
        # Ejemplo: "prim" NO debe matchear en "comprimidos"
        if _matches_word(marca, name_lower, _MARCA_PATTERNS):
            return MARCA_NECESIDAD[marca]

    # 2. Buscar por KEYWORDS (también con word boundary)
    for necesidad, keywords in NECESIDAD_KEYWORDS.items():
        sorted_keywords = sorted(keywords, key=len, reverse=True)
        for kw in sorted_keywords:
            if _matches_word(kw, name_lower, _KEYWORD_PATTERNS):
                return necesidad

    return "otros"


def get_necesidad_label(necesidad_code: str) -> str:
    """Obtiene el label legible de una necesidad."""
    return NECESIDADES.get(necesidad_code, "Otros / Sin clasificar")


# =============================================================================
# API PRINCIPAL - CLASIFICACIÓN CON CONFIANZA
# =============================================================================

# Marcas de alta confianza (0.90)
HIGH_CONFIDENCE_BRANDS = {
    'isdin', 'heliocare', 'avene', 'vichy', 'cerave', 'eucerin',
    'bioderma', 'cantabria', 'sesderma', 'olistic', 'gh ', 'ivb ',
    'neutrogena', 'lierac', 'filorga', 'rilastil', 'blemil',
    'nestle', 'hero baby', 'meritene', 'ensure', 'la roche',
    'endocare', 'martiderm', 'sensilis', 'bella aurora',
}

# Keywords de confianza media-alta (0.75)
MEDIUM_CONFIDENCE_KEYWORDS = {
    'solar', 'spf', 'serum', 'crema', 'gel', 'capsulas',
    'comprimidos', 'locion', 'champu', 'pasta', 'jarabe',
    'gotas', 'spray', 'polvo', 'sobres',
}


def _get_confidence(product_name: str, category: str) -> float:
    """
    Calcula confianza del match basado en el tipo de regla.

    Returns:
        Confianza entre 0.0 y 1.0
    """
    if category == "otros":
        return 0.0

    name_lower = product_name.lower()

    # Marcas conocidas = alta confianza
    for brand in HIGH_CONFIDENCE_BRANDS:
        if brand in name_lower:
            return 0.90

    # Keywords específicos = confianza media-alta
    for kw in MEDIUM_CONFIDENCE_KEYWORDS:
        if kw in name_lower:
            return 0.75

    # Default = confianza media
    return 0.60


def classify_product(
    product_name: str,
    default_category: str = "unknown",
    default_confidence: float = 0.0,
) -> Tuple[str, float]:
    """
    API principal: Clasifica un producto VentaLibre por NECESIDAD.

    Esta es la función recomendada para usar en servicios y callbacks.
    Retorna tanto la categoría como la confianza del match.

    Args:
        product_name: Nombre del producto a clasificar
        default_category: Categoría por defecto si no hay match
        default_confidence: Confianza por defecto si no hay match

    Returns:
        Tuple (ml_category, ml_confidence)
        - ml_category: Código NECESIDAD o default_category
        - ml_confidence: 0.60-0.90 para matches, default_confidence si no

    Example:
        >>> classify_product("HELIOCARE 360 GEL OIL-FREE")
        ("proteccion_solar", 0.90)

        >>> classify_product("PRODUCTO DESCONOCIDO XYZ")
        ("unknown", 0.0)
    """
    if not product_name:
        return default_category, default_confidence

    category = extract_necesidad(product_name)

    if category == "otros":
        return default_category, default_confidence

    confidence = _get_confidence(product_name, category)
    return category, confidence


# Alias para compatibilidad con venta_libre_classifier.py
classify_product_by_keywords = classify_product


# =============================================================================
# TEST
# =============================================================================

if __name__ == "__main__":
    test_products = [
        # Original tests
        "OLISTIC FOR WOMEN 28 FRASCOS 25 ML",
        "HELIOCARE 360º PIGMENT SOLUTION FLUID PROTECTOR SOLAR",
        "BLEMIL OPTIMUM EVOLUTION 1 LATA 800 G",
        "MITOSYL POMADA PROTECTORA 2 TUBOS 65 G",
        "THEALOZ DUO 1 ENVASE 10 ML",
        "COREGA CREMA EXTRA FUERTE ADHESIVO PROTESIS DENTAL",
        "EFFACLAR DUO+ 1 TUBO 40 ML",
        "FISIOCREM GEL ACTIVE 1 TUBO 60 ML",
        "ZZZQUIL NATURA 30 GUMMIES",
        "NATALBEN SUPRA 30 CAPSULAS",
        # Products that were "otros" before
        "GH 12 AZELAIC-N SERUM BI-GEL 30 ML",
        "MAGNESIUM 84 CAPSULAS",
        "OMRON M3 COMFORT TENSIOMETRO",
        "ORAL-B VITALITY 100 CEPILLO ELECTRICO",
        "ARTURO ALBA CREMA FACIAL HIDRATANTE",
        "IVB VITAMINAS COMPLEX 60 CAPSULAS",
        "SENSILIS SERUM RETINOL 30 ML",
        "FARMALASTIC TOBILLERA TALLA M",
        "BIOTRUE SOLUCION LENTES 300 ML",
    ]

    print("=" * 70)
    print("TEST EXTRACCION NECESIDAD")
    print("=" * 70)

    for product in test_products:
        necesidad = extract_necesidad(product)
        label = get_necesidad_label(necesidad)
        print(f"\n{product[:50]}...")
        print(f"  -> {necesidad}: {label}")
