# backend/app/services/narrative_service.py
"""
NarrativeService - Issue #509: Executive Summary con NLG.

Genera resúmenes ejecutivos usando Groq LLM, consumiendo:
- KPIs del dashboard (ventas, margen, tendencia)
- Resultados del Insight Engine (10 reglas)

Principio: El LLM no narra datos, CONECTA PUNTOS y ofrece diagnóstico + acción.

Estructura Three-Layer:
1. EL QUÉ (Hechos): Narración de KPIs principales
2. EL PORQUÉ (Diagnóstico): Conexión de insights, causas raíz
3. EL CÓMO (Acción): Recomendación priorizada con impacto económico
"""

import hashlib
import json
import logging
import os
import re
from datetime import datetime, timedelta, timezone
from typing import Any, Dict, List, Optional, Set, Tuple

from pydantic import ValidationError
from tenacity import (
    retry,
    retry_if_exception_type,
    stop_after_attempt,
    wait_exponential,
)

from ..schemas.narrative import (
    ActionTargetType,
    AlertSeverity,
    NarrativeKPIs,
    NarrativeResponse,
    NarrativeSection,
    ValidCategories,
)
from ..services.insight_engine.models import AuditResult

logger = logging.getLogger(__name__)


# === SYSTEM PROMPT ===

SYSTEM_PROMPT = """Eres un consultor farmacéutico experto que analiza datos de venta libre (OTC/parafarmacia).
Tu objetivo es generar un resumen ejecutivo ESTRUCTURADO y ACCIONABLE para el farmacéutico.

REGLAS CRÍTICAS:
1. SOLO menciona números que aparecen EXACTAMENTE en los datos proporcionados
2. NO inventes estadísticas, porcentajes ni cantidades
3. Usa formato estructurado: QUÉ (hechos) → PORQUÉ (diagnóstico) → CÓMO (acción)
4. Prioriza insights por impacto económico (mayor impacto = mayor prioridad)
5. La acción debe ser ESPECÍFICA y MEDIBLE
6. Responde SIEMPRE en español
7. Usa **negrita** para cifras clave en el análisis

IMPORTANTE sobre categorías:
- Si mencionas una categoría, DEBE ser una que existe en el sistema
- Categorías válidas: dermocosmetica, suplementos, higiene_bucal, higiene_personal, bebe_infantil, etc.
- NO inventes categorías como "Cosmética de Lujo" o similares

IMPORTANTE sobre el tono:
- Si no hay problemas críticos, felicita al farmacéutico por la buena gestión
- Enfócate entonces en oportunidades de crecimiento (upselling, margen)
- Sé constructivo y orientado a la acción
"""

USER_PROMPT_TEMPLATE = """## DATOS DEL PERÍODO ({date_from} a {date_to})

### KPIs Principales
- Ventas totales: €{total_sales:,.2f}
- Margen promedio: {avg_margin:.1f}%
- Productos únicos vendidos: {unique_products:,}
- Tendencia vs período anterior: {trend_pct:+.1f}%

### Insights del Motor de Análisis (ordenados por impacto económico)
{insights_text}

### Oportunidad Total Detectada
€{total_opportunity:,.2f}

## GENERA UN RESUMEN EJECUTIVO EN JSON:
{{
  "headline": "Frase impactante de 10-15 palabras resumiendo el estado del negocio",
  "analysis": "2-3 frases conectando KPIs e insights. Usa **negrita** para cifras clave. Explica el PORQUÉ.",
  "alert": "Si hay insight de severidad HIGH, mensaje de alerta urgente. Si no hay problemas críticos, dejar null.",
  "alert_severity": "critical|warning|info|null (según severidad del problema más grave)",
  "action": "Recomendación específica basada en el insight de mayor impacto económico",
  "action_value": número_del_impacto_economico_de_la_accion,
  "action_button_text": "Texto corto para botón (ej: 'Ver stock crítico', 'Analizar márgenes')",
  "action_target_type": "category|product|l2|null (tipo de navegación)",
  "action_target_id": "id_del_destino_si_aplica (ej: 'higiene_bucal', 'dermocosmetica')"
}}
"""

SILENCE_ADDENDUM = """
NOTA ESPECIAL: No hay alertas críticas ni problemas urgentes detectados.
En este caso, tu resumen debe:
1. Felicitar al farmacéutico por la buena gestión del stock
2. Enfocarte en OPORTUNIDADES de crecimiento (productos de alto margen, tendencias positivas)
3. Sugerir acciones de UPSELLING basadas en los datos
4. El tono debe ser POSITIVO y orientado a oportunidades, no a problemas
"""


class NarrativeService:
    """
    Servicio para generar resúmenes ejecutivos con NLG.

    Consume KPIs + Insight Engine → Narrativa estructurada en 3 capas.
    """

    # Modelo Groq (priorizar inteligencia sobre velocidad - 1 call/día/farmacia)
    GROQ_MODEL = "llama-3.3-70b-versatile"

    def __init__(self, groq_api_key: Optional[str] = None):
        """
        Inicializa el servicio.

        Args:
            groq_api_key: API key de Groq (o GROQ_API_KEY env var)
        """
        self.groq_api_key = groq_api_key or os.getenv("GROQ_API_KEY")
        self._groq_client = None

        # Cache en memoria (5 min TTL para desarrollo, 24h en producción)
        # NOTE: Para multi-worker (Render), considerar Redis
        self._cache: Dict[str, Dict] = {}
        self._cache_ttl_hours = 24

        if self.groq_api_key:
            logger.info("NarrativeService initialized with Groq API")
        else:
            logger.warning("GROQ_API_KEY not set - NarrativeService will return fallback")

    @property
    def groq_client(self):
        """Lazy initialization del cliente Groq."""
        if self._groq_client is None and self.groq_api_key:
            try:
                from groq import Groq

                self._groq_client = Groq(api_key=self.groq_api_key)
                logger.info(f"Groq client initialized: model={self.GROQ_MODEL}")
            except ImportError:
                logger.error("groq package not installed. Run: pip install groq")
        return self._groq_client

    def generate_summary(
        self,
        kpis: NarrativeKPIs,
        audit_result: AuditResult,
        language: str = "es",
    ) -> NarrativeResponse:
        """
        Genera resumen ejecutivo combinando KPIs + Insight Engine.

        Args:
            kpis: KPIs del dashboard de venta libre
            audit_result: Resultado del Insight Engine
            language: Idioma de la narrativa (es, en, ca)

        Returns:
            NarrativeResponse con narrativa estructurada
        """
        # Check cache
        cache_key = self._build_cache_key(kpis, audit_result)
        cached = self._get_cached(cache_key)
        if cached:
            return cached

        # Check for "Escenario de Silencio" (no critical alerts)
        has_critical_alerts = any(
            i.severity in ("high", "medium") for i in audit_result.insights
        )

        # Build prompt
        prompt = self._build_prompt(kpis, audit_result, has_critical_alerts)

        # Call LLM
        try:
            raw_response = self._call_llm(prompt)
            narrative_dict = self._extract_json(raw_response)
            narrative_section = self._parse_narrative(narrative_dict)

            # Validate grounding (anti-hallucination)
            grounding_valid = self._validate_grounding(
                narrative_section, kpis, audit_result
            )

            if not grounding_valid:
                logger.warning(
                    "narrative_service.grounding_failed",
                    extra={"cache_key": cache_key[:16]},
                )

        except Exception as e:
            logger.error(
                "narrative_service.generation_error",
                extra={"error": str(e)},
                exc_info=True,
            )
            # Fallback to static narrative
            narrative_section, grounding_valid = self._generate_fallback(
                kpis, audit_result
            )

        # Build response
        now = datetime.now(timezone.utc)
        response = NarrativeResponse(
            summary=narrative_section,
            generated_at=now.isoformat(),
            cache_until=(now + timedelta(hours=self._cache_ttl_hours)).isoformat(),
            grounding_valid=grounding_valid,
            language=language,
            insights_used=len(audit_result.insights),
            total_opportunity=audit_result.total_opportunity,
        )

        # Cache result
        self._set_cached(cache_key, response)

        return response

    def _build_prompt(
        self,
        kpis: NarrativeKPIs,
        audit_result: AuditResult,
        has_critical_alerts: bool,
    ) -> str:
        """Build the user prompt with grounded data."""
        # Format insights for prompt
        insights_text = self._format_insights(audit_result.insights)

        prompt = USER_PROMPT_TEMPLATE.format(
            date_from=kpis.date_from,
            date_to=kpis.date_to,
            total_sales=kpis.total_sales,
            avg_margin=kpis.avg_margin,
            unique_products=kpis.unique_products,
            trend_pct=kpis.trend_pct,
            insights_text=insights_text,
            total_opportunity=audit_result.total_opportunity,
        )

        # Add silence addendum if no critical alerts
        if not has_critical_alerts:
            prompt += "\n\n" + SILENCE_ADDENDUM

        return prompt

    def _format_insights(self, insights: List[Any]) -> str:
        """Format insights for the prompt."""
        if not insights:
            return "No se detectaron anomalías ni oportunidades críticas."

        lines = []
        for i, insight in enumerate(insights[:5], 1):  # Top 5 by economic value
            severity_emoji = {"high": "🔴", "medium": "🟠", "low": "🟢"}.get(
                insight.severity, "⚪"
            )
            lines.append(
                f"{i}. [{severity_emoji} {insight.severity.upper()}] {insight.title}\n"
                f"   - {insight.description}\n"
                f"   - Impacto económico: €{insight.economic_value:,.2f}"
            )

        return "\n".join(lines)

    @retry(
        stop=stop_after_attempt(3),
        wait=wait_exponential(multiplier=1, min=2, max=10),
        retry=retry_if_exception_type((ValueError, json.JSONDecodeError)),
        reraise=True,
    )
    def _call_llm(self, prompt: str) -> str:
        """Call Groq LLM with retry logic."""
        if not self.groq_client:
            raise ValueError("Groq client not available")

        response = self.groq_client.chat.completions.create(
            model=self.GROQ_MODEL,
            messages=[
                {"role": "system", "content": SYSTEM_PROMPT},
                {"role": "user", "content": prompt},
            ],
            temperature=0.3,  # Low for consistency
            max_tokens=600,
            response_format={"type": "json_object"},
        )
        return response.choices[0].message.content

    def _extract_json(self, text: str) -> Dict[str, Any]:
        """Extract JSON from LLM response with fallbacks."""
        # Attempt 1: Direct JSON
        try:
            return json.loads(text)
        except json.JSONDecodeError:
            pass

        # Attempt 2: Find ```json block
        match = re.search(r"```json\s*(.*?)\s*```", text, re.DOTALL)
        if match:
            try:
                return json.loads(match.group(1))
            except json.JSONDecodeError:
                pass

        # Attempt 3: Find {...}
        match = re.search(r"\{.*\}", text, re.DOTALL)
        if match:
            try:
                return json.loads(match.group(0))
            except json.JSONDecodeError:
                pass

        raise json.JSONDecodeError("No valid JSON found", text, 0)

    def _parse_narrative(self, data: Dict[str, Any]) -> NarrativeSection:
        """Parse LLM output into NarrativeSection."""
        # Map alert_severity string to enum
        alert_severity_str = data.get("alert_severity")
        alert_severity = None
        if alert_severity_str and alert_severity_str != "null":
            try:
                alert_severity = AlertSeverity(alert_severity_str.lower())
            except ValueError:
                alert_severity = AlertSeverity.WARNING

        # Map action_target_type string to enum
        target_type_str = data.get("action_target_type")
        target_type = None
        if target_type_str and target_type_str != "null":
            try:
                target_type = ActionTargetType(target_type_str.lower())
            except ValueError:
                pass

        return NarrativeSection(
            headline=data.get("headline", "Resumen no disponible"),
            analysis=data.get("analysis", ""),
            alert=data.get("alert") if data.get("alert") != "null" else None,
            alert_severity=alert_severity,
            action=data.get("action", "Revisar el panel de insights para más detalles"),
            action_value=data.get("action_value"),
            action_button_text=data.get("action_button_text", "Ver detalles"),
            action_target_type=target_type,
            action_target_id=data.get("action_target_id"),
        )

    def _validate_grounding(
        self,
        narrative: NarrativeSection,
        kpis: NarrativeKPIs,
        audit_result: AuditResult,
    ) -> bool:
        """
        Validate that numbers and categories in narrative match input data.

        Prevents LLM hallucinations by checking:
        1. Numeric values mentioned match KPIs or insight values
        2. Category names mentioned exist in the system

        Returns:
            True if grounding is valid, False if hallucination detected
        """
        # Build set of valid numbers from input
        valid_numbers = self._extract_valid_numbers(kpis, audit_result)

        # Build set of valid categories
        valid_categories = ValidCategories

        # Check analysis text for numbers
        analysis_text = narrative.analysis + " " + narrative.headline
        mentioned_numbers = self._extract_numbers_from_text(analysis_text)

        # Validate numbers (with tolerance for rounding)
        for num in mentioned_numbers:
            if not self._number_is_valid(num, valid_numbers, tolerance=0.05):
                logger.warning(
                    "narrative_service.grounding_number_mismatch",
                    extra={"number": num, "valid_numbers": list(valid_numbers)[:10]},
                )
                return False

        # Check for category mentions (semantic grounding)
        mentioned_categories = self._extract_categories_from_text(analysis_text)
        for cat in mentioned_categories:
            if cat not in valid_categories:
                logger.warning(
                    "narrative_service.grounding_category_mismatch",
                    extra={"category": cat},
                )
                return False

        return True

    def _extract_valid_numbers(
        self, kpis: NarrativeKPIs, audit_result: AuditResult
    ) -> Set[float]:
        """Extract all valid numbers from input data."""
        numbers = set()

        # KPIs
        numbers.add(round(kpis.total_sales, 2))
        numbers.add(round(kpis.avg_margin, 1))
        numbers.add(float(kpis.unique_products))
        numbers.add(round(kpis.trend_pct, 1))

        # Insight economic values
        numbers.add(round(audit_result.total_opportunity, 2))
        for insight in audit_result.insights:
            numbers.add(round(insight.economic_value, 2))

        # Add common rounded versions
        for n in list(numbers):
            numbers.add(round(n, 0))
            numbers.add(round(n, 1))

        return numbers

    def _extract_numbers_from_text(self, text: str) -> List[float]:
        """Extract numeric values from text."""
        # Pattern for numbers with optional currency/percent
        pattern = r"[€$]?\s*(\d{1,3}(?:[.,]\d{3})*(?:[.,]\d+)?)\s*[€%]?"
        matches = re.findall(pattern, text)

        numbers = []
        for match in matches:
            # Normalize number format (Spanish uses . for thousands, , for decimals)
            normalized = match.replace(".", "").replace(",", ".")
            try:
                numbers.append(float(normalized))
            except ValueError:
                continue

        return numbers

    def _extract_categories_from_text(self, text: str) -> Set[str]:
        """Extract category-like words from text."""
        # Look for words that could be category names
        # Pattern: lowercase words with underscores or multiple words in quotes
        text_lower = text.lower()

        categories = set()

        # Known category patterns
        category_patterns = [
            r"\b(dermocosmetica|suplementos|higiene_bucal|higiene_personal)\b",
            r"\b(bebe_infantil|nutricion_deportiva|salud_sexual)\b",
            r"\b(dolor_fiebre|respiratorio|digestivo|piel|bucal)\b",
            r"\b(bienestar|circulatorio|nutricion|movilidad)\b",
        ]

        for pattern in category_patterns:
            matches = re.findall(pattern, text_lower)
            categories.update(matches)

        return categories

    def _number_is_valid(
        self, number: float, valid_numbers: Set[float], tolerance: float = 0.05
    ) -> bool:
        """Check if a number is close to any valid number."""
        for valid in valid_numbers:
            if abs(number - valid) <= abs(valid * tolerance):
                return True
            # Also check if it's an exact match (for percentages, counts)
            if abs(number - valid) < 0.5:
                return True
        return False

    def _generate_fallback(
        self, kpis: NarrativeKPIs, audit_result: AuditResult
    ) -> Tuple[NarrativeSection, bool]:
        """Generate fallback narrative when LLM fails."""
        # Determine overall state
        if audit_result.insights:
            top_insight = audit_result.insights[0]
            headline = f"Detectadas {len(audit_result.insights)} oportunidades de mejora"
            analysis = (
                f"Se identificaron **€{audit_result.total_opportunity:,.2f}** en "
                f"oportunidades. La principal: {top_insight.title}."
            )
            action = top_insight.description
            action_value = top_insight.economic_value
            alert = top_insight.title if top_insight.severity == "high" else None
            alert_severity = (
                AlertSeverity.CRITICAL if top_insight.severity == "high" else None
            )
        else:
            headline = "Excelente gestión de venta libre"
            analysis = (
                f"Con **€{kpis.total_sales:,.2f}** en ventas y un margen del "
                f"**{kpis.avg_margin:.1f}%**, el rendimiento es óptimo."
            )
            action = "Mantén el buen trabajo y explora oportunidades de upselling."
            action_value = None
            alert = None
            alert_severity = None

        return (
            NarrativeSection(
                headline=headline,
                analysis=analysis,
                alert=alert,
                alert_severity=alert_severity,
                action=action,
                action_value=action_value,
                action_button_text="Ver detalles",
            ),
            True,  # Fallback is always "grounded" (we control the text)
        )

    def _build_cache_key(
        self, kpis: NarrativeKPIs, audit_result: AuditResult
    ) -> str:
        """Build cache key from input data."""
        data = {
            "kpis": {
                "total_sales": kpis.total_sales,
                "avg_margin": kpis.avg_margin,
                "date_from": kpis.date_from,
                "date_to": kpis.date_to,
            },
            "insights_count": len(audit_result.insights),
            "total_opportunity": audit_result.total_opportunity,
        }
        return hashlib.sha256(json.dumps(data, sort_keys=True).encode()).hexdigest()[
            :32
        ]

    def _get_cached(self, cache_key: str) -> Optional[NarrativeResponse]:
        """Get cached response if still valid."""
        if cache_key not in self._cache:
            return None

        cached = self._cache[cache_key]
        cache_time = datetime.fromisoformat(cached["timestamp"])
        age_hours = (datetime.now(timezone.utc) - cache_time).total_seconds() / 3600

        if age_hours < self._cache_ttl_hours:
            logger.debug(
                "narrative_service.cache_hit",
                extra={"cache_key": cache_key[:16], "age_hours": age_hours},
            )
            return cached["response"]

        # Expired
        del self._cache[cache_key]
        return None

    def _set_cached(self, cache_key: str, response: NarrativeResponse) -> None:
        """Cache response."""
        self._cache[cache_key] = {
            "response": response,
            "timestamp": datetime.now(timezone.utc).isoformat(),
        }


# Singleton instance
narrative_service = NarrativeService()
