"""
Servicio para gestión de productos en manual_review (requieren revisión manual).

Issue #447: Exportar productos sin enriquecer a CSV para análisis offline.

Clasificación de códigos mejorada:
- Validación de checksum para CN-7 y EAN-13
- Cruce con catálogos (ProductCatalog, NomenclatorLocal)
- Tipos: CN_CATALOGED, CN_OTC, EAN_CATALOGED, EAN_OTC, CN_UNVERIFIED, INTERNAL
"""

import csv
from datetime import date
from decimal import Decimal
from io import StringIO
from typing import Dict, List, Literal, Optional, Tuple
from uuid import UUID

import structlog
from sqlalchemy import case, distinct, func, and_, or_
from sqlalchemy.orm import Session

from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.models.product_catalog import ProductCatalog
from app.models.nomenclator_local import NomenclatorLocal
from app.models.pharmacy import Pharmacy
from app.schemas.manual_review import (
    ManualReviewProduct,
    ManualReviewStats,
)

logger = structlog.get_logger(__name__)

# Tipos de código posibles
CodeType = Literal[
    "CN_CATALOGED",   # CN válido, existe en CIMA/Nomenclator
    "CN_OTC",         # CN válido checksum, NO en catálogos (parafarmacia)
    "EAN_CATALOGED",  # EAN válido, existe en catálogos
    "EAN_OTC",        # EAN válido checksum, NO en catálogos
    "CN_UNVERIFIED",  # 6 dígitos numérico (puede ser CN legacy O código interno)
    "INTERNAL",       # Código interno farmacia (confirmado no-universal)
]


class ManualReviewService:
    """
    Servicio para obtener y exportar productos en manual_review.

    Un producto está en "manual_review" si tiene un registro en SalesEnrichment
    con enrichment_status='manual_review' (no encontrado en catálogos oficiales).

    IMPORTANTE: El sistema de enriquecimiento SIEMPRE crea un registro en
    SalesEnrichment para cada venta. El campo enrichment_status indica el
    resultado del proceso de enriquecimiento:
    - 'enriched': Producto encontrado y enriquecido
    - 'manual_review': No encontrado en catálogos, requiere revisión
    - 'failed': Error durante enriquecimiento
    - 'pending': En proceso
    """

    def __init__(self, db: Session):
        self.db = db
        self._catalog_codes_cache: Optional[set] = None

    def _get_manual_review_sales_subquery(self):
        """
        Subquery para IDs de sales_data que tienen enrichment_status='manual_review'.

        IMPORTANTE: El sistema de enriquecimiento SIEMPRE crea un registro en
        SalesEnrichment para cada venta. El campo enrichment_status indica:
        - 'enriched': Producto encontrado y enriquecido
        - 'manual_review': No encontrado en catálogos, requiere revisión
        - 'failed': Error durante enriquecimiento
        - 'pending': En proceso

        Los productos "sin enriquecer" son aquellos con status='manual_review'.
        """
        return (
            self.db.query(SalesEnrichment.sales_data_id)
            .filter(
                and_(
                    SalesEnrichment.sales_data_id.isnot(None),
                    SalesEnrichment.enrichment_status == "manual_review",
                )
            )
            .scalar_subquery()  # Fix SAWarning: use scalar_subquery() for IN() clause
        )

    def _get_catalog_codes(self) -> set:
        """
        Obtiene conjunto de códigos nacionales en catálogos oficiales.
        Cachea el resultado para evitar queries repetidas.
        """
        if self._catalog_codes_cache is not None:
            return self._catalog_codes_cache

        # Códigos en ProductCatalog (CIMA)
        cima_codes = set(
            row[0] for row in
            self.db.query(ProductCatalog.national_code)
            .filter(ProductCatalog.national_code.isnot(None))
            .all()
            if row[0]
        )

        # Códigos en NomenclatorLocal
        nomenclator_codes = set(
            row[0] for row in
            self.db.query(NomenclatorLocal.national_code)
            .filter(NomenclatorLocal.national_code.isnot(None))
            .all()
            if row[0]
        )

        self._catalog_codes_cache = cima_codes | nomenclator_codes

        logger.debug(
            "manual_review.catalog_codes_loaded",
            cima_count=len(cima_codes),
            nomenclator_count=len(nomenclator_codes),
            total_unique=len(self._catalog_codes_cache),
        )

        return self._catalog_codes_cache

    @staticmethod
    def _validate_ean13_checksum(code: str) -> bool:
        """
        Valida dígito de control EAN-13.

        Algoritmo: suma ponderada (1,3,1,3...) mod 10.
        El dígito 13 debe ser (10 - suma % 10) % 10.
        """
        if len(code) != 13 or not code.isdigit():
            return False
        try:
            total = sum(
                int(d) * (1 if i % 2 == 0 else 3)
                for i, d in enumerate(code[:12])
            )
            check = (10 - (total % 10)) % 10
            return check == int(code[12])
        except (ValueError, IndexError):
            return False

    @staticmethod
    def _validate_cn7_checksum(code: str) -> bool:
        """
        Valida dígito de control CN de 7 dígitos (Módulo 10).

        Algoritmo: suma ponderada (3,1,3,1,3,1) sobre primeros 6 dígitos.
        El dígito 7 debe ser (10 - suma % 10) % 10.
        """
        if len(code) != 7 or not code.isdigit():
            return False
        try:
            weights = [3, 1, 3, 1, 3, 1]
            total = sum(int(code[i]) * weights[i] for i in range(6))
            check = (10 - (total % 10)) % 10
            return check == int(code[6])
        except (ValueError, IndexError):
            return False

    def _determine_code_type(self, code: str) -> Tuple[str, bool, bool]:
        """
        Determina el tipo de código con validación de checksum y cruce con catálogos.

        Args:
            code: Código del producto

        Returns:
            Tuple de (code_type, checksum_valid, in_catalog)

        Tipos posibles:
        - CN_CATALOGED: CN válido, existe en CIMA/Nomenclator
        - CN_OTC: CN válido checksum, NO en catálogos (parafarmacia como Frenadol)
        - EAN_CATALOGED: EAN válido, existe en catálogos
        - EAN_OTC: EAN válido checksum, NO en catálogos
        - CN_UNVERIFIED: 6 dígitos numérico (puede ser CN legacy O código interno)
        - INTERNAL: Código interno de farmacia (confirmado no-universal)
        """
        if not code:
            return "INTERNAL", False, False

        code = code.strip()
        length = len(code)
        is_numeric = code.isdigit()
        catalog_codes = self._get_catalog_codes()
        in_catalog = code in catalog_codes

        # EAN-13: 13 dígitos numéricos
        if length == 13 and is_numeric:
            checksum_valid = self._validate_ean13_checksum(code)
            if checksum_valid:
                if in_catalog:
                    return "EAN_CATALOGED", True, True
                else:
                    return "EAN_OTC", True, False
            else:
                return "INTERNAL", False, in_catalog

        # CN-7: 7 dígitos numéricos con checksum
        if length == 7 and is_numeric:
            checksum_valid = self._validate_cn7_checksum(code)
            if checksum_valid:
                if in_catalog:
                    return "CN_CATALOGED", True, True
                else:
                    return "CN_OTC", True, False
            else:
                return "INTERNAL", False, in_catalog

        # CN-6: 6 dígitos numéricos (formato legacy sin checksum)
        if length == 6 and is_numeric:
            if in_catalog:
                return "CN_CATALOGED", True, True  # Asumimos válido si está en catálogo
            else:
                return "CN_UNVERIFIED", True, False  # 6 dígitos sin validación cruzada

        # Cualquier otro caso: codigo interno
        return "INTERNAL", False, in_catalog

    def _get_code_type_filter_conditions(self, code_type: str):
        """
        Genera condiciones SQL para filtrar por tipo de codigo.

        Issue #448: Soporte para filtro code_type en tabla interactiva.

        Args:
            code_type: Tipo de codigo a filtrar (CN, EAN, INTERNAL)

        Returns:
            SQLAlchemy filter condition o None si no aplica filtro
        """
        cn_length = func.length(SalesData.codigo_nacional)
        ean_length = func.length(SalesData.ean13)

        # Condiciones basadas en longitud (igual que en get_stats)
        has_valid_cn = and_(
            SalesData.codigo_nacional.isnot(None),
            SalesData.codigo_nacional != "",
            cn_length >= 6,
            cn_length <= 7,
        )
        has_valid_ean = and_(
            SalesData.ean13.isnot(None),
            SalesData.ean13 != "",
            ean_length == 13,
        )

        if code_type == "CN":
            return has_valid_cn
        elif code_type == "EAN":
            # EAN sin CN valido (EAN-only)
            return and_(has_valid_ean, ~has_valid_cn)
        elif code_type == "INTERNAL":
            # Ni CN valido ni EAN valido
            return and_(~has_valid_cn, ~has_valid_ean)
        else:
            # Sin filtro - devolver None (incluir todo)
            return None

    def get_manual_review_products(
        self,
        pharmacy_id: Optional[UUID] = None,
        min_sales: Optional[int] = None,
        limit: Optional[int] = None,
        offset: int = 0,
        code_type: Optional[str] = None,
        search: Optional[str] = None,
        order_by: Optional[str] = None,
    ) -> Tuple[List[ManualReviewProduct], int]:
        """
        Obtiene productos en manual_review con metricas agregadas.

        IMPORTANTE: Cada producto se agrupa POR FARMACIA para evitar
        conciliar erroneamente codigos internos entre farmacias distintas.
        Los datos originales (codigo_nacional, ean13) se mantienen sin transformar.

        Un producto esta en "manual_review" si tiene un registro en SalesEnrichment
        con enrichment_status='manual_review' (no encontrado en catalogos oficiales).

        Args:
            pharmacy_id: Filtrar por farmacia especifica (opcional)
            min_sales: Minimo de ventas para incluir (opcional)
            limit: Limite de resultados (opcional)
            offset: Offset para paginacion
            code_type: Filtrar por tipo: CN, EAN, INTERNAL (opcional, Issue #448)
            search: Busqueda por nombre o codigo (opcional, Issue #448)
            order_by: Ordenamiento: amount_desc, sales_desc, name_asc (default: amount_desc, Issue #448)

        Returns:
            Tuple de (lista de productos, total count)
        """
        manual_review_sales_ids = self._get_manual_review_sales_subquery()

        # Query principal: agregar ventas CON enrichment_status='manual_review'
        # AGRUPAMOS POR FARMACIA para no mezclar códigos internos de distintas farmacias
        # NO filtramos por codigo_nacional para incluir productos solo con EAN
        query = (
            self.db.query(
                # Datos originales sin transformar
                SalesData.codigo_nacional.label("codigo_nacional"),
                SalesData.ean13.label("ean13"),
                SalesData.product_name.label("product_name"),
                # Identificación de farmacia
                SalesData.pharmacy_id.label("pharmacy_id"),
                Pharmacy.name.label("pharmacy_name"),
                # Métricas agregadas
                func.count(SalesData.id).label("sale_count"),
                func.sum(SalesData.quantity).label("total_units"),
                func.sum(SalesData.total_amount).label("total_amount"),
                func.avg(SalesData.unit_price).label("avg_price"),
                func.min(SalesData.sale_date).label("first_sale"),
                func.max(SalesData.sale_date).label("last_sale"),
            )
            .join(Pharmacy, SalesData.pharmacy_id == Pharmacy.id)
            .filter(SalesData.id.in_(manual_review_sales_ids))
            # NO filtramos por codigo_nacional - incluimos todos los productos
            # Agrupamos por farmacia + codigo + ean + nombre (producto único por farmacia)
            .group_by(
                SalesData.pharmacy_id,
                Pharmacy.name,
                SalesData.codigo_nacional,
                SalesData.ean13,
                SalesData.product_name,
            )
        )

        # Filtros opcionales
        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Issue #448: Filtro por tipo de codigo (CN, EAN, INTERNAL)
        if code_type and code_type in ("CN", "EAN", "INTERNAL"):
            code_type_condition = self._get_code_type_filter_conditions(code_type)
            if code_type_condition is not None:
                query = query.filter(code_type_condition)

        # Issue #448: Busqueda por nombre o codigo
        if search and search.strip():
            search_term = f"%{search.strip()}%"
            query = query.filter(
                or_(
                    SalesData.product_name.ilike(search_term),
                    SalesData.codigo_nacional.ilike(search_term),
                    SalesData.ean13.ilike(search_term),
                )
            )

        if min_sales:
            query = query.having(func.count(SalesData.id) >= min_sales)

        # Issue #448: Ordenamiento configurable
        # Opciones: amount_desc (default), sales_desc, name_asc
        if order_by == "sales_desc":
            query = query.order_by(func.count(SalesData.id).desc())
        elif order_by == "name_asc":
            query = query.order_by(SalesData.product_name.asc())
        else:
            # Default: amount_desc (mas relevantes primero)
            query = query.order_by(func.sum(SalesData.total_amount).desc())

        # Contar total antes de paginar
        count_query = query.subquery()
        total_count = self.db.query(func.count()).select_from(count_query).scalar() or 0

        # Aplicar paginación
        if limit:
            query = query.limit(limit)
        if offset:
            query = query.offset(offset)

        # Ejecutar y transformar resultados
        results = query.all()
        products = []

        for row in results:
            # Determinar codigo para display y clasificacion
            # Prioridad: codigo_nacional > ean13 > "SIN_CODIGO"
            display_code = row.codigo_nacional or row.ean13 or "SIN_CODIGO"
            product_code_type, checksum_valid, in_catalog = self._determine_code_type(display_code)

            # Convertir datetime a date (MIN/MAX de sale_date devuelve datetime)
            first_sale_date = (
                row.first_sale.date() if hasattr(row.first_sale, 'date') else row.first_sale
            ) if row.first_sale else date.today()
            last_sale_date = (
                row.last_sale.date() if hasattr(row.last_sale, 'date') else row.last_sale
            ) if row.last_sale else date.today()

            products.append(
                ManualReviewProduct(
                    # Datos originales
                    codigo_nacional=row.codigo_nacional,
                    ean13=row.ean13,
                    product_name=row.product_name or "Sin nombre",
                    # Farmacia
                    pharmacy_id=row.pharmacy_id,
                    pharmacy_name=row.pharmacy_name,
                    # Codigo para display
                    product_code=display_code,
                    # Clasificacion
                    code_type=product_code_type,
                    checksum_valid=checksum_valid,
                    in_catalog=in_catalog,
                    # Metricas (pharmacy_count=1 porque agrupamos por farmacia)
                    pharmacy_count=1,
                    sale_count=row.sale_count or 0,
                    total_units=row.total_units or 0,
                    total_amount=Decimal(str(row.total_amount or 0)),
                    avg_price=Decimal(str(round(row.avg_price or 0, 2))),
                    first_sale=first_sale_date,
                    last_sale=last_sale_date,
                )
            )

        logger.info(
            "manual_review.products.fetched",
            total_count=total_count,
            returned_count=len(products),
            pharmacy_id=str(pharmacy_id) if pharmacy_id else None,
            code_type_filter=code_type,
            search=search,
            order_by=order_by,
        )

        return products, total_count

    def get_stats(self, pharmacy_id: Optional[UUID] = None) -> ManualReviewStats:
        """
        Obtiene estadísticas resumen de productos en manual_review.

        Un producto está en "manual_review" si tiene un registro en SalesEnrichment
        con enrichment_status='manual_review' (no encontrado en catálogos oficiales).

        Args:
            pharmacy_id: Filtrar por farmacia (opcional)

        Returns:
            ManualReviewStats con totales y desglose por tipo de código
        """
        manual_review_sales_ids = self._get_manual_review_sales_subquery()

        # Query base para ventas con enrichment_status='manual_review'
        # Incluir ventas con CN válido O con EAN válido (para EAN-only products)
        base_query = (
            self.db.query(SalesData)
            .filter(SalesData.id.in_(manual_review_sales_ids))
            .filter(
                or_(
                    and_(
                        SalesData.codigo_nacional.isnot(None),
                        SalesData.codigo_nacional != "",
                    ),
                    and_(
                        SalesData.ean13.isnot(None),
                        SalesData.ean13 != "",
                    ),
                )
            )
        )

        if pharmacy_id:
            base_query = base_query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Total de líneas de venta
        total_sales = base_query.count()

        # Total de productos únicos (usando CN, EAN o nombre como identificador)
        # Prioridad: codigo_nacional > ean13 > product_name
        cn_length = func.length(SalesData.codigo_nacional)
        ean_length = func.length(SalesData.ean13)

        product_identifier = func.coalesce(
            case(
                (
                    and_(
                        SalesData.codigo_nacional.isnot(None),
                        SalesData.codigo_nacional != "",
                        cn_length >= 6,
                        cn_length <= 7,
                    ),
                    SalesData.codigo_nacional,
                )
            ),
            case(
                (
                    and_(
                        SalesData.ean13.isnot(None),
                        SalesData.ean13 != "",
                        ean_length == 13,
                    ),
                    SalesData.ean13,
                )
            ),
            SalesData.product_name,
        )

        total_products = (
            base_query.with_entities(
                func.count(distinct(product_identifier))
            ).scalar()
            or 0
        )

        # Importe total
        total_amount = (
            base_query.with_entities(func.sum(SalesData.total_amount)).scalar()
            or Decimal("0")
        )

        # Desglose por tipo de código (simplificado para stats - basado en longitud)
        # El desglose detallado con checksum se hace en get_manual_review_products
        # Lógica mejorada para detectar:
        # - CN: codigo_nacional con 6-7 dígitos
        # - EAN: ean13 con 13 dígitos (sin CN válido)
        # - INTERNAL: otros formatos
        # Nota: cn_length y ean_length ya definidos arriba para product_identifier
        has_valid_cn = and_(
            SalesData.codigo_nacional.isnot(None),
            SalesData.codigo_nacional != "",
            cn_length >= 6,
            cn_length <= 7,
        )
        has_valid_ean = and_(
            SalesData.ean13.isnot(None),
            SalesData.ean13 != "",
            ean_length == 13,
        )

        code_type_case = case(
            (has_valid_cn, "CN"),
            (has_valid_ean, "EAN"),  # EAN-only (no valid CN)
            else_="INTERNAL",
        )

        # Usar COALESCE para contar productos únicos por CN o EAN
        unique_code = func.coalesce(
            case((has_valid_cn, SalesData.codigo_nacional)),
            case((has_valid_ean, SalesData.ean13)),
            SalesData.product_name,  # Fallback para INTERNAL
        )

        type_breakdown = (
            base_query.with_entities(
                code_type_case.label("code_type"),
                func.count(distinct(unique_code)).label("count"),
            )
            .group_by(code_type_case)
            .all()
        )

        by_code_type = {"CN": 0, "EAN": 0, "INTERNAL": 0}
        for row in type_breakdown:
            if row.code_type in by_code_type:
                by_code_type[row.code_type] = row.count

        return ManualReviewStats(
            total_products=total_products,
            total_sales=total_sales,
            total_amount=Decimal(str(total_amount)),
            by_code_type=by_code_type,
        )

    def export_to_csv(
        self,
        pharmacy_id: Optional[UUID] = None,
        min_sales: Optional[int] = None,
    ) -> Tuple[StringIO, int]:
        """
        Exporta productos en manual_review a CSV con clasificación mejorada.

        Issue #447: Incluye datos originales y farmacia para máxima trazabilidad.

        Args:
            pharmacy_id: Filtrar por farmacia (opcional)
            min_sales: Mínimo de ventas para incluir (opcional)

        Returns:
            Tuple de (StringIO con CSV, número de filas)

        Columnas del CSV (16 columnas totales):
        - Datos originales: codigo_nacional, ean13, product_name
        - Farmacia: pharmacy_id, pharmacy_name
        - Display: product_code (COALESCE CN > EAN > SIN_CODIGO)
        - Clasificación: code_type, checksum_valid, in_catalog
        - Métricas: pharmacy_count, sale_count, total_units,
                   total_amount, avg_price, first_sale, last_sale
        """
        # Obtener todos los productos (sin límite para exportación)
        products, total_count = self.get_manual_review_products(
            pharmacy_id=pharmacy_id,
            min_sales=min_sales,
            limit=None,
            offset=0,
        )

        # Crear CSV en memoria
        output = StringIO()
        writer = csv.writer(output)

        # Header con todas las columnas para maxima trazabilidad (Issue #447)
        # Incluye datos originales (codigo_nacional, ean13) + identificacion farmacia
        writer.writerow(
            [
                # Datos originales (sin transformar)
                "codigo_nacional",
                "ean13",
                "product_name",
                # Identificacion de farmacia (agrupamos por farmacia)
                "pharmacy_id",
                "pharmacy_name",
                # Codigo para display (COALESCE CN > EAN > SIN_CODIGO)
                "product_code",
                # Clasificacion
                "code_type",
                "checksum_valid",
                "in_catalog",
                # Metricas
                "pharmacy_count",
                "sale_count",
                "total_units",
                "total_amount",
                "avg_price",
                "first_sale",
                "last_sale",
            ]
        )

        # Data rows
        for product in products:
            writer.writerow(
                [
                    # Datos originales
                    product.codigo_nacional or "",
                    product.ean13 or "",
                    product.product_name,
                    # Farmacia
                    str(product.pharmacy_id) if product.pharmacy_id else "",
                    product.pharmacy_name or "",
                    # Display
                    product.product_code,
                    # Clasificacion
                    product.code_type,
                    product.checksum_valid,
                    product.in_catalog,
                    # Metricas
                    product.pharmacy_count,
                    product.sale_count,
                    product.total_units,
                    str(product.total_amount),
                    str(product.avg_price),
                    product.first_sale.isoformat(),
                    product.last_sale.isoformat(),
                ]
            )

        output.seek(0)

        logger.info(
            "manual_review.export.completed",
            row_count=len(products),
            pharmacy_id=str(pharmacy_id) if pharmacy_id else None,
        )

        return output, len(products)
