# backend/app/services/license_client.py
"""
License Client Service for kaiFarma Local (Pivot 2026).

Validates license keys against Hub Central with offline support.

Features:
- HTTP client with retry/timeout for Hub API
- Local SQLite cache for offline operation
- 7-day grace period before kill-switch
- Time travel protection (clock rollback detection)
- Machine binding (anti-clone protection)

Usage:
    from app.services.license_client import LicenseClientService

    # At startup
    license_client = LicenseClientService(db)
    result = await license_client.validate_license("KAI-XXXX-YYYY-ZZZZ")

    if result.kill_switch_active:
        # Enter read-only mode
        pass
"""
import logging
import os
from dataclasses import dataclass
from datetime import datetime
from typing import Optional

import httpx
from sqlalchemy.orm import Session

from app.models.license_cache import GRACE_PERIOD_DAYS, LicenseCache, get_machine_id
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


# Configuration
HUB_URL = os.getenv("HUB_URL", "https://api.kaifarma.es")
REVALIDATION_INTERVAL_HOURS = int(os.getenv("LICENSE_REVALIDATION_HOURS", "24"))
HTTP_TIMEOUT_SECONDS = 10.0
MAX_RETRIES = 3
RETRY_BASE_DELAY = 1.0  # seconds
RETRY_MAX_DELAY = 10.0  # cap at 10 seconds


@dataclass
class LicenseValidationResult:
    """Result of license validation."""

    is_valid: bool
    tier: str = "standard"  # free, standard, premium
    pharmacy_id: Optional[str] = None
    pharmacy_name: Optional[str] = None
    expires_at: Optional[datetime] = None

    # Grace period status
    in_grace_period: bool = False
    grace_period_ends: Optional[datetime] = None
    days_remaining: Optional[int] = None

    # Security status
    kill_switch_active: bool = False
    security_violation: Optional[str] = None  # "time_travel" or "machine_mismatch"

    # Error info
    error_message: Optional[str] = None
    hub_reachable: bool = True


class LicenseClientService:
    """
    License validation client for Hub Central.

    Handles both online (Hub API) and offline (cached) validation
    with security protections against clock manipulation and DB cloning.
    """

    def __init__(self, db: Session):
        """
        Initialize license client.

        Args:
            db: SQLAlchemy session for cache access
        """
        self.db = db
        self.hub_url = HUB_URL.rstrip("/")

    async def validate_license(self, license_key: str) -> LicenseValidationResult:
        """
        Validate a license key with full security checks.

        Flow:
        1. Check for security violations (time travel, machine mismatch)
        2. Try online validation against Hub
        3. Fall back to cached validation if offline
        4. Check grace period and kill-switch status

        Args:
            license_key: License key (KAI-XXXX-YYYY-ZZZZ)

        Returns:
            LicenseValidationResult with full status
        """
        # Get or create cache entry
        cache = self._get_or_create_cache(license_key)

        # Step 1: Check for security violations
        if cache.requires_immediate_revalidation():
            logger.warning(
                "[LICENSE] Security violation detected - forcing online validation",
                extra={"security_status": cache.get_security_status()},
            )
            # Must validate online - no grace period for security violations
            result = await self._validate_online(license_key, cache)
            if not result.hub_reachable:
                # CRITICAL: Security violation detected and Hub is unreachable
                # This could be an attack - log at highest level
                logger.critical(
                    "[LICENSE] SECURITY VIOLATION + OFFLINE - Kill switch activated",
                    extra={
                        "security_violation": cache.get_security_status(),
                        "last_online": cache.last_online_validation,
                        "machine_id_mismatch": cache.is_machine_mismatch(),
                        "time_travel": cache.is_time_travel_detected(),
                        "license_key": license_key[:12] + "...",
                        "action": "kill_switch_activated",
                    }
                )
                # Can't reach Hub with security violation - kill switch
                return LicenseValidationResult(
                    is_valid=False,
                    kill_switch_active=True,
                    security_violation=cache.get_security_status(),
                    error_message="Security violation - Hub unreachable for verification",
                    hub_reachable=False,
                )
            return result

        # Step 2: Check if revalidation needed
        if cache.needs_revalidation(REVALIDATION_INTERVAL_HOURS):
            logger.info(
                "[LICENSE] Revalidation needed",
                extra={"hours_since": REVALIDATION_INTERVAL_HOURS},
            )
            # _validate_online handles both success and offline fallback
            # It returns the appropriate result with hub_reachable flag set correctly
            return await self._validate_online(license_key, cache)

        # Step 3: Use cached validation (no revalidation needed)
        return self._validate_from_cache(cache)

    async def _validate_online(
        self, license_key: str, cache: LicenseCache
    ) -> LicenseValidationResult:
        """
        Validate license against Hub API.

        Args:
            license_key: License key
            cache: Cache entry to update

        Returns:
            Validation result (hub_reachable=False if offline)
        """
        try:
            async with httpx.AsyncClient(timeout=HTTP_TIMEOUT_SECONDS) as client:
                response = await self._call_hub_with_retry(client, license_key)

                if response.status_code == 200:
                    data = response.json()
                    return self._process_hub_success(data, cache)

                elif response.status_code == 401:
                    # Invalid license key
                    return self._process_hub_invalid(
                        "License key not found or inactive", cache
                    )

                elif response.status_code == 403:
                    # License revoked
                    return self._process_hub_invalid(
                        "License has been revoked", cache
                    )

                else:
                    # Unexpected status
                    logger.warning(
                        "[LICENSE] Unexpected Hub response",
                        extra={"status": response.status_code},
                    )
                    cache.mark_offline_validation()
                    self.db.commit()
                    return self._validate_from_cache(cache, hub_reachable=True)

        except httpx.TimeoutException:
            logger.warning("[LICENSE] Hub timeout - using cached validation")
            cache.mark_offline_validation()
            self.db.commit()
            return self._validate_from_cache(cache, hub_reachable=False)

        except httpx.ConnectError:
            logger.warning("[LICENSE] Hub unreachable - using cached validation")
            cache.mark_offline_validation()
            self.db.commit()
            return self._validate_from_cache(cache, hub_reachable=False)

        except Exception as e:
            logger.error(
                "[LICENSE] Validation error - falling back to cache",
                exc_info=True,
                extra={"error_type": type(e).__name__}
            )
            cache.mark_offline_validation()
            self.db.commit()
            return self._validate_from_cache(cache, hub_reachable=False)

    async def _call_hub_with_retry(
        self, client: httpx.AsyncClient, license_key: str
    ) -> httpx.Response:
        """
        Call Hub API with retry logic and exponential backoff.

        Args:
            client: HTTP client
            license_key: License key to validate

        Returns:
            HTTP response

        Raises:
            httpx.TimeoutException or httpx.ConnectError after all retries exhausted
        """
        import asyncio

        last_error = None

        for attempt in range(MAX_RETRIES):
            try:
                response = await client.get(
                    f"{self.hub_url}/api/v1/licenses/validate",
                    headers={"X-License-Key": license_key},
                )
                return response

            except (httpx.TimeoutException, httpx.ConnectError) as e:
                last_error = e
                if attempt < MAX_RETRIES - 1:
                    # Exponential backoff with cap: 1s, 2s, 4s (max 10s)
                    delay = min(RETRY_BASE_DELAY * (2 ** attempt), RETRY_MAX_DELAY)
                    logger.info(
                        f"[LICENSE] Retry {attempt + 1}/{MAX_RETRIES} after {delay}s",
                        extra={"error": str(e), "delay": delay},
                    )
                    await asyncio.sleep(delay)

        raise last_error

    def _process_hub_success(
        self, data: dict, cache: LicenseCache
    ) -> LicenseValidationResult:
        """Process successful Hub validation response."""
        expires_at = None
        if data.get("expires_at"):
            expires_at = datetime.fromisoformat(data["expires_at"].replace("Z", "+00:00"))

        # Update cache
        cache.update_from_hub_response(
            is_valid=True,
            pharmacy_id=data.get("pharmacy_id"),
            pharmacy_name=data.get("pharmacy_name"),
            tier=data.get("tier", "standard"),
            expires_at=expires_at,
        )
        self.db.commit()

        logger.info(
            "[LICENSE] Validation successful",
            extra={
                "pharmacy_name": data.get("pharmacy_name"),
                "tier": data.get("tier"),
            },
        )

        return LicenseValidationResult(
            is_valid=True,
            tier=data.get("tier", "standard"),
            pharmacy_id=data.get("pharmacy_id"),
            pharmacy_name=data.get("pharmacy_name"),
            expires_at=expires_at,
            in_grace_period=False,
            grace_period_ends=cache.grace_period_ends,
            hub_reachable=True,
        )

    def _process_hub_invalid(
        self, error_message: str, cache: LicenseCache
    ) -> LicenseValidationResult:
        """Process invalid license response from Hub."""
        cache.update_from_hub_response(is_valid=False, error=error_message)
        self.db.commit()

        logger.warning(f"[LICENSE] Validation failed: {error_message}")

        return LicenseValidationResult(
            is_valid=False,
            kill_switch_active=True,  # Invalid license = immediate kill
            error_message=error_message,
            hub_reachable=True,
        )

    def _validate_from_cache(
        self, cache: LicenseCache, hub_reachable: bool = True
    ) -> LicenseValidationResult:
        """
        Validate using cached data (offline mode).

        Args:
            cache: Cached license data
            hub_reachable: Whether Hub was reachable

        Returns:
            Validation result based on cache state
        """
        # Check kill-switch
        if cache.is_kill_switch_active():
            logger.warning(
                "[LICENSE] Grace period expired - kill switch active",
                extra={"grace_period_ends": cache.grace_period_ends},
            )
            return LicenseValidationResult(
                is_valid=False,
                tier=cache.tier or "standard",
                pharmacy_id=cache.pharmacy_id,
                pharmacy_name=cache.pharmacy_name,
                kill_switch_active=True,
                error_message="Grace period expired - online validation required",
                hub_reachable=hub_reachable,
            )

        # Check grace period
        if cache.is_in_grace_period():
            days_remaining = cache.days_until_kill_switch()
            logger.info(
                f"[LICENSE] In grace period - {days_remaining} days remaining",
                extra={"grace_period_ends": cache.grace_period_ends},
            )
            return LicenseValidationResult(
                is_valid=True,
                tier=cache.tier or "standard",
                pharmacy_id=cache.pharmacy_id,
                pharmacy_name=cache.pharmacy_name,
                in_grace_period=True,
                grace_period_ends=cache.grace_period_ends,
                days_remaining=days_remaining,
                hub_reachable=hub_reachable,
            )

        # Cache says valid
        if cache.is_valid:
            return LicenseValidationResult(
                is_valid=True,
                tier=cache.tier or "standard",
                pharmacy_id=cache.pharmacy_id,
                pharmacy_name=cache.pharmacy_name,
                expires_at=cache.expires_at,
                hub_reachable=hub_reachable,
            )

        # Cache says invalid
        return LicenseValidationResult(
            is_valid=False,
            kill_switch_active=True,
            error_message=cache.validation_error or "License not validated",
            hub_reachable=hub_reachable,
        )

    def _get_or_create_cache(self, license_key: str) -> LicenseCache:
        """
        Get existing cache entry or create new one.

        Args:
            license_key: License key

        Returns:
            LicenseCache instance
        """
        cache = self.db.query(LicenseCache).filter(LicenseCache.id == 1).first()

        if cache is None:
            # First run - create cache entry
            logger.info("[LICENSE] Creating new license cache entry")
            cache = LicenseCache(
                id=1,
                license_key=license_key,
                is_valid=False,  # Not validated yet
                machine_id=get_machine_id(),
            )
            self.db.add(cache)
            self.db.commit()

        elif cache.license_key != license_key:
            # License key changed - reset cache
            logger.info(
                "[LICENSE] License key changed - resetting cache",
                extra={"old": cache.license_key[:12] + "..."},
            )
            cache.license_key = license_key
            cache.is_valid = False
            cache.last_online_validation = None
            cache.grace_period_ends = None
            cache.machine_id = get_machine_id()
            self.db.commit()

        return cache

    def get_license_status(self) -> dict:
        """
        Get current license status for API endpoint.

        Returns:
            Dictionary with license status
        """
        cache = self.db.query(LicenseCache).filter(LicenseCache.id == 1).first()

        if cache is None:
            return {
                "is_valid": False,
                "tier": "unknown",
                "pharmacy_name": None,
                "in_grace_period": False,
                "kill_switch_active": False,
                "needs_activation": True,
            }

        return cache.to_dict()
