﻿# backend/app/services/laboratory_cache_service.py
"""
Laboratory Cache Service - Redis-based caching para laboratory mappings
FASE 1.3 - Issue #23: Laboratory Mapping Performance Optimization

Funcionalidades:
- Cache layer para endpoints /codes-to-names, /names-to-codes, /generic-laboratories
- TTL diferenciado por tipo de datos (24h mappings, 12h generics)
- Cache invalidation automática en catalog updates
- Cache warming durante startup
- Graceful degradation si Redis no disponible
- Monitoreo de performance y hit rates
"""

import logging
import time
from typing import Any, Dict, List, Optional, Tuple, Union

from sqlalchemy.orm import Session

from ..core.cache_config import CacheConfig, CacheKeyGenerator, CacheSerializer, get_redis, is_cache_available
from ..utils.datetime_utils import utc_now
from ..utils.laboratory_queries import laboratory_query_builder

logger = logging.getLogger(__name__)


class CacheStats:
    """Estadísticas de performance del cache"""

    def __init__(self):
        self.hits = 0
        self.misses = 0
        self.errors = 0
        self.total_operations = 0
        self.avg_response_time = 0.0
        self.cache_size_estimate = 0

    @property
    def hit_rate(self) -> float:
        """Calcular hit rate porcentual"""
        total = self.hits + self.misses
        return (self.hits / total * 100) if total > 0 else 0.0

    @property
    def error_rate(self) -> float:
        """Calcular error rate porcentual"""
        return (self.errors / self.total_operations * 100) if self.total_operations > 0 else 0.0

    def to_dict(self) -> Dict[str, Any]:
        """Convertir a diccionario para serialización"""
        return {
            "hits": self.hits,
            "misses": self.misses,
            "errors": self.errors,
            "total_operations": self.total_operations,
            "hit_rate_percent": round(self.hit_rate, 2),
            "error_rate_percent": round(self.error_rate, 2),
            "avg_response_time_ms": round(self.avg_response_time * 1000, 2),
            "cache_size_estimate_mb": round(self.cache_size_estimate / (1024 * 1024), 2),
            "timestamp": utc_now().isoformat(),
        }


class LaboratoryCacheService:
    """Servicio de cache Redis para laboratory mappings"""

    def __init__(self):
        self._stats = CacheStats()
        self._operation_times: List[float] = []
        self._max_operation_history = 1000

    def _record_operation(self, operation_time: float, cache_hit: bool, error: bool = False) -> None:
        """Registrar estadísticas de operación"""
        self._stats.total_operations += 1

        if error:
            self._stats.errors += 1
        elif cache_hit:
            self._stats.hits += 1
        else:
            self._stats.misses += 1

        # Mantener histórico de tiempos para avg_response_time
        self._operation_times.append(operation_time)
        if len(self._operation_times) > self._max_operation_history:
            self._operation_times.pop(0)

        # Calcular average response time
        if self._operation_times:
            self._stats.avg_response_time = sum(self._operation_times) / len(self._operation_times)

    def _get_cache_key(self, key_pattern: str, **kwargs) -> str:
        """Generar cache key usando el patrón especificado"""
        return key_pattern.format(**kwargs)

    def _set_cache_with_ttl(self, key: str, data: Any, ttl: int) -> bool:
        """Guardar en cache con TTL especificado"""
        try:
            redis_client = get_redis()
            if not redis_client:
                return False

            serialized_data = CacheSerializer.serialize(data)
            redis_client.setex(key, ttl, serialized_data)
            return True

        except Exception as e:
            logger.error(f"[CACHE] Error guardando en cache {key}: {str(e)[:100]}")
            return False

    def _get_cache_data(self, key: str) -> Optional[Any]:
        """Obtener datos desde cache"""
        try:
            redis_client = get_redis()
            if not redis_client:
                return None

            cached_data = redis_client.get(key)
            if cached_data is None:
                return None

            return CacheSerializer.deserialize(cached_data)

        except Exception as e:
            logger.error(f"[CACHE] Error obteniendo desde cache {key}: {str(e)[:100]}")
            return None

    def _invalidate_pattern(self, pattern: str) -> int:
        """Invalidar cache keys que coincidan con el patrón"""
        try:
            redis_client = get_redis()
            if not redis_client:
                return 0

            keys = redis_client.keys(pattern)
            if keys:
                count = redis_client.delete(*keys)
                logger.info(f"[CACHE] Invalidados {count} keys con patrón: {pattern}")
                return count
            return 0

        except Exception as e:
            logger.error(f"[CACHE] Error invalidando patrón {pattern}: {str(e)[:100]}")
            return 0

    def _unpack_cached_data(
        self,
        cached_data: Any,
        return_total: bool
    ) -> Optional[Union[Tuple[Dict[str, str], bool], Tuple[Dict[str, str], bool, int]]]:
        """
        Unpack cached data handling JSON tuple->list conversion.

        JSON serialization converts Python tuples to lists. This method handles
        both formats for backward compatibility with existing cached data.

        Args:
            cached_data: Data retrieved from cache (may be dict, list, or tuple)
            return_total: Whether to return total count as third element

        Returns:
            Tuple of (mapping, cache_hit, [total]) or None if data is corrupted
        """
        is_tuple_like = isinstance(cached_data, (tuple, list)) and len(cached_data) == 2

        if return_total:
            if is_tuple_like:
                mapping_data, total_data = cached_data
                # Defensive: validate mapping is a dict
                if not isinstance(mapping_data, dict):
                    logger.warning("[CACHE] Corrupted cache: mapping is not a dict, treating as cache miss")
                    return None
                return mapping_data, True, total_data
            else:
                # Legacy cache format without total - derive count from mapping
                if not isinstance(cached_data, dict):
                    logger.warning("[CACHE] Corrupted cache: expected dict, treating as cache miss")
                    return None
                return cached_data, True, len(cached_data)
        else:
            if is_tuple_like:
                mapping = cached_data[0]
                # Defensive: validate mapping is a dict
                if not isinstance(mapping, dict):
                    logger.warning("[CACHE] Corrupted cache: mapping is not a dict, treating as cache miss")
                    return None
                return mapping, True
            else:
                if not isinstance(cached_data, dict):
                    logger.warning("[CACHE] Corrupted cache: expected dict, treating as cache miss")
                    return None
                return cached_data, True

    # === MÉTODOS PÚBLICOS PARA LABORATORY MAPPINGS ===

    def get_codes_to_names_cached(
        self,
        db: Session,
        codes: Optional[List[str]] = None,
        page: Optional[int] = None,
        per_page: Optional[int] = None,
        return_total: bool = False,
    ) -> Union[Tuple[Dict[str, str], bool], Tuple[Dict[str, str], bool, int]]:
        """
        Obtener mapeo códigos → nombres con cache
        Returns: (mapping, cache_hit)
        """
        start_time = time.time()
        cache_hit = False

        try:
            # FASE 2.2: Cache keys específicos para paginación
            codes_hash = CacheKeyGenerator.hash_codes(codes or [])
            page_suffix = f":p{page}:{per_page}" if page is not None and per_page is not None else ""
            cache_key = self._get_cache_key(CacheConfig.KEY_CODES_TO_NAMES + page_suffix, hash=codes_hash)

            # Intentar obtener desde cache
            if is_cache_available():
                cached_data = self._get_cache_data(cache_key)
                if cached_data is not None:
                    # Use helper to unpack cached data (handles JSON tuple->list conversion)
                    unpacked = self._unpack_cached_data(cached_data, return_total)
                    if unpacked is not None:
                        operation_time = time.time() - start_time
                        self._record_operation(operation_time, cache_hit=True)
                        logger.debug(f"[CACHE] Cache HIT para codes_to_names: {cache_key}")
                        return unpacked
                    # If unpacked is None, data was corrupted - fall through to cache miss

            # Cache miss - obtener desde BD con paginación
            if return_total or (page is not None and per_page is not None):
                mapping, total_count = self._fetch_codes_to_names_from_db_with_total(db, codes, page, per_page)
            else:
                mapping = self._fetch_codes_to_names_from_db(db, codes)
                total_count = None

            # Guardar en cache
            if is_cache_available() and mapping:
                cache_data = (mapping, total_count) if return_total else mapping
                self._set_cache_with_ttl(cache_key, cache_data, CacheConfig.TTL_LABORATORY_MAPPINGS)
                logger.debug(f"[CACHE] Datos guardados en cache: {cache_key}")

            operation_time = time.time() - start_time
            self._record_operation(operation_time, cache_hit=False)

            if return_total:
                return mapping, False, total_count or 0
            else:
                return mapping, False

        except Exception as e:
            operation_time = time.time() - start_time
            self._record_operation(operation_time, cache_hit=False, error=True)
            logger.error(f"[CACHE] Error en get_codes_to_names_cached: {str(e)[:200]}")
            # Fallback - obtener directamente desde BD
            if return_total:
                fallback_mapping, fallback_total = self._fetch_codes_to_names_from_db_with_total(
                    db, codes, page, per_page
                )
                return fallback_mapping, False, fallback_total
            else:
                return self._fetch_codes_to_names_from_db(db, codes), False

    def get_names_to_codes_cached(
        self,
        db: Session,
        names: Optional[List[str]] = None,
        page: Optional[int] = None,
        per_page: Optional[int] = None,
        return_total: bool = False,
    ) -> Union[Tuple[Dict[str, str], bool], Tuple[Dict[str, str], bool, int]]:
        """
        Obtener mapeo nombres → códigos con cache
        Returns: (mapping, cache_hit)
        """
        start_time = time.time()

        try:
            # FASE 2.2: Cache keys específicos para paginación
            names_hash = CacheKeyGenerator.hash_names(names or [])
            page_suffix = f":p{page}:{per_page}" if page is not None and per_page is not None else ""
            cache_key = self._get_cache_key(CacheConfig.KEY_NAMES_TO_CODES + page_suffix, hash=names_hash)

            # Intentar obtener desde cache
            if is_cache_available():
                cached_data = self._get_cache_data(cache_key)
                if cached_data is not None:
                    # Use helper to unpack cached data (handles JSON tuple->list conversion)
                    unpacked = self._unpack_cached_data(cached_data, return_total)
                    if unpacked is not None:
                        operation_time = time.time() - start_time
                        self._record_operation(operation_time, cache_hit=True)
                        logger.debug(f"[CACHE] Cache HIT para names_to_codes: {cache_key}")
                        return unpacked
                    # If unpacked is None, data was corrupted - fall through to cache miss

            # Cache miss - obtener desde BD con paginación
            if return_total or (page is not None and per_page is not None):
                mapping, total_count = self._fetch_names_to_codes_from_db_with_total(db, names, page, per_page)
            else:
                mapping = self._fetch_names_to_codes_from_db(db, names)
                total_count = None

            # Guardar en cache
            if is_cache_available() and mapping:
                cache_data = (mapping, total_count) if return_total else mapping
                self._set_cache_with_ttl(cache_key, cache_data, CacheConfig.TTL_LABORATORY_MAPPINGS)
                logger.debug(f"[CACHE] Datos guardados en cache: {cache_key}")

            operation_time = time.time() - start_time
            self._record_operation(operation_time, cache_hit=False)

            if return_total:
                return mapping, False, total_count or 0
            else:
                return mapping, False

        except Exception as e:
            operation_time = time.time() - start_time
            self._record_operation(operation_time, cache_hit=False, error=True)
            logger.error(f"[CACHE] Error en get_names_to_codes_cached: {str(e)[:200]}")
            if return_total:
                fallback_mapping, fallback_total = self._fetch_names_to_codes_from_db_with_total(
                    db, names, page, per_page
                )
                return fallback_mapping, False, fallback_total
            else:
                return self._fetch_names_to_codes_from_db(db, names), False

    def get_generic_laboratories_cached(
        self, db: Session, page: int = 1, per_page: int = 50, search: Optional[str] = None, return_total: bool = False
    ) -> Union[Tuple[Dict[str, str], bool], Tuple[Dict[str, str], bool, int]]:
        """
        Obtener laboratorios genéricos con cache
        Returns: (mapping, cache_hit)
        """
        start_time = time.time()

        try:
            # Generar cache key
            query_hash = CacheKeyGenerator.hash_generic_query(page, per_page, search)
            cache_key = self._get_cache_key(CacheConfig.KEY_GENERIC_LABS, query_hash=query_hash)

            # Intentar obtener desde cache
            if is_cache_available():
                cached_data = self._get_cache_data(cache_key)
                if cached_data is not None:
                    # Use helper to unpack cached data (handles JSON tuple->list conversion)
                    unpacked = self._unpack_cached_data(cached_data, return_total)
                    if unpacked is not None:
                        operation_time = time.time() - start_time
                        self._record_operation(operation_time, cache_hit=True)
                        logger.debug(f"[CACHE] Cache HIT para generic_laboratories: {cache_key}")
                        return unpacked
                    # If unpacked is None, data was corrupted - fall through to cache miss

            # Cache miss - obtener desde BD con total count si se requiere
            if return_total:
                mapping, total_count = self._fetch_generic_laboratories_from_db_with_total(db, page, per_page, search)
            else:
                mapping = self._fetch_generic_laboratories_from_db(db, page, per_page, search)
                total_count = None

            # Guardar en cache (TTL más corto para queries dinámicas)
            if is_cache_available() and mapping:
                cache_data = (mapping, total_count) if return_total else mapping
                self._set_cache_with_ttl(cache_key, cache_data, CacheConfig.TTL_GENERIC_LABORATORIES)
                logger.debug(f"[CACHE] Datos guardados en cache: {cache_key}")

            operation_time = time.time() - start_time
            self._record_operation(operation_time, cache_hit=False)

            if return_total:
                return mapping, False, total_count or 0
            else:
                return mapping, False

        except Exception as e:
            operation_time = time.time() - start_time
            self._record_operation(operation_time, cache_hit=False, error=True)
            logger.error(f"[CACHE] Error en get_generic_laboratories_cached: {str(e)[:200]}")
            if return_total:
                fallback_mapping, fallback_total = self._fetch_generic_laboratories_from_db_with_total(
                    db, page, per_page, search
                )
                return fallback_mapping, False, fallback_total
            else:
                return self._fetch_generic_laboratories_from_db(db, page, per_page, search), False

    # === MÉTODOS DE FALLBACK PARA BD (FASE 2.2: Con soporte de paginación) ===

    def _fetch_codes_to_names_from_db_with_total(
        self, db: Session, codes: Optional[List[str]], page: Optional[int] = None, per_page: Optional[int] = None
    ) -> Tuple[Dict[str, str], int]:
        """Obtener codes_to_names con conteo total para paginación"""
        start_time = time.time()

        try:
            # Obtener total count
            count_query, count_params = laboratory_query_builder.build_count_query_for_codes(codes)
            if laboratory_query_builder.validate_query_parameters(count_params):
                count_result = db.execute(count_query, count_params).fetchone()
                total_count = count_result.total if count_result else 0
            else:
                logger.error("[CACHE] Parámetros de count query no válidos")
                total_count = 0

            # Obtener datos paginados
            use_cursor = False
            cursor_id = None
            if page and per_page and total_count > 0:
                use_cursor = laboratory_query_builder.should_use_cursor_pagination(page, total_count)

            query, params = laboratory_query_builder.build_codes_to_names_query(
                codes, page, per_page, use_cursor, cursor_id
            )

            if not laboratory_query_builder.validate_query_parameters(params):
                logger.error("[CACHE] Parámetros de query no válidos en codes_to_names_with_total")
                return {}, 0

            result = db.execute(query, params).fetchall()
            mapping = {}

            for row in result:
                code = row.nomen_codigo_laboratorio
                name = row.nomen_laboratorio
                if code and name and len(code) <= 4 and code.isdigit():
                    mapping[code] = name

            execution_time = time.time() - start_time
            laboratory_query_builder.log_query_execution(
                "codes_to_names_with_total", {**params, **count_params}, execution_time
            )

            return mapping, total_count

        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"[CACHE] Error en fallback codes_to_names_with_total: {str(e)[:200]}")
            laboratory_query_builder.log_query_execution("codes_to_names_with_total_ERROR", {}, execution_time)
            return {}, 0

    def _fetch_names_to_codes_from_db_with_total(
        self, db: Session, names: Optional[List[str]], page: Optional[int] = None, per_page: Optional[int] = None
    ) -> Tuple[Dict[str, str], int]:
        """Obtener names_to_codes con conteo total para paginación"""
        start_time = time.time()

        try:
            # Obtener total count
            count_query, count_params = laboratory_query_builder.build_count_query_for_names(names)
            if laboratory_query_builder.validate_query_parameters(count_params):
                count_result = db.execute(count_query, count_params).fetchone()
                total_count = count_result.total if count_result else 0
            else:
                logger.error("[CACHE] Parámetros de count query no válidos")
                total_count = 0

            # Obtener datos paginados
            use_cursor = False
            cursor_id = None
            if page and per_page and total_count > 0:
                use_cursor = laboratory_query_builder.should_use_cursor_pagination(page, total_count)

            query, params = laboratory_query_builder.build_names_to_codes_query(
                names, page, per_page, use_cursor, cursor_id
            )

            if not laboratory_query_builder.validate_query_parameters(params):
                logger.error("[CACHE] Parámetros de query no válidos en names_to_codes_with_total")
                return {}, 0

            result = db.execute(query, params).fetchall()
            mapping = {}

            for row in result:
                code = row.nomen_codigo_laboratorio
                name = row.nomen_laboratorio
                if code and name and len(code) <= 4 and code.isdigit():
                    mapping[name] = code

            execution_time = time.time() - start_time
            laboratory_query_builder.log_query_execution(
                "names_to_codes_with_total", {**params, **count_params}, execution_time
            )

            return mapping, total_count

        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"[CACHE] Error en fallback names_to_codes_with_total: {str(e)[:200]}")
            laboratory_query_builder.log_query_execution("names_to_codes_with_total_ERROR", {}, execution_time)
            return {}, 0

    def _fetch_generic_laboratories_from_db_with_total(
        self, db: Session, page: int, per_page: int, search: Optional[str]
    ) -> Tuple[Dict[str, str], int]:
        """Obtener generic_laboratories con conteo total para paginación"""
        start_time = time.time()

        try:
            # Obtener total count
            count_query, count_params = laboratory_query_builder.build_count_query_for_generics(search)
            if laboratory_query_builder.validate_query_parameters(count_params):
                count_result = db.execute(count_query, count_params).fetchone()
                total_count = count_result.total if count_result else 0
            else:
                logger.error("[CACHE] Parámetros de count query no válidos")
                total_count = 0

            # Obtener datos paginados
            use_cursor = False
            cursor_id = None
            if total_count > 0:
                use_cursor = laboratory_query_builder.should_use_cursor_pagination(page, total_count)

            query, params = laboratory_query_builder.build_generic_laboratories_query(
                page, per_page, search, use_cursor, cursor_id
            )

            if not laboratory_query_builder.validate_query_parameters(params):
                logger.error("[CACHE] Parámetros de query no válidos en generic_laboratories_with_total")
                return {}, 0

            result = db.execute(query, params).fetchall()
            mapping = {}

            for row in result:
                code = row.nomen_codigo_laboratorio
                name = row.nomen_laboratorio
                if code and name and len(code) <= 4 and code.isdigit():
                    mapping[code] = name

            execution_time = time.time() - start_time
            laboratory_query_builder.log_query_execution(
                f"generic_laboratories_with_total_page_{page}", {**params, **count_params}, execution_time
            )

            return mapping, total_count

        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"[CACHE] Error en fallback generic_laboratories_with_total: {str(e)[:200]}")
            laboratory_query_builder.log_query_execution(
                f"generic_laboratories_with_total_ERROR_page_{page}", {}, execution_time
            )
            return {}, 0

    # === MÉTODOS DE FALLBACK PARA BD ===

    def _fetch_codes_to_names_from_db(self, db: Session, codes: Optional[List[str]]) -> Dict[str, str]:
        """
        Fallback: obtener codes_to_names directamente desde BD
        FASE 2.1: Refactorizado para usar LaboratoryQueryBuilder (elimina duplicación SQL)
        """
        start_time = time.time()
        try:
            # FASE 2.1: Usar query builder en lugar de SQL duplicado
            query, params = laboratory_query_builder.build_codes_to_names_query(codes)

            # Validar parámetros antes de ejecutar
            if not laboratory_query_builder.validate_query_parameters(params):
                logger.error("[CACHE] Parámetros de query no válidos en codes_to_names")
                return {}

            # Ejecutar query optimizada
            result = db.execute(query, params).fetchall()
            mapping = {}

            for row in result:
                code = row.nomen_codigo_laboratorio
                name = row.nomen_laboratorio
                if code and name and len(code) <= 4 and code.isdigit():
                    mapping[code] = name

            # Log performance de la query
            execution_time = time.time() - start_time
            laboratory_query_builder.log_query_execution("codes_to_names", params, execution_time)

            return mapping

        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"[CACHE] Error en fallback codes_to_names: {str(e)[:200]}")
            laboratory_query_builder.log_query_execution("codes_to_names_ERROR", {}, execution_time)
            return {}

    def _fetch_names_to_codes_from_db(self, db: Session, names: Optional[List[str]]) -> Dict[str, str]:
        """
        Fallback: obtener names_to_codes directamente desde BD
        FASE 2.1: Refactorizado para usar LaboratoryQueryBuilder (elimina duplicación SQL)
        """
        start_time = time.time()
        try:
            # FASE 2.1: Usar query builder en lugar de SQL duplicado
            query, params = laboratory_query_builder.build_names_to_codes_query(names)

            # Validar parámetros antes de ejecutar
            if not laboratory_query_builder.validate_query_parameters(params):
                logger.error("[CACHE] Parámetros de query no válidos en names_to_codes")
                return {}

            # Ejecutar query optimizada
            result = db.execute(query, params).fetchall()
            mapping = {}

            for row in result:
                code = row.nomen_codigo_laboratorio
                name = row.nomen_laboratorio
                if code and name and len(code) <= 4 and code.isdigit():
                    mapping[name] = code

            # Log performance de la query
            execution_time = time.time() - start_time
            laboratory_query_builder.log_query_execution("names_to_codes", params, execution_time)

            return mapping

        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"[CACHE] Error en fallback names_to_codes: {str(e)[:200]}")
            laboratory_query_builder.log_query_execution("names_to_codes_ERROR", {}, execution_time)
            return {}

    def _fetch_generic_laboratories_from_db(
        self, db: Session, page: int, per_page: int, search: Optional[str]
    ) -> Dict[str, str]:
        """
        Fallback: obtener generic_laboratories directamente desde BD
        FASE 2.1: Refactorizado para usar LaboratoryQueryBuilder (elimina duplicación SQL compleja)
        """
        start_time = time.time()
        try:
            # FASE 2.1: Usar query builder en lugar de SQL complejo duplicado
            query, params = laboratory_query_builder.build_generic_laboratories_query(page, per_page, search)

            # Validar parámetros antes de ejecutar
            if not laboratory_query_builder.validate_query_parameters(params):
                logger.error("[CACHE] Parámetros de query no válidos en generic_laboratories")
                return {}

            # Ejecutar query optimizada
            result = db.execute(query, params).fetchall()
            mapping = {}

            for row in result:
                code = row.nomen_codigo_laboratorio
                name = row.nomen_laboratorio
                if code and name and len(code) <= 4 and code.isdigit():
                    mapping[code] = name

            # Log performance de la query (especialmente importante para paginación)
            execution_time = time.time() - start_time
            laboratory_query_builder.log_query_execution(f"generic_laboratories_page_{page}", params, execution_time)

            return mapping

        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"[CACHE] Error en fallback generic_laboratories: {str(e)[:200]}")
            laboratory_query_builder.log_query_execution(f"generic_laboratories_ERROR_page_{page}", {}, execution_time)
            return {}

    # === MÉTODOS DE INVALIDACIÓN Y WARMING ===

    def invalidate_all_laboratory_cache(self) -> int:
        """Invalidar todo el cache de laboratory mappings"""
        try:
            total_invalidated = 0

            # Invalidar todos los patterns de laboratory cache
            patterns = ["lab:codes:*", "lab:names:*", "lab:generics:*"]

            for pattern in patterns:
                count = self._invalidate_pattern(pattern)
                total_invalidated += count

            logger.info(f"[CACHE] Cache laboratory invalidado: {total_invalidated} keys")
            return total_invalidated

        except Exception as e:
            logger.error(f"[CACHE] Error invalidando cache laboratory: {str(e)[:100]}")
            return 0

    def warm_cache_on_startup(self, db: Session) -> bool:
        """Calentar cache durante startup de la aplicación"""
        try:
            if not is_cache_available():
                logger.warning("[CACHE] Redis no disponible, saltando cache warming")
                return False

            logger.info("[CACHE] Iniciando cache warming para laboratory mappings...")
            start_time = time.time()

            # 1. Warm mapeo completo codes → names
            logger.info("[CACHE] Warming codes_to_names...")
            codes_mapping, _ = self.get_codes_to_names_cached(db)
            codes_warmed = len(codes_mapping)

            # 2. Warm mapeo completo names → codes
            logger.info("[CACHE] Warming names_to_codes...")
            names_mapping, _ = self.get_names_to_codes_cached(db)
            names_warmed = len(names_mapping)

            # 3. Warm primeras páginas de laboratorios genéricos
            logger.info("[CACHE] Warming generic_laboratories...")
            generics_warmed = 0
            for page in range(1, 4):  # Primeras 3 páginas
                generic_mapping, _ = self.get_generic_laboratories_cached(db, page=page, per_page=50)
                generics_warmed += len(generic_mapping)

            # 4. Marcar warming como completado
            redis_client = get_redis()
            if redis_client:
                warming_info = {
                    "completed_at": utc_now().isoformat(),
                    "codes_warmed": codes_warmed,
                    "names_warmed": names_warmed,
                    "generics_warmed": generics_warmed,
                    "warming_time_seconds": time.time() - start_time,
                }
                redis_client.setex(
                    CacheConfig.KEY_WARMING_STATUS, CacheConfig.TTL_CACHE_STATS, CacheSerializer.serialize(warming_info)
                )

            total_time = time.time() - start_time
            logger.info(
                f"[CACHE] Cache warming completado en {total_time:.2f}s: "
                f"codes={codes_warmed}, names={names_warmed}, generics={generics_warmed}"
            )
            return True

        except Exception as e:
            logger.error(f"[CACHE] Error en cache warming: {str(e)[:200]}")
            return False

    def get_cache_statistics(self) -> Dict[str, Any]:
        """Obtener estadísticas detalladas del cache"""
        stats_dict = self._stats.to_dict()

        # Agregar estadísticas de Redis si está disponible
        try:
            redis_client = get_redis()
            if redis_client:
                info = redis_client.info()
                stats_dict.update(
                    {
                        "redis_connected_clients": info.get("connected_clients", 0),
                        "redis_used_memory_human": info.get("used_memory_human", "N/A"),
                        "redis_total_commands_processed": info.get("total_commands_processed", 0),
                        "redis_keyspace_hits": info.get("keyspace_hits", 0),
                        "redis_keyspace_misses": info.get("keyspace_misses", 0),
                    }
                )

                # Calcular hit rate de Redis
                redis_hits = info.get("keyspace_hits", 0)
                redis_misses = info.get("keyspace_misses", 0)
                redis_total = redis_hits + redis_misses
                if redis_total > 0:
                    stats_dict["redis_hit_rate_percent"] = round(redis_hits / redis_total * 100, 2)

        except Exception as e:
            logger.error(f"[CACHE] Error obteniendo stats Redis: {str(e)[:100]}")

        return stats_dict


# Instancia global del servicio de cache
laboratory_cache_service = LaboratoryCacheService()
