﻿# backend/app/services/invitation_service.py
"""
Servicio para gestión de invitaciones de usuarios
"""

import logging
from datetime import timedelta
from typing import Dict, List, Optional
from uuid import UUID, uuid4

from sqlalchemy.orm import Session

from app.core.security import get_password_hash
from app.models.invitation import Invitation
from app.models.pharmacy import Pharmacy
from app.models.user import User
from app.utils.datetime_utils import utc_now
from app.utils.validators import sanitize_dni_nie

logger = logging.getLogger(__name__)


class InvitationService:
    """Servicio para gestión de invitaciones"""

    @staticmethod
    def create_invitation(
        db: Session,
        pharmacy_id: UUID,
        invited_by: User,
        email: str,
        role: str = "user",
        full_name: Optional[str] = None,
        dni_nie: Optional[str] = None,
        message: Optional[str] = None,
        expires_in_days: int = 7,
    ) -> Dict:
        """
        Crea una nueva invitación

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            invited_by: Usuario que envía la invitación
            email: Email del invitado
            role: Rol asignado al nuevo usuario
            full_name: Nombre completo del invitado
            dni_nie: DNI/NIE del invitado
            message: Mensaje personalizado
            expires_in_days: Días hasta que expire la invitación

        Returns:
            Diccionario con los datos de la invitación
        """
        try:
            # Validar que el rol es válido (solo admin o user)
            if role not in ["admin", "user"]:
                raise ValueError(f"Rol inválido: {role}. Solo se permiten 'admin' o 'user'")

            # Verificar permisos del invitador (solo admin puede invitar)
            if not invited_by.is_superuser and invited_by.role != "admin":
                raise ValueError("Solo los administradores pueden enviar invitaciones")

            # Verificar que el invitador pertenece a la farmacia (si no es admin)
            if not invited_by.is_superuser and str(invited_by.pharmacy_id) != str(pharmacy_id):
                raise ValueError("Solo puedes invitar usuarios a tu propia farmacia")

            # Verificar que la farmacia existe
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
            if not pharmacy:
                raise ValueError("Farmacia no encontrada")

            # Verificar si ya existe un usuario con ese email
            existing_user = db.query(User).filter(User.email == email).first()
            if existing_user:
                raise ValueError("Ya existe un usuario con ese email")

            # Verificar si ya existe una invitación pendiente
            existing_invitation = (
                db.query(Invitation)
                .filter(
                    Invitation.email == email,
                    Invitation.status == "pending",
                    Invitation.pharmacy_id == pharmacy_id,
                )
                .first()
            )

            if existing_invitation:
                # Actualizar invitación existente
                existing_invitation.invited_by_id = invited_by.id
                existing_invitation.role = role
                existing_invitation.full_name = full_name
                existing_invitation.dni_nie = sanitize_dni_nie(dni_nie) if dni_nie else None
                existing_invitation.message = message
                existing_invitation.token = str(uuid4())
                existing_invitation.expires_at = utc_now() + timedelta(days=expires_in_days)
                existing_invitation.updated_at = utc_now()

                db.commit()
                logger.info(f"Invitación actualizada para {email}")
                return existing_invitation.to_dict()

            # Crear nueva invitación
            invitation = Invitation(
                pharmacy_id=pharmacy_id,
                invited_by_id=invited_by.id,
                email=email,
                role=role,
                full_name=full_name,
                dni_nie=sanitize_dni_nie(dni_nie) if dni_nie else None,
                message=message,
                token=str(uuid4()),
                expires_at=utc_now() + timedelta(days=expires_in_days),
                status="pending",
                max_uses="1",
                times_used="0",
            )

            db.add(invitation)
            db.commit()

            logger.info(f"Invitación creada para {email} por {invited_by.email}")

            # TODO: Enviar email con la invitación cuando el servicio esté activo
            # Por ahora, retornar el token para testing
            result = invitation.to_dict()
            result["invitation_url"] = f"/auth/accept-invitation?token={invitation.token}"

            return result

        except Exception as e:
            logger.error(f"Error creando invitación: {e}")
            db.rollback()
            raise

    @staticmethod
    def accept_invitation(
        db: Session,
        token: str,
        password: str,
        username: Optional[str] = None,
    ) -> Dict:
        """
        Acepta una invitación y crea el usuario

        Args:
            db: Sesión de base de datos
            token: Token de la invitación
            password: Contraseña para el nuevo usuario
            username: Username opcional (se genera si no se proporciona)

        Returns:
            Diccionario con los datos del usuario creado
        """
        try:
            # Buscar invitación
            invitation = db.query(Invitation).filter(Invitation.token == token).first()

            if not invitation:
                raise ValueError("Invitación no válida")

            # Validar invitación
            if not invitation.is_valid():
                raise ValueError("La invitación ha expirado o ya fue utilizada")

            # Verificar si ya existe un usuario con ese email
            existing_user = db.query(User).filter(User.email == invitation.email).first()
            if existing_user:
                raise ValueError("Ya existe un usuario con ese email")

            # Generar username si no se proporciona
            if not username:
                username = invitation.email.split("@")[0].lower()
                # Asegurar unicidad
                base_username = username
                counter = 1
                while db.query(User).filter(User.username == username).first():
                    username = f"{base_username}{counter}"
                    counter += 1

            # Crear usuario
            new_user = User(
                email=invitation.email,
                username=username,
                hashed_password=get_password_hash(password),
                full_name=invitation.full_name,
                dni_nie=invitation.dni_nie,
                pharmacy_id=invitation.pharmacy_id,
                role=invitation.role,
                is_active=True,
                is_verified=True,  # Consideramos el email verificado al aceptar invitación
                email_verified_at=utc_now(),
            )

            db.add(new_user)

            # Marcar invitación como aceptada
            invitation.accept(new_user.id)

            db.commit()

            logger.info(f"Usuario {new_user.email} creado desde invitación")

            return {
                "user": new_user.to_dict(),
                "message": "Usuario creado exitosamente",
            }

        except Exception as e:
            logger.error(f"Error aceptando invitación: {e}")
            db.rollback()
            raise

    @staticmethod
    def list_invitations(
        db: Session,
        pharmacy_id: UUID,
        status: Optional[str] = None,
        limit: int = 100,
    ) -> List[Dict]:
        """
        Lista las invitaciones de una farmacia

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            status: Filtrar por estado (opcional)
            limit: Límite de resultados

        Returns:
            Lista de invitaciones
        """
        try:
            query = db.query(Invitation).filter(Invitation.pharmacy_id == pharmacy_id)

            if status:
                query = query.filter(Invitation.status == status)

            invitations = query.order_by(Invitation.created_at.desc()).limit(limit).all()

            return [inv.to_dict() for inv in invitations]

        except Exception as e:
            logger.error(f"Error listando invitaciones: {e}")
            raise

    @staticmethod
    def cancel_invitation(db: Session, invitation_id: UUID, user: User) -> Dict:
        """
        Cancela una invitación

        Args:
            db: Sesión de base de datos
            invitation_id: ID de la invitación
            user: Usuario que cancela la invitación

        Returns:
            Diccionario con el resultado
        """
        try:
            invitation = db.query(Invitation).filter(Invitation.id == invitation_id).first()

            if not invitation:
                raise ValueError("Invitación no encontrada")

            # Verificar permisos
            if not user.is_superuser:
                if not user.has_permission("manage_users"):
                    raise ValueError("No tienes permisos para cancelar invitaciones")

                if str(user.pharmacy_id) != str(invitation.pharmacy_id):
                    raise ValueError("No puedes cancelar invitaciones de otras farmacias")

            # Cancelar invitación
            invitation.cancel()
            db.commit()

            logger.info(f"Invitación {invitation_id} cancelada por {user.email}")

            return {"message": "Invitación cancelada exitosamente"}

        except Exception as e:
            logger.error(f"Error cancelando invitación: {e}")
            db.rollback()
            raise

    @staticmethod
    def validate_token(db: Session, token: str) -> Dict:
        """
        Valida un token de invitación

        Args:
            db: Sesión de base de datos
            token: Token a validar

        Returns:
            Información de la invitación si es válida
        """
        try:
            invitation = db.query(Invitation).filter(Invitation.token == token).first()

            if not invitation:
                return {"valid": False, "error": "Token no válido"}

            if not invitation.is_valid():
                return {"valid": False, "error": "Invitación expirada o ya utilizada"}

            # Obtener información de la farmacia
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == invitation.pharmacy_id).first()

            return {
                "valid": True,
                "email": invitation.email,
                "pharmacy_name": pharmacy.name if pharmacy else "Farmacia",
                "role": invitation.role,
                "full_name": invitation.full_name,
                "message": invitation.message,
            }

        except Exception as e:
            logger.error(f"Error validando token: {e}")
            return {"valid": False, "error": "Error validando invitación"}
