# backend/app/services/intercambiable_group_service.py
"""
Servicio para gestión de grupos intercambiables.

Issue #446: Taxonomía jerárquica de venta libre
ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

Este servicio gestiona:
- CRUD de IntercambiableGroup
- Asignación de productos a grupos (por necesidad/embeddings)
- Actualización de estadísticas
"""

import uuid
from typing import List, Optional, Dict, Any

from sqlalchemy import func, update
from sqlalchemy.orm import Session

import structlog

from app.models.intercambiable_group import IntercambiableGroup
from app.models.sales_enrichment import SalesEnrichment
from app.models.sales_data import SalesData

logger = structlog.get_logger(__name__)


class IntercambiableGroupService:
    """
    Servicio para gestión de grupos intercambiables.

    Los grupos intercambiables agrupan productos de múltiples marcas
    que pueden sustituirse entre sí (mismo nivel 4 de taxonomía).
    """

    def __init__(self, session: Session):
        self.session = session

    # =========================================================================
    # CRUD BÁSICO
    # =========================================================================

    def get_all(
        self,
        validated_only: bool = False,
        necesidad: Optional[str] = None,
        limit: int = 100,
        offset: int = 0
    ) -> List[IntercambiableGroup]:
        """
        Obtiene todos los grupos, opcionalmente filtrados.

        Args:
            validated_only: Solo grupos validados por farmacéutico
            necesidad: Filtrar por necesidad específica
            limit: Máximo de grupos a retornar (default 100)
            offset: Offset para paginación

        Returns:
            Lista de IntercambiableGroup
        """
        query = self.session.query(IntercambiableGroup)

        if validated_only:
            query = query.filter(IntercambiableGroup.validated.is_(True))

        if necesidad:
            query = query.filter(IntercambiableGroup.necesidad == necesidad)

        return query.order_by(IntercambiableGroup.necesidad).offset(offset).limit(limit).all()

    def get_by_id(self, group_id: uuid.UUID) -> Optional[IntercambiableGroup]:
        """Obtiene un grupo por su ID."""
        return self.session.query(IntercambiableGroup).filter(
            IntercambiableGroup.id == group_id
        ).first()

    def get_by_slug(self, slug: str) -> Optional[IntercambiableGroup]:
        """Obtiene un grupo por su slug."""
        return self.session.query(IntercambiableGroup).filter(
            IntercambiableGroup.slug == slug
        ).first()

    def get_by_necesidad(self, necesidad: str) -> List[IntercambiableGroup]:
        """Obtiene todos los grupos de una necesidad."""
        return self.session.query(IntercambiableGroup).filter(
            IntercambiableGroup.necesidad == necesidad
        ).all()

    # =========================================================================
    # ASIGNACIÓN DE PRODUCTOS
    # =========================================================================

    def assign_products_by_necesidad(
        self,
        group_id: uuid.UUID,
        pharmacy_id: Optional[uuid.UUID] = None,
        limit: int = 1000,
        auto_commit: bool = True
    ) -> Dict[str, Any]:
        """
        Asigna productos a un grupo basándose en ml_category (necesidad).

        Este es el método más simple de asignación: productos con la misma
        necesidad que el grupo se asignan automáticamente.

        Args:
            group_id: ID del grupo
            pharmacy_id: Opcional, limitar a una farmacia
            limit: Máximo de productos a asignar
            auto_commit: Si True, hace commit automático (default True)

        Returns:
            Dict con estadísticas de asignación
        """
        group = self.get_by_id(group_id)
        if not group:
            raise ValueError(f"Grupo {group_id} no encontrado")

        # Query base: obtener IDs de productos a asignar
        query = (
            self.session.query(SalesEnrichment.id)
            .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
            .filter(
                SalesEnrichment.product_type == 'venta_libre',
                SalesEnrichment.ml_category == group.necesidad,
                SalesEnrichment.intercambiable_group_id.is_(None)
            )
        )

        # Filtro opcional por subcategoría si el grupo la tiene
        if group.subcategory:
            query = query.filter(
                SalesEnrichment.ml_subcategory == group.subcategory
            )

        # Filtro opcional por farmacia
        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Obtener IDs a actualizar (más eficiente que cargar objetos completos)
        enrichment_ids = [r[0] for r in query.limit(limit).all()]

        # Bulk update - una sola query SQL
        assigned_count = 0
        if enrichment_ids:
            stmt = (
                update(SalesEnrichment)
                .where(SalesEnrichment.id.in_(enrichment_ids))
                .values(intercambiable_group_id=group_id)
            )
            result = self.session.execute(stmt)
            assigned_count = result.rowcount

        if auto_commit:
            self.session.commit()
            # Actualizar estadísticas del grupo solo si auto_commit
            self.update_group_stats(group_id, auto_commit=True)

        logger.info(
            "products_assigned_to_group",
            group_id=str(group_id),
            group_name=group.name,
            necesidad=group.necesidad,
            assigned=assigned_count
        )

        return {
            "group_id": str(group_id),
            "group_name": group.name,
            "necesidad": group.necesidad,
            "assigned_count": assigned_count,
        }

    def assign_all_groups(
        self,
        pharmacy_id: Optional[uuid.UUID] = None,
        validated_only: bool = True
    ) -> Dict[str, Any]:
        """
        Asigna productos a TODOS los grupos intercambiables.

        Optimizado para batch operations: un solo commit para todas
        las asignaciones, evitando N+1 queries y timeouts en Render.

        Args:
            pharmacy_id: Opcional, limitar a una farmacia
            validated_only: Solo usar grupos validados

        Returns:
            Dict con estadísticas de asignación por grupo
        """
        groups = self.get_all(validated_only=validated_only, limit=500)
        results = []
        total_assigned = 0

        # Fase 1: Asignar productos sin commit individual
        for group in groups:
            result = self.assign_products_by_necesidad(
                group_id=group.id,
                pharmacy_id=pharmacy_id,
                auto_commit=False  # Diferir commit
            )
            results.append(result)
            total_assigned += result["assigned_count"]

        # Fase 2: Un solo commit para todas las asignaciones
        self.session.commit()

        # Fase 3: Actualizar estadísticas de todos los grupos
        for group in groups:
            self.update_group_stats(group.id, auto_commit=False)

        # Fase 4: Commit final de estadísticas
        self.session.commit()

        logger.info(
            "all_groups_assigned",
            groups_processed=len(groups),
            total_assigned=total_assigned
        )

        return {
            "groups_processed": len(groups),
            "total_assigned": total_assigned,
            "details": results
        }

    # =========================================================================
    # ESTADÍSTICAS
    # =========================================================================

    def update_group_stats(
        self,
        group_id: uuid.UUID,
        auto_commit: bool = True
    ) -> IntercambiableGroup:
        """
        Actualiza las estadísticas denormalizadas de un grupo.

        Calcula:
        - product_count: productos únicos
        - brand_count: marcas diferentes
        - total_sales_amount: suma de ventas
        - total_sales_count: número de registros

        Args:
            group_id: ID del grupo
            auto_commit: Si True, hace commit automático (default True)
        """
        group = self.get_by_id(group_id)
        if not group:
            raise ValueError(f"Grupo {group_id} no encontrado")

        stats = (
            self.session.query(
                func.count(func.distinct(SalesData.product_name)).label("products"),
                func.count(func.distinct(SalesEnrichment.detected_brand)).label("brands"),
                func.coalesce(func.sum(SalesData.total_amount), 0).label("amount"),
                func.count(SalesData.id).label("count"),
            )
            .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
            .filter(SalesEnrichment.intercambiable_group_id == group_id)
            .first()
        )

        if stats:
            group.product_count = stats.products or 0
            group.brand_count = stats.brands or 0
            group.total_sales_amount = stats.amount or 0
            group.total_sales_count = stats.count or 0

        if auto_commit:
            self.session.commit()

        logger.info(
            "group_stats_updated",
            group_id=str(group_id),
            products=group.product_count,
            brands=group.brand_count,
            sales=float(group.total_sales_amount)
        )

        return group

    def update_all_stats(self) -> List[Dict]:
        """Actualiza estadísticas de todos los grupos."""
        groups = self.get_all()
        results = []

        for group in groups:
            updated = self.update_group_stats(group.id)
            results.append({
                "id": str(updated.id),
                "name": updated.name,
                "product_count": updated.product_count,
                "brand_count": updated.brand_count,
                "total_sales_amount": float(updated.total_sales_amount),
            })

        return results

    # =========================================================================
    # CONSULTAS
    # =========================================================================

    def get_group_products(
        self,
        group_id: uuid.UUID,
        limit: int = 100
    ) -> List[Dict]:
        """
        Obtiene los productos de un grupo con detalles.

        Returns:
            Lista de productos con nombre, marca, ventas
        """
        results = (
            self.session.query(
                SalesData.product_name,
                SalesEnrichment.detected_brand,
                SalesEnrichment.brand_line,
                func.sum(SalesData.total_amount).label("total_ventas"),
                func.count(SalesData.id).label("num_ventas")
            )
            .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
            .filter(SalesEnrichment.intercambiable_group_id == group_id)
            .group_by(
                SalesData.product_name,
                SalesEnrichment.detected_brand,
                SalesEnrichment.brand_line
            )
            .order_by(func.sum(SalesData.total_amount).desc())
            .limit(limit)
            .all()
        )

        return [
            {
                "product_name": r.product_name,
                "brand": r.detected_brand,
                "brand_line": r.brand_line,
                "total_ventas": float(r.total_ventas) if r.total_ventas else 0,
                "num_ventas": r.num_ventas
            }
            for r in results
        ]

    def get_summary(self) -> Dict[str, Any]:
        """
        Obtiene resumen general de grupos intercambiables.

        Returns:
            Dict con totales y estadísticas
        """
        groups = self.get_all()

        total_products = sum(g.product_count for g in groups)
        total_sales = sum(float(g.total_sales_amount or 0) for g in groups)
        validated_count = sum(1 for g in groups if g.validated)

        # Distribución por necesidad
        by_necesidad = {}
        for g in groups:
            if g.necesidad not in by_necesidad:
                by_necesidad[g.necesidad] = {"count": 0, "products": 0}
            by_necesidad[g.necesidad]["count"] += 1
            by_necesidad[g.necesidad]["products"] += g.product_count

        return {
            "total_groups": len(groups),
            "validated_groups": validated_count,
            "total_products_in_groups": total_products,
            "total_sales_in_groups": total_sales,
            "by_necesidad": by_necesidad
        }
