# backend/app/services/insight_engine/rules/trend_rules.py
"""
Reglas de Tendencia para Insight Engine v2.0.

Issue #506: Regla de caídas sostenidas con impacto económico.
Issue #519: Conectado con subcategorías L2 para insights más granulares.

TREND_001: Caída Sostenida - Categorías/subcategorías con caída YoY durante meses consecutivos
"""

from datetime import date
from typing import Any, Optional
from uuid import UUID

import structlog
from sqlalchemy import extract, func
from sqlalchemy.orm import Session

from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from .base import BaseInsightRule
from ..models import InsightResult

logger = structlog.get_logger(__name__)


class CaidaSostenidaRule(BaseInsightRule):
    """
    TREND_001: Caída Sostenida - Categorías/subcategorías con caída YoY consecutiva.

    Detecta categorías donde las ventas han caído más del X% YoY durante
    N meses consecutivos. Indica tendencias preocupantes que requieren acción.

    Issue #519: Cuando use_subcategory=True, analiza a nivel L2 (subcategoría)
    para dar insights más específicos (ej: "Caída en Champús Anticaspa").

    Impacto económico: Ventas perdidas vs año anterior.
    """

    rule_code = "TREND_001"
    category = "trend"
    severity = "high"
    default_config = {
        "yoy_decline_threshold": -15.0,  # % de caída para considerar
        "consecutive_months": 3,  # Meses consecutivos en caída
        "min_monthly_revenue": 200.0,  # € mínimo mensual para considerar
        "max_items_to_show": 10,  # Límite de categorías
        "use_subcategory": True,  # Issue #519: Usar L2 cuando disponible
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa categorías/subcategorías con caídas sostenidas."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            decline_threshold = cfg["yoy_decline_threshold"]
            min_consecutive = cfg["consecutive_months"]
            min_revenue = cfg["min_monthly_revenue"]
            max_items = cfg["max_items_to_show"]
            use_subcategory = cfg.get("use_subcategory", True)

            today = date.today()
            current_year = today.year
            previous_year = current_year - 1

            # Analizar últimos 6 meses
            months_to_analyze = 6
            end_month = today.month
            start_month = max(1, end_month - months_to_analyze + 1)

            # 1. Obtener ventas mensuales por categoría/subcategoría - Año actual
            # Issue #519: Incluir ml_subcategory para análisis L2
            current_sales = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    extract("month", SalesData.sale_date).label("month"),
                    func.sum(SalesData.total_amount).label("revenue"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    extract("year", SalesData.sale_date) == current_year,
                    extract("month", SalesData.sale_date) >= start_month,
                    extract("month", SalesData.sale_date) <= end_month,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesEnrichment.ml_category.isnot(None),
                )
                .group_by(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    extract("month", SalesData.sale_date),
                )
                .all()
            )

            # 2. Obtener ventas mensuales - Año anterior
            previous_sales = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    extract("month", SalesData.sale_date).label("month"),
                    func.sum(SalesData.total_amount).label("revenue"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    extract("year", SalesData.sale_date) == previous_year,
                    extract("month", SalesData.sale_date) >= start_month,
                    extract("month", SalesData.sale_date) <= end_month,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesEnrichment.ml_category.isnot(None),
                )
                .group_by(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    extract("month", SalesData.sale_date),
                )
                .all()
            )

            if not current_sales or not previous_sales:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 3. Organizar datos en estructura con soporte L2
            # Issue #519: Usar L2 cuando disponible
            # {cat_key: {"months": {month: revenue}, "display_name": str, "parent": str|None}}
            current_dict: dict[str, dict] = {}
            for row in current_sales:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory
                month = int(row.month)
                rev = float(row.revenue or 0)

                # Determinar clave de agrupación
                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat
                    parent = l1_cat
                else:
                    cat_key = l1_cat
                    parent = None

                if cat_key not in current_dict:
                    current_dict[cat_key] = {
                        "months": {},
                        "display_name": cat_key,
                        "parent": parent,
                        "is_subcategory": bool(use_subcategory and l2_subcat),
                    }
                current_dict[cat_key]["months"][month] = (
                    current_dict[cat_key]["months"].get(month, 0) + rev
                )

            previous_dict: dict[str, dict] = {}
            for row in previous_sales:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory
                month = int(row.month)
                rev = float(row.revenue or 0)

                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat
                    parent = l1_cat
                else:
                    cat_key = l1_cat
                    parent = None

                if cat_key not in previous_dict:
                    previous_dict[cat_key] = {
                        "months": {},
                        "display_name": cat_key,
                        "parent": parent,
                        "is_subcategory": bool(use_subcategory and l2_subcat),
                    }
                previous_dict[cat_key]["months"][month] = (
                    previous_dict[cat_key]["months"].get(month, 0) + rev
                )

            # 4. Calcular YoY por mes y detectar caídas consecutivas
            declining_categories = []
            has_l2_insights = False

            for cat_key, cat_data in current_dict.items():
                prev_data = previous_dict.get(cat_key)
                current_months = cat_data["months"]
                prev_months = prev_data["months"] if prev_data else {}

                if not prev_months:
                    continue

                # Calcular variación YoY por mes
                monthly_changes = []
                for month in range(start_month, end_month + 1):
                    curr_rev = current_months.get(month, 0)
                    prev_rev = prev_months.get(month, 0)

                    if prev_rev < min_revenue:
                        continue

                    yoy_change = ((curr_rev - prev_rev) / prev_rev) * 100
                    monthly_changes.append({
                        "month": month,
                        "current": curr_rev,
                        "previous": prev_rev,
                        "yoy_pct": yoy_change,
                    })

                if len(monthly_changes) < min_consecutive:
                    continue

                # Detectar rachas de caídas consecutivas
                consecutive_declines = 0
                max_consecutive = 0
                decline_total = 0.0

                for mc in monthly_changes[-min_consecutive:]:  # Últimos N meses
                    if mc["yoy_pct"] <= decline_threshold:
                        consecutive_declines += 1
                        decline_total += mc["previous"] - mc["current"]
                        max_consecutive = max(max_consecutive, consecutive_declines)
                    else:
                        consecutive_declines = 0

                if max_consecutive >= min_consecutive:
                    # Calcular ventas perdidas totales
                    total_current = sum(m["current"] for m in monthly_changes)
                    total_previous = sum(m["previous"] for m in monthly_changes)
                    total_lost = max(0, total_previous - total_current)

                    # YoY promedio
                    avg_yoy = sum(m["yoy_pct"] for m in monthly_changes) / len(monthly_changes)

                    # Issue #519: Incluir info de L2
                    item = {
                        "category": cat_data["display_name"],
                        "consecutive_decline_months": max_consecutive,
                        "avg_yoy_change_pct": round(avg_yoy, 1),
                        "current_period_revenue": round(total_current, 2),
                        "previous_period_revenue": round(total_previous, 2),
                        "revenue_lost": round(total_lost, 2),
                        "monthly_data": monthly_changes[-3:],  # Últimos 3 meses
                    }

                    # Añadir parent_category si es subcategoría L2
                    if cat_data["is_subcategory"] and cat_data["parent"]:
                        item["parent_category"] = cat_data["parent"]
                        has_l2_insights = True

                    declining_categories.append(item)

            if not declining_categories:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 5. Ordenar por ventas perdidas y limitar
            declining_categories.sort(key=lambda x: -x["revenue_lost"])
            top_categories = declining_categories[:max_items]

            # 6. Calcular impacto total
            total_lost = sum(c["revenue_lost"] for c in declining_categories)
            monthly_lost = total_lost / months_to_analyze

            # 7. Calcular impact_score
            # Formula: count_score (0-40) + value_score (0-60) = 0-100
            # Divisor /50: sustained decline typically 500-3000€/month lost → 60 points at 3000€
            count_score = min(40, len(declining_categories) * 10)
            value_score = min(60, monthly_lost / 50)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(declining_categories),
                economic_value=monthly_lost,
            )

            # Issue #519: Ajustar descripción según granularidad L1/L2
            if has_l2_insights:
                level_term = "subcategorías"
                description = (
                    f"Hay {len(declining_categories)} subcategorías con caída de ventas superior "
                    f"al {abs(decline_threshold):.0f}% durante {min_consecutive}+ meses consecutivos. "
                    f"Revisa surtido, precios y competencia local. Análisis detallado por subcategoría L2."
                )
            else:
                level_term = "categorías"
                description = (
                    f"Hay {len(declining_categories)} categorías con caída de ventas superior "
                    f"al {abs(decline_threshold):.0f}% durante {min_consecutive}+ meses consecutivos. "
                    f"Revisa surtido, precios y competencia local."
                )

            return self.create_insight(
                title=f"Caída Sostenida: {len(declining_categories)} {level_term}",
                description=description,
                impact_score=impact_score,
                economic_impact=f"Ventas perdidas: {monthly_lost:,.0f}€/mes",
                economic_value=monthly_lost,
                action_label="Analizar tendencias",
                deeplink="/ventalibre/tendencias?filter=declining",
                affected_items=top_categories,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None
