# backend/app/services/insight_engine/rules/surtido_rules.py
"""
Reglas de Surtido para Insight Engine v2.0.

Issue #506: 3 reglas de surtido con impacto económico.
Issue #519: SURTIDO_001 y SURTIDO_003 conectados con subcategorías L2.

SURTIDO_001: Desequilibrio - Categoría/subcategoría con muchas ventas pero pocos SKUs
SURTIDO_002: Hueco de Surtido - Mercado grande pero farmacia infrarrepresentada
SURTIDO_003: Canibalización - Productos propios que se quitan ventas entre sí
"""

from datetime import date, timedelta
from typing import Any, Optional
from uuid import UUID

import structlog
from sqlalchemy import distinct, func
from sqlalchemy.orm import Session

from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.models.product_catalog_venta_libre import ProductCatalogVentaLibre

from .base import BaseInsightRule
from ..models import InsightResult

logger = structlog.get_logger(__name__)


class DesequilibrioRule(BaseInsightRule):
    """
    SURTIDO_001: Desequilibrio - Categorías/subcategorías con alta demanda pero poco surtido.

    Detecta categorías que representan un % significativo de las ventas
    pero tienen muy pocos SKUs diferentes. Indica potencial de expansión.

    Issue #519: Cuando use_subcategory=True, analiza a nivel L2 (subcategoría).

    Impacto económico: Estimación de ventas adicionales al ampliar surtido.
    """

    rule_code = "SURTIDO_001"
    category = "surtido"
    severity = "medium"
    default_config = {
        "min_sales_pct": 5.0,  # % mínimo de ventas para considerar
        "min_skus": 3,  # Mínimo de SKUs para NO alertar
        "lookback_days": 90,  # Días de ventas a analizar
        "expansion_factor": 0.15,  # Factor de crecimiento esperado al ampliar
        "max_items_to_show": 10,
        "use_subcategory": True,  # Issue #519: Usar L2 cuando disponible
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa categorías/subcategorías con desequilibrio surtido/ventas."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            min_sales_pct = cfg["min_sales_pct"]
            min_skus = cfg["min_skus"]
            lookback = cfg["lookback_days"]
            expansion = cfg["expansion_factor"]
            max_items = cfg["max_items_to_show"]
            use_subcategory = cfg.get("use_subcategory", True)

            today = date.today()
            start_date = today - timedelta(days=lookback)

            # 1. Obtener ventas totales de venta libre
            total_vl_sales = (
                db.query(func.sum(SalesData.total_amount))
                .join(SalesEnrichment, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesEnrichment.product_type == "venta_libre",
                )
                .scalar()
            ) or 0

            if total_vl_sales == 0:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Obtener ventas y SKUs por categoría/subcategoría
            # Issue #519: Incluir ml_subcategory
            category_stats = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    func.sum(SalesData.total_amount).label("revenue"),
                    func.count(distinct(SalesData.codigo_nacional)).label("sku_count"),
                    func.sum(SalesData.quantity).label("units_sold"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesEnrichment.ml_category.isnot(None),
                )
                .group_by(SalesEnrichment.ml_category, SalesEnrichment.ml_subcategory)
                .all()
            )

            if not category_stats:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 3. Identificar categorías/subcategorías desequilibradas
            # Issue #519: Agrupar por L2 cuando disponible
            imbalanced = []
            has_l2_insights = False
            aggregated: dict[str, dict] = {}

            for row in category_stats:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory
                revenue = float(row.revenue or 0)
                sku_count = int(row.sku_count or 0)
                units_sold = int(row.units_sold or 0)

                # Determinar clave de agrupación
                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat
                    parent = l1_cat
                    is_subcat = True
                else:
                    cat_key = l1_cat
                    parent = None
                    is_subcat = False

                if cat_key not in aggregated:
                    aggregated[cat_key] = {
                        "revenue": 0.0,
                        "sku_count": 0,
                        "units_sold": 0,
                        "parent": parent,
                        "is_subcategory": is_subcat,
                    }

                aggregated[cat_key]["revenue"] += revenue
                aggregated[cat_key]["sku_count"] += sku_count
                aggregated[cat_key]["units_sold"] += units_sold

            for cat_key, data in aggregated.items():
                revenue = data["revenue"]
                sku_count = data["sku_count"]
                sales_pct = (revenue / total_vl_sales) * 100

                # Categoría con alta demanda pero pocos SKUs
                if sales_pct >= min_sales_pct and sku_count < min_skus:
                    # Potencial = ventas actuales * factor de expansión
                    potential = revenue * expansion

                    item = {
                        "category": cat_key,
                        "revenue": round(revenue, 2),
                        "sales_pct": round(sales_pct, 1),
                        "sku_count": sku_count,
                        "units_sold": data["units_sold"],
                        "expansion_potential": round(potential, 2),
                    }

                    # Añadir parent_category si es subcategoría L2
                    if data["is_subcategory"] and data["parent"]:
                        item["parent_category"] = data["parent"]
                        has_l2_insights = True

                    imbalanced.append(item)

            if not imbalanced:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 4. Ordenar por potencial y limitar
            imbalanced.sort(key=lambda x: -x["expansion_potential"])
            top_items = imbalanced[:max_items]

            # 5. Calcular impacto total
            total_potential = sum(c["expansion_potential"] for c in imbalanced)
            monthly_potential = total_potential / (lookback / 30)

            # 6. Calcular impact_score
            # Formula: count_score (0-40) + value_score (0-60) = 0-100
            # Divisor /50: assortment imbalance typically 500-3000€/month → 60 points at 3000€
            count_score = min(40, len(imbalanced) * 10)
            value_score = min(60, monthly_potential / 50)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(imbalanced),
                economic_value=monthly_potential,
            )

            # Issue #519: Ajustar descripción según granularidad L1/L2
            if has_l2_insights:
                level_term = "subcategorías"
                description = (
                    f"Hay {len(imbalanced)} subcategorías que generan más del {min_sales_pct}% de "
                    f"ventas pero tienen menos de {min_skus} productos diferentes. Ampliar el "
                    f"surtido podría capturar más demanda. Análisis detallado por subcategoría L2."
                )
            else:
                level_term = "categorías"
                description = (
                    f"Hay {len(imbalanced)} categorías que generan más del {min_sales_pct}% de "
                    f"ventas pero tienen menos de {min_skus} productos diferentes. Ampliar el "
                    f"surtido podría capturar más demanda."
                )

            return self.create_insight(
                title=f"Desequilibrio: {len(imbalanced)} {level_term} con poco surtido",
                description=description,
                impact_score=impact_score,
                economic_impact=f"Potencial: +{monthly_potential:,.0f}€/mes",
                economic_value=monthly_potential,
                action_label=f"Ver {level_term}",
                deeplink="/ventalibre/surtido?filter=imbalanced",
                affected_items=top_items,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None


class HuecoSurtidoRule(BaseInsightRule):
    """
    SURTIDO_002: Hueco de Surtido - Segmentos de mercado infrarrepresentados.

    Detecta segmentos donde el mercado es grande (>30% cuota en otras farmacias)
    pero la farmacia tiene poca presencia (<10% de SKUs del catálogo).

    Requiere datos del catálogo de venta libre para comparar.

    Impacto económico: Ventas potenciales del segmento no cubierto.
    """

    rule_code = "SURTIDO_002"
    category = "surtido"
    severity = "low"
    default_config = {
        "market_share_threshold": 30.0,  # % del mercado que representa el segmento
        "pharmacy_share_threshold": 10.0,  # % mínimo de SKUs que debería tener
        "min_catalog_products": 5,  # Mínimo de productos en catálogo para evaluar
        "lookback_days": 90,
        "max_items_to_show": 10,
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa huecos de surtido vs catálogo."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            market_threshold = cfg["market_share_threshold"]
            pharmacy_threshold = cfg["pharmacy_share_threshold"]
            min_catalog = cfg["min_catalog_products"]
            lookback = cfg["lookback_days"]
            max_items = cfg["max_items_to_show"]

            today = date.today()
            start_date = today - timedelta(days=lookback)

            # 1. Obtener productos del catálogo por categoría (NECESIDAD)
            catalog_by_category = (
                db.query(
                    ProductCatalogVentaLibre.necesidad,
                    func.count(ProductCatalogVentaLibre.id).label("catalog_count"),
                )
                .filter(
                    ProductCatalogVentaLibre.necesidad.isnot(None),
                    ProductCatalogVentaLibre.is_active.is_(True),
                )
                .group_by(ProductCatalogVentaLibre.necesidad)
                .having(func.count(ProductCatalogVentaLibre.id) >= min_catalog)
                .all()
            )

            if not catalog_by_category:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            catalog_dict = {
                row.necesidad: int(row.catalog_count)
                for row in catalog_by_category
            }

            # 2. Obtener productos vendidos por la farmacia por categoría
            pharmacy_products = (
                db.query(
                    SalesEnrichment.ml_category,
                    func.count(distinct(SalesData.codigo_nacional)).label("sku_count"),
                    func.sum(SalesData.total_amount).label("revenue"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesEnrichment.ml_category.isnot(None),
                )
                .group_by(SalesEnrichment.ml_category)
                .all()
            )

            pharmacy_dict = {
                row.ml_category: {
                    "sku_count": int(row.sku_count or 0),
                    "revenue": float(row.revenue or 0),
                }
                for row in pharmacy_products
            }

            # 3. Calcular total del catálogo
            total_catalog = sum(catalog_dict.values())

            # 4. Identificar huecos de surtido
            gaps = []

            for cat, catalog_count in catalog_dict.items():
                # % del catálogo que representa esta categoría
                market_pct = (catalog_count / total_catalog) * 100

                if market_pct < market_threshold:
                    continue  # Categoría pequeña, no relevante

                # SKUs de la farmacia en esta categoría
                pharmacy_data = pharmacy_dict.get(cat, {"sku_count": 0, "revenue": 0})
                pharmacy_skus = pharmacy_data["sku_count"]
                pharmacy_revenue = pharmacy_data["revenue"]

                # % de cobertura de la farmacia
                coverage_pct = (pharmacy_skus / catalog_count) * 100 if catalog_count > 0 else 0

                if coverage_pct < pharmacy_threshold:
                    # Hay un hueco significativo
                    # Potencial = productos no cubiertos * precio promedio estimado
                    uncovered = catalog_count - pharmacy_skus
                    # Estimación conservadora: 50€/producto/mes
                    potential = uncovered * 50 * (lookback / 30)

                    gaps.append({
                        "category": cat,
                        "catalog_products": catalog_count,
                        "pharmacy_products": pharmacy_skus,
                        "coverage_pct": round(coverage_pct, 1),
                        "market_share_pct": round(market_pct, 1),
                        "current_revenue": round(pharmacy_revenue, 2),
                        "uncovered_products": uncovered,
                        "potential_revenue": round(potential, 2),
                    })

            if not gaps:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 5. Ordenar por potencial y limitar
            gaps.sort(key=lambda x: -x["potential_revenue"])
            top_gaps = gaps[:max_items]

            # 6. Calcular impacto total
            total_potential = sum(g["potential_revenue"] for g in gaps)
            monthly_potential = total_potential / (lookback / 30)

            # 7. Calcular impact_score
            count_score = min(30, len(gaps) * 5)
            value_score = min(70, monthly_potential / 100)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(gaps),
                economic_value=monthly_potential,
            )

            return self.create_insight(
                title=f"Huecos de Surtido: {len(gaps)} categorías infracubiertas",
                description=(
                    f"Hay {len(gaps)} categorías importantes del mercado donde tu farmacia "
                    f"tiene menos del {pharmacy_threshold}% de los productos disponibles. "
                    f"Ampliar surtido en estas categorías puede capturar nueva demanda."
                ),
                impact_score=impact_score,
                economic_impact=f"Potencial: +{monthly_potential:,.0f}€/mes",
                economic_value=monthly_potential,
                action_label="Ver oportunidades",
                deeplink="/ventalibre/surtido?filter=gaps",
                affected_items=top_gaps,
                severity_override="low",  # Es más informativo que urgente
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None


class CanibalizacionRule(BaseInsightRule):
    """
    SURTIDO_003: Canibalización - Productos que compiten entre sí.

    Detecta pares de productos en la misma categoría donde las ventas de uno
    parecen quitar ventas del otro (correlación negativa en el tiempo).

    Issue #519: Cuando use_subcategory=True, agrupa por subcategoría L2.

    Esto puede indicar duplicidad innecesaria en el surtido o problemas de
    posicionamiento de precio.

    Impacto económico: Estimación de margen perdido por duplicidad.
    """

    rule_code = "SURTIDO_003"
    category = "surtido"
    severity = "low"
    default_config = {
        "correlation_threshold": -0.5,  # Correlación negativa fuerte
        "min_product_revenue": 100.0,  # € mínimo por producto
        "lookback_days": 90,
        "max_items_to_show": 10,
        "use_subcategory": True,  # Issue #519: Usar L2 cuando disponible
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa canibalización entre productos."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            corr_threshold = cfg["correlation_threshold"]
            min_revenue = cfg["min_product_revenue"]
            lookback = cfg["lookback_days"]
            max_items = cfg["max_items_to_show"]
            use_subcategory = cfg.get("use_subcategory", True)

            today = date.today()
            start_date = today - timedelta(days=lookback)

            # 1. Obtener ventas semanales por producto y categoría/subcategoría
            # Issue #519: Incluir ml_subcategory
            weekly_sales = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    SalesData.codigo_nacional,
                    SalesData.product_name,
                    func.date_trunc("week", SalesData.sale_date).label("week"),
                    func.sum(SalesData.total_amount).label("revenue"),
                    func.sum(SalesData.quantity).label("units"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesEnrichment.ml_category.isnot(None),
                    SalesData.codigo_nacional.isnot(None),
                )
                .group_by(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    SalesData.codigo_nacional,
                    SalesData.product_name,
                    func.date_trunc("week", SalesData.sale_date),
                )
                .all()
            )

            if not weekly_sales:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Organizar datos por categoría/subcategoría y producto
            # Issue #519: Usar L2 cuando disponible
            # {category_key: {product_code: {name: str, weeks: {week: revenue}, parent: str|None}}}
            category_products: dict[str, dict[str, dict]] = {}
            category_meta: dict[str, dict] = {}  # Track metadata per category

            for row in weekly_sales:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory
                code = row.codigo_nacional
                name = row.product_name
                week = str(row.week)
                rev = float(row.revenue or 0)

                # Determinar clave de agrupación
                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat
                    parent = l1_cat
                    is_subcat = True
                else:
                    cat_key = l1_cat
                    parent = None
                    is_subcat = False

                if cat_key not in category_products:
                    category_products[cat_key] = {}
                    category_meta[cat_key] = {
                        "parent": parent,
                        "is_subcategory": is_subcat,
                    }

                if code not in category_products[cat_key]:
                    category_products[cat_key][code] = {"name": name, "weeks": {}, "total": 0}

                category_products[cat_key][code]["weeks"][week] = rev
                category_products[cat_key][code]["total"] += rev

            # 3. Calcular correlaciones entre productos en cada categoría/subcategoría
            cannibalization_pairs = []
            has_l2_insights = False

            for cat_key, products in category_products.items():
                meta = category_meta.get(cat_key, {})

                # Filtrar productos con suficiente revenue
                qualified = {
                    code: data
                    for code, data in products.items()
                    if data["total"] >= min_revenue
                }

                if len(qualified) < 2:
                    continue

                # Obtener todas las semanas
                all_weeks = set()
                for data in qualified.values():
                    all_weeks.update(data["weeks"].keys())
                all_weeks = sorted(all_weeks)

                if len(all_weeks) < 4:  # Mínimo 4 semanas para correlación
                    continue

                # Comparar pares de productos
                codes = list(qualified.keys())
                for i, code1 in enumerate(codes):
                    for code2 in codes[i + 1:]:
                        # Construir series temporales
                        series1 = [qualified[code1]["weeks"].get(w, 0) for w in all_weeks]
                        series2 = [qualified[code2]["weeks"].get(w, 0) for w in all_weeks]

                        # Calcular correlación simple
                        corr = self._calculate_correlation(series1, series2)

                        if corr is not None and corr <= corr_threshold:
                            # Hay canibalización
                            total1 = qualified[code1]["total"]
                            total2 = qualified[code2]["total"]
                            # Estimación: 10% de las ventas del menor son "robadas"
                            cannibalized = min(total1, total2) * 0.10

                            # Issue #519: Incluir info de L2
                            item = {
                                "category": cat_key,
                                "product1_code": code1,
                                "product1_name": qualified[code1]["name"][:40],
                                "product1_revenue": round(total1, 2),
                                "product2_code": code2,
                                "product2_name": qualified[code2]["name"][:40],
                                "product2_revenue": round(total2, 2),
                                "correlation": round(corr, 2),
                                "cannibalized_estimate": round(cannibalized, 2),
                            }

                            # Añadir parent_category si es subcategoría L2
                            if meta.get("is_subcategory") and meta.get("parent"):
                                item["parent_category"] = meta["parent"]
                                has_l2_insights = True

                            cannibalization_pairs.append(item)

            if not cannibalization_pairs:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 4. Ordenar por canibalización y limitar
            cannibalization_pairs.sort(key=lambda x: -x["cannibalized_estimate"])
            top_pairs = cannibalization_pairs[:max_items]

            # 5. Calcular impacto total
            total_cannibalized = sum(p["cannibalized_estimate"] for p in cannibalization_pairs)
            monthly_impact = total_cannibalized / (lookback / 30)

            # 6. Calcular impact_score
            count_score = min(30, len(cannibalization_pairs) * 5)
            value_score = min(70, monthly_impact / 30)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(cannibalization_pairs),
                economic_value=monthly_impact,
            )

            # Issue #519: Ajustar descripción según granularidad L1/L2
            if has_l2_insights:
                description = (
                    f"Se detectaron {len(cannibalization_pairs)} pares de productos que parecen "
                    f"competir entre sí (correlación de ventas negativa). Considera simplificar "
                    f"el surtido o diferenciar precios. Análisis detallado por subcategoría L2."
                )
            else:
                description = (
                    f"Se detectaron {len(cannibalization_pairs)} pares de productos que parecen "
                    f"competir entre sí (correlación de ventas negativa). Considera simplificar "
                    f"el surtido o diferenciar precios."
                )

            return self.create_insight(
                title=f"Canibalización: {len(cannibalization_pairs)} pares de productos",
                description=description,
                impact_score=impact_score,
                economic_impact=f"Est. {monthly_impact:,.0f}€/mes en duplicidad",
                economic_value=monthly_impact,
                action_label="Ver duplicidades",
                deeplink="/ventalibre/surtido?filter=cannibalization",
                affected_items=top_pairs,
                severity_override="low",
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None

    def _calculate_correlation(self, x: list[float], y: list[float]) -> Optional[float]:
        """Calcula correlación de Pearson entre dos series."""
        n = len(x)
        if n < 4 or n != len(y):
            return None

        # Medias
        mean_x = sum(x) / n
        mean_y = sum(y) / n

        # Varianzas y covarianza
        var_x = sum((xi - mean_x) ** 2 for xi in x)
        var_y = sum((yi - mean_y) ** 2 for yi in y)
        cov_xy = sum((xi - mean_x) * (yi - mean_y) for xi, yi in zip(x, y))

        # Correlación
        if var_x == 0 or var_y == 0:
            return None

        corr = cov_xy / (var_x ** 0.5 * var_y ** 0.5)
        return corr
