# backend/app/services/insight_engine/rules/stock_rules.py
"""
Reglas de Stock para Insight Engine v2.0.

Issue #506: 3 reglas de stock con impacto económico.

STOCK_001: Stock Crítico - Productos alta rotación con cobertura crítica
STOCK_002: Sobrestock - Productos baja rotación con exceso de stock
STOCK_003: Caja Atrapada - Capital inmovilizado en productos dead/slow
"""

from datetime import date, timedelta
from typing import Any, Optional
from uuid import UUID

import structlog
from sqlalchemy import func
from sqlalchemy.orm import Session

from app.models.inventory_snapshot import InventorySnapshot
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from .base import BaseInsightRule
from ..models import InsightResult

logger = structlog.get_logger(__name__)


class StockCriticoRule(BaseInsightRule):
    """
    STOCK_001: Stock Crítico - Productos alta rotación con cobertura crítica.

    Detecta productos de Clase A (alta rotación) con menos de X días de cobertura.
    Estos productos generan la mayor parte de los ingresos y su rotura es crítica.

    Impacto económico: Ventas perdidas estimadas = ventas_diarias * días_sin_stock
    """

    rule_code = "STOCK_001"
    category = "stock"
    severity = "high"
    default_config = {
        "coverage_days_critical": 3,  # Días mínimos aceptables
        "min_daily_sales_value": 10.0,  # € mínimo en ventas diarias para considerar
        "lookback_days": 90,  # Días para calcular ventas diarias
        "max_items_to_show": 15,  # Límite de items en respuesta
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa productos con cobertura crítica."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            coverage_threshold = cfg["coverage_days_critical"]
            min_daily_value = cfg["min_daily_sales_value"]
            lookback = cfg["lookback_days"]
            max_items = cfg["max_items_to_show"]

            # 1. Obtener snapshot más reciente
            latest_date = (
                db.query(func.max(InventorySnapshot.snapshot_date))
                .filter(InventorySnapshot.pharmacy_id == pharmacy_id)
                .scalar()
            )

            if not latest_date:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Calcular ventas diarias por producto (últimos N días)
            end_date = date.today()
            start_date = end_date - timedelta(days=lookback)

            sales_subq = (
                db.query(
                    SalesData.codigo_nacional,
                    func.sum(SalesData.quantity).label("units_sold"),
                    func.sum(SalesData.total_price).label("revenue"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesData.sale_date <= end_date,
                    SalesData.codigo_nacional.isnot(None),
                )
                .group_by(SalesData.codigo_nacional)
                .subquery()
            )

            # 3. Obtener productos con stock y calcular cobertura
            products = (
                db.query(
                    InventorySnapshot.product_code,
                    InventorySnapshot.product_name,
                    InventorySnapshot.ean13,
                    InventorySnapshot.stock_quantity,
                    InventorySnapshot.unit_price,
                    sales_subq.c.units_sold,
                    sales_subq.c.revenue,
                )
                .outerjoin(
                    sales_subq,
                    InventorySnapshot.product_code == sales_subq.c.codigo_nacional,
                )
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.snapshot_date == latest_date,
                    InventorySnapshot.stock_quantity > 0,
                )
                .all()
            )

            # 4. Filtrar productos con cobertura crítica y alta rotación
            critical_products = []

            for p in products:
                units_sold = int(p.units_sold or 0)
                if units_sold == 0:
                    continue  # Sin ventas = sin cobertura calculable

                daily_sales = units_sold / lookback
                daily_value = float(p.revenue or 0) / lookback

                # Solo productos con ventas diarias significativas
                if daily_value < min_daily_value:
                    continue

                stock = int(p.stock_quantity or 0)
                coverage_days = stock / daily_sales if daily_sales > 0 else 999

                if coverage_days < coverage_threshold:
                    # Estimar ventas perdidas (días sin stock * ventas diarias)
                    days_without = max(0, coverage_threshold - coverage_days)
                    sales_lost = days_without * daily_value

                    critical_products.append({
                        "product_code": p.product_code,
                        "product_name": p.product_name,
                        "ean": p.ean13,
                        "stock_quantity": stock,
                        "coverage_days": round(coverage_days, 1),
                        "daily_sales_value": round(daily_value, 2),
                        "sales_lost_estimate": round(sales_lost, 2),
                    })

            if not critical_products:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 5. Ordenar por ventas perdidas y limitar
            critical_products.sort(key=lambda x: -x["sales_lost_estimate"])
            top_products = critical_products[:max_items]

            # 6. Calcular impacto económico total
            total_lost = sum(p["sales_lost_estimate"] for p in critical_products)
            monthly_impact = total_lost * 30 / coverage_threshold

            # 7. Calcular impact_score (0-100)
            # Basado en cantidad de productos y valor perdido
            count_score = min(50, len(critical_products) * 5)
            value_score = min(50, monthly_impact / 100)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(critical_products),
                economic_value=monthly_impact,
            )

            return self.create_insight(
                title=f"Stock Crítico: {len(critical_products)} productos de alta rotación",
                description=(
                    f"Tienes {len(critical_products)} productos con alta demanda y menos de "
                    f"{coverage_threshold} días de cobertura. Esto puede resultar en ventas "
                    f"perdidas y clientes insatisfechos."
                ),
                impact_score=impact_score,
                economic_impact=f"Est. +{monthly_impact:,.0f}€/mes en ventas recuperables",
                economic_value=monthly_impact,
                action_label="Ver productos afectados",
                deeplink="/ventalibre/inventario?alert=low_stock",
                affected_items=top_products,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None


class SobrestockRule(BaseInsightRule):
    """
    STOCK_002: Sobrestock - Productos con exceso de stock.

    Detecta productos con más de X días de cobertura (típicamente >180 días).
    El capital está inmovilizado y hay riesgo de caducidad.

    Impacto económico: Valor del stock en exceso que podría liberarse.
    """

    rule_code = "STOCK_002"
    category = "stock"
    severity = "medium"
    default_config = {
        "coverage_days_overstock": 180,  # Días para considerar sobrestock
        "optimal_stock_days": 60,  # Días de stock óptimo
        "min_stock_value": 50.0,  # € mínimo en stock para considerar
        "lookback_days": 90,  # Días para calcular ventas diarias
        "max_items_to_show": 15,  # Límite de items
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa productos con sobrestock."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            overstock_threshold = cfg["coverage_days_overstock"]
            optimal_stock_days = cfg["optimal_stock_days"]
            min_stock_value = cfg["min_stock_value"]
            lookback = cfg["lookback_days"]
            max_items = cfg["max_items_to_show"]

            # 1. Obtener snapshot más reciente
            latest_date = (
                db.query(func.max(InventorySnapshot.snapshot_date))
                .filter(InventorySnapshot.pharmacy_id == pharmacy_id)
                .scalar()
            )

            if not latest_date:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Calcular ventas diarias por producto
            end_date = date.today()
            start_date = end_date - timedelta(days=lookback)

            sales_subq = (
                db.query(
                    SalesData.codigo_nacional,
                    func.sum(SalesData.quantity).label("units_sold"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesData.sale_date <= end_date,
                    SalesData.codigo_nacional.isnot(None),
                )
                .group_by(SalesData.codigo_nacional)
                .subquery()
            )

            # 3. Obtener productos con stock
            products = (
                db.query(
                    InventorySnapshot.product_code,
                    InventorySnapshot.product_name,
                    InventorySnapshot.ean13,
                    InventorySnapshot.stock_quantity,
                    InventorySnapshot.unit_price,
                    InventorySnapshot.unit_cost,
                    sales_subq.c.units_sold,
                )
                .outerjoin(
                    sales_subq,
                    InventorySnapshot.product_code == sales_subq.c.codigo_nacional,
                )
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.snapshot_date == latest_date,
                    InventorySnapshot.stock_quantity > 0,
                )
                .all()
            )

            # 4. Identificar productos con sobrestock
            overstock_products = []

            for p in products:
                stock = int(p.stock_quantity or 0)
                price = float(p.unit_cost or p.unit_price or 0)
                stock_value = stock * price

                if stock_value < min_stock_value:
                    continue

                units_sold = int(p.units_sold or 0)
                daily_sales = units_sold / lookback if units_sold > 0 else 0

                # Cobertura infinita si no hay ventas, o calcular
                if daily_sales == 0:
                    coverage_days = 999  # Sin ventas = cobertura "infinita"
                else:
                    coverage_days = stock / daily_sales

                if coverage_days > overstock_threshold:
                    # Calcular stock en exceso (lo que sobra después de N días)
                    optimal_stock = daily_sales * optimal_stock_days
                    excess_units = max(0, stock - optimal_stock)
                    excess_value = excess_units * price

                    overstock_products.append({
                        "product_code": p.product_code,
                        "product_name": p.product_name,
                        "ean": p.ean13,
                        "stock_quantity": stock,
                        "stock_value": round(stock_value, 2),
                        "coverage_days": round(coverage_days, 0) if coverage_days < 999 else ">365",
                        "excess_value": round(excess_value, 2),
                    })

            if not overstock_products:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 5. Ordenar por valor en exceso y limitar
            overstock_products.sort(key=lambda x: -(x["excess_value"] if isinstance(x["excess_value"], float) else 0))
            top_products = overstock_products[:max_items]

            # 6. Calcular impacto económico
            total_excess = sum(
                p["excess_value"] for p in overstock_products
                if isinstance(p["excess_value"], (int, float))
            )

            # 7. Calcular impact_score
            # Formula: count_score (0-40) + value_score (0-60) = 0-100
            # Divisor /200: overstock typically 2000-12000€ tied up → 60 points at 12000€
            # (higher values than margin rules because stock represents capital lock-up)
            count_score = min(40, len(overstock_products) * 3)
            value_score = min(60, total_excess / 200)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(overstock_products),
                economic_value=total_excess,
            )

            return self.create_insight(
                title=f"Sobrestock: {len(overstock_products)} productos con exceso",
                description=(
                    f"Tienes {len(overstock_products)} productos con más de {overstock_threshold} días "
                    f"de cobertura. Este capital podría reinvertirse en productos de mayor rotación."
                ),
                impact_score=impact_score,
                economic_impact=f"Capital liberable: {total_excess:,.0f}€",
                economic_value=total_excess,
                action_label="Revisar excesos",
                deeplink="/ventalibre/inventario?alert=overstock",
                affected_items=top_products,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None


class CajaAtrapadaRule(BaseInsightRule):
    """
    STOCK_003: Caja Atrapada - Capital inmovilizado en productos sin movimiento.

    Detecta productos dead stock (sin ventas en >180 días) con valor significativo.
    Este capital está completamente atrapado y debería liquidarse.

    Impacto económico: Valor total del stock muerto.
    """

    rule_code = "STOCK_003"
    category = "stock"
    severity = "medium"
    default_config = {
        "dead_stock_days": 180,  # Días sin venta para considerar dead stock
        "trapped_cash_min_eur": 500.0,  # € mínimo total para alertar
        "min_item_value": 20.0,  # € mínimo por item para incluir
        "max_items_to_show": 20,  # Límite de items
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa capital atrapado en dead stock."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            dead_days = cfg["dead_stock_days"]
            min_total = cfg["trapped_cash_min_eur"]
            min_item = cfg["min_item_value"]
            max_items = cfg["max_items_to_show"]

            cutoff_date = date.today() - timedelta(days=dead_days)

            # 1. Obtener snapshot más reciente
            latest_date = (
                db.query(func.max(InventorySnapshot.snapshot_date))
                .filter(InventorySnapshot.pharmacy_id == pharmacy_id)
                .scalar()
            )

            if not latest_date:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Buscar productos sin ventas recientes
            dead_products_query = (
                db.query(
                    InventorySnapshot.product_code,
                    InventorySnapshot.product_name,
                    InventorySnapshot.ean13,
                    InventorySnapshot.stock_quantity,
                    InventorySnapshot.last_sale_date,
                    (
                        InventorySnapshot.stock_quantity *
                        func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
                    ).label("stock_value"),
                )
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.snapshot_date == latest_date,
                    InventorySnapshot.stock_quantity > 0,
                )
                .filter(
                    # Sin venta reciente o nunca vendido
                    (InventorySnapshot.last_sale_date < cutoff_date) |
                    (InventorySnapshot.last_sale_date.is_(None))
                )
            )

            dead_products = dead_products_query.all()

            # 3. Filtrar por valor mínimo y construir lista
            trapped_items = []
            total_trapped = 0.0

            for p in dead_products:
                stock_value = float(p.stock_value or 0)

                if stock_value < min_item:
                    continue

                days_since = (
                    (date.today() - p.last_sale_date).days
                    if p.last_sale_date
                    else None
                )

                trapped_items.append({
                    "product_code": p.product_code,
                    "product_name": p.product_name,
                    "ean": p.ean13,
                    "stock_quantity": int(p.stock_quantity or 0),
                    "stock_value": round(stock_value, 2),
                    "days_since_sale": days_since or f">{dead_days}",
                })
                total_trapped += stock_value

            # 4. Verificar umbral mínimo
            if total_trapped < min_total:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 5. Ordenar por valor y limitar
            trapped_items.sort(key=lambda x: -x["stock_value"])
            top_items = trapped_items[:max_items]

            # 6. Calcular impact_score
            count_score = min(30, len(trapped_items) * 2)
            value_score = min(70, total_trapped / 100)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(trapped_items),
                economic_value=total_trapped,
            )

            return self.create_insight(
                title=f"Caja Atrapada: {total_trapped:,.0f}€ en {len(trapped_items)} productos",
                description=(
                    f"Tienes {total_trapped:,.0f}€ invertidos en {len(trapped_items)} productos "
                    f"que no se han vendido en más de {dead_days} días. Considera liquidar o "
                    f"devolver al proveedor para liberar capital."
                ),
                impact_score=impact_score,
                economic_impact=f"Capital atrapado: {total_trapped:,.0f}€",
                economic_value=total_trapped,
                action_label="Ver dead stock",
                deeplink="/ventalibre/inventario?alert=dead_stock",
                affected_items=top_items,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None
