# backend/app/services/insight_engine/rules/margin_rules.py
"""
Reglas de Margen para Insight Engine v2.0.

Issue #506: 2 reglas de margen con impacto económico.
Issue #519: MARGIN_001 conectado con subcategorías L2 para insights más granulares.

MARGIN_001: Erosión de Margen - Categorías/subcategorías con caída de margen YoY
MARGIN_002: Margen Bajo - Top sellers con margen inferior al umbral
"""

from datetime import date, timedelta
from typing import Any, Optional
from uuid import UUID

import structlog
from sqlalchemy import and_, extract, func
from sqlalchemy.orm import Session

from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from .base import BaseInsightRule
from ..models import InsightResult

logger = structlog.get_logger(__name__)


class ErosionMargenRule(BaseInsightRule):
    """
    MARGIN_001: Erosión de Margen - Categorías/subcategorías con caída significativa YoY.

    Detecta categorías donde el margen ha caído más de X puntos porcentuales
    respecto al mismo período del año anterior.

    Issue #519: Cuando use_subcategory=True, analiza a nivel L2 (subcategoría)
    para dar insights más específicos.

    Impacto económico: Diferencia de margen * ventas = beneficio perdido.
    """

    rule_code = "MARGIN_001"
    category = "margin"
    severity = "high"
    default_config = {
        "erosion_pp": 3,  # Puntos porcentuales de caída para alertar
        "min_category_revenue": 1000.0,  # € mínimo de ventas en categoría
        "lookback_months": 3,  # Meses a comparar
        "max_items_to_show": 10,  # Límite de categorías
        "use_subcategory": True,  # Issue #519: Usar L2 cuando disponible
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa categorías/subcategorías con erosión de margen."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            erosion_threshold = cfg["erosion_pp"]
            min_revenue = cfg["min_category_revenue"]
            lookback = cfg["lookback_months"]
            max_items = cfg["max_items_to_show"]
            use_subcategory = cfg.get("use_subcategory", True)

            today = date.today()
            current_year = today.year
            previous_year = current_year - 1

            # Definir períodos de comparación
            # Período actual: últimos N meses del año actual
            # Período anterior: mismos meses del año anterior
            current_months = list(range(max(1, today.month - lookback + 1), today.month + 1))
            if not current_months:
                current_months = [today.month]

            # 1. Calcular margen promedio por categoría/subcategoría - Año actual
            # Issue #519: Incluir ml_subcategory
            current_margins = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    func.sum(SalesData.total_amount).label("revenue"),
                    func.sum(SalesData.margin_amount).label("total_margin"),
                    func.avg(SalesData.margin_percentage).label("avg_margin_pct"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    extract("year", SalesData.sale_date) == current_year,
                    extract("month", SalesData.sale_date).in_(current_months),
                    SalesEnrichment.ml_category.isnot(None),
                    SalesEnrichment.product_type == "venta_libre",
                    SalesData.margin_percentage.isnot(None),
                )
                .group_by(SalesEnrichment.ml_category, SalesEnrichment.ml_subcategory)
                .having(func.sum(SalesData.total_amount) >= min_revenue)
                .all()
            )

            if not current_margins:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Calcular margen promedio - Año anterior (mismos meses)
            previous_margins = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    func.avg(SalesData.margin_percentage).label("avg_margin_pct"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    extract("year", SalesData.sale_date) == previous_year,
                    extract("month", SalesData.sale_date).in_(current_months),
                    SalesEnrichment.ml_category.isnot(None),
                    SalesEnrichment.product_type == "venta_libre",
                    SalesData.margin_percentage.isnot(None),
                )
                .group_by(SalesEnrichment.ml_category, SalesEnrichment.ml_subcategory)
                .all()
            )

            # Issue #519: Construir diccionario con soporte L2
            prev_margin_dict: dict[str, dict] = {}
            for row in previous_margins:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory

                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat
                else:
                    cat_key = l1_cat

                if cat_key not in prev_margin_dict:
                    prev_margin_dict[cat_key] = {
                        "margin": float(row.avg_margin_pct or 0),
                        "count": 1,
                    }
                else:
                    # Promediar si hay múltiples registros
                    old = prev_margin_dict[cat_key]
                    new_count = old["count"] + 1
                    prev_margin_dict[cat_key] = {
                        "margin": (old["margin"] * old["count"] + float(row.avg_margin_pct or 0)) / new_count,
                        "count": new_count,
                    }

            # 3. Comparar y detectar erosiones
            eroded_categories = []
            has_l2_insights = False
            processed_keys = set()

            for row in current_margins:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory

                # Determinar clave de agrupación
                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat
                    display_name = l2_subcat
                    parent_category = l1_cat
                    is_subcategory = True
                else:
                    cat_key = l1_cat
                    display_name = l1_cat
                    parent_category = None
                    is_subcategory = False

                # Evitar duplicados
                if cat_key in processed_keys:
                    continue
                processed_keys.add(cat_key)

                current_pct = float(row.avg_margin_pct or 0)
                prev_data = prev_margin_dict.get(cat_key)

                if prev_data is None:
                    continue  # No hay datos del año anterior

                prev_pct = prev_data["margin"]
                erosion = prev_pct - current_pct

                if erosion >= erosion_threshold:
                    revenue = float(row.revenue or 0)
                    # Beneficio perdido = (erosión% / 100) * revenue
                    lost_profit = (erosion / 100) * revenue

                    item = {
                        "category": display_name,
                        "current_margin_pct": round(current_pct, 1),
                        "previous_margin_pct": round(prev_pct, 1),
                        "erosion_pp": round(erosion, 1),
                        "revenue": round(revenue, 2),
                        "lost_profit": round(lost_profit, 2),
                    }

                    # Añadir parent_category si es subcategoría L2
                    if is_subcategory and parent_category:
                        item["parent_category"] = parent_category
                        has_l2_insights = True

                    eroded_categories.append(item)

            if not eroded_categories:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 4. Ordenar por beneficio perdido y limitar
            eroded_categories.sort(key=lambda x: -x["lost_profit"])
            top_categories = eroded_categories[:max_items]

            # 5. Calcular impacto total
            total_lost = sum(c["lost_profit"] for c in eroded_categories)
            monthly_impact = total_lost  # Ya es mensualizado por el lookback

            # 6. Calcular impact_score
            # Formula: count_score (0-40) + value_score (0-60) = 0-100
            # Divisor /50: margin erosion typically 500-3000€/month → 60 points at 3000€
            count_score = min(40, len(eroded_categories) * 10)
            value_score = min(60, monthly_impact / 50)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(eroded_categories),
                economic_value=monthly_impact,
            )

            # Issue #519: Ajustar descripción según granularidad L1/L2
            if has_l2_insights:
                level_term = "subcategorías"
                description = (
                    f"Se detectaron {len(eroded_categories)} subcategorías con una caída de margen "
                    f"de {erosion_threshold}+ puntos porcentuales respecto al año anterior. "
                    f"Revisa precios y condiciones de compra. Análisis detallado por subcategoría L2."
                )
            else:
                level_term = "categorías"
                description = (
                    f"Se detectaron {len(eroded_categories)} categorías con una caída de margen "
                    f"de {erosion_threshold}+ puntos porcentuales respecto al año anterior. "
                    f"Revisa precios y condiciones de compra."
                )

            return self.create_insight(
                title=f"Erosión de Margen: {len(eroded_categories)} {level_term} afectadas",
                description=description,
                impact_score=impact_score,
                economic_impact=f"Beneficio perdido: {total_lost:,.0f}€",
                economic_value=total_lost,
                action_label="Analizar márgenes",
                deeplink="/ventalibre/analisis?view=margins",
                affected_items=top_categories,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None


class MargenBajoRule(BaseInsightRule):
    """
    MARGIN_002: Margen Bajo - Top sellers con margen inferior al umbral.

    Detecta productos de alta rotación que tienen un margen bajo.
    Son oportunidades de mejora de rentabilidad con alto impacto.

    Impacto económico: Diferencia entre margen actual y objetivo * ventas.
    """

    rule_code = "MARGIN_002"
    category = "margin"
    severity = "medium"
    default_config = {
        "low_margin_threshold": 15.0,  # % margen considerado bajo
        "target_margin": 25.0,  # % margen objetivo
        "top_n_products": 50,  # Analizar top N por ventas
        "min_monthly_sales": 100.0,  # € mínimo mensual de ventas
        "lookback_days": 90,  # Días de ventas a analizar
        "max_items_to_show": 15,
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa productos con margen bajo."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            low_margin = cfg["low_margin_threshold"]
            target_margin = cfg["target_margin"]
            top_n = cfg["top_n_products"]
            min_sales = cfg["min_monthly_sales"]
            lookback = cfg["lookback_days"]
            max_items = cfg["max_items_to_show"]

            today = date.today()
            start_date = today - timedelta(days=lookback)

            # 1. Obtener top productos por ventas
            top_products = (
                db.query(
                    SalesData.product_name,
                    SalesData.codigo_nacional,
                    SalesData.ean13,
                    func.sum(SalesData.total_amount).label("total_revenue"),
                    func.sum(SalesData.quantity).label("total_units"),
                    func.avg(SalesData.margin_percentage).label("avg_margin"),
                    func.sum(SalesData.margin_amount).label("total_margin"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesData.margin_percentage.isnot(None),
                    SalesData.margin_percentage < low_margin,  # Solo low margin
                )
                .group_by(
                    SalesData.product_name,
                    SalesData.codigo_nacional,
                    SalesData.ean13,
                )
                .having(func.sum(SalesData.total_amount) >= min_sales * (lookback / 30))
                .order_by(func.sum(SalesData.total_amount).desc())
                .limit(top_n)
                .all()
            )

            if not top_products:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Construir lista de productos con margen bajo
            low_margin_products = []
            total_potential = 0.0

            for p in top_products:
                current_margin = float(p.avg_margin or 0)
                revenue = float(p.total_revenue or 0)

                # Potencial = (target - actual) / 100 * revenue
                margin_gap = target_margin - current_margin
                potential_gain = (margin_gap / 100) * revenue

                low_margin_products.append({
                    "product_name": p.product_name,
                    "product_code": p.codigo_nacional,
                    "ean": p.ean13,
                    "revenue": round(revenue, 2),
                    "units_sold": int(p.total_units or 0),
                    "current_margin_pct": round(current_margin, 1),
                    "target_margin_pct": target_margin,
                    "potential_gain": round(potential_gain, 2),
                })
                total_potential += potential_gain

            # 3. Ordenar por potencial y limitar
            low_margin_products.sort(key=lambda x: -x["potential_gain"])
            top_items = low_margin_products[:max_items]

            # 4. Calcular impacto anualizado
            months_analyzed = lookback / 30
            monthly_potential = total_potential / months_analyzed if months_analyzed > 0 else total_potential

            # 5. Calcular impact_score
            # Formula: count_score (0-40) + value_score (0-60) = 0-100
            # Divisor /30: low margin gain typically 300-1800€/month → 60 points at 1800€
            # (smaller per-product impact but more products affected)
            count_score = min(40, len(low_margin_products) * 3)
            value_score = min(60, monthly_potential / 30)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(low_margin_products),
                economic_value=monthly_potential,
            )

            return self.create_insight(
                title=f"Margen Bajo: {len(low_margin_products)} top sellers",
                description=(
                    f"Tienes {len(low_margin_products)} productos de alta rotación con margen "
                    f"inferior al {low_margin}%. Negociar mejores condiciones o revisar "
                    f"precios podría mejorar significativamente la rentabilidad."
                ),
                impact_score=impact_score,
                economic_impact=f"Potencial: +{monthly_potential:,.0f}€/mes",
                economic_value=monthly_potential,
                action_label="Ver productos",
                deeplink="/ventalibre/analisis?view=low_margin",
                affected_items=top_items,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None
