# backend/app/services/insight_engine/rules/hhi_rules.py
"""
Reglas de HHI para Insight Engine v2.0.

Issue #506: Regla de dependencia de proveedor con impacto económico.
Issue #519: Conectado con subcategorías L2 para insights más granulares.

HHI_001: Dependencia Crítica - Categorías/subcategorías con alta concentración de marcas
"""

from datetime import date, timedelta
from typing import Any, Optional
from uuid import UUID

import structlog
from sqlalchemy import func
from sqlalchemy.orm import Session

from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from .base import BaseInsightRule
from ..models import InsightResult

logger = structlog.get_logger(__name__)


class DependenciaCriticaRule(BaseInsightRule):
    """
    HHI_001: Dependencia Crítica - Categorías/subcategorías con alta concentración de marcas.

    Detecta categorías donde una o pocas marcas dominan las ventas (HHI > 4000).
    Esto representa un riesgo de negociación y disponibilidad.

    Issue #519: Cuando use_subcategory=True, analiza a nivel L2 (subcategoría)
    para dar insights más específicos (ej: "Protector Solar 50+" vs "dermocosmetica").

    HHI = Σ(cuota_i × 100)²
    - < 1500: Mercado atomizado (bueno)
    - 1500-2500: Concentración moderada
    - 2500-4000: Alta concentración
    - > 4000: Dependencia crítica (alerta)

    Impacto económico: % de ventas en riesgo por dependencia.
    """

    rule_code = "HHI_001"
    category = "hhi"
    severity = "medium"
    default_config = {
        "hhi_critical": 4000,  # Umbral para dependencia crítica
        "min_category_revenue": 500.0,  # € mínimo en categoría
        "lookback_days": 90,  # Días de ventas a analizar
        "max_items_to_show": 10,  # Límite de categorías
        "use_subcategory": True,  # Issue #519: Usar L2 cuando disponible
    }

    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """Evalúa categorías/subcategorías con alta dependencia de marca."""
        self.log_evaluation_start(pharmacy_id)

        try:
            cfg = self.get_config(config)
            hhi_threshold = cfg["hhi_critical"]
            min_revenue = cfg["min_category_revenue"]
            lookback = cfg["lookback_days"]
            max_items = cfg["max_items_to_show"]
            use_subcategory = cfg.get("use_subcategory", True)

            today = date.today()
            start_date = today - timedelta(days=lookback)

            # 1. Obtener ventas por categoría/subcategoría y marca
            # Issue #519: Incluir ml_subcategory para análisis L2
            sales_by_cat_brand = (
                db.query(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    SalesEnrichment.detected_brand,
                    func.sum(SalesData.total_amount).label("revenue"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= start_date,
                    SalesEnrichment.product_type == "venta_libre",
                    SalesEnrichment.ml_category.isnot(None),
                )
                .group_by(
                    SalesEnrichment.ml_category,
                    SalesEnrichment.ml_subcategory,
                    SalesEnrichment.detected_brand,
                )
                .all()
            )

            if not sales_by_cat_brand:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 2. Agrupar por categoría (o subcategoría si use_subcategory=True) y calcular HHI
            # Issue #519: Usar L2 cuando disponible para granularidad
            category_data: dict[str, dict] = {}

            for row in sales_by_cat_brand:
                l1_cat = row.ml_category
                l2_subcat = row.ml_subcategory
                brand = row.detected_brand or "sin_marca"
                revenue = float(row.revenue or 0)

                # Determinar clave de agrupación: L2 si disponible y habilitado, sino L1
                if use_subcategory and l2_subcat:
                    cat_key = l2_subcat  # Usar subcategoría L2
                    display_name = l2_subcat  # Para mostrar en affected_items
                    parent_category = l1_cat
                else:
                    cat_key = l1_cat  # Fallback a L1
                    display_name = l1_cat
                    parent_category = None

                if cat_key not in category_data:
                    category_data[cat_key] = {
                        "total_revenue": 0.0,
                        "brands": {},
                        "display_name": display_name,
                        "parent_category": parent_category,
                        "is_subcategory": bool(use_subcategory and l2_subcat),
                    }

                category_data[cat_key]["total_revenue"] += revenue
                category_data[cat_key]["brands"][brand] = (
                    category_data[cat_key]["brands"].get(brand, 0) + revenue
                )

            # 3. Calcular HHI por categoría/subcategoría y detectar dependencias
            critical_categories = []
            has_l2_insights = False

            for cat, data in category_data.items():
                total = data["total_revenue"]

                if total < min_revenue:
                    continue

                # Calcular cuotas de mercado
                brand_shares = [
                    (rev / total)
                    for rev in data["brands"].values()
                ]

                # HHI = Σ(cuota_i × 100)²
                hhi = sum((share * 100) ** 2 for share in brand_shares)

                if hhi >= hhi_threshold:
                    # Encontrar marca dominante
                    dominant_brand = max(
                        data["brands"].items(),
                        key=lambda x: x[1],
                    )
                    dominant_share = (dominant_brand[1] / total) * 100

                    # Revenue en riesgo = total * % concentración sobre lo normal
                    # Asumimos que >60% de una marca es riesgo
                    risk_pct = max(0, dominant_share - 60) / 100
                    revenue_at_risk = total * risk_pct

                    # Issue #519: Incluir info de L2 en affected_items
                    item = {
                        "category": data["display_name"],
                        "hhi": round(hhi, 0),
                        "dominant_brand": dominant_brand[0],
                        "dominant_share_pct": round(dominant_share, 1),
                        "total_revenue": round(total, 2),
                        "brands_count": len(data["brands"]),
                        "revenue_at_risk": round(revenue_at_risk, 2),
                    }

                    # Añadir parent_category si es subcategoría L2
                    if data["is_subcategory"] and data["parent_category"]:
                        item["parent_category"] = data["parent_category"]
                        has_l2_insights = True

                    critical_categories.append(item)

            if not critical_categories:
                self.log_evaluation_result(pharmacy_id, found=False)
                return None

            # 4. Ordenar por HHI y limitar
            critical_categories.sort(key=lambda x: -x["hhi"])
            top_categories = critical_categories[:max_items]

            # 5. Calcular impacto total
            total_at_risk = sum(c["revenue_at_risk"] for c in critical_categories)
            total_revenue = sum(c["total_revenue"] for c in critical_categories)

            # 6. Calcular impact_score
            # Formula: count_score (0-40) + value_score (0-60) = 0-100
            count_score = min(40, len(critical_categories) * 10)
            # HHI risk is qualitative - use % of revenue at risk directly (no divisor)
            # 60% revenue at risk → 60 points (linear mapping)
            risk_pct = (total_at_risk / total_revenue * 100) if total_revenue > 0 else 0
            value_score = min(60, risk_pct)
            impact_score = int(count_score + value_score)

            self.log_evaluation_result(
                pharmacy_id,
                found=True,
                items_count=len(critical_categories),
                economic_value=total_at_risk,
            )

            # Issue #519: Ajustar descripción según granularidad L1/L2
            if has_l2_insights:
                level_term = "subcategorías"
                description = (
                    f"Tienes {len(critical_categories)} subcategorías con alta concentración "
                    f"de marca (HHI >{hhi_threshold}). Esto puede afectar tu capacidad de "
                    f"negociación y representa un riesgo si hay problemas de suministro. "
                    f"Análisis detallado por subcategoría L2."
                )
            else:
                level_term = "categorías"
                description = (
                    f"Tienes {len(critical_categories)} categorías con alta concentración "
                    f"de marca (HHI >{hhi_threshold}). Esto puede afectar tu capacidad de "
                    f"negociación y representa un riesgo si hay problemas de suministro."
                )

            return self.create_insight(
                title=f"Dependencia Crítica: {len(critical_categories)} {level_term}",
                description=description,
                impact_score=impact_score,
                economic_impact=f"Revenue en riesgo: {total_at_risk:,.0f}€",
                economic_value=total_at_risk,
                action_label=f"Ver {level_term}",
                deeplink="/ventalibre/analisis?view=hhi",
                affected_items=top_categories,
            )

        except Exception as e:
            self.log_evaluation_error(pharmacy_id, e)
            return None
