# backend/app/services/insight_engine/rules/base.py
"""
Base class para reglas del Insight Engine v2.0.

Issue #506: Framework extensible de reglas con contratos claros.

Cada regla debe:
1. Tener un rule_code único (ej: "STOCK_001")
2. Definir category, severity, y default_config
3. Implementar evaluate() que retorna Optional[InsightResult]
4. Calcular el hash de los affected_items para deduplicación

Ejemplo de implementación:
    class StockCriticoRule(BaseInsightRule):
        rule_code = "STOCK_001"
        category = "stock"
        severity = "high"
        default_config = {"coverage_days_critical": 3, "abc_class": "A"}

        async def evaluate(self, db, pharmacy_id, config) -> Optional[InsightResult]:
            # Query productos con cobertura crítica
            # Calcular impacto económico
            # Retornar InsightResult o None
            pass
"""

import hashlib
from abc import ABC, abstractmethod
from typing import Any, Optional
from uuid import UUID

import structlog
from sqlalchemy.orm import Session

from app.services.insight_engine.models import InsightResult

logger = structlog.get_logger(__name__)


class BaseInsightRule(ABC):
    """
    Base class abstracta para todas las reglas de insights.

    Attributes:
        rule_code: Código único de la regla (ej: "STOCK_001")
        category: Categoría (stock, margin, hhi, trend, surtido)
        severity: Severidad default (high, medium, low)
        default_config: Configuración por defecto con umbrales

    Methods:
        evaluate(): Evalúa la regla y retorna InsightResult o None
        calculate_hash(): Genera hash de affected_items para deduplicación
    """

    # Class attributes to be defined by subclasses
    rule_code: str = ""
    category: str = ""
    severity: str = "medium"
    default_config: dict[str, Any] = {}

    def __init__(self):
        """Initialize rule and validate required attributes."""
        if not self.rule_code:
            raise ValueError(f"{self.__class__.__name__} must define rule_code")
        if not self.category:
            raise ValueError(f"{self.__class__.__name__} must define category")

    @abstractmethod
    async def evaluate(
        self,
        db: Session,
        pharmacy_id: UUID,
        config: dict[str, Any],
    ) -> Optional[InsightResult]:
        """
        Evalúa la regla para una farmacia específica.

        Args:
            db: Sesión SQLAlchemy activa
            pharmacy_id: UUID de la farmacia a evaluar
            config: Configuración con umbrales (merged con default_config)

        Returns:
            InsightResult si la regla detecta un problema/oportunidad
            None si no aplica o no hay datos suficientes

        Raises:
            No debe lanzar excepciones - capturar internamente y loggear
        """
        pass

    def get_config(self, custom_config: Optional[dict[str, Any]] = None) -> dict[str, Any]:
        """
        Merge custom config con default_config.

        Args:
            custom_config: Configuración personalizada (puede ser None)

        Returns:
            Configuración final con defaults aplicados
        """
        merged = dict(self.default_config)
        if custom_config:
            merged.update(custom_config)
        return merged

    def calculate_hash(self, affected_items: list[dict[str, Any]]) -> str:
        """
        Calcula hash SHA256[:16] de los affected_items.

        El hash se usa para:
        - Identificar el mismo insight a través del tiempo
        - Buscar feedback previo (snooze/dismiss)
        - Evitar duplicados en la UI

        Args:
            affected_items: Lista de items afectados

        Returns:
            String de 16 caracteres hexadecimales
        """
        # Sort items por una key consistente para hash estable
        # Usamos str() para serializar de forma determinista
        sorted_items = sorted(
            [str(sorted(item.items())) for item in affected_items]
        )
        data = f"{self.rule_code}:{','.join(sorted_items)}"
        return hashlib.sha256(data.encode()).hexdigest()[:16]

    def create_insight(
        self,
        title: str,
        description: str,
        impact_score: int,
        economic_impact: str,
        economic_value: float,
        action_label: str,
        deeplink: str,
        affected_items: list[dict[str, Any]],
        severity_override: Optional[str] = None,
    ) -> InsightResult:
        """
        Helper para crear InsightResult con valores de la regla.

        Args:
            title: Título del insight
            description: Descripción detallada
            impact_score: Score 0-100
            economic_impact: Texto legible (ej: "+450€/mes")
            economic_value: Valor numérico en €
            action_label: Texto del botón
            deeplink: URL para navegación
            affected_items: Items para drill-down
            severity_override: Sobrescribir severidad default

        Returns:
            InsightResult listo para retornar
        """
        insight_hash = self.calculate_hash(affected_items)

        return InsightResult(
            rule_code=self.rule_code,
            category=self.category,
            severity=severity_override or self.severity,
            title=title,
            description=description,
            impact_score=impact_score,
            economic_impact=economic_impact,
            economic_value=economic_value,
            action_label=action_label,
            deeplink=deeplink,
            affected_items=affected_items,
            insight_hash=insight_hash,
        )

    def log_evaluation_start(self, pharmacy_id: UUID) -> None:
        """Log inicio de evaluación para debugging."""
        logger.debug(
            "insight_rule_evaluation_start",
            rule_code=self.rule_code,
            pharmacy_id=str(pharmacy_id),
        )

    def log_evaluation_result(
        self,
        pharmacy_id: UUID,
        found: bool,
        items_count: int = 0,
        economic_value: float = 0,
    ) -> None:
        """Log resultado de evaluación para métricas."""
        logger.info(
            "insight_rule_evaluation_result",
            rule_code=self.rule_code,
            pharmacy_id=str(pharmacy_id),
            found=found,
            items_count=items_count,
            economic_value=economic_value,
        )

    def log_evaluation_error(self, pharmacy_id: UUID, error: Exception) -> None:
        """Log error durante evaluación."""
        logger.error(
            "insight_rule_evaluation_error",
            rule_code=self.rule_code,
            pharmacy_id=str(pharmacy_id),
            error=str(error),
            exc_info=True,
        )

    @staticmethod
    def resolve_l2_category_key(
        l1_category: str,
        l2_subcategory: Optional[str],
        use_subcategory: bool = True,
    ) -> tuple[str, str, Optional[str], bool]:
        """
        Helper para resolver la clave de agrupación L1/L2.

        Usado por reglas que agrupan por categoría con soporte L2 (Issue #519).
        Evita duplicación del patrón if/else para determinar cat_key.

        Args:
            l1_category: Categoría L1 (ml_category)
            l2_subcategory: Subcategoría L2 (ml_subcategory), puede ser None
            use_subcategory: Si True, usa L2 cuando disponible; si False, siempre L1

        Returns:
            Tuple de (cat_key, display_name, parent_category, is_subcategory)
            - cat_key: Clave para agrupar (L2 si disponible y habilitado, sino L1)
            - display_name: Nombre para mostrar en affected_items
            - parent_category: Categoría padre (L1) si es subcategoría, sino None
            - is_subcategory: True si se usó L2

        Example:
            cat_key, display, parent, is_l2 = self.resolve_l2_category_key(
                row.ml_category, row.ml_subcategory, cfg["use_subcategory"]
            )
        """
        if use_subcategory and l2_subcategory:
            return (l2_subcategory, l2_subcategory, l1_category, True)
        return (l1_category, l1_category, None, False)

    def __repr__(self) -> str:
        return f"<{self.__class__.__name__}(rule_code={self.rule_code})>"
