# backend/app/services/insight_engine/models.py
"""
Data models para Insight Engine v2.0.

Issue #506: Estructuras de datos para resultados de insights.
"""

from dataclasses import dataclass, field
from typing import Any


@dataclass
class InsightResult:
    """
    Resultado de una regla de insight.

    Contiene toda la información necesaria para renderizar en UI
    y calcular el valor total de oportunidades.

    Attributes:
        rule_code: Código único de la regla (ej: "STOCK_001")
        category: Categoría para agrupación (stock, margin, hhi, trend, surtido)
        severity: Severidad (high, medium, low)
        title: Título corto para mostrar en UI
        description: Explicación detallada del problema/oportunidad
        impact_score: Puntuación de impacto 0-100 para ordenación
        economic_impact: Texto legible del impacto (ej: "Est. +450€/mes")
        economic_value: Valor numérico en € para cálculos
        action_label: Texto del botón de acción (ej: "Ver productos")
        deeplink: URL relativa para navegación (ej: "/ventalibre/inventario?filter=critical")
        affected_items: Lista de items afectados para drill-down
        insight_hash: Hash SHA256[:16] para deduplicación y feedback

    Example:
        InsightResult(
            rule_code="STOCK_001",
            category="stock",
            severity="high",
            title="Stock Crítico: 12 productos Clase A",
            description="Tienes 12 productos de alta rotación (Clase A) con menos de 3 días de cobertura.",
            impact_score=85,
            economic_impact="Est. +1.200€/mes en ventas recuperadas",
            economic_value=1200.0,
            action_label="Ver productos afectados",
            deeplink="/ventalibre/inventario?abc_class=A&coverage=critical",
            affected_items=[
                {"ean": "8430000000123", "name": "IBUPROFENO 600MG", "coverage_days": 2, "sales_lost": 150},
                {"ean": "8430000000456", "name": "PARACETAMOL 1G", "coverage_days": 1, "sales_lost": 200},
            ],
            insight_hash="a1b2c3d4e5f6g7h8"
        )
    """

    # Identification
    rule_code: str
    category: str
    severity: str

    # Content
    title: str
    description: str

    # Impact
    impact_score: int  # 0-100
    economic_impact: str  # Human readable
    economic_value: float  # € for calculations

    # Actions
    action_label: str
    deeplink: str

    # Details
    affected_items: list[dict[str, Any]] = field(default_factory=list)
    insight_hash: str = ""

    def __post_init__(self):
        """Validate fields after initialization."""
        # Validate severity
        valid_severities = {"high", "medium", "low"}
        if self.severity not in valid_severities:
            raise ValueError(f"severity must be one of {valid_severities}, got {self.severity}")

        # Validate impact_score range
        if not 0 <= self.impact_score <= 100:
            raise ValueError(f"impact_score must be 0-100, got {self.impact_score}")

        # Validate category
        valid_categories = {"stock", "margin", "hhi", "trend", "surtido"}
        if self.category not in valid_categories:
            raise ValueError(f"category must be one of {valid_categories}, got {self.category}")

    def to_dict(self) -> dict[str, Any]:
        """Convert to dictionary for JSON serialization."""
        return {
            "rule_code": self.rule_code,
            "category": self.category,
            "severity": self.severity,
            "title": self.title,
            "description": self.description,
            "impact_score": self.impact_score,
            "economic_impact": self.economic_impact,
            "economic_value": self.economic_value,
            "action_label": self.action_label,
            "deeplink": self.deeplink,
            "affected_items": self.affected_items,
            "insight_hash": self.insight_hash,
        }

    @classmethod
    def from_dict(cls, data: dict[str, Any]) -> "InsightResult":
        """Create from dictionary."""
        return cls(
            rule_code=data["rule_code"],
            category=data["category"],
            severity=data["severity"],
            title=data["title"],
            description=data["description"],
            impact_score=data["impact_score"],
            economic_impact=data["economic_impact"],
            economic_value=data["economic_value"],
            action_label=data["action_label"],
            deeplink=data["deeplink"],
            affected_items=data.get("affected_items", []),
            insight_hash=data.get("insight_hash", ""),
        )


@dataclass
class AuditResult:
    """
    Resultado completo de un audit de insights.

    Contiene todos los insights generados más metadata del audit.

    Attributes:
        insights: Lista de InsightResult ordenados por economic_value DESC
        total_opportunity: Suma de economic_value de todos los insights
        by_category: Conteo de insights por categoría
        by_severity: Conteo de insights por severidad
        suppressed_count: Número de insights filtrados por feedback
        audit_duration_ms: Tiempo de ejecución del audit en milliseconds
    """

    insights: list[InsightResult] = field(default_factory=list)
    total_opportunity: float = 0.0
    by_category: dict[str, int] = field(default_factory=dict)
    by_severity: dict[str, int] = field(default_factory=dict)
    suppressed_count: int = 0
    audit_duration_ms: int = 0

    def __post_init__(self):
        """Calculate aggregations if not provided."""
        if not self.by_category:
            self.by_category = {}
            for insight in self.insights:
                self.by_category[insight.category] = self.by_category.get(insight.category, 0) + 1

        if not self.by_severity:
            self.by_severity = {}
            for insight in self.insights:
                self.by_severity[insight.severity] = self.by_severity.get(insight.severity, 0) + 1

        if self.total_opportunity == 0.0 and self.insights:
            self.total_opportunity = sum(i.economic_value for i in self.insights)

    def to_dict(self) -> dict[str, Any]:
        """Convert to dictionary for JSON serialization."""
        return {
            "insights": [i.to_dict() for i in self.insights],
            "total_opportunity": self.total_opportunity,
            "by_category": self.by_category,
            "by_severity": self.by_severity,
            "suppressed_count": self.suppressed_count,
            "audit_duration_ms": self.audit_duration_ms,
        }

    @classmethod
    def from_dict(cls, data: dict[str, Any]) -> "AuditResult":
        """Create from dictionary (e.g., from cache)."""
        insights = [
            InsightResult.from_dict(i) for i in data.get("insights", [])
        ]
        return cls(
            insights=insights,
            total_opportunity=data.get("total_opportunity", 0.0),
            by_category=data.get("by_category", {}),
            by_severity=data.get("by_severity", {}),
            suppressed_count=data.get("suppressed_count", 0),
            audit_duration_ms=data.get("audit_duration_ms", 0),
        )
