# backend/app/services/insight_engine/engine.py
"""
Insight Engine Service v2.0 - Orquestador principal.

Issue #506: Motor de detección automática de anomalías con feedback loop.

Responsabilidades:
- Ejecutar todas las reglas de insights
- Filtrar insights suprimidos (snooze/dismiss)
- Ordenar por impacto económico
- Registrar feedback del usuario
- Cachear resultados de audits
"""

import time
from datetime import datetime, timedelta, timezone
from typing import Any, Optional
from uuid import UUID, uuid4

import structlog
from sqlalchemy.orm import Session, lazyload

from app.models.insight_feedback import InsightFeedback, InsightFeedbackAction

from .models import AuditResult, InsightResult
from .rules import ALL_RULES
from .rules.base import BaseInsightRule

logger = structlog.get_logger(__name__)


# Default snooze duration
DEFAULT_SNOOZE_DAYS = 7


class InsightEngineService:
    """
    Servicio principal del Insight Engine.

    Orquesta la ejecución de reglas, manejo de feedback y cache de resultados.

    Uso:
        # Ejecutar audit completo
        result = await insight_engine_service.run_audit(db, pharmacy_id)

        # Registrar feedback
        success = await insight_engine_service.submit_feedback(
            db, pharmacy_id, insight_hash, "snooze"
        )

        # Obtener insights activos (con cache)
        insights = await insight_engine_service.get_active_insights(db, pharmacy_id)
    """

    def __init__(self):
        """Initialize service with all rules."""
        self.rules: list[BaseInsightRule] = [RuleClass() for RuleClass in ALL_RULES]
        # NOTE: In-memory cache works for single-worker deployments.
        # For Render multi-worker (Gunicorn), consider Redis-based cache.
        # Current implementation is acceptable for low-traffic initial deployment.
        self._audit_cache: dict[str, dict] = {}  # {pharmacy_id: {result, timestamp}}
        self._cache_ttl_minutes = 5

        logger.info(
            "insight_engine.initialized",
            rules_count=len(self.rules),
            rule_codes=[r.rule_code for r in self.rules],
        )

    async def run_audit(
        self,
        db: Session,
        pharmacy_id: UUID,
        custom_config: Optional[dict[str, dict[str, Any]]] = None,
        force: bool = False,
    ) -> AuditResult:
        """
        Ejecuta un audit completo de insights para una farmacia.

        Args:
            db: Sesión SQLAlchemy
            pharmacy_id: UUID de la farmacia
            custom_config: Configuración personalizada por regla {rule_code: config}
            force: Si True, ignora cache

        Returns:
            AuditResult con insights ordenados por economic_value DESC
        """
        start_time = time.time()
        pharmacy_key = str(pharmacy_id)

        # Check cache (unless force=True)
        if not force and pharmacy_key in self._audit_cache:
            cached = self._audit_cache[pharmacy_key]
            cache_age = (datetime.now(timezone.utc) - cached["timestamp"]).total_seconds()
            if cache_age < self._cache_ttl_minutes * 60:
                logger.debug(
                    "insight_engine.cache_hit",
                    pharmacy_id=pharmacy_key,
                    cache_age_seconds=int(cache_age),
                )
                return cached["result"]

        logger.info(
            "insight_engine.audit_start",
            pharmacy_id=pharmacy_key,
            rules_count=len(self.rules),
        )

        # 1. Cargar feedbacks activos
        active_suppressions = await self._get_active_suppressions(db, pharmacy_id)
        suppressed_hashes = set(active_suppressions.keys())

        # 2. Ejecutar cada regla
        all_insights: list[InsightResult] = []
        suppressed_count = 0

        for rule in self.rules:
            try:
                # Get custom config for this rule if provided
                rule_config = {}
                if custom_config and rule.rule_code in custom_config:
                    rule_config = custom_config[rule.rule_code]

                insight = await rule.evaluate(db, pharmacy_id, rule_config)

                if insight:
                    # Check if suppressed
                    if insight.insight_hash in suppressed_hashes:
                        suppressed_count += 1
                        logger.debug(
                            "insight_engine.insight_suppressed",
                            rule_code=rule.rule_code,
                            insight_hash=insight.insight_hash,
                        )
                    else:
                        all_insights.append(insight)

            except Exception as e:
                logger.error(
                    "insight_engine.rule_error",
                    rule_code=rule.rule_code,
                    error=str(e),
                    exc_info=True,
                )

        # 3. Ordenar por economic_value DESC
        all_insights.sort(key=lambda x: -x.economic_value)

        # 4. Crear resultado
        elapsed_ms = int((time.time() - start_time) * 1000)

        result = AuditResult(
            insights=all_insights,
            suppressed_count=suppressed_count,
            audit_duration_ms=elapsed_ms,
        )

        # 5. Update cache
        self._audit_cache[pharmacy_key] = {
            "result": result,
            "timestamp": datetime.now(timezone.utc),
        }

        logger.info(
            "insight_engine.audit_complete",
            pharmacy_id=pharmacy_key,
            insights_count=len(all_insights),
            suppressed_count=suppressed_count,
            total_opportunity=result.total_opportunity,
            duration_ms=elapsed_ms,
        )

        return result

    async def get_active_insights(
        self,
        db: Session,
        pharmacy_id: UUID,
        category: Optional[str] = None,
        severity: Optional[str] = None,
    ) -> AuditResult:
        """
        Obtiene insights activos para una farmacia (usa cache si disponible).

        Args:
            db: Sesión SQLAlchemy
            pharmacy_id: UUID de la farmacia
            category: Filtrar por categoría (optional)
            severity: Filtrar por severidad (optional)

        Returns:
            AuditResult filtrado según parámetros
        """
        # Run audit (will use cache if available)
        result = await self.run_audit(db, pharmacy_id)

        # Apply filters if specified
        if category or severity:
            filtered_insights = [
                i for i in result.insights
                if (not category or i.category == category)
                and (not severity or i.severity == severity)
            ]

            return AuditResult(
                insights=filtered_insights,
                suppressed_count=result.suppressed_count,
                audit_duration_ms=result.audit_duration_ms,
            )

        return result

    async def submit_feedback(
        self,
        db: Session,
        pharmacy_id: UUID,
        insight_hash: str,
        action: str,
        rule_code: Optional[str] = None,
        notes: Optional[str] = None,
        snooze_days: int = DEFAULT_SNOOZE_DAYS,
    ) -> bool:
        """
        Registra feedback del usuario sobre un insight.

        Args:
            db: Sesión SQLAlchemy
            pharmacy_id: UUID de la farmacia
            insight_hash: Hash del insight (para identificación)
            action: "snooze", "dismiss", o "resolve"
            rule_code: Código de la regla (opcional, para logging)
            notes: Notas del usuario (opcional)
            snooze_days: Días de snooze (solo si action="snooze")

        Returns:
            True si el feedback se registró correctamente
        """
        try:
            # Validate action
            valid_actions = {a.value for a in InsightFeedbackAction}
            if action not in valid_actions:
                logger.warning(
                    "insight_engine.invalid_action",
                    action=action,
                    valid_actions=list(valid_actions),
                )
                return False

            # Calculate snooze_until if applicable
            snoozed_until = None
            if action == InsightFeedbackAction.SNOOZE.value:
                snoozed_until = datetime.now(timezone.utc) + timedelta(days=snooze_days)

            # Create feedback record
            feedback = InsightFeedback(
                id=uuid4(),
                pharmacy_id=pharmacy_id,
                insight_rule_code=rule_code or "UNKNOWN",
                insight_hash=insight_hash,
                action=action,
                snoozed_until=snoozed_until,
                notes=notes,
            )

            db.add(feedback)
            db.commit()

            # Invalidate cache for this pharmacy
            pharmacy_key = str(pharmacy_id)
            if pharmacy_key in self._audit_cache:
                del self._audit_cache[pharmacy_key]

            logger.info(
                "insight_engine.feedback_submitted",
                pharmacy_id=str(pharmacy_id),
                insight_hash=insight_hash,
                action=action,
                rule_code=rule_code,
                snoozed_until=snoozed_until.isoformat() if snoozed_until else None,
            )

            return True

        except Exception as e:
            logger.error(
                "insight_engine.feedback_error",
                pharmacy_id=str(pharmacy_id),
                insight_hash=insight_hash,
                error=str(e),
                exc_info=True,
            )
            db.rollback()
            return False

    async def _get_active_suppressions(
        self,
        db: Session,
        pharmacy_id: UUID,
    ) -> dict[str, InsightFeedback]:
        """
        Obtiene feedbacks activos que suprimen insights.

        Un feedback suprime un insight si:
        - Es DISMISS o RESOLVE (permanente)
        - Es SNOOZE y snoozed_until > ahora

        Returns:
            Dict de {insight_hash: InsightFeedback}
        """
        now = datetime.now(timezone.utc)

        # Query all feedbacks for this pharmacy
        # Use lazyload('*') to prevent N+1 queries on relationships
        feedbacks = (
            db.query(InsightFeedback)
            .options(lazyload("*"))
            .filter(InsightFeedback.pharmacy_id == pharmacy_id)
            .all()
        )

        # Filter to active suppressions
        suppressions = {}
        for fb in feedbacks:
            if fb.action in (
                InsightFeedbackAction.DISMISS.value,
                InsightFeedbackAction.RESOLVE.value,
            ):
                suppressions[fb.insight_hash] = fb
            elif fb.action == InsightFeedbackAction.SNOOZE.value:
                if fb.snoozed_until and fb.snoozed_until > now:
                    suppressions[fb.insight_hash] = fb

        return suppressions

    async def get_feedback_history(
        self,
        db: Session,
        pharmacy_id: UUID,
        limit: int = 50,
    ) -> list[dict[str, Any]]:
        """
        Obtiene historial de feedback para una farmacia.

        Args:
            db: Sesión SQLAlchemy
            pharmacy_id: UUID de la farmacia
            limit: Máximo de registros a retornar

        Returns:
            Lista de feedbacks ordenados por fecha DESC
        """
        feedbacks = (
            db.query(InsightFeedback)
            .filter(InsightFeedback.pharmacy_id == pharmacy_id)
            .order_by(InsightFeedback.created_at.desc())
            .limit(limit)
            .all()
        )

        return [
            {
                "id": str(fb.id),
                "rule_code": fb.insight_rule_code,
                "insight_hash": fb.insight_hash,
                "action": fb.action,
                "snoozed_until": fb.snoozed_until.isoformat() if fb.snoozed_until else None,
                "notes": fb.notes,
                "created_at": fb.created_at.isoformat(),
                "is_active": fb.is_active_suppression,
            }
            for fb in feedbacks
        ]

    def invalidate_cache(self, pharmacy_id: Optional[UUID] = None) -> None:
        """
        Invalida cache de audits.

        Args:
            pharmacy_id: Si se especifica, solo invalida esa farmacia.
                        Si None, invalida todo el cache.
        """
        if pharmacy_id:
            pharmacy_key = str(pharmacy_id)
            if pharmacy_key in self._audit_cache:
                del self._audit_cache[pharmacy_key]
                logger.debug(
                    "insight_engine.cache_invalidated",
                    pharmacy_id=pharmacy_key,
                )
        else:
            self._audit_cache.clear()
            logger.debug("insight_engine.cache_cleared")

    def get_available_rules(self) -> list[dict[str, Any]]:
        """
        Retorna información sobre las reglas disponibles.

        Returns:
            Lista de reglas con su configuración default
        """
        return [
            {
                "rule_code": rule.rule_code,
                "category": rule.category,
                "severity": rule.severity,
                "default_config": rule.default_config,
            }
            for rule in self.rules
        ]


# Singleton instance
insight_engine_service = InsightEngineService()
