"""
Tier 1 Rules: Ingredient/Product -> Symptom Mapping

This module implements the first layer of the Sandwich Strategy:
deterministic rules based on ingredients and product types.

Philosophy:
- Map ingredients to SYMPTOMS, not to product types
- "Ibuprofeno" -> "dolor" (not "medicamento")
- "Arnica" -> "hematomas" (not "natural")

The rules here are DETERMINISTIC and take priority over LLM suggestions
for Tier 1 and Tier 2 categories (see symptom_taxonomy.py).
"""

from dataclasses import dataclass
from typing import Dict, List, Optional, Set, Tuple
import re


@dataclass
class RuleMatch:
    """Result of a rule match."""
    necesidad: str
    confidence: float
    matched_term: str
    rule_type: str  # "ingredient", "product_type", "brand", "keyword"


# =============================================================================
# TIER 1: INGREDIENT -> SYMPTOM MAPPING
# =============================================================================
# These are active pharmaceutical ingredients and natural compounds
# that definitively indicate the symptom being treated.

INGREDIENT_TO_SYMPTOM: Dict[str, str] = {
    # === DOLOR Y FIEBRE ===
    # Fármacos
    "ibuprofeno": "dolor_fiebre",
    "paracetamol": "dolor_fiebre",
    "dexketoprofeno": "dolor_muscular",
    "naproxeno": "dolor_muscular",
    "diclofenaco": "dolor_muscular",
    "metamizol": "dolor_fiebre",
    "aspirina": "dolor_fiebre",
    "acido acetilsalicilico": "dolor_fiebre",

    # Naturales para dolor
    "arnica": "hematomas",
    "harpagofito": "dolor_articular",
    "capsaicina": "dolor_muscular",
    "mentol": "dolor_muscular",
    "alcanfor": "dolor_muscular",
    "salicilato": "dolor_muscular",

    # === RESPIRATORIO ===
    # Antitusivos
    "dextrometorfano": "tos_seca",
    "codeina": "tos_seca",
    "cloperastina": "tos_seca",

    # Mucolíticos/Expectorantes
    "acetilcisteina": "mucosidad_respiratoria",
    "ambroxol": "mucosidad_respiratoria",
    "bromhexina": "mucosidad_respiratoria",
    "carbocisteina": "mucosidad_respiratoria",
    "guaifenesina": "mucosidad_respiratoria",

    # Descongestivos
    "oximetazolina": "congestion_nasal",
    "xilometazolina": "congestion_nasal",
    "fenilefrina": "congestion_nasal",
    "pseudoefedrina": "gripe_resfriado",

    # Antihistamínicos
    "loratadina": "alergias",
    "cetirizina": "alergias",
    "ebastina": "alergias",
    "difenhidramina": "alergias",
    "clorfenamina": "gripe_resfriado",

    # Naturales respiratorios
    "eucalipto": "congestion_nasal",
    "mentol": "congestion_nasal",
    "propolis": "dolor_garganta",
    "miel": "tos_seca",
    "tomillo": "mucosidad_respiratoria",

    # === DIGESTIVO ===
    # Antiácidos
    "omeprazol": "acidez_reflujo",
    "pantoprazol": "acidez_reflujo",
    "ranitidina": "acidez_reflujo",
    "almagato": "acidez_reflujo",
    "hidroxido de aluminio": "acidez_reflujo",
    "bicarbonato": "acidez_reflujo",

    # Antidiarreicos
    "loperamida": "diarrea",
    "racecadotrilo": "diarrea",

    # Laxantes
    "bisacodilo": "estrenimiento",
    "picosulfato": "estrenimiento",
    "macrogol": "estrenimiento",
    "lactulosa": "estrenimiento",
    "sen": "estrenimiento",
    "plantago": "estrenimiento",
    "fibra": "estrenimiento",

    # Antiflatulentos
    "simeticona": "gases_flatulencia",
    "dimeticona": "gases_flatulencia",

    # Antieméticos
    "dimenhidrinato": "nauseas_vomitos",
    "biodramina": "nauseas_vomitos",
    "jengibre": "nauseas_vomitos",

    # Probióticos
    "lactobacillus": "probioticos",
    "bifidobacterium": "probioticos",
    "saccharomyces": "probioticos",
    "probiotico": "probioticos",

    # === PIEL ===
    # Protección solar
    "octinoxato": "proteccion_solar",
    "avobenzona": "proteccion_solar",
    "oxibenzona": "proteccion_solar",
    "titanio": "proteccion_solar",
    "zinc oxide": "proteccion_solar",
    "spf": "proteccion_solar",

    # Antiacné
    "peroxido de benzoilo": "acne",
    "acido salicilico": "acne",
    "niacinamida": "acne",
    "adapaleno": "acne",

    # Antifúngicos
    "clotrimazol": "hongos_piel",
    "miconazol": "hongos_piel",
    "terbinafina": "hongos_piel",
    "ketoconazol": "hongos_piel",

    # Cicatrizantes
    "centella asiatica": "heridas_cicatrizacion",
    "rosa mosqueta": "heridas_cicatrizacion",
    "alantoina": "heridas_cicatrizacion",

    # Antiedad
    "retinol": "arrugas_antiedad",
    "acido hialuronico": "hidratacion_facial",
    "vitamina c": "arrugas_antiedad",
    "colageno": "arrugas_antiedad",
    "peptidos": "arrugas_antiedad",
    "niacinamide": "arrugas_antiedad",

    # Despigmentantes
    "hidroquinona": "manchas",
    "acido kojico": "manchas",
    "arbutina": "manchas",
    "acido tranexamico": "manchas",

    # Calmantes/Atopia
    "avena": "piel_atopica",
    "urea": "hidratacion_corporal",
    "ceramidas": "piel_atopica",

    # === BUCAL ===
    "fluor": "higiene_dental",
    "clorhexidina": "encias",
    "hexetidina": "higiene_dental",
    "triclosan": "higiene_dental",

    # === OCULAR ===
    "hialuronato": "ojo_seco",
    "carmelosa": "ojo_seco",
    "lagrimas artificiales": "ojo_seco",

    # === CAPILAR ===
    "minoxidil": "caida_cabello",
    "finasterida": "caida_cabello",
    "biotina": "caida_cabello",
    "ketoconazol": "caspa",
    "piroctona": "caspa",
    "zinc piritiona": "caspa",
    "permetrina": "piojos",
    "dimeticona": "piojos",

    # === SUEÑO/BIENESTAR ===
    "melatonina": "sueno_insomnio",
    "valeriana": "sueno_insomnio",
    "pasiflora": "sueno_insomnio",
    "triptofano": "sueno_insomnio",
    "magnesio": "magnesio",
    "ginseng": "energia_vitalidad",
    "guarana": "energia_vitalidad",
    "cafeina": "energia_vitalidad",
    "hiperico": "estado_animo",
    "rhodiola": "estres_ansiedad",
    "ashwagandha": "estres_ansiedad",
    "gaba": "estres_ansiedad",

    # === CIRCULATORIO ===
    "diosmina": "piernas_cansadas",
    "hesperidina": "piernas_cansadas",
    "rusco": "piernas_cansadas",
    "castaño de indias": "varices",
    "hamamelis": "hemorroides",
    "lidocaina": "hemorroides",

    # === NUTRICIÓN ===
    "omega 3": "omega3",
    "omega-3": "omega3",
    "dha": "omega3",
    "epa": "omega3",
    "aceite de pescado": "omega3",
    "hierro": "hierro",
    "ferritina": "hierro",
    "acido folico": "embarazo_lactancia",
    "calcio": "calcio_vitamina_d",
    "vitamina d": "calcio_vitamina_d",
    "colecalciferol": "calcio_vitamina_d",

    # === INFANTIL ===
    "simeticona bebe": "colicos_bebe",
    "pantenol": "dermatitis_panal",
    "oxido de zinc": "dermatitis_panal",

    # === DOLOR Y FIEBRE (Ampliado) ===
    "ketoprofeno": "dolor_muscular",
    "piroxicam": "dolor_articular",
    "indometacina": "dolor_articular",
    "meloxicam": "dolor_articular",
    "aceclofenaco": "dolor_articular",
    "etoricoxib": "dolor_articular",
    "celecoxib": "dolor_articular",
    "tramadol": "dolor_fiebre",
    "codeina fosfato": "dolor_fiebre",
    "buprenorfina": "dolor_fiebre",
    "glucosamina": "dolor_articular",
    "condroitina": "dolor_articular",
    "msm": "dolor_articular",
    "colageno tipo ii": "dolor_articular",

    # === RESPIRATORIO (Ampliado) ===
    "levodropropizina": "tos_seca",
    "butamirato": "tos_seca",
    "benzonatato": "tos_seca",
    "ipratropio": "congestion_nasal",
    "salbutamol": "asma_bronquitis",
    "terbutalina": "asma_bronquitis",
    "budesonida": "asma_bronquitis",
    "beclometasona": "asma_bronquitis",
    "montelukast": "alergias",
    "bilastina": "alergias",
    "fexofenadina": "alergias",
    "desloratadina": "alergias",
    "rupatadina": "alergias",
    "cromoglicato": "alergias",
    "agua de mar": "congestion_nasal",
    "solucion salina": "congestion_nasal",
    "pelargonium": "gripe_resfriado",
    "equinacea": "gripe_resfriado",
    "sauco": "gripe_resfriado",
    "malvavisco": "dolor_garganta",
    "liquen de islandia": "dolor_garganta",
    "erísimo": "dolor_garganta",

    # === DIGESTIVO (Ampliado) ===
    "esomeprazol": "acidez_reflujo",
    "lansoprazol": "acidez_reflujo",
    "rabeprazol": "acidez_reflujo",
    "famotidina": "acidez_reflujo",
    "alginato": "acidez_reflujo",
    "domperidona": "nauseas_vomitos",
    "metoclopramida": "nauseas_vomitos",
    "ondansetron": "nauseas_vomitos",
    "mebeverina": "colon_irritable",
    "trimebutina": "colon_irritable",
    "pinaverio": "colon_irritable",
    "butilescopolamina": "gases_flatulencia",
    "carbon activado": "gases_flatulencia",
    "alcachofa": "digestivo",
    "boldo": "digestivo",
    "cardo mariano": "digestivo",
    "papaina": "digestivo",
    "enzimas digestivas": "digestivo",
    "psyllium": "estrenimiento",
    "ispagula": "estrenimiento",
    "aceite de ricino": "estrenimiento",
    "glicerina": "estrenimiento",
    "lactitol": "estrenimiento",
    "polietilenglicol": "estrenimiento",
    "tanino": "diarrea",
    "atapulgita": "diarrea",
    "diosmectita": "diarrea",
    "nifuroxazida": "diarrea",

    # === PIEL (Ampliado) ===
    "hidrocortisona": "dermatitis",
    "betametasona": "dermatitis",
    "prednicarbato": "dermatitis",
    "clobetasol": "dermatitis",
    "tacrolimus": "piel_atopica",
    "pimecrolimus": "piel_atopica",
    "coaltar": "psoriasis",
    "alquitran hulla": "psoriasis",
    "calcipotriol": "psoriasis",
    "nistatina": "hongos_piel",
    "econazol": "hongos_piel",
    "bifonazol": "hongos_piel",
    "sertaconazol": "hongos_piel",
    "ciclopirox": "hongos_piel",
    "permetrina piel": "sarna",
    "ivermectina": "sarna",
    "aciclovir": "herpes",
    "penciclovir": "herpes",
    "docosanol": "herpes",
    "mupirocina": "heridas_infeccion",
    "fusidico": "heridas_infeccion",
    "neomicina": "heridas_infeccion",
    "bacitracina": "heridas_infeccion",
    "povidona yodada": "heridas_cicatrizacion",
    "cloruro de benzalconio": "heridas_cicatrizacion",
    "sulfadiazina argentica": "quemaduras",
    "biafine": "quemaduras",
    "trolamina": "quemaduras",
    "escina": "hematomas",
    "vitamina k": "hematomas",
    "pentosano polisulfato": "hematomas",

    # === OCULAR (Ampliado) ===
    "povidona": "ojo_seco",
    "hidroxipropil": "ojo_seco",
    "trehalosa": "ojo_seco",
    "hipromelosa": "ojo_seco",
    "polividona": "ojo_seco",
    "cromoglicato sodico": "alergia_ocular",
    "olopatadina": "alergia_ocular",
    "ketotifeno": "alergia_ocular",
    "emedastina": "alergia_ocular",
    "azitromicina oftalmico": "conjuntivitis",
    "tobramicina": "conjuntivitis",
    "cloranfenicol": "conjuntivitis",
    "eufrasia": "ojo_seco",
    "manzanilla ocular": "ojo_seco",

    # === SALUD FEMENINA (Nuevo) ===
    "clotrimazol vaginal": "candidiasis_vaginal",
    "miconazol vaginal": "candidiasis_vaginal",
    "fenticonazol": "candidiasis_vaginal",
    "metronidazol vaginal": "vaginosis",
    "acido lactico": "higiene_intima",
    "lactobacillus vaginal": "flora_vaginal",
    "fitoestrógenos": "menopausia",
    "isoflavonas": "menopausia",
    "cimicifuga": "menopausia",
    "trébol rojo": "menopausia",
    "salvia": "menopausia",
    "sauzgatillo": "ciclo_menstrual",
    "onagra": "ciclo_menstrual",
    "hierro bisglicina": "embarazo_lactancia",
    "iodo": "embarazo_lactancia",
    "yodo": "embarazo_lactancia",

    # === OIDOS (Nuevo) ===
    "agua de mar oidos": "higiene_oidos",
    "aceite oliva oidos": "cerumen",
    "carboglicerina": "cerumen",
    "peroxido de hidrogeno oidos": "cerumen",
    "lidocaina otica": "dolor_oidos",
    "fenazona": "dolor_oidos",
    "neomicina otica": "otitis",
    "polimixina": "otitis",
    "ciprofloxacino otico": "otitis",

    # === NUTRICIÓN CLÍNICA (Nuevo) ===
    "proteina suero": "nutricion_clinica",
    "caseina": "nutricion_clinica",
    "maltodextrina": "nutricion_clinica",
    "mct": "nutricion_clinica",
    "trigliceridos cadena media": "nutricion_clinica",
    "glutamina": "nutricion_clinica",
    "arginina": "nutricion_clinica",
    "carnitina": "energia_vitalidad",
    "creatina": "energia_vitalidad",
    "taurina": "energia_vitalidad",
    "coenzima q10": "energia_vitalidad",
    "ubiquinol": "energia_vitalidad",
    "jalea real": "energia_vitalidad",
    "propóleo": "dolor_garganta",

    # === VITAMINAS Y MINERALES (Ampliado) ===
    "vitamina a": "vitaminas_minerales",
    "vitamina b1": "vitaminas_minerales",
    "vitamina b2": "vitaminas_minerales",
    "vitamina b3": "vitaminas_minerales",
    "vitamina b5": "vitaminas_minerales",
    "vitamina b6": "vitaminas_minerales",
    "vitamina b12": "vitaminas_minerales",
    "vitamina e": "vitaminas_minerales",
    "vitamina k": "vitaminas_minerales",
    "zinc": "vitaminas_minerales",
    "selenio": "vitaminas_minerales",
    "cobre": "vitaminas_minerales",
    "manganeso": "vitaminas_minerales",
    "cromo": "vitaminas_minerales",
    "molibdeno": "vitaminas_minerales",
    "potasio": "vitaminas_minerales",
    "sodio": "vitaminas_minerales",
    "fosforo": "vitaminas_minerales",
    "silicio": "vitaminas_minerales",
    "boro": "vitaminas_minerales",
    "levadura cerveza": "vitaminas_minerales",
    "spirulina": "vitaminas_minerales",
    "chlorella": "vitaminas_minerales",

    # === SISTEMA URINARIO (Nuevo) ===
    "arandano rojo": "infeccion_urinaria",
    "d-manosa": "infeccion_urinaria",
    "gayuba": "infeccion_urinaria",
    "brezo": "infeccion_urinaria",
    "ortosifon": "retencion_liquidos",
    "cola de caballo": "retencion_liquidos",
    "diente de leon": "retencion_liquidos",
    "serenoa repens": "prostata",
    "saw palmetto": "prostata",
    "pygeum": "prostata",
    "ortiga": "prostata",
    "licopeno": "prostata",

    # === HUESOS Y ARTICULACIONES (Nuevo) ===
    "calcio citrato": "calcio_vitamina_d",
    "calcio carbonato": "calcio_vitamina_d",
    "vitamina d3": "calcio_vitamina_d",
    "calcifediol": "calcio_vitamina_d",
    "magnesio citrato": "magnesio",
    "magnesio bisglicinato": "magnesio",
    "magnesio marino": "magnesio",
    "acido hialuronico oral": "dolor_articular",
    "boswellia": "dolor_articular",
    "curcuma": "dolor_articular",
    "jengibre articular": "dolor_articular",
    "bambu": "huesos",
    "silicio organico": "huesos",
}


# =============================================================================
# TIER 1: PRODUCT TYPE -> SYMPTOM MAPPING
# =============================================================================
# Physical product types that definitively indicate the symptom.

PRODUCT_TYPE_TO_SYMPTOM: Dict[str, str] = {
    # Dispositivos dolor
    "parche termico": "dolor_muscular",
    "thermacare": "dolor_muscular",
    "bolsa frio": "hematomas",
    "bolsa calor": "dolor_muscular",
    "muñequera": "dolor_articular",
    "rodillera": "dolor_articular",
    "tobillera": "dolor_articular",
    "codera": "dolor_articular",
    "faja lumbar": "dolor_muscular",

    # Bucal
    "cepillo dental": "higiene_dental",
    "cepillo electrico": "higiene_dental",
    "seda dental": "higiene_dental",
    "hilo dental": "higiene_dental",
    "irrigador": "higiene_dental",
    "colutorio": "higiene_dental",
    "enjuague bucal": "higiene_dental",
    "pasta dientes": "higiene_dental",
    "blanqueador dental": "blanqueamiento",
    "adhesivo protesis": "protesis_dental",
    "limpiador protesis": "protesis_dental",

    # Ortopedia/Movilidad
    "andador": "ayudas_tecnicas",
    "muleta": "ayudas_tecnicas",
    "baston": "ayudas_tecnicas",
    "silla ruedas": "ayudas_tecnicas",
    "silla wc": "ayudas_tecnicas",
    "elevador wc": "ayudas_tecnicas",
    "grua": "ayudas_tecnicas",
    "media compresion": "varices",
    "venda elastica": "compresion_vendajes",
    "venda cohesiva": "compresion_vendajes",

    # Botiquín
    "tirita": "heridas_apositos",
    "aposito": "heridas_apositos",
    "gasa": "heridas_apositos",
    "esparadrapo": "heridas_apositos",
    "termometro": "termometros",
    "tensiometro": "tension_arterial",
    "pulsioximetro": "material_sanitario",
    "jeringuilla": "material_sanitario",
    "guante latex": "material_sanitario",
    "guante nitrilo": "material_sanitario",
    "mascarilla": "material_sanitario",

    # Tests
    "test embarazo": "test_diagnostico",
    "test ovulacion": "fertilidad",
    "test covid": "test_diagnostico",
    "test antigenos": "test_diagnostico",
    "glucometro": "test_diagnostico",
    "tira reactiva": "test_diagnostico",

    # Sexual
    "preservativo": "salud_sexual",
    "condon": "salud_sexual",
    "lubricante intimo": "salud_sexual",
    "lubricante sexual": "salud_sexual",
    "gel lubricante intimo": "salud_sexual",
    # NOTE: Plain "lubricante" removed - too generic, matches eye drops (LUBRISTIL, AQUORAL)
    "gel intimo": "higiene_intima",
    "copa menstrual": "higiene_intima",
    "tampax": "higiene_intima",
    "compresa": "higiene_intima",

    # Infantil
    "panal": "higiene_bebe",
    "toallita bebe": "higiene_bebe",
    "leche infantil": "alimentacion_bebe",
    "potito": "alimentacion_bebe",
    "papilla": "alimentacion_bebe",
    "biberon": "alimentacion_bebe",
    "chupete": "alimentacion_bebe",
    "mordedor": "denticion",
    "sacaleches": "embarazo_lactancia",

    # Ocular
    "lentilla": "lentillas",
    "solucion lentillas": "lentillas",
    "colirio": "ojo_seco",
    "lagrima artificial": "ojo_seco",

    # Capilar
    "champu anticaida": "caida_cabello",
    "champu anticaspa": "caspa",
    "locion capilar": "caida_cabello",
    "tratamiento antipiojos": "piojos",

    # === ORTOPEDIA/AYUDAS TÉCNICAS (Ampliado) ===
    "protector dedil": "ortopedia_pie",
    "protector juanete": "ortopedia_pie",
    "plantilla": "ortopedia_pie",
    "talonera": "ortopedia_pie",
    "corrector hallux": "ortopedia_pie",
    "almohadilla metatarsal": "ortopedia_pie",
    "separador dedos": "ortopedia_pie",
    "cabestrillo": "ayudas_tecnicas",
    "collar cervical": "dolor_muscular",
    "cinturon abdominal": "dolor_muscular",
    "muñequera tunel": "dolor_articular",
    "ortesis": "ayudas_tecnicas",
    "ferula": "ayudas_tecnicas",
    "cojin antiescaras": "material_sanitario",
    "colchon antiescaras": "material_sanitario",

    # === HIGIENE OIDOS (Nuevo) ===
    "spray oidos": "higiene_oidos",
    "limpieza oidos": "higiene_oidos",
    "tapones oidos": "higiene_oidos",

    # === FACIAL/CORPORAL (Ampliado) ===
    "limpiador facial": "limpieza_facial",
    "desmaquillante": "limpieza_facial",
    "tonico facial": "limpieza_facial",
    "agua micelar": "limpieza_facial",
    "exfoliante": "limpieza_facial",
    "mascarilla facial": "hidratacion_facial",
    "serum": "arrugas_antiedad",
    "contorno ojos": "arrugas_antiedad",
    "crema antiarrugas": "arrugas_antiedad",
    "gel anticelulitico": "celulitis",
    "crema reafirmante": "celulitis",
    "antiestrias": "estrias",
    "aceite antiestrias": "estrias",
    "autobronceador": "bronceado",

    # === SALUD FEMENINA (Nuevo) ===
    "ovulo vaginal": "candidiasis_vaginal",
    "gel vaginal": "higiene_intima",
    "jabon intimo": "higiene_intima",
    "hidratante vaginal": "higiene_intima",
    "flora intima": "flora_vaginal",
    "probiotico vaginal": "flora_vaginal",
    "test menopausia": "test_diagnostico",
    "test fertilidad": "fertilidad",

    # === TESTS Y AUTODIAGNÓSTICO (Ampliado) ===
    "test nasal": "test_diagnostico",
    "test antigenos nasal": "test_diagnostico",
    "autotest": "test_diagnostico",
    "autodiagnostico": "test_diagnostico",
    "test saliva": "test_diagnostico",
    "test drogas": "test_diagnostico",
    "alcoholimetro": "test_diagnostico",
    "test celiaco": "test_diagnostico",
    "test alergia": "test_diagnostico",
    "test vih": "test_diagnostico",
    "test hepatitis": "test_diagnostico",
    "medidor glucosa": "diabetes",
    "lanceta": "diabetes",
    "tira glucosa": "diabetes",

    # === NUTRICIÓN INFANTIL (Ampliado) ===
    "cereales bebe": "alimentacion_bebe",
    "galleta bebe": "alimentacion_bebe",
    "snack bebe": "alimentacion_bebe",
    "snack zanahoria": "alimentacion_bebe",
    "snack fruta": "alimentacion_bebe",
    "pure fruta": "alimentacion_bebe",
    "pure verdura": "alimentacion_bebe",
    "leche crecimiento": "alimentacion_bebe",
    "leche etapa 1": "alimentacion_bebe",
    "leche etapa 2": "alimentacion_bebe",
    "leche etapa 3": "alimentacion_bebe",
    "formula infantil": "alimentacion_bebe",
    "leche hidrolizada": "alimentacion_bebe",
    "leche antirregurgitacion": "colicos_bebe",
    "leche anticólico": "colicos_bebe",

    # === NUTRICIÓN CLÍNICA (Nuevo) ===
    "nutricion enteral": "nutricion_clinica",
    "batido hipercalorico": "nutricion_clinica",
    "suplemento proteico": "nutricion_clinica",
    "espesante": "nutricion_clinica",
    "dieta liquida": "nutricion_clinica",

    # === BOTIQUÍN/CURAS (Ampliado) ===
    "sutura adhesiva": "heridas_apositos",
    "steri-strip": "heridas_apositos",
    "parche ocular": "material_sanitario",
    "venda tubular": "compresion_vendajes",
    "malla tubular": "compresion_vendajes",
    "media compresion": "varices",
    "calcetin compresion": "varices",
    "manga compresion": "varices",
    "faja postparto": "embarazo_lactancia",
    "faja postoperatorio": "material_sanitario",

    # === INCONTINENCIA (Nuevo) ===
    "absorvente incontinencia": "incontinencia",
    "panal adulto": "incontinencia",
    "braga absorbente": "incontinencia",
    "protector cama": "incontinencia",
    "empapador": "incontinencia",

    # === DEPORTE/FITNESS (Nuevo) ===
    "vendaje deportivo": "compresion_vendajes",
    "tape deportivo": "compresion_vendajes",
    "spray frio": "dolor_muscular",
    "crema deportiva": "dolor_muscular",
    "gel efecto frio": "dolor_muscular",
    "gel efecto calor": "dolor_muscular",
    "sales bano": "dolor_muscular",
    "magnesio deportivo": "magnesio",
    "bebida isotonica": "hidratacion",
    "gel energetico": "energia_vitalidad",
    "barrita energetica": "energia_vitalidad",

    # === REPELENTES E INSECTOS (Nuevo) ===
    "repelente mosquitos": "picaduras",
    "antimosquitos": "picaduras",
    "after bite": "picaduras",
    "crema picaduras": "picaduras",
    "pulsera antimosquitos": "picaduras",
    "insecticida piojos": "piojos",
    "lendrera": "piojos",

    # === DENTAL ESPECIALIZADO (Ampliado) ===
    "tableta limpiadora": "protesis_dental",
    "limpiador dentadura": "protesis_dental",
    "cepillo ortodoncia": "higiene_dental",
    "cepillo interdental": "higiene_dental",
    "cera ortodoncia": "higiene_dental",
    "gel dental sensible": "sensibilidad_dental",
    "pasta sensibilidad": "sensibilidad_dental",
    "recambio cepillo electrico": "higiene_dental",
    "cabezal recambio": "higiene_dental",
}


# =============================================================================
# TIER 1: BRAND -> SYMPTOM MAPPING
# =============================================================================
# Brands that are so specialized they definitively indicate the symptom.

BRAND_TO_SYMPTOM: Dict[str, str] = {
    # Dolor
    "thermacare": "dolor_muscular",
    "fisiocrem": "dolor_muscular",
    "radio salil": "dolor_muscular",
    "reflex": "dolor_muscular",
    "voltaren": "dolor_muscular",
    "arnidol": "hematomas",
    "thrombocid": "hematomas",

    # Solar
    "isdin": "proteccion_solar",  # Mayoría solar, pero no exclusivo
    "heliocare": "proteccion_solar",
    "avene solar": "proteccion_solar",
    "ladival": "proteccion_solar",
    "eucerin sun": "proteccion_solar",

    # Bucal
    "oral-b": "higiene_dental",
    "oral b": "higiene_dental",
    "philips sonicare": "higiene_dental",
    "lacer": "higiene_dental",
    "fluocaril": "higiene_dental",
    "vitis": "higiene_dental",
    "corega": "protesis_dental",
    "polident": "protesis_dental",
    "kukident": "protesis_dental",

    # Capilar
    "olistic": "caida_cabello",
    "pilexil": "caida_cabello",
    "lambdapil": "caida_cabello",
    "iraltone": "caida_cabello",
    "vichy dercos": "caida_cabello",

    # Infantil
    "blemil": "alimentacion_bebe",
    "nidina": "alimentacion_bebe",
    "nutriben": "alimentacion_bebe",
    "hero baby": "alimentacion_bebe",
    "nestle bebe": "alimentacion_bebe",
    "suavinex": "alimentacion_bebe",
    "chicco": "higiene_bebe",
    "dodot": "higiene_bebe",
    "huggies": "higiene_bebe",
    "mustela": "higiene_bebe",

    # Digestivo
    "almax": "acidez_reflujo",
    "gaviscon": "acidez_reflujo",
    "rennie": "acidez_reflujo",
    "ultra levura": "probioticos",
    "lactoflora": "probioticos",
    "prodefen": "probioticos",

    # Respiratorio
    "frenadol": "gripe_resfriado",
    "pharmagrip": "gripe_resfriado",
    "couldina": "gripe_resfriado",
    "bisolgrip": "gripe_resfriado",
    "bisolvon": "mucosidad_respiratoria",
    "flutox": "tos_seca",
    "rhinomer": "congestion_nasal",
    "rhinospray": "congestion_nasal",

    # Sexual
    "durex": "salud_sexual",
    "control preservativo": "salud_sexual",
    "control condom": "salud_sexual",
    # NOTE: "control" alone is too generic (appears in "control imperfecciones", etc.)

    # Piernas/Circulación
    "daflon": "piernas_cansadas",
    "venoruton": "piernas_cansadas",
    "thrombocid": "varices",

    # Omega/Nutrición
    "nordic oil": "omega3",
    "nutergia": "vitaminas_minerales",
    "solgar": "vitaminas_minerales",

    # Piel especializada
    "avene": "piel_atopica",
    "bioderma": "hidratacion_facial",
    "la roche posay": "piel_atopica",
    "cerave": "piel_atopica",
    "eucerin": "hidratacion_facial",
    "vichy": "hidratacion_facial",
    "sesderma": "arrugas_antiedad",
    "martiderm": "arrugas_antiedad",
    "medik8": "arrugas_antiedad",
    "skinceuticals": "arrugas_antiedad",
    "endocare": "arrugas_antiedad",

    # === DOLOR/MUSCULAR (Ampliado) ===
    "actipatch": "dolor_muscular",
    "artroben": "dolor_articular",
    "articare": "dolor_articular",
    "traumeel": "hematomas",
    "radian": "dolor_muscular",
    "flector": "dolor_muscular",
    "fastum": "dolor_muscular",
    "movial": "dolor_articular",
    "artrohelp": "dolor_articular",
    "epaplus": "dolor_articular",
    "colnatur": "dolor_articular",
    "arkoflex": "dolor_articular",
    "aquilea articulaciones": "dolor_articular",

    # === PROTECCIÓN SOLAR (Ampliado) ===
    "anthelios": "proteccion_solar",
    "avene sunscreen": "proteccion_solar",
    "bioderma photoderm": "proteccion_solar",
    "fotoprotector isdin": "proteccion_solar",
    "fusion water": "proteccion_solar",
    "pediatrics isdin": "proteccion_solar",
    "eucerin sun protection": "proteccion_solar",
    "vichy capital soleil": "proteccion_solar",
    "svr sun secure": "proteccion_solar",
    "sensilis sun": "proteccion_solar",
    "rilastil sun": "proteccion_solar",
    "apivita suncare": "proteccion_solar",
    "lancaster sun": "proteccion_solar",
    "piz buin": "proteccion_solar",

    # === BUCAL (Ampliado) ===
    "kin": "higiene_dental",
    "kin oro": "protesis_dental",
    "kin fresh": "higiene_dental",
    "kin gingival": "encias",
    "gum": "higiene_dental",
    "gum sunstar": "higiene_dental",
    "interprox": "higiene_dental",
    "phb": "higiene_dental",
    "sensodyne": "sensibilidad_dental",
    "parodontax": "encias",
    "perio-aid": "encias",
    "perioxidin": "encias",
    "desensin": "sensibilidad_dental",
    "elmex": "higiene_dental",
    "signal": "higiene_dental",
    "colgate": "higiene_dental",
    "listerine": "higiene_dental",
    "tantum verde": "dolor_garganta",

    # === CAPILAR (Ampliado) ===
    "ducray": "caida_cabello",
    "klorane": "caida_cabello",
    "rene furterer": "caida_cabello",
    "alpecin": "caida_cabello",
    "forte pharma capilar": "caida_cabello",
    "tricovel": "caida_cabello",
    "bioscalin": "caida_cabello",
    "apivita hair": "caida_cabello",
    "isdin lambdapil": "caida_cabello",
    "dercos": "caida_cabello",
    "h&s": "caspa",
    "head shoulders": "caspa",
    "nizoral": "caspa",
    "selsun": "caspa",
    "paranix": "piojos",
    "neositrin": "piojos",
    "fullmarks": "piojos",
    "goibi antipiojos": "piojos",
    "filvit": "piojos",
    "otc antipiojos": "piojos",

    # === ALIMENTACIÓN INFANTIL (Ampliado) ===
    "hero solo": "alimentacion_bebe",
    "hero pedialac": "alimentacion_bebe",
    "nestle nan": "alimentacion_bebe",
    "almirón": "alimentacion_bebe",
    "ordesa": "alimentacion_bebe",
    "similac": "alimentacion_bebe",
    "enfamil": "alimentacion_bebe",
    "aptamil": "alimentacion_bebe",
    "damira": "alimentacion_bebe",
    "humana": "alimentacion_bebe",
    "novalac": "alimentacion_bebe",
    "sanutri": "alimentacion_bebe",
    "hipp": "alimentacion_bebe",
    "babybio": "alimentacion_bebe",
    "smileat": "alimentacion_bebe",
    "holle": "alimentacion_bebe",

    # === HIGIENE BEBÉ (Ampliado) ===
    "mustela bebe": "higiene_bebe",
    "weleda bebe": "higiene_bebe",
    "johnson baby": "higiene_bebe",
    "sebamed bebe": "higiene_bebe",
    "lutsine bebe": "dermatitis_panal",
    "mitosyl": "dermatitis_panal",
    "bepanthen bebe": "dermatitis_panal",
    "eryplast": "dermatitis_panal",
    "nutraisdin": "higiene_bebe",
    "klorane bebe": "higiene_bebe",
    "roc bebe": "higiene_bebe",
    "nuk": "alimentacion_bebe",
    "avent philips": "alimentacion_bebe",
    "mam": "alimentacion_bebe",

    # === DIGESTIVO (Ampliado) ===
    "iberogast": "digestivo",
    "aquilea digestivo": "digestivo",
    "arkocapsulas alcachofa": "digestivo",
    "hepatosil": "digestivo",
    "hepadif": "digestivo",
    "digest": "digestivo",
    "antidiarrhoeicum": "diarrea",
    "fortasec": "diarrea",
    "imodium": "diarrea",
    "tiorfan": "diarrea",
    "duphalac": "estrenimiento",
    "movicol": "estrenimiento",
    "dulcolax": "estrenimiento",
    "micralax": "estrenimiento",
    "aerored": "gases_flatulencia",
    "aero red": "gases_flatulencia",
    "carminativo": "gases_flatulencia",
    "buscapina": "colicos",
    "alka seltzer": "acidez_reflujo",
    "maalox": "acidez_reflujo",
    "phosphalugel": "acidez_reflujo",
    "biogaia": "probioticos",
    "lactibiane": "probioticos",
    "symbioram": "probioticos",
    "reuteri": "probioticos",
    "vivomixx": "probioticos",
    "rotabiotic": "probioticos",

    # === RESPIRATORIO (Ampliado) ===
    "respibien": "congestion_nasal",
    "utabon": "congestion_nasal",
    "nasalmer": "congestion_nasal",
    "rinastel": "congestion_nasal",
    "quinton": "congestion_nasal",
    "physiomer": "congestion_nasal",
    "sterimar": "congestion_nasal",
    "marimer": "congestion_nasal",
    "finasal": "congestion_nasal",
    "flixonase": "alergias",
    "nasonex": "alergias",
    "rhinocort": "alergias",
    "allergodil": "alergias",
    "ebastel": "alergias",
    "aerius": "alergias",
    "clarityne": "alergias",
    "zyrtec": "alergias",
    "mucosan": "mucosidad_respiratoria",
    "fluimucil": "mucosidad_respiratoria",
    "bisolfren": "mucosidad_respiratoria",
    "pectox": "mucosidad_respiratoria",
    "prospantus": "mucosidad_respiratoria",
    "strepsils": "dolor_garganta",
    "lizipaina": "dolor_garganta",
    "anginovag": "dolor_garganta",
    "bucometasana": "dolor_garganta",
    "pharyngis": "dolor_garganta",
    "faringesic": "dolor_garganta",

    # === PIEL ATÓPICA/DERMATITIS (Ampliado) ===
    "dermalex": "dermatitis",
    "leti at4": "piel_atopica",
    "sebamed": "piel_atopica",
    "uriage bariéderm": "heridas_cicatrizacion",
    "a-derma": "piel_atopica",
    "ducray dexyane": "piel_atopica",
    "aderma exomega": "piel_atopica",
    "lipikar": "piel_atopica",
    "xeracalm": "piel_atopica",
    "nutrapon": "piel_atopica",
    "bepanthol": "hidratacion_corporal",
    "neutrogena": "hidratacion_corporal",
    "natural honey": "hidratacion_corporal",
    "atopic piel": "piel_atopica",
    "multilind": "hongos_piel",
    "canesten": "hongos_piel",
    "lamisil": "hongos_piel",
    "daktarin": "hongos_piel",

    # === ANTIEDAD/COSMÉTICA PREMIUM ===
    "filorga": "arrugas_antiedad",
    "nuxe": "hidratacion_facial",
    "caudalie": "arrugas_antiedad",
    "clarins": "arrugas_antiedad",
    "clinique": "hidratacion_facial",
    "estee lauder": "arrugas_antiedad",
    "lancome": "arrugas_antiedad",
    "shiseido": "arrugas_antiedad",
    "sensilis": "arrugas_antiedad",
    "bella aurora": "manchas",
    "neostrata": "arrugas_antiedad",
    "ifc cantabria": "arrugas_antiedad",
    "mesoestetic": "arrugas_antiedad",
    "dr sebagh": "arrugas_antiedad",
    "c vitamina": "arrugas_antiedad",
    "flavo c": "arrugas_antiedad",
    "be+": "arrugas_antiedad",
    "singuladerm": "arrugas_antiedad",

    # === OJOS (Ampliado) ===
    "systane": "ojo_seco",
    "optive": "ojo_seco",
    "hylo": "ojo_seco",
    "thealoz": "ojo_seco",
    "artelac": "ojo_seco",
    "blink": "ojo_seco",
    "siccafluid": "ojo_seco",
    "aquoral": "ojo_seco",
    "ocusan": "ojo_seco",
    "oculotect": "ojo_seco",
    "opatanol": "alergia_ocular",
    "cusicrom": "alergia_ocular",
    "zaditen": "alergia_ocular",
    "optiscreen": "ojo_seco",

    # === OIDOS (Nuevo) ===
    "audispray": "higiene_oidos",
    "cerustop": "cerumen",
    "otix": "dolor_oidos",
    "otalgan": "dolor_oidos",
    "tapon oidos mack": "higiene_oidos",
    "quies": "higiene_oidos",

    # === SALUD FEMENINA (Nuevo) ===
    "gine canesten": "candidiasis_vaginal",
    "gynofit": "higiene_intima",
    "cumlaude gynelaude": "higiene_intima",
    "lactacyd": "higiene_intima",
    "chilly": "higiene_intima",
    "tantum rosa": "higiene_intima",
    "urogyn": "flora_vaginal",
    "donna plus": "flora_vaginal",
    "ginefam": "menopausia",
    "ymea": "menopausia",
    "fisiogen": "menopausia",
    "climater": "menopausia",
    "estromineral": "menopausia",
    "menopace": "menopausia",
    "feminabiane": "menopausia",
    "aquilea menopausia": "menopausia",

    # === CIRCULACIÓN (Ampliado) ===
    "venosmil": "piernas_cansadas",
    "venofytol": "piernas_cansadas",
    "venaston": "piernas_cansadas",
    "venolex": "piernas_cansadas",
    "venaliv": "piernas_cansadas",
    "antistax": "piernas_cansadas",
    "hemoal": "hemorroides",
    "proctolog": "hemorroides",
    "hemorrane": "hemorroides",
    "anso": "hemorroides",
    "ruscoven": "varices",
    "varicofix": "varices",

    # === VITAMINAS/SUPLEMENTOS (Ampliado) ===
    "supradyn": "vitaminas_minerales",
    "centrum": "vitaminas_minerales",
    "pharmaton": "energia_vitalidad",
    "redoxon": "vitaminas_minerales",
    "multicentrum": "vitaminas_minerales",
    "dayamineral": "vitaminas_minerales",
    "berocca": "energia_vitalidad",
    "energizer": "energia_vitalidad",
    "marnys": "vitaminas_minerales",
    "ana maria lajusticia": "vitaminas_minerales",
    "santiveri": "vitaminas_minerales",
    "naturhouse": "vitaminas_minerales",
    "solaray": "vitaminas_minerales",
    "now foods": "vitaminas_minerales",
    "lamberts": "vitaminas_minerales",
    "vitae": "vitaminas_minerales",
    "epadhax": "omega3",
    "omegafort": "omega3",
    "triptomax": "sueno_insomnio",
    "aquilea sueno": "sueno_insomnio",
    "melamil": "sueno_insomnio",
    "valeriana arkocaps": "sueno_insomnio",
    "somnidina": "sueno_insomnio",
    "dormidina": "sueno_insomnio",

    # === ORTOPEDIA (Nuevo) ===
    "farmalastic": "ortopedia_pie",
    "farmalastic feet": "ortopedia_pie",
    "herbitas": "ortopedia_pie",
    "comforsil": "ortopedia_pie",
    "scholl": "ortopedia_pie",
    "dr scholl": "ortopedia_pie",
    "compeed": "heridas_apositos",
    "urgo": "heridas_apositos",
    "tiritas": "heridas_apositos",
    "hansaplast": "heridas_apositos",
    "hartmann": "material_sanitario",
    "medela": "embarazo_lactancia",
    "suavinex lactancia": "embarazo_lactancia",
    "bebe confort": "embarazo_lactancia",

    # === NUTRICIÓN CLÍNICA (Nuevo) ===
    "fresubin": "nutricion_clinica",
    "ensure": "nutricion_clinica",
    "resource": "nutricion_clinica",
    "meritene": "nutricion_clinica",
    "fortimel": "nutricion_clinica",
    "nutridrink": "nutricion_clinica",
    "isosource": "nutricion_clinica",
    "nutrison": "nutricion_clinica",
    "barimix": "nutricion_clinica",

    # === REPELENTES (Nuevo) ===
    "goibi": "picaduras",
    "relec": "picaduras",
    "autan": "picaduras",
    "mosquitero": "picaduras",
    "isdin antimosquitos": "picaduras",
    "afterbite": "picaduras",

    # === MARCAS ACNE ===
    "acniben": "acne",
    "sebiaclear": "acne",
    "effaclar": "acne",
    "normaderm": "acne",
    "serozinc": "acne",
    "sebium": "acne",

    # === DIABETES/CONTROL ===
    "accu-chek": "diabetes",
    "freestyle": "diabetes",
    "onetouch": "diabetes",
    "contour": "diabetes",
    "glucomen": "diabetes",

    # === INCONTINENCIA (Nuevo) ===
    "tena": "incontinencia",
    "hartmann molicare": "incontinencia",
    "lindor": "incontinencia",
    "id expert": "incontinencia",
    "depend": "incontinencia",

    # === URINARIO (Nuevo) ===
    "cistitus": "infeccion_urinaria",
    "urocran": "infeccion_urinaria",
    "cysticlean": "infeccion_urinaria",
    "aquilea prostate": "prostata",
    "prostamol": "prostata",

    # === HERPES/LABIAL ===
    "compeed herpes": "herpes",
    "zovirax": "herpes",
    "aciclovir labial": "herpes",
    "herpescare": "herpes",

    # === FLORA INTESTINAL (Missing from tests) ===
    "ns florabiotic": "flora_intestinal",
    "florabiotic": "flora_intestinal",
    "ns vitans": "vitaminas_minerales",
    "ns digestconfort": "digestivo",
    "ns": "vitaminas_minerales",  # Nutrisanté generic

    # === HIERRO (Missing from tests) ===
    "ferroprotect": "hierro",
    "ferrogradumet": "hierro",
    "ferrosanol": "hierro",
    "tardyferon": "hierro",
    "ferplex": "hierro",
    "venofer": "hierro",
    "kilor": "hierro",
    "ferro sanol": "hierro",
    "floradix": "hierro",

    # === HONGOS PIES (Missing from tests) ===
    "funsol": "hongos_pies",
    "fungarest": "hongos_piel",
    "fungusol": "hongos_piel",

    # === HOMEOPATÍA (Missing from tests) ===
    "oscillococcinum": "homeopatia",
    "boiron": "homeopatia",
    "heel": "homeopatia",
    "lehning": "homeopatia",
    "weleda homeo": "homeopatia",
    "hering": "homeopatia",
    "unda": "homeopatia",

    # === DEFENSAS/INMUNIDAD (Missing from tests) ===
    "arkobiotics": "defensas_inmunidad",
    "arkovital": "defensas_inmunidad",
    "inmunoferon": "defensas_inmunidad",
    "inmunoglukan": "defensas_inmunidad",
    "apiserum defensas": "defensas_inmunidad",
    "sambucol": "defensas_inmunidad",
    "oscillococcinum": "defensas_inmunidad",

    # === SEQUEDAD VAGINAL (Missing from tests) ===
    "deligyn": "sequedad_vaginal",
    "cumlaude lab": "higiene_intima",
    "rilastil cumlaude": "sequedad_vaginal",
    "gynofit": "sequedad_vaginal",
    "replens": "sequedad_vaginal",
    "ainara": "sequedad_vaginal",
    "vagisil": "higiene_intima",

    # === MEMORIA/CONCENTRACIÓN (Missing from tests) ===
    "memorup": "memoria_concentracion",
    "memoboost": "memoria_concentracion",
    "memory plus": "memoria_concentracion",
    "cognitex": "memoria_concentracion",
    "ginkgo biloba": "memoria_concentracion",
    "mind master": "memoria_concentracion",
    "neurozan": "memoria_concentracion",
    "vitalmemory": "memoria_concentracion",

    # === MATERIAL SANITARIO (Missing from tests) ===
    "aposan": "material_sanitario",
    "acofar": "material_sanitario",
    "interapothek": "material_sanitario",
    "kern pharma": "material_sanitario",

    # === MANOS (Missing from tests) ===
    "neutrogena manos": "manos",
    "atrix": "manos",
    "o'keeffe's": "manos",
    "cicalfate manos": "manos",

    # === ALIMENTACIÓN BEBÉ (Missing from tests - HERO patterns) ===
    "hero cereales": "alimentacion_bebe",
    "hero 8 cereales": "alimentacion_bebe",
    "hero galleta": "alimentacion_bebe",
    "hero fruta": "alimentacion_bebe",

    # === ENERGÍA (More specific) ===
    "supradyn energy": "energia_cansancio",
    "multicentrum energy": "energia_cansancio",
    "berocca boost": "energia_cansancio",

    # === SENSIBILIDAD DENTAL (More specific) ===
    "bexident sensibles": "sensibilidad_dental",
    "gum sensivital": "sensibilidad_dental",
    "sensigel": "sensibilidad_dental",

    # === ROJECES/ROSACEA (Missing from tests) ===
    "sensibio ar": "rojeces_rosacea",
    "rosaliac": "rojeces_rosacea",
    "antirougeurs": "rojeces_rosacea",
    "diroseal": "rojeces_rosacea",
    "redness solutions": "rojeces_rosacea",

    # === RETINOL/ANTIEDAD (More specific) ===
    "isdinceutics": "arrugas_antiedad",
    "retinal intense": "arrugas_antiedad",
    "retinol": "arrugas_antiedad",
    "retiage": "arrugas_antiedad",

    # === PROBIÓTICOS (Missing from tests) ===
    "megalevure": "probioticos",
    "enterogermina": "flora_intestinal",
    "florastor": "probioticos",
    "bifidus": "probioticos",
    "bivos": "probioticos",

    # === ESTREÑIMIENTO (Missing from tests) ===
    "dulcosoft": "estrenimiento",
    "laxial": "estrenimiento",
    "plantaben": "estrenimiento",
    "fave de fuca": "estrenimiento",
    "agiolax": "estrenimiento",

    # === HIGIENE ÍNTIMA (Missing from tests) ===
    "melagyn": "higiene_intima",
    "saforelle": "higiene_intima",
    "intimina": "higiene_intima",
    "saugella": "higiene_intima",

    # === EMBARAZO/PRENATAL (Missing from tests) ===
    "femibion": "embarazo_lactancia",
    "natalben": "embarazo_lactancia",
    "gestagyn": "embarazo_lactancia",
    "gynea": "embarazo_lactancia",
    "maternelle": "embarazo_lactancia",
    "natalis": "embarazo_lactancia",

    # === CONGESTIÓN NASAL (Missing from tests) ===
    "normomar": "congestion_nasal",
    "lizipadol": "dolor_garganta",
    "liziphar": "dolor_garganta",

    # === CIRCULACIÓN (Missing from tests) ===
    "venovid": "piernas_cansadas",
    "venosmil": "piernas_cansadas",
    "cinfa venoruton": "piernas_cansadas",

    # === DEFENSAS (Missing from tests) ===
    "olecir": "defensas_inmunidad",
    "animobium": "defensas_inmunidad",
    "zarbees": "dolor_garganta",
    "stimunex": "energia_vitalidad",

    # === DERMATITIS PAÑAL (Missing from tests) ===
    "ozobaby": "dermatitis_panal",
    "halibut": "dermatitis_panal",
    "nutraisdin": "dermatitis_panal",
    "pasta lassar": "dermatitis_panal",

    # === PICADURAS/PIOJOS (Missing from tests) ===
    "kitapic": "picaduras",
    "sting relief": "picaduras",
    "pic roll": "picaduras",

    # === PROTECCIÓN AUDITIVA (Missing from tests) ===
    "moldex": "proteccion_auditiva",
    "tapones oidos": "proteccion_auditiva",
    "howard leight": "proteccion_auditiva",

    # === HEMORROIDES (Missing from tests) ===
    "hemoless": "hemorroides",
    "hemofarm": "hemorroides",
    "synalar rectal": "hemorroides",

    # === SUERO FISIOLÓGICO ===
    "suero fisiologico": "congestion_nasal",
    "solucion salina": "congestion_nasal",

    # === GASES (Missing from tests) ===
    "coligas": "gases_flatulencia",
    "aerogas": "gases_flatulencia",

    # === HIERRO (More brands) ===
    "ferrum": "hierro",
    "ferroplus": "hierro",
    "ferroglicina": "hierro",

    # === CAÍDA CABELLO (More brands) ===
    "aminocomplex": "caida_cabello",
    "aminexil": "caida_cabello",
    "neogenic": "caida_cabello",

    # === AFTAS (Missing from tests) ===
    "aloclair": "aftas",
    "aftex": "aftas",
    "urgo aftas": "aftas",
    "boca sana": "aftas",

    # === IRRITACIÓN PIEL (Missing from tests) ===
    "talquistina": "irritacion_piel",
    "caladryl": "irritacion_piel",
    "calamina": "irritacion_piel",
    "after sun": "quemaduras",

    # === LABIOS (Missing from tests) ===
    "letibalm": "labios",
    "blistex": "labios",
    "carmex": "labios",
    "liposan": "labios",
    "neutrogena lip": "labios",

    # === ESTREÑIMIENTO (More brands) ===
    "melilax": "estrenimiento",
    "laxante": "estrenimiento",
    "evacuol": "estrenimiento",

    # === DIGESTIVO (More brands) ===
    "oliovita": "digestivo",
    "gastroliv": "digestivo",
    "digest complex": "digestivo",

    # === MAGNESIO (Exact match needed) ===
    "magnesium": "magnesio",
    "magnesio carbonato": "magnesio",
    "magnesio marino": "magnesio",

    # === PERFUMERÍA ===
    "iap pharma": "perfumeria",
    "equivalenza": "perfumeria",
    "eau de toilette": "perfumeria",

    # === ANTIEDAD/COSMÉTICA (More brands) ===
    "two poles": "arrugas_antiedad",
    "gh pha": "exfoliacion",
    "arturo alba": "arrugas_antiedad",
    "ivb vitalnatal": "vitaminas_minerales",
}


# =============================================================================
# KEYWORD PATTERNS (Regex)
# =============================================================================
# More complex patterns that require regex matching.

KEYWORD_PATTERNS: List[Tuple[str, str, float]] = [
    # (pattern, necesidad, confidence)

    # Solar (high specificity keywords)
    (r"\bspf\s*\d+", "proteccion_solar", 0.98),
    (r"\bfps\s*\d+", "proteccion_solar", 0.98),
    (r"\bsolar\b", "proteccion_solar", 0.90),
    (r"\bfotoprotector\b", "proteccion_solar", 0.98),
    (r"\bafter\s*sun\b", "proteccion_solar", 0.95),

    # Dental
    (r"\bcepillo.*dental\b", "higiene_dental", 0.98),
    (r"\bcepillo.*electrico\b", "higiene_dental", 0.95),
    (r"\bpasta.*dientes\b", "higiene_dental", 0.98),
    (r"\bcolut[ou]rio\b", "higiene_dental", 0.95),
    (r"\bfluo[ru]\b", "higiene_dental", 0.85),

    # Caída cabello
    (r"\banticaida\b", "caida_cabello", 0.95),
    (r"\bcaida.*cabello\b", "caida_cabello", 0.95),
    (r"\bminoxidil\b", "caida_cabello", 0.98),

    # Omega
    (r"\bomega\s*3\b", "omega3", 0.98),
    (r"\bomega-3\b", "omega3", 0.98),
    (r"\bdha\b.*\bepa\b", "omega3", 0.95),
    (r"\baceite.*pescado\b", "omega3", 0.95),

    # Probióticos
    (r"\bprob[ií]otico", "probioticos", 0.95),
    (r"\blactobacillus\b", "probioticos", 0.95),
    (r"\bbifidobacterium\b", "probioticos", 0.95),

    # Dolor
    (r"\bantiinflamatorio\b", "dolor_muscular", 0.85),
    (r"\banalg[eé]sico\b", "dolor_fiebre", 0.90),
    (r"\bparche.*calor\b", "dolor_muscular", 0.95),
    (r"\bparche.*termico\b", "dolor_muscular", 0.95),

    # Digestivo
    (r"\bantiacido\b", "acidez_reflujo", 0.95),
    (r"\blaxante\b", "estrenimiento", 0.95),
    (r"\bantidiarreico\b", "diarrea", 0.95),

    # Respiratorio
    (r"\bantigrip[ae]l\b", "gripe_resfriado", 0.95),
    (r"\bdescongestivo\b", "congestion_nasal", 0.90),
    (r"\bmucol[ií]tico\b", "mucosidad_respiratoria", 0.95),
    (r"\bexpectorante\b", "mucosidad_respiratoria", 0.95),
    (r"\bantitusivo\b", "tos_seca", 0.95),

    # Sueño
    (r"\bmelatonina\b", "sueno_insomnio", 0.98),
    (r"\bpara\s+dormir\b", "sueno_insomnio", 0.90),
    (r"\binsomnio\b", "sueno_insomnio", 0.95),

    # Infantil
    (r"\bbeb[eé]\b", "alimentacion_bebe", 0.70),  # Low confidence, needs context
    (r"\binfantil\b", "alimentacion_bebe", 0.70),
    (r"\bpa[ñn]al\b", "higiene_bebe", 0.95),

    # Test
    (r"\btest.*embarazo\b", "test_diagnostico", 0.98),
    (r"\btest.*covid\b", "test_diagnostico", 0.98),
    (r"\btest.*ovulacion\b", "fertilidad", 0.98),

    # === TESTS/AUTODIAGNÓSTICO (Ampliado) ===
    (r"\bautodiagn[oó]stico\b", "test_diagnostico", 0.95),
    (r"\bautotest\b", "test_diagnostico", 0.95),
    (r"\btest.*nasal\b", "test_diagnostico", 0.95),
    (r"\btest.*antigenos\b", "test_diagnostico", 0.95),
    (r"\btest.*ant[ií]genos\b", "test_diagnostico", 0.95),
    (r"\btest.*saliva\b", "test_diagnostico", 0.95),
    (r"\btest.*celiaco\b", "test_diagnostico", 0.95),
    (r"\btest.*alergia\b", "test_diagnostico", 0.95),
    (r"\btest.*vih\b", "test_diagnostico", 0.95),
    (r"\btest.*combinado\b", "test_diagnostico", 0.90),
    (r"\bgluc[oó]metro\b", "diabetes", 0.95),

    # === OIDOS (Nuevo) ===
    (r"\blimpieza.*o[ií]dos\b", "higiene_oidos", 0.95),
    (r"\bo[ií]dos.*limpieza\b", "higiene_oidos", 0.95),
    (r"\bspray.*o[ií]dos\b", "higiene_oidos", 0.95),
    (r"\bcerumen\b", "cerumen", 0.95),
    (r"\btap[oó]n.*o[ií]do\b", "higiene_oidos", 0.90),
    (r"\botitis\b", "otitis", 0.95),

    # === SALUD FEMENINA ===
    (r"\bflora.*[ií]ntima\b", "flora_vaginal", 0.95),
    (r"\bflora.*vaginal\b", "flora_vaginal", 0.95),
    (r"\bcandidiasis\b", "candidiasis_vaginal", 0.95),
    (r"\bvaginosis\b", "vaginosis", 0.95),
    (r"\b[oó]vulo.*vaginal\b", "candidiasis_vaginal", 0.95),
    (r"\bhigiene.*[ií]ntima\b", "higiene_intima", 0.95),
    (r"\bjab[oó]n.*[ií]ntimo\b", "higiene_intima", 0.95),
    (r"\bmenopausia\b", "menopausia", 0.95),
    (r"\bsofocos\b", "menopausia", 0.90),
    (r"\bisoflavonas\b", "menopausia", 0.90),
    (r"\bciclo.*menstrual\b", "ciclo_menstrual", 0.90),
    (r"\bdismenorrea\b", "ciclo_menstrual", 0.90),
    (r"\bsindrome.*premenstrual\b", "ciclo_menstrual", 0.90),

    # === ORTOPEDIA/PIES ===
    (r"\bdedil\b", "ortopedia_pie", 0.90),
    (r"\bprotector.*dedil\b", "ortopedia_pie", 0.95),
    (r"\bprotector.*juanete\b", "ortopedia_pie", 0.95),
    (r"\bjuanete\b", "ortopedia_pie", 0.90),
    (r"\bhallux\b", "ortopedia_pie", 0.95),
    (r"\bplantilla\b", "ortopedia_pie", 0.85),
    (r"\btalonera\b", "ortopedia_pie", 0.95),
    (r"\bmetatarsal\b", "ortopedia_pie", 0.95),

    # === NUTRICIÓN CLÍNICA ===
    (r"\bnutrici[oó]n.*cl[ií]nica\b", "nutricion_clinica", 0.95),
    (r"\bnutrici[oó]n.*enteral\b", "nutricion_clinica", 0.95),
    (r"\bhipercal[oó]rico\b", "nutricion_clinica", 0.95),
    (r"\bhiperprot[eé]ico\b", "nutricion_clinica", 0.95),
    (r"\bespesante\b", "nutricion_clinica", 0.90),
    (r"\bdisfagia\b", "nutricion_clinica", 0.95),

    # === ALIMENTACIÓN BEBÉ (Ampliado) ===
    (r"\bleche.*etapa\s*\d\b", "alimentacion_bebe", 0.95),
    (r"\bf[oó]rmula.*infantil\b", "alimentacion_bebe", 0.95),
    (r"\bleche.*continuaci[oó]n\b", "alimentacion_bebe", 0.95),
    (r"\bleche.*crecimiento\b", "alimentacion_bebe", 0.95),
    (r"\bleche.*hidrolizada\b", "alimentacion_bebe", 0.95),
    (r"\bpotito\b", "alimentacion_bebe", 0.95),
    (r"\bpapilla\b", "alimentacion_bebe", 0.90),
    (r"\bcereales.*beb[eé]\b", "alimentacion_bebe", 0.95),
    (r"\bsnack.*zanahoria\b", "alimentacion_bebe", 0.95),
    (r"\bsnack.*fruta\b", "alimentacion_bebe", 0.90),
    (r"\bc[oó]lico.*beb[eé]\b", "colicos_bebe", 0.95),
    (r"\banticolico\b", "colicos_bebe", 0.95),
    (r"\bantirregurgitaci[oó]n\b", "colicos_bebe", 0.95),
    (r"\bdentici[oó]n\b", "denticion", 0.95),
    (r"\bmordedor\b", "denticion", 0.95),

    # === PIEL (Ampliado) ===
    (r"\blimpiador.*facial\b", "limpieza_facial", 0.95),
    (r"\bdesmaquillante\b", "limpieza_facial", 0.95),
    (r"\btonico.*facial\b", "limpieza_facial", 0.95),
    (r"\bagua.*micelar\b", "limpieza_facial", 0.95),
    (r"\bexfoliante\b", "limpieza_facial", 0.90),
    (r"\bserum\b", "arrugas_antiedad", 0.85),
    (r"\bantiarrugas\b", "arrugas_antiedad", 0.95),
    (r"\bantiedad\b", "arrugas_antiedad", 0.95),
    (r"\bcontorno.*ojos\b", "arrugas_antiedad", 0.95),
    (r"\bantimanchas\b", "manchas", 0.95),
    (r"\bdespigmentante\b", "manchas", 0.95),
    (r"\banticelulitico\b", "celulitis", 0.95),
    (r"\bcelulitis\b", "celulitis", 0.90),
    (r"\breafirmante\b", "celulitis", 0.85),
    (r"\bantiestrias\b", "estrias", 0.95),
    (r"\bestr[ií]as\b", "estrias", 0.90),
    (r"\bauotbronceador\b", "bronceado", 0.95),
    (r"\bpiel.*at[oó]pica\b", "piel_atopica", 0.95),
    (r"\batopia\b", "piel_atopica", 0.95),
    (r"\bdermatitis\b", "dermatitis", 0.95),
    (r"\beczema\b", "dermatitis", 0.95),
    (r"\bpsoriasis\b", "psoriasis", 0.95),
    (r"\bherpes.*labial\b", "herpes", 0.95),
    (r"\bcalentura\b", "herpes", 0.85),

    # === HONGOS ===
    (r"\bhongos.*piel\b", "hongos_piel", 0.95),
    (r"\bhongos.*u[ñn]as\b", "hongos_piel", 0.95),
    (r"\bantif[uú]ngico\b", "hongos_piel", 0.95),
    (r"\bpie.*atleta\b", "hongos_piel", 0.95),
    (r"\btinea\b", "hongos_piel", 0.95),

    # === DOLOR ARTICULAR (Ampliado) ===
    (r"\barticulaciones\b", "dolor_articular", 0.85),
    (r"\bartrosis\b", "dolor_articular", 0.95),
    (r"\bartritis\b", "dolor_articular", 0.95),
    (r"\bcondr[oó]it[ií]na\b", "dolor_articular", 0.95),
    (r"\bglucosamina\b", "dolor_articular", 0.95),
    (r"\bcol[aá]geno\b", "dolor_articular", 0.80),

    # === CIRCULACIÓN ===
    (r"\bpiernas.*cansadas\b", "piernas_cansadas", 0.95),
    (r"\bpiernas.*pesadas\b", "piernas_cansadas", 0.95),
    (r"\bvarices\b", "varices", 0.95),
    (r"\bhemorroides\b", "hemorroides", 0.95),
    (r"\balmorranas\b", "hemorroides", 0.95),
    (r"\bfisura.*anal\b", "hemorroides", 0.90),

    # === URINARIO ===
    (r"\binfecci[oó]n.*urinaria\b", "infeccion_urinaria", 0.95),
    (r"\bcistitis\b", "infeccion_urinaria", 0.95),
    (r"\bar[aá]ndano.*rojo\b", "infeccion_urinaria", 0.90),
    (r"\bd-manosa\b", "infeccion_urinaria", 0.95),
    (r"\bpr[oó]stata\b", "prostata", 0.95),
    (r"\bretenci[oó]n.*l[ií]quidos\b", "retencion_liquidos", 0.95),
    (r"\bdiur[eé]tico\b", "retencion_liquidos", 0.90),

    # === REPELENTES ===
    (r"\brepelente.*mosquitos\b", "picaduras", 0.95),
    (r"\bantimosquitos\b", "picaduras", 0.95),
    (r"\bpicaduras\b", "picaduras", 0.90),
    (r"\bafter.*bite\b", "picaduras", 0.95),

    # === PIOJOS ===
    (r"\bantipiojos\b", "piojos", 0.98),
    (r"\banti-piojos\b", "piojos", 0.98),
    (r"\bpiojos\b", "piojos", 0.95),
    (r"\bliendres\b", "piojos", 0.95),
    (r"\blendrera\b", "piojos", 0.95),

    # === OJOS ===
    (r"\bojo.*seco\b", "ojo_seco", 0.95),
    (r"\bojos.*secos\b", "ojo_seco", 0.95),
    (r"\bl[aá]grimas.*artificiales\b", "ojo_seco", 0.95),
    (r"\bcolirio\b", "ojo_seco", 0.85),
    (r"\blubricante.*oft[aá]lm", "ojo_seco", 0.95),
    (r"\boft[aá]lm.*lubricante\b", "ojo_seco", 0.95),
    (r"\bsoluci[oó]n.*oft[aá]lmica\b", "ojo_seco", 0.90),
    (r"\bconjuntivitis\b", "conjuntivitis", 0.95),
    (r"\balergia.*ocular\b", "alergia_ocular", 0.95),
    (r"\blentillas\b", "lentillas", 0.95),
    (r"\blente.*contacto\b", "lentillas", 0.95),

    # === DENTAL (Ampliado) ===
    (r"\bsensibilidad.*dental\b", "sensibilidad_dental", 0.95),
    (r"\bdientes.*sensibles\b", "sensibilidad_dental", 0.95),
    (r"\bgingivitis\b", "encias", 0.95),
    (r"\benc[ií]as\b", "encias", 0.85),
    (r"\bperiodontitis\b", "encias", 0.95),
    (r"\bpr[oó]tesis.*dental\b", "protesis_dental", 0.95),
    (r"\bdentadura.*postiza\b", "protesis_dental", 0.95),
    (r"\blimpiador.*dentadura\b", "protesis_dental", 0.95),
    (r"\btableta.*limpiadora\b", "protesis_dental", 0.90),
    (r"\bortodoncia\b", "higiene_dental", 0.90),
    (r"\binterdental\b", "higiene_dental", 0.95),
    (r"\bblanqueamiento.*dental\b", "blanqueamiento", 0.95),
    (r"\bblanqueador.*dental\b", "blanqueamiento", 0.95),

    # === INCONTINENCIA ===
    (r"\bincontinencia\b", "incontinencia", 0.95),
    (r"\bpa[ñn]al.*adulto\b", "incontinencia", 0.95),
    (r"\babsorbente.*incontinencia\b", "incontinencia", 0.95),
    (r"\bempapador\b", "incontinencia", 0.90),
    (r"\bprotector.*cama\b", "incontinencia", 0.90),

    # === DIABETES ===
    (r"\bdiab[eé]tico\b", "diabetes", 0.90),
    (r"\bdiabetes\b", "diabetes", 0.90),
    (r"\btira.*reactiva\b", "diabetes", 0.90),
    (r"\blanceta\b", "diabetes", 0.85),

    # === MAGNESIO ===
    (r"\bmagnesio\b", "magnesio", 0.85),
    (r"\bcalambres\b", "magnesio", 0.80),

    # === VITAMINAS ===
    (r"\bmultivitam[ií]n", "vitaminas_minerales", 0.90),
    (r"\bvitamina\s*[a-eABCDE]\b", "vitaminas_minerales", 0.85),
    (r"\bvitamina\s*d\d*\b", "calcio_vitamina_d", 0.90),
    (r"\bcalcio\b", "calcio_vitamina_d", 0.80),
    (r"\bhierro\b", "hierro", 0.85),
    (r"\bacido.*folico\b", "embarazo_lactancia", 0.90),

    # === ENERGIA ===
    (r"\benergi[az]ante\b", "energia_vitalidad", 0.90),
    (r"\benerg[ií]a.*vitalidad\b", "energia_vitalidad", 0.95),
    (r"\bjalea.*real\b", "energia_vitalidad", 0.95),
    (r"\bginseng\b", "energia_vitalidad", 0.90),
    (r"\bguarana\b", "energia_vitalidad", 0.90),

    # === CONTROL PESO ===
    (r"\bquemagrasas\b", "control_peso", 0.95),
    (r"\bquema.*grasas\b", "control_peso", 0.95),
    (r"\badelgazante\b", "control_peso", 0.95),
    (r"\bsaciante\b", "control_peso", 0.90),
    (r"\bapetito\b", "control_peso", 0.80),
    (r"\bdrenante\b", "retencion_liquidos", 0.90),

    # === ACNÉ ===
    (r"\bacn[eé]\b", "acne", 0.95),
    (r"\bantiacn[eé]\b", "acne", 0.98),
    (r"\bgranos\b", "acne", 0.80),
    (r"\bespinillas\b", "acne", 0.85),
    (r"\bpiel.*grasa\b", "acne", 0.80),
    (r"\bseborreica\b", "caspa", 0.90),

    # === CASPA ===
    (r"\banticaspa\b", "caspa", 0.98),
    (r"\bcaspa\b", "caspa", 0.95),
    (r"\bseborrea\b", "caspa", 0.90),

    # === QUEMADURAS ===
    (r"\bquemadura\b", "quemaduras", 0.95),
    (r"\bquemaduras\b", "quemaduras", 0.95),
    (r"\balivio.*quemadura\b", "quemaduras", 0.95),

    # === EMBARAZO/LACTANCIA ===
    (r"\bembarazo\b", "embarazo_lactancia", 0.85),
    (r"\blactancia\b", "embarazo_lactancia", 0.90),
    (r"\bsacaleches\b", "embarazo_lactancia", 0.95),
    (r"\bpez[oó]n\b", "embarazo_lactancia", 0.85),
    (r"\bprenatal\b", "embarazo_lactancia", 0.95),
    (r"\bpostnatal\b", "embarazo_lactancia", 0.95),
    (r"\bpostparto\b", "embarazo_lactancia", 0.95),

    # === MATERIAL SANITARIO ===
    (r"\bjeringuilla\b", "material_sanitario", 0.95),
    (r"\bguante.*l[aá]tex\b", "material_sanitario", 0.95),
    (r"\bguante.*nitrilo\b", "material_sanitario", 0.95),
    (r"\bmascarilla\b", "material_sanitario", 0.90),
    (r"\bpulsox[ií]metro\b", "material_sanitario", 0.95),
    (r"\btensi[oó]metro\b", "tension_arterial", 0.95),
    (r"\bterm[oó]metro\b", "termometros", 0.95),

    # === HERIDAS/APOSITOS ===
    (r"\btirita\b", "heridas_apositos", 0.95),
    (r"\baposito\b", "heridas_apositos", 0.95),
    (r"\bgasa.*est[eé]ril\b", "heridas_apositos", 0.95),
    (r"\besparadrapo\b", "heridas_apositos", 0.95),
    (r"\bsutura.*adhesiva\b", "heridas_apositos", 0.95),
    (r"\bsteri.?strip\b", "heridas_apositos", 0.95),

    # === COMPRESIÓN ===
    (r"\bmedia.*compresi[oó]n\b", "varices", 0.95),
    (r"\bcalcet[ií]n.*compresi[oó]n\b", "varices", 0.95),
    (r"\bvenda.*el[aá]stica\b", "compresion_vendajes", 0.95),
    (r"\bvendaje\b", "compresion_vendajes", 0.85),

    # === SEXUAL ===
    (r"\bpreservativo\b", "salud_sexual", 0.98),
    (r"\bcond[oó]n\b", "salud_sexual", 0.98),
    # NOTE: "lubricante" removed here - too generic, matches eye drops (LUBRISTIL)
    # Sexual lubricants should be caught by brand (durex, control) instead

    # === HERO CEREALES (Missing from tests - word boundary) ===
    (r"\bhero\b.*\bcereales\b", "alimentacion_bebe", 0.95),
    (r"\bhero\b.*\bgalleta\b", "alimentacion_bebe", 0.95),
    (r"\bhero\b.*\bfruta\b", "alimentacion_bebe", 0.90),
    (r"\b8\s*cereales\b", "alimentacion_bebe", 0.85),

    # === HIERRO (word boundary) ===
    (r"\bferro\w*\b", "hierro", 0.85),  # Matches ferroprotect, ferrogradumet, etc.
    (r"\bhierro\b", "hierro", 0.90),
    (r"\biron\b", "hierro", 0.85),

    # === DEFENSAS/INMUNIDAD ===
    (r"\bdefensas\b", "defensas_inmunidad", 0.85),
    (r"\binmunidad\b", "defensas_inmunidad", 0.90),
    (r"\binmuno\w*\b", "defensas_inmunidad", 0.85),
    (r"\bsistema.*inmune\b", "defensas_inmunidad", 0.90),

    # === MEMORIA/CONCENTRACIÓN ===
    (r"\bmemoria\b", "memoria_concentracion", 0.85),
    (r"\bconcentraci[oó]n\b", "memoria_concentracion", 0.85),
    (r"\bcognitivo\b", "memoria_concentracion", 0.90),
    (r"\bmental\b", "memoria_concentracion", 0.75),

    # === HOMEOPATÍA ===
    (r"\bhomeopat[ií]a\b", "homeopatia", 0.95),
    (r"\bhomeopat[ií]co\b", "homeopatia", 0.95),
    (r"\bunidosis\b", "homeopatia", 0.80),  # Common in homeopathic products

    # === SEQUEDAD VAGINAL ===
    (r"\bsequedad.*vaginal\b", "sequedad_vaginal", 0.95),
    (r"\bhidratante.*interno\b", "sequedad_vaginal", 0.90),
    (r"\blubricante.*vaginal\b", "sequedad_vaginal", 0.90),

    # === ROJECES/ROSÁCEA ===
    (r"\brojeces\b", "rojeces_rosacea", 0.90),
    (r"\bros[aá]cea\b", "rojeces_rosacea", 0.95),
    (r"\bantirrojeces\b", "rojeces_rosacea", 0.95),
    (r"\bredness\b", "rojeces_rosacea", 0.85),
    (r"\bcuperosis\b", "rojeces_rosacea", 0.95),

    # === MANOS ===
    (r"\bcrema.*manos\b", "manos", 0.95),
    (r"\bmanos.*agrietadas\b", "manos", 0.95),
    (r"\bmanos.*secas\b", "manos", 0.90),

    # === HONGOS PIES ===
    (r"\bhongos.*pies\b", "hongos_pies", 0.95),
    (r"\bpie.*atleta\b", "hongos_pies", 0.95),
    (r"\bmicosis.*pie\b", "hongos_pies", 0.95),
    (r"\bsudor.*pies\b", "hongos_pies", 0.85),
    (r"\bmal.*olor.*pies\b", "hongos_pies", 0.85),

    # === AGUA OXIGENADA (Material sanitario) ===
    (r"\bagua.*oxigenada\b", "material_sanitario", 0.95),
    (r"\balcohol\s*\d+\b", "material_sanitario", 0.90),
    (r"\balgod[oó]n\b", "material_sanitario", 0.85),
    (r"\bpovidona.*yodada\b", "material_sanitario", 0.95),
    (r"\bbetadine\b", "material_sanitario", 0.90),

    # === ENERGY/CANSANCIO (More specific) ===
    (r"\benergy\b", "energia_cansancio", 0.85),
    (r"\bcansancio\b", "energia_cansancio", 0.90),
    (r"\bfatiga\b", "energia_cansancio", 0.90),
    (r"\bast[eé]nia\b", "energia_cansancio", 0.95),

    # === SERVICIOS (to catch non-products) ===
    (r"\bservicio\b", "servicio", 0.98),
    (r"\bspd\b", "servicio", 0.95),
    (r"\bregalo\b", "regalo", 0.95),
    (r"\bnavidad\b", "regalo", 0.80),

    # === TOALLITAS OFTÁLMICAS ===
    (r"\btoallita.*oft[aá]lm", "ojo_seco", 0.90),
    (r"\blimpieza.*parpado", "ojo_seco", 0.90),
    (r"\bblefaritis\b", "ojo_seco", 0.95),
]


class IngredientRulesEngine:
    """
    Tier 1 Rules Engine for the Sandwich Strategy.

    Matches ingredients, product types, brands, and keyword patterns
    to determine the symptom-based NECESIDAD.

    IMPORTANT: Uses word boundaries (\b) to prevent false positives
    like "sen" matching "senior" or "miel" matching "familiar".
    """

    def __init__(self):
        self._compiled_patterns = [
            (re.compile(pattern, re.IGNORECASE), necesidad, conf)
            for pattern, necesidad, conf in KEYWORD_PATTERNS
        ]

        # Pre-compile word boundary patterns for ingredients, brands, product types
        # This prevents false positives like "sen" matching "senior"
        self._ingredient_patterns = {
            ingredient: (re.compile(rf"\b{re.escape(ingredient)}\b", re.IGNORECASE), necesidad)
            for ingredient, necesidad in INGREDIENT_TO_SYMPTOM.items()
        }
        self._brand_patterns = {
            brand: (re.compile(rf"\b{re.escape(brand)}\b", re.IGNORECASE), necesidad)
            for brand, necesidad in BRAND_TO_SYMPTOM.items()
        }
        self._product_type_patterns = {
            ptype: (re.compile(rf"\b{re.escape(ptype)}\b", re.IGNORECASE), necesidad)
            for ptype, necesidad in PRODUCT_TYPE_TO_SYMPTOM.items()
        }

    def match_text(self, text: str) -> List[RuleMatch]:
        """
        Match text against all rules using WORD BOUNDARIES.

        Args:
            text: Text to match (can be product name, semantic_summary, or indicaciones)

        Returns:
            List of RuleMatch objects, sorted by priority then confidence

        Priority order:
        1. Regex patterns (highest specificity, e.g., SPF50)
        2. Product types (physical form indicates use)
        3. Brands (specialized brands)
        4. Ingredients (can be generic)
        """
        if not text:
            return []

        text_normalized = self._normalize(text.lower())
        matches: List[RuleMatch] = []

        # Priority 1: Check regex patterns FIRST (highest specificity)
        # SPF, omega-3, etc. are very specific indicators
        for pattern, necesidad, confidence in self._compiled_patterns:
            if pattern.search(text):
                match_obj = pattern.search(text)
                matches.append(RuleMatch(
                    necesidad=necesidad,
                    confidence=confidence,
                    matched_term=match_obj.group() if match_obj else "",
                    rule_type="keyword"
                ))

        # Priority 2: Check product types WITH WORD BOUNDARIES
        for ptype, (pattern, necesidad) in self._product_type_patterns.items():
            if pattern.search(text_normalized):
                matches.append(RuleMatch(
                    necesidad=necesidad,
                    confidence=0.92,
                    matched_term=ptype,
                    rule_type="product_type"
                ))

        # Priority 3: Check brands WITH WORD BOUNDARIES
        for brand, (pattern, necesidad) in self._brand_patterns.items():
            if pattern.search(text_normalized):
                matches.append(RuleMatch(
                    necesidad=necesidad,
                    confidence=0.90,
                    matched_term=brand,
                    rule_type="brand"
                ))

        # Priority 4: Check ingredients WITH WORD BOUNDARIES (prevents "sen" matching "senior")
        for ingredient, (pattern, necesidad) in self._ingredient_patterns.items():
            if pattern.search(text_normalized):
                matches.append(RuleMatch(
                    necesidad=necesidad,
                    confidence=0.85,  # Lower confidence for ingredients
                    matched_term=ingredient,
                    rule_type="ingredient"
                ))

        # Sort by: 1) rule_type priority, 2) confidence descending
        type_priority = {"keyword": 0, "product_type": 1, "brand": 2, "ingredient": 3}
        matches.sort(key=lambda m: (type_priority.get(m.rule_type, 99), -m.confidence))

        # Remove duplicates (same necesidad)
        seen: Set[str] = set()
        unique_matches = []
        for m in matches:
            if m.necesidad not in seen:
                seen.add(m.necesidad)
                unique_matches.append(m)

        return unique_matches

    def match_enriched(
        self,
        product_name: str,
        semantic_summary: Optional[str] = None,
        indicaciones: Optional[str] = None,
        ingredientes: Optional[str] = None
    ) -> Optional[RuleMatch]:
        """
        Match against enriched product data (Sandwich Strategy Step 2).

        The order of priority:
        1. Ingredients field (most specific)
        2. Indicaciones (therapeutic indication)
        3. Semantic summary (LLM cleaned description)
        4. Product name (original, potentially dirty)

        Returns:
            Best RuleMatch or None if no match
        """
        all_matches: List[RuleMatch] = []

        # Priority 1: Ingredients
        if ingredientes:
            matches = self.match_text(ingredientes)
            for m in matches:
                m.confidence *= 1.0  # Full weight
                all_matches.append(m)

        # Priority 2: Indicaciones
        if indicaciones:
            matches = self.match_text(indicaciones)
            for m in matches:
                m.confidence *= 0.95  # Slightly less weight
                all_matches.append(m)

        # Priority 3: Semantic summary
        if semantic_summary:
            matches = self.match_text(semantic_summary)
            for m in matches:
                m.confidence *= 0.90  # Less weight
                all_matches.append(m)

        # Priority 4: Product name (last resort)
        matches = self.match_text(product_name)
        for m in matches:
            m.confidence *= 0.80  # Lowest weight
            all_matches.append(m)

        if not all_matches:
            return None

        # Return the highest confidence match
        all_matches.sort(key=lambda m: m.confidence, reverse=True)
        return all_matches[0]

    def _normalize(self, text: str) -> str:
        """Normalize text for matching."""
        # Remove accents
        replacements = {
            'á': 'a', 'é': 'e', 'í': 'i', 'ó': 'o', 'ú': 'u',
            'ü': 'u', 'ñ': 'n'
        }
        for old, new in replacements.items():
            text = text.replace(old, new)

        # Remove extra whitespace
        text = ' '.join(text.split())

        return text


# Singleton instance
_rules_engine: Optional[IngredientRulesEngine] = None


def get_rules_engine() -> IngredientRulesEngine:
    """Get or create the singleton rules engine."""
    global _rules_engine
    if _rules_engine is None:
        _rules_engine = IngredientRulesEngine()
    return _rules_engine
