﻿# backend/app/services/homogeneous_groups_service.py
"""
Servicio para gestión de conjuntos homogéneos optimizados
Población automática desde datos del nomenclator y cálculo de PVL
"""

import logging
from typing import Dict, List, Optional

from sqlalchemy import func
from sqlalchemy.orm import Session

from app.external_data.nomenclator_integration import pvp_to_pvl
from app.models.homogeneous_group import HomogeneousGroup
from app.models.pharmacy_partners import PharmacyPartner
from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class HomogeneousGroupsService:
    """
    Servicio para crear y mantener la tabla optimizada de conjuntos homogéneos
    """

    def __init__(self, db: Session):
        self.db = db

    def populate_homogeneous_groups_for_pharmacy(self, pharmacy_id: str) -> Dict:
        """
        Puebla la tabla homogeneous_groups para una farmacia específica.

        Lógica:
        1. Obtener conjuntos homogéneos únicos del nomenclator
        2. Para cada conjunto: filtrar productos en "ALTA" y tomar PVP mínimo
        3. Calcular PVL usando pvp_to_pvl()
        4. Identificar laboratorios partners en cada conjunto
        5. Calcular estadísticas de ventas agregadas

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            Dict con estadísticas de la población
        """
        logger.info(f"[HOMOGENEOUS] Poblando conjuntos homogéneos para farmacia {pharmacy_id}")

        try:
            # 1. Obtener laboratorios partners seleccionados
            partner_labs = (
                self.db.query(PharmacyPartner)
                .filter(
                    PharmacyPartner.pharmacy_id == pharmacy_id,
                    PharmacyPartner.is_selected.is_(True),
                )
                .all()
            )

            partner_names = [p.laboratory_name for p in partner_labs]
            logger.info(f"[HOMOGENEOUS] Encontrados {len(partner_names)} laboratorios partners")

            # 2. Obtener conjuntos homogéneos únicos con productos en ALTA
            homogeneous_codes = (
                self.db.query(ProductCatalog.nomen_codigo_homogeneo)
                .filter(
                    ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                    ProductCatalog.nomen_estado == "ALTA",  # Solo productos dados de alta
                    ProductCatalog.nomen_pvp.isnot(None),
                )
                .distinct()
                .all()
            )

            codes_list = [code[0] for code in homogeneous_codes if code[0]]
            logger.info(f"[HOMOGENEOUS] Encontrados {len(codes_list)} conjuntos homogéneos únicos")

            # 3. Limpiar registros existentes de la farmacia
            deleted = self.db.query(HomogeneousGroup).filter(HomogeneousGroup.pharmacy_id == pharmacy_id).delete()
            logger.info(f"[HOMOGENEOUS] Eliminados {deleted} registros previos")

            processed_groups = 0
            total_savings_base = 0.0

            # 4. Procesar cada conjunto homogéneo
            for homogeneous_code in codes_list:
                group_data = self._process_homogeneous_group(pharmacy_id, homogeneous_code, partner_names)

                if group_data:
                    processed_groups += 1
                    total_savings_base += group_data.get("non_partner_pvl_base", 0)

                    # Crear registro en la tabla
                    homogeneous_group = HomogeneousGroup(**group_data)
                    self.db.add(homogeneous_group)

            # 5. Confirmar cambios
            self.db.commit()
            logger.info(f"[HOMOGENEOUS] Procesamiento completado: {processed_groups} grupos guardados")

            return {
                "total_groups_found": len(codes_list),
                "groups_processed": processed_groups,
                "partner_labs_count": len(partner_names),
                "total_savings_base_eur": round(total_savings_base, 2),
                "pharmacy_id": pharmacy_id,
                "processing_date": utc_now().isoformat(),
            }

        except Exception as e:
            self.db.rollback()
            logger.error(f"[HOMOGENEOUS] Error poblando grupos: {str(e)}")
            raise

    def _process_homogeneous_group(
        self, pharmacy_id: str, homogeneous_code: str, partner_names: List[str]
    ) -> Optional[Dict]:
        """
        Procesa un conjunto homogéneo individual y calcula todas sus métricas.

        Args:
            pharmacy_id: ID de la farmacia
            homogeneous_code: Código del conjunto homogéneo
            partner_names: Lista de laboratorios partners

        Returns:
            Dict con datos del grupo o None si no hay datos suficientes
        """
        try:
            # 1. Obtener productos del conjunto en ALTA con precios
            products_in_group = (
                self.db.query(ProductCatalog)
                .filter(
                    ProductCatalog.nomen_codigo_homogeneo == homogeneous_code,
                    ProductCatalog.nomen_estado == "ALTA",
                    ProductCatalog.nomen_pvp.isnot(None),
                )
                .all()
            )

            if not products_in_group:
                return None

            # 2. Calcular PVP mínimo (clave para PVL del conjunto)
            pvp_values = [float(p.nomen_pvp) for p in products_in_group if p.nomen_pvp]
            if not pvp_values:
                return None

            min_pvp = min(pvp_values)
            max_pvp = max(pvp_values)
            avg_pvp = sum(pvp_values) / len(pvp_values)

            # 3. Calcular PVL usando PVP mínimo
            calculated_pvl = pvp_to_pvl(min_pvp)

            # 4. Obtener información del grupo
            representative_product = products_in_group[0]  # Para metadatos
            homogeneous_name = representative_product.nomen_nombre_homogeneo or f"Grupo {homogeneous_code}"

            # 5. Identificar laboratorios en el conjunto
            labs_in_group = list(set([p.nomen_laboratorio for p in products_in_group if p.nomen_laboratorio]))
            partner_labs_in_group = [lab for lab in labs_in_group if lab in partner_names]

            # 6. Calcular estadísticas de ventas
            sales_stats = self._calculate_sales_stats(pharmacy_id, homogeneous_code, partner_names)

            # 7. Compilar datos del grupo
            group_data = {
                "pharmacy_id": pharmacy_id,
                "homogeneous_code": homogeneous_code,
                "homogeneous_name": homogeneous_name,
                # Precios calculados
                "representative_pvp": min_pvp,  # Usamos el mínimo como representativo
                "calculated_pvl": calculated_pvl,
                "min_pvp_in_group": min_pvp,
                "max_pvp_in_group": max_pvp,
                "avg_pvp_in_group": round(avg_pvp, 4),
                # Información de partners
                "has_partner_labs": len(partner_labs_in_group) > 0,
                "partner_laboratories": partner_labs_in_group,
                "total_labs_in_group": len(labs_in_group),
                "partner_labs_count": len(partner_labs_in_group),
                # Estadísticas de ventas
                **sales_stats,
                # Metadatos terapéuticos
                "main_active_ingredient": representative_product.nomen_principio_activo,
                "requires_prescription": representative_product.cima_requiere_receta,
                # Metadatos de cálculo
                "calculation_metadata": {
                    "pvl_calculation_method": "min_pvp_nomenclator",
                    "products_in_group": len(products_in_group),
                    "labs_identified": labs_in_group,
                    "min_pvp_used": min_pvp,
                    "calculation_date": utc_now().isoformat(),
                },
            }

            return group_data

        except Exception as e:
            logger.error(f"[HOMOGENEOUS] Error procesando grupo {homogeneous_code}: {str(e)}")
            return None

    def _calculate_sales_stats(self, pharmacy_id: str, homogeneous_code: str, partner_names: List[str]) -> Dict:
        """
        Calcula estadísticas de ventas agregadas para un conjunto homogéneo.

        Args:
            pharmacy_id: ID de la farmacia
            homogeneous_code: Código del conjunto homogéneo
            partner_names: Lista de laboratorios partners

        Returns:
            Dict con estadísticas de ventas
        """
        try:
            # Query base para ventas del conjunto homogéneo
            base_query = (
                self.db.query(SalesData)
                .join(SalesEnrichment, SalesData.enrichment)
                .join(ProductCatalog, SalesEnrichment.product_catalog)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    ProductCatalog.nomen_codigo_homogeneo == homogeneous_code,
                )
            )

            # Estadísticas totales
            total_stats = base_query.with_entities(
                func.sum(SalesData.quantity),
                func.count(SalesData.id),
                func.sum(SalesData.total_amount),
                func.max(SalesData.sale_date),
            ).first()

            total_units = int(total_stats[0] or 0)
            total_transactions = int(total_stats[1] or 0)
            total_revenue = float(total_stats[2] or 0)
            last_sale_date = total_stats[3]

            # Estadísticas de partners
            partner_stats = (
                base_query.filter(ProductCatalog.nomen_laboratorio.in_(partner_names))
                .with_entities(func.sum(SalesData.quantity), func.sum(SalesData.total_amount))
                .first()
            )

            partner_units = int(partner_stats[0] or 0)
            partner_revenue = float(partner_stats[1] or 0)

            # Estadísticas NO-partners (oportunidad de ahorro)
            non_partner_units = total_units - partner_units
            non_partner_revenue = total_revenue - partner_revenue

            # Calcular PVL base para descuentos
            # PVL ya está calculado en el grupo, solo multiplicar por unidades
            calculated_pvl = pvp_to_pvl(min(10, 20))  # Placeholder, se corregirá arriba
            non_partner_pvl_base = non_partner_units * calculated_pvl if calculated_pvl else 0

            return {
                "total_units_sold": total_units,
                "total_transactions": total_transactions,
                "total_revenue": round(total_revenue, 2),
                "partner_units_sold": partner_units,
                "partner_revenue": round(partner_revenue, 2),
                "non_partner_units_sold": non_partner_units,
                "non_partner_revenue": round(non_partner_revenue, 2),
                "non_partner_pvl_base": 0,  # Se calculará después con el PVL correcto
                "last_sales_date": last_sale_date,
                "data_period_start": None,  # TODO: Calcular desde ventas
                "data_period_end": None,
            }

        except Exception as e:
            logger.error(f"[HOMOGENEOUS] Error calculando stats ventas para {homogeneous_code}: {str(e)}")
            return {
                "total_units_sold": 0,
                "total_transactions": 0,
                "total_revenue": 0.0,
                "partner_units_sold": 0,
                "partner_revenue": 0.0,
                "non_partner_units_sold": 0,
                "non_partner_revenue": 0.0,
                "non_partner_pvl_base": 0.0,
                "last_sales_date": None,
                "data_period_start": None,
                "data_period_end": None,
            }

    def update_pvl_base_calculations(self, pharmacy_id: str) -> Dict:
        """
        Actualiza los cálculos de PVL base después de poblar los grupos.
        Corrige el campo non_partner_pvl_base usando el PVL calculado correcto.

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            Dict con estadísticas de la actualización
        """
        try:
            groups = (
                self.db.query(HomogeneousGroup)
                .filter(
                    HomogeneousGroup.pharmacy_id == pharmacy_id,
                    HomogeneousGroup.calculated_pvl.isnot(None),
                    HomogeneousGroup.non_partner_units_sold > 0,
                )
                .all()
            )

            updated_count = 0

            for group in groups:
                if group.calculated_pvl and group.non_partner_units_sold:
                    # Calcular PVL base correcto
                    new_pvl_base = float(group.calculated_pvl) * group.non_partner_units_sold
                    group.non_partner_pvl_base = round(new_pvl_base, 2)
                    group.updated_at = utc_now()
                    updated_count += 1

            self.db.commit()
            logger.info(f"[HOMOGENEOUS] Actualizados {updated_count} cálculos PVL base")

            return {
                "groups_updated": updated_count,
                "pharmacy_id": pharmacy_id,
                "update_date": utc_now().isoformat(),
            }

        except Exception as e:
            self.db.rollback()
            logger.error(f"[HOMOGENEOUS] Error actualizando PVL base: {str(e)}")
            raise

    def get_savings_opportunities_summary(self, pharmacy_id: str) -> Dict:
        """
        Obtiene resumen de oportunidades de ahorro para una farmacia.

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            Dict con resumen de oportunidades
        """
        try:
            groups = (
                self.db.query(HomogeneousGroup)
                .filter(
                    HomogeneousGroup.pharmacy_id == pharmacy_id,
                    HomogeneousGroup.has_partner_labs.is_(True),
                    HomogeneousGroup.non_partner_units_sold > 0,
                )
                .all()
            )

            if not groups:
                return {
                    "total_opportunities": 0,
                    "total_pvl_base": 0.0,
                    "groups_with_opportunities": 0,
                }

            total_units = sum(g.non_partner_units_sold for g in groups)
            total_pvl_base = sum(float(g.non_partner_pvl_base or 0) for g in groups)

            # Top 10 oportunidades
            top_opportunities = sorted(
                [g.to_dict() for g in groups],
                key=lambda x: x["non_partner_pvl_base"],
                reverse=True,
            )[:10]

            return {
                "total_opportunities": total_units,
                "total_pvl_base": round(total_pvl_base, 2),
                "groups_with_opportunities": len(groups),
                "top_opportunities": top_opportunities,
                "calculation_note": "Ahorro final = total_pvl_base * (descuento_slider / 100)",
            }

        except Exception as e:
            logger.error(f"[HOMOGENEOUS] Error obteniendo resumen: {str(e)}")
            raise
