﻿"""
Heartbeat tracker con fallback a base de datos.

Este módulo implementa un sistema de heartbeat robusto que puede
funcionar con Redis como sistema primario y PostgreSQL como fallback
cuando Redis no está disponible.
"""

import json
from datetime import datetime
from typing import Any, Dict, Optional

import structlog
from sqlalchemy.orm import Session

from app.core.cache_config import get_redis, REDIS_MODULE_AVAILABLE

# Redis exceptions - optional, not available in local Windows mode
try:
    from redis.exceptions import ConnectionError as RedisConnectionError
    from redis.exceptions import RedisError
except ImportError:
    # Fallback: use generic exceptions when redis module not available
    RedisConnectionError = ConnectionError  # type: ignore
    RedisError = Exception  # type: ignore
from app.models.system_status import SystemComponent, SystemStatus, SystemStatusEnum
from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)

# Configuración de heartbeat
HEARTBEAT_TTL_SECONDS = 1800  # 30 minutos
HEARTBEAT_KEY_PREFIX = "cima:heartbeat"


class HeartbeatTracker:
    """
    Sistema de heartbeat con fallback automático a base de datos.

    Intenta usar Redis primero para mejor performance, pero fallback
    a PostgreSQL si Redis no está disponible.
    """

    def __init__(self, db: Session):
        """
        Inicializar el tracker.

        Args:
            db: Sesión de base de datos
        """
        self.db = db
        self.redis_available = False
        self._check_redis_availability()

    def _check_redis_availability(self) -> bool:
        """Verificar si Redis está disponible."""
        try:
            redis_client = get_redis()
            if not redis_client:
                self.redis_available = False
                return False
            redis_client.ping()
            self.redis_available = True
            logger.debug("heartbeat.redis_available")
            return True
        except (RedisError, RedisConnectionError, Exception) as e:
            self.redis_available = False
            logger.warning("heartbeat.redis_unavailable", event_type="heartbeat", error=str(e))
            return False

    def update_heartbeat(
        self, phase: str, processed: int, total: int, additional_data: Optional[Dict[str, Any]] = None
    ) -> bool:
        """
        Actualizar el heartbeat con el estado actual.

        Args:
            phase: Fase actual de sincronización
            processed: Número de items procesados
            total: Total de items a procesar
            additional_data: Datos adicionales opcionales

        Returns:
            True si se actualizó exitosamente, False en caso contrario
        """
        heartbeat_data = {
            "phase": phase,
            "processed": processed,
            "total": total,
            "progress_percentage": round((processed / total * 100) if total > 0 else 0, 2),
            "timestamp": utc_now().isoformat(),
        }

        if additional_data:
            heartbeat_data.update(additional_data)

        # Intentar Redis primero
        if self._update_redis_heartbeat(heartbeat_data):
            return True

        # Fallback a base de datos
        return self._update_db_heartbeat(heartbeat_data)

    def _update_redis_heartbeat(self, data: Dict[str, Any]) -> bool:
        """
        Actualizar heartbeat en Redis.

        Args:
            data: Datos del heartbeat

        Returns:
            True si exitoso, False si falla
        """
        try:
            redis_client = get_redis()
            if not redis_client:
                self.redis_available = False
                return False

            key = f"{HEARTBEAT_KEY_PREFIX}:current"
            redis_client.setex(key, HEARTBEAT_TTL_SECONDS, json.dumps(data))
            self.redis_available = True
            logger.debug(
                "heartbeat.redis_updated",
                event_type="heartbeat",
                phase=data["phase"],
                progress_percentage=data["progress_percentage"],
            )
            return True

        except (RedisError, RedisConnectionError) as e:
            self.redis_available = False
            logger.warning(
                "heartbeat.redis_update_failed", event_type="heartbeat", error=str(e), error_type=type(e).__name__
            )
            return False

    def _update_db_heartbeat(self, data: Dict[str, Any]) -> bool:
        """
        Actualizar heartbeat en base de datos (fallback).

        Args:
            data: Datos del heartbeat

        Returns:
            True si exitoso, False si falla
        """
        try:
            # Buscar o crear registro usando component CIMA
            heartbeat_status = (
                self.db.query(SystemStatus).filter(SystemStatus.component == SystemComponent.CIMA).first()
            )

            if heartbeat_status:
                # Actualizar datos existentes
                heartbeat_status.details = json.dumps(data)
                heartbeat_status.message = f"Heartbeat: {data.get('phase', 'unknown')}"
                heartbeat_status.progress = int(data.get("progress_percentage", 0))
                heartbeat_status.updated_at = utc_now()
            else:
                # Crear nuevo registro
                heartbeat_status = SystemStatus(
                    component=SystemComponent.CIMA,
                    status=SystemStatusEnum.UPDATING,
                    details=json.dumps(data),
                    message=f"Heartbeat: {data.get('phase', 'unknown')}",
                    progress=int(data.get("progress_percentage", 0)),
                )
                self.db.add(heartbeat_status)

            self.db.commit()

            logger.info(
                "heartbeat.db_fallback_used",
                event_type="heartbeat",
                phase=data["phase"],
                progress_percentage=data["progress_percentage"],
                redis_available=False,
            )
            return True

        except Exception as e:
            self.db.rollback()
            logger.error(
                "heartbeat.db_update_failed", event_type="heartbeat", error=str(e), error_type=type(e).__name__
            )
            return False

    def get_heartbeat(self) -> Optional[Dict[str, Any]]:
        """
        Obtener el último heartbeat.

        Returns:
            Datos del heartbeat o None si no existe
        """
        # Intentar Redis primero
        heartbeat = self._get_redis_heartbeat()
        if heartbeat:
            return heartbeat

        # Fallback a base de datos
        return self._get_db_heartbeat()

    def _get_redis_heartbeat(self) -> Optional[Dict[str, Any]]:
        """
        Obtener heartbeat de Redis.

        Returns:
            Datos del heartbeat o None
        """
        try:
            redis_client = get_redis()
            if not redis_client:
                self.redis_available = False
                return None

            key = f"{HEARTBEAT_KEY_PREFIX}:current"
            data = redis_client.get(key)

            if data:
                self.redis_available = True
                return json.loads(data)

            return None

        except (RedisError, RedisConnectionError, Exception):
            self.redis_available = False
            return None

    def _get_db_heartbeat(self) -> Optional[Dict[str, Any]]:
        """
        Obtener heartbeat de base de datos (fallback).

        Returns:
            Datos del heartbeat o None
        """
        try:
            heartbeat_status = (
                self.db.query(SystemStatus).filter(SystemStatus.component == SystemComponent.CIMA).first()
            )

            if heartbeat_status and heartbeat_status.details:
                return json.loads(heartbeat_status.details)

            return None

        except Exception as e:
            logger.error("heartbeat.db_read_failed", event_type="heartbeat", error=str(e))
            return None

    def get_time_since_heartbeat(self) -> Optional[int]:
        """
        Calcular tiempo desde el último heartbeat.

        Returns:
            Segundos desde el último heartbeat, o None si no existe
        """
        heartbeat = self.get_heartbeat()
        if not heartbeat or "timestamp" not in heartbeat:
            return None

        try:
            last_update = datetime.fromisoformat(heartbeat["timestamp"])
            delta = utc_now() - last_update
            return int(delta.total_seconds())

        except (ValueError, TypeError) as e:
            logger.warning("heartbeat.timestamp_parse_error", event_type="heartbeat", error=str(e))
            return None

    def clear_heartbeat(self) -> None:
        """Limpiar el heartbeat actual."""
        # Limpiar Redis
        try:
            redis_client = get_redis()
            if redis_client:
                key = f"{HEARTBEAT_KEY_PREFIX}:current"
                redis_client.delete(key)
                logger.debug("heartbeat.redis_cleared")
        except (RedisError, RedisConnectionError):
            pass

        # Limpiar base de datos
        try:
            self.db.query(SystemStatus).filter(SystemStatus.component == SystemComponent.CIMA).delete()
            self.db.commit()
            logger.debug("heartbeat.db_cleared")
        except Exception as e:
            self.db.rollback()
            logger.warning("heartbeat.clear_failed", event_type="heartbeat", error=str(e))

    def get_status(self) -> Dict[str, Any]:
        """
        Obtener estado del sistema de heartbeat.

        Returns:
            Diccionario con el estado del tracker
        """
        heartbeat = self.get_heartbeat()
        time_since = self.get_time_since_heartbeat()

        return {
            "redis_available": self.redis_available,
            "has_heartbeat": heartbeat is not None,
            "last_heartbeat": heartbeat,
            "time_since_heartbeat_seconds": time_since,
            "storage_backend": "redis" if self.redis_available else "database",
        }
