﻿"""
Servicio de monitoreo de salud del sistema.
Recopila métricas, detecta problemas y genera alertas inteligentes.
Versión con migración 0e616a9b6e0b aplicada.
"""

import os
from datetime import datetime, timedelta
from typing import Dict, List

import psutil
import structlog
from sqlalchemy import desc, func, text
from sqlalchemy.orm import Session

from app.models.product_catalog import ProductCatalog
from app.models.system_health import DeveloperLog, PerformanceSnapshot, SystemAlert, SystemHealthMetric
from app.models.system_status import SystemComponent, SystemStatus, SystemStatusEnum
from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)


class HealthMonitoringService:
    """
    Servicio principal de monitoreo de salud para xFarma.
    Diseñado específicamente para administradores de farmacia.
    """

    def __init__(self, db: Session):
        self.db = db

    def take_performance_snapshot(self) -> PerformanceSnapshot:
        """
        Captura un snapshot completo del estado del sistema.
        """
        snapshot = PerformanceSnapshot()

        try:
            # Sistema general
            snapshot.cpu_usage_percent = psutil.cpu_percent(interval=1)
            memory = psutil.virtual_memory()
            snapshot.memory_usage_mb = memory.used / (1024 * 1024)
            snapshot.memory_available_mb = memory.available / (1024 * 1024)

            # Base de datos - estadísticas básicas
            snapshot.db_connections_active = self._get_db_connections()
            snapshot.db_connections_idle = 0  # Se podría implementar con queries específicas
            snapshot.db_queries_per_second = 0  # Requiere logs de PostgreSQL
            snapshot.db_avg_query_time_ms = 0  # Requiere logs de PostgreSQL

            # Redis - básico
            snapshot.redis_memory_usage_mb = 0  # Se podría implementar
            snapshot.redis_keys_count = 0
            snapshot.redis_operations_per_second = 0

            # APIs externas - simulado por ahora
            snapshot.cima_api_response_time_ms = None
            snapshot.nomenclator_api_response_time_ms = None
            snapshot.api_errors_last_hour = 0

            # Catálogo - datos reales
            catalog_stats = self._get_catalog_statistics()
            snapshot.catalog_total_products = catalog_stats["total"]
            snapshot.catalog_with_cima = catalog_stats["with_cima"]
            snapshot.catalog_with_nomenclator = catalog_stats["with_nomenclator"]
            snapshot.catalog_freshness_hours = catalog_stats["freshness_hours"]

            # Guardar snapshot
            self.db.add(snapshot)
            self.db.commit()

            logger.info(
                "Performance snapshot captured",
                cpu_percent=snapshot.cpu_usage_percent,
                memory_mb=snapshot.memory_usage_mb,
                catalog_products=snapshot.catalog_total_products,
            )

            return snapshot

        except Exception as e:
            logger.error("Error capturing performance snapshot", error=str(e))
            self.db.rollback()
            raise

    def record_metric(
        self,
        category: str,
        metric_name: str,
        value: float,
        unit: str = None,
        context: Dict = None,
    ) -> SystemHealthMetric:
        """
        Registra una métrica específica del sistema.
        """
        metric = SystemHealthMetric(
            category=category,
            metric_name=metric_name,
            metric_value=value,
            metric_unit=unit,
            context=context or {},
        )

        self.db.add(metric)
        self.db.commit()

        logger.debug(
            "Metric recorded",
            category=category,
            metric=metric_name,
            value=value,
            unit=unit,
        )

        return metric

    def create_alert(
        self,
        alert_type: str,
        severity: str,
        title: str,
        message: str,
        context: Dict = None,
    ) -> SystemAlert:
        """
        Crea una nueva alerta o actualiza una existente.
        """
        # Buscar alerta existente del mismo tipo
        existing_alert = self.db.query(SystemAlert).filter_by(alert_type=alert_type, is_active=True).first()

        if existing_alert:
            # Actualizar alerta existente
            existing_alert.last_occurred = utc_now()
            existing_alert.occurrence_count += 1
            existing_alert.message = message
            existing_alert.context = context or {}
            existing_alert.updated_at = utc_now()
            self.db.commit()

            logger.warning(
                "Alert updated",
                alert_type=alert_type,
                severity=severity,
                occurrences=existing_alert.occurrence_count,
            )

            return existing_alert
        else:
            # Crear nueva alerta
            alert = SystemAlert(
                alert_type=alert_type,
                severity=severity,
                title=title,
                message=message,
                context=context or {},
            )

            self.db.add(alert)
            self.db.commit()

            logger.warning(
                "New alert created",
                alert_type=alert_type,
                severity=severity,
                title=title,
            )

            return alert

    def check_system_health(self) -> Dict:
        """
        Ejecuta chequeos de salud específicos para farmacia.
        Retorna un resumen del estado del sistema.
        """
        health_status = {
            "overall_status": "healthy",
            "checks": {},
            "alerts": [],
            "recommendations": [],
        }

        try:
            # Chequeo 1: Frescura del catálogo
            catalog_check = self._check_catalog_freshness()
            health_status["checks"]["catalog_freshness"] = catalog_check
            if not catalog_check["healthy"]:
                health_status["overall_status"] = "warning"

            # Chequeo 2: Uso de memoria
            memory_check = self._check_memory_usage()
            health_status["checks"]["memory_usage"] = memory_check
            if not memory_check["healthy"]:
                if memory_check["severity"] == "critical":
                    health_status["overall_status"] = "critical"
                elif health_status["overall_status"] == "healthy":
                    health_status["overall_status"] = "warning"

            # Chequeo 3: Errores de sincronización recientes
            sync_check = self._check_recent_sync_errors()
            health_status["checks"]["sync_errors"] = sync_check
            if not sync_check["healthy"]:
                health_status["overall_status"] = "warning"

            # Chequeo 4: Cobertura de productos
            coverage_check = self._check_product_coverage()
            health_status["checks"]["product_coverage"] = coverage_check
            if not coverage_check["healthy"]:
                if health_status["overall_status"] == "healthy":
                    health_status["overall_status"] = "warning"

            # Obtener alertas activas
            active_alerts = (
                self.db.query(SystemAlert)
                .filter_by(is_active=True, is_acknowledged=False)
                .order_by(desc(SystemAlert.last_occurred))
                .limit(10)
                .all()
            )

            health_status["alerts"] = [
                {
                    "id": str(alert.id),
                    "type": alert.alert_type,
                    "severity": alert.severity,
                    "title": alert.title,
                    "message": alert.message,
                    "count": alert.occurrence_count,
                    "last_occurred": alert.last_occurred.isoformat(),
                }
                for alert in active_alerts
            ]

            logger.info(
                "System health check completed",
                overall_status=health_status["overall_status"],
                active_alerts=len(active_alerts),
            )

        except Exception as e:
            logger.error("Error during health check", error=str(e))
            health_status["overall_status"] = "error"
            health_status["error"] = str(e)

        return health_status

    def get_developer_logs(
        self,
        limit: int = 100,
        level: str = None,
        logger_name: str = None,
        since: datetime = None,
    ) -> List[Dict]:
        """
        Obtiene logs técnicos para modo desarrollador.
        """
        query = self.db.query(DeveloperLog)

        if level:
            query = query.filter(DeveloperLog.level == level.upper())
        if logger_name:
            query = query.filter(DeveloperLog.logger_name.ilike(f"%{logger_name}%"))
        if since:
            query = query.filter(DeveloperLog.timestamp >= since)

        logs = query.order_by(desc(DeveloperLog.timestamp)).limit(limit).all()

        return [
            {
                "id": str(log.id),
                "level": log.level,
                "logger": log.logger_name,
                "message": log.message,
                "stack_trace": log.stack_trace,
                "context": log.context,
                "execution_time_ms": log.execution_time_ms,
                "memory_usage_mb": log.memory_usage_mb,
                "timestamp": log.timestamp.isoformat(),
            }
            for log in logs
        ]

    def get_performance_trends(self, hours: int = 24) -> Dict:
        """
        Obtiene tendencias de performance para dashboard de salud.
        """
        since = utc_now() - timedelta(hours=hours)

        snapshots = (
            self.db.query(PerformanceSnapshot)
            .filter(PerformanceSnapshot.snapshot_at >= since)
            .order_by(PerformanceSnapshot.snapshot_at)
            .all()
        )

        trends = {
            "timestamps": [],
            "cpu_usage": [],
            "memory_usage": [],
            "catalog_products": [],
            "memory_available": [],
        }

        for snapshot in snapshots:
            trends["timestamps"].append(snapshot.snapshot_at.isoformat())
            trends["cpu_usage"].append(snapshot.cpu_usage_percent or 0)
            trends["memory_usage"].append(snapshot.memory_usage_mb or 0)
            trends["catalog_products"].append(snapshot.catalog_total_products or 0)
            trends["memory_available"].append(snapshot.memory_available_mb or 0)

        return trends

    # Métodos privados de chequeo

    def _check_catalog_freshness(self) -> Dict:
        """Chequea si el catálogo está actualizado."""
        try:
            # Obtener último estado de sincronización
            cima_status = self.db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()

            if not cima_status or not cima_status.updated_at:
                return {
                    "healthy": False,
                    "message": "No hay información de sincronización CIMA",
                    "severity": "high",
                }

            from datetime import timezone as dt_timezone

            hours_since_update = (datetime.now(dt_timezone.utc) - cima_status.updated_at).total_seconds() / 3600

            if hours_since_update > 240:  # 10 días
                self.create_alert(
                    "cima_outdated",
                    "high",
                    "Catálogo CIMA desactualizado",
                    f"CIMA no se ha actualizado en {hours_since_update:.1f} horas (>10 días)",
                    {"hours_since_update": hours_since_update},
                )
                return {
                    "healthy": False,
                    "message": f"CIMA desactualizado ({hours_since_update:.1f}h)",
                    "severity": "high",
                }
            elif hours_since_update > 168:  # 7 días
                return {
                    "healthy": False,
                    "message": f"CIMA necesita actualización ({hours_since_update:.1f}h)",
                    "severity": "medium",
                }

            return {
                "healthy": True,
                "message": f"Catálogo actualizado ({hours_since_update:.1f}h)",
                "severity": "low",
            }

        except Exception as e:
            logger.error("Error checking catalog freshness", error=str(e))
            return {
                "healthy": False,
                "message": "Error en chequeo",
                "severity": "medium",
            }

    def _check_memory_usage(self) -> Dict:
        """Chequea el uso de memoria del sistema."""
        try:
            memory = psutil.virtual_memory()
            usage_percent = memory.percent

            # En Render, generar alerta si uso > 80%
            is_render = os.getenv("RENDER") == "true"
            threshold = 80 if is_render else 90

            if usage_percent > threshold:
                self.create_alert(
                    "memory_warning",
                    "high" if is_render else "medium",
                    "Uso alto de memoria",
                    f"Uso de memoria: {usage_percent:.1f}% (umbral: {threshold}%)",
                    {"usage_percent": usage_percent, "threshold": threshold},
                )
                return {
                    "healthy": False,
                    "message": f"Memoria alta: {usage_percent:.1f}%",
                    "severity": "critical" if usage_percent > 95 else "high",
                }

            return {
                "healthy": True,
                "message": f"Memoria normal: {usage_percent:.1f}%",
                "severity": "low",
            }

        except Exception as e:
            logger.error("Error checking memory usage", error=str(e))
            return {
                "healthy": False,
                "message": "Error en chequeo memoria",
                "severity": "medium",
            }

    def _check_recent_sync_errors(self) -> Dict:
        """Chequea errores de sincronización recientes."""
        try:
            # Buscar fallos de sincronización en las últimas 24 horas
            since = utc_now() - timedelta(hours=24)

            error_statuses = (
                self.db.query(SystemStatus)
                .filter(
                    SystemStatus.updated_at >= since,
                    SystemStatus.status == SystemStatusEnum.ERROR,
                )
                .count()
            )

            if error_statuses > 3:
                self.create_alert(
                    "sync_failure",
                    "high",
                    "Múltiples fallos de sincronización",
                    f"{error_statuses} fallos de sincronización en 24h",
                    {"error_count": error_statuses},
                )
                return {
                    "healthy": False,
                    "message": f"{error_statuses} errores sincronización (24h)",
                    "severity": "high",
                }

            return {
                "healthy": True,
                "message": f"{error_statuses} errores sincronización (24h)",
                "severity": "low",
            }

        except Exception as e:
            logger.error("Error checking sync errors", error=str(e))
            return {
                "healthy": False,
                "message": "Error en chequeo sync",
                "severity": "medium",
            }

    def _check_product_coverage(self) -> Dict:
        """Chequea la cobertura de datos en el catálogo."""
        try:
            total_products = self.db.query(ProductCatalog).count()
            with_cima = self.db.query(ProductCatalog).filter(ProductCatalog.data_sources.ilike("%cima%")).count()

            if total_products == 0:
                return {
                    "healthy": False,
                    "message": "Catálogo vacío",
                    "severity": "critical",
                }

            coverage_percent = (with_cima / total_products) * 100

            if coverage_percent < 85:
                self.create_alert(
                    "product_mismatch",
                    "medium",
                    "Baja cobertura de productos",
                    f"Solo {coverage_percent:.1f}% productos tienen datos CIMA",
                    {"coverage_percent": coverage_percent},
                )
                return {
                    "healthy": False,
                    "message": f"Cobertura CIMA: {coverage_percent:.1f}%",
                    "severity": "medium",
                }

            return {
                "healthy": True,
                "message": f"Cobertura CIMA: {coverage_percent:.1f}%",
                "severity": "low",
            }

        except Exception as e:
            logger.error("Error checking product coverage", error=str(e))
            return {
                "healthy": False,
                "message": "Error en chequeo cobertura",
                "severity": "medium",
            }

    def _get_db_connections(self) -> int:
        """Obtiene el número de conexiones activas a la base de datos."""
        try:
            result = self.db.execute(text("SELECT count(*) FROM pg_stat_activity WHERE state = 'active'")).scalar()
            return result or 0
        except Exception as e:
            logger.warning(f"Failed to get active database connections: {e}")
            return 0

    def _get_catalog_statistics(self) -> Dict:
        """Obtiene estadísticas del catálogo."""
        try:
            total = self.db.query(ProductCatalog).count()
            with_cima = self.db.query(ProductCatalog).filter(ProductCatalog.data_sources.ilike("%cima%")).count()
            with_nomenclator = (
                self.db.query(ProductCatalog).filter(ProductCatalog.data_sources.ilike("%nomenclator%")).count()
            )

            # Calcular frescura basada en última actualización
            latest_update = self.db.query(func.max(ProductCatalog.updated_at)).scalar()
            freshness_hours = 0
            if latest_update:
                # Asegurar que ambos datetimes tengan la misma timezone awareness
                from datetime import timezone

                now_utc = datetime.now(timezone.utc)
                if latest_update.tzinfo is None:
                    # Si latest_update es naive, convertir a UTC
                    latest_update = latest_update.replace(tzinfo=timezone.utc)

                freshness_hours = (now_utc - latest_update).total_seconds() / 3600

            return {
                "total": total,
                "with_cima": with_cima,
                "with_nomenclator": with_nomenclator,
                "freshness_hours": freshness_hours,
            }

        except Exception as e:
            logger.error("Error getting catalog statistics", error=str(e))
            return {
                "total": 0,
                "with_cima": 0,
                "with_nomenclator": 0,
                "freshness_hours": 0,
            }


def log_to_developer_mode(
    db: Session,
    level: str,
    logger_name: str,
    message: str,
    stack_trace: str = None,
    context: Dict = None,
    execution_time_ms: float = None,
    memory_usage_mb: float = None,
):
    """
    Función helper para registrar logs en modo desarrollador.
    """
    try:
        dev_log = DeveloperLog(
            level=level.upper(),
            logger_name=logger_name,
            message=message,
            stack_trace=stack_trace,
            context=context or {},
            execution_time_ms=execution_time_ms,
            memory_usage_mb=memory_usage_mb,
        )

        db.add(dev_log)
        db.commit()
    except Exception as e:
        # No fallar si no se puede registrar el log
        logger.error("Failed to log to developer mode", error=str(e))
        db.rollback()
