﻿# backend/app/services/generic_opportunities_service.py
"""
Servicio para análisis de oportunidades de medicamentos genéricos por farmacia

Este servicio calcula y mantiene las métricas de oportunidades de descuento
basándose en el catálogo maestro y los datos de ventas específicos por farmacia.
"""

import logging
from datetime import datetime, timedelta
from decimal import Decimal
from typing import Any, Dict, List, Optional

from sqlalchemy import and_, text
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class GenericOpportunitiesService:
    """
    Servicio para calcular oportunidades de genéricos por farmacia

    Funcionalidades:
    - Calcular métricas de oportunidades basadas en datos de ventas reales
    - Actualizar PharmacyHomogeneousMetrics con datos calculados
    - Identificar conjuntos con mayor potencial de ahorro
    - Analizar penetración de partners por farmacia
    """

    def __init__(self):
        self.default_analysis_months = 13  # Meses de análisis por defecto

    def calculate_opportunities_for_pharmacy(
        self, db: Session, pharmacy_id: str, analysis_months: int = None,
        selected_employee_names: Optional[List[str]] = None
    ) -> Dict[str, Any]:
        """
        Calcula todas las oportunidades de genéricos para una farmacia específica

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            analysis_months: Meses de análisis (por defecto usa configuración de farmacia)
            selected_employee_names: Lista de nombres de empleados a filtrar (None o vacío = TODOS)

        Returns:
            Estadísticas completas de oportunidades

        Note:
            Issue #402: Employee filtering (PRO feature)
            DECISIÓN #2: Filtrado solo por employee_name (no employee_code)
            DECISIÓN UX #2: Lista vacía o None = TODOS los empleados (default)
            DECISIÓN UX #3: Incluye ventas sin empleado como "__sin_empleado__"
        """
        from app.models.pharmacy import Pharmacy

        logger.info(f"[OPPORTUNITIES] Calculando oportunidades para farmacia {pharmacy_id}")

        # Obtener datos de la farmacia
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise ValueError(f"Farmacia {pharmacy_id} no encontrada")

        # Determinar período de análisis
        months = analysis_months or int(pharmacy.analysis_period_months or self.default_analysis_months)
        end_date = utc_now()
        start_date = end_date - timedelta(days=months * 30)  # Aproximadamente

        stats = {
            "pharmacy_id": pharmacy_id,
            "analysis_period": {
                "start_date": start_date.isoformat(),
                "end_date": end_date.isoformat(),
                "months": months,
            },
            "partners": self._get_pharmacy_partners(pharmacy),
            "opportunities": [],
            "summary": {
                "total_homogeneous_groups": 0,
                "groups_with_opportunities": 0,
                "total_opportunity_units": 0,
                "total_opportunity_base_amount": 0.0,
                "avg_partner_penetration": 0.0,
            },
            "updated_metrics": 0,
        }

        try:
            # Obtener conjuntos homogéneos con ventas de esta farmacia
            homogeneous_groups = self._get_pharmacy_homogeneous_groups(
                db, pharmacy_id, start_date, end_date, selected_employee_names
            )

            logger.info(f"[OPPORTUNITIES] Procesando {len(homogeneous_groups)} conjuntos homogéneos")

            for group_code in homogeneous_groups:
                try:
                    # Calcular métricas para este conjunto
                    group_metrics = self._calculate_group_metrics(
                        db,
                        pharmacy_id,
                        group_code,
                        start_date,
                        end_date,
                        stats["partners"],
                        selected_employee_names,
                    )

                    if group_metrics:
                        # Actualizar o crear registro en PharmacyHomogeneousMetrics
                        self._update_pharmacy_metrics(db, pharmacy_id, group_code, group_metrics)

                        # Agregar a oportunidades si hay potencial de ahorro
                        if group_metrics.get("non_partner_units_sold", 0) > 0:
                            stats["opportunities"].append(
                                {
                                    "homogeneous_code": group_code,
                                    "homogeneous_name": group_metrics.get("homogeneous_name"),
                                    "opportunity_units": group_metrics["non_partner_units_sold"],
                                    "opportunity_base_amount": float(group_metrics.get("potential_savings_base", 0)),
                                    "partner_penetration": group_metrics.get("partner_penetration_percentage", 0),
                                    "total_units_sold": group_metrics["total_units_sold"],
                                    "calculated_pvl": float(group_metrics.get("calculated_pvl", 0)),
                                }
                            )
                            stats["summary"]["groups_with_opportunities"] += 1
                            stats["summary"]["total_opportunity_units"] += group_metrics["non_partner_units_sold"]
                            stats["summary"]["total_opportunity_base_amount"] += float(
                                group_metrics.get("potential_savings_base", 0)
                            )

                        stats["updated_metrics"] += 1

                except Exception as e:
                    logger.error(f"[OPPORTUNITIES] Error procesando grupo {group_code}: {str(e)}")
                    continue

            # Calcular estadísticas finales
            stats["summary"]["total_homogeneous_groups"] = len(homogeneous_groups)

            if stats["opportunities"]:
                penetrations = [opp["partner_penetration"] for opp in stats["opportunities"]]
                stats["summary"]["avg_partner_penetration"] = sum(penetrations) / len(penetrations)

                # Ordenar oportunidades por potencial (unidades * base monetaria)
                stats["opportunities"].sort(
                    key=lambda x: x["opportunity_units"] * x["opportunity_base_amount"],
                    reverse=True,
                )

            # Commit cambios
            db.commit()

            logger.info(
                f"[OPPORTUNITIES] Completado para farmacia {pharmacy_id}: {stats['summary']['groups_with_opportunities']} oportunidades encontradas"
            )

        except Exception as e:
            logger.error(f"[OPPORTUNITIES] Error calculando oportunidades: {str(e)}")
            db.rollback()
            raise

        return stats

    def _get_pharmacy_partners(self, pharmacy: "Pharmacy") -> List[str]:
        """Obtiene la lista de laboratorios partners de la farmacia"""

        if not pharmacy.partner_laboratories:
            return []

        # La farmacia guarda partners como lista JSON
        partners = pharmacy.partner_laboratories
        if isinstance(partners, list):
            return [p.strip() for p in partners if p and p.strip()]

        return []

    def _get_pharmacy_homogeneous_groups(
        self, db: Session, pharmacy_id: str, start_date: datetime, end_date: datetime,
        selected_employee_names: Optional[List[str]] = None
    ) -> List[str]:
        """
        Obtiene códigos homogéneos que tienen ventas en la farmacia durante el período

        Issue #402: Employee filtering (PRO feature)

        Args:
            db: Database session
            pharmacy_id: ID de la farmacia
            start_date: Fecha inicio del período
            end_date: Fecha fin del período
            selected_employee_names: Lista opcional de nombres de empleados para filtrar.
                                      Si None, incluye todas las ventas.
                                      Usa COALESCE para manejar NULL como '__sin_empleado__'

        Returns:
            Lista de códigos homogéneos únicos con ventas en el período

        Security:
            - SQL injection prevention: PostgreSQL array parameters (= ANY(:employee_names))
        """
        # Issue #402: Construir filtro de empleados
        employee_filter = ""
        params = {
            "pharmacy_id": pharmacy_id,
            "start_date": start_date,
            "end_date": end_date,
        }

        if selected_employee_names:
            employee_filter = "AND COALESCE(sd.employee_name, '__sin_empleado__') = ANY(:employee_names)"
            params["employee_names"] = selected_employee_names

        # Query para obtener códigos homogéneos con ventas en el período
        # Fix Issue #422: nomen_codigo_homogeneo está en product_catalog, no en sales_enrichment
        query = text(
            f"""
            SELECT DISTINCT pc.nomen_codigo_homogeneo
            FROM sales_data sd
            JOIN sales_enrichment se ON sd.id = se.sales_data_id
            JOIN product_catalog pc ON se.product_catalog_id = pc.id
            WHERE sd.pharmacy_id = :pharmacy_id
              AND sd.sale_date >= :start_date
              AND sd.sale_date <= :end_date
              {employee_filter}
              AND pc.nomen_codigo_homogeneo IS NOT NULL
              AND pc.nomen_codigo_homogeneo != ''
            ORDER BY pc.nomen_codigo_homogeneo
        """
        )

        result = db.execute(query, params)

        return [row[0] for row in result.fetchall()]

    def _calculate_group_metrics(
        self,
        db: Session,
        pharmacy_id: str,
        homogeneous_code: str,
        start_date: datetime,
        end_date: datetime,
        partner_labs: List[str],
    ) -> Optional[Dict[str, Any]]:
        """
        Calcula métricas específicas para un conjunto homogéneo y farmacia
        """
        from app.models.homogeneous_group_master import HomogeneousGroupMaster

        # Obtener datos del catálogo maestro
        master_group = (
            db.query(HomogeneousGroupMaster).filter(HomogeneousGroupMaster.homogeneous_code == homogeneous_code).first()
        )

        if not master_group:
            logger.warning(f"[OPPORTUNITIES] No se encontró grupo maestro para código {homogeneous_code}")
            return None

        # Query para obtener todas las ventas del conjunto en el período
        # FIX: Agregar JOIN a product_catalog para obtener campos nomen_* correctos
        query = text(
            """
            SELECT
                sd.id as sales_id,
                sd.quantity,
                sd.total_amount,
                pc.nomen_laboratorio,
                pc.nomen_precio_referencia,
                pc.nomen_pvp,
                pc.nomen_codigo_homogeneo
            FROM sales_data sd
            JOIN sales_enrichment se ON sd.id = se.sales_data_id
            JOIN product_catalog pc ON se.product_catalog_id = pc.id
            WHERE sd.pharmacy_id = :pharmacy_id
              AND sd.sale_date >= :start_date
              AND sd.sale_date <= :end_date
              AND pc.nomen_codigo_homogeneo = :homogeneous_code
            ORDER BY sd.sale_date DESC
        """
        )

        result = db.execute(
            query,
            {
                "pharmacy_id": pharmacy_id,
                "start_date": start_date,
                "end_date": end_date,
                "homogeneous_code": homogeneous_code,
            },
        )

        sales_records = result.fetchall()

        if not sales_records:
            return None

        # Obtener PVL calculado del grupo para usar como fallback
        # Este valor existe incluso cuando productos individuales no tienen precio_referencia
        group_calculated_pvl = float(master_group.calculated_pvl) if master_group.calculated_pvl else 0

        # VALIDACIÓN: PVL no puede ser negativo (constraint de negocio)
        if group_calculated_pvl < 0:
            logger.error(
                f"[OPPORTUNITIES] Invalid negative calculated_pvl: "
                f"grupo={homogeneous_code}, nombre={master_group.homogeneous_name}, "
                f"pvl={group_calculated_pvl}"
            )
            group_calculated_pvl = 0

        # Inicializar métricas
        metrics = {
            "homogeneous_code": homogeneous_code,
            "homogeneous_name": master_group.homogeneous_name,
            "calculated_pvl": master_group.calculated_pvl,
            # Totales
            "total_units_sold": 0,
            "total_revenue": 0.0,
            "total_transactions": len(sales_records),
            # Partners
            "partner_units_sold": 0,
            "partner_revenue": 0.0,
            "partner_transactions": 0,
            # No-partners (oportunidades)
            "non_partner_units_sold": 0,
            "non_partner_revenue": 0.0,
            "non_partner_transactions": 0,
            "potential_savings_base": 0.0,
            # Análisis
            "partner_laboratories_found": set(),
            "non_partner_laboratories_found": set(),
            # Período
            "data_period_start": start_date,
            "data_period_end": end_date,
        }

        # Procesar cada venta
        for record in sales_records:
            quantity_raw = record.quantity or 0
            revenue = float(record.total_amount or 0)
            lab = record.nomen_laboratorio

            # Fallback inteligente para PVL:
            # 1. Precio de referencia del producto individual (más preciso)
            # 2. PVL calculado del conjunto homogéneo (fallback para productos sin precio_referencia)
            # 3. 0 (sin datos disponibles)
            pvl_reference = float(
                record.nomen_precio_referencia or  # Prioridad 1: Precio individual
                group_calculated_pvl or            # Prioridad 2: PVL del grupo
                0                                  # Prioridad 3: Sin dato
            )

            # CORRECCIÓN: Cantidad neta considerando devoluciones (total_amount negativo)
            # Si total_amount < 0 → devolución → restar cantidad
            quantity_net = quantity_raw if revenue >= 0 else -quantity_raw

            # Totales
            metrics["total_units_sold"] += quantity_net
            metrics["total_revenue"] += revenue

            # Determinar si es partner o no
            is_partner = lab in partner_labs if lab else False

            if is_partner:
                metrics["partner_units_sold"] += quantity_net
                metrics["partner_revenue"] += revenue
                metrics["partner_transactions"] += 1
                metrics["partner_laboratories_found"].add(lab)
            else:
                metrics["non_partner_units_sold"] += quantity_net
                metrics["non_partner_revenue"] += revenue
                metrics["non_partner_transactions"] += 1

                # Base para cálculo de ahorro potencial
                if pvl_reference > 0:
                    # Base de ahorro = cantidad neta × PVL de referencia
                    metrics["potential_savings_base"] += quantity_net * pvl_reference
                else:
                    # Logging para productos sin precio disponible (ni individual ni grupo)
                    if quantity_net > 0:
                        logger.warning(
                            f"[OPPORTUNITIES] Sin precio para cálculo de ahorro: "
                            f"grupo={homogeneous_code}, lab={lab}, unidades={quantity_net}, "
                            f"precio_individual={record.nomen_precio_referencia}, "
                            f"pvl_grupo={group_calculated_pvl}"
                        )

                if lab:
                    metrics["non_partner_laboratories_found"].add(lab)

        # Convertir sets a listas para serialización
        metrics["partner_laboratories_found"] = list(metrics["partner_laboratories_found"])
        metrics["non_partner_laboratories_found"] = list(metrics["non_partner_laboratories_found"])

        # Calcular penetración de partners
        if metrics["total_units_sold"] > 0:
            metrics["partner_penetration_percentage"] = round(
                (metrics["partner_units_sold"] / metrics["total_units_sold"]) * 100, 2
            )
        else:
            metrics["partner_penetration_percentage"] = 0.0

        # Score de oportunidad
        if metrics["non_partner_units_sold"] > 0 and metrics["potential_savings_base"] > 0:
            # Normalizar score (0-100) basado en unidades y base monetaria
            units_factor = min(metrics["non_partner_units_sold"] / 100, 1.0) * 50
            amount_factor = min(metrics["potential_savings_base"] / 1000, 1.0) * 50
            metrics["opportunity_score"] = round(units_factor + amount_factor, 2)
        else:
            metrics["opportunity_score"] = 0.0

        return metrics

    def _update_pharmacy_metrics(
        self,
        db: Session,
        pharmacy_id: str,
        homogeneous_code: str,
        metrics: Dict[str, Any],
    ) -> None:
        """
        Actualiza o crea registro en PharmacyHomogeneousMetrics
        """
        from app.models.pharmacy_homogeneous_metrics import PharmacyHomogeneousMetrics

        # Buscar registro existente
        pharmacy_metric = (
            db.query(PharmacyHomogeneousMetrics)
            .filter(
                and_(
                    PharmacyHomogeneousMetrics.pharmacy_id == pharmacy_id,
                    PharmacyHomogeneousMetrics.homogeneous_code == homogeneous_code,
                )
            )
            .first()
        )

        if not pharmacy_metric:
            # Crear nuevo registro
            pharmacy_metric = PharmacyHomogeneousMetrics(pharmacy_id=pharmacy_id, homogeneous_code=homogeneous_code)
            db.add(pharmacy_metric)

        # Actualizar campos
        pharmacy_metric.has_partner_sales = metrics["partner_units_sold"] > 0
        pharmacy_metric.data_period_start = metrics["data_period_start"]
        pharmacy_metric.data_period_end = metrics["data_period_end"]

        # Métricas totales
        pharmacy_metric.total_units_sold = metrics["total_units_sold"]
        pharmacy_metric.total_transactions = metrics["total_transactions"]
        pharmacy_metric.total_revenue = Decimal(str(metrics["total_revenue"]))

        # Métricas de partners
        pharmacy_metric.partner_units_sold = metrics["partner_units_sold"]
        pharmacy_metric.partner_revenue = Decimal(str(metrics["partner_revenue"]))
        pharmacy_metric.partner_transactions = metrics["partner_transactions"]

        # Oportunidades
        pharmacy_metric.non_partner_units_sold = metrics["non_partner_units_sold"]
        pharmacy_metric.non_partner_revenue = Decimal(str(metrics["non_partner_revenue"]))
        pharmacy_metric.non_partner_transactions = metrics["non_partner_transactions"]
        pharmacy_metric.potential_savings_base = Decimal(str(metrics["potential_savings_base"]))

        # Score y análisis
        pharmacy_metric.avg_discount_opportunity = Decimal(str(metrics["opportunity_score"]))

        # Metadatos
        pharmacy_metric.calculation_metadata = {
            "calculation_date": utc_now().isoformat(),
            "analysis_method": "sales_data_aggregation",
            "partner_labs_found": metrics["partner_laboratories_found"],
            "non_partner_labs_found": metrics["non_partner_laboratories_found"],
            "data_quality_score": 0.95,  # Estimación
            "sales_records_processed": metrics["total_transactions"],
        }

        # Timestamp de actualización
        pharmacy_metric.updated_at = utc_now()


# Instancia global del servicio
generic_opportunities_service = GenericOpportunitiesService()
