# backend/app/services/feedback_service_v2.py
"""
Service para Human-in-the-Loop feedback del clasificador - V2.

Issue #457: Refactorizado para usar ProductCatalogVentaLibre.

Beneficios sobre V1:
- Cola de revisión SIN duplicados (cada producto único una vez)
- Correcciones se aplican al catálogo (afectan TODAS las ventas)
- Queries simples y rápidas (sin deduplicación en memoria)
- Estadísticas precisas de productos únicos

Prioridades de revisión (P1 > P2 > P3):
- P1: "otros" (fallback) - El clasificador no supo qué hacer
- P2: confidence < 0.60 - Clasificación dudosa
- P3: Resto (>= 0.60) - Auto-aprobados (90% precisión validada)
"""

from datetime import datetime, timezone
from typing import Optional
from uuid import UUID

from sqlalchemy import and_, case, func, or_
from sqlalchemy.orm import Session

from app.models.product_catalog_venta_libre import ProductCatalogVentaLibre
from app.models.product_correction import (
    CorrectionType,
    OutlierReason,
    OutlierStatus,
    PredictionSource,
    ProductCorrection,
)
from app.models.sales_enrichment import SalesEnrichment


class FeedbackServiceV2:
    """
    Servicio para gestión de feedback humano del clasificador.

    V2: Trabaja directamente con ProductCatalogVentaLibre.
    """

    # Umbrales de confianza
    # 2025-12-20: Bajado de 0.85 a 0.75, luego a 0.60 basado en análisis:
    # - 90% de productos P2 estaban bien clasificados
    # - Añadidas 60+ reglas de marcas para cubrir casos específicos
    LOW_CONFIDENCE_THRESHOLD = 0.60

    # Categorías que indican fallback del clasificador
    FALLBACK_CATEGORIES = {
        "otros", "parafarmacia_otros", "unknown", "sin_clasificar",
        "dolor", "vitaminas_general", "interno_no_venta", "higiene_general",
    }

    def __init__(self, db: Session):
        self.db = db

    def get_review_queue(
        self,
        limit: int = 50,
        offset: int = 0,
        priority: Optional[str] = None,
    ) -> dict:
        """
        Obtiene cola de productos únicos pendientes de revisión.

        Trabaja directamente con ProductCatalogVentaLibre, eliminando
        la necesidad de deduplicación.

        Args:
            limit: Máximo de productos a devolver
            offset: Paginación
            priority: "P1", "P2", "P3" o None para todos

        Returns:
            dict con items, total, y breakdown por prioridad
        """
        # Base query: productos activos, no outliers, no verificados
        base_query = (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.is_outlier == False)
            .filter(ProductCatalogVentaLibre.human_verified == False)
        )

        # Condiciones de prioridad
        p1_condition = or_(
            ProductCatalogVentaLibre.ml_category.in_(self.FALLBACK_CATEGORIES),
            ProductCatalogVentaLibre.ml_category.is_(None),
        )
        p2_condition = and_(
            ~ProductCatalogVentaLibre.ml_category.in_(self.FALLBACK_CATEGORIES),
            ProductCatalogVentaLibre.ml_category.isnot(None),
            ProductCatalogVentaLibre.ml_confidence.isnot(None),
            ProductCatalogVentaLibre.ml_confidence < self.LOW_CONFIDENCE_THRESHOLD,
        )
        p3_condition = and_(
            ~ProductCatalogVentaLibre.ml_category.in_(self.FALLBACK_CATEGORIES),
            ProductCatalogVentaLibre.ml_category.isnot(None),
            or_(
                ProductCatalogVentaLibre.ml_confidence.is_(None),
                ProductCatalogVentaLibre.ml_confidence >= self.LOW_CONFIDENCE_THRESHOLD,
            ),
        )

        # Aplicar filtro de prioridad
        if priority == "P1":
            base_query = base_query.filter(p1_condition)
        elif priority == "P2":
            base_query = base_query.filter(p2_condition)
        elif priority == "P3":
            base_query = base_query.filter(p3_condition)

        # Conteos
        total = base_query.count()

        # Breakdown por prioridad (solo si no hay filtro)
        breakdown = {}
        if priority is None:
            p1_count = base_query.filter(p1_condition).count()
            p2_count = base_query.filter(p2_condition).count()
            p3_count = base_query.filter(p3_condition).count()
            breakdown = {
                "P1_fallback": p1_count,
                "P2_low_confidence": p2_count,
                "P3_other": p3_count,
            }

        # Calcular prioridad para ordenar
        priority_case = case(
            (p1_condition, 1),
            (p2_condition, 2),
            else_=3,
        )

        # Obtener items ordenados por prioridad y ventas
        items = (
            base_query.order_by(
                priority_case,
                ProductCatalogVentaLibre.total_sales_count.desc(),
            )
            .offset(offset)
            .limit(limit)
            .all()
        )

        return {
            "items": [self._serialize_product(p) for p in items],
            "total": total,
            "breakdown": breakdown,
            "offset": offset,
            "limit": limit,
        }

    def _serialize_product(self, product: ProductCatalogVentaLibre) -> dict:
        """Serializa un producto del catálogo para la cola de revisión."""
        return {
            "id": str(product.id),
            "product_name": product.product_name_display,
            "product_name_normalized": product.product_name_normalized,
            "current_category": product.ml_category,
            "confidence_score": product.ml_confidence,
            "prediction_source": product.prediction_source,
            "brand_detected": product.detected_brand,
            "total_sales": product.total_sales_count,
            "pharmacies_count": product.pharmacies_count,
            "first_seen_at": product.first_seen_at.isoformat() if product.first_seen_at else None,
            "last_seen_at": product.last_seen_at.isoformat() if product.last_seen_at else None,
        }

    def approve_classification(
        self,
        product_id: UUID,
        reviewer_notes: Optional[str] = None,
    ) -> ProductCatalogVentaLibre:
        """
        Aprueba la clasificación actual del producto.

        Actualiza el catálogo (afecta a TODAS las ventas de este producto).
        """
        product = self.db.get(ProductCatalogVentaLibre, product_id)
        if not product:
            raise ValueError(f"Product {product_id} not found")

        if product.human_verified:
            raise ValueError(f"Product {product_id} already verified")

        # Marcar como verificado
        product.human_verified = True
        product.verified_category = product.ml_category  # Confirmar categoría actual
        product.verified_at = datetime.now(timezone.utc)
        product.reviewer_notes = reviewer_notes
        product.prediction_source = "human"  # Ahora validado por humano

        try:
            self.db.commit()
            self.db.refresh(product)
        except Exception:
            self.db.rollback()
            raise

        # Actualizar todos los SalesEnrichment que referencian este producto
        self._sync_enrichments_from_catalog(product)

        return product

    def correct_classification(
        self,
        product_id: UUID,
        corrected_category: str,
        reviewer_notes: Optional[str] = None,
    ) -> ProductCatalogVentaLibre:
        """
        Corrige la clasificación del producto.

        Actualiza el catálogo (afecta a TODAS las ventas de este producto).
        """
        product = self.db.get(ProductCatalogVentaLibre, product_id)
        if not product:
            raise ValueError(f"Product {product_id} not found")

        if product.human_verified:
            raise ValueError(f"Product {product_id} already verified")

        # Corregir clasificación
        product.human_verified = True
        product.verified_category = corrected_category
        product.verified_at = datetime.now(timezone.utc)
        product.reviewer_notes = reviewer_notes
        product.prediction_source = "human"

        try:
            self.db.commit()
            self.db.refresh(product)
        except Exception:
            self.db.rollback()
            raise

        # Actualizar todos los SalesEnrichment que referencian este producto
        self._sync_enrichments_from_catalog(product)

        return product

    def mark_as_outlier(
        self,
        product_id: UUID,
        outlier_reason: Optional[OutlierReason] = None,
        reviewer_notes: Optional[str] = None,
    ) -> ProductCatalogVentaLibre:
        """
        Marca un producto como outlier (no clasificable).
        """
        product = self.db.get(ProductCatalogVentaLibre, product_id)
        if not product:
            raise ValueError(f"Product {product_id} not found")

        if product.human_verified:
            raise ValueError(f"Product {product_id} already verified")

        # Marcar como outlier
        product.is_outlier = True
        product.outlier_reason = outlier_reason.value if outlier_reason else OutlierReason.OTRO.value
        product.human_verified = True
        product.verified_at = datetime.now(timezone.utc)
        product.reviewer_notes = reviewer_notes

        try:
            self.db.commit()
            self.db.refresh(product)
        except Exception:
            self.db.rollback()
            raise

        return product

    def _sync_enrichments_from_catalog(self, product: ProductCatalogVentaLibre) -> int:
        """
        Sincroniza todos los SalesEnrichment con la clasificación del catálogo.

        Esto asegura que cuando se corrige un producto en el catálogo,
        TODAS las ventas de ese producto se actualizan automáticamente.

        Returns:
            Número de registros actualizados
        """
        category = product.verified_category or product.ml_category

        updated = (
            self.db.query(SalesEnrichment)
            .filter(SalesEnrichment.venta_libre_product_id == product.id)
            .update({
                SalesEnrichment.ml_category: category,
                SalesEnrichment.ml_confidence: 1.0 if product.human_verified else product.ml_confidence,
            }, synchronize_session=False)
        )

        try:
            self.db.commit()
        except Exception:
            self.db.rollback()
            raise

        return updated

    def get_stats(self) -> dict:
        """Obtiene estadísticas del catálogo de venta libre."""
        total = self.db.query(ProductCatalogVentaLibre).filter(
            ProductCatalogVentaLibre.is_active == True
        ).count()

        verified = self.db.query(ProductCatalogVentaLibre).filter(
            ProductCatalogVentaLibre.is_active == True,
            ProductCatalogVentaLibre.human_verified == True,
        ).count()

        outliers = self.db.query(ProductCatalogVentaLibre).filter(
            ProductCatalogVentaLibre.is_outlier == True,
        ).count()

        pending = self.db.query(ProductCatalogVentaLibre).filter(
            ProductCatalogVentaLibre.is_active == True,
            ProductCatalogVentaLibre.is_outlier == False,
            ProductCatalogVentaLibre.human_verified == False,
        ).count()

        # Productos que necesitan revisión (P1 + P2)
        needs_review = self.db.query(ProductCatalogVentaLibre).filter(
            ProductCatalogVentaLibre.is_active == True,
            ProductCatalogVentaLibre.is_outlier == False,
            ProductCatalogVentaLibre.human_verified == False,
            or_(
                ProductCatalogVentaLibre.ml_category.in_(self.FALLBACK_CATEGORIES),
                ProductCatalogVentaLibre.ml_category.is_(None),
                and_(
                    ProductCatalogVentaLibre.ml_confidence.isnot(None),
                    ProductCatalogVentaLibre.ml_confidence < self.LOW_CONFIDENCE_THRESHOLD,
                ),
            ),
        ).count()

        # Top categorías
        top_categories = (
            self.db.query(
                ProductCatalogVentaLibre.ml_category,
                func.count(ProductCatalogVentaLibre.id).label("count"),
            )
            .filter(ProductCatalogVentaLibre.is_active == True)
            .filter(ProductCatalogVentaLibre.ml_category.isnot(None))
            .group_by(ProductCatalogVentaLibre.ml_category)
            .order_by(func.count(ProductCatalogVentaLibre.id).desc())
            .limit(10)
            .all()
        )

        return {
            "total_products": total,
            "verified": verified,
            "outliers": outliers,
            "pending_review": pending,
            "needs_review": needs_review,
            "verification_rate": round(verified / total * 100, 1) if total > 0 else 0,
            "top_categories": [{"category": cat, "count": cnt} for cat, cnt in top_categories],
        }

    def search_products(
        self,
        query: str,
        limit: int = 20,
        include_verified: bool = False,
    ) -> list[dict]:
        """
        Busca productos en el catálogo por nombre.

        Útil para encontrar productos específicos en la cola de revisión.
        """
        base_query = self.db.query(ProductCatalogVentaLibre).filter(
            ProductCatalogVentaLibre.is_active == True,
            ProductCatalogVentaLibre.product_name_normalized.ilike(f"%{query.lower()}%"),
        )

        if not include_verified:
            base_query = base_query.filter(
                ProductCatalogVentaLibre.human_verified == False
            )

        products = base_query.order_by(
            ProductCatalogVentaLibre.total_sales_count.desc()
        ).limit(limit).all()

        return [self._serialize_product(p) for p in products]

    def get_product_details(self, product_id: UUID) -> dict:
        """
        Obtiene detalles completos de un producto del catálogo.

        Incluye información adicional como variantes del nombre y
        referencias a ventas.
        """
        product = self.db.get(ProductCatalogVentaLibre, product_id)
        if not product:
            raise ValueError(f"Product {product_id} not found")

        # Contar ventas asociadas
        sales_count = (
            self.db.query(func.count(SalesEnrichment.id))
            .filter(SalesEnrichment.venta_libre_product_id == product.id)
            .scalar()
        )

        return {
            **self._serialize_product(product),
            "variants": product.product_name_variants or [],
            "verified_category": product.verified_category,
            "is_outlier": product.is_outlier,
            "outlier_reason": product.outlier_reason,
            "sales_enrichment_count": sales_count,
        }

    def get_verified_products(
        self,
        limit: int = 50,
        offset: int = 0,
        only_corrections: bool = False,
        search_query: Optional[str] = None,
    ) -> dict:
        """
        Obtiene productos YA verificados por humanos.

        Permite revisar decisiones previas y corregir errores humanos.

        Args:
            limit: Máximo de productos
            offset: Paginación
            only_corrections: Si True, solo muestra donde verified_category != ml_category
            search_query: Filtrar por nombre de producto

        Returns:
            dict con items, total
        """
        base_query = (
            self.db.query(ProductCatalogVentaLibre)
            .filter(ProductCatalogVentaLibre.human_verified == True)
            .filter(ProductCatalogVentaLibre.is_outlier == False)
        )

        # Filtro: solo correcciones (donde el humano cambió la categoría)
        if only_corrections:
            base_query = base_query.filter(
                ProductCatalogVentaLibre.verified_category != ProductCatalogVentaLibre.ml_category
            )

        # Filtro: búsqueda por nombre
        if search_query:
            base_query = base_query.filter(
                ProductCatalogVentaLibre.product_name_normalized.ilike(f"%{search_query.lower()}%")
            )

        total = base_query.count()

        # Ordenar por fecha de verificación (más reciente primero)
        items = (
            base_query.order_by(ProductCatalogVentaLibre.verified_at.desc())
            .offset(offset)
            .limit(limit)
            .all()
        )

        return {
            "items": [self._serialize_verified_product(p) for p in items],
            "total": total,
            "offset": offset,
            "limit": limit,
        }

    def _serialize_verified_product(self, product: ProductCatalogVentaLibre) -> dict:
        """Serializa un producto verificado para la vista de correcciones."""
        was_corrected = (
            product.verified_category is not None
            and product.verified_category != product.ml_category
        )
        return {
            "id": str(product.id),
            "product_name": product.product_name_display,
            "ml_category": product.ml_category,
            "verified_category": product.verified_category,
            "was_corrected": was_corrected,
            "verified_at": product.verified_at.isoformat() if product.verified_at else None,
            "reviewer_notes": product.reviewer_notes,
            "confidence_score": product.ml_confidence,
            "total_sales": product.total_sales_count,
        }

    def revert_and_recorrect(
        self,
        product_id: UUID,
        new_category: str,
        reviewer_notes: Optional[str] = None,
    ) -> ProductCatalogVentaLibre:
        """
        Corrige un producto YA verificado (corregir error humano).

        Permite al revisor cambiar una clasificación previa.
        Mantiene historial en reviewer_notes.

        Args:
            product_id: ID del producto
            new_category: Nueva categoría correcta
            reviewer_notes: Notas explicando el cambio

        Returns:
            Producto actualizado
        """
        product = self.db.get(ProductCatalogVentaLibre, product_id)
        if not product:
            raise ValueError(f"Product {product_id} not found")

        if not product.human_verified:
            raise ValueError(f"Product {product_id} was not verified yet")

        # Guardar historial del cambio en notas
        old_category = product.verified_category or product.ml_category
        timestamp = datetime.now(timezone.utc).strftime("%Y-%m-%d %H:%M")
        history_note = f"[{timestamp}] Recorregido: {old_category} → {new_category}"

        if reviewer_notes:
            history_note += f" | Razón: {reviewer_notes}"

        # Append al historial existente
        existing_notes = product.reviewer_notes or ""
        if existing_notes:
            product.reviewer_notes = f"{existing_notes}\n{history_note}"
        else:
            product.reviewer_notes = history_note

        # Actualizar clasificación
        product.verified_category = new_category
        product.verified_at = datetime.now(timezone.utc)

        try:
            self.db.commit()
            self.db.refresh(product)
        except Exception:
            self.db.rollback()
            raise

        # Sincronizar con SalesEnrichment
        self._sync_enrichments_from_catalog(product)

        return product

    def revert_to_pending(
        self,
        product_id: UUID,
        reviewer_notes: Optional[str] = None,
    ) -> ProductCatalogVentaLibre:
        """
        Revierte un producto verificado a estado pendiente.

        Útil cuando el revisor se da cuenta que necesita más información
        antes de decidir.

        Args:
            product_id: ID del producto
            reviewer_notes: Razón del revert

        Returns:
            Producto actualizado
        """
        product = self.db.get(ProductCatalogVentaLibre, product_id)
        if not product:
            raise ValueError(f"Product {product_id} not found")

        if not product.human_verified:
            raise ValueError(f"Product {product_id} was not verified yet")

        # Guardar historial
        timestamp = datetime.now(timezone.utc).strftime("%Y-%m-%d %H:%M")
        history_note = f"[{timestamp}] Revertido a pendiente"
        if reviewer_notes:
            history_note += f" | Razón: {reviewer_notes}"

        existing_notes = product.reviewer_notes or ""
        if existing_notes:
            product.reviewer_notes = f"{existing_notes}\n{history_note}"
        else:
            product.reviewer_notes = history_note

        # Revertir estado
        product.human_verified = False
        product.verified_category = None
        product.verified_at = None

        try:
            self.db.commit()
            self.db.refresh(product)
        except Exception:
            self.db.rollback()
            raise

        return product
