# backend/app/services/excel_generator.py
"""
Generador de Excel Pick-Lists para Suelo (Issue #511).

Usa openpyxl para generar un workbook con 3 hojas operativas:
- Liquidar: Stock muerto (>180 días sin venta)
- Reponer: Stock crítico (<7 días cobertura)
- Promocionar: Alto margen, baja rotación
"""

import io
import structlog
from openpyxl import Workbook
from openpyxl.styles import (
    Alignment,
    Border,
    Font,
    PatternFill,
    Side,
)
from openpyxl.utils import get_column_letter

from app.schemas.reports import SueloReportData, ProductPickItem

logger = structlog.get_logger(__name__)

# Estilos corporativos
STYLES = {
    "header_fill": PatternFill(start_color="2C3E50", end_color="2C3E50", fill_type="solid"),
    "header_font": Font(bold=True, color="FFFFFF", size=10),
    "row_alt_fill": PatternFill(start_color="ECF0F1", end_color="ECF0F1", fill_type="solid"),
    "priority_high_fill": PatternFill(start_color="E74C3C", end_color="E74C3C", fill_type="solid"),
    "priority_medium_fill": PatternFill(start_color="F39C12", end_color="F39C12", fill_type="solid"),
    "priority_low_fill": PatternFill(start_color="27AE60", end_color="27AE60", fill_type="solid"),
    "border": Border(
        left=Side(style="thin", color="BDC3C7"),
        right=Side(style="thin", color="BDC3C7"),
        top=Side(style="thin", color="BDC3C7"),
        bottom=Side(style="thin", color="BDC3C7"),
    ),
}


class SueloExcelGenerator:
    """
    Genera Excel Pick-Lists para Suelo.

    3 hojas con productos accionables:
    - Liquidar: Stock muerto a eliminar
    - Reponer: Stock crítico a pedir
    - Promocionar: Stock a activar con ofertas
    """

    def __init__(self):
        self.column_headers = [
            "Prioridad",
            "Producto",
            "Cod. Nacional",
            "EAN",
            "Ubicación",
            "Valor (€)",
            "Unidades",
            "Razón",
        ]
        self.column_widths = [10, 45, 14, 16, 12, 12, 10, 35]

    def generate(self, data: SueloReportData) -> bytes:
        """
        Genera Excel con 3 hojas de pick-lists.

        Args:
            data: SueloReportData con las 3 listas de productos

        Returns:
            bytes del archivo Excel generado
        """
        logger.info(
            "excel_generator.generate.start",
            pharmacy=data.pharmacy_name,
            liquidation_count=len(data.liquidation_items),
            restock_count=len(data.restock_items),
            promotion_count=len(data.promotion_items),
        )

        wb = Workbook()

        # Crear las 3 hojas
        self._create_sheet(
            wb,
            sheet_name="Liquidar",
            items=data.liquidation_items,
            summary_title="PRODUCTOS A LIQUIDAR",
            summary_value=f"Valor total: {data.liquidation_total_value:,.0f}€".replace(",", "."),
            is_first=True,
        )

        self._create_sheet(
            wb,
            sheet_name="Reponer",
            items=data.restock_items,
            summary_title="PRODUCTOS A REPONER",
            summary_value=f"Urgentes (P1): {data.restock_urgency_count} productos",
            is_first=False,
        )

        self._create_sheet(
            wb,
            sheet_name="Promocionar",
            items=data.promotion_items,
            summary_title="PRODUCTOS A PROMOCIONAR",
            summary_value=f"Margen potencial: {data.promotion_potential_margin:,.0f}€".replace(",", "."),
            is_first=False,
        )

        # Generar bytes
        buffer = io.BytesIO()
        wb.save(buffer)
        buffer.seek(0)

        logger.info("excel_generator.generate.complete")

        return buffer.getvalue()

    def _create_sheet(
        self,
        wb: Workbook,
        sheet_name: str,
        items: list[ProductPickItem],
        summary_title: str,
        summary_value: str,
        is_first: bool = False,
    ) -> None:
        """Crear una hoja con la lista de productos."""
        if is_first:
            ws = wb.active
            ws.title = sheet_name
        else:
            ws = wb.create_sheet(title=sheet_name)

        # Fila 1: Título
        ws.merge_cells("A1:H1")
        title_cell = ws["A1"]
        title_cell.value = summary_title
        title_cell.font = Font(bold=True, size=14, color="2C3E50")
        title_cell.alignment = Alignment(horizontal="left", vertical="center")
        ws.row_dimensions[1].height = 25

        # Fila 2: Resumen
        ws.merge_cells("A2:H2")
        summary_cell = ws["A2"]
        summary_cell.value = summary_value
        summary_cell.font = Font(size=11, color="7F8C8D")
        summary_cell.alignment = Alignment(horizontal="left", vertical="center")
        ws.row_dimensions[2].height = 20

        # Fila 3: Vacía
        ws.row_dimensions[3].height = 10

        # Fila 4: Headers
        header_row = 4
        for col_idx, header in enumerate(self.column_headers, start=1):
            cell = ws.cell(row=header_row, column=col_idx, value=header)
            cell.fill = STYLES["header_fill"]
            cell.font = STYLES["header_font"]
            cell.border = STYLES["border"]
            cell.alignment = Alignment(horizontal="center", vertical="center")

        ws.row_dimensions[header_row].height = 22

        # Filas de datos
        if not items:
            # Fila vacía con mensaje
            ws.merge_cells(f"A5:H5")
            empty_cell = ws["A5"]
            empty_cell.value = "Sin productos en esta lista"
            empty_cell.font = Font(italic=True, color="95A5A6")
            empty_cell.alignment = Alignment(horizontal="center", vertical="center")
        else:
            for row_idx, item in enumerate(items, start=5):
                self._write_item_row(ws, row_idx, item)

                # Alternate row color
                if (row_idx - 5) % 2 == 1:
                    for col_idx in range(1, 9):
                        ws.cell(row=row_idx, column=col_idx).fill = STYLES["row_alt_fill"]

        # Ajustar anchos de columna
        for col_idx, width in enumerate(self.column_widths, start=1):
            ws.column_dimensions[get_column_letter(col_idx)].width = width

        # Freeze panes (header row)
        ws.freeze_panes = "A5"

    def _write_item_row(
        self, ws, row_idx: int, item: ProductPickItem
    ) -> None:
        """Escribir una fila de producto."""
        # Prioridad con color
        priority_cell = ws.cell(row=row_idx, column=1, value=f"P{item.priority}")
        priority_cell.alignment = Alignment(horizontal="center", vertical="center")
        priority_cell.font = Font(bold=True, color="FFFFFF")
        if item.priority == 1:
            priority_cell.fill = STYLES["priority_high_fill"]
        elif item.priority == 2:
            priority_cell.fill = STYLES["priority_medium_fill"]
        else:
            priority_cell.fill = STYLES["priority_low_fill"]

        # Producto
        ws.cell(row=row_idx, column=2, value=item.product_name).alignment = Alignment(
            horizontal="left", vertical="center", wrap_text=True
        )

        # Cod. Nacional
        ws.cell(row=row_idx, column=3, value=item.codigo_nacional).alignment = Alignment(
            horizontal="center", vertical="center"
        )

        # EAN
        ws.cell(row=row_idx, column=4, value=item.ean or "—").alignment = Alignment(
            horizontal="center", vertical="center"
        )

        # Ubicación
        ws.cell(row=row_idx, column=5, value=item.location_hint or "N/A").alignment = Alignment(
            horizontal="center", vertical="center"
        )

        # Valor
        value_cell = ws.cell(row=row_idx, column=6, value=item.value_eur)
        value_cell.number_format = '#,##0.00 €'
        value_cell.alignment = Alignment(horizontal="right", vertical="center")

        # Unidades
        ws.cell(row=row_idx, column=7, value=item.quantity).alignment = Alignment(
            horizontal="center", vertical="center"
        )

        # Razón
        ws.cell(row=row_idx, column=8, value=item.reason).alignment = Alignment(
            horizontal="left", vertical="center", wrap_text=True
        )

        # Borders
        for col_idx in range(1, 9):
            ws.cell(row=row_idx, column=col_idx).border = STYLES["border"]


# Instancia singleton
excel_generator = SueloExcelGenerator()
