# backend/app/services/etl_service.py
"""
Servicio ETL Nocturno - Issue #543

Actualiza las vistas materializadas (mv_*) cada noche a las 4:00 AM
para que los dashboards carguen en <2 segundos.

Características:
- IDEMPOTENTE: Se puede ejecutar múltiples veces sin duplicar datos
- TRANSACCIONAL: Todo o nada, no deja datos a medias
- LOGGED: Registra cada ejecución en etl_run_log
"""

import json
import structlog
from datetime import date, datetime, timedelta, timezone
from typing import Optional
from uuid import UUID

from sqlalchemy import text, func, and_, case
from sqlalchemy.orm import Session

from app.models import SalesData, SalesEnrichment, InventorySnapshot, Pharmacy
from app.models.materialized_views import (
    MVSalesMonthly,
    MVSalesCategory,
    MVSalesBrand,
    MVStockKPIs,
    MVWeeklyKPIs,
    ETLRunLog,
)

logger = structlog.get_logger(__name__)


class ETLService:
    """
    Servicio para actualización nocturna de vistas materializadas.

    Uso:
        etl = ETLService()
        await etl.run_nightly(db, pharmacy_id)  # Para una farmacia
        await etl.run_all_pharmacies(db)         # Para todas
    """

    async def run_nightly(
        self,
        db: Session,
        pharmacy_id: UUID,
        force_full_refresh: bool = False,
    ) -> dict:
        """
        Ejecutar ETL nocturno para una farmacia.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            force_full_refresh: Recalcular todo (no incremental)

        Returns:
            dict con métricas de la ejecución
        """
        run_log = ETLRunLog(
            pharmacy_id=pharmacy_id,
            run_type="nightly",
            started_at=datetime.now(timezone.utc),
            status="running",
        )
        db.add(run_log)
        db.commit()

        logger.info(
            "etl_service.run_nightly.start",
            pharmacy_id=str(pharmacy_id),
            force_full_refresh=force_full_refresh,
        )

        metrics = {
            "tables_updated": 0,
            "rows_processed": 0,
            "details": {},
        }

        try:
            # 1. Actualizar ventas mensuales
            result = await self._refresh_sales_monthly(db, pharmacy_id)
            metrics["details"]["mv_sales_monthly"] = result
            metrics["tables_updated"] += 1
            metrics["rows_processed"] += result.get("rows", 0)

            # 2. Actualizar ventas por categoría
            result = await self._refresh_sales_category(db, pharmacy_id)
            metrics["details"]["mv_sales_category"] = result
            metrics["tables_updated"] += 1
            metrics["rows_processed"] += result.get("rows", 0)

            # 3. Actualizar ventas por marca
            result = await self._refresh_sales_brand(db, pharmacy_id)
            metrics["details"]["mv_sales_brand"] = result
            metrics["tables_updated"] += 1
            metrics["rows_processed"] += result.get("rows", 0)

            # 4. Actualizar KPIs de stock
            result = await self._refresh_stock_kpis(db, pharmacy_id)
            metrics["details"]["mv_stock_kpis"] = result
            metrics["tables_updated"] += 1
            metrics["rows_processed"] += result.get("rows", 0)

            # 5. Actualizar KPIs semanales
            result = await self._refresh_weekly_kpis(db, pharmacy_id)
            metrics["details"]["mv_weekly_kpis"] = result
            metrics["tables_updated"] += 1
            metrics["rows_processed"] += result.get("rows", 0)

            # Actualizar log de éxito
            run_log.status = "success"
            run_log.completed_at = datetime.now(timezone.utc)
            run_log.duration_seconds = int(
                (run_log.completed_at - run_log.started_at).total_seconds()
            )
            run_log.rows_processed = metrics["rows_processed"]
            run_log.tables_updated = metrics["tables_updated"]
            run_log.details = json.dumps(metrics["details"])
            db.commit()

            logger.info(
                "etl_service.run_nightly.success",
                pharmacy_id=str(pharmacy_id),
                duration_seconds=run_log.duration_seconds,
                rows_processed=metrics["rows_processed"],
            )

            return metrics

        except Exception as e:
            logger.error(
                "etl_service.run_nightly.error",
                pharmacy_id=str(pharmacy_id),
                error=str(e),
                exc_info=True,
            )

            run_log.status = "failed"
            run_log.completed_at = datetime.now(timezone.utc)
            run_log.duration_seconds = int(
                (run_log.completed_at - run_log.started_at).total_seconds()
            )
            run_log.error_message = str(e)[:500]
            db.commit()

            raise

    async def run_all_pharmacies(self, db: Session) -> dict:
        """
        Ejecutar ETL para todas las farmacias activas.

        Returns:
            dict con resumen de todas las ejecuciones
        """
        logger.info("etl_service.run_all_pharmacies.start")

        pharmacies = db.query(Pharmacy.id).filter(Pharmacy.is_active == True).all()

        results = {
            "total": len(pharmacies),
            "success": 0,
            "failed": 0,
            "errors": [],
        }

        for (pharmacy_id,) in pharmacies:
            try:
                await self.run_nightly(db, pharmacy_id)
                results["success"] += 1
            except Exception as e:
                results["failed"] += 1
                results["errors"].append({
                    "pharmacy_id": str(pharmacy_id),
                    "error": str(e),
                })

        logger.info(
            "etl_service.run_all_pharmacies.complete",
            total=results["total"],
            success=results["success"],
            failed=results["failed"],
        )

        return results

    async def _refresh_sales_monthly(
        self, db: Session, pharmacy_id: UUID
    ) -> dict:
        """
        Actualizar mv_sales_monthly.

        Estrategia: DELETE + INSERT en transacción (idempotente)
        """
        logger.debug("etl_service.refresh_sales_monthly", pharmacy_id=str(pharmacy_id))

        # Borrar datos existentes de esta farmacia
        db.query(MVSalesMonthly).filter(
            MVSalesMonthly.pharmacy_id == pharmacy_id
        ).delete()

        # Calcular agregados mensuales
        # Agrupamos por mes (primer día del mes)
        query = (
            db.query(
                func.date_trunc('month', SalesData.sale_date).label('year_month'),
                # Venta libre
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'venta_libre', SalesData.total_amount),
                        else_=0
                    )
                ).label('vl_total_sales'),
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'venta_libre', SalesData.quantity),
                        else_=0
                    )
                ).label('vl_total_units'),
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'venta_libre', SalesData.margin_amount),
                        else_=0
                    )
                ).label('vl_total_margin'),
                func.count(
                    func.distinct(
                        case(
                            (SalesEnrichment.product_type == 'venta_libre', SalesData.codigo_nacional),
                            else_=None
                        )
                    )
                ).label('vl_sku_count'),
                # Prescripción
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'prescripcion', SalesData.total_amount),
                        else_=0
                    )
                ).label('rx_total_sales'),
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'prescripcion', SalesData.quantity),
                        else_=0
                    )
                ).label('rx_total_units'),
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'prescripcion', SalesData.margin_amount),
                        else_=0
                    )
                ).label('rx_total_margin'),
                func.count(
                    func.distinct(
                        case(
                            (SalesEnrichment.product_type == 'prescripcion', SalesData.codigo_nacional),
                            else_=None
                        )
                    )
                ).label('rx_sku_count'),
                # Totales
                func.sum(SalesData.total_amount).label('total_sales'),
                func.sum(SalesData.quantity).label('total_units'),
                func.sum(SalesData.margin_amount).label('total_margin'),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .group_by(func.date_trunc('month', SalesData.sale_date))
        )

        rows_inserted = 0
        now = datetime.now(timezone.utc)

        for row in query.all():
            if row.year_month is None:
                continue

            # Calcular % margen
            vl_margin_pct = 0
            if row.vl_total_sales and row.vl_total_sales > 0:
                vl_margin_pct = (float(row.vl_total_margin or 0) / float(row.vl_total_sales)) * 100

            mv = MVSalesMonthly(
                pharmacy_id=pharmacy_id,
                year_month=row.year_month.date() if hasattr(row.year_month, 'date') else row.year_month,
                vl_total_sales=row.vl_total_sales or 0,
                vl_total_units=row.vl_total_units or 0,
                vl_total_margin=row.vl_total_margin or 0,
                vl_margin_percent=vl_margin_pct,
                vl_sku_count=row.vl_sku_count or 0,
                rx_total_sales=row.rx_total_sales or 0,
                rx_total_units=row.rx_total_units or 0,
                rx_total_margin=row.rx_total_margin or 0,
                rx_sku_count=row.rx_sku_count or 0,
                total_sales=row.total_sales or 0,
                total_units=row.total_units or 0,
                total_margin=row.total_margin or 0,
                updated_at=now,
            )
            db.add(mv)
            rows_inserted += 1

        db.commit()

        return {"rows": rows_inserted, "status": "success"}

    async def _refresh_sales_category(
        self, db: Session, pharmacy_id: UUID
    ) -> dict:
        """Actualizar mv_sales_category."""
        logger.debug("etl_service.refresh_sales_category", pharmacy_id=str(pharmacy_id))

        db.query(MVSalesCategory).filter(
            MVSalesCategory.pharmacy_id == pharmacy_id
        ).delete()

        query = (
            db.query(
                func.date_trunc('month', SalesData.sale_date).label('year_month'),
                SalesEnrichment.ml_category,
                func.sum(SalesData.total_amount).label('total_sales'),
                func.sum(SalesData.quantity).label('total_units'),
                func.sum(SalesData.margin_amount).label('total_margin'),
                func.count(func.distinct(SalesData.codigo_nacional)).label('sku_count'),
                func.count(SalesData.id).label('transaction_count'),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesEnrichment.product_type == 'venta_libre',
                SalesEnrichment.ml_category.isnot(None),
            )
            .group_by(
                func.date_trunc('month', SalesData.sale_date),
                SalesEnrichment.ml_category,
            )
        )

        rows_inserted = 0
        now = datetime.now(timezone.utc)

        for row in query.all():
            if row.year_month is None or row.ml_category is None:
                continue

            margin_pct = 0
            if row.total_sales and row.total_sales > 0:
                margin_pct = (float(row.total_margin or 0) / float(row.total_sales)) * 100

            mv = MVSalesCategory(
                pharmacy_id=pharmacy_id,
                year_month=row.year_month.date() if hasattr(row.year_month, 'date') else row.year_month,
                ml_category=row.ml_category,
                total_sales=row.total_sales or 0,
                total_units=row.total_units or 0,
                total_margin=row.total_margin or 0,
                margin_percent=margin_pct,
                sku_count=row.sku_count or 0,
                transaction_count=row.transaction_count or 0,
                updated_at=now,
            )
            db.add(mv)
            rows_inserted += 1

        db.commit()

        return {"rows": rows_inserted, "status": "success"}

    async def _refresh_sales_brand(
        self, db: Session, pharmacy_id: UUID
    ) -> dict:
        """Actualizar mv_sales_brand."""
        logger.debug("etl_service.refresh_sales_brand", pharmacy_id=str(pharmacy_id))

        db.query(MVSalesBrand).filter(
            MVSalesBrand.pharmacy_id == pharmacy_id
        ).delete()

        query = (
            db.query(
                func.date_trunc('month', SalesData.sale_date).label('year_month'),
                SalesEnrichment.detected_brand,
                func.mode().within_group(SalesEnrichment.ml_category).label('ml_category'),
                func.sum(SalesData.total_amount).label('total_sales'),
                func.sum(SalesData.quantity).label('total_units'),
                func.sum(SalesData.margin_amount).label('total_margin'),
                func.count(func.distinct(SalesData.codigo_nacional)).label('sku_count'),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesEnrichment.product_type == 'venta_libre',
                SalesEnrichment.detected_brand.isnot(None),
            )
            .group_by(
                func.date_trunc('month', SalesData.sale_date),
                SalesEnrichment.detected_brand,
            )
        )

        rows_inserted = 0
        now = datetime.now(timezone.utc)

        for row in query.all():
            if row.year_month is None or row.detected_brand is None:
                continue

            margin_pct = 0
            if row.total_sales and row.total_sales > 0:
                margin_pct = (float(row.total_margin or 0) / float(row.total_sales)) * 100

            mv = MVSalesBrand(
                pharmacy_id=pharmacy_id,
                year_month=row.year_month.date() if hasattr(row.year_month, 'date') else row.year_month,
                detected_brand=row.detected_brand,
                ml_category=row.ml_category,
                total_sales=row.total_sales or 0,
                total_units=row.total_units or 0,
                total_margin=row.total_margin or 0,
                margin_percent=margin_pct,
                sku_count=row.sku_count or 0,
                updated_at=now,
            )
            db.add(mv)
            rows_inserted += 1

        db.commit()

        return {"rows": rows_inserted, "status": "success"}

    async def _refresh_stock_kpis(
        self, db: Session, pharmacy_id: UUID
    ) -> dict:
        """Actualizar mv_stock_kpis."""
        logger.debug("etl_service.refresh_stock_kpis", pharmacy_id=str(pharmacy_id))

        db.query(MVStockKPIs).filter(
            MVStockKPIs.pharmacy_id == pharmacy_id
        ).delete()

        today = date.today()
        days_30_ago = today - timedelta(days=30)

        # Subquery: última venta por producto
        last_sale_subq = (
            db.query(
                SalesData.codigo_nacional,
                func.max(SalesData.sale_date).label('last_sale_date'),
                (func.sum(SalesData.quantity) / 30.0).label('daily_velocity'),
            )
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= days_30_ago,
            )
            .group_by(SalesData.codigo_nacional)
            .subquery()
        )

        # Query principal: stock + métricas de venta
        query = (
            db.query(
                InventorySnapshot.product_code,
                InventorySnapshot.product_name,
                InventorySnapshot.stock_quantity,
                InventorySnapshot.stock_value,
                last_sale_subq.c.last_sale_date,
                last_sale_subq.c.daily_velocity,
            )
            .outerjoin(
                last_sale_subq,
                InventorySnapshot.product_code == last_sale_subq.c.codigo_nacional,
            )
            .filter(
                InventorySnapshot.pharmacy_id == pharmacy_id,
                InventorySnapshot.stock_quantity > 0,
            )
        )

        rows_inserted = 0
        now = datetime.now(timezone.utc)

        for row in query.all():
            # Calcular días sin venta
            days_without_sale = 999
            if row.last_sale_date:
                days_without_sale = (today - row.last_sale_date).days

            # Calcular cobertura
            coverage_days = 999
            daily_vel = float(row.daily_velocity or 0)
            if daily_vel > 0:
                coverage_days = int(row.stock_quantity / daily_vel)

            # Clasificaciones
            is_dead = days_without_sale > 180
            is_slow = 90 < days_without_sale <= 180
            is_critical = coverage_days < 3 and daily_vel > 0

            mv = MVStockKPIs(
                pharmacy_id=pharmacy_id,
                product_code=row.product_code,
                product_name=row.product_name,
                stock_qty=row.stock_quantity,
                stock_value=row.stock_value or 0,
                daily_velocity=daily_vel,
                coverage_days=min(coverage_days, 999),
                last_sale_date=row.last_sale_date,
                days_without_sale=min(days_without_sale, 999),
                is_dead_stock=is_dead,
                is_slow_mover=is_slow,
                is_critical=is_critical,
                updated_at=now,
            )
            db.add(mv)
            rows_inserted += 1

        db.commit()

        return {"rows": rows_inserted, "status": "success"}

    async def _refresh_weekly_kpis(
        self, db: Session, pharmacy_id: UUID
    ) -> dict:
        """Actualizar mv_weekly_kpis."""
        logger.debug("etl_service.refresh_weekly_kpis", pharmacy_id=str(pharmacy_id))

        db.query(MVWeeklyKPIs).filter(
            MVWeeklyKPIs.pharmacy_id == pharmacy_id
        ).delete()

        # Calcular KPIs por semana (últimas 52 semanas)
        today = date.today()
        weeks_ago_52 = today - timedelta(weeks=52)

        query = (
            db.query(
                func.date_trunc('week', SalesData.sale_date).label('week_start'),
                func.sum(SalesData.total_amount).label('total_sales'),
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'venta_libre', SalesData.total_amount),
                        else_=0
                    )
                ).label('vl_sales'),
                func.sum(
                    case(
                        (SalesEnrichment.product_type == 'prescripcion', SalesData.total_amount),
                        else_=0
                    )
                ).label('rx_sales'),
                func.sum(SalesData.margin_amount).label('total_margin'),
                func.count(SalesData.id).label('transaction_count'),
                func.count(func.distinct(SalesData.codigo_nacional)).label('unique_products'),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= weeks_ago_52,
            )
            .group_by(func.date_trunc('week', SalesData.sale_date))
        )

        rows_inserted = 0
        now = datetime.now(timezone.utc)

        for row in query.all():
            if row.week_start is None:
                continue

            margin_pct = 0
            if row.total_sales and row.total_sales > 0:
                margin_pct = (float(row.total_margin or 0) / float(row.total_sales)) * 100

            mv = MVWeeklyKPIs(
                pharmacy_id=pharmacy_id,
                week_start=row.week_start.date() if hasattr(row.week_start, 'date') else row.week_start,
                total_sales=row.total_sales or 0,
                vl_sales=row.vl_sales or 0,
                rx_sales=row.rx_sales or 0,
                total_margin=row.total_margin or 0,
                margin_percent=margin_pct,
                transaction_count=row.transaction_count or 0,
                unique_products=row.unique_products or 0,
                updated_at=now,
            )
            db.add(mv)
            rows_inserted += 1

        db.commit()

        return {"rows": rows_inserted, "status": "success"}


# Función helper para el scheduler
def refresh_all_views():
    """
    Función llamada por el ETL scheduler en docker-compose.local.yml.

    Ejecuta el ETL para todas las farmacias.
    """
    import asyncio
    from app.core.database import SessionLocal

    logger.info("etl_service.refresh_all_views.start")

    db = SessionLocal()
    etl = ETLService()

    try:
        result = asyncio.run(etl.run_all_pharmacies(db))
        logger.info(
            "etl_service.refresh_all_views.complete",
            **result,
        )
        return result
    finally:
        db.close()


# Instancia singleton
etl_service = ETLService()
