# backend/app/services/erp_sync_service.py
"""
ERP Sync Service - Direct Database Access to Pharmacy ERPs.

Pivot 2026: Local single-tenant architecture.

Configuration via environment variables:
    ERP_TYPE: farmanager | farmatic | nixfarma | unycop
    ERP_HOST: Database host (default: localhost)
    ERP_PORT: Database port (default: 3306)
    ERP_DATABASE: Database name (default: gesql)
    ERP_USER: Database user
    ERP_PASSWORD: Database password
    ERP_SYNC_INTERVAL: Minutes between syncs (default: 15)

Key features:
- Direct DB connection to pharmacy ERP
- Incremental sync via timestamp (not temporal window)
- Uses erp_id (operation_line) for duplicate detection
- Reuses enrichment pipeline via synthetic FileUpload
"""
import os
import threading
import time
import uuid
from datetime import date, datetime, timedelta
from typing import Optional

import structlog
from sqlalchemy import create_engine, text
from sqlalchemy.orm import Session, sessionmaker
from sqlalchemy.pool import NullPool

from app.models import FileUpload, Pharmacy, SalesData, UploadStatus
from app.models.erp_sync_state import ERPSyncState, SyncStatusEnum
from app.models.file_upload import FileType
from app.services.enrichment_service import EnrichmentService
from app.utils.datetime_utils import utc_now

# Import ERP adapters
from app.erp_adapters.farmanager.adapter import FarmanagerAdapter
from app.erp_adapters.models import PrescriptionType, UnifiedSale

logger = structlog.get_logger(__name__)

# Configuration from environment
ERP_TYPE = os.getenv("ERP_TYPE", "farmanager")
ERP_HOST = os.getenv("ERP_HOST", "localhost")
ERP_PORT = int(os.getenv("ERP_PORT", "3306"))
ERP_DATABASE = os.getenv("ERP_DATABASE", "gesql")
ERP_USER = os.getenv("ERP_USER", "root")
ERP_PASSWORD = os.getenv("ERP_PASSWORD", "")
ERP_CHARSET = os.getenv("ERP_CHARSET", "utf8")
ERP_SYNC_INTERVAL = int(os.getenv("ERP_SYNC_INTERVAL", "15"))  # minutes
ERP_SYNC_BATCH_SIZE = int(os.getenv("ERP_SYNC_BATCH_SIZE", "500"))
ERP_INITIAL_SYNC_DAYS = int(os.getenv("ERP_INITIAL_SYNC_DAYS", "90"))

# File type mapping
FILE_TYPE_MAP = {
    "farmanager": FileType.FARMANAGER,
    "farmatic": FileType.FARMATIC,
    "nixfarma": FileType.NIXFARMA,
    "unycop": FileType.UNYCOP,
}


class ERPSyncService:
    """
    Service for syncing data from pharmacy ERP via direct database access.

    Local single-tenant: Always syncs the ONE pharmacy in the installation.

    Usage:
        service = ERPSyncService()

        # Run sync (checks interval automatically)
        result = service.sync()

        # Force full sync (ignore last_sale_timestamp)
        result = service.sync(force_full=True)

        # Get current state
        state = service.get_state()
    """

    def __init__(self, database_url: Optional[str] = None):
        """
        Initialize sync service.

        Args:
            database_url: Override database URL (default: from DATABASE_URL env)
        """
        db_url = database_url or os.getenv(
            "DATABASE_URL",
            "sqlite:///data/kaifarma.db"
        )

        # SQLite needs check_same_thread=False
        connect_args = {}
        if db_url.startswith("sqlite"):
            connect_args["check_same_thread"] = False

        self.engine = create_engine(
            db_url,
            poolclass=NullPool,  # Thread safety
            connect_args=connect_args,
            echo=False,
        )
        self.SessionLocal = sessionmaker(
            autocommit=False,
            autoflush=False,
            bind=self.engine,
        )

        # Sync lock (prevent concurrent syncs)
        self._sync_lock = threading.Lock()
        self._is_syncing = False

        # Track enrichment threads (prevent resource exhaustion)
        self._enrichment_threads: list = []
        self._enrichment_lock = threading.Lock()
        self._max_concurrent_enrichments = 2

    def get_state(self) -> dict:
        """
        Get current sync state.

        Returns:
            Dict with current state or None if not initialized
        """
        db = self.SessionLocal()
        try:
            state = db.query(ERPSyncState).first()
            if state:
                return state.to_dict()
            return {"status": "not_initialized", "message": "No sync has been run yet"}
        finally:
            db.close()

    def sync(self, force_full: bool = False) -> dict:
        """
        Run ERP sync operation.

        Args:
            force_full: If True, ignore last_sale_timestamp and do full sync

        Returns:
            Dict with sync results
        """
        # Prevent concurrent syncs
        if not self._sync_lock.acquire(blocking=False):
            return {
                "success": False,
                "error": "Sync already in progress",
                "records_synced": 0,
            }

        start_time = time.time()
        db = self.SessionLocal()

        try:
            self._is_syncing = True

            # Get or create sync state
            state = db.query(ERPSyncState).first()
            if not state:
                state = ERPSyncState(id=1)
                db.add(state)
                db.commit()

            # Check if should sync (unless forced)
            if not force_full and not state.should_sync(ERP_SYNC_INTERVAL):
                return {
                    "success": True,
                    "skipped": True,
                    "reason": "Not enough time since last sync",
                    "records_synced": 0,
                }

            # Get the pharmacy (single-tenant: should be exactly one)
            pharmacy = db.query(Pharmacy).filter(Pharmacy.is_active == True).first()
            if not pharmacy:
                return {
                    "success": False,
                    "error": "No active pharmacy found",
                    "records_synced": 0,
                }

            logger.info(
                "erp_sync.starting",
                pharmacy_id=str(pharmacy.id),
                erp_type=ERP_TYPE,
                force_full=force_full,
                last_sale_timestamp=state.last_sale_timestamp,
            )

            # Mark sync started
            state.mark_sync_started()
            db.commit()

            # Create ERP adapter
            adapter = self._create_adapter()

            # Determine sync window
            if force_full or not state.last_sale_timestamp:
                # First sync or forced: fetch last N days
                from_date = date.today() - timedelta(days=ERP_INITIAL_SYNC_DAYS)
                to_date = date.today()
                logger.info(
                    "erp_sync.full_sync",
                    from_date=from_date.isoformat(),
                    to_date=to_date.isoformat(),
                )
            else:
                # Incremental: fetch from last sync timestamp
                from_date = state.last_sale_timestamp.date()
                to_date = date.today()
                logger.info(
                    "erp_sync.incremental_sync",
                    from_date=from_date.isoformat(),
                    to_date=to_date.isoformat(),
                )

            # Create synthetic FileUpload
            file_upload = self._create_synthetic_upload(
                db, pharmacy.id, from_date, to_date
            )
            db.commit()

            # Connect and sync
            with adapter:
                result = self._sync_sales(
                    db=db,
                    adapter=adapter,
                    state=state,
                    pharmacy_id=pharmacy.id,
                    file_upload=file_upload,
                    from_date=from_date,
                    to_date=to_date,
                )

            duration = int(time.time() - start_time)

            # Update state
            if result["success"]:
                state.mark_sync_completed(
                    records_synced=result["records_synced"],
                    duration_seconds=duration,
                    last_sale_timestamp=result.get("max_timestamp"),
                )
                file_upload.status = UploadStatus.COMPLETED
                file_upload.processing_completed_at = utc_now()
            else:
                state.mark_sync_error(result.get("error", "Unknown error"))
                file_upload.status = UploadStatus.ERROR
                file_upload.error_message = result.get("error")

            db.commit()

            # Trigger enrichment (async, non-blocking)
            if result["success"] and result["records_synced"] > 0:
                self._trigger_enrichment_async(
                    pharmacy_id=str(pharmacy.id),
                    upload_id=str(file_upload.id),
                )

            result["duration_seconds"] = duration
            logger.info("erp_sync.completed", **result)
            return result

        except Exception as e:
            logger.exception("erp_sync.error", error=str(e))
            return {
                "success": False,
                "error": str(e),
                "records_synced": 0,
                "duration_seconds": int(time.time() - start_time),
            }
        finally:
            self._is_syncing = False
            self._sync_lock.release()
            db.close()

    def _create_adapter(self):
        """Create ERP adapter based on environment config."""
        if ERP_TYPE == "farmanager":
            return FarmanagerAdapter(
                host=ERP_HOST,
                port=ERP_PORT,
                user=ERP_USER,
                password=ERP_PASSWORD,
                database=ERP_DATABASE,
                charset=ERP_CHARSET,
                read_only=True,  # SAFETY: Always read-only
            )
        else:
            raise NotImplementedError(f"ERP type '{ERP_TYPE}' not yet implemented")

    def _create_synthetic_upload(
        self,
        db: Session,
        pharmacy_id,
        from_date: date,
        to_date: date,
    ) -> FileUpload:
        """
        Create synthetic FileUpload record for ERP sync.

        Allows reusing enrichment pipeline without code changes.
        """
        file_upload = FileUpload(
            id=uuid.uuid4(),
            pharmacy_id=pharmacy_id,
            filename=f"erp_sync_{ERP_TYPE}_{from_date}_{to_date}.synthetic",
            original_filename=f"ERP Direct Sync ({ERP_TYPE})",
            file_type=FILE_TYPE_MAP.get(ERP_TYPE, FileType.UNKNOWN),
            file_size=0,  # No file
            status=UploadStatus.PROCESSING,
            processing_started_at=utc_now(),
            processing_notes=f"ERP Direct Access sync from {from_date} to {to_date}",
        )
        db.add(file_upload)
        return file_upload

    def _sync_sales(
        self,
        db: Session,
        adapter,
        state: ERPSyncState,
        pharmacy_id,
        file_upload: FileUpload,
        from_date: date,
        to_date: date,
    ) -> dict:
        """
        Sync sales data from ERP to SalesData table.

        Duplicate detection via erp_id (operation_line composite).
        """
        records_synced = 0
        records_skipped = 0
        max_timestamp = None
        batch = []

        # Get existing operation IDs for dedup
        existing_ids = self._get_existing_operation_ids(
            db, pharmacy_id, from_date, to_date
        )
        logger.info(
            "erp_sync.existing_records",
            count=len(existing_ids),
        )

        try:
            for sale in adapter.get_sales(from_date, to_date):
                # Duplicate check
                if sale.erp_id in existing_ids:
                    records_skipped += 1
                    continue

                # Transform UnifiedSale → SalesData
                sales_record = self._transform_sale(
                    sale=sale,
                    pharmacy_id=pharmacy_id,
                    upload_id=file_upload.id,
                )
                batch.append(sales_record)

                # Track max timestamp for incremental sync
                if sale.timestamp:
                    if not max_timestamp or sale.timestamp > max_timestamp:
                        max_timestamp = sale.timestamp

                # Batch insert
                if len(batch) >= ERP_SYNC_BATCH_SIZE:
                    db.bulk_save_objects(batch)
                    db.commit()
                    records_synced += len(batch)

                    # Update progress
                    file_upload.rows_processed = records_synced
                    file_upload.processing_notes = (
                        f"Synced {records_synced:,} records, "
                        f"skipped {records_skipped:,} duplicates"
                    )
                    db.commit()

                    logger.info(
                        "erp_sync.batch_committed",
                        records_synced=records_synced,
                        records_skipped=records_skipped,
                    )
                    batch = []

            # Final batch
            if batch:
                db.bulk_save_objects(batch)
                db.commit()
                records_synced += len(batch)

            # Update final counts
            file_upload.rows_total = records_synced + records_skipped
            file_upload.rows_processed = records_synced
            file_upload.rows_duplicates = records_skipped
            db.commit()

            return {
                "success": True,
                "records_synced": records_synced,
                "records_skipped": records_skipped,
                "max_timestamp": max_timestamp,
            }

        except Exception as e:
            db.rollback()
            logger.exception("erp_sync.sales_error", error=str(e))
            return {
                "success": False,
                "error": str(e),
                "records_synced": records_synced,
                "records_skipped": records_skipped,
            }

    def _get_existing_operation_ids(
        self,
        db: Session,
        pharmacy_id,
        from_date: date,
        to_date: date,
    ) -> set:
        """
        Get existing ERP operation IDs for duplicate detection.

        NOTE: operation_id contains composite key "{operation}_{line}"
        from the ERP adapter's erp_id field, ensuring uniqueness per sale line.
        """
        result = db.execute(
            text("""
                SELECT DISTINCT operation_id
                FROM sales_data
                WHERE pharmacy_id = :pharmacy_id
                  AND operation_id IS NOT NULL
                  AND sale_date >= :from_date
                  AND sale_date <= :to_date
            """),
            {
                "pharmacy_id": str(pharmacy_id),
                "from_date": from_date,
                "to_date": to_date,
            },
        )
        return {row[0] for row in result.fetchall()}

    def _transform_sale(
        self,
        sale: UnifiedSale,
        pharmacy_id,
        upload_id,
    ) -> SalesData:
        """
        Transform UnifiedSale (from ERP adapter) to SalesData model.

        Philosophy: Extract minimum from ERP, enrich everything later.
        """
        sale_datetime = sale.timestamp
        sale_date = sale_datetime.date() if sale_datetime else None
        sale_time = sale_datetime.time() if sale_datetime else None

        return SalesData(
            id=uuid.uuid4(),
            pharmacy_id=pharmacy_id,
            upload_id=upload_id,
            # Temporal
            sale_date=sale_datetime,
            sale_time=sale_time,
            year=sale_date.year if sale_date else None,
            month=sale_date.month if sale_date else None,
            day=sale_date.day if sale_date else None,
            week=sale_date.isocalendar()[1] if sale_date else None,
            weekday=sale_date.isoweekday() if sale_date else None,
            # Product (CRITICAL: Always string)
            codigo_nacional=sale.product_code or None,
            ean13=sale.product_code if len(sale.product_code or "") == 13 else None,
            product_name=sale.product_name,
            # Quantities and prices
            quantity=sale.quantity,
            unit_price=sale.pvp,
            sale_price=sale.pvp,
            purchase_price=sale.puc,  # PMC from ERP
            total_amount=sale.pvp * sale.quantity if sale.pvp else None,
            discount_amount=sale.discount,
            # Margin
            margin_amount=(
                (sale.pvp - sale.puc) * sale.quantity
                if sale.pvp and sale.puc
                else None
            ),
            margin_percentage=(
                ((sale.pvp - sale.puc) / sale.pvp * 100)
                if sale.pvp and sale.puc and sale.pvp > 0
                else None
            ),
            # Prescription
            requires_prescription=(sale.prescription_type == PrescriptionType.RECETA),
            is_financed=(sale.prescription_type == PrescriptionType.RECETA),
            # ERP reference (for duplicate detection)
            # NOTE: operation_id stores composite key "{operation}_{line}" from erp_id
            # This ensures uniqueness per sale line, not just per operation
            operation_id=sale.erp_id,
            # Employee/customer
            employee_code=sale.employee_id,
            client_code=sale.customer_id,
            # Metadata
            created_at=utc_now(),
        )

    def _trigger_enrichment_async(
        self,
        pharmacy_id: str,
        upload_id: str,
    ) -> None:
        """Trigger enrichment pipeline in background thread."""
        with self._enrichment_lock:
            # Clean up finished threads
            self._enrichment_threads = [
                t for t in self._enrichment_threads if t.is_alive()
            ]

            # Check concurrent limit
            if len(self._enrichment_threads) >= self._max_concurrent_enrichments:
                logger.warning(
                    "erp_sync.enrichment_throttled",
                    active_threads=len(self._enrichment_threads),
                    max_allowed=self._max_concurrent_enrichments,
                )
                return

        def _run_enrichment():
            db = None
            try:
                db = self.SessionLocal()
                enrichment_service = EnrichmentService(db)

                logger.info(
                    "erp_sync.enrichment_starting",
                    pharmacy_id=pharmacy_id,
                    upload_id=upload_id,
                )

                result = enrichment_service.enrich_sales_batch(
                    pharmacy_id=pharmacy_id,
                    upload_id=upload_id,
                )

                logger.info(
                    "erp_sync.enrichment_completed",
                    pharmacy_id=pharmacy_id,
                    upload_id=upload_id,
                    result=result,
                )
            except Exception as e:
                logger.exception(
                    "erp_sync.enrichment_error",
                    pharmacy_id=pharmacy_id,
                    upload_id=upload_id,
                    error=str(e),
                )
            finally:
                if db:
                    db.close()

        thread = threading.Thread(target=_run_enrichment, daemon=True)

        with self._enrichment_lock:
            self._enrichment_threads.append(thread)

        thread.start()

    def reset_errors(self) -> dict:
        """Reset error state for manual intervention."""
        db = self.SessionLocal()
        try:
            state = db.query(ERPSyncState).first()
            if state:
                state.reset_errors()
                db.commit()
                return {"success": True, "message": "Errors reset"}
            return {"success": False, "message": "No sync state found"}
        finally:
            db.close()


# Singleton instance
_erp_sync_service: Optional[ERPSyncService] = None


def get_erp_sync_service() -> ERPSyncService:
    """Get or create singleton ERPSyncService instance."""
    global _erp_sync_service
    if _erp_sync_service is None:
        _erp_sync_service = ERPSyncService()
    return _erp_sync_service
