﻿# backend/app/services/enrichment_service.py
"""
Servicio de enriquecimiento automático de datos de ventas.
Integra con CIMA, Nomenclator y servicios ML para enriquecer productos.
"""

from typing import Callable, Dict, Optional, Tuple

import structlog
from sqlalchemy import and_, func
from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import Session

from app.exceptions import EnrichmentError, ProductNotFoundError
from app.external_data.cima_integration import CIMAIntegrationService
from app.external_data.nomenclator_integration import nomenclator_integration
from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.services.brand_detection_service import brand_detection_service
from app.services.enrichment_cache import enrichment_cache
from app.services.venta_libre_catalog_service import VentaLibreCatalogService
from app.utils.datetime_utils import utc_now

# Configurar structured logging
logger = structlog.get_logger(__name__)


class EnrichmentService:
    """
    Servicio principal de enriquecimiento de datos de ventas.

    Funcionalidades:
    - Matching automático con catálogos CIMA/Nomenclator
    - Clasificación ML para productos OTC
    - Análisis de oportunidades de genéricos
    - Cálculos farmacéuticos automáticos
    """

    def __init__(self):
        self.cima = CIMAIntegrationService()
        self.nomenclator = nomenclator_integration
        self.cache = enrichment_cache

    def enrich_sales_batch_optimized(
        self,
        db: Session,
        pharmacy_id: str,
        upload_id: Optional[str] = None,
        batch_size: int = 500,
    ) -> Dict:
        """
        Versión optimizada del enriquecimiento en lotes para mejor performance.

        Optimizaciones:
        - Procesamiento en chunks para evitar cargar todo en memoria
        - Bulk operations para reducir queries
        - Batch commits para mejor transaccionalidad
        - Cache de productos para evitar lookups repetidos
        """
        logger.info(
            "enrichment.batch.started",
            pharmacy_id=pharmacy_id,
            upload_id=upload_id,
            batch_size=batch_size,
        )

        start_time = utc_now()

        stats = {
            "processed": 0,
            "enriched": 0,
            "failed": 0,
            "duplicates_skipped": 0,
            "matches_by_code": 0,
            "matches_by_ean": 0,
            "matches_by_name": 0,
            "manual_review": 0,
            "cache_hits": 0,
            "cache_misses": 0,
        }

        # Cache de productos para evitar lookups repetidos
        product_cache = {}

        # Procesamiento en chunks para evitar memory issues
        offset = 0
        while True:
            # 1. Obtener chunk pequeño de datos sin enriquecer
            chunk_query = (
                db.query(SalesData)
                .filter(SalesData.pharmacy_id == pharmacy_id)
                .outerjoin(SalesEnrichment)
                .filter(SalesEnrichment.id.is_(None))
            )

            if upload_id:
                chunk_query = chunk_query.filter(SalesData.upload_id == upload_id)

            sales_chunk = chunk_query.offset(offset).limit(batch_size).all()

            if not sales_chunk:
                break  # No más registros

            logger.info(
                "enrichment.chunk.processing",
                chunk_start=offset,
                chunk_end=offset + len(sales_chunk),
                chunk_size=len(sales_chunk),
            )

            # 2. Preparar códigos únicos para batch lookup
            national_codes = list(
                set(
                    [
                        s.codigo_nacional
                        for s in sales_chunk
                        if s.codigo_nacional and len(str(s.codigo_nacional).strip()) >= 4
                    ]
                )
            )

            # 3. Bulk lookup de productos para este chunk
            if national_codes:
                products = db.query(ProductCatalog).filter(ProductCatalog.national_code.in_(national_codes)).all()

                # Actualizar cache
                for product in products:
                    product_cache[product.national_code] = product

                logger.debug("enrichment.cache.updated", cache_size=len(products))

            # 4. Procesar chunk con cache
            enrichments_to_create = []

            for sales_record in sales_chunk:
                try:
                    enrichment = self._enrich_single_record_cached(db, sales_record, product_cache, stats)

                    if enrichment:
                        enrichments_to_create.append(enrichment)
                        stats["enriched"] += 1

                        if enrichment.match_method == "codigo_nacional":
                            stats["matches_by_code"] += 1
                        elif enrichment.match_method == "name_fuzzy":
                            stats["matches_by_name"] += 1
                    else:
                        stats["failed"] += 1

                    stats["processed"] += 1

                    # FIX Issue #253: Checkpoints de progreso cada 1000 registros
                    if stats["processed"] % 1000 == 0:
                        logger.info(
                            f"🔄 [CHECKPOINT] Progreso enriquecimiento: {stats['processed']} registros procesados - "
                            f"Enriquecidos: {stats['enriched']} ({stats['enriched']/stats['processed']*100:.1f}%) - "
                            f"Fallidos: {stats['failed']} - Cache hits: {stats['cache_hits']}/{stats['cache_hits'] + stats['cache_misses']}"
                        )

                except ProductNotFoundError as e:
                    # Producto no encontrado en catálogo es esperado a veces
                    logger.warning(
                        "enrichment.product.not_found",
                        sales_record_id=str(sales_record.id),
                        product_code=e.details.get("product_code"),
                    )
                    stats["failed"] += 1
                    continue
                except Exception as e:
                    logger.error(
                        "enrichment.record.error",
                        sales_record_id=str(sales_record.id),
                        error=str(e),
                    )
                    stats["failed"] += 1

                    # Si hay muchos errores, lanzar excepción
                    if stats["failed"] > len(sales_chunk) * 0.5:  # Más del 50% fallos
                        raise EnrichmentError(
                            stage="batch_processing",
                            product_count=stats["processed"],
                            reason=f"Demasiados fallos en enriquecimiento: {stats['failed']} de {stats['processed']}",
                        )
                    continue

            # 5. Bulk insert de enriquecimientos
            if enrichments_to_create:
                db.bulk_save_objects(enrichments_to_create)

            # 6. Commit del chunk completo
            db.commit()

            logger.info(
                f"[ENRICHMENT_OPT] Chunk completado - Procesados: {stats['processed']}, "
                f"Enriquecidos: {stats['enriched']}, Cache size: {len(product_cache)}"
            )

            offset += batch_size

            # Limitar cache para evitar memory issues
            if len(product_cache) > 5000:
                product_cache.clear()
                logger.debug("[ENRICHMENT_OPT] Cache limpiado por tamaño")

        logger.info(
            "enrichment.batch.completed",
            pharmacy_id=pharmacy_id,
            processed=stats["processed"],
            enriched=stats["enriched"],
            cache_hits=stats["cache_hits"],
            cache_misses=stats["cache_misses"],
            execution_time_seconds=(utc_now() - start_time).total_seconds(),
        )
        return stats

    # REMOVED: Primera definición de enrich_sales_batch eliminada (duplicada).
    # La versión optimizada está definida más abajo (línea ~1267) con parámetros opcionales adicionales.
    # Esto resuelve el error en Render donde el módulo no se carga correctamente debido a definición duplicada.

    def reenrich_failed_records(self, db: Session, pharmacy_id: str, limit: int = 100) -> Dict:
        """
        Reenriquece registros que fallaron previamente o están en revisión manual.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            limit: Número máximo de registros a procesar

        Returns:
            Diccionario con estadísticas del proceso
        """
        logger.info(f"[REENRICHMENT] Iniciando reenriquecimiento para farmacia {pharmacy_id}")

        # Obtener registros que pueden beneficiarse del reenriquecimiento
        failed_enrichments = (
            db.query(SalesEnrichment)
            .join(SalesData)
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesEnrichment.enrichment_status.in_(["manual_review", "failed"]),
                    SalesData.codigo_nacional.isnot(None),  # Solo los que tienen código nacional
                )
            )
            .limit(limit)
            .all()
        )

        if not failed_enrichments:
            logger.info(f"[REENRICHMENT] No hay registros para reenriquecer en farmacia {pharmacy_id}")
            return {"processed": 0, "reenriched": 0, "still_failed": 0, "errors": 0}

        logger.info(f"[REENRICHMENT] Procesando {len(failed_enrichments)} registros fallidos")

        stats = {
            "processed": 0,
            "reenriched": 0,
            "still_failed": 0,
            "errors": 0,
            "cima_matches": 0,
            "nomenclator_matches": 0,
        }

        for enrichment in failed_enrichments:
            try:
                sales_record = enrichment.sales_data

                # Reenriquecer usando el método de actualización
                result = self._update_existing_enrichment(db, sales_record, enrichment)

                if result and result.enrichment_status == "enriched":
                    stats["reenriched"] += 1

                    # Contar por fuente
                    if "cima" in (result.enrichment_source or "").lower():
                        stats["cima_matches"] += 1
                    if "nomenclator" in (result.enrichment_source or "").lower():
                        stats["nomenclator_matches"] += 1

                    logger.debug(f"[REENRICHED] {sales_record.codigo_nacional}: {result.match_method}")
                else:
                    stats["still_failed"] += 1

                stats["processed"] += 1

                # Commit cada 50 registros
                if stats["processed"] % 50 == 0:
                    db.commit()
                    logger.info(
                        f"[REENRICH PROGRESS] {stats['processed']}/{len(failed_enrichments)} - "
                        f"Exitosos: {stats['reenriched']} - CIMA: {stats['cima_matches']} - "
                        f"Nomenclátor: {stats['nomenclator_matches']}"
                    )

            except Exception as e:
                logger.error("reenrichment.record.error", error=str(e))
                stats["errors"] += 1
                continue

        # Commit final
        db.commit()

        logger.info(f"[REENRICHMENT] Completado - {stats}")
        return stats

    def _enrich_single_record(self, db: Session, sales_record: SalesData) -> Optional[SalesEnrichment]:
        """
        Enriquece un registro individual de ventas.

        Args:
            db: Sesión de base de datos
            sales_record: Registro de venta a enriquecer

        Returns:
            SalesEnrichment creado o None si falló
        """
        start_time = utc_now()

        # Crear registro base de enriquecimiento
        enrichment = SalesEnrichment(sales_data_id=sales_record.id, enrichment_status="pending")

        try:
            # 1. Intentar matching por código nacional (prioridad alta)
            product_catalog = None
            match_method = None
            match_confidence = 0

            if sales_record.codigo_nacional:
                # Ejecutar async en contexto sync
                import asyncio

                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                try:
                    product_catalog = loop.run_until_complete(
                        self._find_by_national_code(db, sales_record.codigo_nacional)
                    )
                finally:
                    loop.close()

                if product_catalog:
                    match_method = "codigo_nacional"
                    match_confidence = 95
                    logger.debug(f"Match por CN: {sales_record.codigo_nacional}")

            # 2. Si no encontró por CN, intentar por EAN13 completo
            if not product_catalog and sales_record.ean13:
                product_catalog = self._find_by_ean(db, sales_record.ean13)
                if product_catalog:
                    match_method = "ean13"
                    match_confidence = 90
                    logger.debug(f"Match por EAN13: {sales_record.ean13}")

            # 3. Si no encontró, intentar matching fuzzy por nombre
            if not product_catalog and sales_record.product_name:
                product_catalog, fuzzy_score = self._find_by_name_fuzzy(db, sales_record.product_name)
                if product_catalog and fuzzy_score > 0.8:
                    match_method = "name_fuzzy"
                    match_confidence = int(fuzzy_score * 100)

            # 4. Llenar datos del enriquecimiento
            enrichment.match_method = match_method
            enrichment.match_confidence = match_confidence

            if product_catalog:
                enrichment.product_catalog_id = product_catalog.id
                # Usar las mismas fuentes que el catálogo
                enrichment.enrichment_source = product_catalog.data_sources or "nomenclator"
                enrichment.enrichment_status = "enriched"

                # Derivar categorías farmacéuticas
                enrichment.therapeutic_category = self._derive_therapeutic_category(product_catalog)
                enrichment.prescription_category = self._derive_prescription_category(product_catalog)
                enrichment.product_type = self._derive_product_type(product_catalog)

                # Verificar si tiene conjunto homogéneo (indica alternativas disponibles)
                if product_catalog.nomen_codigo_homogeneo:
                    enrichment.has_generic_alternative = True
                else:
                    enrichment.has_generic_alternative = False

                enrichment.enriched_at = utc_now()
            else:
                # No se encontró en catálogos, intentar clasificar por prescription_reference_list
                # Issue #444: Productos como tiras reactivas no están en CIMA pero sí en listados oficiales
                from app.models.prescription_reference_list import PrescriptionReferenceList

                reference = None
                if sales_record.codigo_nacional:
                    reference = (
                        db.query(PrescriptionReferenceList)
                        .filter(PrescriptionReferenceList.national_code == sales_record.codigo_nacional)
                        .first()
                    )

                if reference:
                    # Producto encontrado en listados oficiales del Ministerio
                    # Issue #449: Crear entrada en ProductCatalog para que aparezca en analytics
                    from app.models.enums import PrescriptionCategory as PCEnum

                    # Crear ProductCatalog si no existe (clave: national_code)
                    product_catalog = db.query(ProductCatalog).filter(
                        ProductCatalog.national_code == sales_record.codigo_nacional
                    ).first()

                    if not product_catalog:
                        # Crear nueva entrada en ProductCatalog desde prescription_reference_list
                        category_enum = reference.category if isinstance(reference.category, PCEnum) else PCEnum(reference.category)
                        product_catalog = ProductCatalog(
                            national_code=sales_record.codigo_nacional,
                            cima_nombre_comercial=sales_record.product_name,
                            nomen_nombre=sales_record.product_name,
                            xfarma_prescription_category=category_enum,
                            data_sources=f"prescription_reference_list:{reference.reference_source}",
                        )
                        db.add(product_catalog)
                        db.flush()  # Obtener ID
                        logger.info(
                            f"[ENRICHMENT] Creado ProductCatalog para {sales_record.codigo_nacional} "
                            f"desde reference_list: {category_enum.value}"
                        )

                    # Vincular enrichment con product_catalog
                    enrichment.product_catalog_id = product_catalog.id
                    enrichment.enrichment_status = "enriched"
                    enrichment.enrichment_source = f"prescription_reference_list:{reference.reference_source}"
                    enrichment.match_method = "reference_list"
                    enrichment.match_confidence = 90
                    # Usar el nombre de la categoría del enum (ej: "tiras_reactivas_glucosa")
                    enrichment.prescription_category = reference.category.value if hasattr(reference.category, 'value') else str(reference.category)
                    enrichment.product_type = "prescription"  # Todos los productos de referencia son de prescripción
                    logger.info(
                        f"[ENRICHMENT] Producto {sales_record.codigo_nacional} clasificado por reference_list: "
                        f"{enrichment.prescription_category}, product_catalog_id={product_catalog.id}"
                    )
                else:
                    # No se encontró en catálogos NI en reference_list, marcar para clasificación ML
                    enrichment.enrichment_status = "manual_review"
                    enrichment.manual_review_reason = "No encontrado en catálogos oficiales"

                    # Intentar clasificación ML básica por nombre
                    if sales_record.product_name:
                        ml_result = self._classify_with_ml(sales_record.product_name)
                        if ml_result:
                            enrichment.ml_category = ml_result.get("category")
                            enrichment.ml_confidence = ml_result.get("confidence")
                            enrichment.ml_model_version = ml_result.get("model_version", "v1.0")

                        # Issue #446: Brand detection para taxonomía venta libre
                        self._apply_brand_detection(enrichment, sales_record.product_name)

                    # CRÍTICO: Productos sin catálogo con clasificación ML son venta_libre
                    # (Si fueran de prescripción estarían en CIMA/nomenclator)
                    if enrichment.ml_category:
                        enrichment.product_type = "venta_libre"

                        # Issue #474: Registrar en ProductCatalogVentaLibre con CN/EAN
                        try:
                            vl_catalog = VentaLibreCatalogService(db)
                            vl_product, was_created = vl_catalog.find_or_create(
                                product_name=sales_record.product_name,
                                pharmacy_id=sales_record.pharmacy_id,
                                ean13=sales_record.ean13,
                                cn=sales_record.codigo_nacional,
                                ml_category=enrichment.ml_category,
                                ml_confidence=enrichment.ml_confidence,
                                prediction_source=enrichment.ml_model_version,
                                detected_brand=enrichment.detected_brand,
                            )
                            enrichment.venta_libre_product_id = vl_product.id
                            if was_created:
                                logger.debug(
                                    f"[VENTA_LIBRE] Created new catalog entry: "
                                    f"{sales_record.product_name[:40]} (CN: {sales_record.codigo_nacional})"
                                )
                        except Exception as vl_error:
                            # No bloquear enrichment si falla el catálogo
                            logger.warning(
                                f"[VENTA_LIBRE] Error registering catalog entry: {vl_error}"
                            )

            # Calcular tiempo de procesamiento
            processing_time = (utc_now() - start_time).total_seconds() * 1000
            enrichment.processing_time_ms = int(processing_time)

            # Guardar en BD con protección contra constraint violation
            try:
                db.add(enrichment)
                db.flush()  # Para obtener el ID sin hacer commit
                return enrichment
            except IntegrityError as db_error:
                # Error de constraint único, solo hacer log y continuar
                db.rollback()
                logger.debug(f"Enrichment ya existe para {sales_record.id}, omitiendo")
                return None

        except Exception as e:
            logger.error(f"[ERROR] Error enriqueciendo registro {sales_record.id}: {str(e)}")
            enrichment.enrichment_status = "failed"
            enrichment.error_messages = str(e)
            db.add(enrichment)
            db.flush()
            return None

    async def _find_by_national_code(self, db: Session, product_code: str) -> Optional[ProductCatalog]:
        """Buscar producto por código nacional usando cache primero, luego BD"""
        if not product_code or len(str(product_code).strip()) < 4:
            return None

        code = str(product_code).strip()

        # 1. Primero buscar en cache
        if self.cache.enabled:
            cached_data = await self.cache.get_enriched_product(code)
            if cached_data:
                # Reconstruir objeto ProductCatalog desde cache
                # (Por simplicidad, retornamos los datos cacheados como dict)
                logger.debug(f"[CACHE HIT] Producto {code} obtenido desde cache")
                # Para mantener compatibilidad, creamos un objeto temporal
                product = ProductCatalog()
                for key, value in cached_data.items():
                    if hasattr(product, key) and key != "cached_at":
                        setattr(product, key, value)
                return product

        # 2. Si no está en cache, buscar en product_catalog
        product = db.query(ProductCatalog).filter(ProductCatalog.national_code == code).first()

        # 3. Si encontramos el producto, guardarlo en cache
        if product and self.cache.enabled:
            await self.cache.set_enriched_product(code, product.to_dict())
            logger.debug(f"[CACHE SET] Producto {code} guardado en cache")

        # 2. Si no existe, buscar en nomenclator_local y crear en product_catalog
        if not product:
            try:
                from app.models.nomenclator_local import NomenclatorLocal

                nomen_info = db.query(NomenclatorLocal).filter(NomenclatorLocal.national_code == code).first()

                if nomen_info:
                    # Crear producto en product_catalog basado en nomenclator_local (datos oficiales)
                    product = ProductCatalog(
                        national_code=code,
                        cima_nombre_comercial=(
                            nomen_info.product_name[:500] if nomen_info.product_name else "Producto oficial"
                        ),
                        nomen_nombre=(nomen_info.product_name[:500] if nomen_info.product_name else ""),
                        nomen_principio_activo=(
                            nomen_info.active_ingredient[:500] if nomen_info.active_ingredient else None
                        ),
                        nomen_laboratorio=(nomen_info.laboratory[:200] if nomen_info.laboratory else None),
                        nomen_tipo_farmaco=(
                            nomen_info.therapeutic_group[:50] if nomen_info.therapeutic_group else None
                        ),
                        nomen_precio_referencia=nomen_info.reference_price,
                        # ✅ CRÍTICO: Guardar PVL calculado desde nomenclátor (optimización queries)
                        nomen_pvl_calculado=(
                            nomen_info.pvl_calculado
                            if hasattr(nomen_info, "pvl_calculado") and nomen_info.pvl_calculado
                            else None
                        ),
                        cima_es_generico=nomen_info.is_generic or False,
                        cima_requiere_receta=(
                            nomen_info.requires_prescription if nomen_info.requires_prescription is not None else True
                        ),
                        data_sources="nomenclator_oficial",
                        enrichment_confidence=95,  # Muy alta confianza - datos oficiales del Ministerio
                        created_at=utc_now(),
                        updated_at=utc_now(),
                    )

                    db.add(product)
                    db.commit()
                    db.refresh(product)

                    # ✨ NUEVO: Calcular PVL automáticamente si cumple condiciones
                    if product.should_calculate_pvl():
                        if product.calculate_and_set_pvl():
                            db.commit()
                            logger.info(
                                f"✨ Auto-calculated PVL for {code}: PVP {product.nomen_pvp}€ → PVL {product.nomen_pvl_calculado}€"
                            )

                    logger.info(f"✅ Created ProductCatalog from NomenclatorLocal for {code}: {nomen_info.laboratory}")

            except Exception as e:
                logger.warning(f"Error creating product from nomenclator_local for {code}: {str(e)}")

        # 3. Si ya existe product pero faltan datos, enriquecer desde nomenclator_local
        elif product and not product.nomen_laboratorio:
            try:
                from app.models.nomenclator_local import NomenclatorLocal

                nomen_info = db.query(NomenclatorLocal).filter(NomenclatorLocal.national_code == code).first()

                if nomen_info and nomen_info.laboratory:
                    # Completar con datos oficiales del nomenclator
                    product.nomen_laboratorio = nomen_info.laboratory[:200]
                    product.nomen_nombre = (
                        nomen_info.product_name[:500] if nomen_info.product_name else product.nomen_nombre
                    )
                    product.nomen_principio_activo = (
                        nomen_info.active_ingredient[:500]
                        if nomen_info.active_ingredient
                        else product.nomen_principio_activo
                    )
                    product.nomen_precio_referencia = (
                        nomen_info.reference_price if nomen_info.reference_price else product.nomen_precio_referencia
                    )

                    # ✅ CRÍTICO: Guardar PVL calculado desde nomenclátor si está disponible
                    # Esto optimiza queries de análisis de partners (evita calcular PVL en cada query)
                    if hasattr(nomen_info, "pvl_calculado") and nomen_info.pvl_calculado:
                        product.nomen_pvl_calculado = nomen_info.pvl_calculado
                        logger.debug(f"[ENRICH] PVL guardado para {code}: {nomen_info.pvl_calculado}€")

                    product.cima_es_generico = (
                        nomen_info.is_generic if nomen_info.is_generic is not None else product.cima_es_generico
                    )
                    product.cima_requiere_receta = (
                        nomen_info.requires_prescription
                        if nomen_info.requires_prescription is not None
                        else product.cima_requiere_receta
                    )

                    # Actualizar fuentes de datos
                    if product.data_sources:
                        if "nomenclator_oficial" not in product.data_sources:
                            product.data_sources = product.data_sources + ",nomenclator_oficial"
                    else:
                        product.data_sources = "nomenclator_oficial"

                    product.updated_at = utc_now()
                    db.add(product)
                    db.commit()

                    # ✨ NUEVO: Calcular PVL automáticamente si cumple condiciones tras actualizar
                    if product.should_calculate_pvl():
                        if product.calculate_and_set_pvl():
                            db.commit()
                            logger.info(
                                f"✨ Auto-calculated PVL for {code}: PVP {product.nomen_pvp}€ → PVL {product.nomen_pvl_calculado}€"
                            )

                    logger.info(f"✅ Enriched ProductCatalog from NomenclatorLocal for {code}: {nomen_info.laboratory}")

            except Exception as e:
                logger.warning(f"Error enriching product from nomenclator_local for {code}: {str(e)}")

        # 🆕 4. Si TAMPOCO en nomenclator → Buscar en prescription_reference_list (Issue #452)
        # Productos como dietoterápicos (BLEMIL, NOVALAC, ENSURE) están en listados oficiales
        # pero no en CIMA ni nomenclator local
        if not product:
            try:
                from app.models.prescription_reference_list import PrescriptionReferenceList
                from app.models.enums import PrescriptionCategory as PCEnum

                # Ordenar por status (A > B) y updated_at para selección determinística
                # cuando hay duplicados del mismo national_code en diferentes categorías
                reference = (
                    db.query(PrescriptionReferenceList)
                    .filter(PrescriptionReferenceList.national_code == code)
                    .order_by(
                        PrescriptionReferenceList.status.desc().nullslast(),
                        PrescriptionReferenceList.updated_at.desc().nullslast()
                    )
                    .first()
                )

                if reference:
                    # Crear ProductCatalog desde prescription_reference_list
                    category_enum = reference.category if isinstance(reference.category, PCEnum) else PCEnum(reference.category)
                    product = ProductCatalog(
                        national_code=code,
                        nomen_nombre=reference.product_name[:500] if reference.product_name else "Producto oficial",
                        xfarma_prescription_category=category_enum,
                        data_sources=f"prescription_reference_list:{reference.reference_source}",
                        enrichment_confidence=90,  # Alta confianza - listado oficial Ministerio
                        created_at=utc_now(),
                        updated_at=utc_now(),
                    )
                    db.add(product)
                    db.commit()
                    db.refresh(product)

                    logger.info(
                        f"✅ Created ProductCatalog from prescription_reference_list for {code}: "
                        f"{category_enum.value} - {reference.reference_source}"
                    )

            except Exception as e:
                db.rollback()  # Limpiar estado de transacción
                logger.warning(f"Error creating product from prescription_reference_list for {code}: {str(e)}")

        # 🆕 5. Si TAMPOCO en prescription_reference_list → Buscar en CIMAVet (Issue #354, Issue #446)
        # Cascada actualizada: CIMA → Nomenclator → CIMAVet local → CIMAVet API → manual_review
        # Feature flag: CIMAVET_ENABLED (default: false)
        import os

        cimavet_enabled = os.getenv("CIMAVET_ENABLED", "false").lower() == "true"

        if not product:
            if not cimavet_enabled:
                logger.debug(
                    f"[ENRICHMENT] CIMAVet cascade disabled via feature flag - skipping lookup for {code}",
                    national_code=code,
                    feature_flag="CIMAVET_ENABLED=false",
                )
            else:
                # CIMAVet enabled - try lookup
                try:
                    from app.models.product_catalog_vet import ProductCatalogVet

                    # Buscar en catálogo veterinario LOCAL (con prefijo VET-)
                    # Nota: Los códigos en CIMAVet se almacenan con prefijo para evitar colisión
                    vet_code_with_prefix = f"VET-{code}"
                    vet_product = (
                        db.query(ProductCatalogVet)
                        .filter(ProductCatalogVet.national_code == vet_code_with_prefix)
                        .first()
                    )

                    # 🆕 Issue #446: Si no esta en ProductCatalogVet local, consultar API CIMAVet
                    vet_data = None
                    if not vet_product:
                        logger.debug(f"[ENRICHMENT] CN {code} no en CIMAVet local, consultando API...")
                        try:
                            from app.services.cimavet_sync_service import cimavet_sync_service

                            # enrich_from_cimavet es async, ejecutar en contexto sync
                            import asyncio

                            try:
                                loop = asyncio.get_event_loop()
                                if loop.is_running():
                                    # Si ya hay un loop corriendo, crear una tarea
                                    import concurrent.futures
                                    with concurrent.futures.ThreadPoolExecutor() as executor:
                                        future = executor.submit(
                                            asyncio.run,
                                            cimavet_sync_service.enrich_from_cimavet(code)
                                        )
                                        vet_data = future.result(timeout=35)
                                else:
                                    vet_data = loop.run_until_complete(
                                        cimavet_sync_service.enrich_from_cimavet(code)
                                    )
                            except RuntimeError:
                                # No hay loop, crear uno nuevo
                                loop = asyncio.new_event_loop()
                                asyncio.set_event_loop(loop)
                                try:
                                    vet_data = loop.run_until_complete(
                                        cimavet_sync_service.enrich_from_cimavet(code)
                                    )
                                finally:
                                    loop.close()

                            if vet_data:
                                logger.info(f"[ENRICHMENT] ✅ CN {code} encontrado en API CIMAVet: {vet_data.get('nombre_comercial', 'N/A')[:50]}")
                            else:
                                logger.debug(f"[ENRICHMENT] CN {code} no encontrado en API CIMAVet")

                        except Exception as api_error:
                            logger.warning(f"[ENRICHMENT] Error consultando API CIMAVet para {code}: {str(api_error)}")
                            vet_data = None

                    # Crear ProductCatalog desde vet_product (local) o vet_data (API)
                    if vet_product:
                        # Crear producto en product_catalog MARCADO como CIMAVet (desde BD local)
                        product = ProductCatalog(
                            national_code=code,  # Sin prefijo en product_catalog (código original)
                            cima_nombre_comercial=vet_product.vet_nombre_comercial[:500],
                            cima_laboratorio_titular=(
                                vet_product.vet_laboratorio_titular[:200] if vet_product.vet_laboratorio_titular else None
                            ),
                            cima_forma_farmaceutica=(
                                vet_product.vet_forma_farmaceutica[:200] if vet_product.vet_forma_farmaceutica else None
                            ),
                            cima_uso_veterinario=True,  # ← CRÍTICO: Marcar como veterinario
                            # 🆕 CRÍTICO: Transferir vet_requiere_receta → cima_requiere_receta
                            # Este campo determina prescription vs venta_libre para VETERINARIA
                            cima_requiere_receta=vet_product.vet_requiere_receta,
                            # Clasificación automática como VETERINARIA
                            xfarma_prescription_category="VETERINARIA",
                            data_sources="CIMAVet",  # ← Fuente única
                            enrichment_confidence=90,  # Alta confianza (datos oficiales AEMPS)
                            created_at=utc_now(),
                            updated_at=utc_now(),
                        )
                        db.add(product)
                        db.commit()
                        db.refresh(product)

                        logger.info(
                            f"✅ Created ProductCatalog from CIMAVet LOCAL for {code}: "
                            f"{vet_product.vet_nombre_comercial} - "
                            f"Requiere receta vet: {vet_product.vet_requiere_receta}"
                        )
                    elif vet_data:
                        # 🆕 Issue #446: Crear producto desde respuesta API CIMAVet
                        product = ProductCatalog(
                            national_code=code,
                            cima_nombre_comercial=vet_data.get("nombre_comercial", "")[:500],
                            cima_laboratorio_titular=(
                                vet_data.get("laboratorio_titular", "")[:200] if vet_data.get("laboratorio_titular") else None
                            ),
                            cima_forma_farmaceutica=(
                                vet_data.get("forma_farmaceutica", "")[:200] if vet_data.get("forma_farmaceutica") else None
                            ),
                            cima_uso_veterinario=True,  # ← CRÍTICO: Marcar como veterinario
                            cima_requiere_receta=vet_data.get("requiere_receta"),
                            xfarma_prescription_category="VETERINARIA",
                            data_sources="CIMAVet_API",  # ← Fuente: API directa
                            enrichment_confidence=90,
                            created_at=utc_now(),
                            updated_at=utc_now(),
                        )
                        db.add(product)
                        db.commit()
                        db.refresh(product)

                        logger.info(
                            f"✅ Created ProductCatalog from CIMAVet API for {code}: "
                            f"{vet_data.get('nombre_comercial', 'N/A')[:60]} - "
                            f"Requiere receta vet: {vet_data.get('requiere_receta')}"
                        )

                except Exception as e:
                    logger.warning(f"Error creating product from CIMAVet for {code}: {str(e)}")

        return product

    def _update_existing_enrichment(
        self, db: Session, sales_record: SalesData, existing_enrichment: SalesEnrichment
    ) -> Optional[SalesEnrichment]:
        """
        Actualiza un registro de enriquecimiento existente con nuevos datos.
        Este método es más seguro que eliminar y crear uno nuevo.

        Args:
            db: Sesión de base de datos
            sales_record: Registro de venta a reenriquecer
            existing_enrichment: Registro de enriquecimiento existente a actualizar

        Returns:
            SalesEnrichment actualizado o None si falló
        """
        start_time = utc_now()

        # Guardar estado anterior para audit trail
        previous_status = existing_enrichment.enrichment_status
        previous_method = existing_enrichment.match_method

        try:
            # 1. Intentar matching por código nacional (prioridad alta)
            product_catalog = None
            match_method = None
            match_confidence = 0

            if sales_record.codigo_nacional:
                import asyncio

                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                try:
                    product_catalog = loop.run_until_complete(
                        self._find_by_national_code(db, sales_record.codigo_nacional)
                    )
                finally:
                    loop.close()

                if product_catalog:
                    match_method = "codigo_nacional"
                    match_confidence = 95

            # 2. Si no encontró por CN, intentar por EAN13
            if not product_catalog and sales_record.ean13:
                product_catalog = self._find_by_ean(db, sales_record.ean13)
                if product_catalog:
                    match_method = "ean13"
                    match_confidence = 90

            # 3. Si no encontró, intentar matching fuzzy por nombre
            if not product_catalog and sales_record.product_name:
                product_catalog, fuzzy_score = self._find_by_name_fuzzy(db, sales_record.product_name)
                if product_catalog and fuzzy_score > 0.8:
                    match_method = "name_fuzzy"
                    match_confidence = int(fuzzy_score * 100)

            # 4. Actualizar registro existente
            if product_catalog:
                # Éxito: actualizar con nuevos datos
                existing_enrichment.product_catalog_id = product_catalog.id
                existing_enrichment.match_method = match_method
                existing_enrichment.match_confidence = match_confidence
                existing_enrichment.enrichment_source = product_catalog.data_sources or "nomenclator"
                existing_enrichment.enrichment_status = "enriched"

                # Derivar categorías farmacéuticas
                existing_enrichment.therapeutic_category = self._derive_therapeutic_category(product_catalog)
                existing_enrichment.prescription_category = self._derive_prescription_category(product_catalog)
                existing_enrichment.product_type = self._derive_product_type(product_catalog)

                # Verificar si tiene conjunto homogéneo
                if product_catalog.nomen_codigo_homogeneo:
                    existing_enrichment.has_generic_alternative = True
                else:
                    existing_enrichment.has_generic_alternative = False

                existing_enrichment.enriched_at = utc_now()
                existing_enrichment.error_messages = None  # Limpiar errores previos

                logger.info(
                    f"[REENRICH SUCCESS] {sales_record.codigo_nacional}: {previous_status} → enriched ({match_method})"
                )
            else:
                # Sigue sin encontrar en catálogos: intentar prescription_reference_list
                # Issue #444: Productos como tiras reactivas no están en CIMA pero sí en listados oficiales
                from app.models.prescription_reference_list import PrescriptionReferenceList

                reference = None
                if sales_record.codigo_nacional:
                    reference = (
                        db.query(PrescriptionReferenceList)
                        .filter(PrescriptionReferenceList.national_code == sales_record.codigo_nacional)
                        .first()
                    )

                if reference:
                    # Producto encontrado en listados oficiales del Ministerio
                    existing_enrichment.enrichment_status = "enriched"
                    existing_enrichment.enrichment_source = f"prescription_reference_list:{reference.reference_source}"
                    existing_enrichment.match_method = "reference_list"
                    existing_enrichment.match_confidence = 90
                    existing_enrichment.prescription_category = reference.category.value if hasattr(reference.category, 'value') else str(reference.category)
                    existing_enrichment.product_type = "prescription"
                    existing_enrichment.error_messages = None  # Limpiar errores previos
                    logger.info(
                        f"[REENRICH SUCCESS] {sales_record.codigo_nacional}: {previous_status} → enriched (reference_list)"
                    )
                else:
                    # No encontrado en catálogos NI en reference_list: actualizar solo metadatos
                    existing_enrichment.enrichment_status = "manual_review"
                    existing_enrichment.manual_review_reason = "No encontrado tras reenriquecimiento"

                    # Intentar clasificación ML básica
                    if sales_record.product_name:
                        ml_result = self._classify_with_ml(sales_record.product_name)
                        if ml_result:
                            existing_enrichment.ml_category = ml_result.get("category")
                            existing_enrichment.ml_confidence = ml_result.get("confidence")
                            existing_enrichment.ml_model_version = ml_result.get("model_version", "v1.0")

                        # Issue #446: Brand detection para taxonomía venta libre
                        self._apply_brand_detection(existing_enrichment, sales_record.product_name)

                    # CRÍTICO: Productos sin catálogo con clasificación ML son venta_libre
                    if existing_enrichment.ml_category:
                        existing_enrichment.product_type = "venta_libre"

                        # Issue #474: Registrar en ProductCatalogVentaLibre con CN/EAN
                        try:
                            vl_catalog = VentaLibreCatalogService(db)
                            vl_product, was_created = vl_catalog.find_or_create(
                                product_name=sales_record.product_name,
                                pharmacy_id=sales_record.pharmacy_id,
                                ean13=sales_record.ean13,
                                cn=sales_record.codigo_nacional,
                                ml_category=existing_enrichment.ml_category,
                                ml_confidence=existing_enrichment.ml_confidence,
                                prediction_source=existing_enrichment.ml_model_version,
                                detected_brand=existing_enrichment.detected_brand,
                            )
                            existing_enrichment.venta_libre_product_id = vl_product.id
                        except Exception as vl_error:
                            logger.warning(
                                f"[VENTA_LIBRE] Error in reenrich catalog entry: {vl_error}"
                            )

            # 5. Actualizar metadatos de reenriquecimiento
            processing_time = (utc_now() - start_time).total_seconds() * 1000
            existing_enrichment.processing_time_ms = int(processing_time)

            # Guardar cambios
            db.add(existing_enrichment)
            db.flush()

            return existing_enrichment if product_catalog else None

        except Exception as e:
            logger.error(
                "reenrichment.record.error",
                codigo_nacional=sales_record.codigo_nacional,
                error=str(e),
            )
            existing_enrichment.enrichment_status = "failed"
            existing_enrichment.error_messages = str(e)
            db.add(existing_enrichment)
            db.flush()
            return None

    def _find_by_ean(self, db: Session, ean: str) -> Optional[ProductCatalog]:
        """Buscar producto por código EAN en el catálogo"""
        # NOTA: ProductCatalog no tiene campo ean_code actualmente
        # Esta funcionalidad se implementará en futuras versiones
        return None

    def _find_by_name_fuzzy(self, db: Session, product_name: str) -> Tuple[Optional[ProductCatalog], float]:
        """
        Buscar producto por similitud de nombre.
        Retorna el producto y el score de similitud.
        """
        if not product_name or len(product_name.strip()) < 3:
            return None, 0.0

        # Implementación básica - en producción usaríamos PostgreSQL full-text search
        # o librerías como fuzzywuzzy
        clean_name = product_name.strip().upper()

        # Buscar coincidencias exactas primero
        exact_match = (
            db.query(ProductCatalog).filter(ProductCatalog.cima_nombre_comercial.ilike(f"%{clean_name}%")).first()
        )

        if exact_match:
            return exact_match, 0.85

        # Buscar por palabras clave principales
        words = clean_name.split()[:3]  # Primeras 3 palabras
        if len(words) >= 2:
            search_pattern = "%".join(words)
            partial_match = (
                db.query(ProductCatalog)
                .filter(ProductCatalog.cima_nombre_comercial.ilike(f"%{search_pattern}%"))
                .first()
            )

            if partial_match:
                return partial_match, 0.75

        return None, 0.0

    def _derive_therapeutic_category(self, product_catalog: ProductCatalog) -> Optional[str]:
        """Derivar categoría terapéutica del producto"""
        if product_catalog.cima_atc_code:
            # Mapeo básico ATC -> Categoría terapéutica
            atc = product_catalog.cima_atc_code[:1]  # Primera letra del código ATC
            atc_map = {
                "A": "Aparato digestivo y metabolismo",
                "B": "Sangre y órganos formadores de sangre",
                "C": "Aparato cardiovascular",
                "D": "Medicamentos dermatológicos",
                "G": "Aparato genitourinario y hormonas sexuales",
                "H": "Preparados hormonales sistémicos",
                "J": "Antiinfecciosos para uso sistémico",
                "L": "Agentes antineoplásicos e inmunomoduladores",
                "M": "Aparato locomotor",
                "N": "Sistema nervioso",
                "P": "Productos antiparasitarios",
                "R": "Aparato respiratorio",
                "S": "Órganos de los sentidos",
                "V": "Varios",
            }
            return atc_map.get(atc, "No clasificado")

        return None

    def _derive_prescription_category(self, product_catalog: ProductCatalog) -> Optional[str]:
        """Derivar categoría de prescripción"""
        if product_catalog.cima_requiere_receta is True:
            return "Con receta"
        elif product_catalog.cima_requiere_receta is False:
            return "Venta Libre"
        else:
            return "No determinado"

    def _derive_product_type(self, product_catalog: ProductCatalog) -> str:
        """
        Determina el tipo de venta basado en clasificación de prescripción.

        División Jerárquica (Issue #16 Fase 2 + Issue #354):
        ====================================================

        REGLA GENERAL (13 categorías):
        - Tiene xfarma_prescription_category (cualquiera de las 14) → "prescription"
        - NO tiene xfarma_prescription_category (NULL) → "venta_libre"

        EXCEPCIÓN ESPECIAL - VETERINARIA (Issue #354):
        - VETERINARIA es la ÚNICA categoría donde tener la categoría NO determina el tipo
        - Para VETERINARIA, el tipo se determina por cima_requiere_receta:
          * cima_requiere_receta = True → "prescription" (requiere receta veterinaria)
          * cima_requiere_receta = False → "venta_libre" (no requiere receta)
          * cima_requiere_receta = NULL → "prescription" (por seguridad)

        CONTEXTO:
        - xfarma_prescription_category = Categoría del producto (QUÉ es)
        - product_type = Tipo de venta (CÓMO se vende: con/sin receta)
        - Para las 13 categorías NO veterinarias: categoría → automáticamente prescription
        - Para VETERINARIA: se consulta campo adicional vet_requiere_receta importado de CIMAVet

        Args:
            product_catalog: Producto del catálogo

        Returns:
            "prescription": Requiere receta (humana o veterinaria)
            "venta_libre": Venta libre (no requiere receta)
        """

        # EXCEPCIÓN: Categoría VETERINARIA
        # Es la ÚNICA donde tener la categoría NO implica automáticamente prescription
        if product_catalog.xfarma_prescription_category == "VETERINARIA":
            # Consultar campo específico de CIMAVet (vet_requiere_receta)
            # que se mapea a cima_requiere_receta al crear el producto
            if product_catalog.cima_requiere_receta is True:
                return "prescription"  # Requiere receta veterinaria
            elif product_catalog.cima_requiere_receta is False:
                return "venta_libre"  # Venta libre (ej: antiparasitarios básicos)
            else:
                # NULL: asumir prescription por seguridad hasta que se importe dato real
                return "prescription"

        # REGLA GENERAL: Resto de categorías (13 categorías)
        # Tener categoría → automáticamente prescription
        elif product_catalog.xfarma_prescription_category is not None:
            return "prescription"

        # Sin categoría (NULL) → Venta Libre
        # Parafarmacia, cosmética, suplementos, productos no catalogados
        else:
            return "venta_libre"

    def _classify_with_ml(self, product_name: str) -> Optional[Dict]:
        """
        Clasificación ML básica para productos no encontrados en catálogos.
        En producción esto sería un modelo ML entrenado.
        """
        if not product_name:
            return None

        # Clasificación por palabras clave (implementación básica)
        name_upper = product_name.upper()

        # Patrones básicos de clasificación
        if any(word in name_upper for word in ["IBUPROFENO", "PARACETAMOL", "ASPIRINA"]):
            return {
                "category": "Analgésicos/Antiinflamatorios",
                "confidence": 0.80,
                "model_version": "keyword_v1.0",
            }
        elif any(word in name_upper for word in ["VITAMINA", "OMEGA", "CALCIO"]):
            return {
                "category": "Complementos nutricionales",
                "confidence": 0.75,
                "model_version": "keyword_v1.0",
            }
        elif any(word in name_upper for word in ["CREMA", "LOCIÓN", "GEL"]):
            return {
                "category": "Dermofarmacia",
                "confidence": 0.70,
                "model_version": "keyword_v1.0",
            }
        else:
            return {
                "category": "Otros",
                "confidence": 0.50,
                "model_version": "keyword_v1.0",
            }

    def _apply_brand_detection(self, enrichment: SalesEnrichment, product_name: str) -> None:
        """
        Aplica detección de marca a un registro de enriquecimiento.

        Issue #446: Taxonomía jerárquica de venta libre
        ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

        Detecta y asigna:
        - detected_brand: Marca del producto (ej: "isdin", "heliocare")
        - brand_line: Línea de producto (ej: "fusion water", "360")
        - ml_subcategory: Zona/subcategoría (ej: "facial", "corporal")

        Si la detección de marca encuentra una NECESIDAD, puede sobrescribir ml_category
        con una clasificación más precisa.

        Args:
            enrichment: Registro de SalesEnrichment a actualizar
            product_name: Nombre del producto
        """
        if not product_name:
            return

        try:
            result = brand_detection_service.detect(product_name)

            if result.detected_brand:
                enrichment.detected_brand = result.detected_brand
                enrichment.brand_line = result.brand_line
                enrichment.ml_subcategory = result.ml_subcategory

                # Si brand detection encontró una NECESIDAD más específica, usarla
                if result.necesidad:
                    # Solo sobrescribir si tenemos mejor confianza o ml_category es genérico
                    current_category = enrichment.ml_category
                    if (
                        not current_category
                        or current_category in ["Otros", "Dermofarmacia"]
                        or (enrichment.ml_confidence and result.confidence > enrichment.ml_confidence)
                    ):
                        enrichment.ml_category = result.necesidad
                        enrichment.ml_confidence = result.confidence
                        enrichment.ml_model_version = "brand_detection_v1.0"

                logger.debug(
                    "brand_detection.applied",
                    product_name=product_name,
                    brand=result.detected_brand,
                    line=result.brand_line,
                    subcategory=result.ml_subcategory,
                    necesidad=result.necesidad,
                )

        except Exception as e:
            # No fallar el enriquecimiento por error en brand detection
            logger.warning(
                "brand_detection.error",
                product_name=product_name,
                error=str(e),
            )

    def get_enrichment_progress(self, db: Session, pharmacy_id: str) -> Dict:
        """
        Obtiene el progreso del enriquecimiento para una farmacia.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia

        Returns:
            Diccionario con estadísticas de progreso
        """
        # Total de registros de ventas
        total_sales = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy_id).count()

        # Registros enriquecidos exitosamente
        enriched_count = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(SalesEnrichment.enrichment_status == "enriched")
            .count()
        )

        # Registros pendientes (sin enriquecimiento)
        pending_count = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .outerjoin(SalesEnrichment)
            .filter(SalesEnrichment.id.is_(None))
            .count()
        )

        # Registros en revisión manual
        manual_review_count = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(SalesEnrichment.enrichment_status == "manual_review")
            .count()
        )

        # Registros fallidos completamente
        failed_count = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(SalesEnrichment.enrichment_status == "failed")
            .count()
        )

        # Estadísticas por fuente de enriquecimiento
        cima_matches = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "enriched",
                    SalesEnrichment.enrichment_source == "cima",
                )
            )
            .count()
        )

        nomenclator_matches = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "enriched",
                    SalesEnrichment.enrichment_source == "nomenclator",
                )
            )
            .count()
        )

        both_sources_matches = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "enriched",
                    SalesEnrichment.enrichment_source == "nomenclator,cima",
                )
            )
            .count()
        )

        # Estadísticas de productos únicos
        from sqlalchemy import distinct, func

        # Total de productos únicos en ventas
        unique_products_total = (
            db.query(func.count(distinct(SalesData.codigo_nacional)))
            .filter(
                and_(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional.isnot(None),
                )
            )
            .scalar()
            or 0
        )

        # Productos únicos enriquecidos con CIMA
        unique_products_cima = (
            db.query(func.count(distinct(SalesData.codigo_nacional)))
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "enriched",
                    SalesEnrichment.enrichment_source == "cima",
                    SalesData.codigo_nacional.isnot(None),
                )
            )
            .scalar()
            or 0
        )

        # Productos únicos enriquecidos con nomenclator
        unique_products_nomenclator = (
            db.query(func.count(distinct(SalesData.codigo_nacional)))
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "enriched",
                    SalesEnrichment.enrichment_source == "nomenclator",
                    SalesData.codigo_nacional.isnot(None),
                )
            )
            .scalar()
            or 0
        )

        # Productos únicos enriquecidos con ambas fuentes
        unique_products_both = (
            db.query(func.count(distinct(SalesData.codigo_nacional)))
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "enriched",
                    SalesEnrichment.enrichment_source == "nomenclator,cima",
                    SalesData.codigo_nacional.isnot(None),
                )
            )
            .scalar()
            or 0
        )

        # Productos únicos en revisión manual
        unique_products_manual_review = (
            db.query(func.count(distinct(SalesData.codigo_nacional)))
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .join(SalesEnrichment)
            .filter(
                and_(
                    SalesEnrichment.enrichment_status == "manual_review",
                    SalesData.codigo_nacional.isnot(None),
                )
            )
            .scalar()
            or 0
        )

        # Incluir manual_review en la tasa porque también están enriquecidos (tienen datos del catálogo)
        total_enriched = enriched_count + manual_review_count
        enrichment_rate = (total_enriched / total_sales * 100) if total_sales > 0 else 0

        return {
            "total_sales": total_sales,
            "enriched": enriched_count,
            "pending": pending_count,
            "manual_review": manual_review_count,
            "failed": failed_count,
            "enrichment_rate": round(enrichment_rate, 2),
            "cima_matches": cima_matches,
            "nomenclator_matches": nomenclator_matches,
            "both_sources_matches": both_sources_matches,
            "needs_processing": pending_count + manual_review_count > 0,
            # Productos únicos
            "unique_products": {
                "total": unique_products_total,
                "cima": unique_products_cima,
                "nomenclator": unique_products_nomenclator,
                "both_sources": unique_products_both,
                "manual_review": unique_products_manual_review,
            },
        }

    def _enrich_single_record_cached(
        self, db: Session, sales_record: SalesData, product_cache: dict, stats: dict = None
    ) -> Optional[SalesEnrichment]:
        """
        Versión optimizada que usa cache para evitar queries repetidas.
        Diseñada para escala de 200K-400K registros.

        Issue #446: Añadida cascada CIMAVet para productos veterinarios.
        Cuando hay cache miss, delega a _find_by_national_code que incluye
        la cascada completa: CIMA → Nomenclator → CIMAVet → manual_review

        Args:
            db: Sesión de base de datos (para cascada CIMAVet)
            sales_record: Registro de venta a enriquecer
            product_cache: Cache de productos {codigo_nacional: ProductCatalog}
            stats: Diccionario de estadísticas (opcional)

        Returns:
            SalesEnrichment creado o None si no se pudo enriquecer
        """
        start_time = utc_now()

        try:
            # 1. Buscar en cache primero por código nacional
            product_catalog = None
            match_method = None
            match_confidence = 0

            if sales_record.codigo_nacional and sales_record.codigo_nacional in product_cache:
                product_catalog = product_cache[sales_record.codigo_nacional]
                match_method = "codigo_nacional"
                match_confidence = 95
                if stats:
                    stats["cache_hits"] = stats.get("cache_hits", 0) + 1

            # 2. Si no está en cache, intentar fuzzy matching limitado (solo para nombres largos)
            elif sales_record.product_name and len(sales_record.product_name) > 15:
                if stats:
                    stats["cache_misses"] = stats.get("cache_misses", 0) + 1
                # Fuzzy matching limitado para evitar false positives
                best_match = None
                best_score = 0

                # Limitar búsqueda a primeros 200 productos del cache
                for cached_product in list(product_cache.values())[:200]:
                    if cached_product.cima_nombre_comercial:
                        score = self._calculate_similarity(
                            sales_record.product_name,
                            cached_product.cima_nombre_comercial,
                        )
                        if score > 0.85 and score > best_score:
                            best_match = cached_product
                            best_score = score

                if best_match:
                    product_catalog = best_match
                    match_method = "name_fuzzy"
                    match_confidence = int(best_score * 100)

            # 3. Crear enriquecimiento si encontró match
            if product_catalog:
                enrichment = SalesEnrichment(
                    sales_data_id=sales_record.id,
                    product_catalog_id=product_catalog.id,
                    match_method=match_method,
                    match_confidence=match_confidence,
                    enrichment_source=product_catalog.data_sources or "nomenclator",
                    enrichment_status="enriched",
                    enriched_at=utc_now(),
                )

                # Derivar categorías básicas
                enrichment.prescription_category = self._derive_prescription_category(product_catalog)
                enrichment.product_type = self._derive_product_type(product_catalog)

                # Verificar si tiene conjunto homogéneo
                if product_catalog.nomen_codigo_homogeneo:
                    enrichment.has_generic_alternative = True

                # Performance tracking
                processing_time = (utc_now() - start_time).total_seconds() * 1000
                enrichment.processing_time_ms = int(processing_time)

                return enrichment
            else:
                # 🆕 Issue #446: Antes de ir a manual_review, intentar cascada completa
                # Si hay codigo_nacional, delegar a _find_by_national_code que incluye:
                # CIMA → Nomenclator → CIMAVet local → CIMAVet API → manual_review
                if sales_record.codigo_nacional:
                    import asyncio

                    try:
                        # _find_by_national_code es async
                        loop = asyncio.new_event_loop()
                        asyncio.set_event_loop(loop)
                        try:
                            product_catalog = loop.run_until_complete(
                                self._find_by_national_code(db, sales_record.codigo_nacional)
                            )
                        finally:
                            loop.close()

                        if product_catalog:
                            # Encontrado via cascada (probablemente CIMAVet)
                            enrichment = SalesEnrichment(
                                sales_data_id=sales_record.id,
                                product_catalog_id=product_catalog.id,
                                match_method="codigo_nacional",
                                match_confidence=90,  # Alta confianza para cascada
                                enrichment_source=product_catalog.data_sources or "cascada",
                                enrichment_status="enriched",
                                enriched_at=utc_now(),
                            )

                            # Derivar categorías
                            enrichment.prescription_category = self._derive_prescription_category(product_catalog)
                            enrichment.product_type = self._derive_product_type(product_catalog)

                            if product_catalog.nomen_codigo_homogeneo:
                                enrichment.has_generic_alternative = True

                            # Actualizar cache para futuras búsquedas
                            product_cache[sales_record.codigo_nacional] = product_catalog

                            processing_time = (utc_now() - start_time).total_seconds() * 1000
                            enrichment.processing_time_ms = int(processing_time)

                            if stats:
                                stats["cache_misses"] = stats.get("cache_misses", 0) + 1

                            logger.debug(
                                f"[ENRICHMENT CACHED] CN {sales_record.codigo_nacional} "
                                f"enriquecido via cascada: {product_catalog.data_sources}"
                            )
                            return enrichment

                    except Exception as cascade_error:
                        logger.warning(
                            f"[ENRICHMENT CACHED] Error en cascada para {sales_record.codigo_nacional}: {cascade_error}"
                        )
                        # Continuar a manual_review

                # No encontrado en ninguna fuente - crear registro para revisión manual
                enrichment = SalesEnrichment(
                    sales_data_id=sales_record.id,
                    enrichment_status="manual_review",
                    manual_review_reason="No encontrado en enriquecimiento optimizado ni cascada",
                    processing_time_ms=int((utc_now() - start_time).total_seconds() * 1000),
                )

                # Issue #446: Brand detection para taxonomía venta libre
                if sales_record.product_name:
                    self._apply_brand_detection(enrichment, sales_record.product_name)

                # CRÍTICO: Productos sin catálogo con clasificación ML son venta_libre
                if enrichment.ml_category:
                    enrichment.product_type = "venta_libre"

                    # Issue #474: Registrar en ProductCatalogVentaLibre con CN/EAN
                    try:
                        vl_catalog = VentaLibreCatalogService(db)
                        vl_product, was_created = vl_catalog.find_or_create(
                            product_name=sales_record.product_name,
                            pharmacy_id=sales_record.pharmacy_id,
                            ean13=sales_record.ean13,
                            cn=sales_record.codigo_nacional,
                            ml_category=enrichment.ml_category,
                            ml_confidence=enrichment.ml_confidence,
                            prediction_source=enrichment.ml_model_version,
                            detected_brand=enrichment.detected_brand,
                        )
                        enrichment.venta_libre_product_id = vl_product.id
                    except Exception as vl_error:
                        logger.warning(
                            f"[VENTA_LIBRE] Error in batch catalog entry: {vl_error}"
                        )

                if stats:
                    stats["manual_review"] = stats.get("manual_review", 0) + 1
                return enrichment

        except Exception as e:
            # Error - crear registro fallido
            enrichment = SalesEnrichment(
                sales_data_id=sales_record.id,
                enrichment_status="failed",
                error_messages=str(e),
                processing_time_ms=int((utc_now() - start_time).total_seconds() * 1000),
            )
            return enrichment

    def _calculate_similarity(self, str1: str, str2: str) -> float:
        """
        Cálculo simple de similitud optimizado para volumen alto
        """
        if not str1 or not str2:
            return 0.0

        # Normalizar strings
        s1 = str1.upper().strip()
        s2 = str2.upper().strip()

        if s1 == s2:
            return 1.0

        # Similitud basada en palabras comunes (Jaccard)
        words1 = set(s1.split())
        words2 = set(s2.split())

        if not words1 or not words2:
            return 0.0

        intersection = len(words1.intersection(words2))
        union = len(words1.union(words2))

        return intersection / union if union > 0 else 0.0

    def enrich_sales_batch(
        self,
        db: Session,
        pharmacy_id: str,
        upload_id: Optional[str] = None,
        batch_size: int = 1000,
        progress_callback: Optional[Callable] = None,
    ) -> Dict:
        """
        Enriquecimiento optimizado para cualquier escala (desde 1K hasta 400K+ registros).

        NOTA: Esta es la única implementación activa de enrich_sales_batch.
        Definición duplicada anterior fue eliminada en commit 2c66480.

        Características optimizadas:
        - Procesamiento en chunks para evitar OOM
        - Bulk operations para máximo rendimiento
        - Cache inteligente con productos frecuentes
        - Progress tracking para UX en procesos largos
        - Memory-conscious design
        - Auto-adaptación según volumen

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            upload_id: ID de upload específico (opcional)
            batch_size: Tamaño de chunk (default: 1000)
            progress_callback: Callback para tracking de progreso (opcional)

        Returns:
            Diccionario con estadísticas detalladas del enriquecimiento
        """
        logger.info(f"[ENRICHMENT] Iniciando enriquecimiento optimizado para farmacia {pharmacy_id}")

        stats = {
            "processed": 0,
            "enriched": 0,
            "failed": 0,
            "duplicates_skipped": 0,
            "matches_by_code": 0,
            "matches_by_ean": 0,
            "matches_by_name": 0,
            "manual_review": 0,
            "total_chunks": 0,
            "cache_hits": 0,
            "cache_misses": 0,
        }

        # 1. Pre-cargar productos más frecuentes para cache inicial
        logger.info("[ENRICHMENT] Precargando productos frecuentes...")
        frequent_products = (
            db.query(ProductCatalog)
            .join(SalesEnrichment)
            .filter(SalesEnrichment.enrichment_status == "enriched")
            .group_by(ProductCatalog.id)
            .order_by(func.count(SalesEnrichment.id).desc())
            .limit(2000)
            .all()
        )

        product_cache = {p.national_code: p for p in frequent_products}
        logger.info(f"[ENRICHMENT] Cache inicializado con {len(product_cache)} productos frecuentes")

        # 2. Procesamiento en chunks para evitar memory issues
        # ✅ NO usar offset - la query filtra dinámicamente registros sin enrichment
        total_estimated = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .outerjoin(SalesEnrichment)
            .filter(SalesEnrichment.id.is_(None))
            .count()
        )

        if total_estimated == 0:
            logger.info("[ENRICHMENT] No hay registros para procesar")
            return stats

        logger.info(f"[ENRICHMENT] Estimados {total_estimated} registros para procesar en chunks de {batch_size}")

        while True:
            # 2.1. Obtener chunk
            chunk_query = (
                db.query(SalesData)
                .filter(SalesData.pharmacy_id == pharmacy_id)
                .outerjoin(SalesEnrichment)
                .filter(SalesEnrichment.id.is_(None))
            )

            if upload_id:
                chunk_query = chunk_query.filter(SalesData.upload_id == upload_id)

            # ✅ CRÍTICO: NO usar offset - la query filtra dinámicamente por sin enrichment
            # Cada iteración obtiene los PRIMEROS batch_size registros sin enrichment
            sales_chunk = chunk_query.limit(batch_size).all()

            if not sales_chunk:
                break  # No más registros

            stats["total_chunks"] += 1
            logger.info(f"[ENRICHMENT] Procesando chunk {stats['total_chunks']} ({len(sales_chunk)} registros)")

            # 2.2. Extraer códigos únicos del chunk
            chunk_codes = list(
                set(
                    [
                        s.codigo_nacional
                        for s in sales_chunk
                        if s.codigo_nacional and len(str(s.codigo_nacional).strip()) >= 4
                    ]
                )
            )

            # 2.3. Bulk lookup solo para códigos no en cache
            new_codes = [code for code in chunk_codes if code not in product_cache]
            if new_codes:
                new_products = db.query(ProductCatalog).filter(ProductCatalog.national_code.in_(new_codes)).all()

                # Actualizar cache
                for product in new_products:
                    product_cache[product.national_code] = product

                logger.debug(f"[ENRICHMENT] Cache actualizado con {len(new_products)} productos nuevos")

            # 2.4. Procesar chunk con cache
            enrichments_to_create = []

            for sales_record in sales_chunk:
                try:
                    # Verificar cache hit/miss
                    if sales_record.codigo_nacional in product_cache:
                        stats["cache_hits"] += 1
                    else:
                        stats["cache_misses"] += 1

                    enrichment = self._enrich_single_record_cached(db, sales_record, product_cache, None)

                    if enrichment:
                        enrichments_to_create.append(enrichment)

                        if enrichment.enrichment_status == "enriched":
                            stats["enriched"] += 1
                            if enrichment.match_method == "codigo_nacional":
                                stats["matches_by_code"] += 1
                            elif enrichment.match_method == "name_fuzzy":
                                stats["matches_by_name"] += 1
                        elif enrichment.enrichment_status == "manual_review":
                            stats["manual_review"] += 1
                        else:
                            stats["failed"] += 1

                    stats["processed"] += 1

                    # FIX Issue #253: Checkpoints de progreso cada 1000 registros
                    if stats["processed"] % 1000 == 0:
                        logger.info(
                            f"🔄 [CHECKPOINT] Progreso enriquecimiento: {stats['processed']} registros procesados - "
                            f"Enriquecidos: {stats['enriched']} ({stats['enriched']/stats['processed']*100:.1f}%) - "
                            f"Fallidos: {stats['failed']} - Cache hits: {stats['cache_hits']}/{stats['cache_hits'] + stats['cache_misses']}"
                        )

                except ProductNotFoundError as e:
                    # Producto no encontrado en catálogo es esperado a veces
                    logger.warning(
                        "enrichment.product.not_found",
                        sales_record_id=str(sales_record.id),
                        product_code=e.details.get("product_code"),
                    )
                    stats["failed"] += 1
                    continue
                except Exception as e:
                    logger.error(
                        "enrichment.record.error",
                        sales_record_id=str(sales_record.id),
                        error=str(e),
                    )
                    stats["failed"] += 1

                    # Si hay muchos errores, lanzar excepción
                    if stats["failed"] > len(sales_chunk) * 0.5:  # Más del 50% fallos
                        raise EnrichmentError(
                            stage="batch_processing",
                            product_count=stats["processed"],
                            reason=f"Demasiados fallos en enriquecimiento: {stats['failed']} de {stats['processed']}",
                        )
                    continue

            # 2.5. Bulk insert del chunk completo
            if enrichments_to_create:
                db.bulk_save_objects(enrichments_to_create)

            # 2.6. Commit del chunk
            db.commit()

            # 2.7. Progress callback para UI
            if progress_callback:
                progress = {
                    "processed": stats["processed"],
                    "total_estimated": total_estimated,
                    "percentage": ((stats["processed"] / total_estimated * 100) if total_estimated > 0 else 0),
                    "current_chunk": stats["total_chunks"],
                    "cache_efficiency": (
                        (stats["cache_hits"] / (stats["cache_hits"] + stats["cache_misses"]) * 100)
                        if (stats["cache_hits"] + stats["cache_misses"]) > 0
                        else 0
                    ),
                }
                progress_callback(progress)

            logger.info(
                f"[ENRICHMENT] Chunk {stats['total_chunks']} completado - "
                f"Procesados: {stats['processed']}/{total_estimated} ({stats['processed']/total_estimated*100:.1f}%) - "
                f"Cache: {len(product_cache)} productos - "
                f"Hit rate: {stats['cache_hits']/(stats['cache_hits'] + stats['cache_misses'])*100:.1f}%"
            )

            # ✅ NO incrementar offset - la query se auto-actualiza al crear enrichments

            # 2.8. Limitar cache para evitar memory issues
            if len(product_cache) > 10000:
                # Mantener solo productos más usados
                most_used = dict(list(product_cache.items())[:5000])
                product_cache.clear()
                product_cache.update(most_used)
                logger.debug("[ENRICHMENT] Cache optimizado por memoria")

        logger.info(f"[ENRICHMENT] Completado - {stats}")
        return stats


# Instancia global del servicio
enrichment_service = EnrichmentService()
