# backend/app/services/employee_service.py
"""
Employee service for employee filtering feature (Issue #402).

Provides functionality to:
- List unique employees per pharmacy from sales data
- Validate subscription plan access (PRO+ only)
- Support filtering in generic analysis endpoints
"""

import logging
from datetime import date
from decimal import Decimal
from typing import List, Optional
from uuid import UUID

from fastapi import HTTPException
from sqlalchemy import text
from sqlalchemy.orm import Session

from app.core.subscription_limits import can_use_employee_filtering
from app.models.user import User
from app.schemas.employee import EmployeeResponse, EmployeeListResponse

logger = logging.getLogger(__name__)


class EmployeeService:
    """Service for managing pharmacy employees data"""

    def __init__(self, db: Session):
        self.db = db

    def get_pharmacy_employees(
        self,
        pharmacy_id: UUID,
        user: User,
    ) -> EmployeeListResponse:
        """
        Get unique employees for a pharmacy from sales data.

        Issue #402: PRO+ feature - Employee filtering for generic analysis.

        Args:
            pharmacy_id: UUID of the pharmacy
            user: Current authenticated user (for subscription check)

        Returns:
            EmployeeListResponse with list of employees ordered by revenue

        Raises:
            HTTPException 403: If user plan doesn't have access to employee filtering
            HTTPException 404: If pharmacy not found or user doesn't own pharmacy
            HTTPException 500: If database query fails

        Example:
            >>> service = EmployeeService(db)
            >>> result = service.get_pharmacy_employees(pharmacy_id, user)
            >>> result.total_count
            3
            >>> result.employees[0].employee_name
            'María García'

        Note:
            Plan enforcement handled by @require_subscription_plan("pro")
            decorator at API layer (endpoint). Service focuses on business logic.
        """
        # Verify user owns this pharmacy (REGLA #10: 1:1 relationship)
        if user.pharmacy_id != pharmacy_id:
            logger.warning(
                f"[EmployeeService] User {user.email} attempted to access "
                f"employees for pharmacy {pharmacy_id} - Unauthorized"
            )
            raise HTTPException(
                status_code=404,
                detail="Pharmacy not found or access denied",
            )

        logger.info(
            f"[EmployeeService] Fetching employees for pharmacy {pharmacy_id} "
            f"(user: {user.email}, plan: {user.subscription_plan})"
        )

        try:
            # DECISIÓN #2 APROBADA: Usar solo employee_name (sin employee_code)
            # DECISIÓN UX #3 APROBADA: Incluir ventas sin empleado como "__sin_empleado__"
            #
            # Query usa COALESCE para convertir NULLs a '__sin_empleado__'
            # Ordenado por sales_count DESC (más ventas primero)
            query = text("""
                SELECT
                    COALESCE(sd.employee_name, '__sin_empleado__') as employee_name,
                    COUNT(*) as sales_count,
                    MAX(sd.sale_date) as last_sale_date
                FROM sales_data sd
                WHERE sd.pharmacy_id = :pharmacy_id
                GROUP BY COALESCE(sd.employee_name, '__sin_empleado__')
                ORDER BY sales_count DESC
            """)

            result = self.db.execute(
                query,
                {"pharmacy_id": str(pharmacy_id)},
            ).fetchall()

            # Convert to EmployeeResponse objects
            employees = []
            for row in result:
                employee = EmployeeResponse(
                    name=row.employee_name,
                    sales_count=row.sales_count,
                    last_sale_date=row.last_sale_date.date() if row.last_sale_date else date.today(),
                )
                employees.append(employee)

            logger.info(
                f"[EmployeeService] Found {len(employees)} employees for pharmacy {pharmacy_id} "
                f"(includes '__sin_empleado__' if present)"
            )

            return EmployeeListResponse(
                employees=employees,
                total_count=len(employees),
            )

        except Exception as e:
            logger.error(
                f"[EmployeeService] Error fetching employees for pharmacy {pharmacy_id}: {str(e)}",
                exc_info=True,
            )
            raise HTTPException(
                status_code=500,
                detail="Error fetching employees data. Please try again later.",
            )

    def validate_employee_names(
        self,
        pharmacy_id: UUID,
        employee_names: List[str],
    ) -> List[str]:
        """
        Validate that employee names exist for a pharmacy.

        DECISIÓN #2 APROBADA: Validar por employee_name (no employee_code)

        Args:
            pharmacy_id: UUID of the pharmacy
            employee_names: List of employee names to validate (can include '__sin_empleado__')

        Returns:
            List of valid employee names that exist in the pharmacy

        Example:
            >>> service = EmployeeService(db)
            >>> valid_names = service.validate_employee_names(
            ...     pharmacy_id,
            ...     ["María García", "Juan Pérez", "Empleado Inexistente"]
            ... )
            >>> valid_names
            ['María García', 'Juan Pérez']  # 'Empleado Inexistente' doesn't exist
        """
        if not employee_names:
            return []

        try:
            # Use PostgreSQL array parameter (safer than dynamic placeholders)
            query = text("""
                SELECT DISTINCT COALESCE(employee_name, '__sin_empleado__') as employee_name
                FROM sales_data
                WHERE pharmacy_id = :pharmacy_id
                  AND COALESCE(employee_name, '__sin_empleado__') = ANY(:employee_names)
            """)

            result = self.db.execute(
                query,
                {
                    "pharmacy_id": str(pharmacy_id),
                    "employee_names": employee_names,  # PostgreSQL array parameter
                }
            ).fetchall()

            valid_names = [row.employee_name for row in result]

            if len(valid_names) < len(employee_names):
                invalid_names = set(employee_names) - set(valid_names)
                logger.warning(
                    f"[EmployeeService] Invalid employee names for pharmacy {pharmacy_id}: "
                    f"{invalid_names}"
                )

            return valid_names

        except Exception as e:
            logger.error(
                f"[EmployeeService] Error validating employee names: {str(e)}",
                exc_info=True,
            )
            return []

    @staticmethod
    def validate_employee_filtering_access(user: User) -> None:
        """
        Validate that user has access to employee filtering feature.

        Raises HTTPException 403 if user plan doesn't support employee filtering.

        Args:
            user: Current authenticated user

        Raises:
            HTTPException 403: If user plan < PRO

        Example:
            >>> EmployeeService.validate_employee_filtering_access(user)
            # Raises 403 if user.subscription_plan == 'free'
        """
        if not can_use_employee_filtering(user.subscription_plan):
            raise HTTPException(
                status_code=403,
                detail=(
                    f"Employee filtering requires PRO plan or higher. "
                    f"Current plan: {user.subscription_plan}. "
                    f"Please upgrade to access this feature."
                ),
            )

    def get_employee_stats(
        self,
        pharmacy_id: UUID,
        employee_names: List[str],
    ) -> dict:
        """
        Get aggregated statistics for selected employees.

        DECISIÓN #2 APROBADA: Stats por employee_name (no employee_code)

        Useful for analytics endpoints that need employee-level metrics.

        Args:
            pharmacy_id: UUID of the pharmacy
            employee_names: List of employee names to analyze (can include '__sin_empleado__')

        Returns:
            Dict with aggregated stats:
            {
                "total_sales": int,
                "total_revenue": Decimal,
                "avg_sale_amount": Decimal,
                "employees_count": int
            }

        Example:
            >>> service = EmployeeService(db)
            >>> stats = service.get_employee_stats(
            ...     pharmacy_id,
            ...     ["María García", "Juan Pérez"]
            ... )
            >>> stats["total_sales"]
            2750
        """
        if not employee_names:
            return {
                "total_sales": 0,
                "total_revenue": Decimal("0.00"),
                "avg_sale_amount": Decimal("0.00"),
                "employees_count": 0,
            }

        try:
            # Use PostgreSQL array parameter (safer than dynamic placeholders)
            query = text("""
                SELECT
                    COUNT(*) as total_sales,
                    SUM(total_amount) as total_revenue,
                    AVG(total_amount) as avg_sale_amount,
                    COUNT(DISTINCT COALESCE(employee_name, '__sin_empleado__')) as employees_count
                FROM sales_data
                WHERE pharmacy_id = :pharmacy_id
                  AND COALESCE(employee_name, '__sin_empleado__') = ANY(:employee_names)
            """)

            result = self.db.execute(
                query,
                {
                    "pharmacy_id": str(pharmacy_id),
                    "employee_names": employee_names,  # PostgreSQL array parameter
                }
            ).fetchone()

            return {
                "total_sales": result.total_sales or 0,
                "total_revenue": Decimal(str(result.total_revenue or 0)),
                "avg_sale_amount": Decimal(str(result.avg_sale_amount or 0)),
                "employees_count": result.employees_count or 0,
            }

        except Exception as e:
            logger.error(
                f"[EmployeeService] Error fetching employee stats: {str(e)}",
                exc_info=True,
            )
            return {
                "total_sales": 0,
                "total_revenue": Decimal("0.00"),
                "avg_sale_amount": Decimal("0.00"),
                "employees_count": 0,
            }
