# backend/app/services/database_management_service.py
"""
Servicio para gestión de base de datos y backups.

Issue #348 FASE 2: Backend - Services & APIs
Implementa creación de backups, verificación de integridad con HMAC,
y gestión de vistas materializadas.
"""
import hashlib
import hmac
import logging
import os
import subprocess
from datetime import datetime
from pathlib import Path
from typing import List, Optional

from sqlalchemy import text
from sqlalchemy.ext.asyncio import AsyncSession

from app.models.backup_metadata import BackupMetadata
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class DatabaseManagementService:
    """Servicio para gestión de base de datos, backups y vistas materializadas"""

    # FIX 3: HMAC Secret Fail-Secure (strict in production, lenient in development)
    HMAC_SECRET = os.getenv("BACKUP_HMAC_SECRET")
    if not HMAC_SECRET:
        environment = os.getenv("ENVIRONMENT", "development")
        if environment == "production":
            # CRITICAL: En producción NO puede arrancar sin HMAC secret
            logger.error("BACKUP_HMAC_SECRET environment variable is not set")
            raise ValueError(
                "BACKUP_HMAC_SECRET must be configured in environment variables. "
                "This is required for backup integrity verification in production."
            )
        else:
            # WARNING: En desarrollo usar fallback (solo para permitir testing)
            HMAC_SECRET = "dev-backup-secret-insecure-2024"
            logger.warning(
                "BACKUP_HMAC_SECRET not configured - using development fallback. "
                "DO NOT use this in production!"
            )

    @staticmethod
    def _calculate_file_hash(file_path: str) -> str:
        """
        Calcula el hash SHA-256 de un archivo.

        Args:
            file_path: Ruta al archivo

        Returns:
            Hash SHA-256 en hexadecimal
        """
        sha256_hash = hashlib.sha256()
        with open(file_path, "rb") as f:
            for byte_block in iter(lambda: f.read(4096), b""):
                sha256_hash.update(byte_block)
        return sha256_hash.hexdigest()

    @staticmethod
    def _generate_hmac_signature(file_hash: str) -> str:
        """
        Genera una firma HMAC para verificación de integridad.

        Args:
            file_hash: Hash SHA-256 del archivo

        Returns:
            Firma HMAC en hexadecimal
        """
        return hmac.new(
            DatabaseManagementService.HMAC_SECRET.encode(),
            file_hash.encode(),
            hashlib.sha256
        ).hexdigest()

    @staticmethod
    async def create_backup(
        db: AsyncSession,
        backup_path: Optional[str] = None
    ) -> BackupMetadata:
        """
        Crea un backup de la base de datos con verificación de integridad.

        Args:
            db: Sesión de base de datos
            backup_path: Ruta donde guardar el backup (opcional)

        Returns:
            Metadatos del backup creado

        Raises:
            RuntimeError: Si falla la creación del backup
        """
        try:
            # Generar nombre y ruta del backup si no se proporciona
            if not backup_path:
                backup_dir = Path("/app/backups")
                backup_dir.mkdir(parents=True, exist_ok=True)

                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                backup_filename = f"xfarma_backup_{timestamp}.sql"
                backup_path = str(backup_dir / backup_filename)

            # Construir comando pg_dump
            db_url = os.getenv("DATABASE_URL")
            # Extraer componentes de la URL
            # postgresql+asyncpg://user:pass@host:port/dbname
            import re
            match = re.match(
                r"postgresql\+asyncpg://([^:]+):([^@]+)@([^:/]+):(\d+)/(.+)",
                db_url
            )

            if not match:
                raise RuntimeError(f"No se pudo parsear DATABASE_URL: {db_url}")

            user, password, host, port, dbname = match.groups()

            # Crear comando pg_dump
            env = os.environ.copy()
            env["PGPASSWORD"] = password

            cmd = [
                "pg_dump",
                "-h", host,
                "-p", port,
                "-U", user,
                "-d", dbname,
                "-f", backup_path,
                "--verbose",
                "--clean",
                "--if-exists",
                "--no-owner",
                "--no-privileges"
            ]

            # Ejecutar pg_dump
            logger.info(f"Creando backup en {backup_path}...")
            result = subprocess.run(
                cmd,
                env=env,
                capture_output=True,
                text=True
            )

            if result.returncode != 0:
                raise RuntimeError(f"pg_dump falló: {result.stderr}")

            # Calcular hash y firma
            file_hash = DatabaseManagementService._calculate_file_hash(backup_path)
            signature = DatabaseManagementService._generate_hmac_signature(file_hash)

            # Obtener tamaño del archivo
            file_size = os.path.getsize(backup_path)

            # Crear registro en base de datos
            backup_metadata = BackupMetadata(
                file_path=backup_path,
                file_size=file_size,
                sha256_hash=file_hash,
                hmac_signature=signature,
                created_at=utc_now(),
                status="completed",
                tables_included=["all"],  # En un backup completo
                compression_type="none"  # Sin compresión por defecto
            )

            db.add(backup_metadata)
            await db.commit()
            await db.refresh(backup_metadata)

            logger.info(
                f"Backup creado exitosamente: {backup_path} "
                f"(Tamaño: {file_size/1024/1024:.2f} MB, Hash: {file_hash[:8]}...)"
            )

            return backup_metadata

        except Exception as e:
            logger.error(f"Error al crear backup: {str(e)}")

            # Intentar crear registro de fallo
            if 'backup_path' in locals():
                backup_metadata = BackupMetadata(
                    file_path=backup_path,
                    file_size=0,
                    created_at=utc_now(),
                    status="failed",
                    error_message=str(e)
                )
                db.add(backup_metadata)
                await db.commit()

            raise RuntimeError(f"Fallo al crear backup: {str(e)}")

    @staticmethod
    async def list_backups(
        db: AsyncSession,
        limit: int = 20
    ) -> List[BackupMetadata]:
        """
        Lista los backups disponibles con información de verificación.

        Args:
            db: Sesión de base de datos
            limit: Máximo de backups a devolver

        Returns:
            Lista de metadatos de backups
        """
        from sqlalchemy import select

        query = select(BackupMetadata).order_by(
            BackupMetadata.created_at.desc()
        ).limit(limit)

        result = await db.execute(query)
        backups = result.scalars().all()

        # Verificar integridad de cada backup
        for backup in backups:
            if backup.sha256_hash and backup.hmac_signature:
                # Verificar si el archivo existe
                if os.path.exists(backup.file_path):
                    try:
                        # Recalcular hash
                        current_hash = DatabaseManagementService._calculate_file_hash(
                            backup.file_path
                        )
                        # Verificar firma
                        expected_signature = DatabaseManagementService._generate_hmac_signature(
                            current_hash
                        )

                        backup.is_verified = (
                            current_hash == backup.sha256_hash and
                            expected_signature == backup.hmac_signature
                        )

                        if not backup.is_verified:
                            logger.warning(
                                f"Backup {backup.file_path} falló verificación de integridad"
                            )
                    except Exception as e:
                        logger.error(f"Error verificando backup {backup.file_path}: {e}")
                        backup.is_verified = False
                else:
                    backup.is_verified = False
                    backup.error_message = "Archivo no encontrado"

        logger.info(f"Listados {len(backups)} backups")

        return backups

    @staticmethod
    def verify_backup_integrity(
        backup_path: str,
        expected_hash: str,
        signature: str
    ) -> bool:
        """
        Verifica la integridad de un backup con HMAC.

        Args:
            backup_path: Ruta al archivo de backup
            expected_hash: Hash SHA-256 esperado
            signature: Firma HMAC esperada

        Returns:
            True si el backup es íntegro, False en caso contrario
        """
        try:
            if not os.path.exists(backup_path):
                logger.error(f"Archivo de backup no encontrado: {backup_path}")
                return False

            # Calcular hash actual
            current_hash = DatabaseManagementService._calculate_file_hash(backup_path)

            # Verificar hash
            if current_hash != expected_hash:
                logger.error(
                    f"Hash no coincide para {backup_path}: "
                    f"esperado={expected_hash[:8]}..., actual={current_hash[:8]}..."
                )
                return False

            # Verificar firma HMAC
            expected_signature = DatabaseManagementService._generate_hmac_signature(
                current_hash
            )

            if expected_signature != signature:
                logger.error(
                    f"Firma HMAC no coincide para {backup_path}: "
                    f"posible manipulación del archivo"
                )
                return False

            logger.info(f"Backup {backup_path} verificado exitosamente")
            return True

        except Exception as e:
            logger.error(f"Error verificando integridad del backup: {str(e)}")
            return False

    @staticmethod
    async def refresh_storage_stats(db: AsyncSession) -> None:
        """
        Actualiza la vista materializada de estadísticas de almacenamiento.

        Args:
            db: Sesión de base de datos

        Raises:
            RuntimeError: Si falla el refresh
        """
        try:
            logger.info("Actualizando vista materializada pharmacy_storage_stats...")

            # REFRESH MATERIALIZED VIEW
            await db.execute(
                text("REFRESH MATERIALIZED VIEW CONCURRENTLY pharmacy_storage_stats")
            )
            await db.commit()

            # Obtener estadísticas actualizadas para log
            result = await db.execute(
                text("""
                    SELECT
                        COUNT(*) as total_pharmacies,
                        SUM(total_files) as total_files,
                        SUM(total_size_mb) as total_size_mb,
                        SUM(sales_records_count) as total_sales,
                        SUM(enriched_records_count) as total_enriched
                    FROM pharmacy_storage_stats
                """)
            )
            stats = result.fetchone()

            logger.info(
                f"Vista actualizada - Farmacias: {stats.total_pharmacies}, "
                f"Archivos: {stats.total_files}, "
                f"Tamaño total: {stats.total_size_mb:.2f} MB, "
                f"Ventas: {stats.total_sales}, "
                f"Enriquecidas: {stats.total_enriched}"
            )

        except Exception as e:
            logger.error(f"Error actualizando vista materializada: {str(e)}")
            raise RuntimeError(f"Fallo al actualizar estadísticas: {str(e)}")

    @staticmethod
    async def refresh_partner_analytics(db: AsyncSession) -> None:
        """
        Actualiza la vista materializada de análisis de partners/laboratorios.

        Refreshes pharmacy_lab_sales_summary que pre-calcula ventas por (pharmacy_id, laboratory)
        de los últimos 13 meses. Esta view optimiza queries de /initialize (180s → < 2s).

        Args:
            db: Sesión de base de datos

        Raises:
            RuntimeError: Si falla el refresh

        Note:
            - Usa REFRESH MATERIALIZED VIEW CONCURRENTLY (no bloquea lecturas)
            - Should run daily (e.g., 2 AM) para mantener partners suggestions actuales
            - View size: ~5k-10k rows (100 pharmacies × 50-100 labs)
        """
        try:
            logger.info("Actualizando vista materializada pharmacy_lab_sales_summary...")

            # REFRESH MATERIALIZED VIEW CONCURRENTLY (no bloquea lecturas)
            await db.execute(
                text("REFRESH MATERIALIZED VIEW CONCURRENTLY pharmacy_lab_sales_summary")
            )
            await db.commit()

            # Obtener estadísticas actualizadas para log
            result = await db.execute(
                text("""
                    SELECT
                        COUNT(*) as total_rows,
                        COUNT(DISTINCT pharmacy_id) as total_pharmacies,
                        COUNT(DISTINCT laboratory_name) as total_labs,
                        SUM(total_sales) as total_sales_amount,
                        MAX(view_refreshed_at) as last_refresh
                    FROM pharmacy_lab_sales_summary
                """)
            )
            stats = result.fetchone()

            logger.info(
                f"Vista actualizada - Farmacias: {stats.total_pharmacies}, "
                f"Laboratorios: {stats.total_labs}, "
                f"Filas: {stats.total_rows}, "
                f"Ventas totales: €{stats.total_sales_amount:,.2f}, "
                f"Última actualización: {stats.last_refresh}"
            )

        except Exception as e:
            logger.error(f"Error actualizando vista materializada pharmacy_lab_sales_summary: {str(e)}")
            raise RuntimeError(f"Fallo al actualizar análisis de partners: {str(e)}")

    @staticmethod
    def get_database_size(db) -> dict:
        """
        Obtiene el tamaño actual de la base de datos.

        Args:
            db: Sesión de base de datos (síncrona)

        Returns:
            Diccionario con información de tamaño
        """
        try:
            # Obtener tamaño de la base de datos
            result = db.execute(
                text("""
                    SELECT
                        pg_database_size(current_database()) as db_size,
                        pg_size_pretty(pg_database_size(current_database())) as db_size_pretty
                """)
            )
            db_info = result.fetchone()

            # Obtener top 10 tablas por tamaño (incluye row_count para frontend)
            tables_result = db.execute(
                text("""
                    SELECT
                        t.schemaname,
                        t.tablename,
                        pg_size_pretty(pg_total_relation_size(t.schemaname||'.'||t.tablename)) as size,
                        pg_total_relation_size(t.schemaname||'.'||t.tablename) as size_bytes,
                        COALESCE(s.n_live_tup, 0) as row_count
                    FROM pg_tables t
                    LEFT JOIN pg_stat_user_tables s
                        ON t.schemaname = s.schemaname AND t.tablename = s.relname
                    WHERE t.schemaname NOT IN ('pg_catalog', 'information_schema')
                    ORDER BY pg_total_relation_size(t.schemaname||'.'||t.tablename) DESC
                    LIMIT 10
                """)
            )
            rows = tables_result.fetchall()
            top_tables = [
                {
                    "schema": row.schemaname,
                    "table": row.tablename,
                    "size": row.size,
                    "size_bytes": row.size_bytes,
                    "row_count": row.row_count
                }
                for row in rows
            ]

            return {
                "database_size": db_info.db_size,
                "database_size_pretty": db_info.db_size_pretty,
                "top_tables": top_tables
            }

        except Exception as e:
            logger.error(f"Error obteniendo tamaño de base de datos: {str(e)}")
            raise
