"""Customer Order Service - Modulo 1: Cazador de Encargos.

This service manages the tracking of customer orders (encargos) from the ERP.
It syncs pending orders from the ERP adapter and maintains local state for
pharmacist actions (contacted, marked for return, etc.).

Architecture:
    ERP (READ-ONLY) --> Adapter --> CustomerOrderService --> SQLite (Tracking)
                                          |
                                          v
                                    Dashboard Widget

Key Principles:
    - ERP is source of truth for raw data (READ-ONLY, REGLA #20)
    - Local SQLite tracks pharmacist actions (state machine)
    - Reconciliation detects orders that disappeared from ERP
"""

import logging
from datetime import datetime, timedelta
from typing import Optional

from sqlalchemy.orm import Session

from ..erp_adapters.farmanager.adapter import FarmanagerAdapter
from ..erp_adapters.models import UnifiedCustomerOrder
from ..models.tracking import CustomerOrderTracking

logger = logging.getLogger(__name__)


class CustomerOrderService:
    """Service for managing customer order tracking.

    Responsibilities:
        1. Sync pending orders from ERP to local tracking
        2. Update tracking state (contacted, to_return, resolved)
        3. Reconcile orders that disappeared from ERP
        4. Provide filtered views for dashboard widget

    Example:
        service = CustomerOrderService(session, erp_adapter)

        # Sync from ERP
        stats = service.sync_from_erp()
        print(f"Synced {stats['new']} new, {stats['resolved']} resolved")

        # Get pending for dashboard
        orders = service.get_pending_orders(severity_filter=["critical", "urgent"])

        # Mark as contacted
        service.mark_as_contacted("ENC-17399-45123", notes="Llamado, recoge manana")
    """

    def __init__(
        self,
        session: Session,
        erp_adapter: Optional[FarmanagerAdapter] = None,
    ):
        """Initialize service.

        Args:
            session: SQLAlchemy session for local tracking database
            erp_adapter: Farmanager adapter (optional, can be set later)
        """
        self._session = session
        self._erp_adapter = erp_adapter

    def set_erp_adapter(self, adapter: FarmanagerAdapter) -> None:
        """Set or update the ERP adapter."""
        self._erp_adapter = adapter

    # ==========================================================================
    # Sync Operations
    # ==========================================================================

    def sync_from_erp(self) -> dict:
        """Sync pending orders from ERP to local tracking.

        This is the main sync operation that should run periodically (e.g., every 15 min).

        Process:
            1. Get all pending orders from ERP
            2. For each order:
               - If new: Create tracking record with status NEW
               - If exists: Update cached data, update last_seen_in_erp
            3. Reconcile: Orders not seen become RESOLVED

        Returns:
            Dict with sync statistics:
            - new: Number of new orders detected
            - updated: Number of existing orders updated
            - resolved: Number of orders auto-resolved (disappeared from ERP)
            - total_pending: Current count of pending orders
            - total_value: Total PVP value of pending orders
        """
        if self._erp_adapter is None:
            raise RuntimeError("ERP adapter not configured. Call set_erp_adapter first.")

        stats = {
            "new": 0,
            "updated": 0,
            "resolved": 0,
            "total_pending": 0,
            "total_value": 0.0,
            "sync_timestamp": datetime.utcnow().isoformat(),
        }

        # Track which orders we see in this sync
        seen_erp_ids: set[str] = set()

        # Get pending orders from ERP
        with self._erp_adapter:
            for erp_order in self._erp_adapter.get_pending_customer_orders():
                seen_erp_ids.add(erp_order.erp_order_id)

                # Check if we already track this order
                existing = self._session.get(CustomerOrderTracking, erp_order.erp_order_id)

                if existing is None:
                    # New order detected
                    tracking = CustomerOrderTracking(
                        erp_order_id=erp_order.erp_order_id,
                        cached_pvl=float(erp_order.pvp),
                        cached_units=erp_order.units,
                        customer_name=erp_order.customer_name,
                        customer_phone=erp_order.customer_phone,
                        product_description=erp_order.product_name,
                        product_cn=erp_order.product_code,
                        status="NEW",
                        first_detected_at=erp_order.created_at,
                        last_seen_in_erp=datetime.utcnow(),
                    )
                    self._session.add(tracking)
                    stats["new"] += 1
                    logger.info(f"New customer order detected: {erp_order.erp_order_id}")
                else:
                    # Update existing order with fresh data
                    existing.update_from_erp(
                        pvl=float(erp_order.pvp),
                        units=erp_order.units,
                        customer_name=erp_order.customer_name,
                        customer_phone=erp_order.customer_phone,
                        product_description=erp_order.product_name,
                        product_cn=erp_order.product_code,
                    )
                    stats["updated"] += 1

        # Reconcile: Mark as RESOLVED orders that disappeared from ERP
        stats["resolved"] = self._reconcile_missing_orders(seen_erp_ids)

        # Commit all changes
        self._session.commit()

        # Calculate totals
        pending_orders = self.get_pending_orders()
        stats["total_pending"] = len(pending_orders)
        stats["total_value"] = sum(o.economic_value for o in pending_orders)

        logger.info(
            f"Sync completed: {stats['new']} new, {stats['updated']} updated, "
            f"{stats['resolved']} resolved. Total pending: {stats['total_pending']}"
        )

        return stats

    def _reconcile_missing_orders(self, seen_erp_ids: set[str]) -> int:
        """Mark orders as RESOLVED if they disappeared from ERP.

        An order disappears when the customer picks it up or it's returned.
        In either case, ges332.iestado changes from 0 to 1 or 2.

        Args:
            seen_erp_ids: Set of order IDs seen in current ERP sync

        Returns:
            Number of orders marked as resolved
        """
        # Find active orders not seen in this sync
        active_orders = (
            self._session.query(CustomerOrderTracking)
            .filter(CustomerOrderTracking.status.notin_(["RESOLVED"]))
            .all()
        )

        resolved_count = 0
        cutoff = datetime.utcnow() - timedelta(hours=1)  # Grace period

        for order in active_orders:
            if order.erp_order_id not in seen_erp_ids:
                # Check if it was seen recently (grace period)
                if order.last_seen_in_erp and order.last_seen_in_erp < cutoff:
                    order.mark_as_resolved()
                    resolved_count += 1
                    logger.info(
                        f"Order {order.erp_order_id} auto-resolved "
                        f"(last seen: {order.last_seen_in_erp})"
                    )

        return resolved_count

    # ==========================================================================
    # State Transitions
    # ==========================================================================

    def mark_as_contacted(
        self,
        erp_order_id: str,
        notes: Optional[str] = None,
    ) -> CustomerOrderTracking:
        """Mark order as contacted (pharmacist called the customer).

        Args:
            erp_order_id: The ERP order ID (e.g., "ENC-17399-45123")
            notes: Optional notes about the call

        Returns:
            Updated tracking record

        Raises:
            ValueError: If order not found
        """
        order = self._session.get(CustomerOrderTracking, erp_order_id)
        if order is None:
            raise ValueError(f"Order not found: {erp_order_id}")

        order.mark_as_contacted(notes)
        self._session.commit()

        logger.info(f"Order {erp_order_id} marked as CONTACTED")
        return order

    def mark_for_return(
        self,
        erp_order_id: str,
        notes: Optional[str] = None,
    ) -> CustomerOrderTracking:
        """Mark order for return to supplier.

        Args:
            erp_order_id: The ERP order ID
            notes: Optional notes about the return decision

        Returns:
            Updated tracking record

        Raises:
            ValueError: If order not found
        """
        order = self._session.get(CustomerOrderTracking, erp_order_id)
        if order is None:
            raise ValueError(f"Order not found: {erp_order_id}")

        order.mark_for_return(notes)
        self._session.commit()

        logger.info(f"Order {erp_order_id} marked for RETURN")
        return order

    def mark_as_resolved(self, erp_order_id: str) -> CustomerOrderTracking:
        """Manually mark order as resolved.

        Use this when the order was picked up but not detected automatically.

        Args:
            erp_order_id: The ERP order ID

        Returns:
            Updated tracking record

        Raises:
            ValueError: If order not found
        """
        order = self._session.get(CustomerOrderTracking, erp_order_id)
        if order is None:
            raise ValueError(f"Order not found: {erp_order_id}")

        order.mark_as_resolved()
        self._session.commit()

        logger.info(f"Order {erp_order_id} manually marked as RESOLVED")
        return order

    def add_notes(self, erp_order_id: str, notes: str) -> CustomerOrderTracking:
        """Add or update notes for an order.

        Args:
            erp_order_id: The ERP order ID
            notes: Notes to add

        Returns:
            Updated tracking record

        Raises:
            ValueError: If order not found
        """
        order = self._session.get(CustomerOrderTracking, erp_order_id)
        if order is None:
            raise ValueError(f"Order not found: {erp_order_id}")

        order.notes = notes
        self._session.commit()

        return order

    # ==========================================================================
    # Query Operations
    # ==========================================================================

    def get_pending_orders(
        self,
        severity_filter: Optional[list[str]] = None,
        exclude_spam: bool = True,
        limit: int = 100,
    ) -> list[CustomerOrderTracking]:
        """Get pending orders for dashboard display.

        Args:
            severity_filter: Optional list of severities to include
                            (e.g., ["critical", "urgent"])
            exclude_spam: If True, exclude orders contacted recently
            limit: Maximum orders to return

        Returns:
            List of tracking records, sorted by severity (critical first)
        """
        orders = (
            self._session.query(CustomerOrderTracking)
            .filter(CustomerOrderTracking.status.notin_(["RESOLVED"]))
            .all()
        )

        # Filter by severity
        if severity_filter:
            orders = [o for o in orders if o.severity in severity_filter]

        # Exclude spam (recently contacted)
        if exclude_spam:
            orders = [o for o in orders if not o.is_spam]

        # Sort by severity priority (critical > urgent > warning > info)
        severity_order = {"critical": 0, "urgent": 1, "warning": 2, "info": 3, "success": 4}
        orders.sort(key=lambda o: (severity_order.get(o.severity, 99), -o.days_pending))

        return orders[:limit]

    def get_order(self, erp_order_id: str) -> Optional[CustomerOrderTracking]:
        """Get a specific order by ID.

        Args:
            erp_order_id: The ERP order ID

        Returns:
            Tracking record or None
        """
        return self._session.get(CustomerOrderTracking, erp_order_id)

    def get_summary(self) -> dict:
        """Get summary statistics for dashboard widget.

        Returns:
            Dict with:
            - total_pending: Number of pending orders
            - total_value: Total PVP value
            - by_severity: Breakdown by severity level
            - by_status: Breakdown by tracking status
        """
        orders = (
            self._session.query(CustomerOrderTracking)
            .filter(CustomerOrderTracking.status.notin_(["RESOLVED"]))
            .all()
        )

        by_severity = {
            "critical": {"count": 0, "value": 0.0},
            "urgent": {"count": 0, "value": 0.0},
            "warning": {"count": 0, "value": 0.0},
            "info": {"count": 0, "value": 0.0},
        }
        by_status = {
            "NEW": 0,
            "ALERTED": 0,
            "CONTACTED": 0,
            "TO_RETURN": 0,
        }

        total_value = 0.0
        for order in orders:
            value = order.economic_value
            total_value += value

            sev = order.severity
            if sev in by_severity:
                by_severity[sev]["count"] += 1
                by_severity[sev]["value"] += value

            if order.status in by_status:
                by_status[order.status] += 1

        return {
            "total_pending": len(orders),
            "total_value": total_value,
            "by_severity": by_severity,
            "by_status": by_status,
        }

    def get_orders_to_return(self) -> list[CustomerOrderTracking]:
        """Get orders marked for return (for export/report).

        Returns:
            List of orders with status TO_RETURN
        """
        return (
            self._session.query(CustomerOrderTracking)
            .filter(CustomerOrderTracking.status == "TO_RETURN")
            .all()
        )

    def get_recently_resolved(
        self,
        days: int = 7,
        limit: int = 50,
    ) -> list[CustomerOrderTracking]:
        """Get recently resolved orders for history view.

        Args:
            days: Number of days to look back
            limit: Maximum orders to return

        Returns:
            List of resolved orders
        """
        cutoff = datetime.utcnow() - timedelta(days=days)
        return (
            self._session.query(CustomerOrderTracking)
            .filter(CustomerOrderTracking.status == "RESOLVED")
            .filter(CustomerOrderTracking.resolved_at >= cutoff)
            .order_by(CustomerOrderTracking.resolved_at.desc())
            .limit(limit)
            .all()
        )
