# backend/app/services/curated_groups_service.py
"""
Servicio para gestión de Grupos Intercambiables Curados.

Issue #521: Admin UI para gestión de grupos curados.
ADR-004: Simplificación del Sistema de Clasificación
"""

import re
from typing import List, Optional, Tuple
from uuid import UUID

import structlog
from sqlalchemy import func, or_
from sqlalchemy.orm import Session

from app.models.interchangeable_curated_group import (
    InterchangeableCuratedGroup,
    InterchangeableGroupMember,
)
from app.schemas.curated_groups import (
    CuratedGroupCreate,
    CuratedGroupFilter,
    CuratedGroupUpdate,
    GroupMemberCreate,
)

logger = structlog.get_logger(__name__)


class CuratedGroupsService:
    """Servicio para gestión de grupos intercambiables curados."""

    def __init__(self, db: Session):
        self.db = db

    # === GROUP CRUD ===

    def list_groups(
        self,
        filter_params: CuratedGroupFilter,
    ) -> Tuple[List[InterchangeableCuratedGroup], int]:
        """
        Lista grupos curados con filtros y paginación.

        Returns:
            Tuple de (lista de grupos, total count)
        """
        query = self.db.query(InterchangeableCuratedGroup)

        # Apply filters
        if filter_params.necesidad_l1:
            query = query.filter(
                InterchangeableCuratedGroup.necesidad_l1 == filter_params.necesidad_l1
            )
        if filter_params.is_active is not None:
            query = query.filter(
                InterchangeableCuratedGroup.is_active == filter_params.is_active
            )
        if filter_params.source:
            query = query.filter(
                InterchangeableCuratedGroup.source == filter_params.source
            )
        if filter_params.search:
            search_term = f"%{filter_params.search}%"
            query = query.filter(
                or_(
                    InterchangeableCuratedGroup.group_name.ilike(search_term),
                    InterchangeableCuratedGroup.description.ilike(search_term),
                )
            )

        # Get total count before pagination
        total = query.count()

        # Apply ordering (whitelist to prevent injection)
        ALLOWED_ORDER_FIELDS = {
            "total_sales_amount": InterchangeableCuratedGroup.total_sales_amount,
            "group_name": InterchangeableCuratedGroup.group_name,
            "product_count": InterchangeableCuratedGroup.product_count,
            "brand_count": InterchangeableCuratedGroup.brand_count,
            "created_at": InterchangeableCuratedGroup.created_at,
            "updated_at": InterchangeableCuratedGroup.updated_at,
            "necesidad_l1": InterchangeableCuratedGroup.necesidad_l1,
        }
        order_column = ALLOWED_ORDER_FIELDS.get(
            filter_params.order_by,
            InterchangeableCuratedGroup.total_sales_amount,
        )
        if filter_params.order_desc:
            query = query.order_by(order_column.desc().nullslast())
        else:
            query = query.order_by(order_column.asc().nullsfirst())

        # Apply pagination
        offset = (filter_params.page - 1) * filter_params.page_size
        query = query.offset(offset).limit(filter_params.page_size)

        groups = query.all()

        logger.info(
            "curated_groups.list",
            total=total,
            page=filter_params.page,
            filters={
                "l1": filter_params.necesidad_l1,
                "active": filter_params.is_active,
                "source": filter_params.source,
            },
        )

        return groups, total

    def get_group(self, group_id: UUID) -> Optional[InterchangeableCuratedGroup]:
        """Obtiene un grupo por ID."""
        return (
            self.db.query(InterchangeableCuratedGroup)
            .filter(InterchangeableCuratedGroup.id == group_id)
            .first()
        )

    def get_group_by_slug(self, slug: str) -> Optional[InterchangeableCuratedGroup]:
        """Obtiene un grupo por slug."""
        return (
            self.db.query(InterchangeableCuratedGroup)
            .filter(InterchangeableCuratedGroup.group_slug == slug)
            .first()
        )

    def create_group(
        self,
        data: CuratedGroupCreate,
        created_by: str,
    ) -> InterchangeableCuratedGroup:
        """
        Crea un nuevo grupo curado.

        Args:
            data: Datos del grupo
            created_by: Email del usuario que crea

        Returns:
            Grupo creado
        """
        # Generate slug if not provided
        slug = data.group_slug or self._generate_slug(data.group_name)

        # Ensure slug is unique
        existing = self.get_group_by_slug(slug)
        if existing:
            # Append number to make unique
            base_slug = slug
            counter = 1
            while existing:
                slug = f"{base_slug}-{counter}"
                existing = self.get_group_by_slug(slug)
                counter += 1

        group = InterchangeableCuratedGroup(
            group_name=data.group_name,
            group_slug=slug,
            description=data.description,
            necesidad_l1=data.necesidad_l1,
            subcategory_l2=data.subcategory_l2,
            source="manual_curated",
            created_by=created_by,
        )

        self.db.add(group)
        self.db.commit()
        self.db.refresh(group)

        logger.info(
            "curated_groups.created",
            group_id=str(group.id),
            slug=slug,
            created_by=created_by,
        )

        return group

    def update_group(
        self,
        group_id: UUID,
        data: CuratedGroupUpdate,
    ) -> Optional[InterchangeableCuratedGroup]:
        """
        Actualiza un grupo existente.

        Args:
            group_id: ID del grupo
            data: Datos a actualizar

        Returns:
            Grupo actualizado o None si no existe
        """
        group = self.get_group(group_id)
        if not group:
            return None

        # Update only provided fields
        update_data = data.model_dump(exclude_unset=True)
        for field, value in update_data.items():
            setattr(group, field, value)

        self.db.commit()
        self.db.refresh(group)

        logger.info(
            "curated_groups.updated",
            group_id=str(group_id),
            fields=list(update_data.keys()),
        )

        return group

    def delete_group(self, group_id: UUID) -> bool:
        """
        Elimina un grupo (y sus miembros por CASCADE).

        Args:
            group_id: ID del grupo

        Returns:
            True si se eliminó, False si no existía
        """
        group = self.get_group(group_id)
        if not group:
            return False

        self.db.delete(group)
        self.db.commit()

        logger.info("curated_groups.deleted", group_id=str(group_id))

        return True

    # === MEMBERS CRUD ===

    def list_members(
        self,
        group_id: UUID,
    ) -> Tuple[List[InterchangeableGroupMember], int]:
        """Lista miembros de un grupo."""
        query = (
            self.db.query(InterchangeableGroupMember)
            .filter(InterchangeableGroupMember.group_id == group_id)
            .order_by(InterchangeableGroupMember.product_name)
        )

        members = query.all()
        total = len(members)

        return members, total

    def add_member(
        self,
        group_id: UUID,
        data: GroupMemberCreate,
        added_by: str,
    ) -> Optional[InterchangeableGroupMember]:
        """
        Añade un miembro a un grupo.

        Args:
            group_id: ID del grupo
            data: Datos del miembro
            added_by: Email del usuario

        Returns:
            Miembro creado o None si el grupo no existe
        """
        group = self.get_group(group_id)
        if not group:
            return None

        # Check if member already exists (by EAN or CN) - single query to avoid race condition
        if data.ean13 or data.codigo_nacional:
            conditions = []
            if data.ean13:
                conditions.append(InterchangeableGroupMember.ean13 == data.ean13)
            if data.codigo_nacional:
                conditions.append(InterchangeableGroupMember.codigo_nacional == data.codigo_nacional)

            existing = (
                self.db.query(InterchangeableGroupMember)
                .filter(
                    InterchangeableGroupMember.group_id == group_id,
                    or_(*conditions),
                )
                .first()
            )
            if existing:
                logger.warning(
                    "curated_groups.member_exists",
                    group_id=str(group_id),
                    ean13=data.ean13,
                    codigo_nacional=data.codigo_nacional,
                )
                return existing

        member = InterchangeableGroupMember(
            group_id=group_id,
            ean13=data.ean13,
            codigo_nacional=data.codigo_nacional,
            product_name=data.product_name,
            detected_brand=data.detected_brand,
            added_by=added_by,
        )

        self.db.add(member)
        self.db.flush()  # Ensure member is visible to subsequent queries

        # Update group stats
        group.product_count = (group.product_count or 0) + 1
        # Recalculate brand count (always, to handle brand updates)
        brand_count = (
            self.db.query(func.count(func.distinct(InterchangeableGroupMember.detected_brand)))
            .filter(
                InterchangeableGroupMember.group_id == group_id,
                InterchangeableGroupMember.detected_brand.isnot(None),
            )
            .scalar()
        )
        group.brand_count = brand_count or 0

        self.db.commit()
        self.db.refresh(member)

        logger.info(
            "curated_groups.member_added",
            group_id=str(group_id),
            member_id=str(member.id),
            added_by=added_by,
        )

        return member

    def remove_member(self, group_id: UUID, member_id: UUID) -> bool:
        """
        Elimina un miembro de un grupo.

        Args:
            group_id: ID del grupo
            member_id: ID del miembro

        Returns:
            True si se eliminó
        """
        member = (
            self.db.query(InterchangeableGroupMember)
            .filter(
                InterchangeableGroupMember.id == member_id,
                InterchangeableGroupMember.group_id == group_id,
            )
            .first()
        )

        if not member:
            return False

        group = self.get_group(group_id)

        self.db.delete(member)

        # Update group stats
        if group:
            group.product_count = max(0, (group.product_count or 0) - 1)
            # Recalculate brand count
            brand_count = (
                self.db.query(func.count(func.distinct(InterchangeableGroupMember.detected_brand)))
                .filter(InterchangeableGroupMember.group_id == group_id)
                .scalar()
            )
            group.brand_count = brand_count or 0

        self.db.commit()

        logger.info(
            "curated_groups.member_removed",
            group_id=str(group_id),
            member_id=str(member_id),
        )

        return True

    # === STATS ===

    def get_stats(self) -> dict:
        """Obtiene estadísticas generales de grupos curados."""
        total = self.db.query(InterchangeableCuratedGroup).count()
        active = (
            self.db.query(InterchangeableCuratedGroup)
            .filter(InterchangeableCuratedGroup.is_active == True)
            .count()
        )
        legacy = (
            self.db.query(InterchangeableCuratedGroup)
            .filter(InterchangeableCuratedGroup.source == "legacy_clustering")
            .count()
        )
        manual = (
            self.db.query(InterchangeableCuratedGroup)
            .filter(InterchangeableCuratedGroup.source == "manual_curated")
            .count()
        )

        # Total products and brands
        products = self.db.query(InterchangeableGroupMember).count()
        brands = (
            self.db.query(func.count(func.distinct(InterchangeableGroupMember.detected_brand)))
            .scalar()
        ) or 0

        # Total sales
        sales = (
            self.db.query(func.sum(InterchangeableCuratedGroup.total_sales_amount))
            .scalar()
        ) or 0

        return {
            "total_groups": total,
            "active_groups": active,
            "legacy_groups": legacy,
            "manual_groups": manual,
            "total_products": products,
            "total_brands": brands,
            "total_sales": float(sales),
        }

    # === HELPERS ===

    def _generate_slug(self, name: str) -> str:
        """Genera un slug URL-friendly desde un nombre."""
        # Convert to lowercase
        slug = name.lower()
        # Replace accented characters
        replacements = {
            "á": "a", "é": "e", "í": "i", "ó": "o", "ú": "u",
            "ñ": "n", "ü": "u",
        }
        for old, new in replacements.items():
            slug = slug.replace(old, new)
        # Replace non-alphanumeric with hyphen
        slug = re.sub(r"[^a-z0-9]+", "-", slug)
        # Remove leading/trailing hyphens
        slug = slug.strip("-")
        # Limit length
        return slug[:100]
