"""
Servicio de Alertas para Clasificador NECESIDAD (Issue #458 - M6/F7)

Gestiona alertas basadas en métricas del clasificador:
- Genera alertas cuando se cruzan umbrales
- Almacena historial de alertas
- Proporciona API para consultar alertas activas
- Soporta auto-resolución de alertas

Tipos de Alertas:
- OUTLIER_RATE: Tasa de outliers elevada
- LOW_ENTROPY: Distribución sesgada
- MODEL_DECAY: Degradación del modelo
- VALIDATION_BACKLOG: Cola de validación acumulada
- LOW_CONFIDENCE: Confianza promedio baja
"""

from datetime import datetime, timedelta
from enum import Enum
from typing import Any, Dict, List, Optional
from uuid import UUID, uuid4

import structlog
from sqlalchemy import and_, desc, func
from sqlalchemy.orm import Session

from app.models.system_health import SystemAlert
from app.services.classifier_monitoring_service import (
    ALERT_THRESHOLDS,
    ClassifierMonitoringService,
)
from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)


# =============================================================================
# ENUMS
# =============================================================================

class AlertSeverity(str, Enum):
    """Severidad de alertas."""
    INFO = "info"
    WARNING = "warning"
    CRITICAL = "critical"


class AlertStatus(str, Enum):
    """Estado de alertas."""
    ACTIVE = "active"
    ACKNOWLEDGED = "acknowledged"
    RESOLVED = "resolved"
    AUTO_RESOLVED = "auto_resolved"


class AlertType(str, Enum):
    """Tipos de alertas del clasificador."""
    OUTLIER_RATE = "outlier_rate"
    LOW_ENTROPY = "low_entropy"
    MODEL_DECAY = "model_decay"
    VALIDATION_BACKLOG = "validation_backlog"
    LOW_CONFIDENCE = "low_confidence"


# =============================================================================
# ALERT DEFINITIONS
# =============================================================================

ALERT_DEFINITIONS = {
    AlertType.OUTLIER_RATE: {
        "title": "Tasa de Outliers Elevada",
        "description": "El porcentaje de productos fuera del cluster supera el umbral",
        "metric": "outlier_rate",
        "thresholds": {
            AlertSeverity.WARNING: ALERT_THRESHOLDS["outlier_rate_warning"],
            AlertSeverity.CRITICAL: ALERT_THRESHOLDS["outlier_rate_critical"],
        },
        "direction": "above",  # Alerta cuando está por encima
    },
    AlertType.LOW_ENTROPY: {
        "title": "Distribución de Clusters Sesgada",
        "description": "La distribución de productos entre categorías es muy desigual",
        "metric": "cluster_entropy",
        "thresholds": {
            AlertSeverity.WARNING: ALERT_THRESHOLDS["entropy_warning"],
        },
        "direction": "below",  # Alerta cuando está por debajo
    },
    AlertType.LOW_CONFIDENCE: {
        "title": "Confianza del Modelo Baja",
        "description": "La confianza promedio del clasificador ML ha bajado",
        "metric": "avg_ml_confidence",
        "thresholds": {
            AlertSeverity.WARNING: ALERT_THRESHOLDS["confidence_warning"],
        },
        "direction": "below",
    },
    AlertType.VALIDATION_BACKLOG: {
        "title": "Backlog de Validación",
        "description": "Hay muchos productos pendientes de validación humana",
        "metric": "validation_pending",
        "thresholds": {
            AlertSeverity.WARNING: ALERT_THRESHOLDS["pending_warning"],
            AlertSeverity.CRITICAL: ALERT_THRESHOLDS["pending_critical"],
        },
        "direction": "above",
    },
    AlertType.MODEL_DECAY: {
        "title": "Degradación del Modelo",
        "description": "La confianza del modelo ha disminuido significativamente",
        "metric": "model_decay_7d",
        "thresholds": {
            AlertSeverity.WARNING: -ALERT_THRESHOLDS["decay_warning"],  # Negativo
        },
        "direction": "below",  # Decay es negativo cuando empeora
    },
}


# =============================================================================
# SERVICE
# =============================================================================

class ClusteringAlertService:
    """
    Servicio de gestión de alertas para el clasificador NECESIDAD.

    Responsabilidades:
    - Evaluar métricas y generar alertas
    - Almacenar y consultar alertas
    - Gestionar ciclo de vida (acknowledge, resolve)
    - Auto-resolver alertas cuando métricas vuelven a normal
    """

    def __init__(self, db: Session):
        self.db = db
        self.monitoring_service = ClassifierMonitoringService(db)

    # =========================================================================
    # ALERT EVALUATION
    # =========================================================================

    def evaluate_and_create_alerts(
        self,
        snapshot: Optional[Dict] = None,
    ) -> List[Dict]:
        """
        Evaluar métricas actuales y crear alertas si se cruzan umbrales.

        Args:
            snapshot: Snapshot de métricas (si None, toma uno nuevo)

        Returns:
            Lista de alertas creadas o actualizadas
        """
        if snapshot is None:
            snapshot = self.monitoring_service.take_snapshot()

        created_alerts = []

        for alert_type, definition in ALERT_DEFINITIONS.items():
            metric_name = definition["metric"]
            metric_value = snapshot.get(metric_name)

            if metric_value is None:
                continue

            # Determinar severidad
            severity = self._determine_severity(
                value=metric_value,
                thresholds=definition["thresholds"],
                direction=definition["direction"],
            )

            if severity:
                # Verificar si ya existe una alerta activa del mismo tipo
                existing = self._get_active_alert(alert_type)

                if existing:
                    # Actualizar severidad si cambió
                    if existing.get("severity") != severity.value:
                        self._update_alert_severity(existing["id"], severity)
                        logger.info(
                            "clustering_alert.severity_updated",
                            alert_type=alert_type.value,
                            old_severity=existing["severity"],
                            new_severity=severity.value,
                        )
                else:
                    # Crear nueva alerta
                    alert = self._create_alert(
                        alert_type=alert_type,
                        severity=severity,
                        metric_value=metric_value,
                        definition=definition,
                    )
                    created_alerts.append(alert)
            else:
                # Métrica volvió a normal - auto-resolver si hay alerta activa
                self._auto_resolve_alert(alert_type)

        logger.info(
            "clustering_alert.evaluation_complete",
            new_alerts=len(created_alerts),
            active_alerts=self.get_active_alerts_count(),
        )

        return created_alerts

    def _determine_severity(
        self,
        value: float,
        thresholds: Dict[AlertSeverity, float],
        direction: str,
    ) -> Optional[AlertSeverity]:
        """
        Determinar severidad basada en valor y umbrales.
        """
        if direction == "above":
            # Ordenar de mayor a menor severidad
            for severity in [AlertSeverity.CRITICAL, AlertSeverity.WARNING, AlertSeverity.INFO]:
                if severity in thresholds and value > thresholds[severity]:
                    return severity
        else:  # direction == "below"
            for severity in [AlertSeverity.CRITICAL, AlertSeverity.WARNING, AlertSeverity.INFO]:
                if severity in thresholds and value < thresholds[severity]:
                    return severity

        return None

    # =========================================================================
    # ALERT CRUD
    # =========================================================================

    def _create_alert(
        self,
        alert_type: AlertType,
        severity: AlertSeverity,
        metric_value: float,
        definition: Dict,
    ) -> Dict:
        """
        Crear y guardar una nueva alerta.

        Usa el modelo SystemAlert existente con campos:
        - is_active, is_acknowledged (booleanos en lugar de status enum)
        - context (JSON para datos de métricas)
        - first_occurred, last_occurred (timestamps)
        """
        now = utc_now()

        # Preparar mensaje
        threshold = definition["thresholds"].get(severity, 0)
        message = f"{definition['description']}. Valor actual: {metric_value:.2f}, umbral: {threshold:.2f}"

        # Guardar métricas en context JSON
        context = {
            "metric_name": definition["metric"],
            "metric_value": metric_value,
            "threshold_value": threshold,
            "direction": definition["direction"],
        }

        # Crear en SystemAlert (modelo existente)
        alert = SystemAlert(
            alert_type=alert_type.value,
            severity=severity.value,
            title=definition["title"],
            message=message,
            context=context,
            is_active=True,
            is_acknowledged=False,
            first_occurred=now,
            last_occurred=now,
            occurrence_count=1,
        )

        self.db.add(alert)
        self.db.commit()

        logger.info(
            "clustering_alert.created",
            alert_type=alert_type.value,
            severity=severity.value,
            metric_value=metric_value,
        )

        return {
            "id": str(alert.id),
            "type": alert_type.value,
            "severity": severity.value,
            "title": definition["title"],
            "message": message,
            "status": AlertStatus.ACTIVE.value,
            "created_at": now.isoformat(),
        }

    def _get_active_alert(self, alert_type: AlertType) -> Optional[Dict]:
        """
        Obtener alerta activa de un tipo específico.
        """
        alert = self.db.query(SystemAlert).filter(
            and_(
                SystemAlert.alert_type == alert_type.value,
                SystemAlert.is_active == True,  # noqa: E712
            )
        ).first()

        if not alert:
            return None

        # Determinar status basado en is_acknowledged
        if alert.is_acknowledged:
            status = AlertStatus.ACKNOWLEDGED.value
        else:
            status = AlertStatus.ACTIVE.value

        return {
            "id": str(alert.id),
            "type": alert.alert_type,
            "severity": alert.severity,
            "status": status,
        }

    def _update_alert_severity(self, alert_id: str, new_severity: AlertSeverity):
        """
        Actualizar severidad de una alerta existente.
        """
        from uuid import UUID as UUIDType
        alert = self.db.query(SystemAlert).filter(
            SystemAlert.id == UUIDType(alert_id)
        ).first()

        if alert:
            alert.severity = new_severity.value
            alert.last_occurred = utc_now()
            self.db.commit()

    def _auto_resolve_alert(self, alert_type: AlertType):
        """
        Auto-resolver alerta cuando la métrica vuelve a normal.
        """
        alert = self.db.query(SystemAlert).filter(
            and_(
                SystemAlert.alert_type == alert_type.value,
                SystemAlert.is_active == True,  # noqa: E712
            )
        ).first()

        if alert:
            alert.is_active = False
            # Agregar info de auto-resolución en context
            context = alert.context or {}
            context["auto_resolved"] = True
            context["resolved_at"] = utc_now().isoformat()
            alert.context = context
            self.db.commit()

            logger.info(
                "clustering_alert.auto_resolved",
                alert_type=alert_type.value,
            )

    # =========================================================================
    # PUBLIC API
    # =========================================================================

    def get_active_alerts(self) -> List[Dict]:
        """
        Obtener todas las alertas activas.

        Returns:
            Lista de alertas activas ordenadas por severidad
        """
        alerts = self.db.query(SystemAlert).filter(
            SystemAlert.is_active == True  # noqa: E712
        ).order_by(
            # Ordenar por severidad: critical > warning > info
            desc(SystemAlert.severity == AlertSeverity.CRITICAL.value),
            desc(SystemAlert.severity == AlertSeverity.WARNING.value),
            desc(SystemAlert.first_occurred),
        ).all()

        return [
            {
                "id": str(a.id),
                "type": a.alert_type,
                "severity": a.severity,
                "title": a.title,
                "message": a.message,
                "status": AlertStatus.ACKNOWLEDGED.value if a.is_acknowledged else AlertStatus.ACTIVE.value,
                "created_at": a.first_occurred.isoformat() if a.first_occurred else None,
                "acknowledged_at": a.acknowledged_at.isoformat() if a.acknowledged_at else None,
            }
            for a in alerts
        ]

    def get_active_alerts_count(self) -> int:
        """
        Contar alertas activas.
        """
        return self.db.query(func.count(SystemAlert.id)).filter(
            SystemAlert.is_active == True  # noqa: E712
        ).scalar() or 0

    def acknowledge_alert(self, alert_id: str, user_id: Optional[UUID] = None) -> bool:
        """
        Marcar alerta como reconocida.

        Args:
            alert_id: ID de la alerta
            user_id: ID del usuario que reconoce (opcional)

        Returns:
            True si se reconoció correctamente
        """
        from uuid import UUID as UUIDType
        alert = self.db.query(SystemAlert).filter(
            and_(
                SystemAlert.id == UUIDType(alert_id),
                SystemAlert.is_active == True,  # noqa: E712
                SystemAlert.is_acknowledged == False,  # noqa: E712
            )
        ).first()

        if not alert:
            return False

        alert.is_acknowledged = True
        alert.acknowledged_at = utc_now()
        alert.acknowledged_by = str(user_id) if user_id else "system"
        self.db.commit()

        logger.info(
            "clustering_alert.acknowledged",
            alert_id=alert_id,
            user_id=str(user_id) if user_id else None,
        )

        return True

    def resolve_alert(self, alert_id: str, user_id: Optional[UUID] = None) -> bool:
        """
        Resolver alerta manualmente.

        Args:
            alert_id: ID de la alerta
            user_id: ID del usuario que resuelve (opcional)

        Returns:
            True si se resolvió correctamente
        """
        from uuid import UUID as UUIDType
        alert = self.db.query(SystemAlert).filter(
            and_(
                SystemAlert.id == UUIDType(alert_id),
                SystemAlert.is_active == True,  # noqa: E712
            )
        ).first()

        if not alert:
            return False

        alert.is_active = False
        # Guardar info de resolución en context
        context = alert.context or {}
        context["resolved_at"] = utc_now().isoformat()
        context["resolved_by"] = str(user_id) if user_id else "system"
        context["manual_resolution"] = True
        alert.context = context
        self.db.commit()

        logger.info(
            "clustering_alert.resolved",
            alert_id=alert_id,
            user_id=str(user_id) if user_id else None,
        )

        return True

    def get_alert_history(
        self,
        days: int = 30,
        include_resolved: bool = True,
    ) -> List[Dict]:
        """
        Obtener historial de alertas.

        Args:
            days: Días de historial
            include_resolved: Incluir alertas resueltas

        Returns:
            Lista de alertas históricas
        """
        cutoff = utc_now() - timedelta(days=days)

        query = self.db.query(SystemAlert).filter(
            SystemAlert.first_occurred >= cutoff
        )

        if not include_resolved:
            query = query.filter(
                SystemAlert.is_active == True  # noqa: E712
            )

        alerts = query.order_by(desc(SystemAlert.first_occurred)).all()

        return [
            {
                "id": str(a.id),
                "type": a.alert_type,
                "severity": a.severity,
                "title": a.title,
                "message": a.message,
                "status": self._get_alert_status(a),
                "created_at": a.first_occurred.isoformat() if a.first_occurred else None,
                "resolved_at": (a.context or {}).get("resolved_at"),
            }
            for a in alerts
        ]

    def _get_alert_status(self, alert: SystemAlert) -> str:
        """Determinar status de una alerta."""
        if not alert.is_active:
            context = alert.context or {}
            if context.get("auto_resolved"):
                return AlertStatus.AUTO_RESOLVED.value
            return AlertStatus.RESOLVED.value
        if alert.is_acknowledged:
            return AlertStatus.ACKNOWLEDGED.value
        return AlertStatus.ACTIVE.value


# =============================================================================
# MODULE-LEVEL SINGLETON FACTORY
# =============================================================================

def get_clustering_alert_service(db: Session) -> ClusteringAlertService:
    """Factory function para obtener instancia del servicio."""
    return ClusteringAlertService(db)
