"""
Servicio de Monitoreo para Clasificador NECESIDAD (Issue #458 - M6)

Monitorea la salud del clasificador de productos venta libre:
- Entropía de clusters (distribución uniforme vs sesgada)
- Tasa de outliers (productos lejos del centroide)
- Model decay (degradación de confianza ML)
- Queue de validación pendiente
- Historial de métricas para tendencias
- Precisión basada en correcciones humanas (Issue #465)
- Drift de distribución semanal (Issue #466)

KPIs Clave:
- cluster_entropy: Shannon entropy de distribución (alto = más uniforme)
- outlier_rate: % de productos con z-score > 2.5
- avg_ml_confidence: Confianza promedio del modelo
- validation_pending: Productos pendientes de validación humana
- model_decay_7d: Cambio en confianza últimos 7 días
- precision: % de predicciones correctas vs correcciones humanas (Issue #465)
- distribution_drift: Cambios significativos en distribución semanal (Issue #466)
"""

import math
from collections import defaultdict
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Tuple
from uuid import UUID

import structlog
from sqlalchemy import and_, desc, func
from sqlalchemy.orm import Session

from app.models.product_catalog_venta_libre import ProductCatalogVentaLibre
from app.models.product_correction import CorrectionType, ProductCorrection
from app.models.system_health import SystemHealthMetric
from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)


# =============================================================================
# CONSTANTS
# =============================================================================

# Umbrales de alerta
ALERT_THRESHOLDS = {
    "outlier_rate_warning": 0.10,   # 10% outliers = warning
    "outlier_rate_critical": 0.20,  # 20% outliers = critical
    "entropy_warning": 0.5,          # Baja entropía = distribución sesgada
    "confidence_warning": 0.75,      # Confianza < 75% = warning
    "decay_warning": 0.05,           # Decaimiento > 5% semanal = warning
    "pending_warning": 100,          # > 100 pendientes = warning
    "pending_critical": 500,         # > 500 pendientes = critical
    # Issue #465: Precision thresholds
    "precision_warning": 0.70,       # < 70% precision = warning
    "precision_critical": 0.50,      # < 50% precision = critical
    # Issue #466: Drift thresholds
    "drift_warning": 0.10,           # > 10% cambio en categoría = warning
    "drift_critical": 0.20,          # > 20% cambio = critical
}

# Categoría de métricas en SystemHealthMetric
METRIC_CATEGORY = "classifier_monitoring"


class ClassifierMonitoringService:
    """
    Servicio de monitoreo para el clasificador de productos NECESIDAD.

    Calcula y almacena KPIs del clasificador para detectar:
    - Distribución sesgada de categorías
    - Productos outlier que necesitan revisión
    - Degradación del modelo ML
    - Backlog de validación pendiente
    """

    def __init__(self, db: Session):
        self.db = db

    # =========================================================================
    # CORE KPI CALCULATIONS
    # =========================================================================

    def calculate_cluster_entropy(
        self,
        pharmacy_id: Optional[UUID] = None,
    ) -> float:
        """
        Calcular entropía de Shannon de la distribución de clusters.

        Una entropía alta (cercana a log2(N)) indica distribución uniforme.
        Una entropía baja indica sesgo hacia pocas categorías.

        Returns:
            float: Entropía normalizada (0-1), donde 1 = máxima uniformidad
        """
        query = self.db.query(
            ProductCatalogVentaLibre.ml_category,
            func.count(ProductCatalogVentaLibre.id).label("count")
        ).filter(
            ProductCatalogVentaLibre.ml_category.isnot(None)
        )

        if pharmacy_id:
            # Si hay pharmacy_id, filtrar (aunque ProductCatalogVentaLibre es global)
            pass

        category_counts = query.group_by(
            ProductCatalogVentaLibre.ml_category
        ).all()

        if not category_counts:
            return 0.0

        total = sum(count for _, count in category_counts)
        if total == 0:
            return 0.0

        # Calcular entropía de Shannon
        entropy = 0.0
        for _, count in category_counts:
            p = count / total
            if p > 0:
                entropy -= p * math.log2(p)

        # Normalizar por máxima entropía posible
        max_entropy = math.log2(len(category_counts))
        normalized_entropy = entropy / max_entropy if max_entropy > 0 else 0.0

        return round(normalized_entropy, 4)

    def calculate_outlier_rate(
        self,
        z_score_threshold: float = 2.5,
    ) -> Tuple[float, int, int]:
        """
        Calcular tasa de outliers (productos marcados como outlier o con baja confianza).

        Args:
            z_score_threshold: No usado - se mantiene por compatibilidad

        Returns:
            Tuple (rate, outlier_count, total_count)
        """
        # Total de productos activos
        total = self.db.query(func.count(ProductCatalogVentaLibre.id)).filter(
            ProductCatalogVentaLibre.is_active == True,  # noqa: E712
        ).scalar() or 0

        if total == 0:
            return 0.0, 0, 0

        # Contar outliers (marcados o con muy baja confianza < 0.5)
        outliers = self.db.query(func.count(ProductCatalogVentaLibre.id)).filter(
            ProductCatalogVentaLibre.is_active == True,  # noqa: E712
            ProductCatalogVentaLibre.is_outlier == True,  # noqa: E712
        ).scalar() or 0

        rate = outliers / total
        return round(rate, 4), outliers, total

    def calculate_avg_ml_confidence(self) -> float:
        """
        Calcular confianza promedio del modelo ML.

        Returns:
            float: Confianza promedio (0-1)
        """
        result = self.db.query(
            func.avg(ProductCatalogVentaLibre.ml_confidence)
        ).filter(
            ProductCatalogVentaLibre.ml_confidence.isnot(None)
        ).scalar()

        return round(result or 0.0, 4)

    def count_pending_validation(self) -> int:
        """
        Contar productos pendientes de validación humana.

        Returns:
            int: Número de productos no verificados
        """
        return self.db.query(func.count(ProductCatalogVentaLibre.id)).filter(
            ProductCatalogVentaLibre.human_verified == False  # noqa: E712
        ).scalar() or 0

    def calculate_coverage(self) -> Tuple[float, int, int]:
        """
        Calcular cobertura de clasificación (Issue #458).

        Coverage = productos con ml_category / total productos venta libre

        Returns:
            Tuple[float, int, int]: (coverage_percent, classified_count, total_count)
        """
        total_count = self.db.query(func.count(ProductCatalogVentaLibre.id)).scalar() or 0

        classified_count = self.db.query(func.count(ProductCatalogVentaLibre.id)).filter(
            ProductCatalogVentaLibre.ml_category.isnot(None)
        ).scalar() or 0

        if total_count == 0:
            return 0.0, 0, 0

        coverage = classified_count / total_count
        return round(coverage, 4), classified_count, total_count

    def calculate_model_decay(
        self,
        days: int = 7,
    ) -> Optional[float]:
        """
        Calcular decaimiento del modelo comparando métricas históricas.

        Args:
            days: Ventana de días para comparar

        Returns:
            float: Cambio porcentual en confianza (negativo = decay)
        """
        # Buscar métricas de confianza históricas
        cutoff = utc_now() - timedelta(days=days)

        old_metrics = self.db.query(SystemHealthMetric).filter(
            and_(
                SystemHealthMetric.category == METRIC_CATEGORY,
                SystemHealthMetric.metric_name == "avg_ml_confidence",
                SystemHealthMetric.measured_at < cutoff,
            )
        ).order_by(desc(SystemHealthMetric.measured_at)).first()

        if not old_metrics:
            return None

        current_confidence = self.calculate_avg_ml_confidence()
        old_confidence = old_metrics.metric_value

        if old_confidence == 0:
            return None

        decay = (current_confidence - old_confidence) / old_confidence
        return round(decay, 4)

    def calculate_precision(
        self,
        days: int = 30,
    ) -> Tuple[float, int, int]:
        """
        Calcular precisión del clasificador basada en correcciones humanas (Issue #465).

        Precision = (APPROVE count) / (APPROVE + CORRECT count)

        Solo cuenta APPROVE y CORRECT. OUTLIER y SKIP no se incluyen.

        Args:
            days: Ventana temporal en días (default 30 días rolling)

        Returns:
            Tuple (precision, approve_count, total_evaluated)
        """
        cutoff = utc_now() - timedelta(days=days)

        # Optimizado: Single query con conditional count (Code Review #1)
        from sqlalchemy import case

        counts = self.db.query(
            func.sum(case(
                (ProductCorrection.correction_type == CorrectionType.APPROVE.value, 1),
                else_=0
            )).label("approve_count"),
            func.sum(case(
                (ProductCorrection.correction_type == CorrectionType.CORRECT.value, 1),
                else_=0
            )).label("correct_count"),
        ).filter(
            and_(
                ProductCorrection.correction_type.in_([
                    CorrectionType.APPROVE.value,
                    CorrectionType.CORRECT.value
                ]),
                ProductCorrection.created_at >= cutoff,
            )
        ).first()

        approve_count = int(counts.approve_count or 0)
        correct_count = int(counts.correct_count or 0)
        total_evaluated = approve_count + correct_count

        if total_evaluated == 0:
            return 0.0, 0, 0

        precision = approve_count / total_evaluated
        return round(precision, 4), approve_count, total_evaluated

    def calculate_distribution_drift(
        self,
        days_current: int = 7,
        days_previous: int = 7,
    ) -> Dict[str, Any]:
        """
        Calcular drift de distribución de categorías semana a semana (Issue #466).

        Compara distribución de categorías de esta semana vs la anterior
        para detectar cambios anómalos.

        Args:
            days_current: Días de la semana actual (default 7)
            days_previous: Días de la semana anterior (default 7)

        Returns:
            Dict con:
                - has_drift: bool si hay drift significativo
                - max_drift: float mayor cambio porcentual
                - drifts: Dict[category, delta] para categorías con drift > threshold
                - this_week: Dict[category, pct] distribución actual
                - last_week: Dict[category, pct] distribución anterior
        """
        now = utc_now()
        week_start = now - timedelta(days=days_current)
        prev_week_start = week_start - timedelta(days=days_previous)

        # NOTE: Para catálogos grandes (67k+), considera añadir índice en
        # ProductCatalogVentaLibre.updated_at si estas queries son lentas.
        # Ver Code Review #3.

        # Distribución semana actual
        this_week_raw = self.db.query(
            ProductCatalogVentaLibre.ml_category,
            func.count(ProductCatalogVentaLibre.id).label("count")
        ).filter(
            and_(
                ProductCatalogVentaLibre.ml_category.isnot(None),
                ProductCatalogVentaLibre.updated_at >= week_start,
            )
        ).group_by(
            ProductCatalogVentaLibre.ml_category
        ).all()

        # Distribución semana anterior
        last_week_raw = self.db.query(
            ProductCatalogVentaLibre.ml_category,
            func.count(ProductCatalogVentaLibre.id).label("count")
        ).filter(
            and_(
                ProductCatalogVentaLibre.ml_category.isnot(None),
                ProductCatalogVentaLibre.updated_at >= prev_week_start,
                ProductCatalogVentaLibre.updated_at < week_start,
            )
        ).group_by(
            ProductCatalogVentaLibre.ml_category
        ).all()

        # Convertir a porcentajes
        def to_pct_dict(raw_data):
            total = sum(count for _, count in raw_data)
            if total == 0:
                return {}
            return {cat: round(count / total, 4) for cat, count in raw_data}

        this_week = to_pct_dict(this_week_raw)
        last_week = to_pct_dict(last_week_raw)

        # Sin datos de semana anterior, no hay baseline para calcular drift
        if not last_week:
            return {
                "has_drift": False,
                "max_drift": 0.0,
                "drift_count": 0,
                "drifts": {},
                "this_week": this_week,
                "last_week": last_week,
            }

        # Calcular deltas
        all_categories = set(this_week.keys()) | set(last_week.keys())
        drifts = {}
        max_drift = 0.0

        for category in all_categories:
            pct_this = this_week.get(category, 0)
            pct_last = last_week.get(category, 0)
            delta = abs(pct_this - pct_last)

            if delta > ALERT_THRESHOLDS["drift_warning"]:
                drifts[category] = {
                    "this_week_pct": pct_this,
                    "last_week_pct": pct_last,
                    "delta": round(delta, 4),
                    "direction": "up" if pct_this > pct_last else "down",
                }

            if delta > max_drift:
                max_drift = delta

        return {
            "has_drift": len(drifts) > 0,
            "max_drift": round(max_drift, 4),
            "drift_count": len(drifts),
            "drifts": drifts,
            "this_week": this_week,
            "last_week": last_week,
        }

    # =========================================================================
    # SNAPSHOT & RECORDING
    # =========================================================================

    def take_snapshot(self) -> Dict[str, Any]:
        """
        Tomar snapshot completo de todas las métricas del clasificador.

        Returns:
            Dict con todas las métricas
        """
        entropy = self.calculate_cluster_entropy()
        outlier_rate, outlier_count, total_with_umap = self.calculate_outlier_rate()
        avg_confidence = self.calculate_avg_ml_confidence()
        pending_count = self.count_pending_validation()
        decay = self.calculate_model_decay()
        coverage, classified_count, total_count = self.calculate_coverage()  # Issue #458
        # Issue #465: Precision KPI
        precision, approve_count, evaluated_count = self.calculate_precision()
        # Issue #466: Drift Detection
        drift_data = self.calculate_distribution_drift()

        snapshot = {
            "timestamp": utc_now().isoformat(),
            "cluster_entropy": entropy,
            "outlier_rate": outlier_rate,
            "outlier_count": outlier_count,
            "total_with_umap": total_with_umap,
            "avg_ml_confidence": avg_confidence,
            "validation_pending": pending_count,
            "model_decay_7d": decay,
            # Issue #458: Coverage KPI
            "coverage": coverage,
            "classified_count": classified_count,
            "total_count": total_count,
            # Issue #465: Precision KPI
            "precision": precision,
            "precision_approve_count": approve_count,
            "precision_evaluated_count": evaluated_count,
            # Issue #466: Drift Detection
            "has_drift": drift_data["has_drift"],
            "max_drift": drift_data["max_drift"],
            "drift_count": drift_data["drift_count"],
            "drift_details": drift_data["drifts"],
        }

        # Calcular alerts basados en umbrales
        snapshot["alerts"] = self._evaluate_alerts(snapshot)

        logger.info(
            "classifier_monitoring.snapshot",
            entropy=entropy,
            outlier_rate=outlier_rate,
            pending=pending_count,
            precision=precision,
            has_drift=drift_data["has_drift"],
            alerts=len(snapshot["alerts"]),
        )

        return snapshot

    def record_metrics(self, snapshot: Optional[Dict] = None) -> List[SystemHealthMetric]:
        """
        Guardar métricas del snapshot en la base de datos.

        NOTA: Las métricas precision y drift (Issues #465, #466) se calculan
        on-demand y NO se almacenan en SystemHealthMetric. Están disponibles
        en tiempo real vía take_snapshot() y los endpoints /classifier/precision
        y /classifier/drift.

        Args:
            snapshot: Snapshot a guardar (si None, calcula uno nuevo)

        Returns:
            Lista de métricas guardadas
        """
        if snapshot is None:
            snapshot = self.take_snapshot()

        metrics = []
        now = utc_now()

        # Guardar cada métrica
        metric_fields = [
            ("cluster_entropy", snapshot["cluster_entropy"], "ratio"),
            ("outlier_rate", snapshot["outlier_rate"], "ratio"),
            ("outlier_count", snapshot["outlier_count"], "count"),
            ("avg_ml_confidence", snapshot["avg_ml_confidence"], "ratio"),
            ("validation_pending", snapshot["validation_pending"], "count"),
        ]

        if snapshot.get("model_decay_7d") is not None:
            metric_fields.append(
                ("model_decay_7d", snapshot["model_decay_7d"], "ratio")
            )

        for metric_name, value, unit in metric_fields:
            metric = SystemHealthMetric(
                category=METRIC_CATEGORY,
                metric_name=metric_name,
                metric_value=value,
                metric_unit=unit,
                measured_at=now,
            )
            self.db.add(metric)
            metrics.append(metric)

        try:
            self.db.commit()
            logger.info(
                "classifier_monitoring.metrics_recorded",
                count=len(metrics),
            )
        except Exception as e:
            logger.error("classifier_monitoring.record_failed", error=str(e))
            self.db.rollback()
            raise

        return metrics

    def _evaluate_alerts(self, snapshot: Dict) -> List[Dict]:
        """
        Evaluar umbrales y generar alertas.

        Returns:
            Lista de alertas con severidad y mensaje
        """
        alerts = []

        # Outlier rate
        outlier_rate = snapshot.get("outlier_rate", 0)
        if outlier_rate > ALERT_THRESHOLDS["outlier_rate_critical"]:
            alerts.append({
                "severity": "critical",
                "metric": "outlier_rate",
                "value": outlier_rate,
                "threshold": ALERT_THRESHOLDS["outlier_rate_critical"],
                "message": f"Tasa de outliers crítica: {outlier_rate:.1%} (umbral: {ALERT_THRESHOLDS['outlier_rate_critical']:.0%})",
            })
        elif outlier_rate > ALERT_THRESHOLDS["outlier_rate_warning"]:
            alerts.append({
                "severity": "warning",
                "metric": "outlier_rate",
                "value": outlier_rate,
                "threshold": ALERT_THRESHOLDS["outlier_rate_warning"],
                "message": f"Tasa de outliers elevada: {outlier_rate:.1%}",
            })

        # Entropy
        entropy = snapshot.get("cluster_entropy", 1)
        if entropy < ALERT_THRESHOLDS["entropy_warning"]:
            alerts.append({
                "severity": "warning",
                "metric": "cluster_entropy",
                "value": entropy,
                "threshold": ALERT_THRESHOLDS["entropy_warning"],
                "message": f"Distribución de clusters sesgada (entropía: {entropy:.2f})",
            })

        # Confidence
        confidence = snapshot.get("avg_ml_confidence", 1)
        if confidence < ALERT_THRESHOLDS["confidence_warning"]:
            alerts.append({
                "severity": "warning",
                "metric": "avg_ml_confidence",
                "value": confidence,
                "threshold": ALERT_THRESHOLDS["confidence_warning"],
                "message": f"Confianza del modelo baja: {confidence:.1%}",
            })

        # Pending validation
        pending = snapshot.get("validation_pending", 0)
        if pending > ALERT_THRESHOLDS["pending_critical"]:
            alerts.append({
                "severity": "critical",
                "metric": "validation_pending",
                "value": pending,
                "threshold": ALERT_THRESHOLDS["pending_critical"],
                "message": f"Backlog de validación crítico: {pending} productos",
            })
        elif pending > ALERT_THRESHOLDS["pending_warning"]:
            alerts.append({
                "severity": "warning",
                "metric": "validation_pending",
                "value": pending,
                "threshold": ALERT_THRESHOLDS["pending_warning"],
                "message": f"Backlog de validación elevado: {pending} productos",
            })

        # Model decay
        decay = snapshot.get("model_decay_7d")
        if decay is not None and decay < -ALERT_THRESHOLDS["decay_warning"]:
            alerts.append({
                "severity": "warning",
                "metric": "model_decay_7d",
                "value": decay,
                "threshold": -ALERT_THRESHOLDS["decay_warning"],
                "message": f"Degradación del modelo detectada: {decay:.1%} en 7 días",
            })

        # Issue #465: Precision alerts
        precision = snapshot.get("precision")
        evaluated = snapshot.get("precision_evaluated_count", 0)
        if precision is not None and evaluated > 0:
            if precision < ALERT_THRESHOLDS["precision_critical"]:
                alerts.append({
                    "severity": "critical",
                    "metric": "precision",
                    "value": precision,
                    "threshold": ALERT_THRESHOLDS["precision_critical"],
                    "message": f"Precisión crítica: {precision:.1%} (basado en {evaluated} evaluaciones)",
                })
            elif precision < ALERT_THRESHOLDS["precision_warning"]:
                alerts.append({
                    "severity": "warning",
                    "metric": "precision",
                    "value": precision,
                    "threshold": ALERT_THRESHOLDS["precision_warning"],
                    "message": f"Precisión baja: {precision:.1%} (basado en {evaluated} evaluaciones)",
                })

        # Issue #466: Drift alerts
        max_drift = snapshot.get("max_drift", 0)
        drift_details = snapshot.get("drift_details", {})
        if max_drift > ALERT_THRESHOLDS["drift_critical"]:
            # Find the category with max drift
            top_drift_cat = max(drift_details.keys(), key=lambda k: drift_details[k]["delta"]) if drift_details else "desconocida"
            alerts.append({
                "severity": "critical",
                "metric": "distribution_drift",
                "value": max_drift,
                "threshold": ALERT_THRESHOLDS["drift_critical"],
                "message": f"Drift crítico detectado: {top_drift_cat} cambió {max_drift:.1%}",
            })
        elif max_drift > ALERT_THRESHOLDS["drift_warning"]:
            top_drift_cat = max(drift_details.keys(), key=lambda k: drift_details[k]["delta"]) if drift_details else "desconocida"
            alerts.append({
                "severity": "warning",
                "metric": "distribution_drift",
                "value": max_drift,
                "threshold": ALERT_THRESHOLDS["drift_warning"],
                "message": f"Drift detectado: {top_drift_cat} cambió {max_drift:.1%}",
            })

        return alerts

    # =========================================================================
    # HISTORY & TRENDS
    # =========================================================================

    def get_metrics_history(
        self,
        days: int = 30,
        metric_names: Optional[List[str]] = None,
    ) -> Dict[str, List[Dict]]:
        """
        Obtener historial de métricas para gráficas de tendencias.

        Args:
            days: Días de historial a recuperar
            metric_names: Lista de métricas específicas (o todas si None)

        Returns:
            Dict con series temporales por métrica
        """
        cutoff = utc_now() - timedelta(days=days)

        query = self.db.query(SystemHealthMetric).filter(
            and_(
                SystemHealthMetric.category == METRIC_CATEGORY,
                SystemHealthMetric.measured_at >= cutoff,
            )
        )

        if metric_names:
            query = query.filter(
                SystemHealthMetric.metric_name.in_(metric_names)
            )

        metrics = query.order_by(SystemHealthMetric.measured_at).all()

        # Agrupar por nombre de métrica
        history = defaultdict(list)
        for m in metrics:
            history[m.metric_name].append({
                "timestamp": m.measured_at.isoformat(),
                "value": m.metric_value,
            })

        return dict(history)

    def get_category_distribution(self) -> List[Dict]:
        """
        Obtener distribución actual de productos por categoría.

        Returns:
            Lista de categorías con conteos
        """
        distribution = self.db.query(
            ProductCatalogVentaLibre.ml_category,
            func.count(ProductCatalogVentaLibre.id).label("count"),
            func.avg(ProductCatalogVentaLibre.ml_confidence).label("avg_confidence"),
        ).filter(
            ProductCatalogVentaLibre.ml_category.isnot(None)
        ).group_by(
            ProductCatalogVentaLibre.ml_category
        ).order_by(
            desc("count")
        ).all()

        return [
            {
                "category": cat or "sin_clasificar",
                "count": count,
                "avg_confidence": round(conf or 0, 4),
            }
            for cat, count, conf in distribution
        ]

    def get_high_risk_products(
        self,
        limit: int = 20,
        z_score_threshold: float = 2.5,
    ) -> List[Dict]:
        """
        Obtener productos de alto riesgo para revisión prioritaria.

        Prioriza productos con:
        - is_outlier=True (marcados como outliers)
        - Baja confianza ML (< 0.6)
        - No verificados humanamente

        Args:
            limit: Máximo de productos a devolver
            z_score_threshold: No usado - se mantiene por compatibilidad

        Returns:
            Lista de productos de alto riesgo
        """
        from sqlalchemy import or_

        # Productos de alto riesgo: outliers O baja confianza, no verificados
        products = self.db.query(ProductCatalogVentaLibre).filter(
            and_(
                ProductCatalogVentaLibre.is_active == True,  # noqa: E712
                ProductCatalogVentaLibre.human_verified == False,  # noqa: E712
                or_(
                    ProductCatalogVentaLibre.is_outlier == True,  # noqa: E712
                    ProductCatalogVentaLibre.ml_confidence < 0.6,
                )
            )
        ).order_by(
            # Priorizar: outliers primero, luego por menor confianza
            desc(ProductCatalogVentaLibre.is_outlier),
            ProductCatalogVentaLibre.ml_confidence.asc().nullsfirst(),
        ).limit(limit).all()

        return [
            {
                "id": str(p.id),
                "product_name": p.product_name_display,
                "ml_category": p.ml_category,
                "ml_confidence": p.ml_confidence,
                "is_outlier": p.is_outlier,
                "detected_brand": p.detected_brand,
            }
            for p in products
        ]


# =============================================================================
# MODULE-LEVEL SINGLETON FACTORY
# =============================================================================

def get_classifier_monitoring_service(db: Session) -> ClassifierMonitoringService:
    """Factory function para obtener instancia del servicio."""
    return ClassifierMonitoringService(db)
